/*
 * Decompiled with CFR 0.152.
 */
package org.apache.iotdb.db.queryengine.plan.expression.multi.builtin.helper;

import java.time.LocalDate;
import java.time.ZoneId;
import java.time.format.DateTimeParseException;
import java.util.Map;
import javax.annotation.Nonnull;
import org.apache.iotdb.db.exception.sql.SemanticException;
import org.apache.iotdb.db.queryengine.common.SessionInfo;
import org.apache.iotdb.db.queryengine.plan.expression.multi.FunctionExpression;
import org.apache.iotdb.db.queryengine.plan.expression.multi.builtin.BuiltInScalarFunctionHelper;
import org.apache.iotdb.db.queryengine.transformation.api.LayerReader;
import org.apache.iotdb.db.queryengine.transformation.dag.column.ColumnTransformer;
import org.apache.iotdb.db.queryengine.transformation.dag.column.unary.scalar.CastFunctionColumnTransformer;
import org.apache.iotdb.db.queryengine.transformation.dag.transformer.Transformer;
import org.apache.iotdb.db.queryengine.transformation.dag.transformer.unary.scalar.CastFunctionTransformer;
import org.apache.iotdb.db.utils.DateTimeUtils;
import org.apache.tsfile.common.conf.TSFileConfig;
import org.apache.tsfile.enums.TSDataType;
import org.apache.tsfile.read.common.type.Type;
import org.apache.tsfile.read.common.type.TypeFactory;
import org.apache.tsfile.utils.Binary;
import org.apache.tsfile.utils.BytesUtils;
import org.apache.tsfile.utils.DateUtils;

public class CastFunctionHelper
implements BuiltInScalarFunctionHelper {
    public static final String ERROR_MSG = "Unsupported target dataType: %s";

    @Override
    public void checkBuiltInScalarFunctionInputDataType(TSDataType tsDataType) throws SemanticException {
    }

    @Override
    public TSDataType getBuiltInScalarFunctionReturnType(FunctionExpression functionExpression) {
        if (!functionExpression.getFunctionAttributes().containsKey("type")) {
            throw new SemanticException("Function Cast must specify a target data type.");
        }
        return TSDataType.valueOf((String)functionExpression.getFunctionAttributes().get("type").toUpperCase());
    }

    @Override
    public ColumnTransformer getBuiltInScalarFunctionColumnTransformer(FunctionExpression expression, ColumnTransformer columnTransformer) {
        return new CastFunctionColumnTransformer(TypeFactory.getType((TSDataType)this.getBuiltInScalarFunctionReturnType(expression)), columnTransformer, ZoneId.systemDefault());
    }

    @Override
    public Transformer getBuiltInScalarFunctionTransformer(FunctionExpression expression, LayerReader layerReader) {
        return new CastFunctionTransformer(layerReader, this.getBuiltInScalarFunctionReturnType(expression));
    }

    @Override
    public void appendFunctionAttributes(boolean hasExpression, StringBuilder builder, Map<String, String> functionAttributes) {
        builder.append(" AS ");
        builder.append(functionAttributes.entrySet().iterator().next().getValue());
    }

    public static int castLongToInt(long value) {
        if (value > Integer.MAX_VALUE || value < Integer.MIN_VALUE) {
            throw new SemanticException(String.format("long value %d is out of range of integer value.", value));
        }
        return (int)value;
    }

    public static int castFloatToInt(float value) {
        if (value > 2.1474836E9f || value < -2.1474836E9f) {
            throw new SemanticException(String.format("Float value %f is out of range of integer value.", Float.valueOf(value)));
        }
        return Math.round(value);
    }

    public static long castFloatToLong(float value) {
        if (value > 9.223372E18f || value < -9.223372E18f) {
            throw new SemanticException(String.format("Float value %f is out of range of long value.", Float.valueOf(value)));
        }
        return Math.round((double)value);
    }

    public static int castDoubleToInt(double value) {
        if (value > 2.147483647E9 || value < -2.147483648E9) {
            throw new SemanticException(String.format("Double value %f is out of range of integer value.", value));
        }
        return Math.round((float)value);
    }

    public static long castDoubleToLong(double value) {
        if (value > 9.223372036854776E18 || value < -9.223372036854776E18) {
            throw new SemanticException(String.format("Double value %f is out of range of long value.", value));
        }
        return Math.round(value);
    }

    public static float castDoubleToFloat(double value) {
        if (value > 3.4028234663852886E38 || value < -3.4028234663852886E38) {
            throw new SemanticException(String.format("Double value %f is out of range of float value.", value));
        }
        return (float)value;
    }

    public static float castTextToFloat(String value) {
        float f = Float.parseFloat(value);
        if (f == Float.POSITIVE_INFINITY || f == Float.NEGATIVE_INFINITY) {
            throw new SemanticException(String.format("Text value %s is out of range of float value.", value));
        }
        return f;
    }

    public static double castTextToDouble(String value) {
        double d = Double.parseDouble(value);
        if (d == Double.POSITIVE_INFINITY || d == Double.NEGATIVE_INFINITY) {
            throw new SemanticException(String.format("Text value %s is out of range of double value.", value));
        }
        return d;
    }

    public static boolean castTextToBoolean(String value) {
        String lowerCase = value.toLowerCase();
        if (lowerCase.equals("true")) {
            return true;
        }
        if (lowerCase.equals("false")) {
            return false;
        }
        throw new SemanticException(String.format("Invalid text input for boolean type: %s", value));
    }

    public static Object cast(@Nonnull Object value, Type sourceType, Type targetType, SessionInfo session) {
        switch (sourceType.getTypeEnum()) {
            case INT32: {
                int intV = value instanceof Integer ? ((Integer)value).intValue() : ((Long)value).intValue();
                return CastFunctionHelper.castInt(intV, targetType);
            }
            case DATE: {
                int dateV = value instanceof Integer ? ((Integer)value).intValue() : ((Long)value).intValue();
                return CastFunctionHelper.castDate(dateV, targetType, session.getZoneId());
            }
            case INT64: {
                long longV = (Long)value;
                return CastFunctionHelper.castLong(longV, targetType);
            }
            case TIMESTAMP: {
                long timestampV = (Long)value;
                return CastFunctionHelper.castTimestamp(timestampV, targetType, session.getZoneId());
            }
            case FLOAT: {
                float floatV = value instanceof Float ? ((Float)value).floatValue() : ((Double)value).floatValue();
                return CastFunctionHelper.castFloat(floatV, targetType);
            }
            case DOUBLE: {
                double doubleV = (Double)value;
                return CastFunctionHelper.castDouble(doubleV, targetType);
            }
            case BOOLEAN: {
                boolean boolV = (Boolean)value;
                return CastFunctionHelper.castBool(boolV, targetType);
            }
            case TEXT: 
            case STRING: 
            case BLOB: {
                Binary binaryV = (Binary)value;
                return CastFunctionHelper.castBinary(binaryV, targetType, session.getZoneId());
            }
        }
        throw new UnsupportedOperationException(String.format("Unsupported source dataType: %s", sourceType.getTypeEnum()));
    }

    private static Object castInt(int value, Type targetType) {
        switch (targetType.getTypeEnum()) {
            case INT32: 
            case DATE: {
                return value;
            }
            case INT64: 
            case TIMESTAMP: {
                return (long)value;
            }
            case FLOAT: {
                return Float.valueOf(value);
            }
            case DOUBLE: {
                return (double)value;
            }
            case BOOLEAN: {
                return value != 0;
            }
            case TEXT: 
            case STRING: {
                return BytesUtils.valueOf((String)String.valueOf(value));
            }
            case BLOB: {
                return new Binary(BytesUtils.intToBytes((int)value));
            }
        }
        throw new UnsupportedOperationException(String.format(ERROR_MSG, targetType.getTypeEnum()));
    }

    private static Object castDate(int value, Type targetType, ZoneId zoneId) {
        switch (targetType.getTypeEnum()) {
            case INT32: 
            case DATE: {
                return value;
            }
            case INT64: {
                return (long)value;
            }
            case TIMESTAMP: {
                return DateTimeUtils.correctPrecision(DateUtils.parseIntToTimestamp((int)value, (ZoneId)zoneId));
            }
            case FLOAT: {
                return Float.valueOf(value);
            }
            case DOUBLE: {
                return (double)value;
            }
            case BOOLEAN: {
                return value != 0;
            }
            case TEXT: 
            case STRING: {
                return BytesUtils.valueOf((String)DateUtils.formatDate((int)value));
            }
            case BLOB: {
                return new Binary(BytesUtils.intToBytes((int)value));
            }
        }
        throw new UnsupportedOperationException(String.format(ERROR_MSG, targetType.getTypeEnum()));
    }

    private static Object castLong(long value, Type targetType) {
        switch (targetType.getTypeEnum()) {
            case INT32: 
            case DATE: {
                return CastFunctionHelper.castLongToInt(value);
            }
            case INT64: 
            case TIMESTAMP: {
                return value;
            }
            case FLOAT: {
                return Float.valueOf(value);
            }
            case DOUBLE: {
                return (double)value;
            }
            case BOOLEAN: {
                return value != 0L;
            }
            case TEXT: 
            case STRING: {
                return BytesUtils.valueOf((String)String.valueOf(value));
            }
            case BLOB: {
                return new Binary(BytesUtils.longToBytes((long)value));
            }
        }
        throw new UnsupportedOperationException(String.format(ERROR_MSG, targetType.getTypeEnum()));
    }

    private static Object castTimestamp(long value, Type targetType, ZoneId zoneId) {
        switch (targetType.getTypeEnum()) {
            case INT32: {
                return CastFunctionHelper.castLongToInt(value);
            }
            case DATE: {
                return DateUtils.parseDateExpressionToInt((LocalDate)DateTimeUtils.convertToLocalDate(value, zoneId));
            }
            case INT64: 
            case TIMESTAMP: {
                return value;
            }
            case FLOAT: {
                return Float.valueOf(value);
            }
            case DOUBLE: {
                return (double)value;
            }
            case BOOLEAN: {
                return value != 0L;
            }
            case TEXT: 
            case STRING: {
                return BytesUtils.valueOf((String)DateTimeUtils.convertLongToDate(value, zoneId));
            }
            case BLOB: {
                return new Binary(BytesUtils.longToBytes((long)value));
            }
        }
        throw new UnsupportedOperationException(String.format(ERROR_MSG, targetType.getTypeEnum()));
    }

    private static Object castFloat(float value, Type targetType) {
        switch (targetType.getTypeEnum()) {
            case INT32: 
            case DATE: {
                return CastFunctionHelper.castFloatToInt(value);
            }
            case INT64: 
            case TIMESTAMP: {
                return CastFunctionHelper.castFloatToLong(value);
            }
            case FLOAT: {
                return Float.valueOf(value);
            }
            case DOUBLE: {
                return (double)value;
            }
            case BOOLEAN: {
                return value != 0.0f;
            }
            case TEXT: 
            case STRING: {
                return BytesUtils.valueOf((String)String.valueOf(value));
            }
            case BLOB: {
                return new Binary(BytesUtils.floatToBytes((float)value));
            }
        }
        throw new UnsupportedOperationException(String.format(ERROR_MSG, targetType.getTypeEnum()));
    }

    private static Object castDouble(double value, Type targetType) {
        switch (targetType.getTypeEnum()) {
            case INT32: 
            case DATE: {
                return CastFunctionHelper.castDoubleToInt(value);
            }
            case INT64: 
            case TIMESTAMP: {
                return CastFunctionHelper.castDoubleToLong(value);
            }
            case FLOAT: {
                return Float.valueOf(CastFunctionHelper.castDoubleToFloat(value));
            }
            case DOUBLE: {
                return value;
            }
            case BOOLEAN: {
                return value != 0.0;
            }
            case TEXT: 
            case STRING: {
                return BytesUtils.valueOf((String)String.valueOf(value));
            }
            case BLOB: {
                return new Binary(BytesUtils.doubleToBytes((double)value));
            }
        }
        throw new UnsupportedOperationException(String.format(ERROR_MSG, targetType.getTypeEnum()));
    }

    private static Object castBool(boolean value, Type targetType) {
        switch (targetType.getTypeEnum()) {
            case INT32: 
            case DATE: {
                return value ? 1 : 0;
            }
            case INT64: 
            case TIMESTAMP: {
                return value ? 1L : 0L;
            }
            case FLOAT: {
                return Float.valueOf(value ? 1.0f : 0.0f);
            }
            case DOUBLE: {
                return value ? 1.0 : 0.0;
            }
            case BOOLEAN: {
                return value;
            }
            case TEXT: 
            case STRING: {
                return BytesUtils.valueOf((String)String.valueOf(value));
            }
            case BLOB: {
                return new Binary(BytesUtils.boolToBytes((boolean)value));
            }
        }
        throw new UnsupportedOperationException(String.format(ERROR_MSG, targetType.getTypeEnum()));
    }

    private static Object castBinary(Binary value, Type targetType, ZoneId zoneId) {
        String stringValue = value.getStringValue(TSFileConfig.STRING_CHARSET);
        try {
            switch (targetType.getTypeEnum()) {
                case INT32: {
                    return Integer.parseInt(stringValue);
                }
                case DATE: {
                    return DateUtils.parseDateExpressionToInt((String)stringValue);
                }
                case INT64: {
                    return Long.parseLong(stringValue);
                }
                case TIMESTAMP: {
                    return DateTimeUtils.convertDatetimeStrToLong(stringValue, zoneId);
                }
                case FLOAT: {
                    return Float.valueOf(CastFunctionHelper.castTextToFloat(stringValue));
                }
                case DOUBLE: {
                    return CastFunctionHelper.castTextToDouble(stringValue);
                }
                case BOOLEAN: {
                    return CastFunctionHelper.castTextToBoolean(stringValue);
                }
                case TEXT: 
                case STRING: 
                case BLOB: {
                    return value;
                }
            }
            throw new UnsupportedOperationException(String.format(ERROR_MSG, targetType.getTypeEnum()));
        }
        catch (NumberFormatException | DateTimeParseException e) {
            throw new SemanticException(String.format("Cannot cast %s to %s type", stringValue, targetType.getDisplayName()));
        }
    }
}

