/*
 * Decompiled with CFR 0.152.
 */
package org.apache.cassandra.service.paxos.uncommitted;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import java.io.IOException;
import java.nio.ByteBuffer;
import java.util.concurrent.atomic.AtomicReference;
import java.util.zip.CRC32;
import org.apache.cassandra.db.rows.Row;
import org.apache.cassandra.io.util.File;
import org.apache.cassandra.io.util.RandomAccessReader;
import org.apache.cassandra.io.util.SequentialWriter;
import org.apache.cassandra.io.util.SequentialWriterOption;
import org.apache.cassandra.net.Crc;
import org.apache.cassandra.service.ClientState;
import org.apache.cassandra.service.paxos.Ballot;
import org.apache.cassandra.service.paxos.Commit;
import org.apache.cassandra.service.paxos.uncommitted.PaxosRows;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class PaxosBallotTracker {
    private static final Logger logger = LoggerFactory.getLogger(PaxosBallotTracker.class);
    private static final int FILE_VERSION = 0;
    static final String FNAME = "ballot.meta";
    private static final String TMP_FNAME = "ballot.meta.tmp";
    private final File directory;
    private final AtomicReference<Ballot> highBound;
    private volatile Ballot lowBound;

    private PaxosBallotTracker(File directory, Ballot highBound, Ballot lowBound) {
        Preconditions.checkNotNull((Object)lowBound);
        Preconditions.checkNotNull((Object)highBound);
        this.directory = directory;
        this.highBound = new AtomicReference<Ballot>(highBound);
        this.lowBound = lowBound;
    }

    private static void serializeBallot(SequentialWriter writer, CRC32 crc, Ballot ballot) throws IOException {
        ByteBuffer bytes = ballot.toBytes();
        writer.write(bytes);
        crc.update(bytes);
    }

    private static Ballot deserializeBallot(RandomAccessReader reader, CRC32 crc, byte[] bytes) throws IOException {
        reader.readFully(bytes);
        crc.update(bytes);
        return Ballot.deserialize(ByteBuffer.wrap(bytes));
    }

    public static void truncate(File directory) throws IOException {
        logger.info("truncating paxos ballot metadata in {}", (Object)directory);
        PaxosBallotTracker.deleteIfExists(new File(directory, TMP_FNAME));
        PaxosBallotTracker.deleteIfExists(new File(directory, FNAME));
    }

    public static PaxosBallotTracker load(File directory) throws IOException {
        PaxosBallotTracker.deleteIfExists(new File(directory, TMP_FNAME));
        File file = new File(directory, FNAME);
        if (!file.exists()) {
            return new PaxosBallotTracker(directory, Ballot.none(), Ballot.none());
        }
        try (RandomAccessReader reader = RandomAccessReader.open(file);){
            int version = reader.readInt();
            if (version != 0) {
                throw new IOException("Unsupported ballot file version: " + version);
            }
            byte[] bytes = new byte[16];
            CRC32 crc = Crc.crc32();
            Ballot highBallot = PaxosBallotTracker.deserializeBallot(reader, crc, bytes);
            Ballot lowBallot = PaxosBallotTracker.deserializeBallot(reader, crc, bytes);
            int checksum = Integer.reverseBytes(reader.readInt());
            if (!reader.isEOF() || (int)crc.getValue() != checksum) {
                throw new IOException("Ballot file corrupted");
            }
            PaxosBallotTracker paxosBallotTracker = new PaxosBallotTracker(directory, highBallot, lowBallot);
            return paxosBallotTracker;
        }
    }

    private static void deleteIfExists(File file) {
        if (file.exists()) {
            file.delete();
        }
    }

    public synchronized void flush() throws IOException {
        File file = new File(this.directory, TMP_FNAME);
        PaxosBallotTracker.deleteIfExists(file);
        try (SequentialWriter writer = new SequentialWriter(file, SequentialWriterOption.FINISH_ON_CLOSE);){
            CRC32 crc = Crc.crc32();
            writer.writeInt(0);
            PaxosBallotTracker.serializeBallot(writer, crc, this.getHighBound());
            PaxosBallotTracker.serializeBallot(writer, crc, this.getLowBound());
            writer.writeInt(Integer.reverseBytes((int)crc.getValue()));
        }
        file.move(new File(this.directory, FNAME));
    }

    public synchronized void truncate() {
        PaxosBallotTracker.deleteIfExists(new File(this.directory, TMP_FNAME));
        PaxosBallotTracker.deleteIfExists(new File(this.directory, FNAME));
        this.highBound.set(Ballot.none());
        this.lowBound = Ballot.none();
    }

    private void updateHighBound(Ballot current, Ballot next) {
        while (Commit.isAfter(next, current) && !this.highBound.compareAndSet(current, next)) {
            current = this.highBound.get();
        }
    }

    void updateHighBound(Ballot next) {
        this.updateHighBound(this.highBound.get(), next);
    }

    public void onUpdate(Row row) {
        Ballot next;
        Ballot current = this.highBound.get();
        if (current == (next = PaxosRows.getHighBallot(row, current))) {
            return;
        }
        this.updateHighBound(current, next);
    }

    @VisibleForTesting
    void updateHighBoundUnsafe(Ballot expected, Ballot update) {
        this.highBound.compareAndSet(expected, update);
    }

    public File getDirectory() {
        return this.directory;
    }

    public synchronized void updateLowBound(Ballot update) throws IOException {
        if (!Commit.isAfter(update, this.lowBound)) {
            logger.debug("Not updating lower bound with earlier or equal ballot from {} to {}", (Object)this.lowBound, (Object)update);
            return;
        }
        logger.debug("Updating lower bound from {} to {}", (Object)this.lowBound, (Object)update);
        ClientState.getTimestampForPaxos(this.lowBound.unixMicros());
        this.lowBound = update;
        this.flush();
    }

    public Ballot getHighBound() {
        return this.highBound.get();
    }

    public Ballot getLowBound() {
        return this.lowBound;
    }
}

