/*
 * Decompiled with CFR 0.152.
 */
package org.apache.cassandra.sidecar.utils;

import com.codahale.metrics.DefaultSettableGauge;
import com.google.inject.Inject;
import com.google.inject.Singleton;
import io.netty.handler.codec.http.HttpResponseStatus;
import io.vertx.core.Future;
import io.vertx.core.Handler;
import io.vertx.core.Promise;
import io.vertx.core.Vertx;
import io.vertx.ext.web.handler.HttpException;
import java.util.AbstractMap;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentLinkedQueue;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import org.apache.cassandra.sidecar.cluster.instance.InstanceMetadata;
import org.apache.cassandra.sidecar.common.server.TableOperations;
import org.apache.cassandra.sidecar.concurrent.ExecutorPools;
import org.apache.cassandra.sidecar.config.ServiceConfiguration;
import org.apache.cassandra.sidecar.metrics.DeltaGauge;
import org.apache.cassandra.sidecar.metrics.instance.InstanceMetrics;
import org.apache.cassandra.sidecar.utils.InstanceMetadataFetcher;
import org.apache.cassandra.sidecar.utils.SSTableUploadsPathBuilder;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.VisibleForTesting;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@Singleton
public class SSTableImporter {
    private static final Logger LOGGER = LoggerFactory.getLogger(SSTableImporter.class);
    public static final boolean DEFAULT_RESET_LEVEL = true;
    public static final boolean DEFAULT_CLEAR_REPAIRED = true;
    public static final boolean DEFAULT_VERIFY_SSTABLES = true;
    public static final boolean DEFAULT_VERIFY_TOKENS = true;
    public static final boolean DEFAULT_INVALIDATE_CACHES = true;
    public static final boolean DEFAULT_EXTENDED_VERIFY = true;
    public static final boolean DEFAULT_COPY_DATA = false;
    private final Vertx vertx;
    private final ExecutorPools executorPools;
    private final InstanceMetadataFetcher metadataFetcher;
    private final SSTableUploadsPathBuilder uploadPathBuilder;
    @VisibleForTesting
    final Map<ImportId, ImportQueue> importQueuePerHost;

    @Inject
    SSTableImporter(Vertx vertx, InstanceMetadataFetcher metadataFetcher, ServiceConfiguration configuration, ExecutorPools executorPools, SSTableUploadsPathBuilder uploadPathBuilder) {
        this.vertx = vertx;
        this.executorPools = executorPools;
        this.metadataFetcher = metadataFetcher;
        this.uploadPathBuilder = uploadPathBuilder;
        this.importQueuePerHost = new ConcurrentHashMap<ImportId, ImportQueue>();
        executorPools.internal().setPeriodic(configuration.sstableImportConfiguration().executeInterval().toMillis(), (Handler<Long>)((Handler)this::processPendingImports));
    }

    public Future<Void> scheduleImport(ImportOptions options) {
        Promise promise = Promise.promise();
        this.importQueuePerHost.computeIfAbsent(this.key(options), this::initializeQueue).offer(new AbstractMap.SimpleEntry<Promise, ImportOptions>(promise, options));
        return promise.future();
    }

    public boolean cancelImport(ImportOptions options) {
        ImportQueue queue = this.importQueuePerHost.get(this.key(options));
        boolean removed = false;
        if (queue != null) {
            removed = queue.removeIf(tuple -> options.equals(tuple.getValue()));
        }
        LOGGER.debug("Cancel import for options={} was {}removed", (Object)options, (Object)(removed ? "" : "not "));
        return removed;
    }

    private ImportId key(ImportOptions options) {
        return new ImportId(options.host, options.keyspace, options.tableName);
    }

    private ImportQueue initializeQueue(ImportId key) {
        return new ImportQueue();
    }

    private void processPendingImports(Long timerId) {
        this.reportPendingImportPerHost();
        for (ImportQueue queue : this.importQueuePerHost.values()) {
            if (queue.isEmpty()) continue;
            this.executorPools.internal().runBlocking(() -> this.maybeDrainImportQueue(queue));
        }
    }

    @VisibleForTesting
    void maybeDrainImportQueue(ImportQueue queue) {
        if (queue.tryLock()) {
            try {
                this.drainImportQueue(queue);
            }
            finally {
                queue.unlock();
            }
        }
    }

    private void drainImportQueue(ImportQueue queue) {
        AbstractMap.SimpleEntry pair;
        int successCount = 0;
        int failureCount = 0;
        InstanceMetrics instanceMetrics = null;
        while ((pair = (AbstractMap.SimpleEntry)queue.poll()) != null) {
            LOGGER.info("Starting SSTable import session");
            Promise promise = (Promise)pair.getKey();
            ImportOptions options = (ImportOptions)pair.getValue();
            InstanceMetadata instance = this.metadataFetcher.instance(options.host);
            if (instanceMetrics == null) {
                instanceMetrics = instance.metrics();
            }
            try {
                TableOperations tableOperations = instance.delegate().tableOperations();
                long startTime = System.nanoTime();
                List failedDirectories = tableOperations.importNewSSTables(options.keyspace, options.tableName, options.directory, options.resetLevel, options.clearRepaired, options.verifySSTables, options.verifyTokens, options.invalidateCaches, options.extendedVerify, options.copyData);
                long serviceTimeNanos = System.nanoTime() - startTime;
                if (!failedDirectories.isEmpty()) {
                    ++failureCount;
                    LOGGER.error("Failed to import SSTables with options={}, serviceTimeMillis={}, failedDirectories={}", new Object[]{options, TimeUnit.NANOSECONDS.toMillis(serviceTimeNanos), failedDirectories});
                    promise.fail((Throwable)new HttpException(HttpResponseStatus.INTERNAL_SERVER_ERROR.code(), "Failed to import from directories: " + failedDirectories));
                    continue;
                }
                ++successCount;
                LOGGER.info("Successfully imported SSTables with options={}, serviceTimeMillis={}", (Object)options, (Object)TimeUnit.NANOSECONDS.toMillis(serviceTimeNanos));
                promise.complete();
                this.cleanup(options);
            }
            catch (Exception exception) {
                ++failureCount;
                LOGGER.error("Failed to import SSTables with options={}", (Object)options, (Object)exception);
                promise.fail((Throwable)exception);
            }
        }
        if (successCount > 0 || failureCount > 0) {
            LOGGER.info("Finished SSTable import session with successCount={}, failureCount={}", (Object)successCount, (Object)failureCount);
            ((DeltaGauge)instanceMetrics.sstableImport().successfulImports.metric).update(successCount);
            ((DeltaGauge)instanceMetrics.sstableImport().failedImports.metric).update(failureCount);
        }
    }

    private void cleanup(ImportOptions options) {
        this.uploadPathBuilder.resolveUploadIdDirectory(options.host, options.uploadId).compose(this.uploadPathBuilder::isValidDirectory).compose(stagingDirectory -> this.vertx.fileSystem().deleteRecursive(stagingDirectory, true)).onSuccess(v -> LOGGER.debug("Successfully removed staging directory for uploadId={}, instance={}, options={}", new Object[]{options.uploadId, options.host, options})).onFailure(cause -> LOGGER.error("Failed to remove staging directory for uploadId={}, instance={}, options={}", new Object[]{options.uploadId, options.host, options, cause}));
    }

    private void reportPendingImportPerHost() {
        HashMap<String, Integer> aggregates = new HashMap<String, Integer>();
        for (Map.Entry<ImportId, ImportQueue> entry : this.importQueuePerHost.entrySet()) {
            aggregates.compute(entry.getKey().host, (k, v) -> ((ImportQueue)entry.getValue()).size() + (v == null ? 0 : v));
        }
        aggregates.forEach((host, count) -> {
            try {
                InstanceMetadata instance = this.metadataFetcher.instance((String)host);
                ((DefaultSettableGauge)instance.metrics().sstableImport().pendingImports.metric).setValue(count);
            }
            catch (Exception e) {
                LOGGER.warn("Unable to report metrics for host={} pendingImports={}", new Object[]{host, count, e});
            }
        });
    }

    @VisibleForTesting
    static class ImportId {
        private final String host;
        private final int hashCode;

        public ImportId(String host, String keyspace, String table) {
            this.host = host;
            this.hashCode = Objects.hash(host, keyspace, table);
        }

        public boolean equals(Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || this.getClass() != o.getClass()) {
                return false;
            }
            ImportId importId = (ImportId)o;
            return this.hashCode == importId.hashCode && Objects.equals(this.host, importId.host);
        }

        public int hashCode() {
            return this.hashCode;
        }
    }

    public static class ImportOptions {
        @NotNull
        final String host;
        @NotNull
        final String keyspace;
        @NotNull
        final String tableName;
        @NotNull
        final String directory;
        @NotNull
        final String uploadId;
        final boolean resetLevel;
        final boolean clearRepaired;
        final boolean verifySSTables;
        final boolean verifyTokens;
        final boolean invalidateCaches;
        final boolean extendedVerify;
        final boolean copyData;

        private ImportOptions(Builder builder) {
            this.host = Objects.requireNonNull(builder.host, "host is required");
            this.keyspace = Objects.requireNonNull(builder.keyspace, "keyspace is required");
            this.tableName = Objects.requireNonNull(builder.tableName, "tableName is required");
            this.directory = Objects.requireNonNull(builder.directory, "directory is required");
            this.uploadId = Objects.requireNonNull(builder.uploadId, "uploadId is required");
            this.resetLevel = builder.resetLevel;
            this.clearRepaired = builder.clearRepaired;
            this.verifySSTables = builder.verifySSTables;
            this.verifyTokens = builder.verifyTokens;
            this.invalidateCaches = builder.invalidateCaches;
            this.extendedVerify = builder.extendedVerify;
            this.copyData = builder.copyData;
        }

        public String host() {
            return this.host;
        }

        public String directory() {
            return this.directory;
        }

        public String uploadId() {
            return this.uploadId;
        }

        public boolean equals(Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || this.getClass() != o.getClass()) {
                return false;
            }
            ImportOptions options = (ImportOptions)o;
            return this.resetLevel == options.resetLevel && this.clearRepaired == options.clearRepaired && this.verifySSTables == options.verifySSTables && this.verifyTokens == options.verifyTokens && this.invalidateCaches == options.invalidateCaches && this.extendedVerify == options.extendedVerify && this.copyData == options.copyData && this.host.equals(options.host) && this.keyspace.equals(options.keyspace) && this.tableName.equals(options.tableName) && this.directory.equals(options.directory) && this.uploadId.equals(options.uploadId);
        }

        public int hashCode() {
            return Objects.hash(this.host, this.keyspace, this.tableName, this.directory, this.uploadId, this.resetLevel, this.clearRepaired, this.verifySSTables, this.verifyTokens, this.invalidateCaches, this.extendedVerify, this.copyData);
        }

        public String toString() {
            return "ImportOptions{host='" + this.host + "', keyspace='" + this.keyspace + "', tableName='" + this.tableName + "', directory='" + this.directory + "', uploadId='" + this.uploadId + "', resetLevel=" + this.resetLevel + ", clearRepaired=" + this.clearRepaired + ", verifySSTables=" + this.verifySSTables + ", verifyTokens=" + this.verifyTokens + ", invalidateCaches=" + this.invalidateCaches + ", extendedVerify=" + this.extendedVerify + ", copyData=" + this.copyData + "}";
        }

        public static final class Builder {
            private String host;
            private String keyspace;
            private String tableName;
            private String directory;
            private String uploadId;
            private boolean resetLevel = true;
            private boolean clearRepaired = true;
            private boolean verifySSTables = true;
            private boolean verifyTokens = true;
            private boolean invalidateCaches = true;
            private boolean extendedVerify = true;
            private boolean copyData = false;

            public Builder host(@NotNull String host) {
                this.host = host;
                return this;
            }

            public Builder keyspace(@NotNull String keyspace) {
                this.keyspace = keyspace;
                return this;
            }

            public Builder tableName(@NotNull String tableName) {
                this.tableName = tableName;
                return this;
            }

            public Builder directory(@NotNull String directory) {
                this.directory = directory;
                return this;
            }

            public Builder uploadId(@NotNull String uploadId) {
                this.uploadId = uploadId;
                return this;
            }

            public Builder resetLevel(boolean resetLevel) {
                this.resetLevel = resetLevel;
                return this;
            }

            public Builder clearRepaired(boolean clearRepaired) {
                this.clearRepaired = clearRepaired;
                return this;
            }

            public Builder verifySSTables(boolean verifySSTables) {
                this.verifySSTables = verifySSTables;
                return this;
            }

            public Builder verifyTokens(boolean verifyTokens) {
                this.verifyTokens = verifyTokens;
                return this;
            }

            public Builder invalidateCaches(boolean invalidateCaches) {
                this.invalidateCaches = invalidateCaches;
                return this;
            }

            public Builder extendedVerify(boolean extendedVerify) {
                this.extendedVerify = extendedVerify;
                return this;
            }

            public Builder copyData(boolean copyData) {
                this.copyData = copyData;
                return this;
            }

            public ImportOptions build() {
                return new ImportOptions(this);
            }
        }
    }

    static class ImportQueue
    extends ConcurrentLinkedQueue<AbstractMap.SimpleEntry<Promise<Void>, ImportOptions>> {
        private final AtomicBoolean isQueueInUse = new AtomicBoolean(false);

        ImportQueue() {
        }

        public boolean tryLock() {
            return this.isQueueInUse.compareAndSet(false, true);
        }

        public void unlock() {
            this.isQueueInUse.set(false);
        }
    }
}

