/*
 * Decompiled with CFR 0.152.
 */
package software.amazon.awssdk.auth.credentials;

import java.io.IOException;
import java.net.InetAddress;
import java.net.URI;
import java.net.UnknownHostException;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.time.Instant;
import java.time.temporal.ChronoUnit;
import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.Predicate;
import software.amazon.awssdk.annotations.SdkPublicApi;
import software.amazon.awssdk.auth.credentials.AwsCredentials;
import software.amazon.awssdk.auth.credentials.HttpCredentialsProvider;
import software.amazon.awssdk.auth.credentials.internal.ContainerCredentialsRetryPolicy;
import software.amazon.awssdk.auth.credentials.internal.HttpCredentialsLoader;
import software.amazon.awssdk.core.SdkSystemSetting;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.core.util.SdkUserAgent;
import software.amazon.awssdk.regions.util.ResourcesEndpointProvider;
import software.amazon.awssdk.regions.util.ResourcesEndpointRetryPolicy;
import software.amazon.awssdk.utils.ComparableUtils;
import software.amazon.awssdk.utils.StringUtils;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.Validate;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;
import software.amazon.awssdk.utils.cache.CachedSupplier;
import software.amazon.awssdk.utils.cache.NonBlocking;
import software.amazon.awssdk.utils.cache.RefreshResult;

@SdkPublicApi
public final class ContainerCredentialsProvider
implements HttpCredentialsProvider,
ToCopyableBuilder<Builder, ContainerCredentialsProvider> {
    private static final String PROVIDER_NAME = "ContainerCredentialsProvider";
    private static final Predicate<InetAddress> IS_LOOPBACK_ADDRESS = InetAddress::isLoopbackAddress;
    private static final Predicate<InetAddress> ALLOWED_HOSTS_RULES = IS_LOOPBACK_ADDRESS;
    private static final String HTTPS = "https";
    private static final String ECS_CONTAINER_HOST = "169.254.170.2";
    private static final String EKS_CONTAINER_HOST_IPV6 = "[fd00:ec2::23]";
    private static final String EKS_CONTAINER_HOST_IPV4 = "169.254.170.23";
    private static final List<String> VALID_LOOP_BACK_IPV4 = Arrays.asList("169.254.170.2", "169.254.170.23");
    private static final List<String> VALID_LOOP_BACK_IPV6 = Arrays.asList("[fd00:ec2::23]");
    private final String endpoint;
    private final HttpCredentialsLoader httpCredentialsLoader;
    private final CachedSupplier<AwsCredentials> credentialsCache;
    private final Boolean asyncCredentialUpdateEnabled;
    private final String asyncThreadName;

    private ContainerCredentialsProvider(BuilderImpl builder) {
        this.endpoint = builder.endpoint;
        this.asyncCredentialUpdateEnabled = builder.asyncCredentialUpdateEnabled;
        this.asyncThreadName = builder.asyncThreadName;
        this.httpCredentialsLoader = HttpCredentialsLoader.create(PROVIDER_NAME);
        if (Boolean.TRUE.equals(builder.asyncCredentialUpdateEnabled)) {
            Validate.paramNotBlank((CharSequence)builder.asyncThreadName, (String)"asyncThreadName");
            this.credentialsCache = CachedSupplier.builder(this::refreshCredentials).cachedValueName(this.toString()).prefetchStrategy((CachedSupplier.PrefetchStrategy)new NonBlocking(builder.asyncThreadName)).build();
        } else {
            this.credentialsCache = CachedSupplier.builder(this::refreshCredentials).cachedValueName(this.toString()).build();
        }
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public String toString() {
        return ToString.create((String)PROVIDER_NAME);
    }

    private RefreshResult<AwsCredentials> refreshCredentials() {
        HttpCredentialsLoader.LoadedCredentials loadedCredentials = this.httpCredentialsLoader.loadCredentials(new ContainerCredentialsEndpointProvider(this.endpoint));
        Instant expiration = loadedCredentials.getExpiration().orElse(null);
        return RefreshResult.builder((Object)loadedCredentials.getAwsCredentials()).staleTime(this.staleTime(expiration)).prefetchTime(this.prefetchTime(expiration)).build();
    }

    private Instant staleTime(Instant expiration) {
        if (expiration == null) {
            return null;
        }
        return expiration.minus(1L, ChronoUnit.MINUTES);
    }

    private Instant prefetchTime(Instant expiration) {
        Instant oneHourFromNow = Instant.now().plus(1L, ChronoUnit.HOURS);
        if (expiration == null) {
            return oneHourFromNow;
        }
        Instant fifteenMinutesBeforeExpiration = expiration.minus(15L, ChronoUnit.MINUTES);
        return (Instant)ComparableUtils.minimum((Comparable[])new Instant[]{oneHourFromNow, fifteenMinutesBeforeExpiration});
    }

    @Override
    public AwsCredentials resolveCredentials() {
        return (AwsCredentials)this.credentialsCache.get();
    }

    public void close() {
        this.credentialsCache.close();
    }

    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    private static final class BuilderImpl
    implements Builder {
        private String endpoint;
        private Boolean asyncCredentialUpdateEnabled;
        private String asyncThreadName;

        private BuilderImpl() {
            this.asyncThreadName("container-credentials-provider");
        }

        private BuilderImpl(ContainerCredentialsProvider credentialsProvider) {
            this.endpoint = credentialsProvider.endpoint;
            this.asyncCredentialUpdateEnabled = credentialsProvider.asyncCredentialUpdateEnabled;
            this.asyncThreadName = credentialsProvider.asyncThreadName;
        }

        @Override
        public Builder endpoint(String endpoint) {
            this.endpoint = endpoint;
            return this;
        }

        public void setEndpoint(String endpoint) {
            this.endpoint(endpoint);
        }

        @Override
        public Builder asyncCredentialUpdateEnabled(Boolean asyncCredentialUpdateEnabled) {
            this.asyncCredentialUpdateEnabled = asyncCredentialUpdateEnabled;
            return this;
        }

        public void setAsyncCredentialUpdateEnabled(boolean asyncCredentialUpdateEnabled) {
            this.asyncCredentialUpdateEnabled(asyncCredentialUpdateEnabled);
        }

        @Override
        public Builder asyncThreadName(String asyncThreadName) {
            this.asyncThreadName = asyncThreadName;
            return this;
        }

        public void setAsyncThreadName(String asyncThreadName) {
            this.asyncThreadName(asyncThreadName);
        }

        @Override
        public ContainerCredentialsProvider build() {
            return new ContainerCredentialsProvider(this);
        }
    }

    public static interface Builder
    extends HttpCredentialsProvider.Builder<ContainerCredentialsProvider, Builder>,
    CopyableBuilder<Builder, ContainerCredentialsProvider> {
    }

    static final class ContainerCredentialsEndpointProvider
    implements ResourcesEndpointProvider {
        private final String endpoint;

        ContainerCredentialsEndpointProvider(String endpoint) {
            this.endpoint = endpoint;
        }

        public URI endpoint() throws IOException {
            if (!SdkSystemSetting.AWS_CONTAINER_CREDENTIALS_RELATIVE_URI.getStringValue().isPresent() && !SdkSystemSetting.AWS_CONTAINER_CREDENTIALS_FULL_URI.getStringValue().isPresent()) {
                throw SdkClientException.builder().message(String.format("Cannot fetch credentials from container - neither %s or %s environment variables are set.", SdkSystemSetting.AWS_CONTAINER_CREDENTIALS_FULL_URI.environmentVariable(), SdkSystemSetting.AWS_CONTAINER_CREDENTIALS_RELATIVE_URI.environmentVariable())).build();
            }
            try {
                URI resolvedURI = SdkSystemSetting.AWS_CONTAINER_CREDENTIALS_RELATIVE_URI.getStringValue().map(this::createUri).orElseGet(() -> URI.create(SdkSystemSetting.AWS_CONTAINER_CREDENTIALS_FULL_URI.getStringValueOrThrow()));
                this.validateURI(resolvedURI);
                return resolvedURI;
            }
            catch (SdkClientException e) {
                throw e;
            }
            catch (Exception e) {
                throw SdkClientException.builder().message("Unable to fetch credentials from container.").cause((Throwable)e).build();
            }
        }

        public ResourcesEndpointRetryPolicy retryPolicy() {
            return new ContainerCredentialsRetryPolicy();
        }

        public Map<String, String> headers() {
            HashMap<String, String> requestHeaders = new HashMap<String, String>();
            requestHeaders.put("User-Agent", SdkUserAgent.create().userAgent());
            this.getTokenValue().filter(StringUtils::isNotBlank).ifPresent(t -> requestHeaders.put("Authorization", (String)t));
            return requestHeaders;
        }

        private Optional<String> getTokenValue() {
            if (SdkSystemSetting.AWS_CONTAINER_AUTHORIZATION_TOKEN.getStringValue().isPresent()) {
                return SdkSystemSetting.AWS_CONTAINER_AUTHORIZATION_TOKEN.getStringValue();
            }
            return SdkSystemSetting.AWS_CONTAINER_AUTHORIZATION_TOKEN_FILE.getStringValue().map(this::readToken);
        }

        private String readToken(String filePath) {
            Path path = Paths.get(filePath, new String[0]);
            try {
                return new String(Files.readAllBytes(path), StandardCharsets.UTF_8);
            }
            catch (IOException e) {
                throw SdkClientException.create((String)String.format("Failed to read %s.", path.toAbsolutePath()), (Throwable)e);
            }
        }

        private URI createUri(String relativeUri) {
            String host = this.endpoint != null ? this.endpoint : SdkSystemSetting.AWS_CONTAINER_SERVICE_ENDPOINT.getStringValueOrThrow();
            return URI.create(host + relativeUri);
        }

        private URI validateURI(URI uri) {
            if (!this.isHttps(uri) && !this.isAllowedHost(uri.getHost())) {
                String envVarName = SdkSystemSetting.AWS_CONTAINER_CREDENTIALS_FULL_URI.environmentVariable();
                throw SdkClientException.builder().message(String.format("The full URI (%s) contained within environment variable %s has an invalid host. Host should resolve to a loopback address or have the full URI be HTTPS.", uri, envVarName)).build();
            }
            return uri;
        }

        private boolean isHttps(URI endpoint) {
            return Objects.equals(ContainerCredentialsProvider.HTTPS, endpoint.getScheme());
        }

        private boolean isAllowedHost(String host) {
            try {
                InetAddress[] addresses = InetAddress.getAllByName(host);
                return addresses.length > 0 && (Arrays.stream(addresses).allMatch(this::matchesAllowedHostRules) || this.isMetadataServiceEndpoint(host));
            }
            catch (UnknownHostException e) {
                throw SdkClientException.builder().cause((Throwable)e).message(String.format("host (%s) could not be resolved to an IP address.", host)).build();
            }
        }

        private boolean matchesAllowedHostRules(InetAddress inetAddress) {
            return ALLOWED_HOSTS_RULES.test(inetAddress);
        }

        public boolean isMetadataServiceEndpoint(String host) {
            String mode = SdkSystemSetting.AWS_EC2_METADATA_SERVICE_ENDPOINT_MODE.getStringValueOrThrow();
            if ("IPV6".equalsIgnoreCase(mode)) {
                return VALID_LOOP_BACK_IPV6.contains(host);
            }
            return VALID_LOOP_BACK_IPV4.contains(host);
        }
    }
}

