/*
 * Decompiled with CFR 0.152.
 */
package org.apache.druid.common.config;

import com.google.common.base.Supplier;
import com.google.common.collect.ImmutableList;
import com.google.inject.Inject;
import java.util.Arrays;
import java.util.List;
import java.util.Map;
import java.util.concurrent.Callable;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.atomic.AtomicReference;
import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import org.apache.druid.common.config.ConfigManagerConfig;
import org.apache.druid.common.config.ConfigSerde;
import org.apache.druid.java.util.common.concurrent.ScheduledExecutors;
import org.apache.druid.java.util.common.lifecycle.LifecycleStart;
import org.apache.druid.java.util.common.lifecycle.LifecycleStop;
import org.apache.druid.java.util.common.logger.Logger;
import org.apache.druid.metadata.MetadataCASUpdate;
import org.apache.druid.metadata.MetadataStorageConnector;
import org.apache.druid.metadata.MetadataStorageTablesConfig;
import org.joda.time.Duration;

public class ConfigManager {
    private static final Logger log = new Logger(ConfigManager.class);
    private final Object lock = new Object();
    private boolean started = false;
    private final MetadataStorageConnector dbConnector;
    private final Supplier<ConfigManagerConfig> config;
    private final ScheduledExecutorService exec;
    private final ConcurrentMap<String, ConfigHolder> watchedConfigs;
    private final String configTable;
    private volatile PollingCallable poller;

    @Inject
    public ConfigManager(MetadataStorageConnector dbConnector, Supplier<MetadataStorageTablesConfig> dbTables, Supplier<ConfigManagerConfig> config) {
        this.dbConnector = dbConnector;
        this.config = config;
        this.exec = ScheduledExecutors.fixed(1, "config-manager-%s");
        this.watchedConfigs = new ConcurrentHashMap<String, ConfigHolder>();
        this.configTable = ((MetadataStorageTablesConfig)dbTables.get()).getConfigTable();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @LifecycleStart
    public void start() {
        Object object = this.lock;
        synchronized (object) {
            if (this.started) {
                return;
            }
            this.poller = new PollingCallable();
            ScheduledExecutors.scheduleWithFixedDelay(this.exec, new Duration(0L), ((ConfigManagerConfig)this.config.get()).getPollDuration().toStandardDuration(), this.poller);
            this.started = true;
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @LifecycleStop
    public void stop() {
        Object object = this.lock;
        synchronized (object) {
            if (!this.started) {
                return;
            }
            this.poller.stop();
            this.poller = null;
            this.started = false;
        }
    }

    private void poll() {
        for (Map.Entry entry : this.watchedConfigs.entrySet()) {
            try {
                if (!((ConfigHolder)entry.getValue()).swapIfNew(this.dbConnector.lookup(this.configTable, "name", "payload", (String)entry.getKey()))) continue;
                log.info("New value for key[%s] seen.", entry.getKey());
            }
            catch (Exception e) {
                log.warn(e, "Exception when checking property[%s]", entry.getKey());
            }
        }
    }

    public <T> AtomicReference<T> watchConfig(final String key, final ConfigSerde<T> serde) {
        ConfigHolder holder = (ConfigHolder)this.watchedConfigs.get(key);
        if (holder == null) {
            try {
                log.debug("Creating watch for key[%s]", key);
                holder = (ConfigHolder)this.exec.submit(new Callable<ConfigHolder<T>>(){

                    @Override
                    public ConfigHolder<T> call() {
                        if (!ConfigManager.this.started) {
                            ConfigManager.this.watchedConfigs.put(key, new ConfigHolder(null, serde));
                        } else {
                            try {
                                if (!ConfigManager.this.watchedConfigs.containsKey(key)) {
                                    byte[] value = ConfigManager.this.dbConnector.lookup(ConfigManager.this.configTable, "name", "payload", key);
                                    ConfigHolder holder = new ConfigHolder(value, serde);
                                    ConfigManager.this.watchedConfigs.put(key, holder);
                                }
                            }
                            catch (Exception e) {
                                log.warn(e, "Failed loading config for key[%s]", key);
                                ConfigManager.this.watchedConfigs.put(key, new ConfigHolder(null, serde));
                            }
                        }
                        return (ConfigHolder)ConfigManager.this.watchedConfigs.get(key);
                    }
                }).get();
            }
            catch (InterruptedException e) {
                Thread.currentThread().interrupt();
                throw new RuntimeException(e);
            }
            catch (ExecutionException e) {
                throw new RuntimeException(e);
            }
        }
        return holder.getReference();
    }

    public <T> SetResult set(String key, ConfigSerde<T> serde, T obj) {
        return this.set(key, serde, null, obj);
    }

    public <T> SetResult set(String key, ConfigSerde<T> serde, @Nullable byte[] oldValue, T newObject) {
        if (newObject == null || !this.started) {
            if (newObject == null) {
                return SetResult.failure(new IllegalAccessException("input obj is null"));
            }
            return SetResult.failure(new IllegalStateException("configManager is not started yet"));
        }
        byte[] newBytes = serde.serialize(newObject);
        try {
            return this.exec.submit(() -> {
                if (oldValue == null || !((ConfigManagerConfig)this.config.get()).isEnableCompareAndSwap()) {
                    this.dbConnector.insertOrUpdate(this.configTable, "name", "payload", key, newBytes);
                } else {
                    MetadataCASUpdate metadataCASUpdate = this.createMetadataCASUpdate(key, oldValue, newBytes);
                    boolean success = this.dbConnector.compareAndSwap((List<MetadataCASUpdate>)ImmutableList.of((Object)metadataCASUpdate));
                    if (!success) {
                        return SetResult.retryableFailure(new IllegalStateException("Config value has changed"));
                    }
                }
                ConfigHolder configHolder = (ConfigHolder)this.watchedConfigs.get(key);
                if (configHolder != null) {
                    configHolder.swapIfNew(newBytes);
                }
                return SetResult.ok();
            }).get();
        }
        catch (Exception e) {
            log.warn(e, "Failed to set[%s]", key);
            return SetResult.failure(e);
        }
    }

    @Nonnull
    private MetadataCASUpdate createMetadataCASUpdate(String keyValue, byte[] oldValue, byte[] newValue) {
        return new MetadataCASUpdate(this.configTable, "name", "payload", keyValue, oldValue, newValue);
    }

    private class PollingCallable
    implements Callable<ScheduledExecutors.Signal> {
        private volatile boolean stop = false;

        private PollingCallable() {
        }

        void stop() {
            this.stop = true;
        }

        @Override
        public ScheduledExecutors.Signal call() {
            if (this.stop) {
                return ScheduledExecutors.Signal.STOP;
            }
            ConfigManager.this.poll();
            return ScheduledExecutors.Signal.REPEAT;
        }
    }

    private static class ConfigHolder<T> {
        private final AtomicReference<byte[]> rawBytes;
        private final ConfigSerde<T> serde;
        private final AtomicReference<T> reference;

        ConfigHolder(byte[] rawBytes, ConfigSerde<T> serde) {
            this.rawBytes = new AtomicReference<byte[]>(rawBytes);
            this.serde = serde;
            this.reference = new AtomicReference<T>(serde.deserialize(rawBytes));
        }

        public AtomicReference<T> getReference() {
            return this.reference;
        }

        public boolean swapIfNew(byte[] newBytes) {
            if (!Arrays.equals(newBytes, this.rawBytes.get())) {
                this.reference.set(this.serde.deserialize(newBytes));
                this.rawBytes.set(newBytes);
                return true;
            }
            return false;
        }
    }

    public static class SetResult {
        private static final SetResult SUCCESS = new SetResult(null, false);
        private final Exception exception;
        private final boolean retryableException;

        public static SetResult ok() {
            return SUCCESS;
        }

        public static SetResult failure(Exception e) {
            return new SetResult(e, false);
        }

        public static SetResult retryableFailure(Exception e) {
            return new SetResult(e, true);
        }

        private SetResult(@Nullable Exception exception, boolean retryableException) {
            this.exception = exception;
            this.retryableException = retryableException;
        }

        public boolean isOk() {
            return this.exception == null;
        }

        public boolean isRetryable() {
            return this.retryableException;
        }

        public Exception getException() {
            return this.exception;
        }
    }
}

