/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hbase.io.crypto.tls;

import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.lang.invoke.MethodHandles;
import java.nio.charset.Charset;
import java.nio.charset.StandardCharsets;
import java.security.GeneralSecurityException;
import java.security.KeyPair;
import java.security.Security;
import java.security.cert.X509Certificate;
import java.util.Arrays;
import java.util.Objects;
import org.apache.commons.io.FileUtils;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.hbase.io.crypto.tls.KeyStoreFileType;
import org.apache.hadoop.hbase.io.crypto.tls.X509KeyType;
import org.apache.hadoop.hbase.io.crypto.tls.X509TestHelpers;
import org.apache.yetus.audience.InterfaceAudience;
import org.bouncycastle.asn1.x500.X500Name;
import org.bouncycastle.asn1.x500.X500NameBuilder;
import org.bouncycastle.asn1.x500.style.BCStyle;
import org.bouncycastle.asn1.x509.GeneralName;
import org.bouncycastle.asn1.x509.GeneralNames;
import org.bouncycastle.operator.OperatorCreationException;

@InterfaceAudience.Private
public final class X509TestContext {
    private static final String TRUST_STORE_PREFIX = "hbase_test_ca";
    private static final String KEY_STORE_PREFIX = "hbase_test_key";
    private final File tempDir;
    private final Configuration conf;
    private X509Certificate trustStoreCertificate;
    private final char[] trustStorePassword;
    private KeyPair trustStoreKeyPair;
    private File trustStoreJksFile;
    private File trustStorePemFile;
    private File trustStorePkcs12File;
    private File trustStoreBcfksFile;
    private KeyPair keyStoreKeyPair;
    private X509Certificate keyStoreCertificate;
    private final char[] keyStorePassword;
    private File keyStoreJksFile;
    private File keyStorePemFile;
    private File keyStorePkcs12File;
    private File keyStoreBcfksFile;

    private X509TestContext(Configuration conf, File tempDir, KeyPair trustStoreKeyPair, char[] trustStorePassword, KeyPair keyStoreKeyPair, char[] keyStorePassword) throws IOException, GeneralSecurityException, OperatorCreationException {
        if (Security.getProvider("BC") == null) {
            throw new IllegalStateException("BC Security provider was not found");
        }
        this.conf = conf;
        this.tempDir = Objects.requireNonNull(tempDir);
        if (!tempDir.isDirectory()) {
            throw new IllegalArgumentException("Not a directory: " + tempDir);
        }
        this.trustStoreKeyPair = trustStoreKeyPair;
        this.trustStorePassword = Objects.requireNonNull(trustStorePassword);
        this.keyStoreKeyPair = Objects.requireNonNull(keyStoreKeyPair);
        this.keyStorePassword = Objects.requireNonNull(keyStorePassword);
        this.createCertificates(new String[0]);
        this.trustStorePkcs12File = null;
        this.trustStorePemFile = null;
        this.trustStoreJksFile = null;
        this.keyStorePkcs12File = null;
        this.keyStorePemFile = null;
        this.keyStoreJksFile = null;
    }

    private X509TestContext(File tempDir, Configuration conf, X509Certificate trustStoreCertificate, char[] trustStorePassword, KeyPair trustStoreKeyPair, File trustStoreJksFile, File trustStorePemFile, File trustStorePkcs12File, KeyPair keyStoreKeyPair, char[] keyStorePassword, X509Certificate keyStoreCertificate) {
        this.tempDir = tempDir;
        this.conf = conf;
        this.trustStoreCertificate = trustStoreCertificate;
        this.trustStorePassword = trustStorePassword;
        this.trustStoreKeyPair = trustStoreKeyPair;
        this.trustStoreJksFile = trustStoreJksFile;
        this.trustStorePemFile = trustStorePemFile;
        this.trustStorePkcs12File = trustStorePkcs12File;
        this.keyStoreKeyPair = keyStoreKeyPair;
        this.keyStoreCertificate = keyStoreCertificate;
        this.keyStorePassword = keyStorePassword;
        this.keyStorePkcs12File = null;
        this.keyStorePemFile = null;
        this.keyStoreJksFile = null;
    }

    public X509Certificate newCert(X500Name name, String ... subjectAltNames) throws GeneralSecurityException, IOException, OperatorCreationException {
        if (subjectAltNames.length == 0) {
            return X509TestHelpers.newCert(this.trustStoreCertificate, this.trustStoreKeyPair, name, this.keyStoreKeyPair.getPublic());
        }
        GeneralName[] names = new GeneralName[subjectAltNames.length];
        for (int i = 0; i < subjectAltNames.length; ++i) {
            names[i] = new GeneralName(2, subjectAltNames[i]);
        }
        return X509TestHelpers.newCert(this.trustStoreCertificate, this.trustStoreKeyPair, name, this.keyStoreKeyPair.getPublic(), new GeneralNames(names));
    }

    public File getTempDir() {
        return this.tempDir;
    }

    public char[] getTrustStorePassword() {
        return this.trustStorePassword;
    }

    public File getTrustStoreFile(KeyStoreFileType storeFileType) throws IOException {
        switch (storeFileType) {
            case JKS: {
                return this.getTrustStoreJksFile();
            }
            case PEM: {
                return this.getTrustStorePemFile();
            }
            case PKCS12: {
                return this.getTrustStorePkcs12File();
            }
            case BCFKS: {
                return this.getTrustStoreBcfksFile();
            }
        }
        throw new IllegalArgumentException("Invalid trust store type: " + storeFileType + ", must be one of: " + Arrays.toString(KeyStoreFileType.values()));
    }

    private File getTrustStoreJksFile() throws IOException {
        if (this.trustStoreJksFile == null) {
            this.trustStoreJksFile = File.createTempFile(TRUST_STORE_PREFIX, KeyStoreFileType.JKS.getDefaultFileExtension(), this.tempDir);
            this.trustStoreJksFile.deleteOnExit();
            this.generateTrustStoreJksFile();
        }
        return this.trustStoreJksFile;
    }

    private void generateTrustStoreJksFile() throws IOException {
        try (FileOutputStream trustStoreOutputStream = new FileOutputStream(this.trustStoreJksFile);){
            byte[] bytes = X509TestHelpers.certToJavaTrustStoreBytes(this.trustStoreCertificate, this.trustStorePassword);
            trustStoreOutputStream.write(bytes);
            trustStoreOutputStream.flush();
        }
        catch (GeneralSecurityException e) {
            throw new IOException(e);
        }
    }

    private File getTrustStorePemFile() throws IOException {
        if (this.trustStorePemFile == null) {
            this.trustStorePemFile = File.createTempFile(TRUST_STORE_PREFIX, KeyStoreFileType.PEM.getDefaultFileExtension(), this.tempDir);
            this.trustStorePemFile.deleteOnExit();
            this.generateTrustStorePemFile();
        }
        return this.trustStorePemFile;
    }

    private void generateTrustStorePemFile() throws IOException {
        FileUtils.writeStringToFile((File)this.trustStorePemFile, (String)X509TestHelpers.pemEncodeX509Certificate(this.trustStoreCertificate), (Charset)StandardCharsets.US_ASCII, (boolean)false);
    }

    private File getTrustStorePkcs12File() throws IOException {
        if (this.trustStorePkcs12File == null) {
            this.trustStorePkcs12File = File.createTempFile(TRUST_STORE_PREFIX, KeyStoreFileType.PKCS12.getDefaultFileExtension(), this.tempDir);
            this.trustStorePkcs12File.deleteOnExit();
            this.generateTrustStorePkcs12File();
        }
        return this.trustStorePkcs12File;
    }

    private void generateTrustStorePkcs12File() throws IOException {
        try (FileOutputStream trustStoreOutputStream = new FileOutputStream(this.trustStorePkcs12File);){
            byte[] bytes = X509TestHelpers.certToPKCS12TrustStoreBytes(this.trustStoreCertificate, this.trustStorePassword);
            trustStoreOutputStream.write(bytes);
            trustStoreOutputStream.flush();
        }
        catch (GeneralSecurityException e) {
            throw new IOException(e);
        }
    }

    private File getTrustStoreBcfksFile() throws IOException {
        if (this.trustStoreBcfksFile == null) {
            this.trustStoreBcfksFile = File.createTempFile(TRUST_STORE_PREFIX, KeyStoreFileType.BCFKS.getDefaultFileExtension(), this.tempDir);
            this.trustStoreBcfksFile.deleteOnExit();
            this.generateTrustStoreBcfksFile();
        }
        return this.trustStoreBcfksFile;
    }

    private void generateTrustStoreBcfksFile() throws IOException {
        try (FileOutputStream trustStoreOutputStream = new FileOutputStream(this.trustStoreBcfksFile);){
            byte[] bytes = X509TestHelpers.certToBCFKSTrustStoreBytes(this.trustStoreCertificate, this.trustStorePassword);
            trustStoreOutputStream.write(bytes);
            trustStoreOutputStream.flush();
        }
        catch (GeneralSecurityException e) {
            throw new IOException(e);
        }
    }

    public X509Certificate getKeyStoreCertificate() {
        return this.keyStoreCertificate;
    }

    public char[] getKeyStorePassword() {
        return this.keyStorePassword;
    }

    public boolean isKeyStoreEncrypted() {
        return this.keyStorePassword != null;
    }

    public Configuration getConf() {
        return this.conf;
    }

    public File getKeyStoreFile(KeyStoreFileType storeFileType) throws IOException {
        switch (storeFileType) {
            case JKS: {
                return this.getKeyStoreJksFile();
            }
            case PEM: {
                return this.getKeyStorePemFile();
            }
            case PKCS12: {
                return this.getKeyStorePkcs12File();
            }
            case BCFKS: {
                return this.getKeyStoreBcfksFile();
            }
        }
        throw new IllegalArgumentException("Invalid key store type: " + storeFileType + ", must be one of: " + Arrays.toString(KeyStoreFileType.values()));
    }

    private File getKeyStoreJksFile() throws IOException {
        if (this.keyStoreJksFile == null) {
            this.keyStoreJksFile = File.createTempFile(KEY_STORE_PREFIX, KeyStoreFileType.JKS.getDefaultFileExtension(), this.tempDir);
            this.keyStoreJksFile.deleteOnExit();
            this.generateKeyStoreJksFile();
        }
        return this.keyStoreJksFile;
    }

    private void generateKeyStoreJksFile() throws IOException {
        try (FileOutputStream keyStoreOutputStream = new FileOutputStream(this.keyStoreJksFile);){
            byte[] bytes = X509TestHelpers.certAndPrivateKeyToJavaKeyStoreBytes(this.keyStoreCertificate, this.keyStoreKeyPair.getPrivate(), this.keyStorePassword);
            keyStoreOutputStream.write(bytes);
            keyStoreOutputStream.flush();
        }
        catch (GeneralSecurityException e) {
            throw new IOException(e);
        }
    }

    private File getKeyStorePemFile() throws IOException {
        if (this.keyStorePemFile == null) {
            try {
                this.keyStorePemFile = File.createTempFile(KEY_STORE_PREFIX, KeyStoreFileType.PEM.getDefaultFileExtension(), this.tempDir);
                this.keyStorePemFile.deleteOnExit();
                this.generateKeyStorePemFile();
            }
            catch (OperatorCreationException e) {
                throw new IOException(e);
            }
        }
        return this.keyStorePemFile;
    }

    private void generateKeyStorePemFile() throws IOException, OperatorCreationException {
        FileUtils.writeStringToFile((File)this.keyStorePemFile, (String)X509TestHelpers.pemEncodeCertAndPrivateKey(this.keyStoreCertificate, this.keyStoreKeyPair.getPrivate(), this.keyStorePassword), (Charset)StandardCharsets.US_ASCII, (boolean)false);
    }

    private File getKeyStorePkcs12File() throws IOException {
        if (this.keyStorePkcs12File == null) {
            this.keyStorePkcs12File = File.createTempFile(KEY_STORE_PREFIX, KeyStoreFileType.PKCS12.getDefaultFileExtension(), this.tempDir);
            this.keyStorePkcs12File.deleteOnExit();
            this.generateKeyStorePkcs12File();
        }
        return this.keyStorePkcs12File;
    }

    private void generateKeyStorePkcs12File() throws IOException {
        try (FileOutputStream keyStoreOutputStream = new FileOutputStream(this.keyStorePkcs12File);){
            byte[] bytes = X509TestHelpers.certAndPrivateKeyToPKCS12Bytes(this.keyStoreCertificate, this.keyStoreKeyPair.getPrivate(), this.keyStorePassword);
            keyStoreOutputStream.write(bytes);
            keyStoreOutputStream.flush();
        }
        catch (GeneralSecurityException e) {
            throw new IOException(e);
        }
    }

    private File getKeyStoreBcfksFile() throws IOException {
        if (this.keyStoreBcfksFile == null) {
            this.keyStoreBcfksFile = File.createTempFile(KEY_STORE_PREFIX, KeyStoreFileType.BCFKS.getDefaultFileExtension(), this.tempDir);
            this.keyStoreBcfksFile.deleteOnExit();
            this.generateKeyStoreBcfksFile();
        }
        return this.keyStoreBcfksFile;
    }

    private void generateKeyStoreBcfksFile() throws IOException {
        try (FileOutputStream keyStoreOutputStream = new FileOutputStream(this.keyStoreBcfksFile);){
            byte[] bytes = X509TestHelpers.certAndPrivateKeyToBCFKSBytes(this.keyStoreCertificate, this.keyStoreKeyPair.getPrivate(), this.keyStorePassword);
            keyStoreOutputStream.write(bytes);
            keyStoreOutputStream.flush();
        }
        catch (GeneralSecurityException e) {
            throw new IOException(e);
        }
    }

    public void setConfigurations(KeyStoreFileType keyStoreFileType, KeyStoreFileType trustStoreFileType) throws IOException {
        this.setKeystoreConfigurations(keyStoreFileType, this.conf);
        this.conf.set("hbase.rpc.tls.truststore.location", this.getTrustStoreFile(trustStoreFileType).getAbsolutePath());
        this.conf.set("hbase.rpc.tls.truststore.password", String.valueOf(this.getTrustStorePassword()));
        this.conf.set("hbase.rpc.tls.truststore.type", trustStoreFileType.getPropertyValue());
    }

    public void setKeystoreConfigurations(KeyStoreFileType keyStoreFileType, Configuration confToSet) throws IOException {
        confToSet.set("hbase.rpc.tls.keystore.location", this.getKeyStoreFile(keyStoreFileType).getAbsolutePath());
        confToSet.set("hbase.rpc.tls.keystore.password", String.valueOf(this.getKeyStorePassword()));
        confToSet.set("hbase.rpc.tls.keystore.type", keyStoreFileType.getPropertyValue());
    }

    public void clearConfigurations() {
        this.conf.unset("hbase.rpc.tls.keystore.location");
        this.conf.unset("hbase.rpc.tls.keystore.password");
        this.conf.unset("hbase.rpc.tls.keystore.type");
        this.conf.unset("hbase.rpc.tls.truststore.location");
        this.conf.unset("hbase.rpc.tls.truststore.password");
        this.conf.unset("hbase.rpc.tls.truststore.type");
    }

    public X509TestContext cloneWithNewKeystoreCert(X509Certificate cert) {
        return new X509TestContext(this.tempDir, this.conf, this.trustStoreCertificate, this.trustStorePassword, this.trustStoreKeyPair, this.trustStoreJksFile, this.trustStorePemFile, this.trustStorePkcs12File, this.keyStoreKeyPair, this.keyStorePassword, cert);
    }

    public void regenerateStores(X509KeyType keyStoreKeyType, X509KeyType trustStoreKeyType, KeyStoreFileType keyStoreFileType, KeyStoreFileType trustStoreFileType, String ... subjectAltNames) throws GeneralSecurityException, IOException, OperatorCreationException {
        this.trustStoreKeyPair = X509TestHelpers.generateKeyPair(trustStoreKeyType);
        this.keyStoreKeyPair = X509TestHelpers.generateKeyPair(keyStoreKeyType);
        this.createCertificates(subjectAltNames);
        switch (keyStoreFileType) {
            case JKS: {
                this.generateKeyStoreJksFile();
                break;
            }
            case PEM: {
                this.generateKeyStorePemFile();
                break;
            }
            case BCFKS: {
                this.generateKeyStoreBcfksFile();
                break;
            }
            case PKCS12: {
                this.generateKeyStorePkcs12File();
            }
        }
        switch (trustStoreFileType) {
            case JKS: {
                this.generateTrustStoreJksFile();
                break;
            }
            case PEM: {
                this.generateTrustStorePemFile();
                break;
            }
            case PKCS12: {
                this.generateTrustStorePkcs12File();
                break;
            }
            case BCFKS: {
                this.generateTrustStoreBcfksFile();
            }
        }
    }

    private void createCertificates(String ... subjectAltNames) throws GeneralSecurityException, IOException, OperatorCreationException {
        X500NameBuilder caNameBuilder = new X500NameBuilder(BCStyle.INSTANCE);
        caNameBuilder.addRDN(BCStyle.CN, MethodHandles.lookup().lookupClass().getCanonicalName() + " Root CA");
        this.trustStoreCertificate = X509TestHelpers.newSelfSignedCACert(caNameBuilder.build(), this.trustStoreKeyPair);
        X500NameBuilder nameBuilder = new X500NameBuilder(BCStyle.INSTANCE);
        nameBuilder.addRDN(BCStyle.CN, MethodHandles.lookup().lookupClass().getCanonicalName() + " Zookeeper Test");
        this.keyStoreCertificate = this.newCert(nameBuilder.build(), subjectAltNames);
    }

    public static Builder newBuilder(Configuration conf) {
        return new Builder(conf);
    }

    public static class Builder {
        private final Configuration conf;
        private File tempDir;
        private X509KeyType trustStoreKeyType;
        private char[] trustStorePassword;
        private X509KeyType keyStoreKeyType;
        private char[] keyStorePassword;

        public Builder(Configuration conf) {
            this.conf = conf;
            this.trustStoreKeyType = X509KeyType.EC;
            this.keyStoreKeyType = X509KeyType.EC;
        }

        public X509TestContext build() throws IOException, GeneralSecurityException, OperatorCreationException {
            KeyPair trustStoreKeyPair = X509TestHelpers.generateKeyPair(this.trustStoreKeyType);
            KeyPair keyStoreKeyPair = X509TestHelpers.generateKeyPair(this.keyStoreKeyType);
            return new X509TestContext(this.conf, this.tempDir, trustStoreKeyPair, this.trustStorePassword, keyStoreKeyPair, this.keyStorePassword);
        }

        public Builder setTempDir(File tempDir) {
            this.tempDir = tempDir;
            return this;
        }

        public Builder setTrustStoreKeyType(X509KeyType keyType) {
            this.trustStoreKeyType = keyType;
            return this;
        }

        public Builder setTrustStorePassword(char[] password) {
            this.trustStorePassword = password;
            return this;
        }

        public Builder setKeyStoreKeyType(X509KeyType keyType) {
            this.keyStoreKeyType = keyType;
            return this;
        }

        public Builder setKeyStorePassword(char[] password) {
            this.keyStorePassword = password;
            return this;
        }
    }
}

