/*
 * Decompiled with CFR 0.152.
 */
package org.apache.accumulo.server.security.delegation;

import java.io.ByteArrayInputStream;
import java.io.DataInputStream;
import java.io.IOException;
import org.apache.accumulo.fate.zookeeper.ZooReader;
import org.apache.accumulo.server.security.delegation.AuthenticationKey;
import org.apache.accumulo.server.security.delegation.AuthenticationTokenSecretManager;
import org.apache.zookeeper.KeeperException;
import org.apache.zookeeper.WatchedEvent;
import org.apache.zookeeper.Watcher;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class ZooAuthenticationKeyWatcher
implements Watcher {
    private static final Logger log = LoggerFactory.getLogger(ZooAuthenticationKeyWatcher.class);
    private final AuthenticationTokenSecretManager secretManager;
    private final ZooReader zk;
    private final String baseNode;

    public ZooAuthenticationKeyWatcher(AuthenticationTokenSecretManager secretManager, ZooReader zk, String baseNode) {
        this.secretManager = secretManager;
        this.zk = zk;
        this.baseNode = baseNode;
    }

    public void process(WatchedEvent event) {
        if (Watcher.Event.EventType.None == event.getType()) {
            switch (event.getState()) {
                case Disconnected: 
                case Expired: {
                    log.debug("ZooKeeper connection disconnected, clearing secret manager");
                    this.secretManager.removeAllKeys();
                    break;
                }
                case SyncConnected: {
                    log.debug("ZooKeeper reconnected, updating secret manager");
                    try {
                        this.updateAuthKeys();
                    }
                    catch (InterruptedException | KeeperException e) {
                        log.error("Failed to update secret manager after ZooKeeper reconnect");
                    }
                    break;
                }
                default: {
                    log.warn("Unhandled: " + event);
                }
            }
            return;
        }
        String path = event.getPath();
        if (null == path) {
            return;
        }
        if (!path.startsWith(this.baseNode)) {
            log.info("Ignoring event for path: {}", (Object)path);
            return;
        }
        try {
            if (path.equals(this.baseNode)) {
                this.processBaseNode(event);
            } else {
                this.processChildNode(event);
            }
        }
        catch (InterruptedException | KeeperException e) {
            log.error("Failed to communicate with ZooKeeper", e);
        }
    }

    void processBaseNode(WatchedEvent event) throws KeeperException, InterruptedException {
        switch (event.getType()) {
            case NodeDeleted: {
                log.debug("Parent ZNode was deleted, removing all AuthenticationKeys");
                this.secretManager.removeAllKeys();
                break;
            }
            case None: {
                break;
            }
            case NodeCreated: 
            case NodeChildrenChanged: {
                this.updateAuthKeys(event.getPath());
                break;
            }
            case NodeDataChanged: {
                break;
            }
            default: {
                log.warn("Unsupported event type: {}", (Object)event.getType());
            }
        }
    }

    public void updateAuthKeys() throws KeeperException, InterruptedException {
        if (this.zk.exists(this.baseNode, (Watcher)this)) {
            log.info("Added {} existing AuthenticationKeys to local cache from ZooKeeper", (Object)this.updateAuthKeys(this.baseNode));
        }
    }

    private int updateAuthKeys(String path) throws KeeperException, InterruptedException {
        int keysAdded = 0;
        for (String child : this.zk.getChildren(path, (Watcher)this)) {
            String childPath = path + "/" + child;
            try {
                AuthenticationKey key = this.deserializeKey(this.zk.getData(childPath, (Watcher)this, null));
                this.secretManager.addKey(key);
                ++keysAdded;
            }
            catch (KeeperException.NoNodeException e) {
                log.trace("{} was deleted when we tried to access it", (Object)childPath);
            }
        }
        return keysAdded;
    }

    void processChildNode(WatchedEvent event) throws KeeperException, InterruptedException {
        String path = event.getPath();
        switch (event.getType()) {
            case NodeDeleted: {
                if (null == path) {
                    log.error("Got null path for NodeDeleted event");
                    return;
                }
                String childName = path.substring(this.baseNode.length() + 1);
                this.secretManager.removeKey(Integer.parseInt(childName));
                break;
            }
            case None: {
                break;
            }
            case NodeCreated: {
                if (null == path) {
                    log.error("Got null path for NodeCreated event");
                    return;
                }
                AuthenticationKey key = this.deserializeKey(this.zk.getData(path, (Watcher)this, null));
                log.debug("Adding AuthenticationKey with keyId {}", (Object)key.getKeyId());
                this.secretManager.addKey(key);
                break;
            }
            case NodeDataChanged: {
                if (null == path) {
                    log.error("Got null path for NodeDataChanged event");
                    return;
                }
                AuthenticationKey newKey = this.deserializeKey(this.zk.getData(path, (Watcher)this, null));
                this.secretManager.addKey(newKey);
                break;
            }
            case NodeChildrenChanged: {
                log.warn("Unexpected NodeChildrenChanged event for authentication key node {}", (Object)path);
                break;
            }
            default: {
                log.warn("Unsupported event type: {}", (Object)event.getType());
            }
        }
    }

    AuthenticationKey deserializeKey(byte[] serializedKey) {
        AuthenticationKey key = new AuthenticationKey();
        try {
            key.readFields(new DataInputStream(new ByteArrayInputStream(serializedKey)));
        }
        catch (IOException e) {
            throw new AssertionError((Object)"Failed to read from an in-memory buffer");
        }
        return key;
    }
}

