/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hbase.util;

import java.io.IOException;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.hbase.CompoundConfiguration;
import org.apache.hadoop.hbase.DoNotRetryIOException;
import org.apache.hadoop.hbase.TableName;
import org.apache.hadoop.hbase.client.ColumnFamilyDescriptor;
import org.apache.hadoop.hbase.client.TableDescriptor;
import org.apache.hadoop.hbase.regionserver.RegionCoprocessorHost;
import org.apache.hadoop.hbase.regionserver.RegionSplitPolicy;
import org.apache.hadoop.hbase.regionserver.compactions.ExploringCompactionPolicy;
import org.apache.hadoop.hbase.regionserver.compactions.FIFOCompactionPolicy;
import org.apache.hadoop.hbase.shaded.protobuf.generated.WALProtos;
import org.apache.hadoop.hbase.util.BloomFilterUtil;
import org.apache.hadoop.hbase.util.CompressionTest;
import org.apache.hadoop.hbase.util.EncryptionTest;
import org.apache.yetus.audience.InterfaceAudience;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@InterfaceAudience.Private
public final class TableDescriptorChecker {
    private static Logger LOG = LoggerFactory.getLogger(TableDescriptorChecker.class);
    public static final String TABLE_SANITY_CHECKS = "hbase.table.sanity.checks";
    public static final boolean DEFAULT_TABLE_SANITY_CHECKS = true;
    public static final String MASTER_CHECK_COMPRESSION = "hbase.master.check.compression";
    public static final boolean DEFAULT_MASTER_CHECK_COMPRESSION = true;
    public static final String MASTER_CHECK_ENCRYPTION = "hbase.master.check.encryption";
    public static final boolean DEFAULT_MASTER_CHECK_ENCRYPTION = true;

    private TableDescriptorChecker() {
    }

    public static void sanityCheck(Configuration c, TableDescriptor td) throws IOException {
        int regionReplicas;
        String message;
        long flushSize;
        long maxFileSize;
        String tableVal;
        CompoundConfiguration conf = new CompoundConfiguration().add(c).addBytesMap(td.getValues());
        boolean logWarn = false;
        if (!conf.getBoolean(TABLE_SANITY_CHECKS, true)) {
            logWarn = true;
        }
        if ((tableVal = td.getValue(TABLE_SANITY_CHECKS)) != null && !Boolean.valueOf(tableVal).booleanValue()) {
            logWarn = true;
        }
        long maxFileSizeLowerLimit = 0x200000L;
        long l = maxFileSize = td.getValue("MAX_FILESIZE") == null ? conf.getLong("hbase.hregion.max.filesize", maxFileSizeLowerLimit) : Long.parseLong(td.getValue("MAX_FILESIZE"));
        if (maxFileSize < conf.getLong("hbase.hregion.max.filesize.limit", maxFileSizeLowerLimit)) {
            String message2 = "MAX_FILESIZE for table descriptor or \"hbase.hregion.max.filesize\" (" + maxFileSize + ") is too small, which might cause over splitting into unmanageable number of regions.";
            TableDescriptorChecker.warnOrThrowExceptionForFailure(logWarn, message2, null);
        }
        long flushSizeLowerLimit = 0x100000L;
        long l2 = flushSize = td.getValue("MEMSTORE_FLUSHSIZE") == null ? conf.getLong("hbase.hregion.memstore.flush.size", flushSizeLowerLimit) : Long.parseLong(td.getValue("MEMSTORE_FLUSHSIZE"));
        if (flushSize < conf.getLong("hbase.hregion.memstore.flush.size.limit", flushSizeLowerLimit)) {
            message = "MEMSTORE_FLUSHSIZE for table descriptor or \"hbase.hregion.memstore.flush.size\" (" + flushSize + ") is too small, which might cause very frequent flushing.";
            TableDescriptorChecker.warnOrThrowExceptionForFailure(logWarn, message, null);
        }
        try {
            TableDescriptorChecker.checkClassLoading(conf, td);
        }
        catch (Exception ex) {
            TableDescriptorChecker.warnOrThrowExceptionForFailure(logWarn, ex.getMessage(), null);
        }
        if (conf.getBoolean(MASTER_CHECK_COMPRESSION, true)) {
            try {
                TableDescriptorChecker.checkCompression(td);
            }
            catch (IOException e) {
                TableDescriptorChecker.warnOrThrowExceptionForFailure(logWarn, e.getMessage(), e);
            }
        }
        if (conf.getBoolean(MASTER_CHECK_ENCRYPTION, true)) {
            try {
                TableDescriptorChecker.checkEncryption(conf, td);
            }
            catch (IOException e) {
                TableDescriptorChecker.warnOrThrowExceptionForFailure(logWarn, e.getMessage(), e);
            }
        }
        try {
            TableDescriptorChecker.checkCompactionPolicy(conf, td);
        }
        catch (IOException e) {
            TableDescriptorChecker.warnOrThrowExceptionForFailure(false, e.getMessage(), e);
        }
        if (td.getColumnFamilyCount() == 0) {
            message = "Table should have at least one column family.";
            TableDescriptorChecker.warnOrThrowExceptionForFailure(logWarn, message, null);
        }
        if ((regionReplicas = td.getRegionReplication()) < 1) {
            String message3 = "Table region replication should be at least one.";
            TableDescriptorChecker.warnOrThrowExceptionForFailure(logWarn, message3, null);
        }
        if (td.isReadOnly() && TableName.isMetaTableName(td.getTableName())) {
            TableDescriptorChecker.warnOrThrowExceptionForFailure(false, "Meta table can't be set as read only.", null);
        }
        for (ColumnFamilyDescriptor hcd : td.getColumnFamilies()) {
            String message4;
            if (hcd.getTimeToLive() <= 0) {
                message4 = "TTL for column family " + hcd.getNameAsString() + " must be positive.";
                TableDescriptorChecker.warnOrThrowExceptionForFailure(logWarn, message4, null);
            }
            if (hcd.getBlocksize() < 1024 || hcd.getBlocksize() > 0x1000000) {
                message4 = "Block size for column family " + hcd.getNameAsString() + "  must be between 1K and 16MB.";
                TableDescriptorChecker.warnOrThrowExceptionForFailure(logWarn, message4, null);
            }
            if (hcd.getMinVersions() < 0) {
                message4 = "Min versions for column family " + hcd.getNameAsString() + "  must be positive.";
                TableDescriptorChecker.warnOrThrowExceptionForFailure(logWarn, message4, null);
            }
            if (hcd.getMinVersions() > hcd.getMaxVersions()) {
                message4 = "Min versions for column family " + hcd.getNameAsString() + " must be less than the Max versions.";
                TableDescriptorChecker.warnOrThrowExceptionForFailure(logWarn, message4, null);
            }
            TableDescriptorChecker.checkReplicationScope(hcd);
            TableDescriptorChecker.checkBloomFilterType(hcd);
            if (hcd.getDFSReplication() < 0) {
                message4 = "HFile Replication for column family " + hcd.getNameAsString() + "  must be greater than zero.";
                TableDescriptorChecker.warnOrThrowExceptionForFailure(logWarn, message4, null);
            }
            try {
                hcd.getInMemoryCompaction();
            }
            catch (IllegalArgumentException e) {
                TableDescriptorChecker.warnOrThrowExceptionForFailure(logWarn, e.getMessage(), e);
            }
        }
    }

    private static void checkReplicationScope(ColumnFamilyDescriptor cfd) throws IOException {
        WALProtos.ScopeType scop = WALProtos.ScopeType.valueOf(cfd.getScope());
        if (scop == null) {
            String message = "Replication scope for column family " + cfd.getNameAsString() + " is " + cfd.getScope() + " which is invalid.";
            LOG.error(message);
            throw new DoNotRetryIOException(message);
        }
    }

    private static void checkCompactionPolicy(Configuration conf, TableDescriptor td) throws IOException {
        String className = td.getValue("hbase.hstore.defaultengine.compactionpolicy.class");
        if (className == null) {
            className = conf.get("hbase.hstore.defaultengine.compactionpolicy.class", ExploringCompactionPolicy.class.getName());
        }
        int blockingFileCount = 16;
        String sv = td.getValue("hbase.hstore.blockingStoreFiles");
        blockingFileCount = sv != null ? Integer.parseInt(sv) : conf.getInt("hbase.hstore.blockingStoreFiles", blockingFileCount);
        for (ColumnFamilyDescriptor hcd : td.getColumnFamilies()) {
            String compactionPolicy = hcd.getConfigurationValue("hbase.hstore.defaultengine.compactionpolicy.class");
            if (compactionPolicy == null) {
                compactionPolicy = className;
            }
            if (!compactionPolicy.equals(FIFOCompactionPolicy.class.getName())) continue;
            String message = null;
            if (hcd.getTimeToLive() == Integer.MAX_VALUE) {
                message = "Default TTL is not supported for FIFO compaction";
                throw new IOException(message);
            }
            if (hcd.getMinVersions() > 0) {
                message = "MIN_VERSION > 0 is not supported for FIFO compaction";
                throw new IOException(message);
            }
            sv = hcd.getConfigurationValue("hbase.hstore.blockingStoreFiles");
            if (sv != null) {
                blockingFileCount = Integer.parseInt(sv);
            }
            if (blockingFileCount >= 1000) continue;
            message = "Blocking file count 'hbase.hstore.blockingStoreFiles' " + blockingFileCount + " is below recommended minimum of 1000 for column family " + hcd.getNameAsString();
            throw new IOException(message);
        }
    }

    private static void checkBloomFilterType(ColumnFamilyDescriptor cfd) throws IOException {
        CompoundConfiguration conf = new CompoundConfiguration().addStringMap(cfd.getConfiguration());
        try {
            BloomFilterUtil.getBloomFilterParam(cfd.getBloomFilterType(), conf);
        }
        catch (IllegalArgumentException e) {
            throw new DoNotRetryIOException("Failed to get bloom filter param", e);
        }
    }

    public static void checkCompression(TableDescriptor td) throws IOException {
        for (ColumnFamilyDescriptor cfd : td.getColumnFamilies()) {
            CompressionTest.testCompression(cfd.getCompressionType());
            CompressionTest.testCompression(cfd.getCompactionCompressionType());
        }
    }

    public static void checkEncryption(Configuration conf, TableDescriptor td) throws IOException {
        for (ColumnFamilyDescriptor cfd : td.getColumnFamilies()) {
            EncryptionTest.testEncryption(conf, cfd.getEncryptionType(), cfd.getEncryptionKey());
        }
    }

    public static void checkClassLoading(Configuration conf, TableDescriptor td) throws IOException {
        RegionSplitPolicy.getSplitPolicyClass(td, conf);
        RegionCoprocessorHost.testTableCoprocessorAttrs(conf, td);
    }

    private static void warnOrThrowExceptionForFailure(boolean logWarn, String message, Exception cause) throws IOException {
        if (!logWarn) {
            throw new DoNotRetryIOException(message + " Set " + TABLE_SANITY_CHECKS + " to false at conf or table descriptor if you want to bypass sanity checks", cause);
        }
        LOG.warn(message);
    }
}

