(function (global, factory) {
    typeof exports === 'object' && typeof module !== 'undefined' ? factory(exports, require('vue')) :
    typeof define === 'function' && define.amd ? define(['exports', 'vue'], factory) :
    (global = typeof globalThis !== 'undefined' ? globalThis : global || self, factory(global.naive = {}, global.Vue));
})(this, (function (exports, vue) { 'use strict';

    let onceCbs = [];
    const paramsMap = new WeakMap();
    function flushOnceCallbacks() {
      onceCbs.forEach(cb => cb(...paramsMap.get(cb)));
      onceCbs = [];
    }
    function beforeNextFrameOnce(cb, ...params) {
      paramsMap.set(cb, params);
      if (onceCbs.includes(cb)) return;
      onceCbs.push(cb) === 1 && requestAnimationFrame(flushOnceCallbacks);
    }

    function getParentNode$1(node) {
      // document type
      if (node.nodeType === 9) {
        return null;
      }
      return node.parentNode;
    }
    function getScrollParent$1(node) {
      if (node === null) return null;
      const parentNode = getParentNode$1(node);
      if (parentNode === null) {
        return null;
      }
      // Document
      if (parentNode.nodeType === 9) {
        return document.documentElement;
      }
      // Element
      if (parentNode.nodeType === 1) {
        // Firefox want us to check `-x` and `-y` variations as well
        const {
          overflow,
          overflowX,
          overflowY
        } = getComputedStyle(parentNode);
        if (/(auto|scroll|overlay)/.test(overflow + overflowY + overflowX)) {
          return parentNode;
        }
      }
      return getScrollParent$1(parentNode);
    }

    function unwrapElement(target) {
      if (typeof target === 'string') return document.querySelector(target);
      if (typeof target === 'function') return target();
      return target;
    }

    function happensIn(e, dataSetPropName) {
      let {
        target
      } = e;
      while (target) {
        if (target.dataset) {
          if (target.dataset[dataSetPropName] !== undefined) return true;
        }
        target = target.parentElement;
      }
      return false;
    }

    function getPreciseEventTarget(event) {
      return event.composedPath()[0] || null;
    }

    function parseResponsiveProp(reponsiveProp) {
      if (typeof reponsiveProp === "number") {
        return {
          '': reponsiveProp.toString()
        };
      }
      const params = {};
      reponsiveProp.split(/ +/).forEach(pairLiteral => {
        if (pairLiteral === '') return;
        const [prefix, value] = pairLiteral.split(':');
        if (value === undefined) {
          params[''] = prefix;
        } else {
          params[prefix] = value;
        }
      });
      return params;
    }
    function parseResponsivePropValue(reponsiveProp, activeKeyOrSize) {
      var _a;
      if (reponsiveProp === undefined || reponsiveProp === null) return undefined;
      const classObj = parseResponsiveProp(reponsiveProp);
      if (activeKeyOrSize === undefined) return classObj[''];
      if (typeof activeKeyOrSize === 'string') {
        return (_a = classObj[activeKeyOrSize]) !== null && _a !== void 0 ? _a : classObj[''];
      } else if (Array.isArray(activeKeyOrSize)) {
        for (let i = activeKeyOrSize.length - 1; i >= 0; --i) {
          const key = activeKeyOrSize[i];
          if (key in classObj) return classObj[key];
        }
        return classObj[''];
      } else {
        // Here we suppose all the keys are number formatted
        let activeValue = undefined;
        let activeKey = -1;
        Object.keys(classObj).forEach(key => {
          const keyAsNum = Number(key);
          if (!Number.isNaN(keyAsNum) && activeKeyOrSize >= keyAsNum && keyAsNum >= activeKey) {
            activeKey = keyAsNum;
            activeValue = classObj[key];
          }
        });
        return activeValue;
      }
    }

    function depx(value) {
      if (typeof value === 'string') {
        if (value.endsWith('px')) {
          return Number(value.slice(0, value.length - 2));
        }
        return Number(value);
      }
      return value;
    }
    function pxfy(value) {
      if (value === undefined || value === null) return undefined;
      if (typeof value === 'number') return `${value}px`;
      if (value.endsWith('px')) return value;
      return `${value}px`;
    }
    function getMargin(value, position) {
      const parts = value.trim().split(/\s+/g);
      const margin = {
        top: parts[0]
      };
      switch (parts.length) {
        case 1:
          margin.right = parts[0];
          margin.bottom = parts[0];
          margin.left = parts[0];
          break;
        case 2:
          margin.right = parts[1];
          margin.left = parts[1];
          margin.bottom = parts[0];
          break;
        case 3:
          margin.right = parts[1];
          margin.bottom = parts[2];
          margin.left = parts[1];
          break;
        case 4:
          margin.right = parts[1];
          margin.bottom = parts[2];
          margin.left = parts[3];
          break;
        default:
          throw new Error('[seemly/getMargin]:' + value + ' is not a valid value.');
      }
      if (position === undefined) return margin;
      return margin[position];
    }
    function getGap(value, orient) {
      const [rowGap, colGap] = value.split(' ');
      return {
        row: rowGap,
        col: colGap || rowGap
      };
    }

    var colors = {
      black: '#000',
      silver: '#C0C0C0',
      gray: '#808080',
      white: '#FFF',
      maroon: '#800000',
      red: '#F00',
      purple: '#800080',
      fuchsia: '#F0F',
      green: '#008000',
      lime: '#0F0',
      olive: '#808000',
      yellow: '#FF0',
      navy: '#000080',
      blue: '#00F',
      teal: '#008080',
      aqua: '#0FF',
      transparent: '#0000'
    };

    // All the algorithms credit to https://stackoverflow.com/questions/36721830/convert-hsl-to-rgb-and-hex/54014428#54014428
    // original author: Kamil Kiełczewski
    /**
     * @param h 360
     * @param s 100
     * @param l 100
     * @returns [h, s, v] 360, 100, 100
     */
    function hsl2hsv(h, s, l) {
      s /= 100;
      l /= 100;
      const v = s * Math.min(l, 1 - l) + l;
      return [h, v ? (2 - 2 * l / v) * 100 : 0, v * 100];
    }
    /**
     * @param h 360
     * @param s 100
     * @param v 100
     * @returns [h, s, l] 360, 100, 100
     */
    function hsv2hsl(h, s, v) {
      s /= 100;
      v /= 100;
      const l = v - v * s / 2;
      const m = Math.min(l, 1 - l);
      return [h, m ? (v - l) / m * 100 : 0, l * 100];
    }
    /**
     * @param h 360
     * @param s 100
     * @param v 100
     * @returns [r, g, b] 255, 255, 255
     */
    function hsv2rgb(h, s, v) {
      s /= 100;
      v /= 100;
      let f = (n, k = (n + h / 60) % 6) => v - v * s * Math.max(Math.min(k, 4 - k, 1), 0);
      return [f(5) * 255, f(3) * 255, f(1) * 255];
    }
    /**
     * @param r 255
     * @param g 255
     * @param b 255
     * @returns [360, 100, 100]
     */
    function rgb2hsv(r, g, b) {
      r /= 255;
      g /= 255;
      b /= 255;
      let v = Math.max(r, g, b),
        c = v - Math.min(r, g, b);
      let h = c && (v == r ? (g - b) / c : v == g ? 2 + (b - r) / c : 4 + (r - g) / c);
      return [60 * (h < 0 ? h + 6 : h), v && c / v * 100, v * 100];
    }
    /**
     * @param r 255
     * @param g 255
     * @param b 255
     * @returns [360, 100, 100]
     */
    function rgb2hsl(r, g, b) {
      r /= 255;
      g /= 255;
      b /= 255;
      let v = Math.max(r, g, b),
        c = v - Math.min(r, g, b),
        f = 1 - Math.abs(v + v - c - 1);
      let h = c && (v == r ? (g - b) / c : v == g ? 2 + (b - r) / c : 4 + (r - g) / c);
      return [60 * (h < 0 ? h + 6 : h), f ? c / f * 100 : 0, (v + v - c) * 50];
    }
    /**
     * @param h 360
     * @param s 100
     * @param l 100
     * @returns [255, 255, 255]
     */
    function hsl2rgb(h, s, l) {
      s /= 100;
      l /= 100;
      let a = s * Math.min(l, 1 - l);
      let f = (n, k = (n + h / 30) % 12) => l - a * Math.max(Math.min(k - 3, 9 - k, 1), -1);
      return [f(0) * 255, f(8) * 255, f(4) * 255];
    }

    const prefix$1 = '^\\s*';
    const suffix = '\\s*$';
    const percent = '\\s*((\\.\\d+)|(\\d+(\\.\\d*)?))%\\s*'; // 4 offset
    const float = '\\s*((\\.\\d+)|(\\d+(\\.\\d*)?))\\s*'; // 4 offset
    const hex = '([0-9A-Fa-f])';
    const dhex = '([0-9A-Fa-f]{2})';
    const hslRegex = new RegExp(`${prefix$1}hsl\\s*\\(${float},${percent},${percent}\\)${suffix}`);
    const hsvRegex = new RegExp(`${prefix$1}hsv\\s*\\(${float},${percent},${percent}\\)${suffix}`);
    const hslaRegex = new RegExp(`${prefix$1}hsla\\s*\\(${float},${percent},${percent},${float}\\)${suffix}`);
    const hsvaRegex = new RegExp(`${prefix$1}hsva\\s*\\(${float},${percent},${percent},${float}\\)${suffix}`);
    const rgbRegex = new RegExp(`${prefix$1}rgb\\s*\\(${float},${float},${float}\\)${suffix}`);
    const rgbaRegex = new RegExp(`${prefix$1}rgba\\s*\\(${float},${float},${float},${float}\\)${suffix}`);
    const sHexRegex = new RegExp(`${prefix$1}#${hex}${hex}${hex}${suffix}`);
    const hexRegex = new RegExp(`${prefix$1}#${dhex}${dhex}${dhex}${suffix}`);
    const sHexaRegex = new RegExp(`${prefix$1}#${hex}${hex}${hex}${hex}${suffix}`);
    const hexaRegex = new RegExp(`${prefix$1}#${dhex}${dhex}${dhex}${dhex}${suffix}`);
    function parseHex(value) {
      return parseInt(value, 16);
    }
    /**
     * Convert color string to hsla array
     * @param color format like hsl(180, 100%, 100%), hsla(180, 100%, 100%, 1)
     * @returns
     */
    function hsla(color) {
      try {
        let i;
        if (i = hslaRegex.exec(color)) {
          return [roundDeg(i[1]), roundPercent(i[5]), roundPercent(i[9]), roundAlpha(i[13])];
        } else if (i = hslRegex.exec(color)) {
          return [roundDeg(i[1]), roundPercent(i[5]), roundPercent(i[9]), 1];
        }
        throw new Error(`[seemly/hsla]: Invalid color value ${color}.`);
      } catch (e) {
        throw e;
      }
    }
    /**
     * Convert color string to hsva array
     * @param color format like hsv(180, 100%, 100%), hsva(180, 100%, 100%, 1)
     * @returns
     */
    function hsva(color) {
      try {
        let i;
        if (i = hsvaRegex.exec(color)) {
          return [roundDeg(i[1]), roundPercent(i[5]), roundPercent(i[9]), roundAlpha(i[13])];
        } else if (i = hsvRegex.exec(color)) {
          return [roundDeg(i[1]), roundPercent(i[5]), roundPercent(i[9]), 1];
        }
        throw new Error(`[seemly/hsva]: Invalid color value ${color}.`);
      } catch (e) {
        throw e;
      }
    }
    /**
     * Convert color string to rgba array.
     * @param color format like #000[0], #000000[00], rgb(0, 0, 0), rgba(0, 0, 0, 0) and basic color keywords https://www.w3.org/TR/css-color-3/#html4 and transparent
     * @returns
     */
    function rgba(color) {
      try {
        let i;
        if (i = hexRegex.exec(color)) {
          return [parseHex(i[1]), parseHex(i[2]), parseHex(i[3]), 1];
        } else if (i = rgbRegex.exec(color)) {
          return [roundChannel(i[1]), roundChannel(i[5]), roundChannel(i[9]), 1];
        } else if (i = rgbaRegex.exec(color)) {
          return [roundChannel(i[1]), roundChannel(i[5]), roundChannel(i[9]), roundAlpha(i[13])];
        } else if (i = sHexRegex.exec(color)) {
          return [parseHex(i[1] + i[1]), parseHex(i[2] + i[2]), parseHex(i[3] + i[3]), 1];
        } else if (i = hexaRegex.exec(color)) {
          return [parseHex(i[1]), parseHex(i[2]), parseHex(i[3]), roundAlpha(parseHex(i[4]) / 255)];
        } else if (i = sHexaRegex.exec(color)) {
          return [parseHex(i[1] + i[1]), parseHex(i[2] + i[2]), parseHex(i[3] + i[3]), roundAlpha(parseHex(i[4] + i[4]) / 255)];
        } else if (color in colors) {
          return rgba(colors[color]);
        }
        throw new Error(`[seemly/rgba]: Invalid color value ${color}.`);
      } catch (e) {
        throw e;
      }
    }
    function normalizeAlpha$1(alphaValue) {
      return alphaValue > 1 ? 1 : alphaValue < 0 ? 0 : alphaValue;
    }
    function stringifyRgb(r, g, b) {
      return `rgb(${roundChannel(r)}, ${roundChannel(g)}, ${roundChannel(b)})`;
    }
    function stringifyRgba(r, g, b, a) {
      return `rgba(${roundChannel(r)}, ${roundChannel(g)}, ${roundChannel(b)}, ${normalizeAlpha$1(a)})`;
    }
    function compositeChannel(v1, a1, v2, a2, a) {
      return roundChannel((v1 * a1 * (1 - a2) + v2 * a2) / a);
    }
    function composite(background, overlay) {
      if (!Array.isArray(background)) background = rgba(background);
      if (!Array.isArray(overlay)) overlay = rgba(overlay);
      const a1 = background[3];
      const a2 = overlay[3];
      const alpha = roundAlpha(a1 + a2 - a1 * a2);
      return stringifyRgba(compositeChannel(background[0], a1, overlay[0], a2, alpha), compositeChannel(background[1], a1, overlay[1], a2, alpha), compositeChannel(background[2], a1, overlay[2], a2, alpha), alpha);
    }
    function changeColor(base, options) {
      const [r, g, b, a = 1] = Array.isArray(base) ? base : rgba(base);
      if (options.alpha) {
        return stringifyRgba(r, g, b, options.alpha);
      }
      return stringifyRgba(r, g, b, a);
    }
    function scaleColor(base, options) {
      const [r, g, b, a = 1] = Array.isArray(base) ? base : rgba(base);
      const {
        lightness = 1,
        alpha = 1
      } = options;
      return toRgbaString([r * lightness, g * lightness, b * lightness, a * alpha]);
    }
    function roundAlpha(value) {
      const v = Math.round(Number(value) * 100) / 100;
      if (v > 1) return 1;
      if (v < 0) return 0;
      return v;
    }
    function roundDeg(value) {
      const v = Math.round(Number(value));
      if (v >= 360) return 0;
      if (v < 0) return 0;
      return v;
    }
    function roundChannel(value) {
      const v = Math.round(Number(value));
      if (v > 255) return 255;
      if (v < 0) return 0;
      return v;
    }
    function roundPercent(value) {
      const v = Math.round(Number(value));
      if (v > 100) return 100;
      if (v < 0) return 0;
      return v;
    }
    function toRgbString(base) {
      const [r, g, b] = Array.isArray(base) ? base : rgba(base);
      return stringifyRgb(r, g, b);
    }
    function toRgbaString(base) {
      const [r, g, b] = base;
      if (3 in base) {
        return `rgba(${roundChannel(r)}, ${roundChannel(g)}, ${roundChannel(b)}, ${roundAlpha(base[3])})`;
      }
      return `rgba(${roundChannel(r)}, ${roundChannel(g)}, ${roundChannel(b)}, 1)`;
    }
    function toHsvString(base) {
      return `hsv(${roundDeg(base[0])}, ${roundPercent(base[1])}%, ${roundPercent(base[2])}%)`;
    }
    function toHsvaString(base) {
      const [h, s, v] = base;
      if (3 in base) {
        return `hsva(${roundDeg(h)}, ${roundPercent(s)}%, ${roundPercent(v)}%, ${roundAlpha(base[3])})`;
      }
      return `hsva(${roundDeg(h)}, ${roundPercent(s)}%, ${roundPercent(v)}%, 1)`;
    }
    function toHslString(base) {
      return `hsl(${roundDeg(base[0])}, ${roundPercent(base[1])}%, ${roundPercent(base[2])}%)`;
    }
    function toHslaString(base) {
      const [h, s, l] = base;
      if (3 in base) {
        return `hsla(${roundDeg(h)}, ${roundPercent(s)}%, ${roundPercent(l)}%, ${roundAlpha(base[3])})`;
      }
      return `hsla(${roundDeg(h)}, ${roundPercent(s)}%, ${roundPercent(l)}%, 1)`;
    }
    /**
     *
     * @param base [255, 255, 255, 255], [255, 255, 255], any hex string
     * @returns
     */
    function toHexaString(base) {
      if (typeof base === 'string') {
        let i;
        if (i = hexRegex.exec(base)) {
          return `${i[0]}FF`;
        } else if (i = hexaRegex.exec(base)) {
          return i[0];
        } else if (i = sHexRegex.exec(base)) {
          return `#${i[1]}${i[1]}${i[2]}${i[2]}${i[3]}${i[3]}FF`;
        } else if (i = sHexaRegex.exec(base)) {
          return `#${i[1]}${i[1]}${i[2]}${i[2]}${i[3]}${i[3]}${i[4]}${i[4]}`;
        }
        throw new Error(`[seemly/toHexString]: Invalid hex value ${base}.`);
      }
      const hex = `#${base.slice(0, 3).map(unit => roundChannel(unit).toString(16).toUpperCase().padStart(2, '0')).join('')}`;
      const a = base.length === 3 ? 'FF' : roundChannel(base[3] * 255).toString(16).padStart(2, '0').toUpperCase();
      return hex + a;
    }
    /**
     *
     * @param base [255, 255, 255, 255], [255, 255, 255], any hex string
     * @returns
     */
    function toHexString(base) {
      if (typeof base === 'string') {
        let i;
        if (i = hexRegex.exec(base)) {
          return i[0];
        } else if (i = hexaRegex.exec(base)) {
          return i[0].slice(0, 7);
        } else if (i = sHexRegex.exec(base) || sHexaRegex.exec(base)) {
          return `#${i[1]}${i[1]}${i[2]}${i[2]}${i[3]}${i[3]}`;
        }
        throw new Error(`[seemly/toHexString]: Invalid hex value ${base}.`);
      }
      return `#${base.slice(0, 3).map(unit => roundChannel(unit).toString(16).toUpperCase().padStart(2, '0')).join('')}`;
    }

    function createId(length = 8) {
      return Math.random().toString(16).slice(2, 2 + length);
    }
    function repeat(count, v) {
      const ret = [];
      for (let i = 0; i < count; ++i) {
        ret.push(v);
      }
      return ret;
    }
    function indexMap(count, createValue) {
      const ret = [];
      if (!createValue) {
        for (let i = 0; i < count; ++i) {
          ret.push(i);
        }
        return ret;
      }
      for (let i = 0; i < count; ++i) {
        ret.push(createValue(i));
      }
      return ret;
    }

    function getSlot$1(instance, slotName = "default", fallback = []) {
      const slots = instance.$slots;
      const slot = slots[slotName];
      if (slot === void 0) return fallback;
      return slot();
    }

    function getVNodeChildren(vNode, slotName = "default", fallback = []) {
      const {
        children
      } = vNode;
      if (children !== null && typeof children === "object" && !Array.isArray(children)) {
        const slot = children[slotName];
        if (typeof slot === "function") {
          return slot();
        }
      }
      return fallback;
    }

    function keep(object, keys = [], rest) {
      const keepedObject = {};
      keys.forEach(key => {
        keepedObject[key] = object[key];
      });
      return Object.assign(keepedObject, rest);
    }

    function omit(object, keys = [], rest) {
      const omitedObject = {};
      const originalKeys = Object.getOwnPropertyNames(object);
      originalKeys.forEach(originalKey => {
        if (!keys.includes(originalKey)) {
          omitedObject[originalKey] = object[originalKey];
        }
      });
      return Object.assign(omitedObject, rest);
    }

    function flatten$3(vNodes, filterCommentNode = true, result = []) {
      vNodes.forEach(vNode => {
        if (vNode === null) return;
        if (typeof vNode !== "object") {
          if (typeof vNode === "string" || typeof vNode === "number") {
            result.push(vue.createTextVNode(String(vNode)));
          }
          return;
        }
        if (Array.isArray(vNode)) {
          flatten$3(vNode, filterCommentNode, result);
          return;
        }
        if (vNode.type === vue.Fragment) {
          if (vNode.children === null) return;
          if (Array.isArray(vNode.children)) {
            flatten$3(vNode.children, filterCommentNode, result);
          }
        } else {
          if (vNode.type === vue.Comment && filterCommentNode) return;
          result.push(vNode);
        }
      });
      return result;
    }

    function call(funcs, ...args) {
      if (Array.isArray(funcs)) {
        funcs.forEach(func => call(func, ...args));
      } else {
        return funcs(...args);
      }
    }

    function keysOf(obj) {
      return Object.keys(obj);
    }

    function render$1(r, ...args) {
      if (typeof r === "function") {
        return r(...args);
      } else if (typeof r === "string") {
        return vue.createTextVNode(r);
      } else if (typeof r === "number") {
        return vue.createTextVNode(String(r));
      } else {
        return null;
      }
    }

    const warnedMessages = /* @__PURE__ */new Set();
    function warnOnce(location, message) {
      const mergedMessage = `[naive/${location}]: ${message}`;
      if (warnedMessages.has(mergedMessage)) return;
      warnedMessages.add(mergedMessage);
      console.error(mergedMessage);
    }
    function warn$2(location, message) {
      console.error(`[naive/${location}]: ${message}`);
    }
    function error(location, message, error2) {
      console.error(`[naive/${location}]: ${message}`, error2);
    }
    function throwError(location, message) {
      throw new Error(`[naive/${location}]: ${message}`);
    }

    function smallerSize(size) {
      switch (size) {
        case "tiny":
          return "mini";
        case "small":
          return "tiny";
        case "medium":
          return "small";
        case "large":
          return "medium";
        case "huge":
          return "large";
      }
      throw new Error(`${size} has no smaller size.`);
    }

    function getTitleAttribute(value) {
      switch (typeof value) {
        case "string":
          return value || void 0;
        case "number":
          return String(value);
        default:
          return void 0;
      }
    }

    function getFirstSlotVNode(slots, slotName = "default", props = void 0) {
      const slot = slots[slotName];
      if (!slot) {
        warn$2("getFirstSlotVNode", `slot[${slotName}] is empty`);
        return null;
      }
      const slotContent = flatten$3(slot(props));
      if (slotContent.length === 1) {
        return slotContent[0];
      } else {
        warn$2("getFirstSlotVNode", `slot[${slotName}] should have exactly one child`);
        return null;
      }
    }

    function createDataKey(key) {
      return typeof key === "string" ? `s-${key}` : `n-${key}`;
    }

    function createRefSetter(ref) {
      return inst => {
        if (inst) {
          ref.value = inst.$el;
        } else {
          ref.value = null;
        }
      };
    }

    function createInjectionKey(key) {
      return key;
    }

    function ensureValidVNode(vnodes) {
      return vnodes.some(child => {
        if (!vue.isVNode(child)) {
          return true;
        }
        if (child.type === vue.Comment) {
          return false;
        }
        if (child.type === vue.Fragment && !ensureValidVNode(child.children)) {
          return false;
        }
        return true;
      }) ? vnodes : null;
    }
    function resolveSlot(slot, fallback) {
      return slot && ensureValidVNode(slot()) || fallback();
    }
    function resolveSlotWithProps(slot, props, fallback) {
      return slot && ensureValidVNode(slot(props)) || fallback(props);
    }
    function resolveWrappedSlot(slot, wrapper) {
      const children = slot && ensureValidVNode(slot());
      return wrapper(children || null);
    }
    function resolveWrappedSlotWithProps(slot, props, wrapper) {
      const children = slot && ensureValidVNode(slot(props));
      return wrapper(children || null);
    }
    function isSlotEmpty(slot) {
      return !(slot && ensureValidVNode(slot()));
    }

    function mergeEventHandlers(handlers) {
      const filteredHandlers = handlers.filter(handler => handler !== void 0);
      if (filteredHandlers.length === 0) return void 0;
      if (filteredHandlers.length === 1) return filteredHandlers[0];
      return e => {
        handlers.forEach(handler => {
          if (handler) {
            handler(e);
          }
        });
      };
    }

    function isNodeVShowFalse(vNode) {
      const showDir = vNode.dirs?.find(({
        dir
      }) => dir === vue.vShow);
      return !!(showDir && showDir.value === false);
    }

    const Wrapper = vue.defineComponent({
      render() {
        return this.$slots.default?.();
      }
    });

    const pureNumberRegex = /^(\d|\.)+$/;
    const numberRegex = /(\d|\.)+/;
    function formatLength(length, {
      c = 1,
      offset = 0,
      attachPx = true
    } = {}) {
      if (typeof length === "number") {
        const result = (length + offset) * c;
        if (result === 0) return "0";
        return `${result}px`;
      } else if (typeof length === "string") {
        if (pureNumberRegex.test(length)) {
          const result = (Number(length) + offset) * c;
          if (attachPx) {
            if (result === 0) return "0";
            return `${result}px`;
          } else {
            return `${result}`;
          }
        } else {
          const result = numberRegex.exec(length);
          if (!result) return length;
          return length.replace(numberRegex, String((Number(result[0]) + offset) * c));
        }
      }
      return length;
    }

    function color2Class(color) {
      return color.replace(/#|\(|\)|,|\s|\./g, "_");
    }

    function rtlInset(inset) {
      const {
        left,
        right,
        top,
        bottom
      } = getMargin(inset);
      return `${top} ${right} ${bottom} ${left}`;
    }

    function ampCount(selector) {
      let cnt = 0;
      for (let i = 0; i < selector.length; ++i) {
        if (selector[i] === '&') ++cnt;
      }
      return cnt;
    }
    /**
     * Don't just use ',' to separate css selector. For example:
     * x:(a, b) {} will be split into 'x:(a' and 'b)', which is not expected.
     * Make sure comma doesn't exist inside parentheses.
     */
    const separatorRegex = /\s*,(?![^(]*\))\s*/g;
    const extraSpaceRegex = /\s+/g;
    /**
     * selector must includes '&'
     * selector is trimmed
     * every part of amp is trimmed
     */
    function resolveSelectorWithAmp(amp, selector) {
      const nextAmp = [];
      selector.split(separatorRegex).forEach(partialSelector => {
        let round = ampCount(partialSelector);
        // eslint-disable-next-line @typescript-eslint/strict-boolean-expressions
        if (!round) {
          amp.forEach(partialAmp => {
            nextAmp.push(
            // eslint-disable-next-line @typescript-eslint/strict-boolean-expressions
            (partialAmp && partialAmp + ' ') + partialSelector);
          });
          return;
        } else if (round === 1) {
          amp.forEach(partialAmp => {
            nextAmp.push(partialSelector.replace('&', partialAmp));
          });
          return;
        }
        let partialNextAmp = [partialSelector];
        // eslint-disable-next-line @typescript-eslint/strict-boolean-expressions
        while (round--) {
          const nextPartialNextAmp = [];
          partialNextAmp.forEach(selectorItr => {
            amp.forEach(partialAmp => {
              nextPartialNextAmp.push(selectorItr.replace('&', partialAmp));
            });
          });
          partialNextAmp = nextPartialNextAmp;
        }
        partialNextAmp.forEach(part => nextAmp.push(part));
      });
      return nextAmp;
    }
    /**
     * selector mustn't includes '&'
     * selector is trimmed
     */
    function resolveSelector(amp, selector) {
      const nextAmp = [];
      selector.split(separatorRegex).forEach(partialSelector => {
        amp.forEach(partialAmp => {
          // eslint-disable-next-line @typescript-eslint/strict-boolean-expressions
          nextAmp.push((partialAmp && partialAmp + ' ') + partialSelector);
        });
      });
      return nextAmp;
    }
    function parseSelectorPath(selectorPaths) {
      let amp = [''];
      selectorPaths.forEach(selector => {
        // eslint-disable-next-line
        selector = selector && selector.trim();
        if (
        // eslint-disable-next-line @typescript-eslint/strict-boolean-expressions
        !selector) {
          /**
           * if it's a empty selector, do nothing
           */
          return;
        }
        // eslint-disable-next-line @typescript-eslint/strict-boolean-expressions
        if (selector.includes('&')) {
          amp = resolveSelectorWithAmp(amp, selector);
        } else {
          amp = resolveSelector(amp, selector);
        }
      });
      return amp.join(', ').replace(extraSpaceRegex, ' ');
    }

    function removeElement(el) {
      /* istanbul ignore if */
      // eslint-disable-next-line @typescript-eslint/strict-boolean-expressions
      if (!el) return;
      const parentElement = el.parentElement;
      /* istanbul ignore else */
      // eslint-disable-next-line @typescript-eslint/strict-boolean-expressions
      if (parentElement) parentElement.removeChild(el);
    }
    function queryElement(id, parent) {
      return (parent !== null && parent !== void 0 ? parent : document.head).querySelector(`style[cssr-id="${id}"]`);
    }
    function createElement(id) {
      const el = document.createElement('style');
      el.setAttribute('cssr-id', id);
      return el;
    }
    function isMediaOrSupports(selector) {
      // eslint-disable-next-line @typescript-eslint/strict-boolean-expressions
      if (!selector) return false;
      return /^\s*@(s|m)/.test(selector);
    }

    const kebabRegex = /[A-Z]/g;
    function kebabCase(pattern) {
      return pattern.replace(kebabRegex, match => '-' + match.toLowerCase());
    }
    /** TODO: refine it to solve nested object */
    function unwrapProperty(prop, indent = '  ') {
      if (typeof prop === 'object' && prop !== null) {
        return ' {\n' + Object.entries(prop).map(v => {
          return indent + `  ${kebabCase(v[0])}: ${v[1]};`;
        }).join('\n') + '\n' + indent + '}';
      }
      return `: ${prop};`;
    }
    /** unwrap properties */
    function unwrapProperties(props, instance, params) {
      if (typeof props === 'function') {
        return props({
          context: instance.context,
          props: params
        });
      }
      return props;
    }
    function createStyle(selector, props, instance, params) {
      if (!props) return '';
      // eslint-disable-next-line
      const unwrappedProps = unwrapProperties(props, instance, params);
      if (!unwrappedProps) return '';
      if (typeof unwrappedProps === 'string') {
        return `${selector} {\n${unwrappedProps}\n}`;
      }
      const propertyNames = Object.keys(unwrappedProps);
      if (propertyNames.length === 0) {
        if (instance.config.keepEmptyBlock) return selector + ' {\n}';
        return '';
      }
      const statements = selector ? [selector + ' {'] : [];
      propertyNames.forEach(propertyName => {
        const property = unwrappedProps[propertyName];
        if (propertyName === 'raw') {
          statements.push('\n' + property + '\n');
          return;
        }
        propertyName = kebabCase(propertyName);
        if (property !== null && property !== undefined) {
          statements.push(`  ${propertyName}${unwrapProperty(property)}`);
        }
      });
      if (selector) {
        statements.push('}');
      }
      return statements.join('\n');
    }
    function loopCNodeListWithCallback(children, options, callback) {
      /* istanbul ignore if */
      if (!children) return;
      children.forEach(child => {
        if (Array.isArray(child)) {
          loopCNodeListWithCallback(child, options, callback);
        } else if (typeof child === 'function') {
          const grandChildren = child(options);
          if (Array.isArray(grandChildren)) {
            loopCNodeListWithCallback(grandChildren, options, callback);
          } else if (grandChildren) {
            callback(grandChildren);
          }
        } else if (child) {
          callback(child);
        }
      });
    }
    function traverseCNode(node, selectorPaths, styles, instance, params) {
      const $ = node.$;
      let blockSelector = '';
      if (!$ || typeof $ === 'string') {
        if (isMediaOrSupports($)) {
          blockSelector = $;
        } else {
          // as a string selector
          selectorPaths.push($);
        }
      } else if (typeof $ === 'function') {
        const selector = $({
          context: instance.context,
          props: params
        });
        if (isMediaOrSupports(selector)) {
          blockSelector = selector;
        } else {
          // as a lazy selector
          selectorPaths.push(selector);
        }
      } else {
        // as a option selector
        if ($.before) $.before(instance.context);
        if (!$.$ || typeof $.$ === 'string') {
          if (isMediaOrSupports($.$)) {
            blockSelector = $.$;
          } else {
            // as a string selector
            selectorPaths.push($.$);
          }
        } else /* istanbul ignore else */if ($.$) {
            const selector = $.$({
              context: instance.context,
              props: params
            });
            if (isMediaOrSupports(selector)) {
              blockSelector = selector;
            } else {
              // as a lazy selector
              selectorPaths.push(selector);
            }
          }
      }
      const selector = parseSelectorPath(selectorPaths);
      const style = createStyle(selector, node.props, instance, params);
      if (blockSelector) {
        styles.push(`${blockSelector} {`);
      } else if (style.length) {
        styles.push(style);
      }
      if (node.children) {
        loopCNodeListWithCallback(node.children, {
          context: instance.context,
          props: params
        }, childNode => {
          if (typeof childNode === 'string') {
            const style = createStyle(selector, {
              raw: childNode
            }, instance, params);
            styles.push(style);
          } else {
            traverseCNode(childNode, selectorPaths, styles, instance, params);
          }
        });
      }
      selectorPaths.pop();
      if (blockSelector) {
        styles.push('}');
      }
      if ($ && $.after) $.after(instance.context);
    }
    function render(node, instance, props) {
      const styles = [];
      traverseCNode(node, [], styles, instance, props);
      return styles.join('\n\n');
    }

    /* eslint-disable */
    // Inspired by https://github.com/garycourt/murmurhash-js
    // Ported from https://github.com/aappleby/smhasher/blob/61a0530f28277f2e850bfc39600ce61d02b518de/src/MurmurHash2.cpp#L37-L86
    function murmur2(str) {
      // 'm' and 'r' are mixing constants generated offline.
      // They're not really 'magic', they just happen to work well.
      // const m = 0x5bd1e995;
      // const r = 24;
      // Initialize the hash
      var h = 0; // Mix 4 bytes at a time into the hash

      var k,
        i = 0,
        len = str.length;
      for (; len >= 4; ++i, len -= 4) {
        k = str.charCodeAt(i) & 0xff | (str.charCodeAt(++i) & 0xff) << 8 | (str.charCodeAt(++i) & 0xff) << 16 | (str.charCodeAt(++i) & 0xff) << 24;
        k = /* Math.imul(k, m): */
        (k & 0xffff) * 0x5bd1e995 + ((k >>> 16) * 0xe995 << 16);
        k ^= /* k >>> r: */
        k >>> 24;
        h = /* Math.imul(k, m): */
        (k & 0xffff) * 0x5bd1e995 + ((k >>> 16) * 0xe995 << 16) ^ /* Math.imul(h, m): */
        (h & 0xffff) * 0x5bd1e995 + ((h >>> 16) * 0xe995 << 16);
      } // Handle the last few bytes of the input array

      switch (len) {
        case 3:
          h ^= (str.charCodeAt(i + 2) & 0xff) << 16;
        case 2:
          h ^= (str.charCodeAt(i + 1) & 0xff) << 8;
        case 1:
          h ^= str.charCodeAt(i) & 0xff;
          h = /* Math.imul(h, m): */
          (h & 0xffff) * 0x5bd1e995 + ((h >>> 16) * 0xe995 << 16);
      } // Do a few final mixes of the hash to ensure the last few
      // bytes are well-incorporated.

      h ^= h >>> 13;
      h = /* Math.imul(h, m): */
      (h & 0xffff) * 0x5bd1e995 + ((h >>> 16) * 0xe995 << 16);
      return ((h ^ h >>> 15) >>> 0).toString(36);
    }

    /* eslint-disable @typescript-eslint/prefer-ts-expect-error */
    /* eslint-disable @typescript-eslint/strict-boolean-expressions */
    if (typeof window !== 'undefined') {
      window.__cssrContext = {};
    }
    function unmount(instance, node, id, parent) {
      const {
        els
      } = node;
      // If id is undefined, unmount all styles
      if (id === undefined) {
        els.forEach(removeElement);
        node.els = [];
      } else {
        const target = queryElement(id, parent);
        // eslint-disable-next-line
        if (target && els.includes(target)) {
          removeElement(target);
          node.els = els.filter(el => el !== target);
        }
      }
    }
    function addElementToList(els, target) {
      els.push(target);
    }
    function mount(instance, node, id, props, head, force, anchorMetaName, parent, ssrAdapter
    // eslint-disable-next-line @typescript-eslint/no-invalid-void-type
    ) {
      let style;
      if (id === undefined) {
        style = node.render(props);
        id = murmur2(style);
      }
      if (ssrAdapter) {
        ssrAdapter.adapter(id, style !== null && style !== void 0 ? style : node.render(props));
        return;
      }
      if (parent === undefined) {
        parent = document.head;
      }
      const queriedTarget = queryElement(id, parent);
      if (queriedTarget !== null && !force) {
        return queriedTarget;
      }
      const target = queriedTarget !== null && queriedTarget !== void 0 ? queriedTarget : createElement(id);
      if (style === undefined) style = node.render(props);
      target.textContent = style;
      if (queriedTarget !== null) return queriedTarget;
      if (anchorMetaName) {
        const anchorMetaEl = parent.querySelector(`meta[name="${anchorMetaName}"]`);
        if (anchorMetaEl) {
          parent.insertBefore(target, anchorMetaEl);
          addElementToList(node.els, target);
          return target;
        }
      }
      if (head) {
        parent.insertBefore(target, parent.querySelector('style, link'));
      } else {
        parent.appendChild(target);
      }
      addElementToList(node.els, target);
      return target;
    }

    function wrappedRender(props) {
      return render(this, this.instance, props);
    }
    // do not guard node calling, it should throw an error.
    function wrappedMount(options = {}
    // eslint-disable-next-line @typescript-eslint/no-invalid-void-type
    ) {
      const {
        id,
        ssr,
        props,
        head = false,
        force = false,
        anchorMetaName,
        parent
      } = options;
      const targetElement = mount(this.instance, this, id, props, head, force, anchorMetaName, parent, ssr);
      return targetElement;
    }
    function wrappedUnmount(options = {}) {
      /* istanbul ignore next */
      // eslint-disable-next-line @typescript-eslint/strict-boolean-expressions
      const {
        id,
        parent
      } = options;
      unmount(this.instance, this, id, parent);
    }
    const createCNode = function (instance, $, props, children) {
      return {
        instance,
        $,
        props,
        children,
        els: [],
        render: wrappedRender,
        mount: wrappedMount,
        unmount: wrappedUnmount
      };
    };
    const c$2 = function (instance, $, props, children) {
      if (Array.isArray($)) {
        return createCNode(instance, {
          $: null
        }, null, $);
      } else if (Array.isArray(props)) {
        return createCNode(instance, $, null, props);
      } else if (Array.isArray(children)) {
        return createCNode(instance, $, props, children);
      } else {
        return createCNode(instance, $, props, null);
      }
    };

    function CssRender(config = {}) {
      const cssr = {
        c: (...args) => c$2(cssr, ...args),
        use: (plugin, ...args) => plugin.install(cssr, ...args),
        find: queryElement,
        context: {},
        config
      };
      return cssr;
    }

    function exists(id, ssr) {
      if (id === undefined) return false;
      if (ssr) {
        const {
          context: {
            ids
          }
        } = ssr;
        return ids.has(id);
      }
      return queryElement(id) !== null;
    }

    /* eslint-disable @typescript-eslint/restrict-template-expressions */
    /* eslint-disable @typescript-eslint/strict-boolean-expressions */
    function plugin$1(options) {
      let _bPrefix = '.';
      let _ePrefix = '__';
      let _mPrefix = '--';
      let c;
      if (options) {
        let t = options.blockPrefix;
        if (t) {
          _bPrefix = t;
        }
        t = options.elementPrefix;
        if (t) {
          _ePrefix = t;
        }
        t = options.modifierPrefix;
        if (t) {
          _mPrefix = t;
        }
      }
      const _plugin = {
        install(instance) {
          c = instance.c;
          const ctx = instance.context;
          ctx.bem = {};
          ctx.bem.b = null;
          ctx.bem.els = null;
        }
      };
      function b(arg) {
        let memorizedB;
        let memorizedE;
        return {
          before(ctx) {
            memorizedB = ctx.bem.b;
            memorizedE = ctx.bem.els;
            ctx.bem.els = null;
          },
          after(ctx) {
            ctx.bem.b = memorizedB;
            ctx.bem.els = memorizedE;
          },
          $({
            context,
            props
          }) {
            arg = typeof arg === 'string' ? arg : arg({
              context,
              props
            });
            context.bem.b = arg;
            return `${(props === null || props === void 0 ? void 0 : props.bPrefix) || _bPrefix}${context.bem.b}`;
          }
        };
      }
      function e(arg) {
        let memorizedE;
        return {
          before(ctx) {
            memorizedE = ctx.bem.els;
          },
          after(ctx) {
            ctx.bem.els = memorizedE;
          },
          $({
            context,
            props
          }) {
            arg = typeof arg === 'string' ? arg : arg({
              context,
              props
            });
            context.bem.els = arg.split(',').map(v => v.trim());
            return context.bem.els.map(el => `${(props === null || props === void 0 ? void 0 : props.bPrefix) || _bPrefix}${context.bem.b}${_ePrefix}${el}`).join(', ');
          }
        };
      }
      function m(arg) {
        return {
          $({
            context,
            props
          }) {
            arg = typeof arg === 'string' ? arg : arg({
              context,
              props
            });
            const modifiers = arg.split(',').map(v => v.trim());
            function elementToSelector(el) {
              return modifiers.map(modifier => `&${(props === null || props === void 0 ? void 0 : props.bPrefix) || _bPrefix}${context.bem.b}${el !== undefined ? `${_ePrefix}${el}` : ''}${_mPrefix}${modifier}`).join(', ');
            }
            const els = context.bem.els;
            if (els !== null) {
              if (els.length >= 2) {
                throw Error(`[css-render/plugin-bem]: m(${arg}) is invalid, using modifier inside multiple elements is not allowed`);
              }
              return elementToSelector(els[0]);
            } else {
              return elementToSelector();
            }
          }
        };
      }
      function notM(arg) {
        return {
          $({
            context,
            props
          }) {
            arg = typeof arg === 'string' ? arg : arg({
              context,
              props
            });
            const els = context.bem.els;
            if (els !== null && els.length >= 2) {
              throw Error(`[css-render/plugin-bem]: notM(${arg}) is invalid, using modifier inside multiple elements is not allowed`);
            }
            return `&:not(${(props === null || props === void 0 ? void 0 : props.bPrefix) || _bPrefix}${context.bem.b}${els !== null && els.length > 0 ? `${_ePrefix}${els[0]}` : ''}${_mPrefix}${arg})`;
          }
        };
      }
      const cB = (...args) => c(b(args[0]), args[1], args[2]);
      const cE = (...args) => c(e(args[0]), args[1], args[2]);
      const cM = (...args) => c(m(args[0]), args[1], args[2]);
      const cNotM = (...args) => c(notM(args[0]), args[1], args[2]);
      Object.assign(_plugin, {
        cB,
        cE,
        cM,
        cNotM
      });
      return _plugin;
    }

    const namespace = "n";
    const prefix = `.${namespace}-`;
    const elementPrefix = "__";
    const modifierPrefix = "--";
    const cssr = CssRender();
    const plugin = plugin$1({
      blockPrefix: prefix,
      elementPrefix,
      modifierPrefix
    });
    cssr.use(plugin);
    const {
      c: c$1,
      find
    } = cssr;
    const {
      cB,
      cE,
      cM,
      cNotM
    } = plugin;
    function insideModal(style) {
      return c$1(({
        props: {
          bPrefix
        }
      }) => `${bPrefix || prefix}modal, ${bPrefix || prefix}drawer`, [style]);
    }
    function insidePopover(style) {
      return c$1(({
        props: {
          bPrefix
        }
      }) => `${bPrefix || prefix}popover`, [style]);
    }
    function asModal(style) {
      return c$1(({
        props: {
          bPrefix
        }
      }) => `&${bPrefix || prefix}modal`, style);
    }
    const cCB = (...args) => {
      return c$1(">", [cB(...args)]);
    };
    function createKey(prefix2, suffix) {
      return prefix2 + (suffix === "default" ? "" : suffix.replace(/^[a-z]/, startChar => startChar.toUpperCase()));
    }

    let _isJsdom;
    function isJsdom() {
      if (_isJsdom === void 0) {
        _isJsdom = navigator.userAgent.includes("Node.js") || navigator.userAgent.includes("jsdom");
      }
      return _isJsdom;
    }

    const isBrowser$2 = typeof document !== "undefined" && typeof window !== "undefined";

    const eventSet = /* @__PURE__ */new WeakSet();
    function markEventEffectPerformed(event) {
      eventSet.add(event);
    }
    function eventEffectNotPerformed(event) {
      return !eventSet.has(event);
    }

    function useInjectionInstanceCollection(injectionName, collectionKey, registerKeyRef) {
      const injection = vue.inject(injectionName, null);
      if (injection === null) return;
      const vm = vue.getCurrentInstance()?.proxy;
      vue.watch(registerKeyRef, registerInstance);
      registerInstance(registerKeyRef.value);
      vue.onBeforeUnmount(() => {
        registerInstance(void 0, registerKeyRef.value);
      });
      function registerInstance(key, oldKey) {
        if (!injection) return;
        const collection = injection[collectionKey];
        if (oldKey !== void 0) removeInstance(collection, oldKey);
        if (key !== void 0) addInstance(collection, key);
      }
      function removeInstance(collection, key) {
        if (!collection[key]) collection[key] = [];
        collection[key].splice(collection[key].findIndex(instance => instance === vm), 1);
      }
      function addInstance(collection, key) {
        if (!collection[key]) collection[key] = [];
        if (!~collection[key].findIndex(instance => instance === vm)) {
          collection[key].push(vm);
        }
      }
    }
    function useInjectionCollection(injectionName, collectionKey, valueRef) {
      const injection = vue.inject(injectionName, null);
      if (injection === null) return;
      if (!(collectionKey in injection)) {
        injection[collectionKey] = [];
      }
      injection[collectionKey].push(valueRef.value);
      vue.watch(valueRef, (value, prevValue) => {
        const collectionArray = injection[collectionKey];
        const index = collectionArray.findIndex(collectionValue => collectionValue === prevValue);
        if (~index) collectionArray.splice(index, 1);
        collectionArray.push(value);
      });
      vue.onBeforeUnmount(() => {
        const collectionArray = injection[collectionKey];
        const index = collectionArray.findIndex(collectionValue => collectionValue === valueRef.value);
        if (~index) collectionArray.splice(index, 1);
      });
    }
    function useInjectionElementCollection(injectionName, collectionKey, getElement) {
      const injection = vue.inject(injectionName, null);
      if (injection === null) return;
      if (!(collectionKey in injection)) {
        injection[collectionKey] = [];
      }
      vue.onMounted(() => {
        const el = getElement();
        if (!el) return;
        injection[collectionKey].push(el);
      });
      vue.onBeforeUnmount(() => {
        const collectionArray = injection[collectionKey];
        const element = getElement();
        const index = collectionArray.findIndex(collectionElement => collectionElement === element);
        if (~index) collectionArray.splice(index, 1);
      });
    }

    function useDeferredTrue(valueRef, delay, shouldDelayRef) {
      const delayedRef = vue.ref(valueRef.value);
      let timerId = null;
      vue.watch(valueRef, value => {
        if (timerId !== null) window.clearTimeout(timerId);
        if (value === true) {
          if (shouldDelayRef && !shouldDelayRef.value) {
            delayedRef.value = true;
          } else {
            timerId = window.setTimeout(() => {
              delayedRef.value = true;
            }, delay);
          }
        } else {
          delayedRef.value = false;
        }
      });
      return delayedRef;
    }

    function useFalseUntilTruthy(originalRef) {
      const currentRef = vue.ref(!!originalRef.value);
      if (currentRef.value) return vue.readonly(currentRef);
      const stop = vue.watch(originalRef, value => {
        if (value) {
          currentRef.value = true;
          stop();
        }
      });
      return vue.readonly(currentRef);
    }

    function useMemo(getterOrOptions) {
      const computedValueRef = vue.computed(getterOrOptions);
      // Maybe it's not possible to lazy evaluate the value, since we can't make
      // render phase capture the deps behind useMemo
      const valueRef = vue.ref(computedValueRef.value);
      vue.watch(computedValueRef, value => {
        valueRef.value = value;
      });
      if (typeof getterOrOptions === 'function') {
        return valueRef;
      } else {
        return {
          __v_isRef: true,
          get value() {
            return valueRef.value;
          },
          set value(v) {
            getterOrOptions.set(v);
          }
        };
      }
    }

    function hasInstance() {
      return vue.getCurrentInstance() !== null;
    }
    const isBrowser$1 = typeof window !== 'undefined';

    let fontsReady;
    let isFontReady;
    const init$1 = () => {
      var _a, _b;
      fontsReady = isBrowser$1 ? (_b = (_a = document) === null || _a === void 0 ? void 0 : _a.fonts) === null || _b === void 0 ? void 0 : _b.ready : undefined;
      isFontReady = false;
      /* istanbul ignore if */
      if (fontsReady !== undefined) {
        void fontsReady.then(() => {
          isFontReady = true;
        });
      } else {
        isFontReady = true;
      }
    };
    init$1();
    /**
     * Call callback on fontsReady is resolved. If fontsReady is already resolved,
     * callback won't be called.
     */
    function onFontsReady(cb) {
      /* istanbul ignore next */
      if (isFontReady) return;
      let deactivated = false;
      vue.onMounted(() => {
        /* istanbul ignore next */
        if (!isFontReady) {
          fontsReady === null || fontsReady === void 0 ? void 0 : fontsReady.then(() => {
            if (deactivated) return;
            cb();
          });
        }
      });
      vue.onBeforeUnmount(() => {
        deactivated = true;
      });
    }

    function getEventTarget(e) {
      const path = e.composedPath();
      return path[0];
    }

    const traps = {
      mousemoveoutside: new WeakMap(),
      clickoutside: new WeakMap()
    };
    function createTrapHandler(name, el, originalHandler) {
      if (name === 'mousemoveoutside') {
        const moveHandler = e => {
          if (el.contains(getEventTarget(e))) return;
          originalHandler(e);
        };
        return {
          mousemove: moveHandler,
          touchstart: moveHandler
        };
      } else if (name === 'clickoutside') {
        let mouseDownOutside = false;
        const downHandler = e => {
          mouseDownOutside = !el.contains(getEventTarget(e));
        };
        const upHanlder = e => {
          if (!mouseDownOutside) return;
          if (el.contains(getEventTarget(e))) return;
          originalHandler(e);
        };
        return {
          mousedown: downHandler,
          mouseup: upHanlder,
          touchstart: downHandler,
          touchend: upHanlder
        };
      }
      console.error(
      // eslint-disable-next-line @typescript-eslint/restrict-template-expressions
      `[evtd/create-trap-handler]: name \`${name}\` is invalid. This could be a bug of evtd.`);
      return {};
    }
    function ensureTrapHandlers(name, el, handler) {
      const handlers = traps[name];
      let elHandlers = handlers.get(el);
      if (elHandlers === undefined) {
        handlers.set(el, elHandlers = new WeakMap());
      }
      let trapHandler = elHandlers.get(handler);
      if (trapHandler === undefined) {
        elHandlers.set(handler, trapHandler = createTrapHandler(name, el, handler));
      }
      return trapHandler;
    }
    function trapOn(name, el, handler, options) {
      if (name === 'mousemoveoutside' || name === 'clickoutside') {
        const trapHandlers = ensureTrapHandlers(name, el, handler);
        Object.keys(trapHandlers).forEach(key => {
          on(key, document, trapHandlers[key], options);
        });
        return true;
      }
      return false;
    }
    function trapOff(name, el, handler, options) {
      if (name === 'mousemoveoutside' || name === 'clickoutside') {
        const trapHandlers = ensureTrapHandlers(name, el, handler);
        Object.keys(trapHandlers).forEach(key => {
          off(key, document, trapHandlers[key], options);
        });
        return true;
      }
      return false;
    }

    // currently `once` and `passive` is not supported
    function createDelegate() {
      if (typeof window === 'undefined') {
        return {
          on: () => {},
          off: () => {}
        };
      }
      const propagationStopped = new WeakMap();
      const immediatePropagationStopped = new WeakMap();
      function trackPropagation() {
        propagationStopped.set(this, true);
      }
      function trackImmediate() {
        propagationStopped.set(this, true);
        immediatePropagationStopped.set(this, true);
      }
      function spy(event, propName, fn) {
        const source = event[propName];
        event[propName] = function () {
          fn.apply(event, arguments);
          return source.apply(event, arguments);
        };
        return event;
      }
      function unspy(event, propName) {
        event[propName] = Event.prototype[propName];
      }
      const currentTargets = new WeakMap();
      const currentTargetDescriptor = Object.getOwnPropertyDescriptor(Event.prototype, 'currentTarget');
      function getCurrentTarget() {
        var _a;
        return (_a = currentTargets.get(this)) !== null && _a !== void 0 ? _a : null;
      }
      function defineCurrentTarget(event, getter) {
        if (currentTargetDescriptor === undefined) return;
        Object.defineProperty(event, 'currentTarget', {
          configurable: true,
          enumerable: true,
          get: getter !== null && getter !== void 0 ? getter : currentTargetDescriptor.get
        });
      }
      const phaseToTypeToElToHandlers = {
        bubble: {},
        capture: {}
      };
      const typeToWindowEventHandlers = {};
      function createUnifiedHandler() {
        const delegeteHandler = function (e) {
          const {
            type,
            eventPhase,
            bubbles
          } = e;
          const target = getEventTarget(e);
          if (eventPhase === 2) return;
          const phase = eventPhase === 1 ? 'capture' : 'bubble';
          let cursor = target;
          const path = [];
          // collecting bubble path
          while (true) {
            if (cursor === null) cursor = window;
            path.push(cursor);
            if (cursor === window) {
              break;
            }
            // eslint-disable-next-line @typescript-eslint/strict-boolean-expressions
            cursor = cursor.parentNode || null;
          }
          const captureElToHandlers = phaseToTypeToElToHandlers.capture[type];
          const bubbleElToHandlers = phaseToTypeToElToHandlers.bubble[type];
          spy(e, 'stopPropagation', trackPropagation);
          spy(e, 'stopImmediatePropagation', trackImmediate);
          defineCurrentTarget(e, getCurrentTarget);
          if (phase === 'capture') {
            if (captureElToHandlers === undefined) return;
            // capture
            for (let i = path.length - 1; i >= 0; --i) {
              if (propagationStopped.has(e)) break;
              const target = path[i];
              const handlers = captureElToHandlers.get(target);
              if (handlers !== undefined) {
                currentTargets.set(e, target);
                for (const handler of handlers) {
                  if (immediatePropagationStopped.has(e)) break;
                  handler(e);
                }
              }
              if (i === 0 && !bubbles && bubbleElToHandlers !== undefined) {
                const bubbleHandlers = bubbleElToHandlers.get(target);
                if (bubbleHandlers !== undefined) {
                  for (const handler of bubbleHandlers) {
                    if (immediatePropagationStopped.has(e)) break;
                    handler(e);
                  }
                }
              }
            }
          } else if (phase === 'bubble') {
            if (bubbleElToHandlers === undefined) return;
            // bubble
            for (let i = 0; i < path.length; ++i) {
              if (propagationStopped.has(e)) break;
              const target = path[i];
              const handlers = bubbleElToHandlers.get(target);
              if (handlers !== undefined) {
                currentTargets.set(e, target);
                for (const handler of handlers) {
                  if (immediatePropagationStopped.has(e)) break;
                  handler(e);
                }
              }
            }
          }
          unspy(e, 'stopPropagation');
          unspy(e, 'stopImmediatePropagation');
          defineCurrentTarget(e);
        };
        delegeteHandler.displayName = 'evtdUnifiedHandler';
        return delegeteHandler;
      }
      function createUnifiedWindowEventHandler() {
        const delegateHandler = function (e) {
          const {
            type,
            eventPhase
          } = e;
          if (eventPhase !== 2) return;
          const handlers = typeToWindowEventHandlers[type];
          if (handlers === undefined) return;
          handlers.forEach(handler => handler(e));
        };
        delegateHandler.displayName = 'evtdUnifiedWindowEventHandler';
        return delegateHandler;
      }
      const unifiedHandler = createUnifiedHandler();
      const unfiendWindowEventHandler = createUnifiedWindowEventHandler();
      function ensureElToHandlers(phase, type) {
        const phaseHandlers = phaseToTypeToElToHandlers[phase];
        if (phaseHandlers[type] === undefined) {
          phaseHandlers[type] = new Map();
          window.addEventListener(type, unifiedHandler, phase === 'capture');
        }
        return phaseHandlers[type];
      }
      function ensureWindowEventHandlers(type) {
        const windowEventHandlers = typeToWindowEventHandlers[type];
        if (windowEventHandlers === undefined) {
          typeToWindowEventHandlers[type] = new Set();
          window.addEventListener(type, unfiendWindowEventHandler);
        }
        return typeToWindowEventHandlers[type];
      }
      function ensureHandlers(elToHandlers, el) {
        let elHandlers = elToHandlers.get(el);
        if (elHandlers === undefined) {
          elToHandlers.set(el, elHandlers = new Set());
        }
        return elHandlers;
      }
      function handlerExist(el, phase, type, handler) {
        const elToHandlers = phaseToTypeToElToHandlers[phase][type];
        // phase ${type} event has handlers
        if (elToHandlers !== undefined) {
          const handlers = elToHandlers.get(el);
          // phase using el with ${type} event has handlers
          if (handlers !== undefined) {
            if (handlers.has(handler)) return true;
          }
        }
        return false;
      }
      function windowEventHandlerExist(type, handler) {
        const handlers = typeToWindowEventHandlers[type];
        if (handlers !== undefined) {
          if (handlers.has(handler)) {
            return true;
          }
        }
        return false;
      }
      function on(type, el, handler, options) {
        let mergedHandler;
        if (typeof options === 'object' && options.once === true) {
          mergedHandler = e => {
            off(type, el, mergedHandler, options);
            handler(e);
          };
        } else {
          mergedHandler = handler;
        }
        const trapped = trapOn(type, el, mergedHandler, options);
        if (trapped) return;
        const phase = options === true || typeof options === 'object' && options.capture === true ? 'capture' : 'bubble';
        const elToHandlers = ensureElToHandlers(phase, type);
        const handlers = ensureHandlers(elToHandlers, el);
        if (!handlers.has(mergedHandler)) handlers.add(mergedHandler);
        if (el === window) {
          const windowEventHandlers = ensureWindowEventHandlers(type);
          if (!windowEventHandlers.has(mergedHandler)) {
            windowEventHandlers.add(mergedHandler);
          }
        }
      }
      function off(type, el, handler, options) {
        const trapped = trapOff(type, el, handler, options);
        if (trapped) return;
        const capture = options === true || typeof options === 'object' && options.capture === true;
        const phase = capture ? 'capture' : 'bubble';
        const elToHandlers = ensureElToHandlers(phase, type);
        const handlers = ensureHandlers(elToHandlers, el);
        if (el === window) {
          const mirrorPhase = capture ? 'bubble' : 'capture';
          if (!handlerExist(el, mirrorPhase, type, handler) && windowEventHandlerExist(type, handler)) {
            // eslint-disable-next-line @typescript-eslint/no-non-null-assertion
            const windowEventHandlers = typeToWindowEventHandlers[type];
            windowEventHandlers.delete(handler);
            if (windowEventHandlers.size === 0) {
              window.removeEventListener(type, unfiendWindowEventHandler);
              typeToWindowEventHandlers[type] = undefined;
            }
          }
        }
        if (handlers.has(handler)) handlers.delete(handler);
        if (handlers.size === 0) {
          elToHandlers.delete(el);
        }
        if (elToHandlers.size === 0) {
          window.removeEventListener(type, unifiedHandler, phase === 'capture');
          phaseToTypeToElToHandlers[phase][type] = undefined;
        }
      }
      return {
        on: on,
        off: off
      };
    }
    const {
      on,
      off
    } = createDelegate();

    const mousePositionRef = vue.ref(null);
    function clickHandler(e) {
      if (e.clientX > 0 || e.clientY > 0) {
        mousePositionRef.value = {
          x: e.clientX,
          y: e.clientY
        };
      } else {
        // x = 0 & y = 0
        const {
          target
        } = e;
        if (target instanceof Element) {
          const {
            left,
            top,
            width,
            height
          } = target.getBoundingClientRect();
          if (left > 0 || top > 0) {
            // impossible to be triggered by real click
            mousePositionRef.value = {
              x: left + width / 2,
              y: top + height / 2
            };
          } else {
            mousePositionRef.value = {
              x: 0,
              y: 0
            };
          }
        } else {
          mousePositionRef.value = null;
        }
      }
    }
    let usedCount$2 = 0;
    let managable$2 = true;
    function useClickPosition() {
      if (!isBrowser$1) return vue.readonly(vue.ref(null));
      if (usedCount$2 === 0) on('click', document, clickHandler, true);
      const setup = () => {
        usedCount$2 += 1;
      };
      if (managable$2 && (managable$2 = hasInstance())) {
        vue.onBeforeMount(setup);
        vue.onBeforeUnmount(() => {
          usedCount$2 -= 1;
          if (usedCount$2 === 0) off('click', document, clickHandler, true);
        });
      } else {
        setup();
      }
      return vue.readonly(mousePositionRef);
    }

    const clickedTimeRef = vue.ref(undefined);
    let usedCount$1 = 0;
    function handleClick() {
      clickedTimeRef.value = Date.now();
    }
    let managable$1 = true;
    function useClicked(timeout) {
      if (!isBrowser$1) return vue.readonly(vue.ref(false));
      const clickedRef = vue.ref(false);
      let timerId = null;
      function clearTimer() {
        if (timerId !== null) window.clearTimeout(timerId);
      }
      function clickedHandler() {
        clearTimer();
        clickedRef.value = true;
        timerId = window.setTimeout(() => {
          clickedRef.value = false;
        }, timeout);
      }
      if (usedCount$1 === 0) {
        on('click', window, handleClick, true);
      }
      const setup = () => {
        usedCount$1 += 1;
        on('click', window, clickedHandler, true);
      };
      if (managable$1 && (managable$1 = hasInstance())) {
        vue.onBeforeMount(setup);
        vue.onBeforeUnmount(() => {
          usedCount$1 -= 1;
          if (usedCount$1 === 0) {
            off('click', window, handleClick, true);
          }
          off('click', window, clickedHandler, true);
          clearTimer();
        });
      } else {
        setup();
      }
      return vue.readonly(clickedRef);
    }

    /* eslint-disable @typescript-eslint/strict-boolean-expressions */
    /* eslint-disable @typescript-eslint/no-non-null-assertion */
    let usedCount = 0;
    // Mql means media query list
    const supportMatchMedia = typeof window !== 'undefined' && window.matchMedia !== undefined;
    const osTheme = vue.ref(null);
    let darkMql;
    let lightMql;
    function handleDarkMqlChange(e) {
      if (e.matches) {
        osTheme.value = 'dark';
      }
    }
    function handleLightMqlChange(e) {
      if (e.matches) {
        osTheme.value = 'light';
      }
    }
    function init() {
      darkMql = window.matchMedia('(prefers-color-scheme: dark)');
      lightMql = window.matchMedia('(prefers-color-scheme: light)');
      if (darkMql.matches) {
        osTheme.value = 'dark';
      } else if (lightMql.matches) {
        osTheme.value = 'light';
      } else {
        osTheme.value = null;
      }
      if (darkMql.addEventListener) {
        darkMql.addEventListener('change', handleDarkMqlChange);
        lightMql.addEventListener('change', handleLightMqlChange);
      } else if (darkMql.addListener) {
        darkMql.addListener(handleDarkMqlChange);
        lightMql.addListener(handleLightMqlChange);
      }
    }
    function clean() {
      if ('removeEventListener' in darkMql) {
        darkMql.removeEventListener('change', handleDarkMqlChange);
        lightMql.removeEventListener('change', handleLightMqlChange);
      } else if ('removeListener' in darkMql) {
        darkMql.removeListener(handleDarkMqlChange);
        lightMql.removeListener(handleLightMqlChange);
      }
      darkMql = undefined;
      lightMql = undefined;
    }
    let managable = true;
    function useOsTheme() {
      /* istanbul ignore next */
      if (!supportMatchMedia) {
        return vue.readonly(osTheme);
      }
      if (usedCount === 0) init();
      if (managable && (managable = hasInstance())) {
        vue.onBeforeMount(() => {
          usedCount += 1;
        });
        vue.onBeforeUnmount(() => {
          usedCount -= 1;
          if (usedCount === 0) clean();
        });
      }
      return vue.readonly(osTheme);
    }

    function useMergedState(controlledStateRef, uncontrolledStateRef) {
      vue.watch(controlledStateRef, value => {
        if (value !== undefined) {
          uncontrolledStateRef.value = value;
        }
      });
      return vue.computed(() => {
        if (controlledStateRef.value === undefined) {
          return uncontrolledStateRef.value;
        }
        return controlledStateRef.value;
      });
    }

    function isMounted() {
      const isMounted = vue.ref(false);
      vue.onMounted(() => {
        isMounted.value = true;
      });
      return vue.readonly(isMounted);
    }

    function useCompitable(reactive, keys) {
      // @ts-expect-error
      return vue.computed(() => {
        for (const key of keys) {
          if (reactive[key] !== undefined) return reactive[key];
        }
        return reactive[keys[keys.length - 1]];
      });
    }

    const isIos = (typeof window === 'undefined' ? false : /iPad|iPhone|iPod/.test(navigator.platform) || navigator.platform === 'MacIntel' && navigator.maxTouchPoints > 1) &&
    // eslint-disable-next-line @typescript-eslint/strict-boolean-expressions
    !window.MSStream;
    function useIsIos() {
      return isIos;
    }

    /* eslint-disable @typescript-eslint/consistent-type-assertions */
    const defaultBreakpointOptions = {
      // mobile
      // 0 ~ 640 doesn't mean it should display well in all the range,
      // but means you should treat it like a mobile phone.)
      xs: 0,
      s: 640,
      m: 1024,
      l: 1280,
      xl: 1536,
      '2xl': 1920 // normal desktop display
    };
    function createMediaQuery(screenWidth) {
      return `(min-width: ${screenWidth}px)`;
    }
    const mqlMap = {};
    function useBreakpoints(screens = defaultBreakpointOptions) {
      if (!isBrowser$1) return vue.computed(() => []);
      // eslint-disable-next-line @typescript-eslint/strict-boolean-expressions
      if (typeof window.matchMedia !== 'function') return vue.computed(() => []);
      const breakpointStatusRef = vue.ref({});
      const breakpoints = Object.keys(screens);
      const updateBreakpoints = (e, breakpointName) => {
        if (e.matches) breakpointStatusRef.value[breakpointName] = true;else breakpointStatusRef.value[breakpointName] = false;
      };
      breakpoints.forEach(key => {
        const breakpointValue = screens[key];
        let mql;
        let cbs;
        if (mqlMap[breakpointValue] === undefined) {
          mql = window.matchMedia(createMediaQuery(breakpointValue));
          // eslint-disable-next-line @typescript-eslint/strict-boolean-expressions
          if (mql.addEventListener) {
            mql.addEventListener('change', e => {
              cbs.forEach(cb => {
                cb(e, key);
              });
            });
            // eslint-disable-next-line @typescript-eslint/strict-boolean-expressions
          } else if (mql.addListener) {
            mql.addListener(e => {
              cbs.forEach(cb => {
                cb(e, key);
              });
            });
          }
          cbs = new Set();
          mqlMap[breakpointValue] = {
            mql,
            cbs
          };
        } else {
          mql = mqlMap[breakpointValue].mql;
          cbs = mqlMap[breakpointValue].cbs;
        }
        cbs.add(updateBreakpoints);
        if (mql.matches) {
          cbs.forEach(cb => {
            cb(mql, key);
          });
        }
      });
      vue.onBeforeUnmount(() => {
        breakpoints.forEach(breakpoint => {
          const {
            cbs
          } = mqlMap[screens[breakpoint]];
          if (cbs.has(updateBreakpoints)) {
            cbs.delete(updateBreakpoints);
          }
        });
      });
      return vue.computed(() => {
        const {
          value
        } = breakpointStatusRef;
        return breakpoints.filter(key => value[key]);
      });
    }

    function useKeyboard$1(options = {}, enabledRef) {
      const state = vue.reactive({
        ctrl: false,
        command: false,
        win: false,
        shift: false,
        tab: false
      });
      const {
        keydown,
        keyup
      } = options;
      const keydownHandler = e => {
        switch (e.key) {
          case 'Control':
            state.ctrl = true;
            break;
          case 'Meta':
            state.command = true;
            state.win = true;
            break;
          case 'Shift':
            state.shift = true;
            break;
          case 'Tab':
            state.tab = true;
            break;
        }
        if (keydown !== undefined) {
          Object.keys(keydown).forEach(key => {
            if (key !== e.key) return;
            const handler = keydown[key];
            if (typeof handler === 'function') {
              handler(e);
            } else {
              const {
                stop = false,
                prevent = false
              } = handler;
              if (stop) e.stopPropagation();
              if (prevent) e.preventDefault();
              handler.handler(e);
            }
          });
        }
      };
      const keyupHandler = e => {
        switch (e.key) {
          case 'Control':
            state.ctrl = false;
            break;
          case 'Meta':
            state.command = false;
            state.win = false;
            break;
          case 'Shift':
            state.shift = false;
            break;
          case 'Tab':
            state.tab = false;
            break;
        }
        if (keyup !== undefined) {
          Object.keys(keyup).forEach(key => {
            if (key !== e.key) return;
            const handler = keyup[key];
            if (typeof handler === 'function') {
              handler(e);
            } else {
              const {
                stop = false,
                prevent = false
              } = handler;
              if (stop) e.stopPropagation();
              if (prevent) e.preventDefault();
              handler.handler(e);
            }
          });
        }
      };
      const setup = () => {
        if (enabledRef === undefined || enabledRef.value) {
          on('keydown', document, keydownHandler);
          on('keyup', document, keyupHandler);
        }
        if (enabledRef !== undefined) {
          vue.watch(enabledRef, value => {
            if (value) {
              on('keydown', document, keydownHandler);
              on('keyup', document, keyupHandler);
            } else {
              off('keydown', document, keydownHandler);
              off('keyup', document, keyupHandler);
            }
          });
        }
      };
      if (hasInstance()) {
        vue.onBeforeMount(setup);
        vue.onBeforeUnmount(() => {
          if (enabledRef === undefined || enabledRef.value) {
            off('keydown', document, keydownHandler);
            off('keyup', document, keyupHandler);
          }
        });
      } else {
        setup();
      }
      return vue.readonly(state);
    }

    const internalSelectionMenuInjectionKey = createInjectionKey("n-internal-select-menu");
    const internalSelectionMenuBodyInjectionKey = createInjectionKey("n-internal-select-menu-body");

    const modalBodyInjectionKey = createInjectionKey("n-modal-body");
    const modalProviderInjectionKey$1 = createInjectionKey("n-modal-provider");
    const modalInjectionKey = createInjectionKey("n-modal");

    const drawerBodyInjectionKey = createInjectionKey("n-drawer-body");
    const drawerInjectionKey = createInjectionKey("n-drawer");

    const popoverBodyInjectionKey = createInjectionKey("n-popover-body");

    const teleportDisabled = "__disabled__";
    function useAdjustedTo(props) {
      const modal = vue.inject(modalBodyInjectionKey, null);
      const drawer = vue.inject(drawerBodyInjectionKey, null);
      const popover = vue.inject(popoverBodyInjectionKey, null);
      const selectMenu = vue.inject(internalSelectionMenuBodyInjectionKey, null);
      const fullscreenElementRef = vue.ref();
      if (typeof document !== "undefined") {
        fullscreenElementRef.value = document.fullscreenElement;
        const handleFullscreenChange = () => {
          fullscreenElementRef.value = document.fullscreenElement;
        };
        vue.onMounted(() => {
          on("fullscreenchange", document, handleFullscreenChange);
        });
        vue.onBeforeUnmount(() => {
          off("fullscreenchange", document, handleFullscreenChange);
        });
      }
      return useMemo(() => {
        const {
          to
        } = props;
        if (to !== void 0) {
          if (to === false) return teleportDisabled;
          if (to === true) return fullscreenElementRef.value || "body";
          return to;
        }
        if (modal?.value) {
          return modal.value.$el ?? modal.value;
        }
        if (drawer?.value) return drawer.value;
        if (popover?.value) return popover.value;
        if (selectMenu?.value) return selectMenu.value;
        return to ?? (fullscreenElementRef.value || "body");
      });
    }
    useAdjustedTo.tdkey = teleportDisabled;
    useAdjustedTo.propTo = {
      type: [String, Object, Boolean],
      default: void 0
    };

    let houdiniRegistered = false;
    function useHoudini() {
      if (!isBrowser$2) return;
      if (!window.CSS) return;
      if (!houdiniRegistered) {
        houdiniRegistered = true;
        if ("registerProperty" in window?.CSS) {
          try {
            ;
            CSS.registerProperty({
              name: "--n-color-start",
              syntax: "<color>",
              inherits: false,
              initialValue: "#0000"
            });
            CSS.registerProperty({
              name: "--n-color-end",
              syntax: "<color>",
              inherits: false,
              initialValue: "#0000"
            });
          } catch {}
        }
      }
    }

    function getSlot(scope, slots, slotName = 'default') {
      const slot = slots[slotName];
      if (slot === undefined) {
        throw new Error(`[vueuc/${scope}]: slot[${slotName}] is empty.`);
      }
      return slot();
    }
    // o(n) flatten
    function flatten$2(vNodes, filterCommentNode = true, result = []) {
      vNodes.forEach(vNode => {
        if (vNode === null) return;
        if (typeof vNode !== 'object') {
          if (typeof vNode === 'string' || typeof vNode === 'number') {
            result.push(vue.createTextVNode(String(vNode)));
          }
          return;
        }
        if (Array.isArray(vNode)) {
          flatten$2(vNode, filterCommentNode, result);
          return;
        }
        if (vNode.type === vue.Fragment) {
          if (vNode.children === null) return;
          if (Array.isArray(vNode.children)) {
            flatten$2(vNode.children, filterCommentNode, result);
          }
          // rawSlot
        } else if (vNode.type !== vue.Comment) {
          result.push(vNode);
        }
      });
      return result;
    }
    function getFirstVNode(scope, slots, slotName = 'default') {
      const slot = slots[slotName];
      if (slot === undefined) {
        throw new Error(`[vueuc/${scope}]: slot[${slotName}] is empty.`);
      }
      const content = flatten$2(slot());
      // vue will normalize the slot, so slot must be an array
      if (content.length === 1) {
        return content[0];
      } else {
        throw new Error(`[vueuc/${scope}]: slot[${slotName}] should have exactly one child.`);
      }
    }

    let viewMeasurer = null;
    function ensureViewBoundingRect() {
      if (viewMeasurer === null) {
        viewMeasurer = document.getElementById('v-binder-view-measurer');
        if (viewMeasurer === null) {
          viewMeasurer = document.createElement('div');
          viewMeasurer.id = 'v-binder-view-measurer';
          const {
            style
          } = viewMeasurer;
          style.position = 'fixed';
          style.left = '0';
          style.right = '0';
          style.top = '0';
          style.bottom = '0';
          style.pointerEvents = 'none';
          style.visibility = 'hidden';
          document.body.appendChild(viewMeasurer);
        }
      }
      return viewMeasurer.getBoundingClientRect();
    }
    function getPointRect(x, y) {
      const viewRect = ensureViewBoundingRect();
      return {
        top: y,
        left: x,
        height: 0,
        width: 0,
        right: viewRect.width - x,
        bottom: viewRect.height - y
      };
    }
    function getRect$1(el) {
      const elRect = el.getBoundingClientRect();
      const viewRect = ensureViewBoundingRect();
      return {
        left: elRect.left - viewRect.left,
        top: elRect.top - viewRect.top,
        bottom: viewRect.height + viewRect.top - elRect.bottom,
        right: viewRect.width + viewRect.left - elRect.right,
        width: elRect.width,
        height: elRect.height
      };
    }
    function getParentNode(node) {
      // document type
      if (node.nodeType === 9) {
        return null;
      }
      return node.parentNode;
    }
    function getScrollParent(node) {
      if (node === null) return null;
      const parentNode = getParentNode(node);
      if (parentNode === null) {
        return null;
      }
      // Document
      if (parentNode.nodeType === 9) {
        return document;
      }
      // Element
      if (parentNode.nodeType === 1) {
        // Firefox want us to check `-x` and `-y` variations as well
        const {
          overflow,
          overflowX,
          overflowY
        } = getComputedStyle(parentNode);
        if (/(auto|scroll|overlay)/.test(overflow + overflowY + overflowX)) {
          return parentNode;
        }
      }
      return getScrollParent(parentNode);
    }

    /* eslint-disable @typescript-eslint/no-non-null-assertion */
    const Binder = vue.defineComponent({
      name: 'Binder',
      props: {
        syncTargetWithParent: Boolean,
        syncTarget: {
          type: Boolean,
          default: true
        }
      },
      setup(props) {
        var _a;
        vue.provide('VBinder', (_a = vue.getCurrentInstance()) === null || _a === void 0 ? void 0 : _a.proxy);
        const VBinder = vue.inject('VBinder', null);
        const targetRef = vue.ref(null);
        /**
         * If there's no nested vbinder, we can simply set the target ref.
         *
         * However, when it comes to:
         * <VBinder> <- syncTarget = false
         *
         *              Should hold target DOM ref, but can't get it directly from
         *              its VTarget. So if there are nested VBinder, we should:
         *              1. Stop setting target DOM from level-1 VTarget
         *              2. Set target DOM from level-2 VTarget
         *              For (1), we need `syncTarget` to `false`
         *              For (2), we need to set `syncTargetWithParent` to `true` on
         *              level-2 VBinder
         *   <VTarget>
         *     <VBinder> <- syncTargetWithParent = true
         *       <VTarget>target</VTarget>
         *     </VBinder>
         *     <VFollower>
         *       content1
         *     </VFollower>
         *   </VTarget>
         *   <VFollower>
         *     content2
         *   </VFollower>
         * </VBinder>
         */
        const setTargetRef = el => {
          targetRef.value = el;
          // eslint-disable-next-line @typescript-eslint/strict-boolean-expressions
          if (VBinder && props.syncTargetWithParent) {
            VBinder.setTargetRef(el);
          }
        };
        // scroll related
        let scrollableNodes = [];
        const ensureScrollListener = () => {
          let cursor = targetRef.value;
          while (true) {
            cursor = getScrollParent(cursor);
            if (cursor === null) break;
            scrollableNodes.push(cursor);
          }
          for (const el of scrollableNodes) {
            on('scroll', el, onScroll, true);
          }
        };
        const removeScrollListeners = () => {
          for (const el of scrollableNodes) {
            off('scroll', el, onScroll, true);
          }
          scrollableNodes = [];
        };
        const followerScrollListeners = new Set();
        const addScrollListener = listener => {
          if (followerScrollListeners.size === 0) {
            ensureScrollListener();
          }
          if (!followerScrollListeners.has(listener)) {
            followerScrollListeners.add(listener);
          }
        };
        const removeScrollListener = listener => {
          if (followerScrollListeners.has(listener)) {
            followerScrollListeners.delete(listener);
          }
          if (followerScrollListeners.size === 0) {
            removeScrollListeners();
          }
        };
        const onScroll = () => {
          beforeNextFrameOnce(onScrollRaf);
        };
        const onScrollRaf = () => {
          followerScrollListeners.forEach(listener => listener());
        };
        // resize related
        const followerResizeListeners = new Set();
        const addResizeListener = listener => {
          if (followerResizeListeners.size === 0) {
            on('resize', window, onResize);
          }
          if (!followerResizeListeners.has(listener)) {
            followerResizeListeners.add(listener);
          }
        };
        const removeResizeListener = listener => {
          if (followerResizeListeners.has(listener)) {
            followerResizeListeners.delete(listener);
          }
          if (followerResizeListeners.size === 0) {
            off('resize', window, onResize);
          }
        };
        const onResize = () => {
          followerResizeListeners.forEach(listener => listener());
        };
        vue.onBeforeUnmount(() => {
          off('resize', window, onResize);
          removeScrollListeners();
        });
        return {
          targetRef,
          setTargetRef,
          addScrollListener,
          removeScrollListener,
          addResizeListener,
          removeResizeListener
        };
      },
      render() {
        return getSlot('binder', this.$slots);
      }
    });

    /* eslint-disable @typescript-eslint/no-non-null-assertion */
    var VTarget = vue.defineComponent({
      name: 'Target',
      setup() {
        const {
          setTargetRef,
          syncTarget
        } = vue.inject('VBinder');
        const setTargetDirective = {
          mounted: setTargetRef,
          updated: setTargetRef
        };
        return {
          syncTarget,
          setTargetDirective
        };
      },
      render() {
        const {
          syncTarget,
          setTargetDirective
        } = this;
        /**
         * If you are using VBinder as a child of VBinder, the children wouldn't be
         * a valid DOM or component that can be attached to by directive.
         * So we won't sync target on those kind of situation and control the
         * target sync logic manually.
         */
        if (syncTarget) {
          return vue.withDirectives(getFirstVNode('follower', this.$slots), [[setTargetDirective]]);
        }
        return getFirstVNode('follower', this.$slots);
      }
    });

    const ctxKey$1 = '@@mmoContext';
    const mousemoveoutside = {
      mounted(el, {
        value
      }) {
        el[ctxKey$1] = {
          handler: undefined
        };
        if (typeof value === 'function') {
          el[ctxKey$1].handler = value;
          on('mousemoveoutside', el, value);
        }
      },
      updated(el, {
        value
      }) {
        const ctx = el[ctxKey$1];
        if (typeof value === 'function') {
          if (ctx.handler) {
            if (ctx.handler !== value) {
              off('mousemoveoutside', el, ctx.handler);
              ctx.handler = value;
              on('mousemoveoutside', el, value);
            }
          } else {
            el[ctxKey$1].handler = value;
            on('mousemoveoutside', el, value);
          }
        } else {
          if (ctx.handler) {
            off('mousemoveoutside', el, ctx.handler);
            ctx.handler = undefined;
          }
        }
      },
      unmounted(el) {
        const {
          handler
        } = el[ctxKey$1];
        if (handler) {
          off('mousemoveoutside', el, handler);
        }
        el[ctxKey$1].handler = undefined;
      }
    };

    const ctxKey = '@@coContext';
    const clickoutside = {
      mounted(el, {
        value,
        modifiers
      }) {
        el[ctxKey] = {
          handler: undefined
        };
        if (typeof value === 'function') {
          el[ctxKey].handler = value;
          on('clickoutside', el, value, {
            capture: modifiers.capture
          });
        }
      },
      updated(el, {
        value,
        modifiers
      }) {
        const ctx = el[ctxKey];
        if (typeof value === 'function') {
          if (ctx.handler) {
            if (ctx.handler !== value) {
              off('clickoutside', el, ctx.handler, {
                capture: modifiers.capture
              });
              ctx.handler = value;
              on('clickoutside', el, value, {
                capture: modifiers.capture
              });
            }
          } else {
            el[ctxKey].handler = value;
            on('clickoutside', el, value, {
              capture: modifiers.capture
            });
          }
        } else {
          if (ctx.handler) {
            off('clickoutside', el, ctx.handler, {
              capture: modifiers.capture
            });
            ctx.handler = undefined;
          }
        }
      },
      unmounted(el, {
        modifiers
      }) {
        const {
          handler
        } = el[ctxKey];
        if (handler) {
          off('clickoutside', el, handler, {
            capture: modifiers.capture
          });
        }
        el[ctxKey].handler = undefined;
      }
    };

    function warn$1(location, message) {
      console.error(`[vdirs/${location}]: ${message}`);
    }

    class ZIndexManager {
      constructor() {
        this.elementZIndex = new Map();
        this.nextZIndex = 2000;
      }
      get elementCount() {
        return this.elementZIndex.size;
      }
      ensureZIndex(el, zIndex) {
        const {
          elementZIndex
        } = this;
        if (zIndex !== undefined) {
          el.style.zIndex = `${zIndex}`;
          elementZIndex.delete(el);
          return;
        }
        const {
          nextZIndex
        } = this;
        if (elementZIndex.has(el)) {
          const currentZIndex = elementZIndex.get(el);
          if (currentZIndex + 1 === this.nextZIndex) return;
        }
        el.style.zIndex = `${nextZIndex}`;
        elementZIndex.set(el, nextZIndex);
        this.nextZIndex = nextZIndex + 1;
        this.squashState();
      }
      unregister(el, zIndex) {
        const {
          elementZIndex
        } = this;
        if (elementZIndex.has(el)) {
          elementZIndex.delete(el);
        } else if (zIndex === undefined) {
          warn$1('z-index-manager/unregister-element', 'Element not found when unregistering.');
        }
        this.squashState();
      }
      squashState() {
        const {
          elementCount
        } = this;
        if (!elementCount) {
          this.nextZIndex = 2000;
        }
        if (this.nextZIndex - elementCount > 2500) this.rearrange();
      }
      rearrange() {
        const elementZIndexPair = Array.from(this.elementZIndex.entries());
        elementZIndexPair.sort((pair1, pair2) => {
          return pair1[1] - pair2[1];
        });
        this.nextZIndex = 2000;
        elementZIndexPair.forEach(pair => {
          const el = pair[0];
          const zIndex = this.nextZIndex++;
          if (`${zIndex}` !== el.style.zIndex) el.style.zIndex = `${zIndex}`;
        });
      }
    }
    var zIndexManager = new ZIndexManager();

    const ctx = '@@ziContext';
    // We don't expect manually bound zindex should be changed
    const zindexable = {
      mounted(el, bindings) {
        const {
          value = {}
        } = bindings;
        const {
          zIndex,
          enabled
        } = value;
        el[ctx] = {
          enabled: !!enabled,
          initialized: false
        };
        if (enabled) {
          zIndexManager.ensureZIndex(el, zIndex);
          el[ctx].initialized = true;
        }
      },
      updated(el, bindings) {
        const {
          value = {}
        } = bindings;
        const {
          zIndex,
          enabled
        } = value;
        const cachedEnabled = el[ctx].enabled;
        if (enabled && !cachedEnabled) {
          zIndexManager.ensureZIndex(el, zIndex);
          el[ctx].initialized = true;
        }
        el[ctx].enabled = !!enabled;
      },
      unmounted(el, bindings) {
        if (!el[ctx].initialized) return;
        const {
          value = {}
        } = bindings;
        const {
          zIndex
        } = value;
        zIndexManager.unregister(el, zIndex);
      }
    };

    const ssrContextKey = '@css-render/vue3-ssr';
    function createStyleString(id, style) {
      return `<style cssr-id="${id}">\n${style}\n</style>`;
    }
    function ssrAdapter(id, style, ssrContext) {
      const {
        styles,
        ids
      } = ssrContext;
      // we need to impl other options to make it behaves the same as the client side
      if (ids.has(id)) return;
      if (styles !== null) {
        ids.add(id);
        styles.push(createStyleString(id, style));
      }
    }
    const isBrowser = typeof document !== 'undefined';
    function useSsrAdapter() {
      if (isBrowser) return undefined;
      const context = vue.inject(ssrContextKey, null);
      if (context === null) return undefined;
      return {
        adapter: (id, style) => ssrAdapter(id, style, context),
        context
      };
    }

    function warn(location, message) {
      console.error(`[vueuc/${location}]: ${message}`);
    }

    const {
      c
    } = CssRender();
    const cssrAnchorMetaName$1 = 'vueuc-style';

    function lowBit(n) {
      return n & -n;
    }
    class FinweckTree {
      /**
       * @param l length of the array
       * @param min min value of the array
       */
      constructor(l, min) {
        this.l = l;
        this.min = min;
        const ft = new Array(l + 1);
        for (let i = 0; i < l + 1; ++i) {
          ft[i] = 0;
        }
        this.ft = ft;
      }
      /**
       * Add arr[i] by n, start from 0
       * @param i the index of the element to be added
       * @param n the value to be added
       */
      add(i, n) {
        if (n === 0) return;
        const {
          l,
          ft
        } = this;
        i += 1;
        while (i <= l) {
          ft[i] += n;
          i += lowBit(i);
        }
      }
      /**
       * Get the value of index i
       * @param i index
       * @returns value of the index
       */
      get(i) {
        return this.sum(i + 1) - this.sum(i);
      }
      /**
       * Get the sum of first i elements
       * @param i count of head elements to be added
       * @returns the sum of first i elements
       */
      sum(i) {
        if (i === undefined) i = this.l;
        if (i <= 0) return 0;
        const {
          ft,
          min,
          l
        } = this;
        if (i > l) throw new Error('[FinweckTree.sum]: `i` is larger than length.');
        let ret = i * min;
        while (i > 0) {
          ret += ft[i];
          i -= lowBit(i);
        }
        return ret;
      }
      /**
       * Get the largest count of head elements whose sum are <= threshold
       * @param threshold
       * @returns the largest count of head elements whose sum are <= threshold
       */
      getBound(threshold) {
        let l = 0;
        let r = this.l;
        while (r > l) {
          const m = Math.floor((l + r) / 2);
          const sumM = this.sum(m);
          if (sumM > threshold) {
            r = m;
            continue;
          } else if (sumM < threshold) {
            if (l === m) {
              if (this.sum(l + 1) <= threshold) return l + 1;
              return m;
            }
            l = m;
          } else {
            return m;
          }
        }
        return l;
      }
    }

    function resolveTo(selector) {
      if (typeof selector === 'string') {
        return document.querySelector(selector);
      }
      return selector();
    }

    var LazyTeleport = vue.defineComponent({
      name: 'LazyTeleport',
      props: {
        to: {
          type: [String, Object],
          default: undefined
        },
        disabled: Boolean,
        show: {
          type: Boolean,
          required: true
        }
      },
      setup(props) {
        return {
          showTeleport: useFalseUntilTruthy(vue.toRef(props, 'show')),
          mergedTo: vue.computed(() => {
            const {
              to
            } = props;
            return to !== null && to !== void 0 ? to : 'body';
          })
        };
      },
      render() {
        return this.showTeleport ? this.disabled ? getSlot('lazy-teleport', this.$slots) : vue.h(vue.Teleport, {
          disabled: this.disabled,
          to: this.mergedTo
        }, getSlot('lazy-teleport', this.$slots)) : null;
      }
    });

    const oppositionPositions = {
      top: 'bottom',
      bottom: 'top',
      left: 'right',
      right: 'left'
    };
    const oppositeAligns = {
      start: 'end',
      center: 'center',
      end: 'start'
    };
    const propToCompare = {
      top: 'height',
      bottom: 'height',
      left: 'width',
      right: 'width'
    };
    const transformOrigins = {
      'bottom-start': 'top left',
      bottom: 'top center',
      'bottom-end': 'top right',
      'top-start': 'bottom left',
      top: 'bottom center',
      'top-end': 'bottom right',
      'right-start': 'top left',
      right: 'center left',
      'right-end': 'bottom left',
      'left-start': 'top right',
      left: 'center right',
      'left-end': 'bottom right'
    };
    const overlapTransformOrigin = {
      'bottom-start': 'bottom left',
      bottom: 'bottom center',
      'bottom-end': 'bottom right',
      'top-start': 'top left',
      top: 'top center',
      'top-end': 'top right',
      'right-start': 'top right',
      right: 'center right',
      'right-end': 'bottom right',
      'left-start': 'top left',
      left: 'center left',
      'left-end': 'bottom left'
    };
    const oppositeAlignCssPositionProps = {
      'bottom-start': 'right',
      'bottom-end': 'left',
      'top-start': 'right',
      'top-end': 'left',
      'right-start': 'bottom',
      'right-end': 'top',
      'left-start': 'bottom',
      'left-end': 'top'
    };
    const keepOffsetDirection = {
      top: true,
      bottom: false,
      left: true,
      right: false // left--
    };
    const cssPositionToOppositeAlign = {
      top: 'end',
      bottom: 'start',
      left: 'end',
      right: 'start'
    };
    function getPlacementAndOffsetOfFollower(placement, targetRect, followerRect, shift, flip, overlap) {
      if (!flip || overlap) {
        return {
          placement: placement,
          top: 0,
          left: 0
        };
      }
      const [position, align] = placement.split('-');
      let properAlign = align !== null && align !== void 0 ? align : 'center';
      let properOffset = {
        top: 0,
        left: 0
      };
      const deriveOffset = (oppositeAlignCssSizeProp, alignCssPositionProp, offsetVertically) => {
        let left = 0;
        let top = 0;
        const diff = followerRect[oppositeAlignCssSizeProp] - targetRect[alignCssPositionProp] - targetRect[oppositeAlignCssSizeProp];
        if (diff > 0 && shift) {
          if (offsetVertically) {
            //       screen border
            // |-----------------------------------------|
            // |                    | f  |               |
            // |                    | o  |               |
            // |                    | l  |               |
            // |                    | l  |----           |
            // |                    | o  |tar |          |
            // |                    | w  |get |          |
            // |                    | e  |    |          |
            // |                    | r  |----           |
            // |                     ----                |
            // |-----------------------------------------|
            top = keepOffsetDirection[alignCssPositionProp] ? diff : -diff;
          } else {
            //       screen border
            // |----------------------------------------|
            // |                                        |
            // |          ----------                    |
            // |          | target |                    |
            // |       ----------------------------------
            // |       |       follower                 |
            // |       ----------------------------------
            // |                                        |
            // |----------------------------------------|
            left = keepOffsetDirection[alignCssPositionProp] ? diff : -diff;
          }
        }
        return {
          left,
          top
        };
      };
      const offsetVertically = position === 'left' || position === 'right';
      // choose proper placement for non-center align
      if (properAlign !== 'center') {
        const oppositeAlignCssPositionProp = oppositeAlignCssPositionProps[placement];
        const currentAlignCssPositionProp = oppositionPositions[oppositeAlignCssPositionProp];
        const oppositeAlignCssSizeProp = propToCompare[oppositeAlignCssPositionProp];
        // if follower rect is larger than target rect in align direction
        // ----------[ target ]---------|
        // ----------[     follower     ]
        if (followerRect[oppositeAlignCssSizeProp] > targetRect[oppositeAlignCssSizeProp]) {
          if (
          // current space is not enough
          // ----------[ target ]---------|
          // -------[     follower        ]
          targetRect[oppositeAlignCssPositionProp] + targetRect[oppositeAlignCssSizeProp] < followerRect[oppositeAlignCssSizeProp]) {
            const followerOverTargetSize = (followerRect[oppositeAlignCssSizeProp] - targetRect[oppositeAlignCssSizeProp]) / 2;
            if (targetRect[oppositeAlignCssPositionProp] < followerOverTargetSize || targetRect[currentAlignCssPositionProp] < followerOverTargetSize) {
              // opposite align has larger space
              // -------[ target ]-----------|
              // -------[     follower     ]-|
              if (targetRect[oppositeAlignCssPositionProp] < targetRect[currentAlignCssPositionProp]) {
                properAlign = oppositeAligns[align];
                properOffset = deriveOffset(oppositeAlignCssSizeProp, currentAlignCssPositionProp, offsetVertically);
              } else {
                // ----------------[ target ]----|
                // --------[   follower     ]----|
                properOffset = deriveOffset(oppositeAlignCssSizeProp, oppositeAlignCssPositionProp, offsetVertically);
              }
            } else {
              // 'center' align is better
              // ------------[ target ]--------|
              // -------[       follower    ]--|
              properAlign = 'center';
            }
          }
        } else if (followerRect[oppositeAlignCssSizeProp] < targetRect[oppositeAlignCssSizeProp]) {
          // TODO: maybe center is better
          if (targetRect[currentAlignCssPositionProp] < 0 &&
          // opposite align has larger space
          // ------------[   target   ]
          // ----------------[follower]
          targetRect[oppositeAlignCssPositionProp] > targetRect[currentAlignCssPositionProp]) {
            properAlign = oppositeAligns[align];
          }
        }
      } else {
        const possibleAlternativeAlignCssPositionProp1 = position === 'bottom' || position === 'top' ? 'left' : 'top';
        const possibleAlternativeAlignCssPositionProp2 = oppositionPositions[possibleAlternativeAlignCssPositionProp1];
        const alternativeAlignCssSizeProp = propToCompare[possibleAlternativeAlignCssPositionProp1];
        const followerOverTargetSize = (followerRect[alternativeAlignCssSizeProp] - targetRect[alternativeAlignCssSizeProp]) / 2;
        if (
        // center is not enough
        // ----------- [ target ]--|
        // -------[     follower     ]
        targetRect[possibleAlternativeAlignCssPositionProp1] < followerOverTargetSize || targetRect[possibleAlternativeAlignCssPositionProp2] < followerOverTargetSize) {
          // alternative 2 position's space is larger
          if (targetRect[possibleAlternativeAlignCssPositionProp1] > targetRect[possibleAlternativeAlignCssPositionProp2]) {
            properAlign = cssPositionToOppositeAlign[possibleAlternativeAlignCssPositionProp1];
            properOffset = deriveOffset(alternativeAlignCssSizeProp, possibleAlternativeAlignCssPositionProp1, offsetVertically);
          } else {
            // alternative 1 position's space is larger
            properAlign = cssPositionToOppositeAlign[possibleAlternativeAlignCssPositionProp2];
            properOffset = deriveOffset(alternativeAlignCssSizeProp, possibleAlternativeAlignCssPositionProp2, offsetVertically);
          }
        }
      }
      let properPosition = position;
      if (
      // space is not enough
      targetRect[position] < followerRect[propToCompare[position]] &&
      // opposite position's space is larger
      targetRect[position] < targetRect[oppositionPositions[position]]) {
        properPosition = oppositionPositions[position];
      }
      return {
        placement: properAlign !== 'center' ? `${properPosition}-${properAlign}` : properPosition,
        left: properOffset.left,
        top: properOffset.top
      };
    }
    function getProperTransformOrigin(placement, overlap) {
      if (overlap) return overlapTransformOrigin[placement];
      return transformOrigins[placement];
    }
    // ------------
    // |  offset  |
    // |          |
    // | [target] |
    // |          |
    // ------------
    // TODO: refactor it to remove dup logic
    function getOffset$1(placement, offsetRect, targetRect, offsetTopToStandardPlacement, offsetLeftToStandardPlacement, overlap) {
      if (overlap) {
        switch (placement) {
          case 'bottom-start':
            return {
              top: `${Math.round(targetRect.top - offsetRect.top + targetRect.height)}px`,
              left: `${Math.round(targetRect.left - offsetRect.left)}px`,
              transform: 'translateY(-100%)'
            };
          case 'bottom-end':
            return {
              top: `${Math.round(targetRect.top - offsetRect.top + targetRect.height)}px`,
              left: `${Math.round(targetRect.left - offsetRect.left + targetRect.width)}px`,
              transform: 'translateX(-100%) translateY(-100%)'
            };
          case 'top-start':
            return {
              top: `${Math.round(targetRect.top - offsetRect.top)}px`,
              left: `${Math.round(targetRect.left - offsetRect.left)}px`,
              transform: ''
            };
          case 'top-end':
            return {
              top: `${Math.round(targetRect.top - offsetRect.top)}px`,
              left: `${Math.round(targetRect.left - offsetRect.left + targetRect.width)}px`,
              transform: 'translateX(-100%)'
            };
          case 'right-start':
            return {
              top: `${Math.round(targetRect.top - offsetRect.top)}px`,
              left: `${Math.round(targetRect.left - offsetRect.left + targetRect.width)}px`,
              transform: 'translateX(-100%)'
            };
          case 'right-end':
            return {
              top: `${Math.round(targetRect.top - offsetRect.top + targetRect.height)}px`,
              left: `${Math.round(targetRect.left - offsetRect.left + targetRect.width)}px`,
              transform: 'translateX(-100%) translateY(-100%)'
            };
          case 'left-start':
            return {
              top: `${Math.round(targetRect.top - offsetRect.top)}px`,
              left: `${Math.round(targetRect.left - offsetRect.left)}px`,
              transform: ''
            };
          case 'left-end':
            return {
              top: `${Math.round(targetRect.top - offsetRect.top + targetRect.height)}px`,
              left: `${Math.round(targetRect.left - offsetRect.left)}px`,
              transform: 'translateY(-100%)'
            };
          case 'top':
            return {
              top: `${Math.round(targetRect.top - offsetRect.top)}px`,
              left: `${Math.round(targetRect.left - offsetRect.left + targetRect.width / 2)}px`,
              transform: 'translateX(-50%)'
            };
          case 'right':
            return {
              top: `${Math.round(targetRect.top - offsetRect.top + targetRect.height / 2)}px`,
              left: `${Math.round(targetRect.left - offsetRect.left + targetRect.width)}px`,
              transform: 'translateX(-100%) translateY(-50%)'
            };
          case 'left':
            return {
              top: `${Math.round(targetRect.top - offsetRect.top + targetRect.height / 2)}px`,
              left: `${Math.round(targetRect.left - offsetRect.left)}px`,
              transform: 'translateY(-50%)'
            };
          case 'bottom':
          default:
            return {
              top: `${Math.round(targetRect.top - offsetRect.top + targetRect.height)}px`,
              left: `${Math.round(targetRect.left - offsetRect.left + targetRect.width / 2)}px`,
              transform: 'translateX(-50%) translateY(-100%)'
            };
        }
      }
      switch (placement) {
        case 'bottom-start':
          return {
            top: `${Math.round(targetRect.top - offsetRect.top + targetRect.height + offsetTopToStandardPlacement)}px`,
            left: `${Math.round(targetRect.left - offsetRect.left + offsetLeftToStandardPlacement)}px`,
            transform: ''
          };
        case 'bottom-end':
          return {
            top: `${Math.round(targetRect.top - offsetRect.top + targetRect.height + offsetTopToStandardPlacement)}px`,
            left: `${Math.round(targetRect.left - offsetRect.left + targetRect.width + offsetLeftToStandardPlacement)}px`,
            transform: 'translateX(-100%)'
          };
        case 'top-start':
          return {
            top: `${Math.round(targetRect.top - offsetRect.top + offsetTopToStandardPlacement)}px`,
            left: `${Math.round(targetRect.left - offsetRect.left + offsetLeftToStandardPlacement)}px`,
            transform: 'translateY(-100%)'
          };
        case 'top-end':
          return {
            top: `${Math.round(targetRect.top - offsetRect.top + offsetTopToStandardPlacement)}px`,
            left: `${Math.round(targetRect.left - offsetRect.left + targetRect.width + offsetLeftToStandardPlacement)}px`,
            transform: 'translateX(-100%) translateY(-100%)'
          };
        case 'right-start':
          return {
            top: `${Math.round(targetRect.top - offsetRect.top + offsetTopToStandardPlacement)}px`,
            left: `${Math.round(targetRect.left - offsetRect.left + targetRect.width + offsetLeftToStandardPlacement)}px`,
            transform: ''
          };
        case 'right-end':
          return {
            top: `${Math.round(targetRect.top - offsetRect.top + targetRect.height + offsetTopToStandardPlacement)}px`,
            left: `${Math.round(targetRect.left - offsetRect.left + targetRect.width + offsetLeftToStandardPlacement)}px`,
            transform: 'translateY(-100%)'
          };
        case 'left-start':
          return {
            top: `${Math.round(targetRect.top - offsetRect.top + offsetTopToStandardPlacement)}px`,
            left: `${Math.round(targetRect.left - offsetRect.left + offsetLeftToStandardPlacement)}px`,
            transform: 'translateX(-100%)'
          };
        case 'left-end':
          return {
            top: `${Math.round(targetRect.top - offsetRect.top + targetRect.height + offsetTopToStandardPlacement)}px`,
            left: `${Math.round(targetRect.left - offsetRect.left + offsetLeftToStandardPlacement)}px`,
            transform: 'translateX(-100%) translateY(-100%)'
          };
        case 'top':
          return {
            top: `${Math.round(targetRect.top - offsetRect.top + offsetTopToStandardPlacement)}px`,
            left: `${Math.round(targetRect.left - offsetRect.left + targetRect.width / 2 + offsetLeftToStandardPlacement)}px`,
            transform: 'translateY(-100%) translateX(-50%)'
          };
        case 'right':
          return {
            top: `${Math.round(targetRect.top - offsetRect.top + targetRect.height / 2 + offsetTopToStandardPlacement)}px`,
            left: `${Math.round(targetRect.left - offsetRect.left + targetRect.width + offsetLeftToStandardPlacement)}px`,
            transform: 'translateY(-50%)'
          };
        case 'left':
          return {
            top: `${Math.round(targetRect.top - offsetRect.top + targetRect.height / 2 + offsetTopToStandardPlacement)}px`,
            left: `${Math.round(targetRect.left - offsetRect.left + offsetLeftToStandardPlacement)}px`,
            transform: 'translateY(-50%) translateX(-100%)'
          };
        case 'bottom':
        default:
          return {
            top: `${Math.round(targetRect.top - offsetRect.top + targetRect.height + offsetTopToStandardPlacement)}px`,
            left: `${Math.round(targetRect.left - offsetRect.left + targetRect.width / 2 + offsetLeftToStandardPlacement)}px`,
            transform: 'translateX(-50%)'
          };
      }
    }

    /* eslint-disable @typescript-eslint/no-non-null-assertion */
    const style$1F = c([c('.v-binder-follower-container', {
      position: 'absolute',
      left: '0',
      right: '0',
      top: '0',
      height: '0',
      pointerEvents: 'none',
      zIndex: 'auto'
    }), c('.v-binder-follower-content', {
      position: 'absolute',
      zIndex: 'auto'
    }, [c('> *', {
      pointerEvents: 'all'
    })])]);
    var VFollower = vue.defineComponent({
      name: 'Follower',
      inheritAttrs: false,
      props: {
        show: Boolean,
        enabled: {
          type: Boolean,
          default: undefined
        },
        placement: {
          type: String,
          default: 'bottom'
        },
        syncTrigger: {
          type: Array,
          default: ['resize', 'scroll']
        },
        to: [String, Object],
        flip: {
          type: Boolean,
          default: true
        },
        internalShift: Boolean,
        x: Number,
        y: Number,
        width: String,
        minWidth: String,
        containerClass: String,
        teleportDisabled: Boolean,
        zindexable: {
          type: Boolean,
          default: true
        },
        zIndex: Number,
        overlap: Boolean
      },
      setup(props) {
        const VBinder = vue.inject('VBinder');
        const mergedEnabledRef = useMemo(() => {
          return props.enabled !== undefined ? props.enabled : props.show;
        });
        const followerRef = vue.ref(null);
        const offsetContainerRef = vue.ref(null);
        const ensureListeners = () => {
          const {
            syncTrigger
          } = props;
          if (syncTrigger.includes('scroll')) {
            VBinder.addScrollListener(syncPosition);
          }
          if (syncTrigger.includes('resize')) {
            VBinder.addResizeListener(syncPosition);
          }
        };
        const removeListeners = () => {
          VBinder.removeScrollListener(syncPosition);
          VBinder.removeResizeListener(syncPosition);
        };
        vue.onMounted(() => {
          if (mergedEnabledRef.value) {
            syncPosition();
            ensureListeners();
          }
        });
        const ssrAdapter = useSsrAdapter();
        style$1F.mount({
          id: 'vueuc/binder',
          head: true,
          anchorMetaName: cssrAnchorMetaName$1,
          ssr: ssrAdapter
        });
        vue.onBeforeUnmount(() => {
          removeListeners();
        });
        onFontsReady(() => {
          if (mergedEnabledRef.value) {
            syncPosition();
          }
        });
        const syncPosition = () => {
          if (!mergedEnabledRef.value) {
            return;
          }
          const follower = followerRef.value;
          // sometimes watched props change before its dom is ready
          // for example: show=false, x=undefined, y=undefined
          //              show=true,  x=0,         y=0
          // will cause error
          // I may optimize the watch start point later
          if (follower === null) return;
          const target = VBinder.targetRef;
          const {
            x,
            y,
            overlap
          } = props;
          const targetRect = x !== undefined && y !== undefined ? getPointRect(x, y) : getRect$1(target);
          follower.style.setProperty('--v-target-width', `${Math.round(targetRect.width)}px`);
          follower.style.setProperty('--v-target-height', `${Math.round(targetRect.height)}px`);
          const {
            width,
            minWidth,
            placement,
            internalShift,
            flip
          } = props;
          follower.setAttribute('v-placement', placement);
          if (overlap) {
            follower.setAttribute('v-overlap', '');
          } else {
            follower.removeAttribute('v-overlap');
          }
          const {
            style
          } = follower;
          if (width === 'target') {
            style.width = `${targetRect.width}px`;
          } else if (width !== undefined) {
            style.width = width;
          } else {
            style.width = '';
          }
          if (minWidth === 'target') {
            style.minWidth = `${targetRect.width}px`;
          } else if (minWidth !== undefined) {
            style.minWidth = minWidth;
          } else {
            style.minWidth = '';
          }
          const followerRect = getRect$1(follower);
          const offsetContainerRect = getRect$1(offsetContainerRef.value);
          const {
            left: offsetLeftToStandardPlacement,
            top: offsetTopToStandardPlacement,
            placement: properPlacement
          } = getPlacementAndOffsetOfFollower(placement, targetRect, followerRect, internalShift, flip, overlap);
          const properTransformOrigin = getProperTransformOrigin(properPlacement, overlap);
          const {
            left,
            top,
            transform
          } = getOffset$1(properPlacement, offsetContainerRect, targetRect, offsetTopToStandardPlacement, offsetLeftToStandardPlacement, overlap);
          // we assume that the content size doesn't change after flip,
          // nor we need to make sync logic more complex
          follower.setAttribute('v-placement', properPlacement);
          follower.style.setProperty('--v-offset-left', `${Math.round(offsetLeftToStandardPlacement)}px`);
          follower.style.setProperty('--v-offset-top', `${Math.round(offsetTopToStandardPlacement)}px`);
          follower.style.transform = `translateX(${left}) translateY(${top}) ${transform}`;
          follower.style.setProperty('--v-transform-origin', properTransformOrigin);
          follower.style.transformOrigin = properTransformOrigin;
        };
        vue.watch(mergedEnabledRef, value => {
          if (value) {
            ensureListeners();
            syncOnNextTick();
          } else {
            removeListeners();
          }
        });
        const syncOnNextTick = () => {
          vue.nextTick().then(syncPosition).catch(e => console.error(e));
        };
        ['placement', 'x', 'y', 'internalShift', 'flip', 'width', 'overlap', 'minWidth'].forEach(prop => {
          vue.watch(vue.toRef(props, prop), syncPosition);
        });
        ['teleportDisabled'].forEach(prop => {
          vue.watch(vue.toRef(props, prop), syncOnNextTick);
        });
        vue.watch(vue.toRef(props, 'syncTrigger'), value => {
          if (!value.includes('resize')) {
            VBinder.removeResizeListener(syncPosition);
          } else {
            VBinder.addResizeListener(syncPosition);
          }
          if (!value.includes('scroll')) {
            VBinder.removeScrollListener(syncPosition);
          } else {
            VBinder.addScrollListener(syncPosition);
          }
        });
        const isMountedRef = isMounted();
        const mergedToRef = useMemo(() => {
          const {
            to
          } = props;
          if (to !== undefined) return to;
          if (isMountedRef.value) {
            // TODO: find proper container
            return undefined;
          }
          return undefined;
        });
        return {
          VBinder,
          mergedEnabled: mergedEnabledRef,
          offsetContainerRef,
          followerRef,
          mergedTo: mergedToRef,
          syncPosition
        };
      },
      render() {
        return vue.h(LazyTeleport, {
          show: this.show,
          to: this.mergedTo,
          disabled: this.teleportDisabled
        }, {
          default: () => {
            var _a, _b;
            const vNode = vue.h('div', {
              class: ['v-binder-follower-container', this.containerClass],
              ref: 'offsetContainerRef'
            }, [vue.h('div', {
              class: 'v-binder-follower-content',
              ref: 'followerRef'
            }, (_b = (_a = this.$slots).default) === null || _b === void 0 ? void 0 : _b.call(_a))]);
            if (this.zindexable) {
              return vue.withDirectives(vNode, [[zindexable, {
                enabled: this.mergedEnabled,
                zIndex: this.zIndex
              }]]);
            }
            return vNode;
          }
        });
      }
    });

    var resizeObservers = [];

    var hasActiveObservations = function () {
      return resizeObservers.some(function (ro) {
        return ro.activeTargets.length > 0;
      });
    };

    var hasSkippedObservations = function () {
      return resizeObservers.some(function (ro) {
        return ro.skippedTargets.length > 0;
      });
    };

    var msg = 'ResizeObserver loop completed with undelivered notifications.';
    var deliverResizeLoopError = function () {
      var event;
      if (typeof ErrorEvent === 'function') {
        event = new ErrorEvent('error', {
          message: msg
        });
      } else {
        event = document.createEvent('Event');
        event.initEvent('error', false, false);
        event.message = msg;
      }
      window.dispatchEvent(event);
    };

    var ResizeObserverBoxOptions;
    (function (ResizeObserverBoxOptions) {
      ResizeObserverBoxOptions["BORDER_BOX"] = "border-box";
      ResizeObserverBoxOptions["CONTENT_BOX"] = "content-box";
      ResizeObserverBoxOptions["DEVICE_PIXEL_CONTENT_BOX"] = "device-pixel-content-box";
    })(ResizeObserverBoxOptions || (ResizeObserverBoxOptions = {}));

    var freeze = function (obj) {
      return Object.freeze(obj);
    };

    var ResizeObserverSize = function () {
      function ResizeObserverSize(inlineSize, blockSize) {
        this.inlineSize = inlineSize;
        this.blockSize = blockSize;
        freeze(this);
      }
      return ResizeObserverSize;
    }();

    var DOMRectReadOnly = function () {
      function DOMRectReadOnly(x, y, width, height) {
        this.x = x;
        this.y = y;
        this.width = width;
        this.height = height;
        this.top = this.y;
        this.left = this.x;
        this.bottom = this.top + this.height;
        this.right = this.left + this.width;
        return freeze(this);
      }
      DOMRectReadOnly.prototype.toJSON = function () {
        var _a = this,
          x = _a.x,
          y = _a.y,
          top = _a.top,
          right = _a.right,
          bottom = _a.bottom,
          left = _a.left,
          width = _a.width,
          height = _a.height;
        return {
          x: x,
          y: y,
          top: top,
          right: right,
          bottom: bottom,
          left: left,
          width: width,
          height: height
        };
      };
      DOMRectReadOnly.fromRect = function (rectangle) {
        return new DOMRectReadOnly(rectangle.x, rectangle.y, rectangle.width, rectangle.height);
      };
      return DOMRectReadOnly;
    }();

    var isSVG = function (target) {
      return target instanceof SVGElement && 'getBBox' in target;
    };
    var isHidden = function (target) {
      if (isSVG(target)) {
        var _a = target.getBBox(),
          width = _a.width,
          height = _a.height;
        return !width && !height;
      }
      var _b = target,
        offsetWidth = _b.offsetWidth,
        offsetHeight = _b.offsetHeight;
      return !(offsetWidth || offsetHeight || target.getClientRects().length);
    };
    var isElement = function (obj) {
      var _a;
      if (obj instanceof Element) {
        return true;
      }
      var scope = (_a = obj === null || obj === void 0 ? void 0 : obj.ownerDocument) === null || _a === void 0 ? void 0 : _a.defaultView;
      return !!(scope && obj instanceof scope.Element);
    };
    var isReplacedElement = function (target) {
      switch (target.tagName) {
        case 'INPUT':
          if (target.type !== 'image') {
            break;
          }
        case 'VIDEO':
        case 'AUDIO':
        case 'EMBED':
        case 'OBJECT':
        case 'CANVAS':
        case 'IFRAME':
        case 'IMG':
          return true;
      }
      return false;
    };

    var global$1 = typeof window !== 'undefined' ? window : {};

    var cache = new WeakMap();
    var scrollRegexp = /auto|scroll/;
    var verticalRegexp = /^tb|vertical/;
    var IE = /msie|trident/i.test(global$1.navigator && global$1.navigator.userAgent);
    var parseDimension = function (pixel) {
      return parseFloat(pixel || '0');
    };
    var size = function (inlineSize, blockSize, switchSizes) {
      if (inlineSize === void 0) {
        inlineSize = 0;
      }
      if (blockSize === void 0) {
        blockSize = 0;
      }
      if (switchSizes === void 0) {
        switchSizes = false;
      }
      return new ResizeObserverSize((switchSizes ? blockSize : inlineSize) || 0, (switchSizes ? inlineSize : blockSize) || 0);
    };
    var zeroBoxes = freeze({
      devicePixelContentBoxSize: size(),
      borderBoxSize: size(),
      contentBoxSize: size(),
      contentRect: new DOMRectReadOnly(0, 0, 0, 0)
    });
    var calculateBoxSizes = function (target, forceRecalculation) {
      if (forceRecalculation === void 0) {
        forceRecalculation = false;
      }
      if (cache.has(target) && !forceRecalculation) {
        return cache.get(target);
      }
      if (isHidden(target)) {
        cache.set(target, zeroBoxes);
        return zeroBoxes;
      }
      var cs = getComputedStyle(target);
      var svg = isSVG(target) && target.ownerSVGElement && target.getBBox();
      var removePadding = !IE && cs.boxSizing === 'border-box';
      var switchSizes = verticalRegexp.test(cs.writingMode || '');
      var canScrollVertically = !svg && scrollRegexp.test(cs.overflowY || '');
      var canScrollHorizontally = !svg && scrollRegexp.test(cs.overflowX || '');
      var paddingTop = svg ? 0 : parseDimension(cs.paddingTop);
      var paddingRight = svg ? 0 : parseDimension(cs.paddingRight);
      var paddingBottom = svg ? 0 : parseDimension(cs.paddingBottom);
      var paddingLeft = svg ? 0 : parseDimension(cs.paddingLeft);
      var borderTop = svg ? 0 : parseDimension(cs.borderTopWidth);
      var borderRight = svg ? 0 : parseDimension(cs.borderRightWidth);
      var borderBottom = svg ? 0 : parseDimension(cs.borderBottomWidth);
      var borderLeft = svg ? 0 : parseDimension(cs.borderLeftWidth);
      var horizontalPadding = paddingLeft + paddingRight;
      var verticalPadding = paddingTop + paddingBottom;
      var horizontalBorderArea = borderLeft + borderRight;
      var verticalBorderArea = borderTop + borderBottom;
      var horizontalScrollbarThickness = !canScrollHorizontally ? 0 : target.offsetHeight - verticalBorderArea - target.clientHeight;
      var verticalScrollbarThickness = !canScrollVertically ? 0 : target.offsetWidth - horizontalBorderArea - target.clientWidth;
      var widthReduction = removePadding ? horizontalPadding + horizontalBorderArea : 0;
      var heightReduction = removePadding ? verticalPadding + verticalBorderArea : 0;
      var contentWidth = svg ? svg.width : parseDimension(cs.width) - widthReduction - verticalScrollbarThickness;
      var contentHeight = svg ? svg.height : parseDimension(cs.height) - heightReduction - horizontalScrollbarThickness;
      var borderBoxWidth = contentWidth + horizontalPadding + verticalScrollbarThickness + horizontalBorderArea;
      var borderBoxHeight = contentHeight + verticalPadding + horizontalScrollbarThickness + verticalBorderArea;
      var boxes = freeze({
        devicePixelContentBoxSize: size(Math.round(contentWidth * devicePixelRatio), Math.round(contentHeight * devicePixelRatio), switchSizes),
        borderBoxSize: size(borderBoxWidth, borderBoxHeight, switchSizes),
        contentBoxSize: size(contentWidth, contentHeight, switchSizes),
        contentRect: new DOMRectReadOnly(paddingLeft, paddingTop, contentWidth, contentHeight)
      });
      cache.set(target, boxes);
      return boxes;
    };
    var calculateBoxSize = function (target, observedBox, forceRecalculation) {
      var _a = calculateBoxSizes(target, forceRecalculation),
        borderBoxSize = _a.borderBoxSize,
        contentBoxSize = _a.contentBoxSize,
        devicePixelContentBoxSize = _a.devicePixelContentBoxSize;
      switch (observedBox) {
        case ResizeObserverBoxOptions.DEVICE_PIXEL_CONTENT_BOX:
          return devicePixelContentBoxSize;
        case ResizeObserverBoxOptions.BORDER_BOX:
          return borderBoxSize;
        default:
          return contentBoxSize;
      }
    };

    var ResizeObserverEntry = function () {
      function ResizeObserverEntry(target) {
        var boxes = calculateBoxSizes(target);
        this.target = target;
        this.contentRect = boxes.contentRect;
        this.borderBoxSize = freeze([boxes.borderBoxSize]);
        this.contentBoxSize = freeze([boxes.contentBoxSize]);
        this.devicePixelContentBoxSize = freeze([boxes.devicePixelContentBoxSize]);
      }
      return ResizeObserverEntry;
    }();

    var calculateDepthForNode = function (node) {
      if (isHidden(node)) {
        return Infinity;
      }
      var depth = 0;
      var parent = node.parentNode;
      while (parent) {
        depth += 1;
        parent = parent.parentNode;
      }
      return depth;
    };

    var broadcastActiveObservations = function () {
      var shallowestDepth = Infinity;
      var callbacks = [];
      resizeObservers.forEach(function processObserver(ro) {
        if (ro.activeTargets.length === 0) {
          return;
        }
        var entries = [];
        ro.activeTargets.forEach(function processTarget(ot) {
          var entry = new ResizeObserverEntry(ot.target);
          var targetDepth = calculateDepthForNode(ot.target);
          entries.push(entry);
          ot.lastReportedSize = calculateBoxSize(ot.target, ot.observedBox);
          if (targetDepth < shallowestDepth) {
            shallowestDepth = targetDepth;
          }
        });
        callbacks.push(function resizeObserverCallback() {
          ro.callback.call(ro.observer, entries, ro.observer);
        });
        ro.activeTargets.splice(0, ro.activeTargets.length);
      });
      for (var _i = 0, callbacks_1 = callbacks; _i < callbacks_1.length; _i++) {
        var callback = callbacks_1[_i];
        callback();
      }
      return shallowestDepth;
    };

    var gatherActiveObservationsAtDepth = function (depth) {
      resizeObservers.forEach(function processObserver(ro) {
        ro.activeTargets.splice(0, ro.activeTargets.length);
        ro.skippedTargets.splice(0, ro.skippedTargets.length);
        ro.observationTargets.forEach(function processTarget(ot) {
          if (ot.isActive()) {
            if (calculateDepthForNode(ot.target) > depth) {
              ro.activeTargets.push(ot);
            } else {
              ro.skippedTargets.push(ot);
            }
          }
        });
      });
    };

    var process$1 = function () {
      var depth = 0;
      gatherActiveObservationsAtDepth(depth);
      while (hasActiveObservations()) {
        depth = broadcastActiveObservations();
        gatherActiveObservationsAtDepth(depth);
      }
      if (hasSkippedObservations()) {
        deliverResizeLoopError();
      }
      return depth > 0;
    };

    var trigger;
    var callbacks = [];
    var notify = function () {
      return callbacks.splice(0).forEach(function (cb) {
        return cb();
      });
    };
    var queueMicroTask = function (callback) {
      if (!trigger) {
        var toggle_1 = 0;
        var el_1 = document.createTextNode('');
        var config = {
          characterData: true
        };
        new MutationObserver(function () {
          return notify();
        }).observe(el_1, config);
        trigger = function () {
          el_1.textContent = "".concat(toggle_1 ? toggle_1-- : toggle_1++);
        };
      }
      callbacks.push(callback);
      trigger();
    };

    var queueResizeObserver = function (cb) {
      queueMicroTask(function ResizeObserver() {
        requestAnimationFrame(cb);
      });
    };

    var watching = 0;
    var isWatching = function () {
      return !!watching;
    };
    var CATCH_PERIOD = 250;
    var observerConfig = {
      attributes: true,
      characterData: true,
      childList: true,
      subtree: true
    };
    var events = ['resize', 'load', 'transitionend', 'animationend', 'animationstart', 'animationiteration', 'keyup', 'keydown', 'mouseup', 'mousedown', 'mouseover', 'mouseout', 'blur', 'focus'];
    var time$1 = function (timeout) {
      if (timeout === void 0) {
        timeout = 0;
      }
      return Date.now() + timeout;
    };
    var scheduled = false;
    var Scheduler = function () {
      function Scheduler() {
        var _this = this;
        this.stopped = true;
        this.listener = function () {
          return _this.schedule();
        };
      }
      Scheduler.prototype.run = function (timeout) {
        var _this = this;
        if (timeout === void 0) {
          timeout = CATCH_PERIOD;
        }
        if (scheduled) {
          return;
        }
        scheduled = true;
        var until = time$1(timeout);
        queueResizeObserver(function () {
          var elementsHaveResized = false;
          try {
            elementsHaveResized = process$1();
          } finally {
            scheduled = false;
            timeout = until - time$1();
            if (!isWatching()) {
              return;
            }
            if (elementsHaveResized) {
              _this.run(1000);
            } else if (timeout > 0) {
              _this.run(timeout);
            } else {
              _this.start();
            }
          }
        });
      };
      Scheduler.prototype.schedule = function () {
        this.stop();
        this.run();
      };
      Scheduler.prototype.observe = function () {
        var _this = this;
        var cb = function () {
          return _this.observer && _this.observer.observe(document.body, observerConfig);
        };
        document.body ? cb() : global$1.addEventListener('DOMContentLoaded', cb);
      };
      Scheduler.prototype.start = function () {
        var _this = this;
        if (this.stopped) {
          this.stopped = false;
          this.observer = new MutationObserver(this.listener);
          this.observe();
          events.forEach(function (name) {
            return global$1.addEventListener(name, _this.listener, true);
          });
        }
      };
      Scheduler.prototype.stop = function () {
        var _this = this;
        if (!this.stopped) {
          this.observer && this.observer.disconnect();
          events.forEach(function (name) {
            return global$1.removeEventListener(name, _this.listener, true);
          });
          this.stopped = true;
        }
      };
      return Scheduler;
    }();
    var scheduler = new Scheduler();
    var updateCount = function (n) {
      !watching && n > 0 && scheduler.start();
      watching += n;
      !watching && scheduler.stop();
    };

    var skipNotifyOnElement = function (target) {
      return !isSVG(target) && !isReplacedElement(target) && getComputedStyle(target).display === 'inline';
    };
    var ResizeObservation = function () {
      function ResizeObservation(target, observedBox) {
        this.target = target;
        this.observedBox = observedBox || ResizeObserverBoxOptions.CONTENT_BOX;
        this.lastReportedSize = {
          inlineSize: 0,
          blockSize: 0
        };
      }
      ResizeObservation.prototype.isActive = function () {
        var size = calculateBoxSize(this.target, this.observedBox, true);
        if (skipNotifyOnElement(this.target)) {
          this.lastReportedSize = size;
        }
        if (this.lastReportedSize.inlineSize !== size.inlineSize || this.lastReportedSize.blockSize !== size.blockSize) {
          return true;
        }
        return false;
      };
      return ResizeObservation;
    }();

    var ResizeObserverDetail = function () {
      function ResizeObserverDetail(resizeObserver, callback) {
        this.activeTargets = [];
        this.skippedTargets = [];
        this.observationTargets = [];
        this.observer = resizeObserver;
        this.callback = callback;
      }
      return ResizeObserverDetail;
    }();

    var observerMap = new WeakMap();
    var getObservationIndex = function (observationTargets, target) {
      for (var i = 0; i < observationTargets.length; i += 1) {
        if (observationTargets[i].target === target) {
          return i;
        }
      }
      return -1;
    };
    var ResizeObserverController = function () {
      function ResizeObserverController() {}
      ResizeObserverController.connect = function (resizeObserver, callback) {
        var detail = new ResizeObserverDetail(resizeObserver, callback);
        observerMap.set(resizeObserver, detail);
      };
      ResizeObserverController.observe = function (resizeObserver, target, options) {
        var detail = observerMap.get(resizeObserver);
        var firstObservation = detail.observationTargets.length === 0;
        if (getObservationIndex(detail.observationTargets, target) < 0) {
          firstObservation && resizeObservers.push(detail);
          detail.observationTargets.push(new ResizeObservation(target, options && options.box));
          updateCount(1);
          scheduler.schedule();
        }
      };
      ResizeObserverController.unobserve = function (resizeObserver, target) {
        var detail = observerMap.get(resizeObserver);
        var index = getObservationIndex(detail.observationTargets, target);
        var lastObservation = detail.observationTargets.length === 1;
        if (index >= 0) {
          lastObservation && resizeObservers.splice(resizeObservers.indexOf(detail), 1);
          detail.observationTargets.splice(index, 1);
          updateCount(-1);
        }
      };
      ResizeObserverController.disconnect = function (resizeObserver) {
        var _this = this;
        var detail = observerMap.get(resizeObserver);
        detail.observationTargets.slice().forEach(function (ot) {
          return _this.unobserve(resizeObserver, ot.target);
        });
        detail.activeTargets.splice(0, detail.activeTargets.length);
      };
      return ResizeObserverController;
    }();

    var ResizeObserver = function () {
      function ResizeObserver(callback) {
        if (arguments.length === 0) {
          throw new TypeError("Failed to construct 'ResizeObserver': 1 argument required, but only 0 present.");
        }
        if (typeof callback !== 'function') {
          throw new TypeError("Failed to construct 'ResizeObserver': The callback provided as parameter 1 is not a function.");
        }
        ResizeObserverController.connect(this, callback);
      }
      ResizeObserver.prototype.observe = function (target, options) {
        if (arguments.length === 0) {
          throw new TypeError("Failed to execute 'observe' on 'ResizeObserver': 1 argument required, but only 0 present.");
        }
        if (!isElement(target)) {
          throw new TypeError("Failed to execute 'observe' on 'ResizeObserver': parameter 1 is not of type 'Element");
        }
        ResizeObserverController.observe(this, target, options);
      };
      ResizeObserver.prototype.unobserve = function (target) {
        if (arguments.length === 0) {
          throw new TypeError("Failed to execute 'unobserve' on 'ResizeObserver': 1 argument required, but only 0 present.");
        }
        if (!isElement(target)) {
          throw new TypeError("Failed to execute 'unobserve' on 'ResizeObserver': parameter 1 is not of type 'Element");
        }
        ResizeObserverController.unobserve(this, target);
      };
      ResizeObserver.prototype.disconnect = function () {
        ResizeObserverController.disconnect(this);
      };
      ResizeObserver.toString = function () {
        return 'function ResizeObserver () { [polyfill code] }';
      };
      return ResizeObserver;
    }();

    class ResizeObserverDelegate {
      constructor() {
        this.handleResize = this.handleResize.bind(this);
        this.observer = new (typeof window !== 'undefined' && window.ResizeObserver || ResizeObserver)(this.handleResize);
        this.elHandlersMap = new Map();
      }
      handleResize(entries) {
        for (const entry of entries) {
          const handler = this.elHandlersMap.get(entry.target);
          if (handler !== undefined) {
            handler(entry);
          }
        }
      }
      registerHandler(el, handler) {
        this.elHandlersMap.set(el, handler);
        this.observer.observe(el);
      }
      unregisterHandler(el) {
        if (!this.elHandlersMap.has(el)) {
          return;
        }
        this.elHandlersMap.delete(el);
        this.observer.unobserve(el);
      }
    }
    var resizeObserverManager = new ResizeObserverDelegate();

    var VResizeObserver = vue.defineComponent({
      name: 'ResizeObserver',
      props: {
        onResize: Function
      },
      setup(props) {
        let registered = false;
        // eslint-disable-next-line @typescript-eslint/no-non-null-assertion
        const proxy = vue.getCurrentInstance().proxy;
        function handleResize(entry) {
          const {
            onResize
          } = props;
          if (onResize !== undefined) onResize(entry);
        }
        vue.onMounted(() => {
          const el = proxy.$el;
          if (el === undefined) {
            warn('resize-observer', '$el does not exist.');
            return;
          }
          if (el.nextElementSibling !== el.nextSibling) {
            if (el.nodeType === 3 && el.nodeValue !== '') {
              warn('resize-observer', '$el can not be observed (it may be a text node).');
              return;
            }
          }
          if (el.nextElementSibling !== null) {
            resizeObserverManager.registerHandler(el.nextElementSibling, handleResize);
            registered = true;
          }
        });
        vue.onBeforeUnmount(() => {
          if (registered) {
            resizeObserverManager.unregisterHandler(proxy.$el.nextElementSibling);
          }
        });
      },
      render() {
        return vue.renderSlot(this.$slots, 'default');
      }
    });

    let maybeTouch;
    function ensureMaybeTouch() {
      if (typeof document === 'undefined') return false;
      if (maybeTouch === undefined) {
        if ('matchMedia' in window) {
          maybeTouch = window.matchMedia('(pointer:coarse)').matches;
        } else {
          maybeTouch = false;
        }
      }
      return maybeTouch;
    }
    let wheelScale;
    function ensureWheelScale() {
      if (typeof document === 'undefined') return 1;
      if (wheelScale === undefined) {
        wheelScale = 'chrome' in window ? window.devicePixelRatio : 1;
      }
      return wheelScale;
    }

    const xScrollInjextionKey = 'VVirtualListXScroll';

    function setupXScroll({
      columnsRef,
      renderColRef,
      renderItemWithColsRef
    }) {
      const listWidthRef = vue.ref(0);
      const scrollLeftRef = vue.ref(0);
      const xFinweckTreeRef = vue.computed(() => {
        const columns = columnsRef.value;
        if (columns.length === 0) {
          return null;
        }
        const ft = new FinweckTree(columns.length, 0);
        columns.forEach((column, index) => {
          ft.add(index, column.width);
        });
        return ft;
      });
      const startIndexRef = useMemo(() => {
        const xFinweckTree = xFinweckTreeRef.value;
        if (xFinweckTree !== null) {
          return Math.max(xFinweckTree.getBound(scrollLeftRef.value) - 1, 0);
        } else {
          return 0;
        }
      });
      const getLeft = index => {
        const xFinweckTree = xFinweckTreeRef.value;
        if (xFinweckTree !== null) {
          return xFinweckTree.sum(index);
        } else {
          return 0;
        }
      };
      const endIndexRef = useMemo(() => {
        const xFinweckTree = xFinweckTreeRef.value;
        if (xFinweckTree !== null) {
          return Math.min(xFinweckTree.getBound(scrollLeftRef.value + listWidthRef.value) + 1, columnsRef.value.length - 1);
        } else {
          return 0;
        }
      });
      vue.provide(xScrollInjextionKey, {
        startIndexRef,
        endIndexRef,
        columnsRef,
        renderColRef,
        renderItemWithColsRef,
        getLeft
      });
      return {
        setListWidth(value) {
          listWidthRef.value = value;
        },
        scrollLeftRef
      };
    }

    const VirtualListRow = vue.defineComponent({
      name: 'VirtualListRow',
      props: {
        index: {
          type: Number,
          required: true
        },
        item: {
          type: Object,
          required: true
        }
      },
      setup() {
        const {
          startIndexRef,
          endIndexRef,
          columnsRef,
          getLeft,
          renderColRef,
          renderItemWithColsRef
        } =
        // eslint-disable-next-line @typescript-eslint/no-non-null-assertion
        vue.inject(xScrollInjextionKey);
        return {
          startIndex: startIndexRef,
          endIndex: endIndexRef,
          columns: columnsRef,
          renderCol: renderColRef,
          renderItemWithCols: renderItemWithColsRef,
          getLeft
        };
      },
      render() {
        const {
          startIndex,
          endIndex,
          columns,
          renderCol,
          renderItemWithCols,
          getLeft,
          item
        } = this;
        if (renderItemWithCols != null) {
          return renderItemWithCols({
            itemIndex: this.index,
            startColIndex: startIndex,
            endColIndex: endIndex,
            allColumns: columns,
            item,
            getLeft
          });
        }
        if (renderCol != null) {
          const items = [];
          for (let i = startIndex; i <= endIndex; ++i) {
            const column = columns[i];
            items.push(renderCol({
              column,
              left: getLeft(i),
              item
            }));
          }
          return items;
        }
        return null;
      }
    });

    /* eslint-disable no-void */
    /* eslint-disable @typescript-eslint/restrict-plus-operands */
    const styles$1 = c('.v-vl', {
      maxHeight: 'inherit',
      height: '100%',
      overflow: 'auto',
      minWidth: '1px' // a zero width container won't be scrollable
    }, [c('&:not(.v-vl--show-scrollbar)', {
      scrollbarWidth: 'none'
    }, [c('&::-webkit-scrollbar, &::-webkit-scrollbar-track-piece, &::-webkit-scrollbar-thumb', {
      width: 0,
      height: 0,
      display: 'none'
    })])]);
    var VVirtualList = vue.defineComponent({
      name: 'VirtualList',
      inheritAttrs: false,
      props: {
        showScrollbar: {
          type: Boolean,
          default: true
        },
        columns: {
          type: Array,
          default: () => []
        },
        renderCol: Function,
        renderItemWithCols: Function,
        items: {
          type: Array,
          default: () => []
        },
        // it is suppose to be the min height
        itemSize: {
          type: Number,
          required: true
        },
        itemResizable: Boolean,
        itemsStyle: [String, Object],
        visibleItemsTag: {
          type: [String, Object],
          default: 'div'
        },
        visibleItemsProps: Object,
        ignoreItemResize: Boolean,
        onScroll: Function,
        onWheel: Function,
        onResize: Function,
        defaultScrollKey: [Number, String],
        defaultScrollIndex: Number,
        keyField: {
          type: String,
          default: 'key'
        },
        // Whether it is a good API?
        // ResizeObserver + footer & header is not enough.
        // Too complex for simple case
        paddingTop: {
          type: [Number, String],
          default: 0
        },
        paddingBottom: {
          type: [Number, String],
          default: 0
        }
      },
      setup(props) {
        const ssrAdapter = useSsrAdapter();
        styles$1.mount({
          id: 'vueuc/virtual-list',
          head: true,
          anchorMetaName: cssrAnchorMetaName$1,
          ssr: ssrAdapter
        });
        vue.onMounted(() => {
          const {
            defaultScrollIndex,
            defaultScrollKey
          } = props;
          if (defaultScrollIndex !== undefined && defaultScrollIndex !== null) {
            scrollTo({
              index: defaultScrollIndex
            });
          } else if (defaultScrollKey !== undefined && defaultScrollKey !== null) {
            scrollTo({
              key: defaultScrollKey
            });
          }
        });
        let isDeactivated = false;
        let activateStateInitialized = false;
        vue.onActivated(() => {
          isDeactivated = false;
          if (!activateStateInitialized) {
            activateStateInitialized = true;
            return;
          }
          // remount
          scrollTo({
            top: scrollTopRef.value,
            left: scrollLeftRef.value
          });
        });
        vue.onDeactivated(() => {
          isDeactivated = true;
          if (!activateStateInitialized) {
            activateStateInitialized = true;
          }
        });
        const totalWidthRef = useMemo(() => {
          if (props.renderCol == null && props.renderItemWithCols == null) return undefined;
          if (props.columns.length === 0) return undefined;
          let width = 0;
          props.columns.forEach(column => {
            width += column.width;
          });
          return width;
        });
        const keyIndexMapRef = vue.computed(() => {
          const map = new Map();
          const {
            keyField
          } = props;
          props.items.forEach((item, index) => {
            map.set(item[keyField], index);
          });
          return map;
        });
        const {
          scrollLeftRef,
          setListWidth
        } = setupXScroll({
          columnsRef: vue.toRef(props, 'columns'),
          renderColRef: vue.toRef(props, 'renderCol'),
          renderItemWithColsRef: vue.toRef(props, 'renderItemWithCols')
        });
        const listElRef = vue.ref(null);
        const listHeightRef = vue.ref(undefined);
        const keyToHeightOffset = new Map();
        const finweckTreeRef = vue.computed(() => {
          const {
            items,
            itemSize,
            keyField
          } = props;
          const ft = new FinweckTree(items.length, itemSize);
          items.forEach((item, index) => {
            const key = item[keyField];
            const heightOffset = keyToHeightOffset.get(key);
            if (heightOffset !== undefined) {
              ft.add(index, heightOffset);
            }
          });
          return ft;
        });
        const finweckTreeUpdateTrigger = vue.ref(0);
        const scrollTopRef = vue.ref(0);
        const startIndexRef = useMemo(() => {
          return Math.max(finweckTreeRef.value.getBound(scrollTopRef.value - depx(props.paddingTop)) - 1, 0);
        });
        const viewportItemsRef = vue.computed(() => {
          const {
            value: listHeight
          } = listHeightRef;
          if (listHeight === undefined) return [];
          const {
            items,
            itemSize
          } = props;
          const startIndex = startIndexRef.value;
          const endIndex = Math.min(startIndex + Math.ceil(listHeight / itemSize + 1), items.length - 1);
          const viewportItems = [];
          for (let i = startIndex; i <= endIndex; ++i) {
            viewportItems.push(items[i]);
          }
          return viewportItems;
        });
        const scrollTo = (options, y) => {
          if (typeof options === 'number') {
            scrollToPosition(options, y, 'auto');
            return;
          }
          const {
            left,
            top,
            index,
            key,
            position,
            behavior,
            debounce = true
          } = options;
          if (left !== undefined || top !== undefined) {
            scrollToPosition(left, top, behavior);
          } else if (index !== undefined) {
            scrollToIndex(index, behavior, debounce);
          } else if (key !== undefined) {
            const toIndex = keyIndexMapRef.value.get(key);
            if (toIndex !== undefined) scrollToIndex(toIndex, behavior, debounce);
          } else if (position === 'bottom') {
            scrollToPosition(0, Number.MAX_SAFE_INTEGER, behavior);
          } else if (position === 'top') {
            scrollToPosition(0, 0, behavior);
          }
        };
        let anchorIndex;
        let anchorTimerId = null;
        function scrollToIndex(index, behavior, debounce) {
          const {
            value: ft
          } = finweckTreeRef;
          const targetTop = ft.sum(index) + depx(props.paddingTop);
          if (!debounce) {
            listElRef.value.scrollTo({
              left: 0,
              top: targetTop,
              behavior
            });
          } else {
            anchorIndex = index;
            if (anchorTimerId !== null) {
              window.clearTimeout(anchorTimerId);
            }
            anchorTimerId = window.setTimeout(() => {
              anchorIndex = undefined;
              anchorTimerId = null;
            }, 16); // use 0 ms may be ealier than resize callback...
            const {
              scrollTop,
              offsetHeight
            } = listElRef.value;
            if (targetTop > scrollTop) {
              const itemSize = ft.get(index);
              if (targetTop + itemSize <= scrollTop + offsetHeight) ; else {
                listElRef.value.scrollTo({
                  left: 0,
                  top: targetTop + itemSize - offsetHeight,
                  behavior
                });
              }
            } else {
              listElRef.value.scrollTo({
                left: 0,
                top: targetTop,
                behavior
              });
            }
          }
        }
        function scrollToPosition(left, top, behavior) {
          listElRef.value.scrollTo({
            left,
            top,
            behavior
          });
        }
        function handleItemResize(key, entry) {
          var _a, _b, _c;
          if (isDeactivated) return;
          if (props.ignoreItemResize) return;
          if (isHideByVShow(entry.target)) return;
          const {
            value: ft
          } = finweckTreeRef;
          const index = keyIndexMapRef.value.get(key);
          const previousHeight = ft.get(index);
          const height = (_c = (_b = (_a = entry.borderBoxSize) === null || _a === void 0 ? void 0 : _a[0]) === null || _b === void 0 ? void 0 : _b.blockSize) !== null && _c !== void 0 ? _c : entry.contentRect.height;
          if (height === previousHeight) return;
          // height offset based on itemSize
          // used when rebuild the finweck tree
          const offset = height - props.itemSize;
          if (offset === 0) {
            keyToHeightOffset.delete(key);
          } else {
            keyToHeightOffset.set(key, height - props.itemSize);
          }
          // delta height based on finweck tree data
          const delta = height - previousHeight;
          if (delta === 0) return;
          ft.add(index, delta);
          const listEl = listElRef.value;
          if (listEl != null) {
            if (anchorIndex === undefined) {
              const previousHeightSum = ft.sum(index);
              if (listEl.scrollTop > previousHeightSum) {
                listEl.scrollBy(0, delta);
              }
            } else {
              if (index < anchorIndex) {
                listEl.scrollBy(0, delta);
              } else if (index === anchorIndex) {
                const previousHeightSum = ft.sum(index);
                if (height + previousHeightSum >
                // Note, listEl shouldn't have border, nor offsetHeight won't be
                // correct
                listEl.scrollTop + listEl.offsetHeight) {
                  listEl.scrollBy(0, delta);
                }
              }
            }
            syncViewport();
          }
          finweckTreeUpdateTrigger.value++;
        }
        const mayUseWheel = !ensureMaybeTouch();
        let wheelCatched = false;
        function handleListScroll(e) {
          var _a;
          (_a = props.onScroll) === null || _a === void 0 ? void 0 : _a.call(props, e);
          if (!mayUseWheel || !wheelCatched) {
            syncViewport();
          }
        }
        function handleListWheel(e) {
          var _a;
          (_a = props.onWheel) === null || _a === void 0 ? void 0 : _a.call(props, e);
          if (mayUseWheel) {
            const listEl = listElRef.value;
            if (listEl != null) {
              if (e.deltaX === 0) {
                if (listEl.scrollTop === 0 && e.deltaY <= 0) {
                  return;
                }
                if (listEl.scrollTop + listEl.offsetHeight >= listEl.scrollHeight && e.deltaY >= 0) {
                  return;
                }
              }
              e.preventDefault();
              listEl.scrollTop += e.deltaY / ensureWheelScale();
              listEl.scrollLeft += e.deltaX / ensureWheelScale();
              syncViewport();
              wheelCatched = true;
              beforeNextFrameOnce(() => {
                wheelCatched = false;
              });
            }
          }
        }
        function handleListResize(entry) {
          if (isDeactivated) return;
          // List is HTMLElement
          if (isHideByVShow(entry.target)) return;
          // If height is same, return
          if (entry.contentRect.height === listHeightRef.value) return;
          listHeightRef.value = entry.contentRect.height;
          setListWidth(entry.contentRect.width);
          const {
            onResize
          } = props;
          if (onResize !== undefined) onResize(entry);
        }
        function syncViewport() {
          const {
            value: listEl
          } = listElRef;
          // sometime ref el can be null
          // https://github.com/TuSimple/naive-ui/issues/811
          if (listEl == null) return;
          scrollTopRef.value = listEl.scrollTop;
          scrollLeftRef.value = listEl.scrollLeft;
        }
        function isHideByVShow(el) {
          let cursor = el;
          while (cursor !== null) {
            if (cursor.style.display === 'none') return true;
            cursor = cursor.parentElement;
          }
          return false;
        }
        return {
          listHeight: listHeightRef,
          listStyle: {
            overflow: 'auto'
          },
          keyToIndex: keyIndexMapRef,
          itemsStyle: vue.computed(() => {
            const {
              itemResizable
            } = props;
            const height = pxfy(finweckTreeRef.value.sum());
            // eslint-disable-next-line @typescript-eslint/no-unused-expressions
            finweckTreeUpdateTrigger.value;
            return [props.itemsStyle, {
              boxSizing: 'content-box',
              width: pxfy(totalWidthRef.value),
              height: itemResizable ? '' : height,
              minHeight: itemResizable ? height : '',
              paddingTop: pxfy(props.paddingTop),
              paddingBottom: pxfy(props.paddingBottom)
            }];
          }),
          visibleItemsStyle: vue.computed(() => {
            // eslint-disable-next-line @typescript-eslint/no-unused-expressions
            finweckTreeUpdateTrigger.value;
            return {
              transform: `translateY(${pxfy(finweckTreeRef.value.sum(startIndexRef.value))})`
            };
          }),
          viewportItems: viewportItemsRef,
          listElRef,
          itemsElRef: vue.ref(null),
          scrollTo,
          handleListResize,
          handleListScroll,
          handleListWheel,
          handleItemResize
        };
      },
      render() {
        const {
          itemResizable,
          keyField,
          keyToIndex,
          visibleItemsTag
        } = this;
        return vue.h(VResizeObserver, {
          onResize: this.handleListResize
        }, {
          default: () => {
            var _a, _b;
            return vue.h('div', vue.mergeProps(this.$attrs, {
              class: ['v-vl', this.showScrollbar && 'v-vl--show-scrollbar'],
              onScroll: this.handleListScroll,
              onWheel: this.handleListWheel,
              ref: 'listElRef'
            }), [this.items.length !== 0 ? vue.h('div', {
              ref: 'itemsElRef',
              class: 'v-vl-items',
              style: this.itemsStyle
            }, [vue.h(visibleItemsTag, Object.assign({
              class: 'v-vl-visible-items',
              style: this.visibleItemsStyle
            }, this.visibleItemsProps), {
              default: () => {
                const {
                  renderCol,
                  renderItemWithCols
                } = this;
                return this.viewportItems.map(item => {
                  const key = item[keyField];
                  const index = keyToIndex.get(key);
                  const renderedCols = renderCol != null ? vue.h(VirtualListRow, {
                    index,
                    item
                  }) : undefined;
                  const renderedItemWithCols = renderItemWithCols != null ? vue.h(VirtualListRow, {
                    index,
                    item
                  }) : undefined;
                  const itemVNode = this.$slots.default({
                    item,
                    renderedCols,
                    renderedItemWithCols,
                    index
                  })[0];
                  if (itemResizable) {
                    return vue.h(VResizeObserver, {
                      key,
                      onResize: entry => this.handleItemResize(key, entry)
                    }, {
                      default: () => itemVNode
                    });
                  }
                  itemVNode.key = key;
                  return itemVNode;
                });
              }
            })]) : (_b = (_a = this.$slots).empty) === null || _b === void 0 ? void 0 : _b.call(_a)]);
          }
        });
      }
    });

    const styles = c('.v-x-scroll', {
      overflow: 'auto',
      scrollbarWidth: 'none'
    }, [c('&::-webkit-scrollbar', {
      width: 0,
      height: 0
    })]);
    var VXScroll = vue.defineComponent({
      name: 'XScroll',
      props: {
        disabled: Boolean,
        onScroll: Function
      },
      setup() {
        const selfRef = vue.ref(null);
        function handleWheel(e) {
          const preventYWheel = e.currentTarget.offsetWidth < e.currentTarget.scrollWidth;
          if (!preventYWheel || e.deltaY === 0) return;
          e.currentTarget.scrollLeft += e.deltaY + e.deltaX;
          e.preventDefault();
        }
        const ssrAdapter = useSsrAdapter();
        styles.mount({
          id: 'vueuc/x-scroll',
          head: true,
          anchorMetaName: cssrAnchorMetaName$1,
          ssr: ssrAdapter
        });
        const exposedMethods = {
          scrollTo(...args) {
            var _a;
            (_a = selfRef.value) === null || _a === void 0 ? void 0 : _a.scrollTo(...args);
          }
        };
        return Object.assign({
          selfRef,
          handleWheel
        }, exposedMethods);
      },
      render() {
        return vue.h('div', {
          ref: 'selfRef',
          onScroll: this.onScroll,
          onWheel: this.disabled ? undefined : this.handleWheel,
          class: 'v-x-scroll'
        }, this.$slots);
      }
    });

    /* eslint-disable @typescript-eslint/strict-boolean-expressions */
    const hiddenAttr = 'v-hidden';
    const style$1E = c('[v-hidden]', {
      display: 'none!important'
    });
    var VOverflow = vue.defineComponent({
      name: 'Overflow',
      props: {
        getCounter: Function,
        getTail: Function,
        updateCounter: Function,
        onUpdateCount: Function,
        onUpdateOverflow: Function
      },
      setup(props, {
        slots
      }) {
        const selfRef = vue.ref(null);
        const counterRef = vue.ref(null);
        function deriveCounter(options) {
          const {
            value: self
          } = selfRef;
          const {
            getCounter,
            getTail
          } = props;
          let counter;
          if (getCounter !== undefined) counter = getCounter();else {
            counter = counterRef.value;
          }
          if (!self || !counter) return;
          if (counter.hasAttribute(hiddenAttr)) {
            counter.removeAttribute(hiddenAttr);
          }
          const {
            children
          } = self;
          if (options.showAllItemsBeforeCalculate) {
            for (const child of children) {
              if (child.hasAttribute(hiddenAttr)) {
                child.removeAttribute(hiddenAttr);
              }
            }
          }
          const containerWidth = self.offsetWidth;
          const childWidths = [];
          const tail = slots.tail ? getTail === null || getTail === void 0 ? void 0 : getTail() : null;
          let childWidthSum = tail ? tail.offsetWidth : 0;
          let overflow = false;
          const len = self.children.length - (slots.tail ? 1 : 0);
          for (let i = 0; i < len - 1; ++i) {
            if (i < 0) continue;
            const child = children[i];
            if (overflow) {
              if (!child.hasAttribute(hiddenAttr)) {
                child.setAttribute(hiddenAttr, '');
              }
              continue;
            } else if (child.hasAttribute(hiddenAttr)) {
              child.removeAttribute(hiddenAttr);
            }
            const childWidth = child.offsetWidth;
            childWidthSum += childWidth;
            childWidths[i] = childWidth;
            if (childWidthSum > containerWidth) {
              const {
                updateCounter
              } = props;
              for (let j = i; j >= 0; --j) {
                const restCount = len - 1 - j;
                if (updateCounter !== undefined) {
                  updateCounter(restCount);
                } else {
                  counter.textContent = `${restCount}`;
                }
                const counterWidth = counter.offsetWidth;
                childWidthSum -= childWidths[j];
                if (childWidthSum + counterWidth <= containerWidth || j === 0) {
                  overflow = true;
                  i = j - 1;
                  if (tail) {
                    // tail too long or 1st element too long
                    // we only consider tail now
                    if (i === -1) {
                      tail.style.maxWidth = `${containerWidth - counterWidth}px`;
                      tail.style.boxSizing = 'border-box';
                    } else {
                      tail.style.maxWidth = '';
                    }
                  }
                  const {
                    onUpdateCount
                  } = props;
                  if (onUpdateCount) onUpdateCount(restCount);
                  break;
                }
              }
            }
          }
          const {
            onUpdateOverflow
          } = props;
          if (!overflow) {
            if (onUpdateOverflow !== undefined) {
              onUpdateOverflow(false);
            }
            counter.setAttribute(hiddenAttr, '');
          } else {
            if (onUpdateOverflow !== undefined) {
              onUpdateOverflow(true);
            }
          }
        }
        const ssrAdapter = useSsrAdapter();
        style$1E.mount({
          id: 'vueuc/overflow',
          head: true,
          anchorMetaName: cssrAnchorMetaName$1,
          ssr: ssrAdapter
        });
        vue.onMounted(() => deriveCounter({
          showAllItemsBeforeCalculate: false
        }));
        // besides onMounted, other case should be manually triggered, or we shoud watch items
        return {
          selfRef,
          counterRef,
          sync: deriveCounter
        };
      },
      render() {
        const {
          $slots
        } = this;
        // eslint-disable-next-line @typescript-eslint/no-floating-promises
        vue.nextTick(() => this.sync({
          showAllItemsBeforeCalculate: false
        }));
        // It shouldn't have border
        return vue.h('div', {
          class: 'v-overflow',
          ref: 'selfRef'
        }, [vue.renderSlot($slots, 'default'),
        // $slots.counter should only has 1 element
        $slots.counter ? $slots.counter() : vue.h('span', {
          style: {
            display: 'inline-block'
          },
          ref: 'counterRef'
        }),
        // $slots.tail should only has 1 element
        $slots.tail ? $slots.tail() : null]);
      }
    });

    /* eslint-disable @typescript-eslint/strict-boolean-expressions */
    // ref https://www.w3.org/TR/wai-aria-practices-1.1/examples/dialog-modal/js/dialog.js
    function isHTMLElement(node) {
      return node instanceof HTMLElement;
    }
    function focusFirstDescendant(node) {
      for (let i = 0; i < node.childNodes.length; i++) {
        const child = node.childNodes[i];
        if (isHTMLElement(child)) {
          if (attemptFocus(child) || focusFirstDescendant(child)) {
            return true;
          }
        }
      }
      return false;
    }
    function focusLastDescendant(element) {
      for (let i = element.childNodes.length - 1; i >= 0; i--) {
        const child = element.childNodes[i];
        if (isHTMLElement(child)) {
          if (attemptFocus(child) || focusLastDescendant(child)) {
            return true;
          }
        }
      }
      return false;
    }
    function attemptFocus(element) {
      if (!isFocusable(element)) {
        return false;
      }
      try {
        element.focus({
          preventScroll: true
        });
      } catch (e) {}
      return document.activeElement === element;
    }
    function isFocusable(element) {
      if (element.tabIndex > 0 || element.tabIndex === 0 && element.getAttribute('tabIndex') !== null) {
        return true;
      }
      if (element.getAttribute('disabled')) {
        return false;
      }
      switch (element.nodeName) {
        case 'A':
          return !!element.href && element.rel !== 'ignore';
        case 'INPUT':
          return element.type !== 'hidden' && element.type !== 'file';
        case 'BUTTON':
        case 'SELECT':
        case 'TEXTAREA':
          return true;
        default:
          return false;
      }
    }

    let stack = [];
    const FocusTrap = vue.defineComponent({
      name: 'FocusTrap',
      props: {
        disabled: Boolean,
        active: Boolean,
        autoFocus: {
          type: Boolean,
          default: true
        },
        onEsc: Function,
        initialFocusTo: String,
        finalFocusTo: String,
        returnFocusOnDeactivated: {
          type: Boolean,
          default: true
        }
      },
      setup(props) {
        const id = createId();
        const focusableStartRef = vue.ref(null);
        const focusableEndRef = vue.ref(null);
        let activated = false;
        let ignoreInternalFocusChange = false;
        const lastFocusedElement = typeof document === 'undefined' ? null : document.activeElement;
        function isCurrentActive() {
          const currentActiveId = stack[stack.length - 1];
          return currentActiveId === id;
        }
        function handleDocumentKeydown(e) {
          var _a;
          if (e.code === 'Escape') {
            if (isCurrentActive()) {
              (_a = props.onEsc) === null || _a === void 0 ? void 0 : _a.call(props, e);
            }
          }
        }
        vue.onMounted(() => {
          vue.watch(() => props.active, value => {
            if (value) {
              activate();
              on('keydown', document, handleDocumentKeydown);
            } else {
              off('keydown', document, handleDocumentKeydown);
              if (activated) {
                deactivate();
              }
            }
          }, {
            immediate: true
          });
        });
        vue.onBeforeUnmount(() => {
          off('keydown', document, handleDocumentKeydown);
          if (activated) deactivate();
        });
        function handleDocumentFocus(e) {
          if (ignoreInternalFocusChange) return;
          if (isCurrentActive()) {
            const mainEl = getMainEl();
            if (mainEl === null) return;
            if (mainEl.contains(getPreciseEventTarget(e))) return;
            // I don't handle shift + tab status since it's too tricky to handle
            // Not impossible but I need to sleep
            resetFocusTo('first');
          }
        }
        function getMainEl() {
          const focusableStartEl = focusableStartRef.value;
          if (focusableStartEl === null) return null;
          let mainEl = focusableStartEl;
          while (true) {
            mainEl = mainEl.nextSibling;
            if (mainEl === null) break;
            if (mainEl instanceof Element && mainEl.tagName === 'DIV') {
              break;
            }
          }
          return mainEl;
        }
        function activate() {
          var _a;
          if (props.disabled) return;
          stack.push(id);
          if (props.autoFocus) {
            const {
              initialFocusTo
            } = props;
            if (initialFocusTo === undefined) {
              resetFocusTo('first');
            } else {
              (_a = resolveTo(initialFocusTo)) === null || _a === void 0 ? void 0 : _a.focus({
                preventScroll: true
              });
            }
          }
          activated = true;
          document.addEventListener('focus', handleDocumentFocus, true);
        }
        function deactivate() {
          var _a;
          if (props.disabled) return;
          document.removeEventListener('focus', handleDocumentFocus, true);
          stack = stack.filter(idInStack => idInStack !== id);
          if (isCurrentActive()) return;
          const {
            finalFocusTo
          } = props;
          if (finalFocusTo !== undefined) {
            (_a = resolveTo(finalFocusTo)) === null || _a === void 0 ? void 0 : _a.focus({
              preventScroll: true
            });
          } else if (props.returnFocusOnDeactivated) {
            if (lastFocusedElement instanceof HTMLElement) {
              ignoreInternalFocusChange = true;
              lastFocusedElement.focus({
                preventScroll: true
              });
              ignoreInternalFocusChange = false;
            }
          }
        }
        function resetFocusTo(target) {
          if (!isCurrentActive()) return;
          if (props.active) {
            const focusableStartEl = focusableStartRef.value;
            const focusableEndEl = focusableEndRef.value;
            if (focusableStartEl !== null && focusableEndEl !== null) {
              const mainEl = getMainEl();
              if (mainEl == null || mainEl === focusableEndEl) {
                ignoreInternalFocusChange = true;
                focusableStartEl.focus({
                  preventScroll: true
                });
                ignoreInternalFocusChange = false;
                return;
              }
              ignoreInternalFocusChange = true;
              const focused = target === 'first' ? focusFirstDescendant(mainEl) : focusLastDescendant(mainEl);
              ignoreInternalFocusChange = false;
              if (!focused) {
                ignoreInternalFocusChange = true;
                focusableStartEl.focus({
                  preventScroll: true
                });
                ignoreInternalFocusChange = false;
              }
            }
          }
        }
        function handleStartFocus(e) {
          if (ignoreInternalFocusChange) return;
          const mainEl = getMainEl();
          if (mainEl === null) return;
          if (e.relatedTarget !== null && mainEl.contains(e.relatedTarget)) {
            // if it comes from inner, focus last
            resetFocusTo('last');
          } else {
            // otherwise focus first
            resetFocusTo('first');
          }
        }
        function handleEndFocus(e) {
          if (ignoreInternalFocusChange) return;
          if (e.relatedTarget !== null && e.relatedTarget === focusableStartRef.value) {
            // if it comes from first, focus last
            resetFocusTo('last');
          } else {
            // otherwise focus first
            resetFocusTo('first');
          }
        }
        return {
          focusableStartRef,
          focusableEndRef,
          focusableStyle: 'position: absolute; height: 0; width: 0;',
          handleStartFocus,
          handleEndFocus
        };
      },
      render() {
        const {
          default: defaultSlot
        } = this.$slots;
        if (defaultSlot === undefined) return null;
        if (this.disabled) return defaultSlot();
        const {
          active,
          focusableStyle
        } = this;
        return vue.h(vue.Fragment, null, [vue.h('div', {
          'aria-hidden': 'true',
          tabindex: active ? '0' : '-1',
          ref: 'focusableStartRef',
          style: focusableStyle,
          onFocus: this.handleStartFocus
        }), defaultSlot(), vue.h('div', {
          'aria-hidden': 'true',
          style: focusableStyle,
          ref: 'focusableEndRef',
          tabindex: active ? '0' : '-1',
          onFocus: this.handleEndFocus
        })]);
      }
    });

    function useOnResize(elRef, onResize) {
      if (onResize) {
        vue.onMounted(() => {
          const {
            value: el
          } = elRef;
          if (el) {
            resizeObserverManager.registerHandler(el, onResize);
          }
        });
        vue.onBeforeUnmount(() => {
          const {
            value: el
          } = elRef;
          if (el) {
            resizeObserverManager.unregisterHandler(el);
          }
        });
      }
    }

    let lockCount = 0;
    let originalMarginRight = "";
    let originalOverflow = "";
    let originalOverflowX = "";
    let originalOverflowY = "";
    const lockHtmlScrollRightCompensationRef = vue.ref("0px");
    function useLockHtmlScroll(lockRef) {
      if (typeof document === "undefined") return;
      const el = document.documentElement;
      let watchStopHandle;
      let activated = false;
      const unlock = () => {
        el.style.marginRight = originalMarginRight;
        el.style.overflow = originalOverflow;
        el.style.overflowX = originalOverflowX;
        el.style.overflowY = originalOverflowY;
        lockHtmlScrollRightCompensationRef.value = "0px";
      };
      vue.onMounted(() => {
        watchStopHandle = vue.watch(lockRef, value => {
          if (value) {
            if (!lockCount) {
              const scrollbarWidth = window.innerWidth - el.offsetWidth;
              if (scrollbarWidth > 0) {
                originalMarginRight = el.style.marginRight;
                el.style.marginRight = `${scrollbarWidth}px`;
                lockHtmlScrollRightCompensationRef.value = `${scrollbarWidth}px`;
              }
              originalOverflow = el.style.overflow;
              originalOverflowX = el.style.overflowX;
              originalOverflowY = el.style.overflowY;
              el.style.overflow = "hidden";
              el.style.overflowX = "hidden";
              el.style.overflowY = "hidden";
            }
            activated = true;
            lockCount++;
          } else {
            lockCount--;
            if (!lockCount) {
              unlock();
            }
            activated = false;
          }
        }, {
          immediate: true
        });
      });
      vue.onBeforeUnmount(() => {
        watchStopHandle?.();
        if (activated) {
          lockCount--;
          if (!lockCount) {
            unlock();
          }
          activated = false;
        }
      });
    }

    const isComposingRef = vue.ref(false);
    function compositionStartHandler() {
      isComposingRef.value = true;
    }
    function compositionEndHandler() {
      isComposingRef.value = false;
    }
    let mountedCount = 0;
    function useIsComposing() {
      if (isBrowser$2) {
        vue.onBeforeMount(() => {
          if (!mountedCount) {
            window.addEventListener("compositionstart", compositionStartHandler);
            window.addEventListener("compositionend", compositionEndHandler);
          }
          mountedCount++;
        });
        vue.onBeforeUnmount(() => {
          if (mountedCount <= 1) {
            window.removeEventListener("compositionstart", compositionStartHandler);
            window.removeEventListener("compositionend", compositionEndHandler);
            mountedCount = 0;
          } else {
            mountedCount--;
          }
        });
      }
      return isComposingRef;
    }

    function useReactivated(callback) {
      const isDeactivatedRef = {
        isDeactivated: false
      };
      let activateStateInitialized = false;
      vue.onActivated(() => {
        isDeactivatedRef.isDeactivated = false;
        if (!activateStateInitialized) {
          activateStateInitialized = true;
          return;
        }
        callback();
      });
      vue.onDeactivated(() => {
        isDeactivatedRef.isDeactivated = true;
        if (!activateStateInitialized) {
          activateStateInitialized = true;
        }
      });
      return isDeactivatedRef;
    }

    function isDocument(node) {
      return node.nodeName === "#document";
    }

    function download(url, name) {
      if (!url) return;
      const a = document.createElement("a");
      a.href = url;
      if (name !== void 0) {
        a.download = name;
      }
      document.body.appendChild(a);
      a.click();
      document.body.removeChild(a);
    }

    const formItemInjectionKey = createInjectionKey("n-form-item");
    function useFormItem(props, {
      defaultSize = "medium",
      mergedSize,
      mergedDisabled
    } = {}) {
      const NFormItem = vue.inject(formItemInjectionKey, null);
      vue.provide(formItemInjectionKey, null);
      const mergedSizeRef = vue.computed(mergedSize ? () => mergedSize(NFormItem) : () => {
        const {
          size
        } = props;
        if (size) return size;
        if (NFormItem) {
          const {
            mergedSize: mergedSize2
          } = NFormItem;
          if (mergedSize2.value !== void 0) {
            return mergedSize2.value;
          }
        }
        return defaultSize;
      });
      const mergedDisabledRef = vue.computed(mergedDisabled ? () => mergedDisabled(NFormItem) : () => {
        const {
          disabled
        } = props;
        if (disabled !== void 0) {
          return disabled;
        }
        if (NFormItem) {
          return NFormItem.disabled.value;
        }
        return false;
      });
      const mergedStatusRef = vue.computed(() => {
        const {
          status
        } = props;
        if (status) return status;
        return NFormItem?.mergedValidationStatus.value;
      });
      vue.onBeforeUnmount(() => {
        if (NFormItem) {
          NFormItem.restoreValidation();
        }
      });
      return {
        mergedSizeRef,
        mergedDisabledRef,
        mergedStatusRef,
        nTriggerFormBlur() {
          if (NFormItem) {
            NFormItem.handleContentBlur();
          }
        },
        nTriggerFormChange() {
          if (NFormItem) {
            NFormItem.handleContentChange();
          }
        },
        nTriggerFormFocus() {
          if (NFormItem) {
            NFormItem.handleContentFocus();
          }
        },
        nTriggerFormInput() {
          if (NFormItem) {
            NFormItem.handleContentInput();
          }
        }
      };
    }

    var commonjsGlobal = typeof globalThis !== 'undefined' ? globalThis : typeof window !== 'undefined' ? window : typeof global !== 'undefined' ? global : typeof self !== 'undefined' ? self : {};

    var lodash = {exports: {}};

    /**
     * @license
     * Lodash <https://lodash.com/>
     * Copyright OpenJS Foundation and other contributors <https://openjsf.org/>
     * Released under MIT license <https://lodash.com/license>
     * Based on Underscore.js 1.8.3 <http://underscorejs.org/LICENSE>
     * Copyright Jeremy Ashkenas, DocumentCloud and Investigative Reporters & Editors
     */
    lodash.exports;

    (function (module, exports) {
    (function(){/** Used as a safe reference for `undefined` in pre-ES5 environments. */var undefined$1;/** Used as the semantic version number. */var VERSION='4.17.21';/** Used as the size to enable large array optimizations. */var LARGE_ARRAY_SIZE=200;/** Error message constants. */var CORE_ERROR_TEXT='Unsupported core-js use. Try https://npms.io/search?q=ponyfill.',FUNC_ERROR_TEXT='Expected a function',INVALID_TEMPL_VAR_ERROR_TEXT='Invalid `variable` option passed into `_.template`';/** Used to stand-in for `undefined` hash values. */var HASH_UNDEFINED='__lodash_hash_undefined__';/** Used as the maximum memoize cache size. */var MAX_MEMOIZE_SIZE=500;/** Used as the internal argument placeholder. */var PLACEHOLDER='__lodash_placeholder__';/** Used to compose bitmasks for cloning. */var CLONE_DEEP_FLAG=1,CLONE_FLAT_FLAG=2,CLONE_SYMBOLS_FLAG=4;/** Used to compose bitmasks for value comparisons. */var COMPARE_PARTIAL_FLAG=1,COMPARE_UNORDERED_FLAG=2;/** Used to compose bitmasks for function metadata. */var WRAP_BIND_FLAG=1,WRAP_BIND_KEY_FLAG=2,WRAP_CURRY_BOUND_FLAG=4,WRAP_CURRY_FLAG=8,WRAP_CURRY_RIGHT_FLAG=16,WRAP_PARTIAL_FLAG=32,WRAP_PARTIAL_RIGHT_FLAG=64,WRAP_ARY_FLAG=128,WRAP_REARG_FLAG=256,WRAP_FLIP_FLAG=512;/** Used as default options for `_.truncate`. */var DEFAULT_TRUNC_LENGTH=30,DEFAULT_TRUNC_OMISSION='...';/** Used to detect hot functions by number of calls within a span of milliseconds. */var HOT_COUNT=800,HOT_SPAN=16;/** Used to indicate the type of lazy iteratees. */var LAZY_FILTER_FLAG=1,LAZY_MAP_FLAG=2,LAZY_WHILE_FLAG=3;/** Used as references for various `Number` constants. */var INFINITY=1/0,MAX_SAFE_INTEGER=9007199254740991,MAX_INTEGER=1.7976931348623157e+308,NAN=0/0;/** Used as references for the maximum length and index of an array. */var MAX_ARRAY_LENGTH=4294967295,MAX_ARRAY_INDEX=MAX_ARRAY_LENGTH-1,HALF_MAX_ARRAY_LENGTH=MAX_ARRAY_LENGTH>>>1;/** Used to associate wrap methods with their bit flags. */var wrapFlags=[['ary',WRAP_ARY_FLAG],['bind',WRAP_BIND_FLAG],['bindKey',WRAP_BIND_KEY_FLAG],['curry',WRAP_CURRY_FLAG],['curryRight',WRAP_CURRY_RIGHT_FLAG],['flip',WRAP_FLIP_FLAG],['partial',WRAP_PARTIAL_FLAG],['partialRight',WRAP_PARTIAL_RIGHT_FLAG],['rearg',WRAP_REARG_FLAG]];/** `Object#toString` result references. */var argsTag='[object Arguments]',arrayTag='[object Array]',asyncTag='[object AsyncFunction]',boolTag='[object Boolean]',dateTag='[object Date]',domExcTag='[object DOMException]',errorTag='[object Error]',funcTag='[object Function]',genTag='[object GeneratorFunction]',mapTag='[object Map]',numberTag='[object Number]',nullTag='[object Null]',objectTag='[object Object]',promiseTag='[object Promise]',proxyTag='[object Proxy]',regexpTag='[object RegExp]',setTag='[object Set]',stringTag='[object String]',symbolTag='[object Symbol]',undefinedTag='[object Undefined]',weakMapTag='[object WeakMap]',weakSetTag='[object WeakSet]';var arrayBufferTag='[object ArrayBuffer]',dataViewTag='[object DataView]',float32Tag='[object Float32Array]',float64Tag='[object Float64Array]',int8Tag='[object Int8Array]',int16Tag='[object Int16Array]',int32Tag='[object Int32Array]',uint8Tag='[object Uint8Array]',uint8ClampedTag='[object Uint8ClampedArray]',uint16Tag='[object Uint16Array]',uint32Tag='[object Uint32Array]';/** Used to match empty string literals in compiled template source. */var reEmptyStringLeading=/\b__p \+= '';/g,reEmptyStringMiddle=/\b(__p \+=) '' \+/g,reEmptyStringTrailing=/(__e\(.*?\)|\b__t\)) \+\n'';/g;/** Used to match HTML entities and HTML characters. */var reEscapedHtml=/&(?:amp|lt|gt|quot|#39);/g,reUnescapedHtml=/[&<>"']/g,reHasEscapedHtml=RegExp(reEscapedHtml.source),reHasUnescapedHtml=RegExp(reUnescapedHtml.source);/** Used to match template delimiters. */var reEscape=/<%-([\s\S]+?)%>/g,reEvaluate=/<%([\s\S]+?)%>/g,reInterpolate=/<%=([\s\S]+?)%>/g;/** Used to match property names within property paths. */var reIsDeepProp=/\.|\[(?:[^[\]]*|(["'])(?:(?!\1)[^\\]|\\.)*?\1)\]/,reIsPlainProp=/^\w*$/,rePropName=/[^.[\]]+|\[(?:(-?\d+(?:\.\d+)?)|(["'])((?:(?!\2)[^\\]|\\.)*?)\2)\]|(?=(?:\.|\[\])(?:\.|\[\]|$))/g;/**
    	   * Used to match `RegExp`
    	   * [syntax characters](http://ecma-international.org/ecma-262/7.0/#sec-patterns).
    	   */var reRegExpChar=/[\\^$.*+?()[\]{}|]/g,reHasRegExpChar=RegExp(reRegExpChar.source);/** Used to match leading whitespace. */var reTrimStart=/^\s+/;/** Used to match a single whitespace character. */var reWhitespace=/\s/;/** Used to match wrap detail comments. */var reWrapComment=/\{(?:\n\/\* \[wrapped with .+\] \*\/)?\n?/,reWrapDetails=/\{\n\/\* \[wrapped with (.+)\] \*/,reSplitDetails=/,? & /;/** Used to match words composed of alphanumeric characters. */var reAsciiWord=/[^\x00-\x2f\x3a-\x40\x5b-\x60\x7b-\x7f]+/g;/**
    	   * Used to validate the `validate` option in `_.template` variable.
    	   *
    	   * Forbids characters which could potentially change the meaning of the function argument definition:
    	   * - "()," (modification of function parameters)
    	   * - "=" (default value)
    	   * - "[]{}" (destructuring of function parameters)
    	   * - "/" (beginning of a comment)
    	   * - whitespace
    	   */var reForbiddenIdentifierChars=/[()=,{}\[\]\/\s]/;/** Used to match backslashes in property paths. */var reEscapeChar=/\\(\\)?/g;/**
    	   * Used to match
    	   * [ES template delimiters](http://ecma-international.org/ecma-262/7.0/#sec-template-literal-lexical-components).
    	   */var reEsTemplate=/\$\{([^\\}]*(?:\\.[^\\}]*)*)\}/g;/** Used to match `RegExp` flags from their coerced string values. */var reFlags=/\w*$/;/** Used to detect bad signed hexadecimal string values. */var reIsBadHex=/^[-+]0x[0-9a-f]+$/i;/** Used to detect binary string values. */var reIsBinary=/^0b[01]+$/i;/** Used to detect host constructors (Safari). */var reIsHostCtor=/^\[object .+?Constructor\]$/;/** Used to detect octal string values. */var reIsOctal=/^0o[0-7]+$/i;/** Used to detect unsigned integer values. */var reIsUint=/^(?:0|[1-9]\d*)$/;/** Used to match Latin Unicode letters (excluding mathematical operators). */var reLatin=/[\xc0-\xd6\xd8-\xf6\xf8-\xff\u0100-\u017f]/g;/** Used to ensure capturing order of template delimiters. */var reNoMatch=/($^)/;/** Used to match unescaped characters in compiled string literals. */var reUnescapedString=/['\n\r\u2028\u2029\\]/g;/** Used to compose unicode character classes. */var rsAstralRange='\\ud800-\\udfff',rsComboMarksRange='\\u0300-\\u036f',reComboHalfMarksRange='\\ufe20-\\ufe2f',rsComboSymbolsRange='\\u20d0-\\u20ff',rsComboRange=rsComboMarksRange+reComboHalfMarksRange+rsComboSymbolsRange,rsDingbatRange='\\u2700-\\u27bf',rsLowerRange='a-z\\xdf-\\xf6\\xf8-\\xff',rsMathOpRange='\\xac\\xb1\\xd7\\xf7',rsNonCharRange='\\x00-\\x2f\\x3a-\\x40\\x5b-\\x60\\x7b-\\xbf',rsPunctuationRange='\\u2000-\\u206f',rsSpaceRange=' \\t\\x0b\\f\\xa0\\ufeff\\n\\r\\u2028\\u2029\\u1680\\u180e\\u2000\\u2001\\u2002\\u2003\\u2004\\u2005\\u2006\\u2007\\u2008\\u2009\\u200a\\u202f\\u205f\\u3000',rsUpperRange='A-Z\\xc0-\\xd6\\xd8-\\xde',rsVarRange='\\ufe0e\\ufe0f',rsBreakRange=rsMathOpRange+rsNonCharRange+rsPunctuationRange+rsSpaceRange;/** Used to compose unicode capture groups. */var rsApos="['\u2019]",rsAstral='['+rsAstralRange+']',rsBreak='['+rsBreakRange+']',rsCombo='['+rsComboRange+']',rsDigits='\\d+',rsDingbat='['+rsDingbatRange+']',rsLower='['+rsLowerRange+']',rsMisc='[^'+rsAstralRange+rsBreakRange+rsDigits+rsDingbatRange+rsLowerRange+rsUpperRange+']',rsFitz='\\ud83c[\\udffb-\\udfff]',rsModifier='(?:'+rsCombo+'|'+rsFitz+')',rsNonAstral='[^'+rsAstralRange+']',rsRegional='(?:\\ud83c[\\udde6-\\uddff]){2}',rsSurrPair='[\\ud800-\\udbff][\\udc00-\\udfff]',rsUpper='['+rsUpperRange+']',rsZWJ='\\u200d';/** Used to compose unicode regexes. */var rsMiscLower='(?:'+rsLower+'|'+rsMisc+')',rsMiscUpper='(?:'+rsUpper+'|'+rsMisc+')',rsOptContrLower='(?:'+rsApos+'(?:d|ll|m|re|s|t|ve))?',rsOptContrUpper='(?:'+rsApos+'(?:D|LL|M|RE|S|T|VE))?',reOptMod=rsModifier+'?',rsOptVar='['+rsVarRange+']?',rsOptJoin='(?:'+rsZWJ+'(?:'+[rsNonAstral,rsRegional,rsSurrPair].join('|')+')'+rsOptVar+reOptMod+')*',rsOrdLower='\\d*(?:1st|2nd|3rd|(?![123])\\dth)(?=\\b|[A-Z_])',rsOrdUpper='\\d*(?:1ST|2ND|3RD|(?![123])\\dTH)(?=\\b|[a-z_])',rsSeq=rsOptVar+reOptMod+rsOptJoin,rsEmoji='(?:'+[rsDingbat,rsRegional,rsSurrPair].join('|')+')'+rsSeq,rsSymbol='(?:'+[rsNonAstral+rsCombo+'?',rsCombo,rsRegional,rsSurrPair,rsAstral].join('|')+')';/** Used to match apostrophes. */var reApos=RegExp(rsApos,'g');/**
    	   * Used to match [combining diacritical marks](https://en.wikipedia.org/wiki/Combining_Diacritical_Marks) and
    	   * [combining diacritical marks for symbols](https://en.wikipedia.org/wiki/Combining_Diacritical_Marks_for_Symbols).
    	   */var reComboMark=RegExp(rsCombo,'g');/** Used to match [string symbols](https://mathiasbynens.be/notes/javascript-unicode). */var reUnicode=RegExp(rsFitz+'(?='+rsFitz+')|'+rsSymbol+rsSeq,'g');/** Used to match complex or compound words. */var reUnicodeWord=RegExp([rsUpper+'?'+rsLower+'+'+rsOptContrLower+'(?='+[rsBreak,rsUpper,'$'].join('|')+')',rsMiscUpper+'+'+rsOptContrUpper+'(?='+[rsBreak,rsUpper+rsMiscLower,'$'].join('|')+')',rsUpper+'?'+rsMiscLower+'+'+rsOptContrLower,rsUpper+'+'+rsOptContrUpper,rsOrdUpper,rsOrdLower,rsDigits,rsEmoji].join('|'),'g');/** Used to detect strings with [zero-width joiners or code points from the astral planes](http://eev.ee/blog/2015/09/12/dark-corners-of-unicode/). */var reHasUnicode=RegExp('['+rsZWJ+rsAstralRange+rsComboRange+rsVarRange+']');/** Used to detect strings that need a more robust regexp to match words. */var reHasUnicodeWord=/[a-z][A-Z]|[A-Z]{2}[a-z]|[0-9][a-zA-Z]|[a-zA-Z][0-9]|[^a-zA-Z0-9 ]/;/** Used to assign default `context` object properties. */var contextProps=['Array','Buffer','DataView','Date','Error','Float32Array','Float64Array','Function','Int8Array','Int16Array','Int32Array','Map','Math','Object','Promise','RegExp','Set','String','Symbol','TypeError','Uint8Array','Uint8ClampedArray','Uint16Array','Uint32Array','WeakMap','_','clearTimeout','isFinite','parseInt','setTimeout'];/** Used to make template sourceURLs easier to identify. */var templateCounter=-1;/** Used to identify `toStringTag` values of typed arrays. */var typedArrayTags={};typedArrayTags[float32Tag]=typedArrayTags[float64Tag]=typedArrayTags[int8Tag]=typedArrayTags[int16Tag]=typedArrayTags[int32Tag]=typedArrayTags[uint8Tag]=typedArrayTags[uint8ClampedTag]=typedArrayTags[uint16Tag]=typedArrayTags[uint32Tag]=true;typedArrayTags[argsTag]=typedArrayTags[arrayTag]=typedArrayTags[arrayBufferTag]=typedArrayTags[boolTag]=typedArrayTags[dataViewTag]=typedArrayTags[dateTag]=typedArrayTags[errorTag]=typedArrayTags[funcTag]=typedArrayTags[mapTag]=typedArrayTags[numberTag]=typedArrayTags[objectTag]=typedArrayTags[regexpTag]=typedArrayTags[setTag]=typedArrayTags[stringTag]=typedArrayTags[weakMapTag]=false;/** Used to identify `toStringTag` values supported by `_.clone`. */var cloneableTags={};cloneableTags[argsTag]=cloneableTags[arrayTag]=cloneableTags[arrayBufferTag]=cloneableTags[dataViewTag]=cloneableTags[boolTag]=cloneableTags[dateTag]=cloneableTags[float32Tag]=cloneableTags[float64Tag]=cloneableTags[int8Tag]=cloneableTags[int16Tag]=cloneableTags[int32Tag]=cloneableTags[mapTag]=cloneableTags[numberTag]=cloneableTags[objectTag]=cloneableTags[regexpTag]=cloneableTags[setTag]=cloneableTags[stringTag]=cloneableTags[symbolTag]=cloneableTags[uint8Tag]=cloneableTags[uint8ClampedTag]=cloneableTags[uint16Tag]=cloneableTags[uint32Tag]=true;cloneableTags[errorTag]=cloneableTags[funcTag]=cloneableTags[weakMapTag]=false;/** Used to map Latin Unicode letters to basic Latin letters. */var deburredLetters={// Latin-1 Supplement block.
    	'\xc0':'A','\xc1':'A','\xc2':'A','\xc3':'A','\xc4':'A','\xc5':'A','\xe0':'a','\xe1':'a','\xe2':'a','\xe3':'a','\xe4':'a','\xe5':'a','\xc7':'C','\xe7':'c','\xd0':'D','\xf0':'d','\xc8':'E','\xc9':'E','\xca':'E','\xcb':'E','\xe8':'e','\xe9':'e','\xea':'e','\xeb':'e','\xcc':'I','\xcd':'I','\xce':'I','\xcf':'I','\xec':'i','\xed':'i','\xee':'i','\xef':'i','\xd1':'N','\xf1':'n','\xd2':'O','\xd3':'O','\xd4':'O','\xd5':'O','\xd6':'O','\xd8':'O','\xf2':'o','\xf3':'o','\xf4':'o','\xf5':'o','\xf6':'o','\xf8':'o','\xd9':'U','\xda':'U','\xdb':'U','\xdc':'U','\xf9':'u','\xfa':'u','\xfb':'u','\xfc':'u','\xdd':'Y','\xfd':'y','\xff':'y','\xc6':'Ae','\xe6':'ae','\xde':'Th','\xfe':'th','\xdf':'ss',// Latin Extended-A block.
    	'\u0100':'A','\u0102':'A','\u0104':'A','\u0101':'a','\u0103':'a','\u0105':'a','\u0106':'C','\u0108':'C','\u010a':'C','\u010c':'C','\u0107':'c','\u0109':'c','\u010b':'c','\u010d':'c','\u010e':'D','\u0110':'D','\u010f':'d','\u0111':'d','\u0112':'E','\u0114':'E','\u0116':'E','\u0118':'E','\u011a':'E','\u0113':'e','\u0115':'e','\u0117':'e','\u0119':'e','\u011b':'e','\u011c':'G','\u011e':'G','\u0120':'G','\u0122':'G','\u011d':'g','\u011f':'g','\u0121':'g','\u0123':'g','\u0124':'H','\u0126':'H','\u0125':'h','\u0127':'h','\u0128':'I','\u012a':'I','\u012c':'I','\u012e':'I','\u0130':'I','\u0129':'i','\u012b':'i','\u012d':'i','\u012f':'i','\u0131':'i','\u0134':'J','\u0135':'j','\u0136':'K','\u0137':'k','\u0138':'k','\u0139':'L','\u013b':'L','\u013d':'L','\u013f':'L','\u0141':'L','\u013a':'l','\u013c':'l','\u013e':'l','\u0140':'l','\u0142':'l','\u0143':'N','\u0145':'N','\u0147':'N','\u014a':'N','\u0144':'n','\u0146':'n','\u0148':'n','\u014b':'n','\u014c':'O','\u014e':'O','\u0150':'O','\u014d':'o','\u014f':'o','\u0151':'o','\u0154':'R','\u0156':'R','\u0158':'R','\u0155':'r','\u0157':'r','\u0159':'r','\u015a':'S','\u015c':'S','\u015e':'S','\u0160':'S','\u015b':'s','\u015d':'s','\u015f':'s','\u0161':'s','\u0162':'T','\u0164':'T','\u0166':'T','\u0163':'t','\u0165':'t','\u0167':'t','\u0168':'U','\u016a':'U','\u016c':'U','\u016e':'U','\u0170':'U','\u0172':'U','\u0169':'u','\u016b':'u','\u016d':'u','\u016f':'u','\u0171':'u','\u0173':'u','\u0174':'W','\u0175':'w','\u0176':'Y','\u0177':'y','\u0178':'Y','\u0179':'Z','\u017b':'Z','\u017d':'Z','\u017a':'z','\u017c':'z','\u017e':'z','\u0132':'IJ','\u0133':'ij','\u0152':'Oe','\u0153':'oe','\u0149':"'n",'\u017f':'s'};/** Used to map characters to HTML entities. */var htmlEscapes={'&':'&amp;','<':'&lt;','>':'&gt;','"':'&quot;',"'":'&#39;'};/** Used to map HTML entities to characters. */var htmlUnescapes={'&amp;':'&','&lt;':'<','&gt;':'>','&quot;':'"','&#39;':"'"};/** Used to escape characters for inclusion in compiled string literals. */var stringEscapes={'\\':'\\',"'":"'",'\n':'n','\r':'r','\u2028':'u2028','\u2029':'u2029'};/** Built-in method references without a dependency on `root`. */var freeParseFloat=parseFloat,freeParseInt=parseInt;/** Detect free variable `global` from Node.js. */var freeGlobal=typeof commonjsGlobal=='object'&&commonjsGlobal&&commonjsGlobal.Object===Object&&commonjsGlobal;/** Detect free variable `self`. */var freeSelf=typeof self=='object'&&self&&self.Object===Object&&self;/** Used as a reference to the global object. */var root=freeGlobal||freeSelf||Function('return this')();/** Detect free variable `exports`. */var freeExports=exports&&!exports.nodeType&&exports;/** Detect free variable `module`. */var freeModule=freeExports&&'object'=='object'&&module&&!module.nodeType&&module;/** Detect the popular CommonJS extension `module.exports`. */var moduleExports=freeModule&&freeModule.exports===freeExports;/** Detect free variable `process` from Node.js. */var freeProcess=moduleExports&&freeGlobal.process;/** Used to access faster Node.js helpers. */var nodeUtil=function(){try{// Use `util.types` for Node.js 10+.
    	var types=freeModule&&freeModule.require&&freeModule.require('util').types;if(types){return types;}// Legacy `process.binding('util')` for Node.js < 10.
    	return freeProcess&&freeProcess.binding&&freeProcess.binding('util');}catch(e){}}();/* Node.js helper references. */var nodeIsArrayBuffer=nodeUtil&&nodeUtil.isArrayBuffer,nodeIsDate=nodeUtil&&nodeUtil.isDate,nodeIsMap=nodeUtil&&nodeUtil.isMap,nodeIsRegExp=nodeUtil&&nodeUtil.isRegExp,nodeIsSet=nodeUtil&&nodeUtil.isSet,nodeIsTypedArray=nodeUtil&&nodeUtil.isTypedArray;/*--------------------------------------------------------------------------*//**
    	   * A faster alternative to `Function#apply`, this function invokes `func`
    	   * with the `this` binding of `thisArg` and the arguments of `args`.
    	   *
    	   * @private
    	   * @param {Function} func The function to invoke.
    	   * @param {*} thisArg The `this` binding of `func`.
    	   * @param {Array} args The arguments to invoke `func` with.
    	   * @returns {*} Returns the result of `func`.
    	   */function apply(func,thisArg,args){switch(args.length){case 0:return func.call(thisArg);case 1:return func.call(thisArg,args[0]);case 2:return func.call(thisArg,args[0],args[1]);case 3:return func.call(thisArg,args[0],args[1],args[2]);}return func.apply(thisArg,args);}/**
    	   * A specialized version of `baseAggregator` for arrays.
    	   *
    	   * @private
    	   * @param {Array} [array] The array to iterate over.
    	   * @param {Function} setter The function to set `accumulator` values.
    	   * @param {Function} iteratee The iteratee to transform keys.
    	   * @param {Object} accumulator The initial aggregated object.
    	   * @returns {Function} Returns `accumulator`.
    	   */function arrayAggregator(array,setter,iteratee,accumulator){var index=-1,length=array==null?0:array.length;while(++index<length){var value=array[index];setter(accumulator,value,iteratee(value),array);}return accumulator;}/**
    	   * A specialized version of `_.forEach` for arrays without support for
    	   * iteratee shorthands.
    	   *
    	   * @private
    	   * @param {Array} [array] The array to iterate over.
    	   * @param {Function} iteratee The function invoked per iteration.
    	   * @returns {Array} Returns `array`.
    	   */function arrayEach(array,iteratee){var index=-1,length=array==null?0:array.length;while(++index<length){if(iteratee(array[index],index,array)===false){break;}}return array;}/**
    	   * A specialized version of `_.forEachRight` for arrays without support for
    	   * iteratee shorthands.
    	   *
    	   * @private
    	   * @param {Array} [array] The array to iterate over.
    	   * @param {Function} iteratee The function invoked per iteration.
    	   * @returns {Array} Returns `array`.
    	   */function arrayEachRight(array,iteratee){var length=array==null?0:array.length;while(length--){if(iteratee(array[length],length,array)===false){break;}}return array;}/**
    	   * A specialized version of `_.every` for arrays without support for
    	   * iteratee shorthands.
    	   *
    	   * @private
    	   * @param {Array} [array] The array to iterate over.
    	   * @param {Function} predicate The function invoked per iteration.
    	   * @returns {boolean} Returns `true` if all elements pass the predicate check,
    	   *  else `false`.
    	   */function arrayEvery(array,predicate){var index=-1,length=array==null?0:array.length;while(++index<length){if(!predicate(array[index],index,array)){return false;}}return true;}/**
    	   * A specialized version of `_.filter` for arrays without support for
    	   * iteratee shorthands.
    	   *
    	   * @private
    	   * @param {Array} [array] The array to iterate over.
    	   * @param {Function} predicate The function invoked per iteration.
    	   * @returns {Array} Returns the new filtered array.
    	   */function arrayFilter(array,predicate){var index=-1,length=array==null?0:array.length,resIndex=0,result=[];while(++index<length){var value=array[index];if(predicate(value,index,array)){result[resIndex++]=value;}}return result;}/**
    	   * A specialized version of `_.includes` for arrays without support for
    	   * specifying an index to search from.
    	   *
    	   * @private
    	   * @param {Array} [array] The array to inspect.
    	   * @param {*} target The value to search for.
    	   * @returns {boolean} Returns `true` if `target` is found, else `false`.
    	   */function arrayIncludes(array,value){var length=array==null?0:array.length;return !!length&&baseIndexOf(array,value,0)>-1;}/**
    	   * This function is like `arrayIncludes` except that it accepts a comparator.
    	   *
    	   * @private
    	   * @param {Array} [array] The array to inspect.
    	   * @param {*} target The value to search for.
    	   * @param {Function} comparator The comparator invoked per element.
    	   * @returns {boolean} Returns `true` if `target` is found, else `false`.
    	   */function arrayIncludesWith(array,value,comparator){var index=-1,length=array==null?0:array.length;while(++index<length){if(comparator(value,array[index])){return true;}}return false;}/**
    	   * A specialized version of `_.map` for arrays without support for iteratee
    	   * shorthands.
    	   *
    	   * @private
    	   * @param {Array} [array] The array to iterate over.
    	   * @param {Function} iteratee The function invoked per iteration.
    	   * @returns {Array} Returns the new mapped array.
    	   */function arrayMap(array,iteratee){var index=-1,length=array==null?0:array.length,result=Array(length);while(++index<length){result[index]=iteratee(array[index],index,array);}return result;}/**
    	   * Appends the elements of `values` to `array`.
    	   *
    	   * @private
    	   * @param {Array} array The array to modify.
    	   * @param {Array} values The values to append.
    	   * @returns {Array} Returns `array`.
    	   */function arrayPush(array,values){var index=-1,length=values.length,offset=array.length;while(++index<length){array[offset+index]=values[index];}return array;}/**
    	   * A specialized version of `_.reduce` for arrays without support for
    	   * iteratee shorthands.
    	   *
    	   * @private
    	   * @param {Array} [array] The array to iterate over.
    	   * @param {Function} iteratee The function invoked per iteration.
    	   * @param {*} [accumulator] The initial value.
    	   * @param {boolean} [initAccum] Specify using the first element of `array` as
    	   *  the initial value.
    	   * @returns {*} Returns the accumulated value.
    	   */function arrayReduce(array,iteratee,accumulator,initAccum){var index=-1,length=array==null?0:array.length;if(initAccum&&length){accumulator=array[++index];}while(++index<length){accumulator=iteratee(accumulator,array[index],index,array);}return accumulator;}/**
    	   * A specialized version of `_.reduceRight` for arrays without support for
    	   * iteratee shorthands.
    	   *
    	   * @private
    	   * @param {Array} [array] The array to iterate over.
    	   * @param {Function} iteratee The function invoked per iteration.
    	   * @param {*} [accumulator] The initial value.
    	   * @param {boolean} [initAccum] Specify using the last element of `array` as
    	   *  the initial value.
    	   * @returns {*} Returns the accumulated value.
    	   */function arrayReduceRight(array,iteratee,accumulator,initAccum){var length=array==null?0:array.length;if(initAccum&&length){accumulator=array[--length];}while(length--){accumulator=iteratee(accumulator,array[length],length,array);}return accumulator;}/**
    	   * A specialized version of `_.some` for arrays without support for iteratee
    	   * shorthands.
    	   *
    	   * @private
    	   * @param {Array} [array] The array to iterate over.
    	   * @param {Function} predicate The function invoked per iteration.
    	   * @returns {boolean} Returns `true` if any element passes the predicate check,
    	   *  else `false`.
    	   */function arraySome(array,predicate){var index=-1,length=array==null?0:array.length;while(++index<length){if(predicate(array[index],index,array)){return true;}}return false;}/**
    	   * Gets the size of an ASCII `string`.
    	   *
    	   * @private
    	   * @param {string} string The string inspect.
    	   * @returns {number} Returns the string size.
    	   */var asciiSize=baseProperty('length');/**
    	   * Converts an ASCII `string` to an array.
    	   *
    	   * @private
    	   * @param {string} string The string to convert.
    	   * @returns {Array} Returns the converted array.
    	   */function asciiToArray(string){return string.split('');}/**
    	   * Splits an ASCII `string` into an array of its words.
    	   *
    	   * @private
    	   * @param {string} The string to inspect.
    	   * @returns {Array} Returns the words of `string`.
    	   */function asciiWords(string){return string.match(reAsciiWord)||[];}/**
    	   * The base implementation of methods like `_.findKey` and `_.findLastKey`,
    	   * without support for iteratee shorthands, which iterates over `collection`
    	   * using `eachFunc`.
    	   *
    	   * @private
    	   * @param {Array|Object} collection The collection to inspect.
    	   * @param {Function} predicate The function invoked per iteration.
    	   * @param {Function} eachFunc The function to iterate over `collection`.
    	   * @returns {*} Returns the found element or its key, else `undefined`.
    	   */function baseFindKey(collection,predicate,eachFunc){var result;eachFunc(collection,function(value,key,collection){if(predicate(value,key,collection)){result=key;return false;}});return result;}/**
    	   * The base implementation of `_.findIndex` and `_.findLastIndex` without
    	   * support for iteratee shorthands.
    	   *
    	   * @private
    	   * @param {Array} array The array to inspect.
    	   * @param {Function} predicate The function invoked per iteration.
    	   * @param {number} fromIndex The index to search from.
    	   * @param {boolean} [fromRight] Specify iterating from right to left.
    	   * @returns {number} Returns the index of the matched value, else `-1`.
    	   */function baseFindIndex(array,predicate,fromIndex,fromRight){var length=array.length,index=fromIndex+(fromRight?1:-1);while(fromRight?index--:++index<length){if(predicate(array[index],index,array)){return index;}}return -1;}/**
    	   * The base implementation of `_.indexOf` without `fromIndex` bounds checks.
    	   *
    	   * @private
    	   * @param {Array} array The array to inspect.
    	   * @param {*} value The value to search for.
    	   * @param {number} fromIndex The index to search from.
    	   * @returns {number} Returns the index of the matched value, else `-1`.
    	   */function baseIndexOf(array,value,fromIndex){return value===value?strictIndexOf(array,value,fromIndex):baseFindIndex(array,baseIsNaN,fromIndex);}/**
    	   * This function is like `baseIndexOf` except that it accepts a comparator.
    	   *
    	   * @private
    	   * @param {Array} array The array to inspect.
    	   * @param {*} value The value to search for.
    	   * @param {number} fromIndex The index to search from.
    	   * @param {Function} comparator The comparator invoked per element.
    	   * @returns {number} Returns the index of the matched value, else `-1`.
    	   */function baseIndexOfWith(array,value,fromIndex,comparator){var index=fromIndex-1,length=array.length;while(++index<length){if(comparator(array[index],value)){return index;}}return -1;}/**
    	   * The base implementation of `_.isNaN` without support for number objects.
    	   *
    	   * @private
    	   * @param {*} value The value to check.
    	   * @returns {boolean} Returns `true` if `value` is `NaN`, else `false`.
    	   */function baseIsNaN(value){return value!==value;}/**
    	   * The base implementation of `_.mean` and `_.meanBy` without support for
    	   * iteratee shorthands.
    	   *
    	   * @private
    	   * @param {Array} array The array to iterate over.
    	   * @param {Function} iteratee The function invoked per iteration.
    	   * @returns {number} Returns the mean.
    	   */function baseMean(array,iteratee){var length=array==null?0:array.length;return length?baseSum(array,iteratee)/length:NAN;}/**
    	   * The base implementation of `_.property` without support for deep paths.
    	   *
    	   * @private
    	   * @param {string} key The key of the property to get.
    	   * @returns {Function} Returns the new accessor function.
    	   */function baseProperty(key){return function(object){return object==null?undefined$1:object[key];};}/**
    	   * The base implementation of `_.propertyOf` without support for deep paths.
    	   *
    	   * @private
    	   * @param {Object} object The object to query.
    	   * @returns {Function} Returns the new accessor function.
    	   */function basePropertyOf(object){return function(key){return object==null?undefined$1:object[key];};}/**
    	   * The base implementation of `_.reduce` and `_.reduceRight`, without support
    	   * for iteratee shorthands, which iterates over `collection` using `eachFunc`.
    	   *
    	   * @private
    	   * @param {Array|Object} collection The collection to iterate over.
    	   * @param {Function} iteratee The function invoked per iteration.
    	   * @param {*} accumulator The initial value.
    	   * @param {boolean} initAccum Specify using the first or last element of
    	   *  `collection` as the initial value.
    	   * @param {Function} eachFunc The function to iterate over `collection`.
    	   * @returns {*} Returns the accumulated value.
    	   */function baseReduce(collection,iteratee,accumulator,initAccum,eachFunc){eachFunc(collection,function(value,index,collection){accumulator=initAccum?(initAccum=false,value):iteratee(accumulator,value,index,collection);});return accumulator;}/**
    	   * The base implementation of `_.sortBy` which uses `comparer` to define the
    	   * sort order of `array` and replaces criteria objects with their corresponding
    	   * values.
    	   *
    	   * @private
    	   * @param {Array} array The array to sort.
    	   * @param {Function} comparer The function to define sort order.
    	   * @returns {Array} Returns `array`.
    	   */function baseSortBy(array,comparer){var length=array.length;array.sort(comparer);while(length--){array[length]=array[length].value;}return array;}/**
    	   * The base implementation of `_.sum` and `_.sumBy` without support for
    	   * iteratee shorthands.
    	   *
    	   * @private
    	   * @param {Array} array The array to iterate over.
    	   * @param {Function} iteratee The function invoked per iteration.
    	   * @returns {number} Returns the sum.
    	   */function baseSum(array,iteratee){var result,index=-1,length=array.length;while(++index<length){var current=iteratee(array[index]);if(current!==undefined$1){result=result===undefined$1?current:result+current;}}return result;}/**
    	   * The base implementation of `_.times` without support for iteratee shorthands
    	   * or max array length checks.
    	   *
    	   * @private
    	   * @param {number} n The number of times to invoke `iteratee`.
    	   * @param {Function} iteratee The function invoked per iteration.
    	   * @returns {Array} Returns the array of results.
    	   */function baseTimes(n,iteratee){var index=-1,result=Array(n);while(++index<n){result[index]=iteratee(index);}return result;}/**
    	   * The base implementation of `_.toPairs` and `_.toPairsIn` which creates an array
    	   * of key-value pairs for `object` corresponding to the property names of `props`.
    	   *
    	   * @private
    	   * @param {Object} object The object to query.
    	   * @param {Array} props The property names to get values for.
    	   * @returns {Object} Returns the key-value pairs.
    	   */function baseToPairs(object,props){return arrayMap(props,function(key){return [key,object[key]];});}/**
    	   * The base implementation of `_.trim`.
    	   *
    	   * @private
    	   * @param {string} string The string to trim.
    	   * @returns {string} Returns the trimmed string.
    	   */function baseTrim(string){return string?string.slice(0,trimmedEndIndex(string)+1).replace(reTrimStart,''):string;}/**
    	   * The base implementation of `_.unary` without support for storing metadata.
    	   *
    	   * @private
    	   * @param {Function} func The function to cap arguments for.
    	   * @returns {Function} Returns the new capped function.
    	   */function baseUnary(func){return function(value){return func(value);};}/**
    	   * The base implementation of `_.values` and `_.valuesIn` which creates an
    	   * array of `object` property values corresponding to the property names
    	   * of `props`.
    	   *
    	   * @private
    	   * @param {Object} object The object to query.
    	   * @param {Array} props The property names to get values for.
    	   * @returns {Object} Returns the array of property values.
    	   */function baseValues(object,props){return arrayMap(props,function(key){return object[key];});}/**
    	   * Checks if a `cache` value for `key` exists.
    	   *
    	   * @private
    	   * @param {Object} cache The cache to query.
    	   * @param {string} key The key of the entry to check.
    	   * @returns {boolean} Returns `true` if an entry for `key` exists, else `false`.
    	   */function cacheHas(cache,key){return cache.has(key);}/**
    	   * Used by `_.trim` and `_.trimStart` to get the index of the first string symbol
    	   * that is not found in the character symbols.
    	   *
    	   * @private
    	   * @param {Array} strSymbols The string symbols to inspect.
    	   * @param {Array} chrSymbols The character symbols to find.
    	   * @returns {number} Returns the index of the first unmatched string symbol.
    	   */function charsStartIndex(strSymbols,chrSymbols){var index=-1,length=strSymbols.length;while(++index<length&&baseIndexOf(chrSymbols,strSymbols[index],0)>-1){}return index;}/**
    	   * Used by `_.trim` and `_.trimEnd` to get the index of the last string symbol
    	   * that is not found in the character symbols.
    	   *
    	   * @private
    	   * @param {Array} strSymbols The string symbols to inspect.
    	   * @param {Array} chrSymbols The character symbols to find.
    	   * @returns {number} Returns the index of the last unmatched string symbol.
    	   */function charsEndIndex(strSymbols,chrSymbols){var index=strSymbols.length;while(index--&&baseIndexOf(chrSymbols,strSymbols[index],0)>-1){}return index;}/**
    	   * Gets the number of `placeholder` occurrences in `array`.
    	   *
    	   * @private
    	   * @param {Array} array The array to inspect.
    	   * @param {*} placeholder The placeholder to search for.
    	   * @returns {number} Returns the placeholder count.
    	   */function countHolders(array,placeholder){var length=array.length,result=0;while(length--){if(array[length]===placeholder){++result;}}return result;}/**
    	   * Used by `_.deburr` to convert Latin-1 Supplement and Latin Extended-A
    	   * letters to basic Latin letters.
    	   *
    	   * @private
    	   * @param {string} letter The matched letter to deburr.
    	   * @returns {string} Returns the deburred letter.
    	   */var deburrLetter=basePropertyOf(deburredLetters);/**
    	   * Used by `_.escape` to convert characters to HTML entities.
    	   *
    	   * @private
    	   * @param {string} chr The matched character to escape.
    	   * @returns {string} Returns the escaped character.
    	   */var escapeHtmlChar=basePropertyOf(htmlEscapes);/**
    	   * Used by `_.template` to escape characters for inclusion in compiled string literals.
    	   *
    	   * @private
    	   * @param {string} chr The matched character to escape.
    	   * @returns {string} Returns the escaped character.
    	   */function escapeStringChar(chr){return '\\'+stringEscapes[chr];}/**
    	   * Gets the value at `key` of `object`.
    	   *
    	   * @private
    	   * @param {Object} [object] The object to query.
    	   * @param {string} key The key of the property to get.
    	   * @returns {*} Returns the property value.
    	   */function getValue(object,key){return object==null?undefined$1:object[key];}/**
    	   * Checks if `string` contains Unicode symbols.
    	   *
    	   * @private
    	   * @param {string} string The string to inspect.
    	   * @returns {boolean} Returns `true` if a symbol is found, else `false`.
    	   */function hasUnicode(string){return reHasUnicode.test(string);}/**
    	   * Checks if `string` contains a word composed of Unicode symbols.
    	   *
    	   * @private
    	   * @param {string} string The string to inspect.
    	   * @returns {boolean} Returns `true` if a word is found, else `false`.
    	   */function hasUnicodeWord(string){return reHasUnicodeWord.test(string);}/**
    	   * Converts `iterator` to an array.
    	   *
    	   * @private
    	   * @param {Object} iterator The iterator to convert.
    	   * @returns {Array} Returns the converted array.
    	   */function iteratorToArray(iterator){var data,result=[];while(!(data=iterator.next()).done){result.push(data.value);}return result;}/**
    	   * Converts `map` to its key-value pairs.
    	   *
    	   * @private
    	   * @param {Object} map The map to convert.
    	   * @returns {Array} Returns the key-value pairs.
    	   */function mapToArray(map){var index=-1,result=Array(map.size);map.forEach(function(value,key){result[++index]=[key,value];});return result;}/**
    	   * Creates a unary function that invokes `func` with its argument transformed.
    	   *
    	   * @private
    	   * @param {Function} func The function to wrap.
    	   * @param {Function} transform The argument transform.
    	   * @returns {Function} Returns the new function.
    	   */function overArg(func,transform){return function(arg){return func(transform(arg));};}/**
    	   * Replaces all `placeholder` elements in `array` with an internal placeholder
    	   * and returns an array of their indexes.
    	   *
    	   * @private
    	   * @param {Array} array The array to modify.
    	   * @param {*} placeholder The placeholder to replace.
    	   * @returns {Array} Returns the new array of placeholder indexes.
    	   */function replaceHolders(array,placeholder){var index=-1,length=array.length,resIndex=0,result=[];while(++index<length){var value=array[index];if(value===placeholder||value===PLACEHOLDER){array[index]=PLACEHOLDER;result[resIndex++]=index;}}return result;}/**
    	   * Converts `set` to an array of its values.
    	   *
    	   * @private
    	   * @param {Object} set The set to convert.
    	   * @returns {Array} Returns the values.
    	   */function setToArray(set){var index=-1,result=Array(set.size);set.forEach(function(value){result[++index]=value;});return result;}/**
    	   * Converts `set` to its value-value pairs.
    	   *
    	   * @private
    	   * @param {Object} set The set to convert.
    	   * @returns {Array} Returns the value-value pairs.
    	   */function setToPairs(set){var index=-1,result=Array(set.size);set.forEach(function(value){result[++index]=[value,value];});return result;}/**
    	   * A specialized version of `_.indexOf` which performs strict equality
    	   * comparisons of values, i.e. `===`.
    	   *
    	   * @private
    	   * @param {Array} array The array to inspect.
    	   * @param {*} value The value to search for.
    	   * @param {number} fromIndex The index to search from.
    	   * @returns {number} Returns the index of the matched value, else `-1`.
    	   */function strictIndexOf(array,value,fromIndex){var index=fromIndex-1,length=array.length;while(++index<length){if(array[index]===value){return index;}}return -1;}/**
    	   * A specialized version of `_.lastIndexOf` which performs strict equality
    	   * comparisons of values, i.e. `===`.
    	   *
    	   * @private
    	   * @param {Array} array The array to inspect.
    	   * @param {*} value The value to search for.
    	   * @param {number} fromIndex The index to search from.
    	   * @returns {number} Returns the index of the matched value, else `-1`.
    	   */function strictLastIndexOf(array,value,fromIndex){var index=fromIndex+1;while(index--){if(array[index]===value){return index;}}return index;}/**
    	   * Gets the number of symbols in `string`.
    	   *
    	   * @private
    	   * @param {string} string The string to inspect.
    	   * @returns {number} Returns the string size.
    	   */function stringSize(string){return hasUnicode(string)?unicodeSize(string):asciiSize(string);}/**
    	   * Converts `string` to an array.
    	   *
    	   * @private
    	   * @param {string} string The string to convert.
    	   * @returns {Array} Returns the converted array.
    	   */function stringToArray(string){return hasUnicode(string)?unicodeToArray(string):asciiToArray(string);}/**
    	   * Used by `_.trim` and `_.trimEnd` to get the index of the last non-whitespace
    	   * character of `string`.
    	   *
    	   * @private
    	   * @param {string} string The string to inspect.
    	   * @returns {number} Returns the index of the last non-whitespace character.
    	   */function trimmedEndIndex(string){var index=string.length;while(index--&&reWhitespace.test(string.charAt(index))){}return index;}/**
    	   * Used by `_.unescape` to convert HTML entities to characters.
    	   *
    	   * @private
    	   * @param {string} chr The matched character to unescape.
    	   * @returns {string} Returns the unescaped character.
    	   */var unescapeHtmlChar=basePropertyOf(htmlUnescapes);/**
    	   * Gets the size of a Unicode `string`.
    	   *
    	   * @private
    	   * @param {string} string The string inspect.
    	   * @returns {number} Returns the string size.
    	   */function unicodeSize(string){var result=reUnicode.lastIndex=0;while(reUnicode.test(string)){++result;}return result;}/**
    	   * Converts a Unicode `string` to an array.
    	   *
    	   * @private
    	   * @param {string} string The string to convert.
    	   * @returns {Array} Returns the converted array.
    	   */function unicodeToArray(string){return string.match(reUnicode)||[];}/**
    	   * Splits a Unicode `string` into an array of its words.
    	   *
    	   * @private
    	   * @param {string} The string to inspect.
    	   * @returns {Array} Returns the words of `string`.
    	   */function unicodeWords(string){return string.match(reUnicodeWord)||[];}/*--------------------------------------------------------------------------*//**
    	   * Create a new pristine `lodash` function using the `context` object.
    	   *
    	   * @static
    	   * @memberOf _
    	   * @since 1.1.0
    	   * @category Util
    	   * @param {Object} [context=root] The context object.
    	   * @returns {Function} Returns a new `lodash` function.
    	   * @example
    	   *
    	   * _.mixin({ 'foo': _.constant('foo') });
    	   *
    	   * var lodash = _.runInContext();
    	   * lodash.mixin({ 'bar': lodash.constant('bar') });
    	   *
    	   * _.isFunction(_.foo);
    	   * // => true
    	   * _.isFunction(_.bar);
    	   * // => false
    	   *
    	   * lodash.isFunction(lodash.foo);
    	   * // => false
    	   * lodash.isFunction(lodash.bar);
    	   * // => true
    	   *
    	   * // Create a suped-up `defer` in Node.js.
    	   * var defer = _.runInContext({ 'setTimeout': setImmediate }).defer;
    	   */var runInContext=function runInContext(context){context=context==null?root:_.defaults(root.Object(),context,_.pick(root,contextProps));/** Built-in constructor references. */var Array=context.Array,Date=context.Date,Error=context.Error,Function=context.Function,Math=context.Math,Object=context.Object,RegExp=context.RegExp,String=context.String,TypeError=context.TypeError;/** Used for built-in method references. */var arrayProto=Array.prototype,funcProto=Function.prototype,objectProto=Object.prototype;/** Used to detect overreaching core-js shims. */var coreJsData=context['__core-js_shared__'];/** Used to resolve the decompiled source of functions. */var funcToString=funcProto.toString;/** Used to check objects for own properties. */var hasOwnProperty=objectProto.hasOwnProperty;/** Used to generate unique IDs. */var idCounter=0;/** Used to detect methods masquerading as native. */var maskSrcKey=function(){var uid=/[^.]+$/.exec(coreJsData&&coreJsData.keys&&coreJsData.keys.IE_PROTO||'');return uid?'Symbol(src)_1.'+uid:'';}();/**
    	     * Used to resolve the
    	     * [`toStringTag`](http://ecma-international.org/ecma-262/7.0/#sec-object.prototype.tostring)
    	     * of values.
    	     */var nativeObjectToString=objectProto.toString;/** Used to infer the `Object` constructor. */var objectCtorString=funcToString.call(Object);/** Used to restore the original `_` reference in `_.noConflict`. */var oldDash=root._;/** Used to detect if a method is native. */var reIsNative=RegExp('^'+funcToString.call(hasOwnProperty).replace(reRegExpChar,'\\$&').replace(/hasOwnProperty|(function).*?(?=\\\()| for .+?(?=\\\])/g,'$1.*?')+'$');/** Built-in value references. */var Buffer=moduleExports?context.Buffer:undefined$1,Symbol=context.Symbol,Uint8Array=context.Uint8Array,allocUnsafe=Buffer?Buffer.allocUnsafe:undefined$1,getPrototype=overArg(Object.getPrototypeOf,Object),objectCreate=Object.create,propertyIsEnumerable=objectProto.propertyIsEnumerable,splice=arrayProto.splice,spreadableSymbol=Symbol?Symbol.isConcatSpreadable:undefined$1,symIterator=Symbol?Symbol.iterator:undefined$1,symToStringTag=Symbol?Symbol.toStringTag:undefined$1;var defineProperty=function(){try{var func=getNative(Object,'defineProperty');func({},'',{});return func;}catch(e){}}();/** Mocked built-ins. */var ctxClearTimeout=context.clearTimeout!==root.clearTimeout&&context.clearTimeout,ctxNow=Date&&Date.now!==root.Date.now&&Date.now,ctxSetTimeout=context.setTimeout!==root.setTimeout&&context.setTimeout;/* Built-in method references for those with the same name as other `lodash` methods. */var nativeCeil=Math.ceil,nativeFloor=Math.floor,nativeGetSymbols=Object.getOwnPropertySymbols,nativeIsBuffer=Buffer?Buffer.isBuffer:undefined$1,nativeIsFinite=context.isFinite,nativeJoin=arrayProto.join,nativeKeys=overArg(Object.keys,Object),nativeMax=Math.max,nativeMin=Math.min,nativeNow=Date.now,nativeParseInt=context.parseInt,nativeRandom=Math.random,nativeReverse=arrayProto.reverse;/* Built-in method references that are verified to be native. */var DataView=getNative(context,'DataView'),Map=getNative(context,'Map'),Promise=getNative(context,'Promise'),Set=getNative(context,'Set'),WeakMap=getNative(context,'WeakMap'),nativeCreate=getNative(Object,'create');/** Used to store function metadata. */var metaMap=WeakMap&&new WeakMap();/** Used to lookup unminified function names. */var realNames={};/** Used to detect maps, sets, and weakmaps. */var dataViewCtorString=toSource(DataView),mapCtorString=toSource(Map),promiseCtorString=toSource(Promise),setCtorString=toSource(Set),weakMapCtorString=toSource(WeakMap);/** Used to convert symbols to primitives and strings. */var symbolProto=Symbol?Symbol.prototype:undefined$1,symbolValueOf=symbolProto?symbolProto.valueOf:undefined$1,symbolToString=symbolProto?symbolProto.toString:undefined$1;/*------------------------------------------------------------------------*//**
    	     * Creates a `lodash` object which wraps `value` to enable implicit method
    	     * chain sequences. Methods that operate on and return arrays, collections,
    	     * and functions can be chained together. Methods that retrieve a single value
    	     * or may return a primitive value will automatically end the chain sequence
    	     * and return the unwrapped value. Otherwise, the value must be unwrapped
    	     * with `_#value`.
    	     *
    	     * Explicit chain sequences, which must be unwrapped with `_#value`, may be
    	     * enabled using `_.chain`.
    	     *
    	     * The execution of chained methods is lazy, that is, it's deferred until
    	     * `_#value` is implicitly or explicitly called.
    	     *
    	     * Lazy evaluation allows several methods to support shortcut fusion.
    	     * Shortcut fusion is an optimization to merge iteratee calls; this avoids
    	     * the creation of intermediate arrays and can greatly reduce the number of
    	     * iteratee executions. Sections of a chain sequence qualify for shortcut
    	     * fusion if the section is applied to an array and iteratees accept only
    	     * one argument. The heuristic for whether a section qualifies for shortcut
    	     * fusion is subject to change.
    	     *
    	     * Chaining is supported in custom builds as long as the `_#value` method is
    	     * directly or indirectly included in the build.
    	     *
    	     * In addition to lodash methods, wrappers have `Array` and `String` methods.
    	     *
    	     * The wrapper `Array` methods are:
    	     * `concat`, `join`, `pop`, `push`, `shift`, `sort`, `splice`, and `unshift`
    	     *
    	     * The wrapper `String` methods are:
    	     * `replace` and `split`
    	     *
    	     * The wrapper methods that support shortcut fusion are:
    	     * `at`, `compact`, `drop`, `dropRight`, `dropWhile`, `filter`, `find`,
    	     * `findLast`, `head`, `initial`, `last`, `map`, `reject`, `reverse`, `slice`,
    	     * `tail`, `take`, `takeRight`, `takeRightWhile`, `takeWhile`, and `toArray`
    	     *
    	     * The chainable wrapper methods are:
    	     * `after`, `ary`, `assign`, `assignIn`, `assignInWith`, `assignWith`, `at`,
    	     * `before`, `bind`, `bindAll`, `bindKey`, `castArray`, `chain`, `chunk`,
    	     * `commit`, `compact`, `concat`, `conforms`, `constant`, `countBy`, `create`,
    	     * `curry`, `debounce`, `defaults`, `defaultsDeep`, `defer`, `delay`,
    	     * `difference`, `differenceBy`, `differenceWith`, `drop`, `dropRight`,
    	     * `dropRightWhile`, `dropWhile`, `extend`, `extendWith`, `fill`, `filter`,
    	     * `flatMap`, `flatMapDeep`, `flatMapDepth`, `flatten`, `flattenDeep`,
    	     * `flattenDepth`, `flip`, `flow`, `flowRight`, `fromPairs`, `functions`,
    	     * `functionsIn`, `groupBy`, `initial`, `intersection`, `intersectionBy`,
    	     * `intersectionWith`, `invert`, `invertBy`, `invokeMap`, `iteratee`, `keyBy`,
    	     * `keys`, `keysIn`, `map`, `mapKeys`, `mapValues`, `matches`, `matchesProperty`,
    	     * `memoize`, `merge`, `mergeWith`, `method`, `methodOf`, `mixin`, `negate`,
    	     * `nthArg`, `omit`, `omitBy`, `once`, `orderBy`, `over`, `overArgs`,
    	     * `overEvery`, `overSome`, `partial`, `partialRight`, `partition`, `pick`,
    	     * `pickBy`, `plant`, `property`, `propertyOf`, `pull`, `pullAll`, `pullAllBy`,
    	     * `pullAllWith`, `pullAt`, `push`, `range`, `rangeRight`, `rearg`, `reject`,
    	     * `remove`, `rest`, `reverse`, `sampleSize`, `set`, `setWith`, `shuffle`,
    	     * `slice`, `sort`, `sortBy`, `splice`, `spread`, `tail`, `take`, `takeRight`,
    	     * `takeRightWhile`, `takeWhile`, `tap`, `throttle`, `thru`, `toArray`,
    	     * `toPairs`, `toPairsIn`, `toPath`, `toPlainObject`, `transform`, `unary`,
    	     * `union`, `unionBy`, `unionWith`, `uniq`, `uniqBy`, `uniqWith`, `unset`,
    	     * `unshift`, `unzip`, `unzipWith`, `update`, `updateWith`, `values`,
    	     * `valuesIn`, `without`, `wrap`, `xor`, `xorBy`, `xorWith`, `zip`,
    	     * `zipObject`, `zipObjectDeep`, and `zipWith`
    	     *
    	     * The wrapper methods that are **not** chainable by default are:
    	     * `add`, `attempt`, `camelCase`, `capitalize`, `ceil`, `clamp`, `clone`,
    	     * `cloneDeep`, `cloneDeepWith`, `cloneWith`, `conformsTo`, `deburr`,
    	     * `defaultTo`, `divide`, `each`, `eachRight`, `endsWith`, `eq`, `escape`,
    	     * `escapeRegExp`, `every`, `find`, `findIndex`, `findKey`, `findLast`,
    	     * `findLastIndex`, `findLastKey`, `first`, `floor`, `forEach`, `forEachRight`,
    	     * `forIn`, `forInRight`, `forOwn`, `forOwnRight`, `get`, `gt`, `gte`, `has`,
    	     * `hasIn`, `head`, `identity`, `includes`, `indexOf`, `inRange`, `invoke`,
    	     * `isArguments`, `isArray`, `isArrayBuffer`, `isArrayLike`, `isArrayLikeObject`,
    	     * `isBoolean`, `isBuffer`, `isDate`, `isElement`, `isEmpty`, `isEqual`,
    	     * `isEqualWith`, `isError`, `isFinite`, `isFunction`, `isInteger`, `isLength`,
    	     * `isMap`, `isMatch`, `isMatchWith`, `isNaN`, `isNative`, `isNil`, `isNull`,
    	     * `isNumber`, `isObject`, `isObjectLike`, `isPlainObject`, `isRegExp`,
    	     * `isSafeInteger`, `isSet`, `isString`, `isUndefined`, `isTypedArray`,
    	     * `isWeakMap`, `isWeakSet`, `join`, `kebabCase`, `last`, `lastIndexOf`,
    	     * `lowerCase`, `lowerFirst`, `lt`, `lte`, `max`, `maxBy`, `mean`, `meanBy`,
    	     * `min`, `minBy`, `multiply`, `noConflict`, `noop`, `now`, `nth`, `pad`,
    	     * `padEnd`, `padStart`, `parseInt`, `pop`, `random`, `reduce`, `reduceRight`,
    	     * `repeat`, `result`, `round`, `runInContext`, `sample`, `shift`, `size`,
    	     * `snakeCase`, `some`, `sortedIndex`, `sortedIndexBy`, `sortedLastIndex`,
    	     * `sortedLastIndexBy`, `startCase`, `startsWith`, `stubArray`, `stubFalse`,
    	     * `stubObject`, `stubString`, `stubTrue`, `subtract`, `sum`, `sumBy`,
    	     * `template`, `times`, `toFinite`, `toInteger`, `toJSON`, `toLength`,
    	     * `toLower`, `toNumber`, `toSafeInteger`, `toString`, `toUpper`, `trim`,
    	     * `trimEnd`, `trimStart`, `truncate`, `unescape`, `uniqueId`, `upperCase`,
    	     * `upperFirst`, `value`, and `words`
    	     *
    	     * @name _
    	     * @constructor
    	     * @category Seq
    	     * @param {*} value The value to wrap in a `lodash` instance.
    	     * @returns {Object} Returns the new `lodash` wrapper instance.
    	     * @example
    	     *
    	     * function square(n) {
    	     *   return n * n;
    	     * }
    	     *
    	     * var wrapped = _([1, 2, 3]);
    	     *
    	     * // Returns an unwrapped value.
    	     * wrapped.reduce(_.add);
    	     * // => 6
    	     *
    	     * // Returns a wrapped value.
    	     * var squares = wrapped.map(square);
    	     *
    	     * _.isArray(squares);
    	     * // => false
    	     *
    	     * _.isArray(squares.value());
    	     * // => true
    	     */function lodash(value){if(isObjectLike(value)&&!isArray(value)&&!(value instanceof LazyWrapper)){if(value instanceof LodashWrapper){return value;}if(hasOwnProperty.call(value,'__wrapped__')){return wrapperClone(value);}}return new LodashWrapper(value);}/**
    	     * The base implementation of `_.create` without support for assigning
    	     * properties to the created object.
    	     *
    	     * @private
    	     * @param {Object} proto The object to inherit from.
    	     * @returns {Object} Returns the new object.
    	     */var baseCreate=function(){function object(){}return function(proto){if(!isObject(proto)){return {};}if(objectCreate){return objectCreate(proto);}object.prototype=proto;var result=new object();object.prototype=undefined$1;return result;};}();/**
    	     * The function whose prototype chain sequence wrappers inherit from.
    	     *
    	     * @private
    	     */function baseLodash(){// No operation performed.
    	}/**
    	     * The base constructor for creating `lodash` wrapper objects.
    	     *
    	     * @private
    	     * @param {*} value The value to wrap.
    	     * @param {boolean} [chainAll] Enable explicit method chain sequences.
    	     */function LodashWrapper(value,chainAll){this.__wrapped__=value;this.__actions__=[];this.__chain__=!!chainAll;this.__index__=0;this.__values__=undefined$1;}/**
    	     * By default, the template delimiters used by lodash are like those in
    	     * embedded Ruby (ERB) as well as ES2015 template strings. Change the
    	     * following template settings to use alternative delimiters.
    	     *
    	     * @static
    	     * @memberOf _
    	     * @type {Object}
    	     */lodash.templateSettings={/**
    	       * Used to detect `data` property values to be HTML-escaped.
    	       *
    	       * @memberOf _.templateSettings
    	       * @type {RegExp}
    	       */'escape':reEscape,/**
    	       * Used to detect code to be evaluated.
    	       *
    	       * @memberOf _.templateSettings
    	       * @type {RegExp}
    	       */'evaluate':reEvaluate,/**
    	       * Used to detect `data` property values to inject.
    	       *
    	       * @memberOf _.templateSettings
    	       * @type {RegExp}
    	       */'interpolate':reInterpolate,/**
    	       * Used to reference the data object in the template text.
    	       *
    	       * @memberOf _.templateSettings
    	       * @type {string}
    	       */'variable':'',/**
    	       * Used to import variables into the compiled template.
    	       *
    	       * @memberOf _.templateSettings
    	       * @type {Object}
    	       */'imports':{/**
    	         * A reference to the `lodash` function.
    	         *
    	         * @memberOf _.templateSettings.imports
    	         * @type {Function}
    	         */'_':lodash}};// Ensure wrappers are instances of `baseLodash`.
    	lodash.prototype=baseLodash.prototype;lodash.prototype.constructor=lodash;LodashWrapper.prototype=baseCreate(baseLodash.prototype);LodashWrapper.prototype.constructor=LodashWrapper;/*------------------------------------------------------------------------*//**
    	     * Creates a lazy wrapper object which wraps `value` to enable lazy evaluation.
    	     *
    	     * @private
    	     * @constructor
    	     * @param {*} value The value to wrap.
    	     */function LazyWrapper(value){this.__wrapped__=value;this.__actions__=[];this.__dir__=1;this.__filtered__=false;this.__iteratees__=[];this.__takeCount__=MAX_ARRAY_LENGTH;this.__views__=[];}/**
    	     * Creates a clone of the lazy wrapper object.
    	     *
    	     * @private
    	     * @name clone
    	     * @memberOf LazyWrapper
    	     * @returns {Object} Returns the cloned `LazyWrapper` object.
    	     */function lazyClone(){var result=new LazyWrapper(this.__wrapped__);result.__actions__=copyArray(this.__actions__);result.__dir__=this.__dir__;result.__filtered__=this.__filtered__;result.__iteratees__=copyArray(this.__iteratees__);result.__takeCount__=this.__takeCount__;result.__views__=copyArray(this.__views__);return result;}/**
    	     * Reverses the direction of lazy iteration.
    	     *
    	     * @private
    	     * @name reverse
    	     * @memberOf LazyWrapper
    	     * @returns {Object} Returns the new reversed `LazyWrapper` object.
    	     */function lazyReverse(){if(this.__filtered__){var result=new LazyWrapper(this);result.__dir__=-1;result.__filtered__=true;}else {result=this.clone();result.__dir__*=-1;}return result;}/**
    	     * Extracts the unwrapped value from its lazy wrapper.
    	     *
    	     * @private
    	     * @name value
    	     * @memberOf LazyWrapper
    	     * @returns {*} Returns the unwrapped value.
    	     */function lazyValue(){var array=this.__wrapped__.value(),dir=this.__dir__,isArr=isArray(array),isRight=dir<0,arrLength=isArr?array.length:0,view=getView(0,arrLength,this.__views__),start=view.start,end=view.end,length=end-start,index=isRight?end:start-1,iteratees=this.__iteratees__,iterLength=iteratees.length,resIndex=0,takeCount=nativeMin(length,this.__takeCount__);if(!isArr||!isRight&&arrLength==length&&takeCount==length){return baseWrapperValue(array,this.__actions__);}var result=[];outer:while(length--&&resIndex<takeCount){index+=dir;var iterIndex=-1,value=array[index];while(++iterIndex<iterLength){var data=iteratees[iterIndex],iteratee=data.iteratee,type=data.type,computed=iteratee(value);if(type==LAZY_MAP_FLAG){value=computed;}else if(!computed){if(type==LAZY_FILTER_FLAG){continue outer;}else {break outer;}}}result[resIndex++]=value;}return result;}// Ensure `LazyWrapper` is an instance of `baseLodash`.
    	LazyWrapper.prototype=baseCreate(baseLodash.prototype);LazyWrapper.prototype.constructor=LazyWrapper;/*------------------------------------------------------------------------*//**
    	     * Creates a hash object.
    	     *
    	     * @private
    	     * @constructor
    	     * @param {Array} [entries] The key-value pairs to cache.
    	     */function Hash(entries){var index=-1,length=entries==null?0:entries.length;this.clear();while(++index<length){var entry=entries[index];this.set(entry[0],entry[1]);}}/**
    	     * Removes all key-value entries from the hash.
    	     *
    	     * @private
    	     * @name clear
    	     * @memberOf Hash
    	     */function hashClear(){this.__data__=nativeCreate?nativeCreate(null):{};this.size=0;}/**
    	     * Removes `key` and its value from the hash.
    	     *
    	     * @private
    	     * @name delete
    	     * @memberOf Hash
    	     * @param {Object} hash The hash to modify.
    	     * @param {string} key The key of the value to remove.
    	     * @returns {boolean} Returns `true` if the entry was removed, else `false`.
    	     */function hashDelete(key){var result=this.has(key)&&delete this.__data__[key];this.size-=result?1:0;return result;}/**
    	     * Gets the hash value for `key`.
    	     *
    	     * @private
    	     * @name get
    	     * @memberOf Hash
    	     * @param {string} key The key of the value to get.
    	     * @returns {*} Returns the entry value.
    	     */function hashGet(key){var data=this.__data__;if(nativeCreate){var result=data[key];return result===HASH_UNDEFINED?undefined$1:result;}return hasOwnProperty.call(data,key)?data[key]:undefined$1;}/**
    	     * Checks if a hash value for `key` exists.
    	     *
    	     * @private
    	     * @name has
    	     * @memberOf Hash
    	     * @param {string} key The key of the entry to check.
    	     * @returns {boolean} Returns `true` if an entry for `key` exists, else `false`.
    	     */function hashHas(key){var data=this.__data__;return nativeCreate?data[key]!==undefined$1:hasOwnProperty.call(data,key);}/**
    	     * Sets the hash `key` to `value`.
    	     *
    	     * @private
    	     * @name set
    	     * @memberOf Hash
    	     * @param {string} key The key of the value to set.
    	     * @param {*} value The value to set.
    	     * @returns {Object} Returns the hash instance.
    	     */function hashSet(key,value){var data=this.__data__;this.size+=this.has(key)?0:1;data[key]=nativeCreate&&value===undefined$1?HASH_UNDEFINED:value;return this;}// Add methods to `Hash`.
    	Hash.prototype.clear=hashClear;Hash.prototype['delete']=hashDelete;Hash.prototype.get=hashGet;Hash.prototype.has=hashHas;Hash.prototype.set=hashSet;/*------------------------------------------------------------------------*//**
    	     * Creates an list cache object.
    	     *
    	     * @private
    	     * @constructor
    	     * @param {Array} [entries] The key-value pairs to cache.
    	     */function ListCache(entries){var index=-1,length=entries==null?0:entries.length;this.clear();while(++index<length){var entry=entries[index];this.set(entry[0],entry[1]);}}/**
    	     * Removes all key-value entries from the list cache.
    	     *
    	     * @private
    	     * @name clear
    	     * @memberOf ListCache
    	     */function listCacheClear(){this.__data__=[];this.size=0;}/**
    	     * Removes `key` and its value from the list cache.
    	     *
    	     * @private
    	     * @name delete
    	     * @memberOf ListCache
    	     * @param {string} key The key of the value to remove.
    	     * @returns {boolean} Returns `true` if the entry was removed, else `false`.
    	     */function listCacheDelete(key){var data=this.__data__,index=assocIndexOf(data,key);if(index<0){return false;}var lastIndex=data.length-1;if(index==lastIndex){data.pop();}else {splice.call(data,index,1);}--this.size;return true;}/**
    	     * Gets the list cache value for `key`.
    	     *
    	     * @private
    	     * @name get
    	     * @memberOf ListCache
    	     * @param {string} key The key of the value to get.
    	     * @returns {*} Returns the entry value.
    	     */function listCacheGet(key){var data=this.__data__,index=assocIndexOf(data,key);return index<0?undefined$1:data[index][1];}/**
    	     * Checks if a list cache value for `key` exists.
    	     *
    	     * @private
    	     * @name has
    	     * @memberOf ListCache
    	     * @param {string} key The key of the entry to check.
    	     * @returns {boolean} Returns `true` if an entry for `key` exists, else `false`.
    	     */function listCacheHas(key){return assocIndexOf(this.__data__,key)>-1;}/**
    	     * Sets the list cache `key` to `value`.
    	     *
    	     * @private
    	     * @name set
    	     * @memberOf ListCache
    	     * @param {string} key The key of the value to set.
    	     * @param {*} value The value to set.
    	     * @returns {Object} Returns the list cache instance.
    	     */function listCacheSet(key,value){var data=this.__data__,index=assocIndexOf(data,key);if(index<0){++this.size;data.push([key,value]);}else {data[index][1]=value;}return this;}// Add methods to `ListCache`.
    	ListCache.prototype.clear=listCacheClear;ListCache.prototype['delete']=listCacheDelete;ListCache.prototype.get=listCacheGet;ListCache.prototype.has=listCacheHas;ListCache.prototype.set=listCacheSet;/*------------------------------------------------------------------------*//**
    	     * Creates a map cache object to store key-value pairs.
    	     *
    	     * @private
    	     * @constructor
    	     * @param {Array} [entries] The key-value pairs to cache.
    	     */function MapCache(entries){var index=-1,length=entries==null?0:entries.length;this.clear();while(++index<length){var entry=entries[index];this.set(entry[0],entry[1]);}}/**
    	     * Removes all key-value entries from the map.
    	     *
    	     * @private
    	     * @name clear
    	     * @memberOf MapCache
    	     */function mapCacheClear(){this.size=0;this.__data__={'hash':new Hash(),'map':new(Map||ListCache)(),'string':new Hash()};}/**
    	     * Removes `key` and its value from the map.
    	     *
    	     * @private
    	     * @name delete
    	     * @memberOf MapCache
    	     * @param {string} key The key of the value to remove.
    	     * @returns {boolean} Returns `true` if the entry was removed, else `false`.
    	     */function mapCacheDelete(key){var result=getMapData(this,key)['delete'](key);this.size-=result?1:0;return result;}/**
    	     * Gets the map value for `key`.
    	     *
    	     * @private
    	     * @name get
    	     * @memberOf MapCache
    	     * @param {string} key The key of the value to get.
    	     * @returns {*} Returns the entry value.
    	     */function mapCacheGet(key){return getMapData(this,key).get(key);}/**
    	     * Checks if a map value for `key` exists.
    	     *
    	     * @private
    	     * @name has
    	     * @memberOf MapCache
    	     * @param {string} key The key of the entry to check.
    	     * @returns {boolean} Returns `true` if an entry for `key` exists, else `false`.
    	     */function mapCacheHas(key){return getMapData(this,key).has(key);}/**
    	     * Sets the map `key` to `value`.
    	     *
    	     * @private
    	     * @name set
    	     * @memberOf MapCache
    	     * @param {string} key The key of the value to set.
    	     * @param {*} value The value to set.
    	     * @returns {Object} Returns the map cache instance.
    	     */function mapCacheSet(key,value){var data=getMapData(this,key),size=data.size;data.set(key,value);this.size+=data.size==size?0:1;return this;}// Add methods to `MapCache`.
    	MapCache.prototype.clear=mapCacheClear;MapCache.prototype['delete']=mapCacheDelete;MapCache.prototype.get=mapCacheGet;MapCache.prototype.has=mapCacheHas;MapCache.prototype.set=mapCacheSet;/*------------------------------------------------------------------------*//**
    	     *
    	     * Creates an array cache object to store unique values.
    	     *
    	     * @private
    	     * @constructor
    	     * @param {Array} [values] The values to cache.
    	     */function SetCache(values){var index=-1,length=values==null?0:values.length;this.__data__=new MapCache();while(++index<length){this.add(values[index]);}}/**
    	     * Adds `value` to the array cache.
    	     *
    	     * @private
    	     * @name add
    	     * @memberOf SetCache
    	     * @alias push
    	     * @param {*} value The value to cache.
    	     * @returns {Object} Returns the cache instance.
    	     */function setCacheAdd(value){this.__data__.set(value,HASH_UNDEFINED);return this;}/**
    	     * Checks if `value` is in the array cache.
    	     *
    	     * @private
    	     * @name has
    	     * @memberOf SetCache
    	     * @param {*} value The value to search for.
    	     * @returns {number} Returns `true` if `value` is found, else `false`.
    	     */function setCacheHas(value){return this.__data__.has(value);}// Add methods to `SetCache`.
    	SetCache.prototype.add=SetCache.prototype.push=setCacheAdd;SetCache.prototype.has=setCacheHas;/*------------------------------------------------------------------------*//**
    	     * Creates a stack cache object to store key-value pairs.
    	     *
    	     * @private
    	     * @constructor
    	     * @param {Array} [entries] The key-value pairs to cache.
    	     */function Stack(entries){var data=this.__data__=new ListCache(entries);this.size=data.size;}/**
    	     * Removes all key-value entries from the stack.
    	     *
    	     * @private
    	     * @name clear
    	     * @memberOf Stack
    	     */function stackClear(){this.__data__=new ListCache();this.size=0;}/**
    	     * Removes `key` and its value from the stack.
    	     *
    	     * @private
    	     * @name delete
    	     * @memberOf Stack
    	     * @param {string} key The key of the value to remove.
    	     * @returns {boolean} Returns `true` if the entry was removed, else `false`.
    	     */function stackDelete(key){var data=this.__data__,result=data['delete'](key);this.size=data.size;return result;}/**
    	     * Gets the stack value for `key`.
    	     *
    	     * @private
    	     * @name get
    	     * @memberOf Stack
    	     * @param {string} key The key of the value to get.
    	     * @returns {*} Returns the entry value.
    	     */function stackGet(key){return this.__data__.get(key);}/**
    	     * Checks if a stack value for `key` exists.
    	     *
    	     * @private
    	     * @name has
    	     * @memberOf Stack
    	     * @param {string} key The key of the entry to check.
    	     * @returns {boolean} Returns `true` if an entry for `key` exists, else `false`.
    	     */function stackHas(key){return this.__data__.has(key);}/**
    	     * Sets the stack `key` to `value`.
    	     *
    	     * @private
    	     * @name set
    	     * @memberOf Stack
    	     * @param {string} key The key of the value to set.
    	     * @param {*} value The value to set.
    	     * @returns {Object} Returns the stack cache instance.
    	     */function stackSet(key,value){var data=this.__data__;if(data instanceof ListCache){var pairs=data.__data__;if(!Map||pairs.length<LARGE_ARRAY_SIZE-1){pairs.push([key,value]);this.size=++data.size;return this;}data=this.__data__=new MapCache(pairs);}data.set(key,value);this.size=data.size;return this;}// Add methods to `Stack`.
    	Stack.prototype.clear=stackClear;Stack.prototype['delete']=stackDelete;Stack.prototype.get=stackGet;Stack.prototype.has=stackHas;Stack.prototype.set=stackSet;/*------------------------------------------------------------------------*//**
    	     * Creates an array of the enumerable property names of the array-like `value`.
    	     *
    	     * @private
    	     * @param {*} value The value to query.
    	     * @param {boolean} inherited Specify returning inherited property names.
    	     * @returns {Array} Returns the array of property names.
    	     */function arrayLikeKeys(value,inherited){var isArr=isArray(value),isArg=!isArr&&isArguments(value),isBuff=!isArr&&!isArg&&isBuffer(value),isType=!isArr&&!isArg&&!isBuff&&isTypedArray(value),skipIndexes=isArr||isArg||isBuff||isType,result=skipIndexes?baseTimes(value.length,String):[],length=result.length;for(var key in value){if((inherited||hasOwnProperty.call(value,key))&&!(skipIndexes&&(// Safari 9 has enumerable `arguments.length` in strict mode.
    	key=='length'||// Node.js 0.10 has enumerable non-index properties on buffers.
    	isBuff&&(key=='offset'||key=='parent')||// PhantomJS 2 has enumerable non-index properties on typed arrays.
    	isType&&(key=='buffer'||key=='byteLength'||key=='byteOffset')||// Skip index properties.
    	isIndex(key,length)))){result.push(key);}}return result;}/**
    	     * A specialized version of `_.sample` for arrays.
    	     *
    	     * @private
    	     * @param {Array} array The array to sample.
    	     * @returns {*} Returns the random element.
    	     */function arraySample(array){var length=array.length;return length?array[baseRandom(0,length-1)]:undefined$1;}/**
    	     * A specialized version of `_.sampleSize` for arrays.
    	     *
    	     * @private
    	     * @param {Array} array The array to sample.
    	     * @param {number} n The number of elements to sample.
    	     * @returns {Array} Returns the random elements.
    	     */function arraySampleSize(array,n){return shuffleSelf(copyArray(array),baseClamp(n,0,array.length));}/**
    	     * A specialized version of `_.shuffle` for arrays.
    	     *
    	     * @private
    	     * @param {Array} array The array to shuffle.
    	     * @returns {Array} Returns the new shuffled array.
    	     */function arrayShuffle(array){return shuffleSelf(copyArray(array));}/**
    	     * This function is like `assignValue` except that it doesn't assign
    	     * `undefined` values.
    	     *
    	     * @private
    	     * @param {Object} object The object to modify.
    	     * @param {string} key The key of the property to assign.
    	     * @param {*} value The value to assign.
    	     */function assignMergeValue(object,key,value){if(value!==undefined$1&&!eq(object[key],value)||value===undefined$1&&!(key in object)){baseAssignValue(object,key,value);}}/**
    	     * Assigns `value` to `key` of `object` if the existing value is not equivalent
    	     * using [`SameValueZero`](http://ecma-international.org/ecma-262/7.0/#sec-samevaluezero)
    	     * for equality comparisons.
    	     *
    	     * @private
    	     * @param {Object} object The object to modify.
    	     * @param {string} key The key of the property to assign.
    	     * @param {*} value The value to assign.
    	     */function assignValue(object,key,value){var objValue=object[key];if(!(hasOwnProperty.call(object,key)&&eq(objValue,value))||value===undefined$1&&!(key in object)){baseAssignValue(object,key,value);}}/**
    	     * Gets the index at which the `key` is found in `array` of key-value pairs.
    	     *
    	     * @private
    	     * @param {Array} array The array to inspect.
    	     * @param {*} key The key to search for.
    	     * @returns {number} Returns the index of the matched value, else `-1`.
    	     */function assocIndexOf(array,key){var length=array.length;while(length--){if(eq(array[length][0],key)){return length;}}return -1;}/**
    	     * Aggregates elements of `collection` on `accumulator` with keys transformed
    	     * by `iteratee` and values set by `setter`.
    	     *
    	     * @private
    	     * @param {Array|Object} collection The collection to iterate over.
    	     * @param {Function} setter The function to set `accumulator` values.
    	     * @param {Function} iteratee The iteratee to transform keys.
    	     * @param {Object} accumulator The initial aggregated object.
    	     * @returns {Function} Returns `accumulator`.
    	     */function baseAggregator(collection,setter,iteratee,accumulator){baseEach(collection,function(value,key,collection){setter(accumulator,value,iteratee(value),collection);});return accumulator;}/**
    	     * The base implementation of `_.assign` without support for multiple sources
    	     * or `customizer` functions.
    	     *
    	     * @private
    	     * @param {Object} object The destination object.
    	     * @param {Object} source The source object.
    	     * @returns {Object} Returns `object`.
    	     */function baseAssign(object,source){return object&&copyObject(source,keys(source),object);}/**
    	     * The base implementation of `_.assignIn` without support for multiple sources
    	     * or `customizer` functions.
    	     *
    	     * @private
    	     * @param {Object} object The destination object.
    	     * @param {Object} source The source object.
    	     * @returns {Object} Returns `object`.
    	     */function baseAssignIn(object,source){return object&&copyObject(source,keysIn(source),object);}/**
    	     * The base implementation of `assignValue` and `assignMergeValue` without
    	     * value checks.
    	     *
    	     * @private
    	     * @param {Object} object The object to modify.
    	     * @param {string} key The key of the property to assign.
    	     * @param {*} value The value to assign.
    	     */function baseAssignValue(object,key,value){if(key=='__proto__'&&defineProperty){defineProperty(object,key,{'configurable':true,'enumerable':true,'value':value,'writable':true});}else {object[key]=value;}}/**
    	     * The base implementation of `_.at` without support for individual paths.
    	     *
    	     * @private
    	     * @param {Object} object The object to iterate over.
    	     * @param {string[]} paths The property paths to pick.
    	     * @returns {Array} Returns the picked elements.
    	     */function baseAt(object,paths){var index=-1,length=paths.length,result=Array(length),skip=object==null;while(++index<length){result[index]=skip?undefined$1:get(object,paths[index]);}return result;}/**
    	     * The base implementation of `_.clamp` which doesn't coerce arguments.
    	     *
    	     * @private
    	     * @param {number} number The number to clamp.
    	     * @param {number} [lower] The lower bound.
    	     * @param {number} upper The upper bound.
    	     * @returns {number} Returns the clamped number.
    	     */function baseClamp(number,lower,upper){if(number===number){if(upper!==undefined$1){number=number<=upper?number:upper;}if(lower!==undefined$1){number=number>=lower?number:lower;}}return number;}/**
    	     * The base implementation of `_.clone` and `_.cloneDeep` which tracks
    	     * traversed objects.
    	     *
    	     * @private
    	     * @param {*} value The value to clone.
    	     * @param {boolean} bitmask The bitmask flags.
    	     *  1 - Deep clone
    	     *  2 - Flatten inherited properties
    	     *  4 - Clone symbols
    	     * @param {Function} [customizer] The function to customize cloning.
    	     * @param {string} [key] The key of `value`.
    	     * @param {Object} [object] The parent object of `value`.
    	     * @param {Object} [stack] Tracks traversed objects and their clone counterparts.
    	     * @returns {*} Returns the cloned value.
    	     */function baseClone(value,bitmask,customizer,key,object,stack){var result,isDeep=bitmask&CLONE_DEEP_FLAG,isFlat=bitmask&CLONE_FLAT_FLAG,isFull=bitmask&CLONE_SYMBOLS_FLAG;if(customizer){result=object?customizer(value,key,object,stack):customizer(value);}if(result!==undefined$1){return result;}if(!isObject(value)){return value;}var isArr=isArray(value);if(isArr){result=initCloneArray(value);if(!isDeep){return copyArray(value,result);}}else {var tag=getTag(value),isFunc=tag==funcTag||tag==genTag;if(isBuffer(value)){return cloneBuffer(value,isDeep);}if(tag==objectTag||tag==argsTag||isFunc&&!object){result=isFlat||isFunc?{}:initCloneObject(value);if(!isDeep){return isFlat?copySymbolsIn(value,baseAssignIn(result,value)):copySymbols(value,baseAssign(result,value));}}else {if(!cloneableTags[tag]){return object?value:{};}result=initCloneByTag(value,tag,isDeep);}}// Check for circular references and return its corresponding clone.
    	stack||(stack=new Stack());var stacked=stack.get(value);if(stacked){return stacked;}stack.set(value,result);if(isSet(value)){value.forEach(function(subValue){result.add(baseClone(subValue,bitmask,customizer,subValue,value,stack));});}else if(isMap(value)){value.forEach(function(subValue,key){result.set(key,baseClone(subValue,bitmask,customizer,key,value,stack));});}var keysFunc=isFull?isFlat?getAllKeysIn:getAllKeys:isFlat?keysIn:keys;var props=isArr?undefined$1:keysFunc(value);arrayEach(props||value,function(subValue,key){if(props){key=subValue;subValue=value[key];}// Recursively populate clone (susceptible to call stack limits).
    	assignValue(result,key,baseClone(subValue,bitmask,customizer,key,value,stack));});return result;}/**
    	     * The base implementation of `_.conforms` which doesn't clone `source`.
    	     *
    	     * @private
    	     * @param {Object} source The object of property predicates to conform to.
    	     * @returns {Function} Returns the new spec function.
    	     */function baseConforms(source){var props=keys(source);return function(object){return baseConformsTo(object,source,props);};}/**
    	     * The base implementation of `_.conformsTo` which accepts `props` to check.
    	     *
    	     * @private
    	     * @param {Object} object The object to inspect.
    	     * @param {Object} source The object of property predicates to conform to.
    	     * @returns {boolean} Returns `true` if `object` conforms, else `false`.
    	     */function baseConformsTo(object,source,props){var length=props.length;if(object==null){return !length;}object=Object(object);while(length--){var key=props[length],predicate=source[key],value=object[key];if(value===undefined$1&&!(key in object)||!predicate(value)){return false;}}return true;}/**
    	     * The base implementation of `_.delay` and `_.defer` which accepts `args`
    	     * to provide to `func`.
    	     *
    	     * @private
    	     * @param {Function} func The function to delay.
    	     * @param {number} wait The number of milliseconds to delay invocation.
    	     * @param {Array} args The arguments to provide to `func`.
    	     * @returns {number|Object} Returns the timer id or timeout object.
    	     */function baseDelay(func,wait,args){if(typeof func!='function'){throw new TypeError(FUNC_ERROR_TEXT);}return setTimeout(function(){func.apply(undefined$1,args);},wait);}/**
    	     * The base implementation of methods like `_.difference` without support
    	     * for excluding multiple arrays or iteratee shorthands.
    	     *
    	     * @private
    	     * @param {Array} array The array to inspect.
    	     * @param {Array} values The values to exclude.
    	     * @param {Function} [iteratee] The iteratee invoked per element.
    	     * @param {Function} [comparator] The comparator invoked per element.
    	     * @returns {Array} Returns the new array of filtered values.
    	     */function baseDifference(array,values,iteratee,comparator){var index=-1,includes=arrayIncludes,isCommon=true,length=array.length,result=[],valuesLength=values.length;if(!length){return result;}if(iteratee){values=arrayMap(values,baseUnary(iteratee));}if(comparator){includes=arrayIncludesWith;isCommon=false;}else if(values.length>=LARGE_ARRAY_SIZE){includes=cacheHas;isCommon=false;values=new SetCache(values);}outer:while(++index<length){var value=array[index],computed=iteratee==null?value:iteratee(value);value=comparator||value!==0?value:0;if(isCommon&&computed===computed){var valuesIndex=valuesLength;while(valuesIndex--){if(values[valuesIndex]===computed){continue outer;}}result.push(value);}else if(!includes(values,computed,comparator)){result.push(value);}}return result;}/**
    	     * The base implementation of `_.forEach` without support for iteratee shorthands.
    	     *
    	     * @private
    	     * @param {Array|Object} collection The collection to iterate over.
    	     * @param {Function} iteratee The function invoked per iteration.
    	     * @returns {Array|Object} Returns `collection`.
    	     */var baseEach=createBaseEach(baseForOwn);/**
    	     * The base implementation of `_.forEachRight` without support for iteratee shorthands.
    	     *
    	     * @private
    	     * @param {Array|Object} collection The collection to iterate over.
    	     * @param {Function} iteratee The function invoked per iteration.
    	     * @returns {Array|Object} Returns `collection`.
    	     */var baseEachRight=createBaseEach(baseForOwnRight,true);/**
    	     * The base implementation of `_.every` without support for iteratee shorthands.
    	     *
    	     * @private
    	     * @param {Array|Object} collection The collection to iterate over.
    	     * @param {Function} predicate The function invoked per iteration.
    	     * @returns {boolean} Returns `true` if all elements pass the predicate check,
    	     *  else `false`
    	     */function baseEvery(collection,predicate){var result=true;baseEach(collection,function(value,index,collection){result=!!predicate(value,index,collection);return result;});return result;}/**
    	     * The base implementation of methods like `_.max` and `_.min` which accepts a
    	     * `comparator` to determine the extremum value.
    	     *
    	     * @private
    	     * @param {Array} array The array to iterate over.
    	     * @param {Function} iteratee The iteratee invoked per iteration.
    	     * @param {Function} comparator The comparator used to compare values.
    	     * @returns {*} Returns the extremum value.
    	     */function baseExtremum(array,iteratee,comparator){var index=-1,length=array.length;while(++index<length){var value=array[index],current=iteratee(value);if(current!=null&&(computed===undefined$1?current===current&&!isSymbol(current):comparator(current,computed))){var computed=current,result=value;}}return result;}/**
    	     * The base implementation of `_.fill` without an iteratee call guard.
    	     *
    	     * @private
    	     * @param {Array} array The array to fill.
    	     * @param {*} value The value to fill `array` with.
    	     * @param {number} [start=0] The start position.
    	     * @param {number} [end=array.length] The end position.
    	     * @returns {Array} Returns `array`.
    	     */function baseFill(array,value,start,end){var length=array.length;start=toInteger(start);if(start<0){start=-start>length?0:length+start;}end=end===undefined$1||end>length?length:toInteger(end);if(end<0){end+=length;}end=start>end?0:toLength(end);while(start<end){array[start++]=value;}return array;}/**
    	     * The base implementation of `_.filter` without support for iteratee shorthands.
    	     *
    	     * @private
    	     * @param {Array|Object} collection The collection to iterate over.
    	     * @param {Function} predicate The function invoked per iteration.
    	     * @returns {Array} Returns the new filtered array.
    	     */function baseFilter(collection,predicate){var result=[];baseEach(collection,function(value,index,collection){if(predicate(value,index,collection)){result.push(value);}});return result;}/**
    	     * The base implementation of `_.flatten` with support for restricting flattening.
    	     *
    	     * @private
    	     * @param {Array} array The array to flatten.
    	     * @param {number} depth The maximum recursion depth.
    	     * @param {boolean} [predicate=isFlattenable] The function invoked per iteration.
    	     * @param {boolean} [isStrict] Restrict to values that pass `predicate` checks.
    	     * @param {Array} [result=[]] The initial result value.
    	     * @returns {Array} Returns the new flattened array.
    	     */function baseFlatten(array,depth,predicate,isStrict,result){var index=-1,length=array.length;predicate||(predicate=isFlattenable);result||(result=[]);while(++index<length){var value=array[index];if(depth>0&&predicate(value)){if(depth>1){// Recursively flatten arrays (susceptible to call stack limits).
    	baseFlatten(value,depth-1,predicate,isStrict,result);}else {arrayPush(result,value);}}else if(!isStrict){result[result.length]=value;}}return result;}/**
    	     * The base implementation of `baseForOwn` which iterates over `object`
    	     * properties returned by `keysFunc` and invokes `iteratee` for each property.
    	     * Iteratee functions may exit iteration early by explicitly returning `false`.
    	     *
    	     * @private
    	     * @param {Object} object The object to iterate over.
    	     * @param {Function} iteratee The function invoked per iteration.
    	     * @param {Function} keysFunc The function to get the keys of `object`.
    	     * @returns {Object} Returns `object`.
    	     */var baseFor=createBaseFor();/**
    	     * This function is like `baseFor` except that it iterates over properties
    	     * in the opposite order.
    	     *
    	     * @private
    	     * @param {Object} object The object to iterate over.
    	     * @param {Function} iteratee The function invoked per iteration.
    	     * @param {Function} keysFunc The function to get the keys of `object`.
    	     * @returns {Object} Returns `object`.
    	     */var baseForRight=createBaseFor(true);/**
    	     * The base implementation of `_.forOwn` without support for iteratee shorthands.
    	     *
    	     * @private
    	     * @param {Object} object The object to iterate over.
    	     * @param {Function} iteratee The function invoked per iteration.
    	     * @returns {Object} Returns `object`.
    	     */function baseForOwn(object,iteratee){return object&&baseFor(object,iteratee,keys);}/**
    	     * The base implementation of `_.forOwnRight` without support for iteratee shorthands.
    	     *
    	     * @private
    	     * @param {Object} object The object to iterate over.
    	     * @param {Function} iteratee The function invoked per iteration.
    	     * @returns {Object} Returns `object`.
    	     */function baseForOwnRight(object,iteratee){return object&&baseForRight(object,iteratee,keys);}/**
    	     * The base implementation of `_.functions` which creates an array of
    	     * `object` function property names filtered from `props`.
    	     *
    	     * @private
    	     * @param {Object} object The object to inspect.
    	     * @param {Array} props The property names to filter.
    	     * @returns {Array} Returns the function names.
    	     */function baseFunctions(object,props){return arrayFilter(props,function(key){return isFunction(object[key]);});}/**
    	     * The base implementation of `_.get` without support for default values.
    	     *
    	     * @private
    	     * @param {Object} object The object to query.
    	     * @param {Array|string} path The path of the property to get.
    	     * @returns {*} Returns the resolved value.
    	     */function baseGet(object,path){path=castPath(path,object);var index=0,length=path.length;while(object!=null&&index<length){object=object[toKey(path[index++])];}return index&&index==length?object:undefined$1;}/**
    	     * The base implementation of `getAllKeys` and `getAllKeysIn` which uses
    	     * `keysFunc` and `symbolsFunc` to get the enumerable property names and
    	     * symbols of `object`.
    	     *
    	     * @private
    	     * @param {Object} object The object to query.
    	     * @param {Function} keysFunc The function to get the keys of `object`.
    	     * @param {Function} symbolsFunc The function to get the symbols of `object`.
    	     * @returns {Array} Returns the array of property names and symbols.
    	     */function baseGetAllKeys(object,keysFunc,symbolsFunc){var result=keysFunc(object);return isArray(object)?result:arrayPush(result,symbolsFunc(object));}/**
    	     * The base implementation of `getTag` without fallbacks for buggy environments.
    	     *
    	     * @private
    	     * @param {*} value The value to query.
    	     * @returns {string} Returns the `toStringTag`.
    	     */function baseGetTag(value){if(value==null){return value===undefined$1?undefinedTag:nullTag;}return symToStringTag&&symToStringTag in Object(value)?getRawTag(value):objectToString(value);}/**
    	     * The base implementation of `_.gt` which doesn't coerce arguments.
    	     *
    	     * @private
    	     * @param {*} value The value to compare.
    	     * @param {*} other The other value to compare.
    	     * @returns {boolean} Returns `true` if `value` is greater than `other`,
    	     *  else `false`.
    	     */function baseGt(value,other){return value>other;}/**
    	     * The base implementation of `_.has` without support for deep paths.
    	     *
    	     * @private
    	     * @param {Object} [object] The object to query.
    	     * @param {Array|string} key The key to check.
    	     * @returns {boolean} Returns `true` if `key` exists, else `false`.
    	     */function baseHas(object,key){return object!=null&&hasOwnProperty.call(object,key);}/**
    	     * The base implementation of `_.hasIn` without support for deep paths.
    	     *
    	     * @private
    	     * @param {Object} [object] The object to query.
    	     * @param {Array|string} key The key to check.
    	     * @returns {boolean} Returns `true` if `key` exists, else `false`.
    	     */function baseHasIn(object,key){return object!=null&&key in Object(object);}/**
    	     * The base implementation of `_.inRange` which doesn't coerce arguments.
    	     *
    	     * @private
    	     * @param {number} number The number to check.
    	     * @param {number} start The start of the range.
    	     * @param {number} end The end of the range.
    	     * @returns {boolean} Returns `true` if `number` is in the range, else `false`.
    	     */function baseInRange(number,start,end){return number>=nativeMin(start,end)&&number<nativeMax(start,end);}/**
    	     * The base implementation of methods like `_.intersection`, without support
    	     * for iteratee shorthands, that accepts an array of arrays to inspect.
    	     *
    	     * @private
    	     * @param {Array} arrays The arrays to inspect.
    	     * @param {Function} [iteratee] The iteratee invoked per element.
    	     * @param {Function} [comparator] The comparator invoked per element.
    	     * @returns {Array} Returns the new array of shared values.
    	     */function baseIntersection(arrays,iteratee,comparator){var includes=comparator?arrayIncludesWith:arrayIncludes,length=arrays[0].length,othLength=arrays.length,othIndex=othLength,caches=Array(othLength),maxLength=Infinity,result=[];while(othIndex--){var array=arrays[othIndex];if(othIndex&&iteratee){array=arrayMap(array,baseUnary(iteratee));}maxLength=nativeMin(array.length,maxLength);caches[othIndex]=!comparator&&(iteratee||length>=120&&array.length>=120)?new SetCache(othIndex&&array):undefined$1;}array=arrays[0];var index=-1,seen=caches[0];outer:while(++index<length&&result.length<maxLength){var value=array[index],computed=iteratee?iteratee(value):value;value=comparator||value!==0?value:0;if(!(seen?cacheHas(seen,computed):includes(result,computed,comparator))){othIndex=othLength;while(--othIndex){var cache=caches[othIndex];if(!(cache?cacheHas(cache,computed):includes(arrays[othIndex],computed,comparator))){continue outer;}}if(seen){seen.push(computed);}result.push(value);}}return result;}/**
    	     * The base implementation of `_.invert` and `_.invertBy` which inverts
    	     * `object` with values transformed by `iteratee` and set by `setter`.
    	     *
    	     * @private
    	     * @param {Object} object The object to iterate over.
    	     * @param {Function} setter The function to set `accumulator` values.
    	     * @param {Function} iteratee The iteratee to transform values.
    	     * @param {Object} accumulator The initial inverted object.
    	     * @returns {Function} Returns `accumulator`.
    	     */function baseInverter(object,setter,iteratee,accumulator){baseForOwn(object,function(value,key,object){setter(accumulator,iteratee(value),key,object);});return accumulator;}/**
    	     * The base implementation of `_.invoke` without support for individual
    	     * method arguments.
    	     *
    	     * @private
    	     * @param {Object} object The object to query.
    	     * @param {Array|string} path The path of the method to invoke.
    	     * @param {Array} args The arguments to invoke the method with.
    	     * @returns {*} Returns the result of the invoked method.
    	     */function baseInvoke(object,path,args){path=castPath(path,object);object=parent(object,path);var func=object==null?object:object[toKey(last(path))];return func==null?undefined$1:apply(func,object,args);}/**
    	     * The base implementation of `_.isArguments`.
    	     *
    	     * @private
    	     * @param {*} value The value to check.
    	     * @returns {boolean} Returns `true` if `value` is an `arguments` object,
    	     */function baseIsArguments(value){return isObjectLike(value)&&baseGetTag(value)==argsTag;}/**
    	     * The base implementation of `_.isArrayBuffer` without Node.js optimizations.
    	     *
    	     * @private
    	     * @param {*} value The value to check.
    	     * @returns {boolean} Returns `true` if `value` is an array buffer, else `false`.
    	     */function baseIsArrayBuffer(value){return isObjectLike(value)&&baseGetTag(value)==arrayBufferTag;}/**
    	     * The base implementation of `_.isDate` without Node.js optimizations.
    	     *
    	     * @private
    	     * @param {*} value The value to check.
    	     * @returns {boolean} Returns `true` if `value` is a date object, else `false`.
    	     */function baseIsDate(value){return isObjectLike(value)&&baseGetTag(value)==dateTag;}/**
    	     * The base implementation of `_.isEqual` which supports partial comparisons
    	     * and tracks traversed objects.
    	     *
    	     * @private
    	     * @param {*} value The value to compare.
    	     * @param {*} other The other value to compare.
    	     * @param {boolean} bitmask The bitmask flags.
    	     *  1 - Unordered comparison
    	     *  2 - Partial comparison
    	     * @param {Function} [customizer] The function to customize comparisons.
    	     * @param {Object} [stack] Tracks traversed `value` and `other` objects.
    	     * @returns {boolean} Returns `true` if the values are equivalent, else `false`.
    	     */function baseIsEqual(value,other,bitmask,customizer,stack){if(value===other){return true;}if(value==null||other==null||!isObjectLike(value)&&!isObjectLike(other)){return value!==value&&other!==other;}return baseIsEqualDeep(value,other,bitmask,customizer,baseIsEqual,stack);}/**
    	     * A specialized version of `baseIsEqual` for arrays and objects which performs
    	     * deep comparisons and tracks traversed objects enabling objects with circular
    	     * references to be compared.
    	     *
    	     * @private
    	     * @param {Object} object The object to compare.
    	     * @param {Object} other The other object to compare.
    	     * @param {number} bitmask The bitmask flags. See `baseIsEqual` for more details.
    	     * @param {Function} customizer The function to customize comparisons.
    	     * @param {Function} equalFunc The function to determine equivalents of values.
    	     * @param {Object} [stack] Tracks traversed `object` and `other` objects.
    	     * @returns {boolean} Returns `true` if the objects are equivalent, else `false`.
    	     */function baseIsEqualDeep(object,other,bitmask,customizer,equalFunc,stack){var objIsArr=isArray(object),othIsArr=isArray(other),objTag=objIsArr?arrayTag:getTag(object),othTag=othIsArr?arrayTag:getTag(other);objTag=objTag==argsTag?objectTag:objTag;othTag=othTag==argsTag?objectTag:othTag;var objIsObj=objTag==objectTag,othIsObj=othTag==objectTag,isSameTag=objTag==othTag;if(isSameTag&&isBuffer(object)){if(!isBuffer(other)){return false;}objIsArr=true;objIsObj=false;}if(isSameTag&&!objIsObj){stack||(stack=new Stack());return objIsArr||isTypedArray(object)?equalArrays(object,other,bitmask,customizer,equalFunc,stack):equalByTag(object,other,objTag,bitmask,customizer,equalFunc,stack);}if(!(bitmask&COMPARE_PARTIAL_FLAG)){var objIsWrapped=objIsObj&&hasOwnProperty.call(object,'__wrapped__'),othIsWrapped=othIsObj&&hasOwnProperty.call(other,'__wrapped__');if(objIsWrapped||othIsWrapped){var objUnwrapped=objIsWrapped?object.value():object,othUnwrapped=othIsWrapped?other.value():other;stack||(stack=new Stack());return equalFunc(objUnwrapped,othUnwrapped,bitmask,customizer,stack);}}if(!isSameTag){return false;}stack||(stack=new Stack());return equalObjects(object,other,bitmask,customizer,equalFunc,stack);}/**
    	     * The base implementation of `_.isMap` without Node.js optimizations.
    	     *
    	     * @private
    	     * @param {*} value The value to check.
    	     * @returns {boolean} Returns `true` if `value` is a map, else `false`.
    	     */function baseIsMap(value){return isObjectLike(value)&&getTag(value)==mapTag;}/**
    	     * The base implementation of `_.isMatch` without support for iteratee shorthands.
    	     *
    	     * @private
    	     * @param {Object} object The object to inspect.
    	     * @param {Object} source The object of property values to match.
    	     * @param {Array} matchData The property names, values, and compare flags to match.
    	     * @param {Function} [customizer] The function to customize comparisons.
    	     * @returns {boolean} Returns `true` if `object` is a match, else `false`.
    	     */function baseIsMatch(object,source,matchData,customizer){var index=matchData.length,length=index,noCustomizer=!customizer;if(object==null){return !length;}object=Object(object);while(index--){var data=matchData[index];if(noCustomizer&&data[2]?data[1]!==object[data[0]]:!(data[0]in object)){return false;}}while(++index<length){data=matchData[index];var key=data[0],objValue=object[key],srcValue=data[1];if(noCustomizer&&data[2]){if(objValue===undefined$1&&!(key in object)){return false;}}else {var stack=new Stack();if(customizer){var result=customizer(objValue,srcValue,key,object,source,stack);}if(!(result===undefined$1?baseIsEqual(srcValue,objValue,COMPARE_PARTIAL_FLAG|COMPARE_UNORDERED_FLAG,customizer,stack):result)){return false;}}}return true;}/**
    	     * The base implementation of `_.isNative` without bad shim checks.
    	     *
    	     * @private
    	     * @param {*} value The value to check.
    	     * @returns {boolean} Returns `true` if `value` is a native function,
    	     *  else `false`.
    	     */function baseIsNative(value){if(!isObject(value)||isMasked(value)){return false;}var pattern=isFunction(value)?reIsNative:reIsHostCtor;return pattern.test(toSource(value));}/**
    	     * The base implementation of `_.isRegExp` without Node.js optimizations.
    	     *
    	     * @private
    	     * @param {*} value The value to check.
    	     * @returns {boolean} Returns `true` if `value` is a regexp, else `false`.
    	     */function baseIsRegExp(value){return isObjectLike(value)&&baseGetTag(value)==regexpTag;}/**
    	     * The base implementation of `_.isSet` without Node.js optimizations.
    	     *
    	     * @private
    	     * @param {*} value The value to check.
    	     * @returns {boolean} Returns `true` if `value` is a set, else `false`.
    	     */function baseIsSet(value){return isObjectLike(value)&&getTag(value)==setTag;}/**
    	     * The base implementation of `_.isTypedArray` without Node.js optimizations.
    	     *
    	     * @private
    	     * @param {*} value The value to check.
    	     * @returns {boolean} Returns `true` if `value` is a typed array, else `false`.
    	     */function baseIsTypedArray(value){return isObjectLike(value)&&isLength(value.length)&&!!typedArrayTags[baseGetTag(value)];}/**
    	     * The base implementation of `_.iteratee`.
    	     *
    	     * @private
    	     * @param {*} [value=_.identity] The value to convert to an iteratee.
    	     * @returns {Function} Returns the iteratee.
    	     */function baseIteratee(value){// Don't store the `typeof` result in a variable to avoid a JIT bug in Safari 9.
    	// See https://bugs.webkit.org/show_bug.cgi?id=156034 for more details.
    	if(typeof value=='function'){return value;}if(value==null){return identity;}if(typeof value=='object'){return isArray(value)?baseMatchesProperty(value[0],value[1]):baseMatches(value);}return property(value);}/**
    	     * The base implementation of `_.keys` which doesn't treat sparse arrays as dense.
    	     *
    	     * @private
    	     * @param {Object} object The object to query.
    	     * @returns {Array} Returns the array of property names.
    	     */function baseKeys(object){if(!isPrototype(object)){return nativeKeys(object);}var result=[];for(var key in Object(object)){if(hasOwnProperty.call(object,key)&&key!='constructor'){result.push(key);}}return result;}/**
    	     * The base implementation of `_.keysIn` which doesn't treat sparse arrays as dense.
    	     *
    	     * @private
    	     * @param {Object} object The object to query.
    	     * @returns {Array} Returns the array of property names.
    	     */function baseKeysIn(object){if(!isObject(object)){return nativeKeysIn(object);}var isProto=isPrototype(object),result=[];for(var key in object){if(!(key=='constructor'&&(isProto||!hasOwnProperty.call(object,key)))){result.push(key);}}return result;}/**
    	     * The base implementation of `_.lt` which doesn't coerce arguments.
    	     *
    	     * @private
    	     * @param {*} value The value to compare.
    	     * @param {*} other The other value to compare.
    	     * @returns {boolean} Returns `true` if `value` is less than `other`,
    	     *  else `false`.
    	     */function baseLt(value,other){return value<other;}/**
    	     * The base implementation of `_.map` without support for iteratee shorthands.
    	     *
    	     * @private
    	     * @param {Array|Object} collection The collection to iterate over.
    	     * @param {Function} iteratee The function invoked per iteration.
    	     * @returns {Array} Returns the new mapped array.
    	     */function baseMap(collection,iteratee){var index=-1,result=isArrayLike(collection)?Array(collection.length):[];baseEach(collection,function(value,key,collection){result[++index]=iteratee(value,key,collection);});return result;}/**
    	     * The base implementation of `_.matches` which doesn't clone `source`.
    	     *
    	     * @private
    	     * @param {Object} source The object of property values to match.
    	     * @returns {Function} Returns the new spec function.
    	     */function baseMatches(source){var matchData=getMatchData(source);if(matchData.length==1&&matchData[0][2]){return matchesStrictComparable(matchData[0][0],matchData[0][1]);}return function(object){return object===source||baseIsMatch(object,source,matchData);};}/**
    	     * The base implementation of `_.matchesProperty` which doesn't clone `srcValue`.
    	     *
    	     * @private
    	     * @param {string} path The path of the property to get.
    	     * @param {*} srcValue The value to match.
    	     * @returns {Function} Returns the new spec function.
    	     */function baseMatchesProperty(path,srcValue){if(isKey(path)&&isStrictComparable(srcValue)){return matchesStrictComparable(toKey(path),srcValue);}return function(object){var objValue=get(object,path);return objValue===undefined$1&&objValue===srcValue?hasIn(object,path):baseIsEqual(srcValue,objValue,COMPARE_PARTIAL_FLAG|COMPARE_UNORDERED_FLAG);};}/**
    	     * The base implementation of `_.merge` without support for multiple sources.
    	     *
    	     * @private
    	     * @param {Object} object The destination object.
    	     * @param {Object} source The source object.
    	     * @param {number} srcIndex The index of `source`.
    	     * @param {Function} [customizer] The function to customize merged values.
    	     * @param {Object} [stack] Tracks traversed source values and their merged
    	     *  counterparts.
    	     */function baseMerge(object,source,srcIndex,customizer,stack){if(object===source){return;}baseFor(source,function(srcValue,key){stack||(stack=new Stack());if(isObject(srcValue)){baseMergeDeep(object,source,key,srcIndex,baseMerge,customizer,stack);}else {var newValue=customizer?customizer(safeGet(object,key),srcValue,key+'',object,source,stack):undefined$1;if(newValue===undefined$1){newValue=srcValue;}assignMergeValue(object,key,newValue);}},keysIn);}/**
    	     * A specialized version of `baseMerge` for arrays and objects which performs
    	     * deep merges and tracks traversed objects enabling objects with circular
    	     * references to be merged.
    	     *
    	     * @private
    	     * @param {Object} object The destination object.
    	     * @param {Object} source The source object.
    	     * @param {string} key The key of the value to merge.
    	     * @param {number} srcIndex The index of `source`.
    	     * @param {Function} mergeFunc The function to merge values.
    	     * @param {Function} [customizer] The function to customize assigned values.
    	     * @param {Object} [stack] Tracks traversed source values and their merged
    	     *  counterparts.
    	     */function baseMergeDeep(object,source,key,srcIndex,mergeFunc,customizer,stack){var objValue=safeGet(object,key),srcValue=safeGet(source,key),stacked=stack.get(srcValue);if(stacked){assignMergeValue(object,key,stacked);return;}var newValue=customizer?customizer(objValue,srcValue,key+'',object,source,stack):undefined$1;var isCommon=newValue===undefined$1;if(isCommon){var isArr=isArray(srcValue),isBuff=!isArr&&isBuffer(srcValue),isTyped=!isArr&&!isBuff&&isTypedArray(srcValue);newValue=srcValue;if(isArr||isBuff||isTyped){if(isArray(objValue)){newValue=objValue;}else if(isArrayLikeObject(objValue)){newValue=copyArray(objValue);}else if(isBuff){isCommon=false;newValue=cloneBuffer(srcValue,true);}else if(isTyped){isCommon=false;newValue=cloneTypedArray(srcValue,true);}else {newValue=[];}}else if(isPlainObject(srcValue)||isArguments(srcValue)){newValue=objValue;if(isArguments(objValue)){newValue=toPlainObject(objValue);}else if(!isObject(objValue)||isFunction(objValue)){newValue=initCloneObject(srcValue);}}else {isCommon=false;}}if(isCommon){// Recursively merge objects and arrays (susceptible to call stack limits).
    	stack.set(srcValue,newValue);mergeFunc(newValue,srcValue,srcIndex,customizer,stack);stack['delete'](srcValue);}assignMergeValue(object,key,newValue);}/**
    	     * The base implementation of `_.nth` which doesn't coerce arguments.
    	     *
    	     * @private
    	     * @param {Array} array The array to query.
    	     * @param {number} n The index of the element to return.
    	     * @returns {*} Returns the nth element of `array`.
    	     */function baseNth(array,n){var length=array.length;if(!length){return;}n+=n<0?length:0;return isIndex(n,length)?array[n]:undefined$1;}/**
    	     * The base implementation of `_.orderBy` without param guards.
    	     *
    	     * @private
    	     * @param {Array|Object} collection The collection to iterate over.
    	     * @param {Function[]|Object[]|string[]} iteratees The iteratees to sort by.
    	     * @param {string[]} orders The sort orders of `iteratees`.
    	     * @returns {Array} Returns the new sorted array.
    	     */function baseOrderBy(collection,iteratees,orders){if(iteratees.length){iteratees=arrayMap(iteratees,function(iteratee){if(isArray(iteratee)){return function(value){return baseGet(value,iteratee.length===1?iteratee[0]:iteratee);};}return iteratee;});}else {iteratees=[identity];}var index=-1;iteratees=arrayMap(iteratees,baseUnary(getIteratee()));var result=baseMap(collection,function(value,key,collection){var criteria=arrayMap(iteratees,function(iteratee){return iteratee(value);});return {'criteria':criteria,'index':++index,'value':value};});return baseSortBy(result,function(object,other){return compareMultiple(object,other,orders);});}/**
    	     * The base implementation of `_.pick` without support for individual
    	     * property identifiers.
    	     *
    	     * @private
    	     * @param {Object} object The source object.
    	     * @param {string[]} paths The property paths to pick.
    	     * @returns {Object} Returns the new object.
    	     */function basePick(object,paths){return basePickBy(object,paths,function(value,path){return hasIn(object,path);});}/**
    	     * The base implementation of  `_.pickBy` without support for iteratee shorthands.
    	     *
    	     * @private
    	     * @param {Object} object The source object.
    	     * @param {string[]} paths The property paths to pick.
    	     * @param {Function} predicate The function invoked per property.
    	     * @returns {Object} Returns the new object.
    	     */function basePickBy(object,paths,predicate){var index=-1,length=paths.length,result={};while(++index<length){var path=paths[index],value=baseGet(object,path);if(predicate(value,path)){baseSet(result,castPath(path,object),value);}}return result;}/**
    	     * A specialized version of `baseProperty` which supports deep paths.
    	     *
    	     * @private
    	     * @param {Array|string} path The path of the property to get.
    	     * @returns {Function} Returns the new accessor function.
    	     */function basePropertyDeep(path){return function(object){return baseGet(object,path);};}/**
    	     * The base implementation of `_.pullAllBy` without support for iteratee
    	     * shorthands.
    	     *
    	     * @private
    	     * @param {Array} array The array to modify.
    	     * @param {Array} values The values to remove.
    	     * @param {Function} [iteratee] The iteratee invoked per element.
    	     * @param {Function} [comparator] The comparator invoked per element.
    	     * @returns {Array} Returns `array`.
    	     */function basePullAll(array,values,iteratee,comparator){var indexOf=comparator?baseIndexOfWith:baseIndexOf,index=-1,length=values.length,seen=array;if(array===values){values=copyArray(values);}if(iteratee){seen=arrayMap(array,baseUnary(iteratee));}while(++index<length){var fromIndex=0,value=values[index],computed=iteratee?iteratee(value):value;while((fromIndex=indexOf(seen,computed,fromIndex,comparator))>-1){if(seen!==array){splice.call(seen,fromIndex,1);}splice.call(array,fromIndex,1);}}return array;}/**
    	     * The base implementation of `_.pullAt` without support for individual
    	     * indexes or capturing the removed elements.
    	     *
    	     * @private
    	     * @param {Array} array The array to modify.
    	     * @param {number[]} indexes The indexes of elements to remove.
    	     * @returns {Array} Returns `array`.
    	     */function basePullAt(array,indexes){var length=array?indexes.length:0,lastIndex=length-1;while(length--){var index=indexes[length];if(length==lastIndex||index!==previous){var previous=index;if(isIndex(index)){splice.call(array,index,1);}else {baseUnset(array,index);}}}return array;}/**
    	     * The base implementation of `_.random` without support for returning
    	     * floating-point numbers.
    	     *
    	     * @private
    	     * @param {number} lower The lower bound.
    	     * @param {number} upper The upper bound.
    	     * @returns {number} Returns the random number.
    	     */function baseRandom(lower,upper){return lower+nativeFloor(nativeRandom()*(upper-lower+1));}/**
    	     * The base implementation of `_.range` and `_.rangeRight` which doesn't
    	     * coerce arguments.
    	     *
    	     * @private
    	     * @param {number} start The start of the range.
    	     * @param {number} end The end of the range.
    	     * @param {number} step The value to increment or decrement by.
    	     * @param {boolean} [fromRight] Specify iterating from right to left.
    	     * @returns {Array} Returns the range of numbers.
    	     */function baseRange(start,end,step,fromRight){var index=-1,length=nativeMax(nativeCeil((end-start)/(step||1)),0),result=Array(length);while(length--){result[fromRight?length:++index]=start;start+=step;}return result;}/**
    	     * The base implementation of `_.repeat` which doesn't coerce arguments.
    	     *
    	     * @private
    	     * @param {string} string The string to repeat.
    	     * @param {number} n The number of times to repeat the string.
    	     * @returns {string} Returns the repeated string.
    	     */function baseRepeat(string,n){var result='';if(!string||n<1||n>MAX_SAFE_INTEGER){return result;}// Leverage the exponentiation by squaring algorithm for a faster repeat.
    	// See https://en.wikipedia.org/wiki/Exponentiation_by_squaring for more details.
    	do{if(n%2){result+=string;}n=nativeFloor(n/2);if(n){string+=string;}}while(n);return result;}/**
    	     * The base implementation of `_.rest` which doesn't validate or coerce arguments.
    	     *
    	     * @private
    	     * @param {Function} func The function to apply a rest parameter to.
    	     * @param {number} [start=func.length-1] The start position of the rest parameter.
    	     * @returns {Function} Returns the new function.
    	     */function baseRest(func,start){return setToString(overRest(func,start,identity),func+'');}/**
    	     * The base implementation of `_.sample`.
    	     *
    	     * @private
    	     * @param {Array|Object} collection The collection to sample.
    	     * @returns {*} Returns the random element.
    	     */function baseSample(collection){return arraySample(values(collection));}/**
    	     * The base implementation of `_.sampleSize` without param guards.
    	     *
    	     * @private
    	     * @param {Array|Object} collection The collection to sample.
    	     * @param {number} n The number of elements to sample.
    	     * @returns {Array} Returns the random elements.
    	     */function baseSampleSize(collection,n){var array=values(collection);return shuffleSelf(array,baseClamp(n,0,array.length));}/**
    	     * The base implementation of `_.set`.
    	     *
    	     * @private
    	     * @param {Object} object The object to modify.
    	     * @param {Array|string} path The path of the property to set.
    	     * @param {*} value The value to set.
    	     * @param {Function} [customizer] The function to customize path creation.
    	     * @returns {Object} Returns `object`.
    	     */function baseSet(object,path,value,customizer){if(!isObject(object)){return object;}path=castPath(path,object);var index=-1,length=path.length,lastIndex=length-1,nested=object;while(nested!=null&&++index<length){var key=toKey(path[index]),newValue=value;if(key==='__proto__'||key==='constructor'||key==='prototype'){return object;}if(index!=lastIndex){var objValue=nested[key];newValue=customizer?customizer(objValue,key,nested):undefined$1;if(newValue===undefined$1){newValue=isObject(objValue)?objValue:isIndex(path[index+1])?[]:{};}}assignValue(nested,key,newValue);nested=nested[key];}return object;}/**
    	     * The base implementation of `setData` without support for hot loop shorting.
    	     *
    	     * @private
    	     * @param {Function} func The function to associate metadata with.
    	     * @param {*} data The metadata.
    	     * @returns {Function} Returns `func`.
    	     */var baseSetData=!metaMap?identity:function(func,data){metaMap.set(func,data);return func;};/**
    	     * The base implementation of `setToString` without support for hot loop shorting.
    	     *
    	     * @private
    	     * @param {Function} func The function to modify.
    	     * @param {Function} string The `toString` result.
    	     * @returns {Function} Returns `func`.
    	     */var baseSetToString=!defineProperty?identity:function(func,string){return defineProperty(func,'toString',{'configurable':true,'enumerable':false,'value':constant(string),'writable':true});};/**
    	     * The base implementation of `_.shuffle`.
    	     *
    	     * @private
    	     * @param {Array|Object} collection The collection to shuffle.
    	     * @returns {Array} Returns the new shuffled array.
    	     */function baseShuffle(collection){return shuffleSelf(values(collection));}/**
    	     * The base implementation of `_.slice` without an iteratee call guard.
    	     *
    	     * @private
    	     * @param {Array} array The array to slice.
    	     * @param {number} [start=0] The start position.
    	     * @param {number} [end=array.length] The end position.
    	     * @returns {Array} Returns the slice of `array`.
    	     */function baseSlice(array,start,end){var index=-1,length=array.length;if(start<0){start=-start>length?0:length+start;}end=end>length?length:end;if(end<0){end+=length;}length=start>end?0:end-start>>>0;start>>>=0;var result=Array(length);while(++index<length){result[index]=array[index+start];}return result;}/**
    	     * The base implementation of `_.some` without support for iteratee shorthands.
    	     *
    	     * @private
    	     * @param {Array|Object} collection The collection to iterate over.
    	     * @param {Function} predicate The function invoked per iteration.
    	     * @returns {boolean} Returns `true` if any element passes the predicate check,
    	     *  else `false`.
    	     */function baseSome(collection,predicate){var result;baseEach(collection,function(value,index,collection){result=predicate(value,index,collection);return !result;});return !!result;}/**
    	     * The base implementation of `_.sortedIndex` and `_.sortedLastIndex` which
    	     * performs a binary search of `array` to determine the index at which `value`
    	     * should be inserted into `array` in order to maintain its sort order.
    	     *
    	     * @private
    	     * @param {Array} array The sorted array to inspect.
    	     * @param {*} value The value to evaluate.
    	     * @param {boolean} [retHighest] Specify returning the highest qualified index.
    	     * @returns {number} Returns the index at which `value` should be inserted
    	     *  into `array`.
    	     */function baseSortedIndex(array,value,retHighest){var low=0,high=array==null?low:array.length;if(typeof value=='number'&&value===value&&high<=HALF_MAX_ARRAY_LENGTH){while(low<high){var mid=low+high>>>1,computed=array[mid];if(computed!==null&&!isSymbol(computed)&&(retHighest?computed<=value:computed<value)){low=mid+1;}else {high=mid;}}return high;}return baseSortedIndexBy(array,value,identity,retHighest);}/**
    	     * The base implementation of `_.sortedIndexBy` and `_.sortedLastIndexBy`
    	     * which invokes `iteratee` for `value` and each element of `array` to compute
    	     * their sort ranking. The iteratee is invoked with one argument; (value).
    	     *
    	     * @private
    	     * @param {Array} array The sorted array to inspect.
    	     * @param {*} value The value to evaluate.
    	     * @param {Function} iteratee The iteratee invoked per element.
    	     * @param {boolean} [retHighest] Specify returning the highest qualified index.
    	     * @returns {number} Returns the index at which `value` should be inserted
    	     *  into `array`.
    	     */function baseSortedIndexBy(array,value,iteratee,retHighest){var low=0,high=array==null?0:array.length;if(high===0){return 0;}value=iteratee(value);var valIsNaN=value!==value,valIsNull=value===null,valIsSymbol=isSymbol(value),valIsUndefined=value===undefined$1;while(low<high){var mid=nativeFloor((low+high)/2),computed=iteratee(array[mid]),othIsDefined=computed!==undefined$1,othIsNull=computed===null,othIsReflexive=computed===computed,othIsSymbol=isSymbol(computed);if(valIsNaN){var setLow=retHighest||othIsReflexive;}else if(valIsUndefined){setLow=othIsReflexive&&(retHighest||othIsDefined);}else if(valIsNull){setLow=othIsReflexive&&othIsDefined&&(retHighest||!othIsNull);}else if(valIsSymbol){setLow=othIsReflexive&&othIsDefined&&!othIsNull&&(retHighest||!othIsSymbol);}else if(othIsNull||othIsSymbol){setLow=false;}else {setLow=retHighest?computed<=value:computed<value;}if(setLow){low=mid+1;}else {high=mid;}}return nativeMin(high,MAX_ARRAY_INDEX);}/**
    	     * The base implementation of `_.sortedUniq` and `_.sortedUniqBy` without
    	     * support for iteratee shorthands.
    	     *
    	     * @private
    	     * @param {Array} array The array to inspect.
    	     * @param {Function} [iteratee] The iteratee invoked per element.
    	     * @returns {Array} Returns the new duplicate free array.
    	     */function baseSortedUniq(array,iteratee){var index=-1,length=array.length,resIndex=0,result=[];while(++index<length){var value=array[index],computed=iteratee?iteratee(value):value;if(!index||!eq(computed,seen)){var seen=computed;result[resIndex++]=value===0?0:value;}}return result;}/**
    	     * The base implementation of `_.toNumber` which doesn't ensure correct
    	     * conversions of binary, hexadecimal, or octal string values.
    	     *
    	     * @private
    	     * @param {*} value The value to process.
    	     * @returns {number} Returns the number.
    	     */function baseToNumber(value){if(typeof value=='number'){return value;}if(isSymbol(value)){return NAN;}return +value;}/**
    	     * The base implementation of `_.toString` which doesn't convert nullish
    	     * values to empty strings.
    	     *
    	     * @private
    	     * @param {*} value The value to process.
    	     * @returns {string} Returns the string.
    	     */function baseToString(value){// Exit early for strings to avoid a performance hit in some environments.
    	if(typeof value=='string'){return value;}if(isArray(value)){// Recursively convert values (susceptible to call stack limits).
    	return arrayMap(value,baseToString)+'';}if(isSymbol(value)){return symbolToString?symbolToString.call(value):'';}var result=value+'';return result=='0'&&1/value==-INFINITY?'-0':result;}/**
    	     * The base implementation of `_.uniqBy` without support for iteratee shorthands.
    	     *
    	     * @private
    	     * @param {Array} array The array to inspect.
    	     * @param {Function} [iteratee] The iteratee invoked per element.
    	     * @param {Function} [comparator] The comparator invoked per element.
    	     * @returns {Array} Returns the new duplicate free array.
    	     */function baseUniq(array,iteratee,comparator){var index=-1,includes=arrayIncludes,length=array.length,isCommon=true,result=[],seen=result;if(comparator){isCommon=false;includes=arrayIncludesWith;}else if(length>=LARGE_ARRAY_SIZE){var set=iteratee?null:createSet(array);if(set){return setToArray(set);}isCommon=false;includes=cacheHas;seen=new SetCache();}else {seen=iteratee?[]:result;}outer:while(++index<length){var value=array[index],computed=iteratee?iteratee(value):value;value=comparator||value!==0?value:0;if(isCommon&&computed===computed){var seenIndex=seen.length;while(seenIndex--){if(seen[seenIndex]===computed){continue outer;}}if(iteratee){seen.push(computed);}result.push(value);}else if(!includes(seen,computed,comparator)){if(seen!==result){seen.push(computed);}result.push(value);}}return result;}/**
    	     * The base implementation of `_.unset`.
    	     *
    	     * @private
    	     * @param {Object} object The object to modify.
    	     * @param {Array|string} path The property path to unset.
    	     * @returns {boolean} Returns `true` if the property is deleted, else `false`.
    	     */function baseUnset(object,path){path=castPath(path,object);object=parent(object,path);return object==null||delete object[toKey(last(path))];}/**
    	     * The base implementation of `_.update`.
    	     *
    	     * @private
    	     * @param {Object} object The object to modify.
    	     * @param {Array|string} path The path of the property to update.
    	     * @param {Function} updater The function to produce the updated value.
    	     * @param {Function} [customizer] The function to customize path creation.
    	     * @returns {Object} Returns `object`.
    	     */function baseUpdate(object,path,updater,customizer){return baseSet(object,path,updater(baseGet(object,path)),customizer);}/**
    	     * The base implementation of methods like `_.dropWhile` and `_.takeWhile`
    	     * without support for iteratee shorthands.
    	     *
    	     * @private
    	     * @param {Array} array The array to query.
    	     * @param {Function} predicate The function invoked per iteration.
    	     * @param {boolean} [isDrop] Specify dropping elements instead of taking them.
    	     * @param {boolean} [fromRight] Specify iterating from right to left.
    	     * @returns {Array} Returns the slice of `array`.
    	     */function baseWhile(array,predicate,isDrop,fromRight){var length=array.length,index=fromRight?length:-1;while((fromRight?index--:++index<length)&&predicate(array[index],index,array)){}return isDrop?baseSlice(array,fromRight?0:index,fromRight?index+1:length):baseSlice(array,fromRight?index+1:0,fromRight?length:index);}/**
    	     * The base implementation of `wrapperValue` which returns the result of
    	     * performing a sequence of actions on the unwrapped `value`, where each
    	     * successive action is supplied the return value of the previous.
    	     *
    	     * @private
    	     * @param {*} value The unwrapped value.
    	     * @param {Array} actions Actions to perform to resolve the unwrapped value.
    	     * @returns {*} Returns the resolved value.
    	     */function baseWrapperValue(value,actions){var result=value;if(result instanceof LazyWrapper){result=result.value();}return arrayReduce(actions,function(result,action){return action.func.apply(action.thisArg,arrayPush([result],action.args));},result);}/**
    	     * The base implementation of methods like `_.xor`, without support for
    	     * iteratee shorthands, that accepts an array of arrays to inspect.
    	     *
    	     * @private
    	     * @param {Array} arrays The arrays to inspect.
    	     * @param {Function} [iteratee] The iteratee invoked per element.
    	     * @param {Function} [comparator] The comparator invoked per element.
    	     * @returns {Array} Returns the new array of values.
    	     */function baseXor(arrays,iteratee,comparator){var length=arrays.length;if(length<2){return length?baseUniq(arrays[0]):[];}var index=-1,result=Array(length);while(++index<length){var array=arrays[index],othIndex=-1;while(++othIndex<length){if(othIndex!=index){result[index]=baseDifference(result[index]||array,arrays[othIndex],iteratee,comparator);}}}return baseUniq(baseFlatten(result,1),iteratee,comparator);}/**
    	     * This base implementation of `_.zipObject` which assigns values using `assignFunc`.
    	     *
    	     * @private
    	     * @param {Array} props The property identifiers.
    	     * @param {Array} values The property values.
    	     * @param {Function} assignFunc The function to assign values.
    	     * @returns {Object} Returns the new object.
    	     */function baseZipObject(props,values,assignFunc){var index=-1,length=props.length,valsLength=values.length,result={};while(++index<length){var value=index<valsLength?values[index]:undefined$1;assignFunc(result,props[index],value);}return result;}/**
    	     * Casts `value` to an empty array if it's not an array like object.
    	     *
    	     * @private
    	     * @param {*} value The value to inspect.
    	     * @returns {Array|Object} Returns the cast array-like object.
    	     */function castArrayLikeObject(value){return isArrayLikeObject(value)?value:[];}/**
    	     * Casts `value` to `identity` if it's not a function.
    	     *
    	     * @private
    	     * @param {*} value The value to inspect.
    	     * @returns {Function} Returns cast function.
    	     */function castFunction(value){return typeof value=='function'?value:identity;}/**
    	     * Casts `value` to a path array if it's not one.
    	     *
    	     * @private
    	     * @param {*} value The value to inspect.
    	     * @param {Object} [object] The object to query keys on.
    	     * @returns {Array} Returns the cast property path array.
    	     */function castPath(value,object){if(isArray(value)){return value;}return isKey(value,object)?[value]:stringToPath(toString(value));}/**
    	     * A `baseRest` alias which can be replaced with `identity` by module
    	     * replacement plugins.
    	     *
    	     * @private
    	     * @type {Function}
    	     * @param {Function} func The function to apply a rest parameter to.
    	     * @returns {Function} Returns the new function.
    	     */var castRest=baseRest;/**
    	     * Casts `array` to a slice if it's needed.
    	     *
    	     * @private
    	     * @param {Array} array The array to inspect.
    	     * @param {number} start The start position.
    	     * @param {number} [end=array.length] The end position.
    	     * @returns {Array} Returns the cast slice.
    	     */function castSlice(array,start,end){var length=array.length;end=end===undefined$1?length:end;return !start&&end>=length?array:baseSlice(array,start,end);}/**
    	     * A simple wrapper around the global [`clearTimeout`](https://mdn.io/clearTimeout).
    	     *
    	     * @private
    	     * @param {number|Object} id The timer id or timeout object of the timer to clear.
    	     */var clearTimeout=ctxClearTimeout||function(id){return root.clearTimeout(id);};/**
    	     * Creates a clone of  `buffer`.
    	     *
    	     * @private
    	     * @param {Buffer} buffer The buffer to clone.
    	     * @param {boolean} [isDeep] Specify a deep clone.
    	     * @returns {Buffer} Returns the cloned buffer.
    	     */function cloneBuffer(buffer,isDeep){if(isDeep){return buffer.slice();}var length=buffer.length,result=allocUnsafe?allocUnsafe(length):new buffer.constructor(length);buffer.copy(result);return result;}/**
    	     * Creates a clone of `arrayBuffer`.
    	     *
    	     * @private
    	     * @param {ArrayBuffer} arrayBuffer The array buffer to clone.
    	     * @returns {ArrayBuffer} Returns the cloned array buffer.
    	     */function cloneArrayBuffer(arrayBuffer){var result=new arrayBuffer.constructor(arrayBuffer.byteLength);new Uint8Array(result).set(new Uint8Array(arrayBuffer));return result;}/**
    	     * Creates a clone of `dataView`.
    	     *
    	     * @private
    	     * @param {Object} dataView The data view to clone.
    	     * @param {boolean} [isDeep] Specify a deep clone.
    	     * @returns {Object} Returns the cloned data view.
    	     */function cloneDataView(dataView,isDeep){var buffer=isDeep?cloneArrayBuffer(dataView.buffer):dataView.buffer;return new dataView.constructor(buffer,dataView.byteOffset,dataView.byteLength);}/**
    	     * Creates a clone of `regexp`.
    	     *
    	     * @private
    	     * @param {Object} regexp The regexp to clone.
    	     * @returns {Object} Returns the cloned regexp.
    	     */function cloneRegExp(regexp){var result=new regexp.constructor(regexp.source,reFlags.exec(regexp));result.lastIndex=regexp.lastIndex;return result;}/**
    	     * Creates a clone of the `symbol` object.
    	     *
    	     * @private
    	     * @param {Object} symbol The symbol object to clone.
    	     * @returns {Object} Returns the cloned symbol object.
    	     */function cloneSymbol(symbol){return symbolValueOf?Object(symbolValueOf.call(symbol)):{};}/**
    	     * Creates a clone of `typedArray`.
    	     *
    	     * @private
    	     * @param {Object} typedArray The typed array to clone.
    	     * @param {boolean} [isDeep] Specify a deep clone.
    	     * @returns {Object} Returns the cloned typed array.
    	     */function cloneTypedArray(typedArray,isDeep){var buffer=isDeep?cloneArrayBuffer(typedArray.buffer):typedArray.buffer;return new typedArray.constructor(buffer,typedArray.byteOffset,typedArray.length);}/**
    	     * Compares values to sort them in ascending order.
    	     *
    	     * @private
    	     * @param {*} value The value to compare.
    	     * @param {*} other The other value to compare.
    	     * @returns {number} Returns the sort order indicator for `value`.
    	     */function compareAscending(value,other){if(value!==other){var valIsDefined=value!==undefined$1,valIsNull=value===null,valIsReflexive=value===value,valIsSymbol=isSymbol(value);var othIsDefined=other!==undefined$1,othIsNull=other===null,othIsReflexive=other===other,othIsSymbol=isSymbol(other);if(!othIsNull&&!othIsSymbol&&!valIsSymbol&&value>other||valIsSymbol&&othIsDefined&&othIsReflexive&&!othIsNull&&!othIsSymbol||valIsNull&&othIsDefined&&othIsReflexive||!valIsDefined&&othIsReflexive||!valIsReflexive){return 1;}if(!valIsNull&&!valIsSymbol&&!othIsSymbol&&value<other||othIsSymbol&&valIsDefined&&valIsReflexive&&!valIsNull&&!valIsSymbol||othIsNull&&valIsDefined&&valIsReflexive||!othIsDefined&&valIsReflexive||!othIsReflexive){return -1;}}return 0;}/**
    	     * Used by `_.orderBy` to compare multiple properties of a value to another
    	     * and stable sort them.
    	     *
    	     * If `orders` is unspecified, all values are sorted in ascending order. Otherwise,
    	     * specify an order of "desc" for descending or "asc" for ascending sort order
    	     * of corresponding values.
    	     *
    	     * @private
    	     * @param {Object} object The object to compare.
    	     * @param {Object} other The other object to compare.
    	     * @param {boolean[]|string[]} orders The order to sort by for each property.
    	     * @returns {number} Returns the sort order indicator for `object`.
    	     */function compareMultiple(object,other,orders){var index=-1,objCriteria=object.criteria,othCriteria=other.criteria,length=objCriteria.length,ordersLength=orders.length;while(++index<length){var result=compareAscending(objCriteria[index],othCriteria[index]);if(result){if(index>=ordersLength){return result;}var order=orders[index];return result*(order=='desc'?-1:1);}}// Fixes an `Array#sort` bug in the JS engine embedded in Adobe applications
    	// that causes it, under certain circumstances, to provide the same value for
    	// `object` and `other`. See https://github.com/jashkenas/underscore/pull/1247
    	// for more details.
    	//
    	// This also ensures a stable sort in V8 and other engines.
    	// See https://bugs.chromium.org/p/v8/issues/detail?id=90 for more details.
    	return object.index-other.index;}/**
    	     * Creates an array that is the composition of partially applied arguments,
    	     * placeholders, and provided arguments into a single array of arguments.
    	     *
    	     * @private
    	     * @param {Array} args The provided arguments.
    	     * @param {Array} partials The arguments to prepend to those provided.
    	     * @param {Array} holders The `partials` placeholder indexes.
    	     * @params {boolean} [isCurried] Specify composing for a curried function.
    	     * @returns {Array} Returns the new array of composed arguments.
    	     */function composeArgs(args,partials,holders,isCurried){var argsIndex=-1,argsLength=args.length,holdersLength=holders.length,leftIndex=-1,leftLength=partials.length,rangeLength=nativeMax(argsLength-holdersLength,0),result=Array(leftLength+rangeLength),isUncurried=!isCurried;while(++leftIndex<leftLength){result[leftIndex]=partials[leftIndex];}while(++argsIndex<holdersLength){if(isUncurried||argsIndex<argsLength){result[holders[argsIndex]]=args[argsIndex];}}while(rangeLength--){result[leftIndex++]=args[argsIndex++];}return result;}/**
    	     * This function is like `composeArgs` except that the arguments composition
    	     * is tailored for `_.partialRight`.
    	     *
    	     * @private
    	     * @param {Array} args The provided arguments.
    	     * @param {Array} partials The arguments to append to those provided.
    	     * @param {Array} holders The `partials` placeholder indexes.
    	     * @params {boolean} [isCurried] Specify composing for a curried function.
    	     * @returns {Array} Returns the new array of composed arguments.
    	     */function composeArgsRight(args,partials,holders,isCurried){var argsIndex=-1,argsLength=args.length,holdersIndex=-1,holdersLength=holders.length,rightIndex=-1,rightLength=partials.length,rangeLength=nativeMax(argsLength-holdersLength,0),result=Array(rangeLength+rightLength),isUncurried=!isCurried;while(++argsIndex<rangeLength){result[argsIndex]=args[argsIndex];}var offset=argsIndex;while(++rightIndex<rightLength){result[offset+rightIndex]=partials[rightIndex];}while(++holdersIndex<holdersLength){if(isUncurried||argsIndex<argsLength){result[offset+holders[holdersIndex]]=args[argsIndex++];}}return result;}/**
    	     * Copies the values of `source` to `array`.
    	     *
    	     * @private
    	     * @param {Array} source The array to copy values from.
    	     * @param {Array} [array=[]] The array to copy values to.
    	     * @returns {Array} Returns `array`.
    	     */function copyArray(source,array){var index=-1,length=source.length;array||(array=Array(length));while(++index<length){array[index]=source[index];}return array;}/**
    	     * Copies properties of `source` to `object`.
    	     *
    	     * @private
    	     * @param {Object} source The object to copy properties from.
    	     * @param {Array} props The property identifiers to copy.
    	     * @param {Object} [object={}] The object to copy properties to.
    	     * @param {Function} [customizer] The function to customize copied values.
    	     * @returns {Object} Returns `object`.
    	     */function copyObject(source,props,object,customizer){var isNew=!object;object||(object={});var index=-1,length=props.length;while(++index<length){var key=props[index];var newValue=customizer?customizer(object[key],source[key],key,object,source):undefined$1;if(newValue===undefined$1){newValue=source[key];}if(isNew){baseAssignValue(object,key,newValue);}else {assignValue(object,key,newValue);}}return object;}/**
    	     * Copies own symbols of `source` to `object`.
    	     *
    	     * @private
    	     * @param {Object} source The object to copy symbols from.
    	     * @param {Object} [object={}] The object to copy symbols to.
    	     * @returns {Object} Returns `object`.
    	     */function copySymbols(source,object){return copyObject(source,getSymbols(source),object);}/**
    	     * Copies own and inherited symbols of `source` to `object`.
    	     *
    	     * @private
    	     * @param {Object} source The object to copy symbols from.
    	     * @param {Object} [object={}] The object to copy symbols to.
    	     * @returns {Object} Returns `object`.
    	     */function copySymbolsIn(source,object){return copyObject(source,getSymbolsIn(source),object);}/**
    	     * Creates a function like `_.groupBy`.
    	     *
    	     * @private
    	     * @param {Function} setter The function to set accumulator values.
    	     * @param {Function} [initializer] The accumulator object initializer.
    	     * @returns {Function} Returns the new aggregator function.
    	     */function createAggregator(setter,initializer){return function(collection,iteratee){var func=isArray(collection)?arrayAggregator:baseAggregator,accumulator=initializer?initializer():{};return func(collection,setter,getIteratee(iteratee,2),accumulator);};}/**
    	     * Creates a function like `_.assign`.
    	     *
    	     * @private
    	     * @param {Function} assigner The function to assign values.
    	     * @returns {Function} Returns the new assigner function.
    	     */function createAssigner(assigner){return baseRest(function(object,sources){var index=-1,length=sources.length,customizer=length>1?sources[length-1]:undefined$1,guard=length>2?sources[2]:undefined$1;customizer=assigner.length>3&&typeof customizer=='function'?(length--,customizer):undefined$1;if(guard&&isIterateeCall(sources[0],sources[1],guard)){customizer=length<3?undefined$1:customizer;length=1;}object=Object(object);while(++index<length){var source=sources[index];if(source){assigner(object,source,index,customizer);}}return object;});}/**
    	     * Creates a `baseEach` or `baseEachRight` function.
    	     *
    	     * @private
    	     * @param {Function} eachFunc The function to iterate over a collection.
    	     * @param {boolean} [fromRight] Specify iterating from right to left.
    	     * @returns {Function} Returns the new base function.
    	     */function createBaseEach(eachFunc,fromRight){return function(collection,iteratee){if(collection==null){return collection;}if(!isArrayLike(collection)){return eachFunc(collection,iteratee);}var length=collection.length,index=fromRight?length:-1,iterable=Object(collection);while(fromRight?index--:++index<length){if(iteratee(iterable[index],index,iterable)===false){break;}}return collection;};}/**
    	     * Creates a base function for methods like `_.forIn` and `_.forOwn`.
    	     *
    	     * @private
    	     * @param {boolean} [fromRight] Specify iterating from right to left.
    	     * @returns {Function} Returns the new base function.
    	     */function createBaseFor(fromRight){return function(object,iteratee,keysFunc){var index=-1,iterable=Object(object),props=keysFunc(object),length=props.length;while(length--){var key=props[fromRight?length:++index];if(iteratee(iterable[key],key,iterable)===false){break;}}return object;};}/**
    	     * Creates a function that wraps `func` to invoke it with the optional `this`
    	     * binding of `thisArg`.
    	     *
    	     * @private
    	     * @param {Function} func The function to wrap.
    	     * @param {number} bitmask The bitmask flags. See `createWrap` for more details.
    	     * @param {*} [thisArg] The `this` binding of `func`.
    	     * @returns {Function} Returns the new wrapped function.
    	     */function createBind(func,bitmask,thisArg){var isBind=bitmask&WRAP_BIND_FLAG,Ctor=createCtor(func);function wrapper(){var fn=this&&this!==root&&this instanceof wrapper?Ctor:func;return fn.apply(isBind?thisArg:this,arguments);}return wrapper;}/**
    	     * Creates a function like `_.lowerFirst`.
    	     *
    	     * @private
    	     * @param {string} methodName The name of the `String` case method to use.
    	     * @returns {Function} Returns the new case function.
    	     */function createCaseFirst(methodName){return function(string){string=toString(string);var strSymbols=hasUnicode(string)?stringToArray(string):undefined$1;var chr=strSymbols?strSymbols[0]:string.charAt(0);var trailing=strSymbols?castSlice(strSymbols,1).join(''):string.slice(1);return chr[methodName]()+trailing;};}/**
    	     * Creates a function like `_.camelCase`.
    	     *
    	     * @private
    	     * @param {Function} callback The function to combine each word.
    	     * @returns {Function} Returns the new compounder function.
    	     */function createCompounder(callback){return function(string){return arrayReduce(words(deburr(string).replace(reApos,'')),callback,'');};}/**
    	     * Creates a function that produces an instance of `Ctor` regardless of
    	     * whether it was invoked as part of a `new` expression or by `call` or `apply`.
    	     *
    	     * @private
    	     * @param {Function} Ctor The constructor to wrap.
    	     * @returns {Function} Returns the new wrapped function.
    	     */function createCtor(Ctor){return function(){// Use a `switch` statement to work with class constructors. See
    	// http://ecma-international.org/ecma-262/7.0/#sec-ecmascript-function-objects-call-thisargument-argumentslist
    	// for more details.
    	var args=arguments;switch(args.length){case 0:return new Ctor();case 1:return new Ctor(args[0]);case 2:return new Ctor(args[0],args[1]);case 3:return new Ctor(args[0],args[1],args[2]);case 4:return new Ctor(args[0],args[1],args[2],args[3]);case 5:return new Ctor(args[0],args[1],args[2],args[3],args[4]);case 6:return new Ctor(args[0],args[1],args[2],args[3],args[4],args[5]);case 7:return new Ctor(args[0],args[1],args[2],args[3],args[4],args[5],args[6]);}var thisBinding=baseCreate(Ctor.prototype),result=Ctor.apply(thisBinding,args);// Mimic the constructor's `return` behavior.
    	// See https://es5.github.io/#x13.2.2 for more details.
    	return isObject(result)?result:thisBinding;};}/**
    	     * Creates a function that wraps `func` to enable currying.
    	     *
    	     * @private
    	     * @param {Function} func The function to wrap.
    	     * @param {number} bitmask The bitmask flags. See `createWrap` for more details.
    	     * @param {number} arity The arity of `func`.
    	     * @returns {Function} Returns the new wrapped function.
    	     */function createCurry(func,bitmask,arity){var Ctor=createCtor(func);function wrapper(){var length=arguments.length,args=Array(length),index=length,placeholder=getHolder(wrapper);while(index--){args[index]=arguments[index];}var holders=length<3&&args[0]!==placeholder&&args[length-1]!==placeholder?[]:replaceHolders(args,placeholder);length-=holders.length;if(length<arity){return createRecurry(func,bitmask,createHybrid,wrapper.placeholder,undefined$1,args,holders,undefined$1,undefined$1,arity-length);}var fn=this&&this!==root&&this instanceof wrapper?Ctor:func;return apply(fn,this,args);}return wrapper;}/**
    	     * Creates a `_.find` or `_.findLast` function.
    	     *
    	     * @private
    	     * @param {Function} findIndexFunc The function to find the collection index.
    	     * @returns {Function} Returns the new find function.
    	     */function createFind(findIndexFunc){return function(collection,predicate,fromIndex){var iterable=Object(collection);if(!isArrayLike(collection)){var iteratee=getIteratee(predicate,3);collection=keys(collection);predicate=function(key){return iteratee(iterable[key],key,iterable);};}var index=findIndexFunc(collection,predicate,fromIndex);return index>-1?iterable[iteratee?collection[index]:index]:undefined$1;};}/**
    	     * Creates a `_.flow` or `_.flowRight` function.
    	     *
    	     * @private
    	     * @param {boolean} [fromRight] Specify iterating from right to left.
    	     * @returns {Function} Returns the new flow function.
    	     */function createFlow(fromRight){return flatRest(function(funcs){var length=funcs.length,index=length,prereq=LodashWrapper.prototype.thru;if(fromRight){funcs.reverse();}while(index--){var func=funcs[index];if(typeof func!='function'){throw new TypeError(FUNC_ERROR_TEXT);}if(prereq&&!wrapper&&getFuncName(func)=='wrapper'){var wrapper=new LodashWrapper([],true);}}index=wrapper?index:length;while(++index<length){func=funcs[index];var funcName=getFuncName(func),data=funcName=='wrapper'?getData(func):undefined$1;if(data&&isLaziable(data[0])&&data[1]==(WRAP_ARY_FLAG|WRAP_CURRY_FLAG|WRAP_PARTIAL_FLAG|WRAP_REARG_FLAG)&&!data[4].length&&data[9]==1){wrapper=wrapper[getFuncName(data[0])].apply(wrapper,data[3]);}else {wrapper=func.length==1&&isLaziable(func)?wrapper[funcName]():wrapper.thru(func);}}return function(){var args=arguments,value=args[0];if(wrapper&&args.length==1&&isArray(value)){return wrapper.plant(value).value();}var index=0,result=length?funcs[index].apply(this,args):value;while(++index<length){result=funcs[index].call(this,result);}return result;};});}/**
    	     * Creates a function that wraps `func` to invoke it with optional `this`
    	     * binding of `thisArg`, partial application, and currying.
    	     *
    	     * @private
    	     * @param {Function|string} func The function or method name to wrap.
    	     * @param {number} bitmask The bitmask flags. See `createWrap` for more details.
    	     * @param {*} [thisArg] The `this` binding of `func`.
    	     * @param {Array} [partials] The arguments to prepend to those provided to
    	     *  the new function.
    	     * @param {Array} [holders] The `partials` placeholder indexes.
    	     * @param {Array} [partialsRight] The arguments to append to those provided
    	     *  to the new function.
    	     * @param {Array} [holdersRight] The `partialsRight` placeholder indexes.
    	     * @param {Array} [argPos] The argument positions of the new function.
    	     * @param {number} [ary] The arity cap of `func`.
    	     * @param {number} [arity] The arity of `func`.
    	     * @returns {Function} Returns the new wrapped function.
    	     */function createHybrid(func,bitmask,thisArg,partials,holders,partialsRight,holdersRight,argPos,ary,arity){var isAry=bitmask&WRAP_ARY_FLAG,isBind=bitmask&WRAP_BIND_FLAG,isBindKey=bitmask&WRAP_BIND_KEY_FLAG,isCurried=bitmask&(WRAP_CURRY_FLAG|WRAP_CURRY_RIGHT_FLAG),isFlip=bitmask&WRAP_FLIP_FLAG,Ctor=isBindKey?undefined$1:createCtor(func);function wrapper(){var length=arguments.length,args=Array(length),index=length;while(index--){args[index]=arguments[index];}if(isCurried){var placeholder=getHolder(wrapper),holdersCount=countHolders(args,placeholder);}if(partials){args=composeArgs(args,partials,holders,isCurried);}if(partialsRight){args=composeArgsRight(args,partialsRight,holdersRight,isCurried);}length-=holdersCount;if(isCurried&&length<arity){var newHolders=replaceHolders(args,placeholder);return createRecurry(func,bitmask,createHybrid,wrapper.placeholder,thisArg,args,newHolders,argPos,ary,arity-length);}var thisBinding=isBind?thisArg:this,fn=isBindKey?thisBinding[func]:func;length=args.length;if(argPos){args=reorder(args,argPos);}else if(isFlip&&length>1){args.reverse();}if(isAry&&ary<length){args.length=ary;}if(this&&this!==root&&this instanceof wrapper){fn=Ctor||createCtor(fn);}return fn.apply(thisBinding,args);}return wrapper;}/**
    	     * Creates a function like `_.invertBy`.
    	     *
    	     * @private
    	     * @param {Function} setter The function to set accumulator values.
    	     * @param {Function} toIteratee The function to resolve iteratees.
    	     * @returns {Function} Returns the new inverter function.
    	     */function createInverter(setter,toIteratee){return function(object,iteratee){return baseInverter(object,setter,toIteratee(iteratee),{});};}/**
    	     * Creates a function that performs a mathematical operation on two values.
    	     *
    	     * @private
    	     * @param {Function} operator The function to perform the operation.
    	     * @param {number} [defaultValue] The value used for `undefined` arguments.
    	     * @returns {Function} Returns the new mathematical operation function.
    	     */function createMathOperation(operator,defaultValue){return function(value,other){var result;if(value===undefined$1&&other===undefined$1){return defaultValue;}if(value!==undefined$1){result=value;}if(other!==undefined$1){if(result===undefined$1){return other;}if(typeof value=='string'||typeof other=='string'){value=baseToString(value);other=baseToString(other);}else {value=baseToNumber(value);other=baseToNumber(other);}result=operator(value,other);}return result;};}/**
    	     * Creates a function like `_.over`.
    	     *
    	     * @private
    	     * @param {Function} arrayFunc The function to iterate over iteratees.
    	     * @returns {Function} Returns the new over function.
    	     */function createOver(arrayFunc){return flatRest(function(iteratees){iteratees=arrayMap(iteratees,baseUnary(getIteratee()));return baseRest(function(args){var thisArg=this;return arrayFunc(iteratees,function(iteratee){return apply(iteratee,thisArg,args);});});});}/**
    	     * Creates the padding for `string` based on `length`. The `chars` string
    	     * is truncated if the number of characters exceeds `length`.
    	     *
    	     * @private
    	     * @param {number} length The padding length.
    	     * @param {string} [chars=' '] The string used as padding.
    	     * @returns {string} Returns the padding for `string`.
    	     */function createPadding(length,chars){chars=chars===undefined$1?' ':baseToString(chars);var charsLength=chars.length;if(charsLength<2){return charsLength?baseRepeat(chars,length):chars;}var result=baseRepeat(chars,nativeCeil(length/stringSize(chars)));return hasUnicode(chars)?castSlice(stringToArray(result),0,length).join(''):result.slice(0,length);}/**
    	     * Creates a function that wraps `func` to invoke it with the `this` binding
    	     * of `thisArg` and `partials` prepended to the arguments it receives.
    	     *
    	     * @private
    	     * @param {Function} func The function to wrap.
    	     * @param {number} bitmask The bitmask flags. See `createWrap` for more details.
    	     * @param {*} thisArg The `this` binding of `func`.
    	     * @param {Array} partials The arguments to prepend to those provided to
    	     *  the new function.
    	     * @returns {Function} Returns the new wrapped function.
    	     */function createPartial(func,bitmask,thisArg,partials){var isBind=bitmask&WRAP_BIND_FLAG,Ctor=createCtor(func);function wrapper(){var argsIndex=-1,argsLength=arguments.length,leftIndex=-1,leftLength=partials.length,args=Array(leftLength+argsLength),fn=this&&this!==root&&this instanceof wrapper?Ctor:func;while(++leftIndex<leftLength){args[leftIndex]=partials[leftIndex];}while(argsLength--){args[leftIndex++]=arguments[++argsIndex];}return apply(fn,isBind?thisArg:this,args);}return wrapper;}/**
    	     * Creates a `_.range` or `_.rangeRight` function.
    	     *
    	     * @private
    	     * @param {boolean} [fromRight] Specify iterating from right to left.
    	     * @returns {Function} Returns the new range function.
    	     */function createRange(fromRight){return function(start,end,step){if(step&&typeof step!='number'&&isIterateeCall(start,end,step)){end=step=undefined$1;}// Ensure the sign of `-0` is preserved.
    	start=toFinite(start);if(end===undefined$1){end=start;start=0;}else {end=toFinite(end);}step=step===undefined$1?start<end?1:-1:toFinite(step);return baseRange(start,end,step,fromRight);};}/**
    	     * Creates a function that performs a relational operation on two values.
    	     *
    	     * @private
    	     * @param {Function} operator The function to perform the operation.
    	     * @returns {Function} Returns the new relational operation function.
    	     */function createRelationalOperation(operator){return function(value,other){if(!(typeof value=='string'&&typeof other=='string')){value=toNumber(value);other=toNumber(other);}return operator(value,other);};}/**
    	     * Creates a function that wraps `func` to continue currying.
    	     *
    	     * @private
    	     * @param {Function} func The function to wrap.
    	     * @param {number} bitmask The bitmask flags. See `createWrap` for more details.
    	     * @param {Function} wrapFunc The function to create the `func` wrapper.
    	     * @param {*} placeholder The placeholder value.
    	     * @param {*} [thisArg] The `this` binding of `func`.
    	     * @param {Array} [partials] The arguments to prepend to those provided to
    	     *  the new function.
    	     * @param {Array} [holders] The `partials` placeholder indexes.
    	     * @param {Array} [argPos] The argument positions of the new function.
    	     * @param {number} [ary] The arity cap of `func`.
    	     * @param {number} [arity] The arity of `func`.
    	     * @returns {Function} Returns the new wrapped function.
    	     */function createRecurry(func,bitmask,wrapFunc,placeholder,thisArg,partials,holders,argPos,ary,arity){var isCurry=bitmask&WRAP_CURRY_FLAG,newHolders=isCurry?holders:undefined$1,newHoldersRight=isCurry?undefined$1:holders,newPartials=isCurry?partials:undefined$1,newPartialsRight=isCurry?undefined$1:partials;bitmask|=isCurry?WRAP_PARTIAL_FLAG:WRAP_PARTIAL_RIGHT_FLAG;bitmask&=~(isCurry?WRAP_PARTIAL_RIGHT_FLAG:WRAP_PARTIAL_FLAG);if(!(bitmask&WRAP_CURRY_BOUND_FLAG)){bitmask&=~(WRAP_BIND_FLAG|WRAP_BIND_KEY_FLAG);}var newData=[func,bitmask,thisArg,newPartials,newHolders,newPartialsRight,newHoldersRight,argPos,ary,arity];var result=wrapFunc.apply(undefined$1,newData);if(isLaziable(func)){setData(result,newData);}result.placeholder=placeholder;return setWrapToString(result,func,bitmask);}/**
    	     * Creates a function like `_.round`.
    	     *
    	     * @private
    	     * @param {string} methodName The name of the `Math` method to use when rounding.
    	     * @returns {Function} Returns the new round function.
    	     */function createRound(methodName){var func=Math[methodName];return function(number,precision){number=toNumber(number);precision=precision==null?0:nativeMin(toInteger(precision),292);if(precision&&nativeIsFinite(number)){// Shift with exponential notation to avoid floating-point issues.
    	// See [MDN](https://mdn.io/round#Examples) for more details.
    	var pair=(toString(number)+'e').split('e'),value=func(pair[0]+'e'+(+pair[1]+precision));pair=(toString(value)+'e').split('e');return +(pair[0]+'e'+(+pair[1]-precision));}return func(number);};}/**
    	     * Creates a set object of `values`.
    	     *
    	     * @private
    	     * @param {Array} values The values to add to the set.
    	     * @returns {Object} Returns the new set.
    	     */var createSet=!(Set&&1/setToArray(new Set([,-0]))[1]==INFINITY)?noop:function(values){return new Set(values);};/**
    	     * Creates a `_.toPairs` or `_.toPairsIn` function.
    	     *
    	     * @private
    	     * @param {Function} keysFunc The function to get the keys of a given object.
    	     * @returns {Function} Returns the new pairs function.
    	     */function createToPairs(keysFunc){return function(object){var tag=getTag(object);if(tag==mapTag){return mapToArray(object);}if(tag==setTag){return setToPairs(object);}return baseToPairs(object,keysFunc(object));};}/**
    	     * Creates a function that either curries or invokes `func` with optional
    	     * `this` binding and partially applied arguments.
    	     *
    	     * @private
    	     * @param {Function|string} func The function or method name to wrap.
    	     * @param {number} bitmask The bitmask flags.
    	     *    1 - `_.bind`
    	     *    2 - `_.bindKey`
    	     *    4 - `_.curry` or `_.curryRight` of a bound function
    	     *    8 - `_.curry`
    	     *   16 - `_.curryRight`
    	     *   32 - `_.partial`
    	     *   64 - `_.partialRight`
    	     *  128 - `_.rearg`
    	     *  256 - `_.ary`
    	     *  512 - `_.flip`
    	     * @param {*} [thisArg] The `this` binding of `func`.
    	     * @param {Array} [partials] The arguments to be partially applied.
    	     * @param {Array} [holders] The `partials` placeholder indexes.
    	     * @param {Array} [argPos] The argument positions of the new function.
    	     * @param {number} [ary] The arity cap of `func`.
    	     * @param {number} [arity] The arity of `func`.
    	     * @returns {Function} Returns the new wrapped function.
    	     */function createWrap(func,bitmask,thisArg,partials,holders,argPos,ary,arity){var isBindKey=bitmask&WRAP_BIND_KEY_FLAG;if(!isBindKey&&typeof func!='function'){throw new TypeError(FUNC_ERROR_TEXT);}var length=partials?partials.length:0;if(!length){bitmask&=~(WRAP_PARTIAL_FLAG|WRAP_PARTIAL_RIGHT_FLAG);partials=holders=undefined$1;}ary=ary===undefined$1?ary:nativeMax(toInteger(ary),0);arity=arity===undefined$1?arity:toInteger(arity);length-=holders?holders.length:0;if(bitmask&WRAP_PARTIAL_RIGHT_FLAG){var partialsRight=partials,holdersRight=holders;partials=holders=undefined$1;}var data=isBindKey?undefined$1:getData(func);var newData=[func,bitmask,thisArg,partials,holders,partialsRight,holdersRight,argPos,ary,arity];if(data){mergeData(newData,data);}func=newData[0];bitmask=newData[1];thisArg=newData[2];partials=newData[3];holders=newData[4];arity=newData[9]=newData[9]===undefined$1?isBindKey?0:func.length:nativeMax(newData[9]-length,0);if(!arity&&bitmask&(WRAP_CURRY_FLAG|WRAP_CURRY_RIGHT_FLAG)){bitmask&=~(WRAP_CURRY_FLAG|WRAP_CURRY_RIGHT_FLAG);}if(!bitmask||bitmask==WRAP_BIND_FLAG){var result=createBind(func,bitmask,thisArg);}else if(bitmask==WRAP_CURRY_FLAG||bitmask==WRAP_CURRY_RIGHT_FLAG){result=createCurry(func,bitmask,arity);}else if((bitmask==WRAP_PARTIAL_FLAG||bitmask==(WRAP_BIND_FLAG|WRAP_PARTIAL_FLAG))&&!holders.length){result=createPartial(func,bitmask,thisArg,partials);}else {result=createHybrid.apply(undefined$1,newData);}var setter=data?baseSetData:setData;return setWrapToString(setter(result,newData),func,bitmask);}/**
    	     * Used by `_.defaults` to customize its `_.assignIn` use to assign properties
    	     * of source objects to the destination object for all destination properties
    	     * that resolve to `undefined`.
    	     *
    	     * @private
    	     * @param {*} objValue The destination value.
    	     * @param {*} srcValue The source value.
    	     * @param {string} key The key of the property to assign.
    	     * @param {Object} object The parent object of `objValue`.
    	     * @returns {*} Returns the value to assign.
    	     */function customDefaultsAssignIn(objValue,srcValue,key,object){if(objValue===undefined$1||eq(objValue,objectProto[key])&&!hasOwnProperty.call(object,key)){return srcValue;}return objValue;}/**
    	     * Used by `_.defaultsDeep` to customize its `_.merge` use to merge source
    	     * objects into destination objects that are passed thru.
    	     *
    	     * @private
    	     * @param {*} objValue The destination value.
    	     * @param {*} srcValue The source value.
    	     * @param {string} key The key of the property to merge.
    	     * @param {Object} object The parent object of `objValue`.
    	     * @param {Object} source The parent object of `srcValue`.
    	     * @param {Object} [stack] Tracks traversed source values and their merged
    	     *  counterparts.
    	     * @returns {*} Returns the value to assign.
    	     */function customDefaultsMerge(objValue,srcValue,key,object,source,stack){if(isObject(objValue)&&isObject(srcValue)){// Recursively merge objects and arrays (susceptible to call stack limits).
    	stack.set(srcValue,objValue);baseMerge(objValue,srcValue,undefined$1,customDefaultsMerge,stack);stack['delete'](srcValue);}return objValue;}/**
    	     * Used by `_.omit` to customize its `_.cloneDeep` use to only clone plain
    	     * objects.
    	     *
    	     * @private
    	     * @param {*} value The value to inspect.
    	     * @param {string} key The key of the property to inspect.
    	     * @returns {*} Returns the uncloned value or `undefined` to defer cloning to `_.cloneDeep`.
    	     */function customOmitClone(value){return isPlainObject(value)?undefined$1:value;}/**
    	     * A specialized version of `baseIsEqualDeep` for arrays with support for
    	     * partial deep comparisons.
    	     *
    	     * @private
    	     * @param {Array} array The array to compare.
    	     * @param {Array} other The other array to compare.
    	     * @param {number} bitmask The bitmask flags. See `baseIsEqual` for more details.
    	     * @param {Function} customizer The function to customize comparisons.
    	     * @param {Function} equalFunc The function to determine equivalents of values.
    	     * @param {Object} stack Tracks traversed `array` and `other` objects.
    	     * @returns {boolean} Returns `true` if the arrays are equivalent, else `false`.
    	     */function equalArrays(array,other,bitmask,customizer,equalFunc,stack){var isPartial=bitmask&COMPARE_PARTIAL_FLAG,arrLength=array.length,othLength=other.length;if(arrLength!=othLength&&!(isPartial&&othLength>arrLength)){return false;}// Check that cyclic values are equal.
    	var arrStacked=stack.get(array);var othStacked=stack.get(other);if(arrStacked&&othStacked){return arrStacked==other&&othStacked==array;}var index=-1,result=true,seen=bitmask&COMPARE_UNORDERED_FLAG?new SetCache():undefined$1;stack.set(array,other);stack.set(other,array);// Ignore non-index properties.
    	while(++index<arrLength){var arrValue=array[index],othValue=other[index];if(customizer){var compared=isPartial?customizer(othValue,arrValue,index,other,array,stack):customizer(arrValue,othValue,index,array,other,stack);}if(compared!==undefined$1){if(compared){continue;}result=false;break;}// Recursively compare arrays (susceptible to call stack limits).
    	if(seen){if(!arraySome(other,function(othValue,othIndex){if(!cacheHas(seen,othIndex)&&(arrValue===othValue||equalFunc(arrValue,othValue,bitmask,customizer,stack))){return seen.push(othIndex);}})){result=false;break;}}else if(!(arrValue===othValue||equalFunc(arrValue,othValue,bitmask,customizer,stack))){result=false;break;}}stack['delete'](array);stack['delete'](other);return result;}/**
    	     * A specialized version of `baseIsEqualDeep` for comparing objects of
    	     * the same `toStringTag`.
    	     *
    	     * **Note:** This function only supports comparing values with tags of
    	     * `Boolean`, `Date`, `Error`, `Number`, `RegExp`, or `String`.
    	     *
    	     * @private
    	     * @param {Object} object The object to compare.
    	     * @param {Object} other The other object to compare.
    	     * @param {string} tag The `toStringTag` of the objects to compare.
    	     * @param {number} bitmask The bitmask flags. See `baseIsEqual` for more details.
    	     * @param {Function} customizer The function to customize comparisons.
    	     * @param {Function} equalFunc The function to determine equivalents of values.
    	     * @param {Object} stack Tracks traversed `object` and `other` objects.
    	     * @returns {boolean} Returns `true` if the objects are equivalent, else `false`.
    	     */function equalByTag(object,other,tag,bitmask,customizer,equalFunc,stack){switch(tag){case dataViewTag:if(object.byteLength!=other.byteLength||object.byteOffset!=other.byteOffset){return false;}object=object.buffer;other=other.buffer;case arrayBufferTag:if(object.byteLength!=other.byteLength||!equalFunc(new Uint8Array(object),new Uint8Array(other))){return false;}return true;case boolTag:case dateTag:case numberTag:// Coerce booleans to `1` or `0` and dates to milliseconds.
    	// Invalid dates are coerced to `NaN`.
    	return eq(+object,+other);case errorTag:return object.name==other.name&&object.message==other.message;case regexpTag:case stringTag:// Coerce regexes to strings and treat strings, primitives and objects,
    	// as equal. See http://www.ecma-international.org/ecma-262/7.0/#sec-regexp.prototype.tostring
    	// for more details.
    	return object==other+'';case mapTag:var convert=mapToArray;case setTag:var isPartial=bitmask&COMPARE_PARTIAL_FLAG;convert||(convert=setToArray);if(object.size!=other.size&&!isPartial){return false;}// Assume cyclic values are equal.
    	var stacked=stack.get(object);if(stacked){return stacked==other;}bitmask|=COMPARE_UNORDERED_FLAG;// Recursively compare objects (susceptible to call stack limits).
    	stack.set(object,other);var result=equalArrays(convert(object),convert(other),bitmask,customizer,equalFunc,stack);stack['delete'](object);return result;case symbolTag:if(symbolValueOf){return symbolValueOf.call(object)==symbolValueOf.call(other);}}return false;}/**
    	     * A specialized version of `baseIsEqualDeep` for objects with support for
    	     * partial deep comparisons.
    	     *
    	     * @private
    	     * @param {Object} object The object to compare.
    	     * @param {Object} other The other object to compare.
    	     * @param {number} bitmask The bitmask flags. See `baseIsEqual` for more details.
    	     * @param {Function} customizer The function to customize comparisons.
    	     * @param {Function} equalFunc The function to determine equivalents of values.
    	     * @param {Object} stack Tracks traversed `object` and `other` objects.
    	     * @returns {boolean} Returns `true` if the objects are equivalent, else `false`.
    	     */function equalObjects(object,other,bitmask,customizer,equalFunc,stack){var isPartial=bitmask&COMPARE_PARTIAL_FLAG,objProps=getAllKeys(object),objLength=objProps.length,othProps=getAllKeys(other),othLength=othProps.length;if(objLength!=othLength&&!isPartial){return false;}var index=objLength;while(index--){var key=objProps[index];if(!(isPartial?key in other:hasOwnProperty.call(other,key))){return false;}}// Check that cyclic values are equal.
    	var objStacked=stack.get(object);var othStacked=stack.get(other);if(objStacked&&othStacked){return objStacked==other&&othStacked==object;}var result=true;stack.set(object,other);stack.set(other,object);var skipCtor=isPartial;while(++index<objLength){key=objProps[index];var objValue=object[key],othValue=other[key];if(customizer){var compared=isPartial?customizer(othValue,objValue,key,other,object,stack):customizer(objValue,othValue,key,object,other,stack);}// Recursively compare objects (susceptible to call stack limits).
    	if(!(compared===undefined$1?objValue===othValue||equalFunc(objValue,othValue,bitmask,customizer,stack):compared)){result=false;break;}skipCtor||(skipCtor=key=='constructor');}if(result&&!skipCtor){var objCtor=object.constructor,othCtor=other.constructor;// Non `Object` object instances with different constructors are not equal.
    	if(objCtor!=othCtor&&'constructor'in object&&'constructor'in other&&!(typeof objCtor=='function'&&objCtor instanceof objCtor&&typeof othCtor=='function'&&othCtor instanceof othCtor)){result=false;}}stack['delete'](object);stack['delete'](other);return result;}/**
    	     * A specialized version of `baseRest` which flattens the rest array.
    	     *
    	     * @private
    	     * @param {Function} func The function to apply a rest parameter to.
    	     * @returns {Function} Returns the new function.
    	     */function flatRest(func){return setToString(overRest(func,undefined$1,flatten),func+'');}/**
    	     * Creates an array of own enumerable property names and symbols of `object`.
    	     *
    	     * @private
    	     * @param {Object} object The object to query.
    	     * @returns {Array} Returns the array of property names and symbols.
    	     */function getAllKeys(object){return baseGetAllKeys(object,keys,getSymbols);}/**
    	     * Creates an array of own and inherited enumerable property names and
    	     * symbols of `object`.
    	     *
    	     * @private
    	     * @param {Object} object The object to query.
    	     * @returns {Array} Returns the array of property names and symbols.
    	     */function getAllKeysIn(object){return baseGetAllKeys(object,keysIn,getSymbolsIn);}/**
    	     * Gets metadata for `func`.
    	     *
    	     * @private
    	     * @param {Function} func The function to query.
    	     * @returns {*} Returns the metadata for `func`.
    	     */var getData=!metaMap?noop:function(func){return metaMap.get(func);};/**
    	     * Gets the name of `func`.
    	     *
    	     * @private
    	     * @param {Function} func The function to query.
    	     * @returns {string} Returns the function name.
    	     */function getFuncName(func){var result=func.name+'',array=realNames[result],length=hasOwnProperty.call(realNames,result)?array.length:0;while(length--){var data=array[length],otherFunc=data.func;if(otherFunc==null||otherFunc==func){return data.name;}}return result;}/**
    	     * Gets the argument placeholder value for `func`.
    	     *
    	     * @private
    	     * @param {Function} func The function to inspect.
    	     * @returns {*} Returns the placeholder value.
    	     */function getHolder(func){var object=hasOwnProperty.call(lodash,'placeholder')?lodash:func;return object.placeholder;}/**
    	     * Gets the appropriate "iteratee" function. If `_.iteratee` is customized,
    	     * this function returns the custom method, otherwise it returns `baseIteratee`.
    	     * If arguments are provided, the chosen function is invoked with them and
    	     * its result is returned.
    	     *
    	     * @private
    	     * @param {*} [value] The value to convert to an iteratee.
    	     * @param {number} [arity] The arity of the created iteratee.
    	     * @returns {Function} Returns the chosen function or its result.
    	     */function getIteratee(){var result=lodash.iteratee||iteratee;result=result===iteratee?baseIteratee:result;return arguments.length?result(arguments[0],arguments[1]):result;}/**
    	     * Gets the data for `map`.
    	     *
    	     * @private
    	     * @param {Object} map The map to query.
    	     * @param {string} key The reference key.
    	     * @returns {*} Returns the map data.
    	     */function getMapData(map,key){var data=map.__data__;return isKeyable(key)?data[typeof key=='string'?'string':'hash']:data.map;}/**
    	     * Gets the property names, values, and compare flags of `object`.
    	     *
    	     * @private
    	     * @param {Object} object The object to query.
    	     * @returns {Array} Returns the match data of `object`.
    	     */function getMatchData(object){var result=keys(object),length=result.length;while(length--){var key=result[length],value=object[key];result[length]=[key,value,isStrictComparable(value)];}return result;}/**
    	     * Gets the native function at `key` of `object`.
    	     *
    	     * @private
    	     * @param {Object} object The object to query.
    	     * @param {string} key The key of the method to get.
    	     * @returns {*} Returns the function if it's native, else `undefined`.
    	     */function getNative(object,key){var value=getValue(object,key);return baseIsNative(value)?value:undefined$1;}/**
    	     * A specialized version of `baseGetTag` which ignores `Symbol.toStringTag` values.
    	     *
    	     * @private
    	     * @param {*} value The value to query.
    	     * @returns {string} Returns the raw `toStringTag`.
    	     */function getRawTag(value){var isOwn=hasOwnProperty.call(value,symToStringTag),tag=value[symToStringTag];try{value[symToStringTag]=undefined$1;var unmasked=true;}catch(e){}var result=nativeObjectToString.call(value);if(unmasked){if(isOwn){value[symToStringTag]=tag;}else {delete value[symToStringTag];}}return result;}/**
    	     * Creates an array of the own enumerable symbols of `object`.
    	     *
    	     * @private
    	     * @param {Object} object The object to query.
    	     * @returns {Array} Returns the array of symbols.
    	     */var getSymbols=!nativeGetSymbols?stubArray:function(object){if(object==null){return [];}object=Object(object);return arrayFilter(nativeGetSymbols(object),function(symbol){return propertyIsEnumerable.call(object,symbol);});};/**
    	     * Creates an array of the own and inherited enumerable symbols of `object`.
    	     *
    	     * @private
    	     * @param {Object} object The object to query.
    	     * @returns {Array} Returns the array of symbols.
    	     */var getSymbolsIn=!nativeGetSymbols?stubArray:function(object){var result=[];while(object){arrayPush(result,getSymbols(object));object=getPrototype(object);}return result;};/**
    	     * Gets the `toStringTag` of `value`.
    	     *
    	     * @private
    	     * @param {*} value The value to query.
    	     * @returns {string} Returns the `toStringTag`.
    	     */var getTag=baseGetTag;// Fallback for data views, maps, sets, and weak maps in IE 11 and promises in Node.js < 6.
    	if(DataView&&getTag(new DataView(new ArrayBuffer(1)))!=dataViewTag||Map&&getTag(new Map())!=mapTag||Promise&&getTag(Promise.resolve())!=promiseTag||Set&&getTag(new Set())!=setTag||WeakMap&&getTag(new WeakMap())!=weakMapTag){getTag=function(value){var result=baseGetTag(value),Ctor=result==objectTag?value.constructor:undefined$1,ctorString=Ctor?toSource(Ctor):'';if(ctorString){switch(ctorString){case dataViewCtorString:return dataViewTag;case mapCtorString:return mapTag;case promiseCtorString:return promiseTag;case setCtorString:return setTag;case weakMapCtorString:return weakMapTag;}}return result;};}/**
    	     * Gets the view, applying any `transforms` to the `start` and `end` positions.
    	     *
    	     * @private
    	     * @param {number} start The start of the view.
    	     * @param {number} end The end of the view.
    	     * @param {Array} transforms The transformations to apply to the view.
    	     * @returns {Object} Returns an object containing the `start` and `end`
    	     *  positions of the view.
    	     */function getView(start,end,transforms){var index=-1,length=transforms.length;while(++index<length){var data=transforms[index],size=data.size;switch(data.type){case'drop':start+=size;break;case'dropRight':end-=size;break;case'take':end=nativeMin(end,start+size);break;case'takeRight':start=nativeMax(start,end-size);break;}}return {'start':start,'end':end};}/**
    	     * Extracts wrapper details from the `source` body comment.
    	     *
    	     * @private
    	     * @param {string} source The source to inspect.
    	     * @returns {Array} Returns the wrapper details.
    	     */function getWrapDetails(source){var match=source.match(reWrapDetails);return match?match[1].split(reSplitDetails):[];}/**
    	     * Checks if `path` exists on `object`.
    	     *
    	     * @private
    	     * @param {Object} object The object to query.
    	     * @param {Array|string} path The path to check.
    	     * @param {Function} hasFunc The function to check properties.
    	     * @returns {boolean} Returns `true` if `path` exists, else `false`.
    	     */function hasPath(object,path,hasFunc){path=castPath(path,object);var index=-1,length=path.length,result=false;while(++index<length){var key=toKey(path[index]);if(!(result=object!=null&&hasFunc(object,key))){break;}object=object[key];}if(result||++index!=length){return result;}length=object==null?0:object.length;return !!length&&isLength(length)&&isIndex(key,length)&&(isArray(object)||isArguments(object));}/**
    	     * Initializes an array clone.
    	     *
    	     * @private
    	     * @param {Array} array The array to clone.
    	     * @returns {Array} Returns the initialized clone.
    	     */function initCloneArray(array){var length=array.length,result=new array.constructor(length);// Add properties assigned by `RegExp#exec`.
    	if(length&&typeof array[0]=='string'&&hasOwnProperty.call(array,'index')){result.index=array.index;result.input=array.input;}return result;}/**
    	     * Initializes an object clone.
    	     *
    	     * @private
    	     * @param {Object} object The object to clone.
    	     * @returns {Object} Returns the initialized clone.
    	     */function initCloneObject(object){return typeof object.constructor=='function'&&!isPrototype(object)?baseCreate(getPrototype(object)):{};}/**
    	     * Initializes an object clone based on its `toStringTag`.
    	     *
    	     * **Note:** This function only supports cloning values with tags of
    	     * `Boolean`, `Date`, `Error`, `Map`, `Number`, `RegExp`, `Set`, or `String`.
    	     *
    	     * @private
    	     * @param {Object} object The object to clone.
    	     * @param {string} tag The `toStringTag` of the object to clone.
    	     * @param {boolean} [isDeep] Specify a deep clone.
    	     * @returns {Object} Returns the initialized clone.
    	     */function initCloneByTag(object,tag,isDeep){var Ctor=object.constructor;switch(tag){case arrayBufferTag:return cloneArrayBuffer(object);case boolTag:case dateTag:return new Ctor(+object);case dataViewTag:return cloneDataView(object,isDeep);case float32Tag:case float64Tag:case int8Tag:case int16Tag:case int32Tag:case uint8Tag:case uint8ClampedTag:case uint16Tag:case uint32Tag:return cloneTypedArray(object,isDeep);case mapTag:return new Ctor();case numberTag:case stringTag:return new Ctor(object);case regexpTag:return cloneRegExp(object);case setTag:return new Ctor();case symbolTag:return cloneSymbol(object);}}/**
    	     * Inserts wrapper `details` in a comment at the top of the `source` body.
    	     *
    	     * @private
    	     * @param {string} source The source to modify.
    	     * @returns {Array} details The details to insert.
    	     * @returns {string} Returns the modified source.
    	     */function insertWrapDetails(source,details){var length=details.length;if(!length){return source;}var lastIndex=length-1;details[lastIndex]=(length>1?'& ':'')+details[lastIndex];details=details.join(length>2?', ':' ');return source.replace(reWrapComment,'{\n/* [wrapped with '+details+'] */\n');}/**
    	     * Checks if `value` is a flattenable `arguments` object or array.
    	     *
    	     * @private
    	     * @param {*} value The value to check.
    	     * @returns {boolean} Returns `true` if `value` is flattenable, else `false`.
    	     */function isFlattenable(value){return isArray(value)||isArguments(value)||!!(spreadableSymbol&&value&&value[spreadableSymbol]);}/**
    	     * Checks if `value` is a valid array-like index.
    	     *
    	     * @private
    	     * @param {*} value The value to check.
    	     * @param {number} [length=MAX_SAFE_INTEGER] The upper bounds of a valid index.
    	     * @returns {boolean} Returns `true` if `value` is a valid index, else `false`.
    	     */function isIndex(value,length){var type=typeof value;length=length==null?MAX_SAFE_INTEGER:length;return !!length&&(type=='number'||type!='symbol'&&reIsUint.test(value))&&value>-1&&value%1==0&&value<length;}/**
    	     * Checks if the given arguments are from an iteratee call.
    	     *
    	     * @private
    	     * @param {*} value The potential iteratee value argument.
    	     * @param {*} index The potential iteratee index or key argument.
    	     * @param {*} object The potential iteratee object argument.
    	     * @returns {boolean} Returns `true` if the arguments are from an iteratee call,
    	     *  else `false`.
    	     */function isIterateeCall(value,index,object){if(!isObject(object)){return false;}var type=typeof index;if(type=='number'?isArrayLike(object)&&isIndex(index,object.length):type=='string'&&index in object){return eq(object[index],value);}return false;}/**
    	     * Checks if `value` is a property name and not a property path.
    	     *
    	     * @private
    	     * @param {*} value The value to check.
    	     * @param {Object} [object] The object to query keys on.
    	     * @returns {boolean} Returns `true` if `value` is a property name, else `false`.
    	     */function isKey(value,object){if(isArray(value)){return false;}var type=typeof value;if(type=='number'||type=='symbol'||type=='boolean'||value==null||isSymbol(value)){return true;}return reIsPlainProp.test(value)||!reIsDeepProp.test(value)||object!=null&&value in Object(object);}/**
    	     * Checks if `value` is suitable for use as unique object key.
    	     *
    	     * @private
    	     * @param {*} value The value to check.
    	     * @returns {boolean} Returns `true` if `value` is suitable, else `false`.
    	     */function isKeyable(value){var type=typeof value;return type=='string'||type=='number'||type=='symbol'||type=='boolean'?value!=='__proto__':value===null;}/**
    	     * Checks if `func` has a lazy counterpart.
    	     *
    	     * @private
    	     * @param {Function} func The function to check.
    	     * @returns {boolean} Returns `true` if `func` has a lazy counterpart,
    	     *  else `false`.
    	     */function isLaziable(func){var funcName=getFuncName(func),other=lodash[funcName];if(typeof other!='function'||!(funcName in LazyWrapper.prototype)){return false;}if(func===other){return true;}var data=getData(other);return !!data&&func===data[0];}/**
    	     * Checks if `func` has its source masked.
    	     *
    	     * @private
    	     * @param {Function} func The function to check.
    	     * @returns {boolean} Returns `true` if `func` is masked, else `false`.
    	     */function isMasked(func){return !!maskSrcKey&&maskSrcKey in func;}/**
    	     * Checks if `func` is capable of being masked.
    	     *
    	     * @private
    	     * @param {*} value The value to check.
    	     * @returns {boolean} Returns `true` if `func` is maskable, else `false`.
    	     */var isMaskable=coreJsData?isFunction:stubFalse;/**
    	     * Checks if `value` is likely a prototype object.
    	     *
    	     * @private
    	     * @param {*} value The value to check.
    	     * @returns {boolean} Returns `true` if `value` is a prototype, else `false`.
    	     */function isPrototype(value){var Ctor=value&&value.constructor,proto=typeof Ctor=='function'&&Ctor.prototype||objectProto;return value===proto;}/**
    	     * Checks if `value` is suitable for strict equality comparisons, i.e. `===`.
    	     *
    	     * @private
    	     * @param {*} value The value to check.
    	     * @returns {boolean} Returns `true` if `value` if suitable for strict
    	     *  equality comparisons, else `false`.
    	     */function isStrictComparable(value){return value===value&&!isObject(value);}/**
    	     * A specialized version of `matchesProperty` for source values suitable
    	     * for strict equality comparisons, i.e. `===`.
    	     *
    	     * @private
    	     * @param {string} key The key of the property to get.
    	     * @param {*} srcValue The value to match.
    	     * @returns {Function} Returns the new spec function.
    	     */function matchesStrictComparable(key,srcValue){return function(object){if(object==null){return false;}return object[key]===srcValue&&(srcValue!==undefined$1||key in Object(object));};}/**
    	     * A specialized version of `_.memoize` which clears the memoized function's
    	     * cache when it exceeds `MAX_MEMOIZE_SIZE`.
    	     *
    	     * @private
    	     * @param {Function} func The function to have its output memoized.
    	     * @returns {Function} Returns the new memoized function.
    	     */function memoizeCapped(func){var result=memoize(func,function(key){if(cache.size===MAX_MEMOIZE_SIZE){cache.clear();}return key;});var cache=result.cache;return result;}/**
    	     * Merges the function metadata of `source` into `data`.
    	     *
    	     * Merging metadata reduces the number of wrappers used to invoke a function.
    	     * This is possible because methods like `_.bind`, `_.curry`, and `_.partial`
    	     * may be applied regardless of execution order. Methods like `_.ary` and
    	     * `_.rearg` modify function arguments, making the order in which they are
    	     * executed important, preventing the merging of metadata. However, we make
    	     * an exception for a safe combined case where curried functions have `_.ary`
    	     * and or `_.rearg` applied.
    	     *
    	     * @private
    	     * @param {Array} data The destination metadata.
    	     * @param {Array} source The source metadata.
    	     * @returns {Array} Returns `data`.
    	     */function mergeData(data,source){var bitmask=data[1],srcBitmask=source[1],newBitmask=bitmask|srcBitmask,isCommon=newBitmask<(WRAP_BIND_FLAG|WRAP_BIND_KEY_FLAG|WRAP_ARY_FLAG);var isCombo=srcBitmask==WRAP_ARY_FLAG&&bitmask==WRAP_CURRY_FLAG||srcBitmask==WRAP_ARY_FLAG&&bitmask==WRAP_REARG_FLAG&&data[7].length<=source[8]||srcBitmask==(WRAP_ARY_FLAG|WRAP_REARG_FLAG)&&source[7].length<=source[8]&&bitmask==WRAP_CURRY_FLAG;// Exit early if metadata can't be merged.
    	if(!(isCommon||isCombo)){return data;}// Use source `thisArg` if available.
    	if(srcBitmask&WRAP_BIND_FLAG){data[2]=source[2];// Set when currying a bound function.
    	newBitmask|=bitmask&WRAP_BIND_FLAG?0:WRAP_CURRY_BOUND_FLAG;}// Compose partial arguments.
    	var value=source[3];if(value){var partials=data[3];data[3]=partials?composeArgs(partials,value,source[4]):value;data[4]=partials?replaceHolders(data[3],PLACEHOLDER):source[4];}// Compose partial right arguments.
    	value=source[5];if(value){partials=data[5];data[5]=partials?composeArgsRight(partials,value,source[6]):value;data[6]=partials?replaceHolders(data[5],PLACEHOLDER):source[6];}// Use source `argPos` if available.
    	value=source[7];if(value){data[7]=value;}// Use source `ary` if it's smaller.
    	if(srcBitmask&WRAP_ARY_FLAG){data[8]=data[8]==null?source[8]:nativeMin(data[8],source[8]);}// Use source `arity` if one is not provided.
    	if(data[9]==null){data[9]=source[9];}// Use source `func` and merge bitmasks.
    	data[0]=source[0];data[1]=newBitmask;return data;}/**
    	     * This function is like
    	     * [`Object.keys`](http://ecma-international.org/ecma-262/7.0/#sec-object.keys)
    	     * except that it includes inherited enumerable properties.
    	     *
    	     * @private
    	     * @param {Object} object The object to query.
    	     * @returns {Array} Returns the array of property names.
    	     */function nativeKeysIn(object){var result=[];if(object!=null){for(var key in Object(object)){result.push(key);}}return result;}/**
    	     * Converts `value` to a string using `Object.prototype.toString`.
    	     *
    	     * @private
    	     * @param {*} value The value to convert.
    	     * @returns {string} Returns the converted string.
    	     */function objectToString(value){return nativeObjectToString.call(value);}/**
    	     * A specialized version of `baseRest` which transforms the rest array.
    	     *
    	     * @private
    	     * @param {Function} func The function to apply a rest parameter to.
    	     * @param {number} [start=func.length-1] The start position of the rest parameter.
    	     * @param {Function} transform The rest array transform.
    	     * @returns {Function} Returns the new function.
    	     */function overRest(func,start,transform){start=nativeMax(start===undefined$1?func.length-1:start,0);return function(){var args=arguments,index=-1,length=nativeMax(args.length-start,0),array=Array(length);while(++index<length){array[index]=args[start+index];}index=-1;var otherArgs=Array(start+1);while(++index<start){otherArgs[index]=args[index];}otherArgs[start]=transform(array);return apply(func,this,otherArgs);};}/**
    	     * Gets the parent value at `path` of `object`.
    	     *
    	     * @private
    	     * @param {Object} object The object to query.
    	     * @param {Array} path The path to get the parent value of.
    	     * @returns {*} Returns the parent value.
    	     */function parent(object,path){return path.length<2?object:baseGet(object,baseSlice(path,0,-1));}/**
    	     * Reorder `array` according to the specified indexes where the element at
    	     * the first index is assigned as the first element, the element at
    	     * the second index is assigned as the second element, and so on.
    	     *
    	     * @private
    	     * @param {Array} array The array to reorder.
    	     * @param {Array} indexes The arranged array indexes.
    	     * @returns {Array} Returns `array`.
    	     */function reorder(array,indexes){var arrLength=array.length,length=nativeMin(indexes.length,arrLength),oldArray=copyArray(array);while(length--){var index=indexes[length];array[length]=isIndex(index,arrLength)?oldArray[index]:undefined$1;}return array;}/**
    	     * Gets the value at `key`, unless `key` is "__proto__" or "constructor".
    	     *
    	     * @private
    	     * @param {Object} object The object to query.
    	     * @param {string} key The key of the property to get.
    	     * @returns {*} Returns the property value.
    	     */function safeGet(object,key){if(key==='constructor'&&typeof object[key]==='function'){return;}if(key=='__proto__'){return;}return object[key];}/**
    	     * Sets metadata for `func`.
    	     *
    	     * **Note:** If this function becomes hot, i.e. is invoked a lot in a short
    	     * period of time, it will trip its breaker and transition to an identity
    	     * function to avoid garbage collection pauses in V8. See
    	     * [V8 issue 2070](https://bugs.chromium.org/p/v8/issues/detail?id=2070)
    	     * for more details.
    	     *
    	     * @private
    	     * @param {Function} func The function to associate metadata with.
    	     * @param {*} data The metadata.
    	     * @returns {Function} Returns `func`.
    	     */var setData=shortOut(baseSetData);/**
    	     * A simple wrapper around the global [`setTimeout`](https://mdn.io/setTimeout).
    	     *
    	     * @private
    	     * @param {Function} func The function to delay.
    	     * @param {number} wait The number of milliseconds to delay invocation.
    	     * @returns {number|Object} Returns the timer id or timeout object.
    	     */var setTimeout=ctxSetTimeout||function(func,wait){return root.setTimeout(func,wait);};/**
    	     * Sets the `toString` method of `func` to return `string`.
    	     *
    	     * @private
    	     * @param {Function} func The function to modify.
    	     * @param {Function} string The `toString` result.
    	     * @returns {Function} Returns `func`.
    	     */var setToString=shortOut(baseSetToString);/**
    	     * Sets the `toString` method of `wrapper` to mimic the source of `reference`
    	     * with wrapper details in a comment at the top of the source body.
    	     *
    	     * @private
    	     * @param {Function} wrapper The function to modify.
    	     * @param {Function} reference The reference function.
    	     * @param {number} bitmask The bitmask flags. See `createWrap` for more details.
    	     * @returns {Function} Returns `wrapper`.
    	     */function setWrapToString(wrapper,reference,bitmask){var source=reference+'';return setToString(wrapper,insertWrapDetails(source,updateWrapDetails(getWrapDetails(source),bitmask)));}/**
    	     * Creates a function that'll short out and invoke `identity` instead
    	     * of `func` when it's called `HOT_COUNT` or more times in `HOT_SPAN`
    	     * milliseconds.
    	     *
    	     * @private
    	     * @param {Function} func The function to restrict.
    	     * @returns {Function} Returns the new shortable function.
    	     */function shortOut(func){var count=0,lastCalled=0;return function(){var stamp=nativeNow(),remaining=HOT_SPAN-(stamp-lastCalled);lastCalled=stamp;if(remaining>0){if(++count>=HOT_COUNT){return arguments[0];}}else {count=0;}return func.apply(undefined$1,arguments);};}/**
    	     * A specialized version of `_.shuffle` which mutates and sets the size of `array`.
    	     *
    	     * @private
    	     * @param {Array} array The array to shuffle.
    	     * @param {number} [size=array.length] The size of `array`.
    	     * @returns {Array} Returns `array`.
    	     */function shuffleSelf(array,size){var index=-1,length=array.length,lastIndex=length-1;size=size===undefined$1?length:size;while(++index<size){var rand=baseRandom(index,lastIndex),value=array[rand];array[rand]=array[index];array[index]=value;}array.length=size;return array;}/**
    	     * Converts `string` to a property path array.
    	     *
    	     * @private
    	     * @param {string} string The string to convert.
    	     * @returns {Array} Returns the property path array.
    	     */var stringToPath=memoizeCapped(function(string){var result=[];if(string.charCodeAt(0)===46/* . */){result.push('');}string.replace(rePropName,function(match,number,quote,subString){result.push(quote?subString.replace(reEscapeChar,'$1'):number||match);});return result;});/**
    	     * Converts `value` to a string key if it's not a string or symbol.
    	     *
    	     * @private
    	     * @param {*} value The value to inspect.
    	     * @returns {string|symbol} Returns the key.
    	     */function toKey(value){if(typeof value=='string'||isSymbol(value)){return value;}var result=value+'';return result=='0'&&1/value==-INFINITY?'-0':result;}/**
    	     * Converts `func` to its source code.
    	     *
    	     * @private
    	     * @param {Function} func The function to convert.
    	     * @returns {string} Returns the source code.
    	     */function toSource(func){if(func!=null){try{return funcToString.call(func);}catch(e){}try{return func+'';}catch(e){}}return '';}/**
    	     * Updates wrapper `details` based on `bitmask` flags.
    	     *
    	     * @private
    	     * @returns {Array} details The details to modify.
    	     * @param {number} bitmask The bitmask flags. See `createWrap` for more details.
    	     * @returns {Array} Returns `details`.
    	     */function updateWrapDetails(details,bitmask){arrayEach(wrapFlags,function(pair){var value='_.'+pair[0];if(bitmask&pair[1]&&!arrayIncludes(details,value)){details.push(value);}});return details.sort();}/**
    	     * Creates a clone of `wrapper`.
    	     *
    	     * @private
    	     * @param {Object} wrapper The wrapper to clone.
    	     * @returns {Object} Returns the cloned wrapper.
    	     */function wrapperClone(wrapper){if(wrapper instanceof LazyWrapper){return wrapper.clone();}var result=new LodashWrapper(wrapper.__wrapped__,wrapper.__chain__);result.__actions__=copyArray(wrapper.__actions__);result.__index__=wrapper.__index__;result.__values__=wrapper.__values__;return result;}/*------------------------------------------------------------------------*//**
    	     * Creates an array of elements split into groups the length of `size`.
    	     * If `array` can't be split evenly, the final chunk will be the remaining
    	     * elements.
    	     *
    	     * @static
    	     * @memberOf _
    	     * @since 3.0.0
    	     * @category Array
    	     * @param {Array} array The array to process.
    	     * @param {number} [size=1] The length of each chunk
    	     * @param- {Object} [guard] Enables use as an iteratee for methods like `_.map`.
    	     * @returns {Array} Returns the new array of chunks.
    	     * @example
    	     *
    	     * _.chunk(['a', 'b', 'c', 'd'], 2);
    	     * // => [['a', 'b'], ['c', 'd']]
    	     *
    	     * _.chunk(['a', 'b', 'c', 'd'], 3);
    	     * // => [['a', 'b', 'c'], ['d']]
    	     */function chunk(array,size,guard){if(guard?isIterateeCall(array,size,guard):size===undefined$1){size=1;}else {size=nativeMax(toInteger(size),0);}var length=array==null?0:array.length;if(!length||size<1){return [];}var index=0,resIndex=0,result=Array(nativeCeil(length/size));while(index<length){result[resIndex++]=baseSlice(array,index,index+=size);}return result;}/**
    	     * Creates an array with all falsey values removed. The values `false`, `null`,
    	     * `0`, `""`, `undefined`, and `NaN` are falsey.
    	     *
    	     * @static
    	     * @memberOf _
    	     * @since 0.1.0
    	     * @category Array
    	     * @param {Array} array The array to compact.
    	     * @returns {Array} Returns the new array of filtered values.
    	     * @example
    	     *
    	     * _.compact([0, 1, false, 2, '', 3]);
    	     * // => [1, 2, 3]
    	     */function compact(array){var index=-1,length=array==null?0:array.length,resIndex=0,result=[];while(++index<length){var value=array[index];if(value){result[resIndex++]=value;}}return result;}/**
    	     * Creates a new array concatenating `array` with any additional arrays
    	     * and/or values.
    	     *
    	     * @static
    	     * @memberOf _
    	     * @since 4.0.0
    	     * @category Array
    	     * @param {Array} array The array to concatenate.
    	     * @param {...*} [values] The values to concatenate.
    	     * @returns {Array} Returns the new concatenated array.
    	     * @example
    	     *
    	     * var array = [1];
    	     * var other = _.concat(array, 2, [3], [[4]]);
    	     *
    	     * console.log(other);
    	     * // => [1, 2, 3, [4]]
    	     *
    	     * console.log(array);
    	     * // => [1]
    	     */function concat(){var length=arguments.length;if(!length){return [];}var args=Array(length-1),array=arguments[0],index=length;while(index--){args[index-1]=arguments[index];}return arrayPush(isArray(array)?copyArray(array):[array],baseFlatten(args,1));}/**
    	     * Creates an array of `array` values not included in the other given arrays
    	     * using [`SameValueZero`](http://ecma-international.org/ecma-262/7.0/#sec-samevaluezero)
    	     * for equality comparisons. The order and references of result values are
    	     * determined by the first array.
    	     *
    	     * **Note:** Unlike `_.pullAll`, this method returns a new array.
    	     *
    	     * @static
    	     * @memberOf _
    	     * @since 0.1.0
    	     * @category Array
    	     * @param {Array} array The array to inspect.
    	     * @param {...Array} [values] The values to exclude.
    	     * @returns {Array} Returns the new array of filtered values.
    	     * @see _.without, _.xor
    	     * @example
    	     *
    	     * _.difference([2, 1], [2, 3]);
    	     * // => [1]
    	     */var difference=baseRest(function(array,values){return isArrayLikeObject(array)?baseDifference(array,baseFlatten(values,1,isArrayLikeObject,true)):[];});/**
    	     * This method is like `_.difference` except that it accepts `iteratee` which
    	     * is invoked for each element of `array` and `values` to generate the criterion
    	     * by which they're compared. The order and references of result values are
    	     * determined by the first array. The iteratee is invoked with one argument:
    	     * (value).
    	     *
    	     * **Note:** Unlike `_.pullAllBy`, this method returns a new array.
    	     *
    	     * @static
    	     * @memberOf _
    	     * @since 4.0.0
    	     * @category Array
    	     * @param {Array} array The array to inspect.
    	     * @param {...Array} [values] The values to exclude.
    	     * @param {Function} [iteratee=_.identity] The iteratee invoked per element.
    	     * @returns {Array} Returns the new array of filtered values.
    	     * @example
    	     *
    	     * _.differenceBy([2.1, 1.2], [2.3, 3.4], Math.floor);
    	     * // => [1.2]
    	     *
    	     * // The `_.property` iteratee shorthand.
    	     * _.differenceBy([{ 'x': 2 }, { 'x': 1 }], [{ 'x': 1 }], 'x');
    	     * // => [{ 'x': 2 }]
    	     */var differenceBy=baseRest(function(array,values){var iteratee=last(values);if(isArrayLikeObject(iteratee)){iteratee=undefined$1;}return isArrayLikeObject(array)?baseDifference(array,baseFlatten(values,1,isArrayLikeObject,true),getIteratee(iteratee,2)):[];});/**
    	     * This method is like `_.difference` except that it accepts `comparator`
    	     * which is invoked to compare elements of `array` to `values`. The order and
    	     * references of result values are determined by the first array. The comparator
    	     * is invoked with two arguments: (arrVal, othVal).
    	     *
    	     * **Note:** Unlike `_.pullAllWith`, this method returns a new array.
    	     *
    	     * @static
    	     * @memberOf _
    	     * @since 4.0.0
    	     * @category Array
    	     * @param {Array} array The array to inspect.
    	     * @param {...Array} [values] The values to exclude.
    	     * @param {Function} [comparator] The comparator invoked per element.
    	     * @returns {Array} Returns the new array of filtered values.
    	     * @example
    	     *
    	     * var objects = [{ 'x': 1, 'y': 2 }, { 'x': 2, 'y': 1 }];
    	     *
    	     * _.differenceWith(objects, [{ 'x': 1, 'y': 2 }], _.isEqual);
    	     * // => [{ 'x': 2, 'y': 1 }]
    	     */var differenceWith=baseRest(function(array,values){var comparator=last(values);if(isArrayLikeObject(comparator)){comparator=undefined$1;}return isArrayLikeObject(array)?baseDifference(array,baseFlatten(values,1,isArrayLikeObject,true),undefined$1,comparator):[];});/**
    	     * Creates a slice of `array` with `n` elements dropped from the beginning.
    	     *
    	     * @static
    	     * @memberOf _
    	     * @since 0.5.0
    	     * @category Array
    	     * @param {Array} array The array to query.
    	     * @param {number} [n=1] The number of elements to drop.
    	     * @param- {Object} [guard] Enables use as an iteratee for methods like `_.map`.
    	     * @returns {Array} Returns the slice of `array`.
    	     * @example
    	     *
    	     * _.drop([1, 2, 3]);
    	     * // => [2, 3]
    	     *
    	     * _.drop([1, 2, 3], 2);
    	     * // => [3]
    	     *
    	     * _.drop([1, 2, 3], 5);
    	     * // => []
    	     *
    	     * _.drop([1, 2, 3], 0);
    	     * // => [1, 2, 3]
    	     */function drop(array,n,guard){var length=array==null?0:array.length;if(!length){return [];}n=guard||n===undefined$1?1:toInteger(n);return baseSlice(array,n<0?0:n,length);}/**
    	     * Creates a slice of `array` with `n` elements dropped from the end.
    	     *
    	     * @static
    	     * @memberOf _
    	     * @since 3.0.0
    	     * @category Array
    	     * @param {Array} array The array to query.
    	     * @param {number} [n=1] The number of elements to drop.
    	     * @param- {Object} [guard] Enables use as an iteratee for methods like `_.map`.
    	     * @returns {Array} Returns the slice of `array`.
    	     * @example
    	     *
    	     * _.dropRight([1, 2, 3]);
    	     * // => [1, 2]
    	     *
    	     * _.dropRight([1, 2, 3], 2);
    	     * // => [1]
    	     *
    	     * _.dropRight([1, 2, 3], 5);
    	     * // => []
    	     *
    	     * _.dropRight([1, 2, 3], 0);
    	     * // => [1, 2, 3]
    	     */function dropRight(array,n,guard){var length=array==null?0:array.length;if(!length){return [];}n=guard||n===undefined$1?1:toInteger(n);n=length-n;return baseSlice(array,0,n<0?0:n);}/**
    	     * Creates a slice of `array` excluding elements dropped from the end.
    	     * Elements are dropped until `predicate` returns falsey. The predicate is
    	     * invoked with three arguments: (value, index, array).
    	     *
    	     * @static
    	     * @memberOf _
    	     * @since 3.0.0
    	     * @category Array
    	     * @param {Array} array The array to query.
    	     * @param {Function} [predicate=_.identity] The function invoked per iteration.
    	     * @returns {Array} Returns the slice of `array`.
    	     * @example
    	     *
    	     * var users = [
    	     *   { 'user': 'barney',  'active': true },
    	     *   { 'user': 'fred',    'active': false },
    	     *   { 'user': 'pebbles', 'active': false }
    	     * ];
    	     *
    	     * _.dropRightWhile(users, function(o) { return !o.active; });
    	     * // => objects for ['barney']
    	     *
    	     * // The `_.matches` iteratee shorthand.
    	     * _.dropRightWhile(users, { 'user': 'pebbles', 'active': false });
    	     * // => objects for ['barney', 'fred']
    	     *
    	     * // The `_.matchesProperty` iteratee shorthand.
    	     * _.dropRightWhile(users, ['active', false]);
    	     * // => objects for ['barney']
    	     *
    	     * // The `_.property` iteratee shorthand.
    	     * _.dropRightWhile(users, 'active');
    	     * // => objects for ['barney', 'fred', 'pebbles']
    	     */function dropRightWhile(array,predicate){return array&&array.length?baseWhile(array,getIteratee(predicate,3),true,true):[];}/**
    	     * Creates a slice of `array` excluding elements dropped from the beginning.
    	     * Elements are dropped until `predicate` returns falsey. The predicate is
    	     * invoked with three arguments: (value, index, array).
    	     *
    	     * @static
    	     * @memberOf _
    	     * @since 3.0.0
    	     * @category Array
    	     * @param {Array} array The array to query.
    	     * @param {Function} [predicate=_.identity] The function invoked per iteration.
    	     * @returns {Array} Returns the slice of `array`.
    	     * @example
    	     *
    	     * var users = [
    	     *   { 'user': 'barney',  'active': false },
    	     *   { 'user': 'fred',    'active': false },
    	     *   { 'user': 'pebbles', 'active': true }
    	     * ];
    	     *
    	     * _.dropWhile(users, function(o) { return !o.active; });
    	     * // => objects for ['pebbles']
    	     *
    	     * // The `_.matches` iteratee shorthand.
    	     * _.dropWhile(users, { 'user': 'barney', 'active': false });
    	     * // => objects for ['fred', 'pebbles']
    	     *
    	     * // The `_.matchesProperty` iteratee shorthand.
    	     * _.dropWhile(users, ['active', false]);
    	     * // => objects for ['pebbles']
    	     *
    	     * // The `_.property` iteratee shorthand.
    	     * _.dropWhile(users, 'active');
    	     * // => objects for ['barney', 'fred', 'pebbles']
    	     */function dropWhile(array,predicate){return array&&array.length?baseWhile(array,getIteratee(predicate,3),true):[];}/**
    	     * Fills elements of `array` with `value` from `start` up to, but not
    	     * including, `end`.
    	     *
    	     * **Note:** This method mutates `array`.
    	     *
    	     * @static
    	     * @memberOf _
    	     * @since 3.2.0
    	     * @category Array
    	     * @param {Array} array The array to fill.
    	     * @param {*} value The value to fill `array` with.
    	     * @param {number} [start=0] The start position.
    	     * @param {number} [end=array.length] The end position.
    	     * @returns {Array} Returns `array`.
    	     * @example
    	     *
    	     * var array = [1, 2, 3];
    	     *
    	     * _.fill(array, 'a');
    	     * console.log(array);
    	     * // => ['a', 'a', 'a']
    	     *
    	     * _.fill(Array(3), 2);
    	     * // => [2, 2, 2]
    	     *
    	     * _.fill([4, 6, 8, 10], '*', 1, 3);
    	     * // => [4, '*', '*', 10]
    	     */function fill(array,value,start,end){var length=array==null?0:array.length;if(!length){return [];}if(start&&typeof start!='number'&&isIterateeCall(array,value,start)){start=0;end=length;}return baseFill(array,value,start,end);}/**
    	     * This method is like `_.find` except that it returns the index of the first
    	     * element `predicate` returns truthy for instead of the element itself.
    	     *
    	     * @static
    	     * @memberOf _
    	     * @since 1.1.0
    	     * @category Array
    	     * @param {Array} array The array to inspect.
    	     * @param {Function} [predicate=_.identity] The function invoked per iteration.
    	     * @param {number} [fromIndex=0] The index to search from.
    	     * @returns {number} Returns the index of the found element, else `-1`.
    	     * @example
    	     *
    	     * var users = [
    	     *   { 'user': 'barney',  'active': false },
    	     *   { 'user': 'fred',    'active': false },
    	     *   { 'user': 'pebbles', 'active': true }
    	     * ];
    	     *
    	     * _.findIndex(users, function(o) { return o.user == 'barney'; });
    	     * // => 0
    	     *
    	     * // The `_.matches` iteratee shorthand.
    	     * _.findIndex(users, { 'user': 'fred', 'active': false });
    	     * // => 1
    	     *
    	     * // The `_.matchesProperty` iteratee shorthand.
    	     * _.findIndex(users, ['active', false]);
    	     * // => 0
    	     *
    	     * // The `_.property` iteratee shorthand.
    	     * _.findIndex(users, 'active');
    	     * // => 2
    	     */function findIndex(array,predicate,fromIndex){var length=array==null?0:array.length;if(!length){return -1;}var index=fromIndex==null?0:toInteger(fromIndex);if(index<0){index=nativeMax(length+index,0);}return baseFindIndex(array,getIteratee(predicate,3),index);}/**
    	     * This method is like `_.findIndex` except that it iterates over elements
    	     * of `collection` from right to left.
    	     *
    	     * @static
    	     * @memberOf _
    	     * @since 2.0.0
    	     * @category Array
    	     * @param {Array} array The array to inspect.
    	     * @param {Function} [predicate=_.identity] The function invoked per iteration.
    	     * @param {number} [fromIndex=array.length-1] The index to search from.
    	     * @returns {number} Returns the index of the found element, else `-1`.
    	     * @example
    	     *
    	     * var users = [
    	     *   { 'user': 'barney',  'active': true },
    	     *   { 'user': 'fred',    'active': false },
    	     *   { 'user': 'pebbles', 'active': false }
    	     * ];
    	     *
    	     * _.findLastIndex(users, function(o) { return o.user == 'pebbles'; });
    	     * // => 2
    	     *
    	     * // The `_.matches` iteratee shorthand.
    	     * _.findLastIndex(users, { 'user': 'barney', 'active': true });
    	     * // => 0
    	     *
    	     * // The `_.matchesProperty` iteratee shorthand.
    	     * _.findLastIndex(users, ['active', false]);
    	     * // => 2
    	     *
    	     * // The `_.property` iteratee shorthand.
    	     * _.findLastIndex(users, 'active');
    	     * // => 0
    	     */function findLastIndex(array,predicate,fromIndex){var length=array==null?0:array.length;if(!length){return -1;}var index=length-1;if(fromIndex!==undefined$1){index=toInteger(fromIndex);index=fromIndex<0?nativeMax(length+index,0):nativeMin(index,length-1);}return baseFindIndex(array,getIteratee(predicate,3),index,true);}/**
    	     * Flattens `array` a single level deep.
    	     *
    	     * @static
    	     * @memberOf _
    	     * @since 0.1.0
    	     * @category Array
    	     * @param {Array} array The array to flatten.
    	     * @returns {Array} Returns the new flattened array.
    	     * @example
    	     *
    	     * _.flatten([1, [2, [3, [4]], 5]]);
    	     * // => [1, 2, [3, [4]], 5]
    	     */function flatten(array){var length=array==null?0:array.length;return length?baseFlatten(array,1):[];}/**
    	     * Recursively flattens `array`.
    	     *
    	     * @static
    	     * @memberOf _
    	     * @since 3.0.0
    	     * @category Array
    	     * @param {Array} array The array to flatten.
    	     * @returns {Array} Returns the new flattened array.
    	     * @example
    	     *
    	     * _.flattenDeep([1, [2, [3, [4]], 5]]);
    	     * // => [1, 2, 3, 4, 5]
    	     */function flattenDeep(array){var length=array==null?0:array.length;return length?baseFlatten(array,INFINITY):[];}/**
    	     * Recursively flatten `array` up to `depth` times.
    	     *
    	     * @static
    	     * @memberOf _
    	     * @since 4.4.0
    	     * @category Array
    	     * @param {Array} array The array to flatten.
    	     * @param {number} [depth=1] The maximum recursion depth.
    	     * @returns {Array} Returns the new flattened array.
    	     * @example
    	     *
    	     * var array = [1, [2, [3, [4]], 5]];
    	     *
    	     * _.flattenDepth(array, 1);
    	     * // => [1, 2, [3, [4]], 5]
    	     *
    	     * _.flattenDepth(array, 2);
    	     * // => [1, 2, 3, [4], 5]
    	     */function flattenDepth(array,depth){var length=array==null?0:array.length;if(!length){return [];}depth=depth===undefined$1?1:toInteger(depth);return baseFlatten(array,depth);}/**
    	     * The inverse of `_.toPairs`; this method returns an object composed
    	     * from key-value `pairs`.
    	     *
    	     * @static
    	     * @memberOf _
    	     * @since 4.0.0
    	     * @category Array
    	     * @param {Array} pairs The key-value pairs.
    	     * @returns {Object} Returns the new object.
    	     * @example
    	     *
    	     * _.fromPairs([['a', 1], ['b', 2]]);
    	     * // => { 'a': 1, 'b': 2 }
    	     */function fromPairs(pairs){var index=-1,length=pairs==null?0:pairs.length,result={};while(++index<length){var pair=pairs[index];result[pair[0]]=pair[1];}return result;}/**
    	     * Gets the first element of `array`.
    	     *
    	     * @static
    	     * @memberOf _
    	     * @since 0.1.0
    	     * @alias first
    	     * @category Array
    	     * @param {Array} array The array to query.
    	     * @returns {*} Returns the first element of `array`.
    	     * @example
    	     *
    	     * _.head([1, 2, 3]);
    	     * // => 1
    	     *
    	     * _.head([]);
    	     * // => undefined
    	     */function head(array){return array&&array.length?array[0]:undefined$1;}/**
    	     * Gets the index at which the first occurrence of `value` is found in `array`
    	     * using [`SameValueZero`](http://ecma-international.org/ecma-262/7.0/#sec-samevaluezero)
    	     * for equality comparisons. If `fromIndex` is negative, it's used as the
    	     * offset from the end of `array`.
    	     *
    	     * @static
    	     * @memberOf _
    	     * @since 0.1.0
    	     * @category Array
    	     * @param {Array} array The array to inspect.
    	     * @param {*} value The value to search for.
    	     * @param {number} [fromIndex=0] The index to search from.
    	     * @returns {number} Returns the index of the matched value, else `-1`.
    	     * @example
    	     *
    	     * _.indexOf([1, 2, 1, 2], 2);
    	     * // => 1
    	     *
    	     * // Search from the `fromIndex`.
    	     * _.indexOf([1, 2, 1, 2], 2, 2);
    	     * // => 3
    	     */function indexOf(array,value,fromIndex){var length=array==null?0:array.length;if(!length){return -1;}var index=fromIndex==null?0:toInteger(fromIndex);if(index<0){index=nativeMax(length+index,0);}return baseIndexOf(array,value,index);}/**
    	     * Gets all but the last element of `array`.
    	     *
    	     * @static
    	     * @memberOf _
    	     * @since 0.1.0
    	     * @category Array
    	     * @param {Array} array The array to query.
    	     * @returns {Array} Returns the slice of `array`.
    	     * @example
    	     *
    	     * _.initial([1, 2, 3]);
    	     * // => [1, 2]
    	     */function initial(array){var length=array==null?0:array.length;return length?baseSlice(array,0,-1):[];}/**
    	     * Creates an array of unique values that are included in all given arrays
    	     * using [`SameValueZero`](http://ecma-international.org/ecma-262/7.0/#sec-samevaluezero)
    	     * for equality comparisons. The order and references of result values are
    	     * determined by the first array.
    	     *
    	     * @static
    	     * @memberOf _
    	     * @since 0.1.0
    	     * @category Array
    	     * @param {...Array} [arrays] The arrays to inspect.
    	     * @returns {Array} Returns the new array of intersecting values.
    	     * @example
    	     *
    	     * _.intersection([2, 1], [2, 3]);
    	     * // => [2]
    	     */var intersection=baseRest(function(arrays){var mapped=arrayMap(arrays,castArrayLikeObject);return mapped.length&&mapped[0]===arrays[0]?baseIntersection(mapped):[];});/**
    	     * This method is like `_.intersection` except that it accepts `iteratee`
    	     * which is invoked for each element of each `arrays` to generate the criterion
    	     * by which they're compared. The order and references of result values are
    	     * determined by the first array. The iteratee is invoked with one argument:
    	     * (value).
    	     *
    	     * @static
    	     * @memberOf _
    	     * @since 4.0.0
    	     * @category Array
    	     * @param {...Array} [arrays] The arrays to inspect.
    	     * @param {Function} [iteratee=_.identity] The iteratee invoked per element.
    	     * @returns {Array} Returns the new array of intersecting values.
    	     * @example
    	     *
    	     * _.intersectionBy([2.1, 1.2], [2.3, 3.4], Math.floor);
    	     * // => [2.1]
    	     *
    	     * // The `_.property` iteratee shorthand.
    	     * _.intersectionBy([{ 'x': 1 }], [{ 'x': 2 }, { 'x': 1 }], 'x');
    	     * // => [{ 'x': 1 }]
    	     */var intersectionBy=baseRest(function(arrays){var iteratee=last(arrays),mapped=arrayMap(arrays,castArrayLikeObject);if(iteratee===last(mapped)){iteratee=undefined$1;}else {mapped.pop();}return mapped.length&&mapped[0]===arrays[0]?baseIntersection(mapped,getIteratee(iteratee,2)):[];});/**
    	     * This method is like `_.intersection` except that it accepts `comparator`
    	     * which is invoked to compare elements of `arrays`. The order and references
    	     * of result values are determined by the first array. The comparator is
    	     * invoked with two arguments: (arrVal, othVal).
    	     *
    	     * @static
    	     * @memberOf _
    	     * @since 4.0.0
    	     * @category Array
    	     * @param {...Array} [arrays] The arrays to inspect.
    	     * @param {Function} [comparator] The comparator invoked per element.
    	     * @returns {Array} Returns the new array of intersecting values.
    	     * @example
    	     *
    	     * var objects = [{ 'x': 1, 'y': 2 }, { 'x': 2, 'y': 1 }];
    	     * var others = [{ 'x': 1, 'y': 1 }, { 'x': 1, 'y': 2 }];
    	     *
    	     * _.intersectionWith(objects, others, _.isEqual);
    	     * // => [{ 'x': 1, 'y': 2 }]
    	     */var intersectionWith=baseRest(function(arrays){var comparator=last(arrays),mapped=arrayMap(arrays,castArrayLikeObject);comparator=typeof comparator=='function'?comparator:undefined$1;if(comparator){mapped.pop();}return mapped.length&&mapped[0]===arrays[0]?baseIntersection(mapped,undefined$1,comparator):[];});/**
    	     * Converts all elements in `array` into a string separated by `separator`.
    	     *
    	     * @static
    	     * @memberOf _
    	     * @since 4.0.0
    	     * @category Array
    	     * @param {Array} array The array to convert.
    	     * @param {string} [separator=','] The element separator.
    	     * @returns {string} Returns the joined string.
    	     * @example
    	     *
    	     * _.join(['a', 'b', 'c'], '~');
    	     * // => 'a~b~c'
    	     */function join(array,separator){return array==null?'':nativeJoin.call(array,separator);}/**
    	     * Gets the last element of `array`.
    	     *
    	     * @static
    	     * @memberOf _
    	     * @since 0.1.0
    	     * @category Array
    	     * @param {Array} array The array to query.
    	     * @returns {*} Returns the last element of `array`.
    	     * @example
    	     *
    	     * _.last([1, 2, 3]);
    	     * // => 3
    	     */function last(array){var length=array==null?0:array.length;return length?array[length-1]:undefined$1;}/**
    	     * This method is like `_.indexOf` except that it iterates over elements of
    	     * `array` from right to left.
    	     *
    	     * @static
    	     * @memberOf _
    	     * @since 0.1.0
    	     * @category Array
    	     * @param {Array} array The array to inspect.
    	     * @param {*} value The value to search for.
    	     * @param {number} [fromIndex=array.length-1] The index to search from.
    	     * @returns {number} Returns the index of the matched value, else `-1`.
    	     * @example
    	     *
    	     * _.lastIndexOf([1, 2, 1, 2], 2);
    	     * // => 3
    	     *
    	     * // Search from the `fromIndex`.
    	     * _.lastIndexOf([1, 2, 1, 2], 2, 2);
    	     * // => 1
    	     */function lastIndexOf(array,value,fromIndex){var length=array==null?0:array.length;if(!length){return -1;}var index=length;if(fromIndex!==undefined$1){index=toInteger(fromIndex);index=index<0?nativeMax(length+index,0):nativeMin(index,length-1);}return value===value?strictLastIndexOf(array,value,index):baseFindIndex(array,baseIsNaN,index,true);}/**
    	     * Gets the element at index `n` of `array`. If `n` is negative, the nth
    	     * element from the end is returned.
    	     *
    	     * @static
    	     * @memberOf _
    	     * @since 4.11.0
    	     * @category Array
    	     * @param {Array} array The array to query.
    	     * @param {number} [n=0] The index of the element to return.
    	     * @returns {*} Returns the nth element of `array`.
    	     * @example
    	     *
    	     * var array = ['a', 'b', 'c', 'd'];
    	     *
    	     * _.nth(array, 1);
    	     * // => 'b'
    	     *
    	     * _.nth(array, -2);
    	     * // => 'c';
    	     */function nth(array,n){return array&&array.length?baseNth(array,toInteger(n)):undefined$1;}/**
    	     * Removes all given values from `array` using
    	     * [`SameValueZero`](http://ecma-international.org/ecma-262/7.0/#sec-samevaluezero)
    	     * for equality comparisons.
    	     *
    	     * **Note:** Unlike `_.without`, this method mutates `array`. Use `_.remove`
    	     * to remove elements from an array by predicate.
    	     *
    	     * @static
    	     * @memberOf _
    	     * @since 2.0.0
    	     * @category Array
    	     * @param {Array} array The array to modify.
    	     * @param {...*} [values] The values to remove.
    	     * @returns {Array} Returns `array`.
    	     * @example
    	     *
    	     * var array = ['a', 'b', 'c', 'a', 'b', 'c'];
    	     *
    	     * _.pull(array, 'a', 'c');
    	     * console.log(array);
    	     * // => ['b', 'b']
    	     */var pull=baseRest(pullAll);/**
    	     * This method is like `_.pull` except that it accepts an array of values to remove.
    	     *
    	     * **Note:** Unlike `_.difference`, this method mutates `array`.
    	     *
    	     * @static
    	     * @memberOf _
    	     * @since 4.0.0
    	     * @category Array
    	     * @param {Array} array The array to modify.
    	     * @param {Array} values The values to remove.
    	     * @returns {Array} Returns `array`.
    	     * @example
    	     *
    	     * var array = ['a', 'b', 'c', 'a', 'b', 'c'];
    	     *
    	     * _.pullAll(array, ['a', 'c']);
    	     * console.log(array);
    	     * // => ['b', 'b']
    	     */function pullAll(array,values){return array&&array.length&&values&&values.length?basePullAll(array,values):array;}/**
    	     * This method is like `_.pullAll` except that it accepts `iteratee` which is
    	     * invoked for each element of `array` and `values` to generate the criterion
    	     * by which they're compared. The iteratee is invoked with one argument: (value).
    	     *
    	     * **Note:** Unlike `_.differenceBy`, this method mutates `array`.
    	     *
    	     * @static
    	     * @memberOf _
    	     * @since 4.0.0
    	     * @category Array
    	     * @param {Array} array The array to modify.
    	     * @param {Array} values The values to remove.
    	     * @param {Function} [iteratee=_.identity] The iteratee invoked per element.
    	     * @returns {Array} Returns `array`.
    	     * @example
    	     *
    	     * var array = [{ 'x': 1 }, { 'x': 2 }, { 'x': 3 }, { 'x': 1 }];
    	     *
    	     * _.pullAllBy(array, [{ 'x': 1 }, { 'x': 3 }], 'x');
    	     * console.log(array);
    	     * // => [{ 'x': 2 }]
    	     */function pullAllBy(array,values,iteratee){return array&&array.length&&values&&values.length?basePullAll(array,values,getIteratee(iteratee,2)):array;}/**
    	     * This method is like `_.pullAll` except that it accepts `comparator` which
    	     * is invoked to compare elements of `array` to `values`. The comparator is
    	     * invoked with two arguments: (arrVal, othVal).
    	     *
    	     * **Note:** Unlike `_.differenceWith`, this method mutates `array`.
    	     *
    	     * @static
    	     * @memberOf _
    	     * @since 4.6.0
    	     * @category Array
    	     * @param {Array} array The array to modify.
    	     * @param {Array} values The values to remove.
    	     * @param {Function} [comparator] The comparator invoked per element.
    	     * @returns {Array} Returns `array`.
    	     * @example
    	     *
    	     * var array = [{ 'x': 1, 'y': 2 }, { 'x': 3, 'y': 4 }, { 'x': 5, 'y': 6 }];
    	     *
    	     * _.pullAllWith(array, [{ 'x': 3, 'y': 4 }], _.isEqual);
    	     * console.log(array);
    	     * // => [{ 'x': 1, 'y': 2 }, { 'x': 5, 'y': 6 }]
    	     */function pullAllWith(array,values,comparator){return array&&array.length&&values&&values.length?basePullAll(array,values,undefined$1,comparator):array;}/**
    	     * Removes elements from `array` corresponding to `indexes` and returns an
    	     * array of removed elements.
    	     *
    	     * **Note:** Unlike `_.at`, this method mutates `array`.
    	     *
    	     * @static
    	     * @memberOf _
    	     * @since 3.0.0
    	     * @category Array
    	     * @param {Array} array The array to modify.
    	     * @param {...(number|number[])} [indexes] The indexes of elements to remove.
    	     * @returns {Array} Returns the new array of removed elements.
    	     * @example
    	     *
    	     * var array = ['a', 'b', 'c', 'd'];
    	     * var pulled = _.pullAt(array, [1, 3]);
    	     *
    	     * console.log(array);
    	     * // => ['a', 'c']
    	     *
    	     * console.log(pulled);
    	     * // => ['b', 'd']
    	     */var pullAt=flatRest(function(array,indexes){var length=array==null?0:array.length,result=baseAt(array,indexes);basePullAt(array,arrayMap(indexes,function(index){return isIndex(index,length)?+index:index;}).sort(compareAscending));return result;});/**
    	     * Removes all elements from `array` that `predicate` returns truthy for
    	     * and returns an array of the removed elements. The predicate is invoked
    	     * with three arguments: (value, index, array).
    	     *
    	     * **Note:** Unlike `_.filter`, this method mutates `array`. Use `_.pull`
    	     * to pull elements from an array by value.
    	     *
    	     * @static
    	     * @memberOf _
    	     * @since 2.0.0
    	     * @category Array
    	     * @param {Array} array The array to modify.
    	     * @param {Function} [predicate=_.identity] The function invoked per iteration.
    	     * @returns {Array} Returns the new array of removed elements.
    	     * @example
    	     *
    	     * var array = [1, 2, 3, 4];
    	     * var evens = _.remove(array, function(n) {
    	     *   return n % 2 == 0;
    	     * });
    	     *
    	     * console.log(array);
    	     * // => [1, 3]
    	     *
    	     * console.log(evens);
    	     * // => [2, 4]
    	     */function remove(array,predicate){var result=[];if(!(array&&array.length)){return result;}var index=-1,indexes=[],length=array.length;predicate=getIteratee(predicate,3);while(++index<length){var value=array[index];if(predicate(value,index,array)){result.push(value);indexes.push(index);}}basePullAt(array,indexes);return result;}/**
    	     * Reverses `array` so that the first element becomes the last, the second
    	     * element becomes the second to last, and so on.
    	     *
    	     * **Note:** This method mutates `array` and is based on
    	     * [`Array#reverse`](https://mdn.io/Array/reverse).
    	     *
    	     * @static
    	     * @memberOf _
    	     * @since 4.0.0
    	     * @category Array
    	     * @param {Array} array The array to modify.
    	     * @returns {Array} Returns `array`.
    	     * @example
    	     *
    	     * var array = [1, 2, 3];
    	     *
    	     * _.reverse(array);
    	     * // => [3, 2, 1]
    	     *
    	     * console.log(array);
    	     * // => [3, 2, 1]
    	     */function reverse(array){return array==null?array:nativeReverse.call(array);}/**
    	     * Creates a slice of `array` from `start` up to, but not including, `end`.
    	     *
    	     * **Note:** This method is used instead of
    	     * [`Array#slice`](https://mdn.io/Array/slice) to ensure dense arrays are
    	     * returned.
    	     *
    	     * @static
    	     * @memberOf _
    	     * @since 3.0.0
    	     * @category Array
    	     * @param {Array} array The array to slice.
    	     * @param {number} [start=0] The start position.
    	     * @param {number} [end=array.length] The end position.
    	     * @returns {Array} Returns the slice of `array`.
    	     */function slice(array,start,end){var length=array==null?0:array.length;if(!length){return [];}if(end&&typeof end!='number'&&isIterateeCall(array,start,end)){start=0;end=length;}else {start=start==null?0:toInteger(start);end=end===undefined$1?length:toInteger(end);}return baseSlice(array,start,end);}/**
    	     * Uses a binary search to determine the lowest index at which `value`
    	     * should be inserted into `array` in order to maintain its sort order.
    	     *
    	     * @static
    	     * @memberOf _
    	     * @since 0.1.0
    	     * @category Array
    	     * @param {Array} array The sorted array to inspect.
    	     * @param {*} value The value to evaluate.
    	     * @returns {number} Returns the index at which `value` should be inserted
    	     *  into `array`.
    	     * @example
    	     *
    	     * _.sortedIndex([30, 50], 40);
    	     * // => 1
    	     */function sortedIndex(array,value){return baseSortedIndex(array,value);}/**
    	     * This method is like `_.sortedIndex` except that it accepts `iteratee`
    	     * which is invoked for `value` and each element of `array` to compute their
    	     * sort ranking. The iteratee is invoked with one argument: (value).
    	     *
    	     * @static
    	     * @memberOf _
    	     * @since 4.0.0
    	     * @category Array
    	     * @param {Array} array The sorted array to inspect.
    	     * @param {*} value The value to evaluate.
    	     * @param {Function} [iteratee=_.identity] The iteratee invoked per element.
    	     * @returns {number} Returns the index at which `value` should be inserted
    	     *  into `array`.
    	     * @example
    	     *
    	     * var objects = [{ 'x': 4 }, { 'x': 5 }];
    	     *
    	     * _.sortedIndexBy(objects, { 'x': 4 }, function(o) { return o.x; });
    	     * // => 0
    	     *
    	     * // The `_.property` iteratee shorthand.
    	     * _.sortedIndexBy(objects, { 'x': 4 }, 'x');
    	     * // => 0
    	     */function sortedIndexBy(array,value,iteratee){return baseSortedIndexBy(array,value,getIteratee(iteratee,2));}/**
    	     * This method is like `_.indexOf` except that it performs a binary
    	     * search on a sorted `array`.
    	     *
    	     * @static
    	     * @memberOf _
    	     * @since 4.0.0
    	     * @category Array
    	     * @param {Array} array The array to inspect.
    	     * @param {*} value The value to search for.
    	     * @returns {number} Returns the index of the matched value, else `-1`.
    	     * @example
    	     *
    	     * _.sortedIndexOf([4, 5, 5, 5, 6], 5);
    	     * // => 1
    	     */function sortedIndexOf(array,value){var length=array==null?0:array.length;if(length){var index=baseSortedIndex(array,value);if(index<length&&eq(array[index],value)){return index;}}return -1;}/**
    	     * This method is like `_.sortedIndex` except that it returns the highest
    	     * index at which `value` should be inserted into `array` in order to
    	     * maintain its sort order.
    	     *
    	     * @static
    	     * @memberOf _
    	     * @since 3.0.0
    	     * @category Array
    	     * @param {Array} array The sorted array to inspect.
    	     * @param {*} value The value to evaluate.
    	     * @returns {number} Returns the index at which `value` should be inserted
    	     *  into `array`.
    	     * @example
    	     *
    	     * _.sortedLastIndex([4, 5, 5, 5, 6], 5);
    	     * // => 4
    	     */function sortedLastIndex(array,value){return baseSortedIndex(array,value,true);}/**
    	     * This method is like `_.sortedLastIndex` except that it accepts `iteratee`
    	     * which is invoked for `value` and each element of `array` to compute their
    	     * sort ranking. The iteratee is invoked with one argument: (value).
    	     *
    	     * @static
    	     * @memberOf _
    	     * @since 4.0.0
    	     * @category Array
    	     * @param {Array} array The sorted array to inspect.
    	     * @param {*} value The value to evaluate.
    	     * @param {Function} [iteratee=_.identity] The iteratee invoked per element.
    	     * @returns {number} Returns the index at which `value` should be inserted
    	     *  into `array`.
    	     * @example
    	     *
    	     * var objects = [{ 'x': 4 }, { 'x': 5 }];
    	     *
    	     * _.sortedLastIndexBy(objects, { 'x': 4 }, function(o) { return o.x; });
    	     * // => 1
    	     *
    	     * // The `_.property` iteratee shorthand.
    	     * _.sortedLastIndexBy(objects, { 'x': 4 }, 'x');
    	     * // => 1
    	     */function sortedLastIndexBy(array,value,iteratee){return baseSortedIndexBy(array,value,getIteratee(iteratee,2),true);}/**
    	     * This method is like `_.lastIndexOf` except that it performs a binary
    	     * search on a sorted `array`.
    	     *
    	     * @static
    	     * @memberOf _
    	     * @since 4.0.0
    	     * @category Array
    	     * @param {Array} array The array to inspect.
    	     * @param {*} value The value to search for.
    	     * @returns {number} Returns the index of the matched value, else `-1`.
    	     * @example
    	     *
    	     * _.sortedLastIndexOf([4, 5, 5, 5, 6], 5);
    	     * // => 3
    	     */function sortedLastIndexOf(array,value){var length=array==null?0:array.length;if(length){var index=baseSortedIndex(array,value,true)-1;if(eq(array[index],value)){return index;}}return -1;}/**
    	     * This method is like `_.uniq` except that it's designed and optimized
    	     * for sorted arrays.
    	     *
    	     * @static
    	     * @memberOf _
    	     * @since 4.0.0
    	     * @category Array
    	     * @param {Array} array The array to inspect.
    	     * @returns {Array} Returns the new duplicate free array.
    	     * @example
    	     *
    	     * _.sortedUniq([1, 1, 2]);
    	     * // => [1, 2]
    	     */function sortedUniq(array){return array&&array.length?baseSortedUniq(array):[];}/**
    	     * This method is like `_.uniqBy` except that it's designed and optimized
    	     * for sorted arrays.
    	     *
    	     * @static
    	     * @memberOf _
    	     * @since 4.0.0
    	     * @category Array
    	     * @param {Array} array The array to inspect.
    	     * @param {Function} [iteratee] The iteratee invoked per element.
    	     * @returns {Array} Returns the new duplicate free array.
    	     * @example
    	     *
    	     * _.sortedUniqBy([1.1, 1.2, 2.3, 2.4], Math.floor);
    	     * // => [1.1, 2.3]
    	     */function sortedUniqBy(array,iteratee){return array&&array.length?baseSortedUniq(array,getIteratee(iteratee,2)):[];}/**
    	     * Gets all but the first element of `array`.
    	     *
    	     * @static
    	     * @memberOf _
    	     * @since 4.0.0
    	     * @category Array
    	     * @param {Array} array The array to query.
    	     * @returns {Array} Returns the slice of `array`.
    	     * @example
    	     *
    	     * _.tail([1, 2, 3]);
    	     * // => [2, 3]
    	     */function tail(array){var length=array==null?0:array.length;return length?baseSlice(array,1,length):[];}/**
    	     * Creates a slice of `array` with `n` elements taken from the beginning.
    	     *
    	     * @static
    	     * @memberOf _
    	     * @since 0.1.0
    	     * @category Array
    	     * @param {Array} array The array to query.
    	     * @param {number} [n=1] The number of elements to take.
    	     * @param- {Object} [guard] Enables use as an iteratee for methods like `_.map`.
    	     * @returns {Array} Returns the slice of `array`.
    	     * @example
    	     *
    	     * _.take([1, 2, 3]);
    	     * // => [1]
    	     *
    	     * _.take([1, 2, 3], 2);
    	     * // => [1, 2]
    	     *
    	     * _.take([1, 2, 3], 5);
    	     * // => [1, 2, 3]
    	     *
    	     * _.take([1, 2, 3], 0);
    	     * // => []
    	     */function take(array,n,guard){if(!(array&&array.length)){return [];}n=guard||n===undefined$1?1:toInteger(n);return baseSlice(array,0,n<0?0:n);}/**
    	     * Creates a slice of `array` with `n` elements taken from the end.
    	     *
    	     * @static
    	     * @memberOf _
    	     * @since 3.0.0
    	     * @category Array
    	     * @param {Array} array The array to query.
    	     * @param {number} [n=1] The number of elements to take.
    	     * @param- {Object} [guard] Enables use as an iteratee for methods like `_.map`.
    	     * @returns {Array} Returns the slice of `array`.
    	     * @example
    	     *
    	     * _.takeRight([1, 2, 3]);
    	     * // => [3]
    	     *
    	     * _.takeRight([1, 2, 3], 2);
    	     * // => [2, 3]
    	     *
    	     * _.takeRight([1, 2, 3], 5);
    	     * // => [1, 2, 3]
    	     *
    	     * _.takeRight([1, 2, 3], 0);
    	     * // => []
    	     */function takeRight(array,n,guard){var length=array==null?0:array.length;if(!length){return [];}n=guard||n===undefined$1?1:toInteger(n);n=length-n;return baseSlice(array,n<0?0:n,length);}/**
    	     * Creates a slice of `array` with elements taken from the end. Elements are
    	     * taken until `predicate` returns falsey. The predicate is invoked with
    	     * three arguments: (value, index, array).
    	     *
    	     * @static
    	     * @memberOf _
    	     * @since 3.0.0
    	     * @category Array
    	     * @param {Array} array The array to query.
    	     * @param {Function} [predicate=_.identity] The function invoked per iteration.
    	     * @returns {Array} Returns the slice of `array`.
    	     * @example
    	     *
    	     * var users = [
    	     *   { 'user': 'barney',  'active': true },
    	     *   { 'user': 'fred',    'active': false },
    	     *   { 'user': 'pebbles', 'active': false }
    	     * ];
    	     *
    	     * _.takeRightWhile(users, function(o) { return !o.active; });
    	     * // => objects for ['fred', 'pebbles']
    	     *
    	     * // The `_.matches` iteratee shorthand.
    	     * _.takeRightWhile(users, { 'user': 'pebbles', 'active': false });
    	     * // => objects for ['pebbles']
    	     *
    	     * // The `_.matchesProperty` iteratee shorthand.
    	     * _.takeRightWhile(users, ['active', false]);
    	     * // => objects for ['fred', 'pebbles']
    	     *
    	     * // The `_.property` iteratee shorthand.
    	     * _.takeRightWhile(users, 'active');
    	     * // => []
    	     */function takeRightWhile(array,predicate){return array&&array.length?baseWhile(array,getIteratee(predicate,3),false,true):[];}/**
    	     * Creates a slice of `array` with elements taken from the beginning. Elements
    	     * are taken until `predicate` returns falsey. The predicate is invoked with
    	     * three arguments: (value, index, array).
    	     *
    	     * @static
    	     * @memberOf _
    	     * @since 3.0.0
    	     * @category Array
    	     * @param {Array} array The array to query.
    	     * @param {Function} [predicate=_.identity] The function invoked per iteration.
    	     * @returns {Array} Returns the slice of `array`.
    	     * @example
    	     *
    	     * var users = [
    	     *   { 'user': 'barney',  'active': false },
    	     *   { 'user': 'fred',    'active': false },
    	     *   { 'user': 'pebbles', 'active': true }
    	     * ];
    	     *
    	     * _.takeWhile(users, function(o) { return !o.active; });
    	     * // => objects for ['barney', 'fred']
    	     *
    	     * // The `_.matches` iteratee shorthand.
    	     * _.takeWhile(users, { 'user': 'barney', 'active': false });
    	     * // => objects for ['barney']
    	     *
    	     * // The `_.matchesProperty` iteratee shorthand.
    	     * _.takeWhile(users, ['active', false]);
    	     * // => objects for ['barney', 'fred']
    	     *
    	     * // The `_.property` iteratee shorthand.
    	     * _.takeWhile(users, 'active');
    	     * // => []
    	     */function takeWhile(array,predicate){return array&&array.length?baseWhile(array,getIteratee(predicate,3)):[];}/**
    	     * Creates an array of unique values, in order, from all given arrays using
    	     * [`SameValueZero`](http://ecma-international.org/ecma-262/7.0/#sec-samevaluezero)
    	     * for equality comparisons.
    	     *
    	     * @static
    	     * @memberOf _
    	     * @since 0.1.0
    	     * @category Array
    	     * @param {...Array} [arrays] The arrays to inspect.
    	     * @returns {Array} Returns the new array of combined values.
    	     * @example
    	     *
    	     * _.union([2], [1, 2]);
    	     * // => [2, 1]
    	     */var union=baseRest(function(arrays){return baseUniq(baseFlatten(arrays,1,isArrayLikeObject,true));});/**
    	     * This method is like `_.union` except that it accepts `iteratee` which is
    	     * invoked for each element of each `arrays` to generate the criterion by
    	     * which uniqueness is computed. Result values are chosen from the first
    	     * array in which the value occurs. The iteratee is invoked with one argument:
    	     * (value).
    	     *
    	     * @static
    	     * @memberOf _
    	     * @since 4.0.0
    	     * @category Array
    	     * @param {...Array} [arrays] The arrays to inspect.
    	     * @param {Function} [iteratee=_.identity] The iteratee invoked per element.
    	     * @returns {Array} Returns the new array of combined values.
    	     * @example
    	     *
    	     * _.unionBy([2.1], [1.2, 2.3], Math.floor);
    	     * // => [2.1, 1.2]
    	     *
    	     * // The `_.property` iteratee shorthand.
    	     * _.unionBy([{ 'x': 1 }], [{ 'x': 2 }, { 'x': 1 }], 'x');
    	     * // => [{ 'x': 1 }, { 'x': 2 }]
    	     */var unionBy=baseRest(function(arrays){var iteratee=last(arrays);if(isArrayLikeObject(iteratee)){iteratee=undefined$1;}return baseUniq(baseFlatten(arrays,1,isArrayLikeObject,true),getIteratee(iteratee,2));});/**
    	     * This method is like `_.union` except that it accepts `comparator` which
    	     * is invoked to compare elements of `arrays`. Result values are chosen from
    	     * the first array in which the value occurs. The comparator is invoked
    	     * with two arguments: (arrVal, othVal).
    	     *
    	     * @static
    	     * @memberOf _
    	     * @since 4.0.0
    	     * @category Array
    	     * @param {...Array} [arrays] The arrays to inspect.
    	     * @param {Function} [comparator] The comparator invoked per element.
    	     * @returns {Array} Returns the new array of combined values.
    	     * @example
    	     *
    	     * var objects = [{ 'x': 1, 'y': 2 }, { 'x': 2, 'y': 1 }];
    	     * var others = [{ 'x': 1, 'y': 1 }, { 'x': 1, 'y': 2 }];
    	     *
    	     * _.unionWith(objects, others, _.isEqual);
    	     * // => [{ 'x': 1, 'y': 2 }, { 'x': 2, 'y': 1 }, { 'x': 1, 'y': 1 }]
    	     */var unionWith=baseRest(function(arrays){var comparator=last(arrays);comparator=typeof comparator=='function'?comparator:undefined$1;return baseUniq(baseFlatten(arrays,1,isArrayLikeObject,true),undefined$1,comparator);});/**
    	     * Creates a duplicate-free version of an array, using
    	     * [`SameValueZero`](http://ecma-international.org/ecma-262/7.0/#sec-samevaluezero)
    	     * for equality comparisons, in which only the first occurrence of each element
    	     * is kept. The order of result values is determined by the order they occur
    	     * in the array.
    	     *
    	     * @static
    	     * @memberOf _
    	     * @since 0.1.0
    	     * @category Array
    	     * @param {Array} array The array to inspect.
    	     * @returns {Array} Returns the new duplicate free array.
    	     * @example
    	     *
    	     * _.uniq([2, 1, 2]);
    	     * // => [2, 1]
    	     */function uniq(array){return array&&array.length?baseUniq(array):[];}/**
    	     * This method is like `_.uniq` except that it accepts `iteratee` which is
    	     * invoked for each element in `array` to generate the criterion by which
    	     * uniqueness is computed. The order of result values is determined by the
    	     * order they occur in the array. The iteratee is invoked with one argument:
    	     * (value).
    	     *
    	     * @static
    	     * @memberOf _
    	     * @since 4.0.0
    	     * @category Array
    	     * @param {Array} array The array to inspect.
    	     * @param {Function} [iteratee=_.identity] The iteratee invoked per element.
    	     * @returns {Array} Returns the new duplicate free array.
    	     * @example
    	     *
    	     * _.uniqBy([2.1, 1.2, 2.3], Math.floor);
    	     * // => [2.1, 1.2]
    	     *
    	     * // The `_.property` iteratee shorthand.
    	     * _.uniqBy([{ 'x': 1 }, { 'x': 2 }, { 'x': 1 }], 'x');
    	     * // => [{ 'x': 1 }, { 'x': 2 }]
    	     */function uniqBy(array,iteratee){return array&&array.length?baseUniq(array,getIteratee(iteratee,2)):[];}/**
    	     * This method is like `_.uniq` except that it accepts `comparator` which
    	     * is invoked to compare elements of `array`. The order of result values is
    	     * determined by the order they occur in the array.The comparator is invoked
    	     * with two arguments: (arrVal, othVal).
    	     *
    	     * @static
    	     * @memberOf _
    	     * @since 4.0.0
    	     * @category Array
    	     * @param {Array} array The array to inspect.
    	     * @param {Function} [comparator] The comparator invoked per element.
    	     * @returns {Array} Returns the new duplicate free array.
    	     * @example
    	     *
    	     * var objects = [{ 'x': 1, 'y': 2 }, { 'x': 2, 'y': 1 }, { 'x': 1, 'y': 2 }];
    	     *
    	     * _.uniqWith(objects, _.isEqual);
    	     * // => [{ 'x': 1, 'y': 2 }, { 'x': 2, 'y': 1 }]
    	     */function uniqWith(array,comparator){comparator=typeof comparator=='function'?comparator:undefined$1;return array&&array.length?baseUniq(array,undefined$1,comparator):[];}/**
    	     * This method is like `_.zip` except that it accepts an array of grouped
    	     * elements and creates an array regrouping the elements to their pre-zip
    	     * configuration.
    	     *
    	     * @static
    	     * @memberOf _
    	     * @since 1.2.0
    	     * @category Array
    	     * @param {Array} array The array of grouped elements to process.
    	     * @returns {Array} Returns the new array of regrouped elements.
    	     * @example
    	     *
    	     * var zipped = _.zip(['a', 'b'], [1, 2], [true, false]);
    	     * // => [['a', 1, true], ['b', 2, false]]
    	     *
    	     * _.unzip(zipped);
    	     * // => [['a', 'b'], [1, 2], [true, false]]
    	     */function unzip(array){if(!(array&&array.length)){return [];}var length=0;array=arrayFilter(array,function(group){if(isArrayLikeObject(group)){length=nativeMax(group.length,length);return true;}});return baseTimes(length,function(index){return arrayMap(array,baseProperty(index));});}/**
    	     * This method is like `_.unzip` except that it accepts `iteratee` to specify
    	     * how regrouped values should be combined. The iteratee is invoked with the
    	     * elements of each group: (...group).
    	     *
    	     * @static
    	     * @memberOf _
    	     * @since 3.8.0
    	     * @category Array
    	     * @param {Array} array The array of grouped elements to process.
    	     * @param {Function} [iteratee=_.identity] The function to combine
    	     *  regrouped values.
    	     * @returns {Array} Returns the new array of regrouped elements.
    	     * @example
    	     *
    	     * var zipped = _.zip([1, 2], [10, 20], [100, 200]);
    	     * // => [[1, 10, 100], [2, 20, 200]]
    	     *
    	     * _.unzipWith(zipped, _.add);
    	     * // => [3, 30, 300]
    	     */function unzipWith(array,iteratee){if(!(array&&array.length)){return [];}var result=unzip(array);if(iteratee==null){return result;}return arrayMap(result,function(group){return apply(iteratee,undefined$1,group);});}/**
    	     * Creates an array excluding all given values using
    	     * [`SameValueZero`](http://ecma-international.org/ecma-262/7.0/#sec-samevaluezero)
    	     * for equality comparisons.
    	     *
    	     * **Note:** Unlike `_.pull`, this method returns a new array.
    	     *
    	     * @static
    	     * @memberOf _
    	     * @since 0.1.0
    	     * @category Array
    	     * @param {Array} array The array to inspect.
    	     * @param {...*} [values] The values to exclude.
    	     * @returns {Array} Returns the new array of filtered values.
    	     * @see _.difference, _.xor
    	     * @example
    	     *
    	     * _.without([2, 1, 2, 3], 1, 2);
    	     * // => [3]
    	     */var without=baseRest(function(array,values){return isArrayLikeObject(array)?baseDifference(array,values):[];});/**
    	     * Creates an array of unique values that is the
    	     * [symmetric difference](https://en.wikipedia.org/wiki/Symmetric_difference)
    	     * of the given arrays. The order of result values is determined by the order
    	     * they occur in the arrays.
    	     *
    	     * @static
    	     * @memberOf _
    	     * @since 2.4.0
    	     * @category Array
    	     * @param {...Array} [arrays] The arrays to inspect.
    	     * @returns {Array} Returns the new array of filtered values.
    	     * @see _.difference, _.without
    	     * @example
    	     *
    	     * _.xor([2, 1], [2, 3]);
    	     * // => [1, 3]
    	     */var xor=baseRest(function(arrays){return baseXor(arrayFilter(arrays,isArrayLikeObject));});/**
    	     * This method is like `_.xor` except that it accepts `iteratee` which is
    	     * invoked for each element of each `arrays` to generate the criterion by
    	     * which by which they're compared. The order of result values is determined
    	     * by the order they occur in the arrays. The iteratee is invoked with one
    	     * argument: (value).
    	     *
    	     * @static
    	     * @memberOf _
    	     * @since 4.0.0
    	     * @category Array
    	     * @param {...Array} [arrays] The arrays to inspect.
    	     * @param {Function} [iteratee=_.identity] The iteratee invoked per element.
    	     * @returns {Array} Returns the new array of filtered values.
    	     * @example
    	     *
    	     * _.xorBy([2.1, 1.2], [2.3, 3.4], Math.floor);
    	     * // => [1.2, 3.4]
    	     *
    	     * // The `_.property` iteratee shorthand.
    	     * _.xorBy([{ 'x': 1 }], [{ 'x': 2 }, { 'x': 1 }], 'x');
    	     * // => [{ 'x': 2 }]
    	     */var xorBy=baseRest(function(arrays){var iteratee=last(arrays);if(isArrayLikeObject(iteratee)){iteratee=undefined$1;}return baseXor(arrayFilter(arrays,isArrayLikeObject),getIteratee(iteratee,2));});/**
    	     * This method is like `_.xor` except that it accepts `comparator` which is
    	     * invoked to compare elements of `arrays`. The order of result values is
    	     * determined by the order they occur in the arrays. The comparator is invoked
    	     * with two arguments: (arrVal, othVal).
    	     *
    	     * @static
    	     * @memberOf _
    	     * @since 4.0.0
    	     * @category Array
    	     * @param {...Array} [arrays] The arrays to inspect.
    	     * @param {Function} [comparator] The comparator invoked per element.
    	     * @returns {Array} Returns the new array of filtered values.
    	     * @example
    	     *
    	     * var objects = [{ 'x': 1, 'y': 2 }, { 'x': 2, 'y': 1 }];
    	     * var others = [{ 'x': 1, 'y': 1 }, { 'x': 1, 'y': 2 }];
    	     *
    	     * _.xorWith(objects, others, _.isEqual);
    	     * // => [{ 'x': 2, 'y': 1 }, { 'x': 1, 'y': 1 }]
    	     */var xorWith=baseRest(function(arrays){var comparator=last(arrays);comparator=typeof comparator=='function'?comparator:undefined$1;return baseXor(arrayFilter(arrays,isArrayLikeObject),undefined$1,comparator);});/**
    	     * Creates an array of grouped elements, the first of which contains the
    	     * first elements of the given arrays, the second of which contains the
    	     * second elements of the given arrays, and so on.
    	     *
    	     * @static
    	     * @memberOf _
    	     * @since 0.1.0
    	     * @category Array
    	     * @param {...Array} [arrays] The arrays to process.
    	     * @returns {Array} Returns the new array of grouped elements.
    	     * @example
    	     *
    	     * _.zip(['a', 'b'], [1, 2], [true, false]);
    	     * // => [['a', 1, true], ['b', 2, false]]
    	     */var zip=baseRest(unzip);/**
    	     * This method is like `_.fromPairs` except that it accepts two arrays,
    	     * one of property identifiers and one of corresponding values.
    	     *
    	     * @static
    	     * @memberOf _
    	     * @since 0.4.0
    	     * @category Array
    	     * @param {Array} [props=[]] The property identifiers.
    	     * @param {Array} [values=[]] The property values.
    	     * @returns {Object} Returns the new object.
    	     * @example
    	     *
    	     * _.zipObject(['a', 'b'], [1, 2]);
    	     * // => { 'a': 1, 'b': 2 }
    	     */function zipObject(props,values){return baseZipObject(props||[],values||[],assignValue);}/**
    	     * This method is like `_.zipObject` except that it supports property paths.
    	     *
    	     * @static
    	     * @memberOf _
    	     * @since 4.1.0
    	     * @category Array
    	     * @param {Array} [props=[]] The property identifiers.
    	     * @param {Array} [values=[]] The property values.
    	     * @returns {Object} Returns the new object.
    	     * @example
    	     *
    	     * _.zipObjectDeep(['a.b[0].c', 'a.b[1].d'], [1, 2]);
    	     * // => { 'a': { 'b': [{ 'c': 1 }, { 'd': 2 }] } }
    	     */function zipObjectDeep(props,values){return baseZipObject(props||[],values||[],baseSet);}/**
    	     * This method is like `_.zip` except that it accepts `iteratee` to specify
    	     * how grouped values should be combined. The iteratee is invoked with the
    	     * elements of each group: (...group).
    	     *
    	     * @static
    	     * @memberOf _
    	     * @since 3.8.0
    	     * @category Array
    	     * @param {...Array} [arrays] The arrays to process.
    	     * @param {Function} [iteratee=_.identity] The function to combine
    	     *  grouped values.
    	     * @returns {Array} Returns the new array of grouped elements.
    	     * @example
    	     *
    	     * _.zipWith([1, 2], [10, 20], [100, 200], function(a, b, c) {
    	     *   return a + b + c;
    	     * });
    	     * // => [111, 222]
    	     */var zipWith=baseRest(function(arrays){var length=arrays.length,iteratee=length>1?arrays[length-1]:undefined$1;iteratee=typeof iteratee=='function'?(arrays.pop(),iteratee):undefined$1;return unzipWith(arrays,iteratee);});/*------------------------------------------------------------------------*//**
    	     * Creates a `lodash` wrapper instance that wraps `value` with explicit method
    	     * chain sequences enabled. The result of such sequences must be unwrapped
    	     * with `_#value`.
    	     *
    	     * @static
    	     * @memberOf _
    	     * @since 1.3.0
    	     * @category Seq
    	     * @param {*} value The value to wrap.
    	     * @returns {Object} Returns the new `lodash` wrapper instance.
    	     * @example
    	     *
    	     * var users = [
    	     *   { 'user': 'barney',  'age': 36 },
    	     *   { 'user': 'fred',    'age': 40 },
    	     *   { 'user': 'pebbles', 'age': 1 }
    	     * ];
    	     *
    	     * var youngest = _
    	     *   .chain(users)
    	     *   .sortBy('age')
    	     *   .map(function(o) {
    	     *     return o.user + ' is ' + o.age;
    	     *   })
    	     *   .head()
    	     *   .value();
    	     * // => 'pebbles is 1'
    	     */function chain(value){var result=lodash(value);result.__chain__=true;return result;}/**
    	     * This method invokes `interceptor` and returns `value`. The interceptor
    	     * is invoked with one argument; (value). The purpose of this method is to
    	     * "tap into" a method chain sequence in order to modify intermediate results.
    	     *
    	     * @static
    	     * @memberOf _
    	     * @since 0.1.0
    	     * @category Seq
    	     * @param {*} value The value to provide to `interceptor`.
    	     * @param {Function} interceptor The function to invoke.
    	     * @returns {*} Returns `value`.
    	     * @example
    	     *
    	     * _([1, 2, 3])
    	     *  .tap(function(array) {
    	     *    // Mutate input array.
    	     *    array.pop();
    	     *  })
    	     *  .reverse()
    	     *  .value();
    	     * // => [2, 1]
    	     */function tap(value,interceptor){interceptor(value);return value;}/**
    	     * This method is like `_.tap` except that it returns the result of `interceptor`.
    	     * The purpose of this method is to "pass thru" values replacing intermediate
    	     * results in a method chain sequence.
    	     *
    	     * @static
    	     * @memberOf _
    	     * @since 3.0.0
    	     * @category Seq
    	     * @param {*} value The value to provide to `interceptor`.
    	     * @param {Function} interceptor The function to invoke.
    	     * @returns {*} Returns the result of `interceptor`.
    	     * @example
    	     *
    	     * _('  abc  ')
    	     *  .chain()
    	     *  .trim()
    	     *  .thru(function(value) {
    	     *    return [value];
    	     *  })
    	     *  .value();
    	     * // => ['abc']
    	     */function thru(value,interceptor){return interceptor(value);}/**
    	     * This method is the wrapper version of `_.at`.
    	     *
    	     * @name at
    	     * @memberOf _
    	     * @since 1.0.0
    	     * @category Seq
    	     * @param {...(string|string[])} [paths] The property paths to pick.
    	     * @returns {Object} Returns the new `lodash` wrapper instance.
    	     * @example
    	     *
    	     * var object = { 'a': [{ 'b': { 'c': 3 } }, 4] };
    	     *
    	     * _(object).at(['a[0].b.c', 'a[1]']).value();
    	     * // => [3, 4]
    	     */var wrapperAt=flatRest(function(paths){var length=paths.length,start=length?paths[0]:0,value=this.__wrapped__,interceptor=function(object){return baseAt(object,paths);};if(length>1||this.__actions__.length||!(value instanceof LazyWrapper)||!isIndex(start)){return this.thru(interceptor);}value=value.slice(start,+start+(length?1:0));value.__actions__.push({'func':thru,'args':[interceptor],'thisArg':undefined$1});return new LodashWrapper(value,this.__chain__).thru(function(array){if(length&&!array.length){array.push(undefined$1);}return array;});});/**
    	     * Creates a `lodash` wrapper instance with explicit method chain sequences enabled.
    	     *
    	     * @name chain
    	     * @memberOf _
    	     * @since 0.1.0
    	     * @category Seq
    	     * @returns {Object} Returns the new `lodash` wrapper instance.
    	     * @example
    	     *
    	     * var users = [
    	     *   { 'user': 'barney', 'age': 36 },
    	     *   { 'user': 'fred',   'age': 40 }
    	     * ];
    	     *
    	     * // A sequence without explicit chaining.
    	     * _(users).head();
    	     * // => { 'user': 'barney', 'age': 36 }
    	     *
    	     * // A sequence with explicit chaining.
    	     * _(users)
    	     *   .chain()
    	     *   .head()
    	     *   .pick('user')
    	     *   .value();
    	     * // => { 'user': 'barney' }
    	     */function wrapperChain(){return chain(this);}/**
    	     * Executes the chain sequence and returns the wrapped result.
    	     *
    	     * @name commit
    	     * @memberOf _
    	     * @since 3.2.0
    	     * @category Seq
    	     * @returns {Object} Returns the new `lodash` wrapper instance.
    	     * @example
    	     *
    	     * var array = [1, 2];
    	     * var wrapped = _(array).push(3);
    	     *
    	     * console.log(array);
    	     * // => [1, 2]
    	     *
    	     * wrapped = wrapped.commit();
    	     * console.log(array);
    	     * // => [1, 2, 3]
    	     *
    	     * wrapped.last();
    	     * // => 3
    	     *
    	     * console.log(array);
    	     * // => [1, 2, 3]
    	     */function wrapperCommit(){return new LodashWrapper(this.value(),this.__chain__);}/**
    	     * Gets the next value on a wrapped object following the
    	     * [iterator protocol](https://mdn.io/iteration_protocols#iterator).
    	     *
    	     * @name next
    	     * @memberOf _
    	     * @since 4.0.0
    	     * @category Seq
    	     * @returns {Object} Returns the next iterator value.
    	     * @example
    	     *
    	     * var wrapped = _([1, 2]);
    	     *
    	     * wrapped.next();
    	     * // => { 'done': false, 'value': 1 }
    	     *
    	     * wrapped.next();
    	     * // => { 'done': false, 'value': 2 }
    	     *
    	     * wrapped.next();
    	     * // => { 'done': true, 'value': undefined }
    	     */function wrapperNext(){if(this.__values__===undefined$1){this.__values__=toArray(this.value());}var done=this.__index__>=this.__values__.length,value=done?undefined$1:this.__values__[this.__index__++];return {'done':done,'value':value};}/**
    	     * Enables the wrapper to be iterable.
    	     *
    	     * @name Symbol.iterator
    	     * @memberOf _
    	     * @since 4.0.0
    	     * @category Seq
    	     * @returns {Object} Returns the wrapper object.
    	     * @example
    	     *
    	     * var wrapped = _([1, 2]);
    	     *
    	     * wrapped[Symbol.iterator]() === wrapped;
    	     * // => true
    	     *
    	     * Array.from(wrapped);
    	     * // => [1, 2]
    	     */function wrapperToIterator(){return this;}/**
    	     * Creates a clone of the chain sequence planting `value` as the wrapped value.
    	     *
    	     * @name plant
    	     * @memberOf _
    	     * @since 3.2.0
    	     * @category Seq
    	     * @param {*} value The value to plant.
    	     * @returns {Object} Returns the new `lodash` wrapper instance.
    	     * @example
    	     *
    	     * function square(n) {
    	     *   return n * n;
    	     * }
    	     *
    	     * var wrapped = _([1, 2]).map(square);
    	     * var other = wrapped.plant([3, 4]);
    	     *
    	     * other.value();
    	     * // => [9, 16]
    	     *
    	     * wrapped.value();
    	     * // => [1, 4]
    	     */function wrapperPlant(value){var result,parent=this;while(parent instanceof baseLodash){var clone=wrapperClone(parent);clone.__index__=0;clone.__values__=undefined$1;if(result){previous.__wrapped__=clone;}else {result=clone;}var previous=clone;parent=parent.__wrapped__;}previous.__wrapped__=value;return result;}/**
    	     * This method is the wrapper version of `_.reverse`.
    	     *
    	     * **Note:** This method mutates the wrapped array.
    	     *
    	     * @name reverse
    	     * @memberOf _
    	     * @since 0.1.0
    	     * @category Seq
    	     * @returns {Object} Returns the new `lodash` wrapper instance.
    	     * @example
    	     *
    	     * var array = [1, 2, 3];
    	     *
    	     * _(array).reverse().value()
    	     * // => [3, 2, 1]
    	     *
    	     * console.log(array);
    	     * // => [3, 2, 1]
    	     */function wrapperReverse(){var value=this.__wrapped__;if(value instanceof LazyWrapper){var wrapped=value;if(this.__actions__.length){wrapped=new LazyWrapper(this);}wrapped=wrapped.reverse();wrapped.__actions__.push({'func':thru,'args':[reverse],'thisArg':undefined$1});return new LodashWrapper(wrapped,this.__chain__);}return this.thru(reverse);}/**
    	     * Executes the chain sequence to resolve the unwrapped value.
    	     *
    	     * @name value
    	     * @memberOf _
    	     * @since 0.1.0
    	     * @alias toJSON, valueOf
    	     * @category Seq
    	     * @returns {*} Returns the resolved unwrapped value.
    	     * @example
    	     *
    	     * _([1, 2, 3]).value();
    	     * // => [1, 2, 3]
    	     */function wrapperValue(){return baseWrapperValue(this.__wrapped__,this.__actions__);}/*------------------------------------------------------------------------*//**
    	     * Creates an object composed of keys generated from the results of running
    	     * each element of `collection` thru `iteratee`. The corresponding value of
    	     * each key is the number of times the key was returned by `iteratee`. The
    	     * iteratee is invoked with one argument: (value).
    	     *
    	     * @static
    	     * @memberOf _
    	     * @since 0.5.0
    	     * @category Collection
    	     * @param {Array|Object} collection The collection to iterate over.
    	     * @param {Function} [iteratee=_.identity] The iteratee to transform keys.
    	     * @returns {Object} Returns the composed aggregate object.
    	     * @example
    	     *
    	     * _.countBy([6.1, 4.2, 6.3], Math.floor);
    	     * // => { '4': 1, '6': 2 }
    	     *
    	     * // The `_.property` iteratee shorthand.
    	     * _.countBy(['one', 'two', 'three'], 'length');
    	     * // => { '3': 2, '5': 1 }
    	     */var countBy=createAggregator(function(result,value,key){if(hasOwnProperty.call(result,key)){++result[key];}else {baseAssignValue(result,key,1);}});/**
    	     * Checks if `predicate` returns truthy for **all** elements of `collection`.
    	     * Iteration is stopped once `predicate` returns falsey. The predicate is
    	     * invoked with three arguments: (value, index|key, collection).
    	     *
    	     * **Note:** This method returns `true` for
    	     * [empty collections](https://en.wikipedia.org/wiki/Empty_set) because
    	     * [everything is true](https://en.wikipedia.org/wiki/Vacuous_truth) of
    	     * elements of empty collections.
    	     *
    	     * @static
    	     * @memberOf _
    	     * @since 0.1.0
    	     * @category Collection
    	     * @param {Array|Object} collection The collection to iterate over.
    	     * @param {Function} [predicate=_.identity] The function invoked per iteration.
    	     * @param- {Object} [guard] Enables use as an iteratee for methods like `_.map`.
    	     * @returns {boolean} Returns `true` if all elements pass the predicate check,
    	     *  else `false`.
    	     * @example
    	     *
    	     * _.every([true, 1, null, 'yes'], Boolean);
    	     * // => false
    	     *
    	     * var users = [
    	     *   { 'user': 'barney', 'age': 36, 'active': false },
    	     *   { 'user': 'fred',   'age': 40, 'active': false }
    	     * ];
    	     *
    	     * // The `_.matches` iteratee shorthand.
    	     * _.every(users, { 'user': 'barney', 'active': false });
    	     * // => false
    	     *
    	     * // The `_.matchesProperty` iteratee shorthand.
    	     * _.every(users, ['active', false]);
    	     * // => true
    	     *
    	     * // The `_.property` iteratee shorthand.
    	     * _.every(users, 'active');
    	     * // => false
    	     */function every(collection,predicate,guard){var func=isArray(collection)?arrayEvery:baseEvery;if(guard&&isIterateeCall(collection,predicate,guard)){predicate=undefined$1;}return func(collection,getIteratee(predicate,3));}/**
    	     * Iterates over elements of `collection`, returning an array of all elements
    	     * `predicate` returns truthy for. The predicate is invoked with three
    	     * arguments: (value, index|key, collection).
    	     *
    	     * **Note:** Unlike `_.remove`, this method returns a new array.
    	     *
    	     * @static
    	     * @memberOf _
    	     * @since 0.1.0
    	     * @category Collection
    	     * @param {Array|Object} collection The collection to iterate over.
    	     * @param {Function} [predicate=_.identity] The function invoked per iteration.
    	     * @returns {Array} Returns the new filtered array.
    	     * @see _.reject
    	     * @example
    	     *
    	     * var users = [
    	     *   { 'user': 'barney', 'age': 36, 'active': true },
    	     *   { 'user': 'fred',   'age': 40, 'active': false }
    	     * ];
    	     *
    	     * _.filter(users, function(o) { return !o.active; });
    	     * // => objects for ['fred']
    	     *
    	     * // The `_.matches` iteratee shorthand.
    	     * _.filter(users, { 'age': 36, 'active': true });
    	     * // => objects for ['barney']
    	     *
    	     * // The `_.matchesProperty` iteratee shorthand.
    	     * _.filter(users, ['active', false]);
    	     * // => objects for ['fred']
    	     *
    	     * // The `_.property` iteratee shorthand.
    	     * _.filter(users, 'active');
    	     * // => objects for ['barney']
    	     *
    	     * // Combining several predicates using `_.overEvery` or `_.overSome`.
    	     * _.filter(users, _.overSome([{ 'age': 36 }, ['age', 40]]));
    	     * // => objects for ['fred', 'barney']
    	     */function filter(collection,predicate){var func=isArray(collection)?arrayFilter:baseFilter;return func(collection,getIteratee(predicate,3));}/**
    	     * Iterates over elements of `collection`, returning the first element
    	     * `predicate` returns truthy for. The predicate is invoked with three
    	     * arguments: (value, index|key, collection).
    	     *
    	     * @static
    	     * @memberOf _
    	     * @since 0.1.0
    	     * @category Collection
    	     * @param {Array|Object} collection The collection to inspect.
    	     * @param {Function} [predicate=_.identity] The function invoked per iteration.
    	     * @param {number} [fromIndex=0] The index to search from.
    	     * @returns {*} Returns the matched element, else `undefined`.
    	     * @example
    	     *
    	     * var users = [
    	     *   { 'user': 'barney',  'age': 36, 'active': true },
    	     *   { 'user': 'fred',    'age': 40, 'active': false },
    	     *   { 'user': 'pebbles', 'age': 1,  'active': true }
    	     * ];
    	     *
    	     * _.find(users, function(o) { return o.age < 40; });
    	     * // => object for 'barney'
    	     *
    	     * // The `_.matches` iteratee shorthand.
    	     * _.find(users, { 'age': 1, 'active': true });
    	     * // => object for 'pebbles'
    	     *
    	     * // The `_.matchesProperty` iteratee shorthand.
    	     * _.find(users, ['active', false]);
    	     * // => object for 'fred'
    	     *
    	     * // The `_.property` iteratee shorthand.
    	     * _.find(users, 'active');
    	     * // => object for 'barney'
    	     */var find=createFind(findIndex);/**
    	     * This method is like `_.find` except that it iterates over elements of
    	     * `collection` from right to left.
    	     *
    	     * @static
    	     * @memberOf _
    	     * @since 2.0.0
    	     * @category Collection
    	     * @param {Array|Object} collection The collection to inspect.
    	     * @param {Function} [predicate=_.identity] The function invoked per iteration.
    	     * @param {number} [fromIndex=collection.length-1] The index to search from.
    	     * @returns {*} Returns the matched element, else `undefined`.
    	     * @example
    	     *
    	     * _.findLast([1, 2, 3, 4], function(n) {
    	     *   return n % 2 == 1;
    	     * });
    	     * // => 3
    	     */var findLast=createFind(findLastIndex);/**
    	     * Creates a flattened array of values by running each element in `collection`
    	     * thru `iteratee` and flattening the mapped results. The iteratee is invoked
    	     * with three arguments: (value, index|key, collection).
    	     *
    	     * @static
    	     * @memberOf _
    	     * @since 4.0.0
    	     * @category Collection
    	     * @param {Array|Object} collection The collection to iterate over.
    	     * @param {Function} [iteratee=_.identity] The function invoked per iteration.
    	     * @returns {Array} Returns the new flattened array.
    	     * @example
    	     *
    	     * function duplicate(n) {
    	     *   return [n, n];
    	     * }
    	     *
    	     * _.flatMap([1, 2], duplicate);
    	     * // => [1, 1, 2, 2]
    	     */function flatMap(collection,iteratee){return baseFlatten(map(collection,iteratee),1);}/**
    	     * This method is like `_.flatMap` except that it recursively flattens the
    	     * mapped results.
    	     *
    	     * @static
    	     * @memberOf _
    	     * @since 4.7.0
    	     * @category Collection
    	     * @param {Array|Object} collection The collection to iterate over.
    	     * @param {Function} [iteratee=_.identity] The function invoked per iteration.
    	     * @returns {Array} Returns the new flattened array.
    	     * @example
    	     *
    	     * function duplicate(n) {
    	     *   return [[[n, n]]];
    	     * }
    	     *
    	     * _.flatMapDeep([1, 2], duplicate);
    	     * // => [1, 1, 2, 2]
    	     */function flatMapDeep(collection,iteratee){return baseFlatten(map(collection,iteratee),INFINITY);}/**
    	     * This method is like `_.flatMap` except that it recursively flattens the
    	     * mapped results up to `depth` times.
    	     *
    	     * @static
    	     * @memberOf _
    	     * @since 4.7.0
    	     * @category Collection
    	     * @param {Array|Object} collection The collection to iterate over.
    	     * @param {Function} [iteratee=_.identity] The function invoked per iteration.
    	     * @param {number} [depth=1] The maximum recursion depth.
    	     * @returns {Array} Returns the new flattened array.
    	     * @example
    	     *
    	     * function duplicate(n) {
    	     *   return [[[n, n]]];
    	     * }
    	     *
    	     * _.flatMapDepth([1, 2], duplicate, 2);
    	     * // => [[1, 1], [2, 2]]
    	     */function flatMapDepth(collection,iteratee,depth){depth=depth===undefined$1?1:toInteger(depth);return baseFlatten(map(collection,iteratee),depth);}/**
    	     * Iterates over elements of `collection` and invokes `iteratee` for each element.
    	     * The iteratee is invoked with three arguments: (value, index|key, collection).
    	     * Iteratee functions may exit iteration early by explicitly returning `false`.
    	     *
    	     * **Note:** As with other "Collections" methods, objects with a "length"
    	     * property are iterated like arrays. To avoid this behavior use `_.forIn`
    	     * or `_.forOwn` for object iteration.
    	     *
    	     * @static
    	     * @memberOf _
    	     * @since 0.1.0
    	     * @alias each
    	     * @category Collection
    	     * @param {Array|Object} collection The collection to iterate over.
    	     * @param {Function} [iteratee=_.identity] The function invoked per iteration.
    	     * @returns {Array|Object} Returns `collection`.
    	     * @see _.forEachRight
    	     * @example
    	     *
    	     * _.forEach([1, 2], function(value) {
    	     *   console.log(value);
    	     * });
    	     * // => Logs `1` then `2`.
    	     *
    	     * _.forEach({ 'a': 1, 'b': 2 }, function(value, key) {
    	     *   console.log(key);
    	     * });
    	     * // => Logs 'a' then 'b' (iteration order is not guaranteed).
    	     */function forEach(collection,iteratee){var func=isArray(collection)?arrayEach:baseEach;return func(collection,getIteratee(iteratee,3));}/**
    	     * This method is like `_.forEach` except that it iterates over elements of
    	     * `collection` from right to left.
    	     *
    	     * @static
    	     * @memberOf _
    	     * @since 2.0.0
    	     * @alias eachRight
    	     * @category Collection
    	     * @param {Array|Object} collection The collection to iterate over.
    	     * @param {Function} [iteratee=_.identity] The function invoked per iteration.
    	     * @returns {Array|Object} Returns `collection`.
    	     * @see _.forEach
    	     * @example
    	     *
    	     * _.forEachRight([1, 2], function(value) {
    	     *   console.log(value);
    	     * });
    	     * // => Logs `2` then `1`.
    	     */function forEachRight(collection,iteratee){var func=isArray(collection)?arrayEachRight:baseEachRight;return func(collection,getIteratee(iteratee,3));}/**
    	     * Creates an object composed of keys generated from the results of running
    	     * each element of `collection` thru `iteratee`. The order of grouped values
    	     * is determined by the order they occur in `collection`. The corresponding
    	     * value of each key is an array of elements responsible for generating the
    	     * key. The iteratee is invoked with one argument: (value).
    	     *
    	     * @static
    	     * @memberOf _
    	     * @since 0.1.0
    	     * @category Collection
    	     * @param {Array|Object} collection The collection to iterate over.
    	     * @param {Function} [iteratee=_.identity] The iteratee to transform keys.
    	     * @returns {Object} Returns the composed aggregate object.
    	     * @example
    	     *
    	     * _.groupBy([6.1, 4.2, 6.3], Math.floor);
    	     * // => { '4': [4.2], '6': [6.1, 6.3] }
    	     *
    	     * // The `_.property` iteratee shorthand.
    	     * _.groupBy(['one', 'two', 'three'], 'length');
    	     * // => { '3': ['one', 'two'], '5': ['three'] }
    	     */var groupBy=createAggregator(function(result,value,key){if(hasOwnProperty.call(result,key)){result[key].push(value);}else {baseAssignValue(result,key,[value]);}});/**
    	     * Checks if `value` is in `collection`. If `collection` is a string, it's
    	     * checked for a substring of `value`, otherwise
    	     * [`SameValueZero`](http://ecma-international.org/ecma-262/7.0/#sec-samevaluezero)
    	     * is used for equality comparisons. If `fromIndex` is negative, it's used as
    	     * the offset from the end of `collection`.
    	     *
    	     * @static
    	     * @memberOf _
    	     * @since 0.1.0
    	     * @category Collection
    	     * @param {Array|Object|string} collection The collection to inspect.
    	     * @param {*} value The value to search for.
    	     * @param {number} [fromIndex=0] The index to search from.
    	     * @param- {Object} [guard] Enables use as an iteratee for methods like `_.reduce`.
    	     * @returns {boolean} Returns `true` if `value` is found, else `false`.
    	     * @example
    	     *
    	     * _.includes([1, 2, 3], 1);
    	     * // => true
    	     *
    	     * _.includes([1, 2, 3], 1, 2);
    	     * // => false
    	     *
    	     * _.includes({ 'a': 1, 'b': 2 }, 1);
    	     * // => true
    	     *
    	     * _.includes('abcd', 'bc');
    	     * // => true
    	     */function includes(collection,value,fromIndex,guard){collection=isArrayLike(collection)?collection:values(collection);fromIndex=fromIndex&&!guard?toInteger(fromIndex):0;var length=collection.length;if(fromIndex<0){fromIndex=nativeMax(length+fromIndex,0);}return isString(collection)?fromIndex<=length&&collection.indexOf(value,fromIndex)>-1:!!length&&baseIndexOf(collection,value,fromIndex)>-1;}/**
    	     * Invokes the method at `path` of each element in `collection`, returning
    	     * an array of the results of each invoked method. Any additional arguments
    	     * are provided to each invoked method. If `path` is a function, it's invoked
    	     * for, and `this` bound to, each element in `collection`.
    	     *
    	     * @static
    	     * @memberOf _
    	     * @since 4.0.0
    	     * @category Collection
    	     * @param {Array|Object} collection The collection to iterate over.
    	     * @param {Array|Function|string} path The path of the method to invoke or
    	     *  the function invoked per iteration.
    	     * @param {...*} [args] The arguments to invoke each method with.
    	     * @returns {Array} Returns the array of results.
    	     * @example
    	     *
    	     * _.invokeMap([[5, 1, 7], [3, 2, 1]], 'sort');
    	     * // => [[1, 5, 7], [1, 2, 3]]
    	     *
    	     * _.invokeMap([123, 456], String.prototype.split, '');
    	     * // => [['1', '2', '3'], ['4', '5', '6']]
    	     */var invokeMap=baseRest(function(collection,path,args){var index=-1,isFunc=typeof path=='function',result=isArrayLike(collection)?Array(collection.length):[];baseEach(collection,function(value){result[++index]=isFunc?apply(path,value,args):baseInvoke(value,path,args);});return result;});/**
    	     * Creates an object composed of keys generated from the results of running
    	     * each element of `collection` thru `iteratee`. The corresponding value of
    	     * each key is the last element responsible for generating the key. The
    	     * iteratee is invoked with one argument: (value).
    	     *
    	     * @static
    	     * @memberOf _
    	     * @since 4.0.0
    	     * @category Collection
    	     * @param {Array|Object} collection The collection to iterate over.
    	     * @param {Function} [iteratee=_.identity] The iteratee to transform keys.
    	     * @returns {Object} Returns the composed aggregate object.
    	     * @example
    	     *
    	     * var array = [
    	     *   { 'dir': 'left', 'code': 97 },
    	     *   { 'dir': 'right', 'code': 100 }
    	     * ];
    	     *
    	     * _.keyBy(array, function(o) {
    	     *   return String.fromCharCode(o.code);
    	     * });
    	     * // => { 'a': { 'dir': 'left', 'code': 97 }, 'd': { 'dir': 'right', 'code': 100 } }
    	     *
    	     * _.keyBy(array, 'dir');
    	     * // => { 'left': { 'dir': 'left', 'code': 97 }, 'right': { 'dir': 'right', 'code': 100 } }
    	     */var keyBy=createAggregator(function(result,value,key){baseAssignValue(result,key,value);});/**
    	     * Creates an array of values by running each element in `collection` thru
    	     * `iteratee`. The iteratee is invoked with three arguments:
    	     * (value, index|key, collection).
    	     *
    	     * Many lodash methods are guarded to work as iteratees for methods like
    	     * `_.every`, `_.filter`, `_.map`, `_.mapValues`, `_.reject`, and `_.some`.
    	     *
    	     * The guarded methods are:
    	     * `ary`, `chunk`, `curry`, `curryRight`, `drop`, `dropRight`, `every`,
    	     * `fill`, `invert`, `parseInt`, `random`, `range`, `rangeRight`, `repeat`,
    	     * `sampleSize`, `slice`, `some`, `sortBy`, `split`, `take`, `takeRight`,
    	     * `template`, `trim`, `trimEnd`, `trimStart`, and `words`
    	     *
    	     * @static
    	     * @memberOf _
    	     * @since 0.1.0
    	     * @category Collection
    	     * @param {Array|Object} collection The collection to iterate over.
    	     * @param {Function} [iteratee=_.identity] The function invoked per iteration.
    	     * @returns {Array} Returns the new mapped array.
    	     * @example
    	     *
    	     * function square(n) {
    	     *   return n * n;
    	     * }
    	     *
    	     * _.map([4, 8], square);
    	     * // => [16, 64]
    	     *
    	     * _.map({ 'a': 4, 'b': 8 }, square);
    	     * // => [16, 64] (iteration order is not guaranteed)
    	     *
    	     * var users = [
    	     *   { 'user': 'barney' },
    	     *   { 'user': 'fred' }
    	     * ];
    	     *
    	     * // The `_.property` iteratee shorthand.
    	     * _.map(users, 'user');
    	     * // => ['barney', 'fred']
    	     */function map(collection,iteratee){var func=isArray(collection)?arrayMap:baseMap;return func(collection,getIteratee(iteratee,3));}/**
    	     * This method is like `_.sortBy` except that it allows specifying the sort
    	     * orders of the iteratees to sort by. If `orders` is unspecified, all values
    	     * are sorted in ascending order. Otherwise, specify an order of "desc" for
    	     * descending or "asc" for ascending sort order of corresponding values.
    	     *
    	     * @static
    	     * @memberOf _
    	     * @since 4.0.0
    	     * @category Collection
    	     * @param {Array|Object} collection The collection to iterate over.
    	     * @param {Array[]|Function[]|Object[]|string[]} [iteratees=[_.identity]]
    	     *  The iteratees to sort by.
    	     * @param {string[]} [orders] The sort orders of `iteratees`.
    	     * @param- {Object} [guard] Enables use as an iteratee for methods like `_.reduce`.
    	     * @returns {Array} Returns the new sorted array.
    	     * @example
    	     *
    	     * var users = [
    	     *   { 'user': 'fred',   'age': 48 },
    	     *   { 'user': 'barney', 'age': 34 },
    	     *   { 'user': 'fred',   'age': 40 },
    	     *   { 'user': 'barney', 'age': 36 }
    	     * ];
    	     *
    	     * // Sort by `user` in ascending order and by `age` in descending order.
    	     * _.orderBy(users, ['user', 'age'], ['asc', 'desc']);
    	     * // => objects for [['barney', 36], ['barney', 34], ['fred', 48], ['fred', 40]]
    	     */function orderBy(collection,iteratees,orders,guard){if(collection==null){return [];}if(!isArray(iteratees)){iteratees=iteratees==null?[]:[iteratees];}orders=guard?undefined$1:orders;if(!isArray(orders)){orders=orders==null?[]:[orders];}return baseOrderBy(collection,iteratees,orders);}/**
    	     * Creates an array of elements split into two groups, the first of which
    	     * contains elements `predicate` returns truthy for, the second of which
    	     * contains elements `predicate` returns falsey for. The predicate is
    	     * invoked with one argument: (value).
    	     *
    	     * @static
    	     * @memberOf _
    	     * @since 3.0.0
    	     * @category Collection
    	     * @param {Array|Object} collection The collection to iterate over.
    	     * @param {Function} [predicate=_.identity] The function invoked per iteration.
    	     * @returns {Array} Returns the array of grouped elements.
    	     * @example
    	     *
    	     * var users = [
    	     *   { 'user': 'barney',  'age': 36, 'active': false },
    	     *   { 'user': 'fred',    'age': 40, 'active': true },
    	     *   { 'user': 'pebbles', 'age': 1,  'active': false }
    	     * ];
    	     *
    	     * _.partition(users, function(o) { return o.active; });
    	     * // => objects for [['fred'], ['barney', 'pebbles']]
    	     *
    	     * // The `_.matches` iteratee shorthand.
    	     * _.partition(users, { 'age': 1, 'active': false });
    	     * // => objects for [['pebbles'], ['barney', 'fred']]
    	     *
    	     * // The `_.matchesProperty` iteratee shorthand.
    	     * _.partition(users, ['active', false]);
    	     * // => objects for [['barney', 'pebbles'], ['fred']]
    	     *
    	     * // The `_.property` iteratee shorthand.
    	     * _.partition(users, 'active');
    	     * // => objects for [['fred'], ['barney', 'pebbles']]
    	     */var partition=createAggregator(function(result,value,key){result[key?0:1].push(value);},function(){return [[],[]];});/**
    	     * Reduces `collection` to a value which is the accumulated result of running
    	     * each element in `collection` thru `iteratee`, where each successive
    	     * invocation is supplied the return value of the previous. If `accumulator`
    	     * is not given, the first element of `collection` is used as the initial
    	     * value. The iteratee is invoked with four arguments:
    	     * (accumulator, value, index|key, collection).
    	     *
    	     * Many lodash methods are guarded to work as iteratees for methods like
    	     * `_.reduce`, `_.reduceRight`, and `_.transform`.
    	     *
    	     * The guarded methods are:
    	     * `assign`, `defaults`, `defaultsDeep`, `includes`, `merge`, `orderBy`,
    	     * and `sortBy`
    	     *
    	     * @static
    	     * @memberOf _
    	     * @since 0.1.0
    	     * @category Collection
    	     * @param {Array|Object} collection The collection to iterate over.
    	     * @param {Function} [iteratee=_.identity] The function invoked per iteration.
    	     * @param {*} [accumulator] The initial value.
    	     * @returns {*} Returns the accumulated value.
    	     * @see _.reduceRight
    	     * @example
    	     *
    	     * _.reduce([1, 2], function(sum, n) {
    	     *   return sum + n;
    	     * }, 0);
    	     * // => 3
    	     *
    	     * _.reduce({ 'a': 1, 'b': 2, 'c': 1 }, function(result, value, key) {
    	     *   (result[value] || (result[value] = [])).push(key);
    	     *   return result;
    	     * }, {});
    	     * // => { '1': ['a', 'c'], '2': ['b'] } (iteration order is not guaranteed)
    	     */function reduce(collection,iteratee,accumulator){var func=isArray(collection)?arrayReduce:baseReduce,initAccum=arguments.length<3;return func(collection,getIteratee(iteratee,4),accumulator,initAccum,baseEach);}/**
    	     * This method is like `_.reduce` except that it iterates over elements of
    	     * `collection` from right to left.
    	     *
    	     * @static
    	     * @memberOf _
    	     * @since 0.1.0
    	     * @category Collection
    	     * @param {Array|Object} collection The collection to iterate over.
    	     * @param {Function} [iteratee=_.identity] The function invoked per iteration.
    	     * @param {*} [accumulator] The initial value.
    	     * @returns {*} Returns the accumulated value.
    	     * @see _.reduce
    	     * @example
    	     *
    	     * var array = [[0, 1], [2, 3], [4, 5]];
    	     *
    	     * _.reduceRight(array, function(flattened, other) {
    	     *   return flattened.concat(other);
    	     * }, []);
    	     * // => [4, 5, 2, 3, 0, 1]
    	     */function reduceRight(collection,iteratee,accumulator){var func=isArray(collection)?arrayReduceRight:baseReduce,initAccum=arguments.length<3;return func(collection,getIteratee(iteratee,4),accumulator,initAccum,baseEachRight);}/**
    	     * The opposite of `_.filter`; this method returns the elements of `collection`
    	     * that `predicate` does **not** return truthy for.
    	     *
    	     * @static
    	     * @memberOf _
    	     * @since 0.1.0
    	     * @category Collection
    	     * @param {Array|Object} collection The collection to iterate over.
    	     * @param {Function} [predicate=_.identity] The function invoked per iteration.
    	     * @returns {Array} Returns the new filtered array.
    	     * @see _.filter
    	     * @example
    	     *
    	     * var users = [
    	     *   { 'user': 'barney', 'age': 36, 'active': false },
    	     *   { 'user': 'fred',   'age': 40, 'active': true }
    	     * ];
    	     *
    	     * _.reject(users, function(o) { return !o.active; });
    	     * // => objects for ['fred']
    	     *
    	     * // The `_.matches` iteratee shorthand.
    	     * _.reject(users, { 'age': 40, 'active': true });
    	     * // => objects for ['barney']
    	     *
    	     * // The `_.matchesProperty` iteratee shorthand.
    	     * _.reject(users, ['active', false]);
    	     * // => objects for ['fred']
    	     *
    	     * // The `_.property` iteratee shorthand.
    	     * _.reject(users, 'active');
    	     * // => objects for ['barney']
    	     */function reject(collection,predicate){var func=isArray(collection)?arrayFilter:baseFilter;return func(collection,negate(getIteratee(predicate,3)));}/**
    	     * Gets a random element from `collection`.
    	     *
    	     * @static
    	     * @memberOf _
    	     * @since 2.0.0
    	     * @category Collection
    	     * @param {Array|Object} collection The collection to sample.
    	     * @returns {*} Returns the random element.
    	     * @example
    	     *
    	     * _.sample([1, 2, 3, 4]);
    	     * // => 2
    	     */function sample(collection){var func=isArray(collection)?arraySample:baseSample;return func(collection);}/**
    	     * Gets `n` random elements at unique keys from `collection` up to the
    	     * size of `collection`.
    	     *
    	     * @static
    	     * @memberOf _
    	     * @since 4.0.0
    	     * @category Collection
    	     * @param {Array|Object} collection The collection to sample.
    	     * @param {number} [n=1] The number of elements to sample.
    	     * @param- {Object} [guard] Enables use as an iteratee for methods like `_.map`.
    	     * @returns {Array} Returns the random elements.
    	     * @example
    	     *
    	     * _.sampleSize([1, 2, 3], 2);
    	     * // => [3, 1]
    	     *
    	     * _.sampleSize([1, 2, 3], 4);
    	     * // => [2, 3, 1]
    	     */function sampleSize(collection,n,guard){if(guard?isIterateeCall(collection,n,guard):n===undefined$1){n=1;}else {n=toInteger(n);}var func=isArray(collection)?arraySampleSize:baseSampleSize;return func(collection,n);}/**
    	     * Creates an array of shuffled values, using a version of the
    	     * [Fisher-Yates shuffle](https://en.wikipedia.org/wiki/Fisher-Yates_shuffle).
    	     *
    	     * @static
    	     * @memberOf _
    	     * @since 0.1.0
    	     * @category Collection
    	     * @param {Array|Object} collection The collection to shuffle.
    	     * @returns {Array} Returns the new shuffled array.
    	     * @example
    	     *
    	     * _.shuffle([1, 2, 3, 4]);
    	     * // => [4, 1, 3, 2]
    	     */function shuffle(collection){var func=isArray(collection)?arrayShuffle:baseShuffle;return func(collection);}/**
    	     * Gets the size of `collection` by returning its length for array-like
    	     * values or the number of own enumerable string keyed properties for objects.
    	     *
    	     * @static
    	     * @memberOf _
    	     * @since 0.1.0
    	     * @category Collection
    	     * @param {Array|Object|string} collection The collection to inspect.
    	     * @returns {number} Returns the collection size.
    	     * @example
    	     *
    	     * _.size([1, 2, 3]);
    	     * // => 3
    	     *
    	     * _.size({ 'a': 1, 'b': 2 });
    	     * // => 2
    	     *
    	     * _.size('pebbles');
    	     * // => 7
    	     */function size(collection){if(collection==null){return 0;}if(isArrayLike(collection)){return isString(collection)?stringSize(collection):collection.length;}var tag=getTag(collection);if(tag==mapTag||tag==setTag){return collection.size;}return baseKeys(collection).length;}/**
    	     * Checks if `predicate` returns truthy for **any** element of `collection`.
    	     * Iteration is stopped once `predicate` returns truthy. The predicate is
    	     * invoked with three arguments: (value, index|key, collection).
    	     *
    	     * @static
    	     * @memberOf _
    	     * @since 0.1.0
    	     * @category Collection
    	     * @param {Array|Object} collection The collection to iterate over.
    	     * @param {Function} [predicate=_.identity] The function invoked per iteration.
    	     * @param- {Object} [guard] Enables use as an iteratee for methods like `_.map`.
    	     * @returns {boolean} Returns `true` if any element passes the predicate check,
    	     *  else `false`.
    	     * @example
    	     *
    	     * _.some([null, 0, 'yes', false], Boolean);
    	     * // => true
    	     *
    	     * var users = [
    	     *   { 'user': 'barney', 'active': true },
    	     *   { 'user': 'fred',   'active': false }
    	     * ];
    	     *
    	     * // The `_.matches` iteratee shorthand.
    	     * _.some(users, { 'user': 'barney', 'active': false });
    	     * // => false
    	     *
    	     * // The `_.matchesProperty` iteratee shorthand.
    	     * _.some(users, ['active', false]);
    	     * // => true
    	     *
    	     * // The `_.property` iteratee shorthand.
    	     * _.some(users, 'active');
    	     * // => true
    	     */function some(collection,predicate,guard){var func=isArray(collection)?arraySome:baseSome;if(guard&&isIterateeCall(collection,predicate,guard)){predicate=undefined$1;}return func(collection,getIteratee(predicate,3));}/**
    	     * Creates an array of elements, sorted in ascending order by the results of
    	     * running each element in a collection thru each iteratee. This method
    	     * performs a stable sort, that is, it preserves the original sort order of
    	     * equal elements. The iteratees are invoked with one argument: (value).
    	     *
    	     * @static
    	     * @memberOf _
    	     * @since 0.1.0
    	     * @category Collection
    	     * @param {Array|Object} collection The collection to iterate over.
    	     * @param {...(Function|Function[])} [iteratees=[_.identity]]
    	     *  The iteratees to sort by.
    	     * @returns {Array} Returns the new sorted array.
    	     * @example
    	     *
    	     * var users = [
    	     *   { 'user': 'fred',   'age': 48 },
    	     *   { 'user': 'barney', 'age': 36 },
    	     *   { 'user': 'fred',   'age': 30 },
    	     *   { 'user': 'barney', 'age': 34 }
    	     * ];
    	     *
    	     * _.sortBy(users, [function(o) { return o.user; }]);
    	     * // => objects for [['barney', 36], ['barney', 34], ['fred', 48], ['fred', 30]]
    	     *
    	     * _.sortBy(users, ['user', 'age']);
    	     * // => objects for [['barney', 34], ['barney', 36], ['fred', 30], ['fred', 48]]
    	     */var sortBy=baseRest(function(collection,iteratees){if(collection==null){return [];}var length=iteratees.length;if(length>1&&isIterateeCall(collection,iteratees[0],iteratees[1])){iteratees=[];}else if(length>2&&isIterateeCall(iteratees[0],iteratees[1],iteratees[2])){iteratees=[iteratees[0]];}return baseOrderBy(collection,baseFlatten(iteratees,1),[]);});/*------------------------------------------------------------------------*//**
    	     * Gets the timestamp of the number of milliseconds that have elapsed since
    	     * the Unix epoch (1 January 1970 00:00:00 UTC).
    	     *
    	     * @static
    	     * @memberOf _
    	     * @since 2.4.0
    	     * @category Date
    	     * @returns {number} Returns the timestamp.
    	     * @example
    	     *
    	     * _.defer(function(stamp) {
    	     *   console.log(_.now() - stamp);
    	     * }, _.now());
    	     * // => Logs the number of milliseconds it took for the deferred invocation.
    	     */var now=ctxNow||function(){return root.Date.now();};/*------------------------------------------------------------------------*//**
    	     * The opposite of `_.before`; this method creates a function that invokes
    	     * `func` once it's called `n` or more times.
    	     *
    	     * @static
    	     * @memberOf _
    	     * @since 0.1.0
    	     * @category Function
    	     * @param {number} n The number of calls before `func` is invoked.
    	     * @param {Function} func The function to restrict.
    	     * @returns {Function} Returns the new restricted function.
    	     * @example
    	     *
    	     * var saves = ['profile', 'settings'];
    	     *
    	     * var done = _.after(saves.length, function() {
    	     *   console.log('done saving!');
    	     * });
    	     *
    	     * _.forEach(saves, function(type) {
    	     *   asyncSave({ 'type': type, 'complete': done });
    	     * });
    	     * // => Logs 'done saving!' after the two async saves have completed.
    	     */function after(n,func){if(typeof func!='function'){throw new TypeError(FUNC_ERROR_TEXT);}n=toInteger(n);return function(){if(--n<1){return func.apply(this,arguments);}};}/**
    	     * Creates a function that invokes `func`, with up to `n` arguments,
    	     * ignoring any additional arguments.
    	     *
    	     * @static
    	     * @memberOf _
    	     * @since 3.0.0
    	     * @category Function
    	     * @param {Function} func The function to cap arguments for.
    	     * @param {number} [n=func.length] The arity cap.
    	     * @param- {Object} [guard] Enables use as an iteratee for methods like `_.map`.
    	     * @returns {Function} Returns the new capped function.
    	     * @example
    	     *
    	     * _.map(['6', '8', '10'], _.ary(parseInt, 1));
    	     * // => [6, 8, 10]
    	     */function ary(func,n,guard){n=guard?undefined$1:n;n=func&&n==null?func.length:n;return createWrap(func,WRAP_ARY_FLAG,undefined$1,undefined$1,undefined$1,undefined$1,n);}/**
    	     * Creates a function that invokes `func`, with the `this` binding and arguments
    	     * of the created function, while it's called less than `n` times. Subsequent
    	     * calls to the created function return the result of the last `func` invocation.
    	     *
    	     * @static
    	     * @memberOf _
    	     * @since 3.0.0
    	     * @category Function
    	     * @param {number} n The number of calls at which `func` is no longer invoked.
    	     * @param {Function} func The function to restrict.
    	     * @returns {Function} Returns the new restricted function.
    	     * @example
    	     *
    	     * jQuery(element).on('click', _.before(5, addContactToList));
    	     * // => Allows adding up to 4 contacts to the list.
    	     */function before(n,func){var result;if(typeof func!='function'){throw new TypeError(FUNC_ERROR_TEXT);}n=toInteger(n);return function(){if(--n>0){result=func.apply(this,arguments);}if(n<=1){func=undefined$1;}return result;};}/**
    	     * Creates a function that invokes `func` with the `this` binding of `thisArg`
    	     * and `partials` prepended to the arguments it receives.
    	     *
    	     * The `_.bind.placeholder` value, which defaults to `_` in monolithic builds,
    	     * may be used as a placeholder for partially applied arguments.
    	     *
    	     * **Note:** Unlike native `Function#bind`, this method doesn't set the "length"
    	     * property of bound functions.
    	     *
    	     * @static
    	     * @memberOf _
    	     * @since 0.1.0
    	     * @category Function
    	     * @param {Function} func The function to bind.
    	     * @param {*} thisArg The `this` binding of `func`.
    	     * @param {...*} [partials] The arguments to be partially applied.
    	     * @returns {Function} Returns the new bound function.
    	     * @example
    	     *
    	     * function greet(greeting, punctuation) {
    	     *   return greeting + ' ' + this.user + punctuation;
    	     * }
    	     *
    	     * var object = { 'user': 'fred' };
    	     *
    	     * var bound = _.bind(greet, object, 'hi');
    	     * bound('!');
    	     * // => 'hi fred!'
    	     *
    	     * // Bound with placeholders.
    	     * var bound = _.bind(greet, object, _, '!');
    	     * bound('hi');
    	     * // => 'hi fred!'
    	     */var bind=baseRest(function(func,thisArg,partials){var bitmask=WRAP_BIND_FLAG;if(partials.length){var holders=replaceHolders(partials,getHolder(bind));bitmask|=WRAP_PARTIAL_FLAG;}return createWrap(func,bitmask,thisArg,partials,holders);});/**
    	     * Creates a function that invokes the method at `object[key]` with `partials`
    	     * prepended to the arguments it receives.
    	     *
    	     * This method differs from `_.bind` by allowing bound functions to reference
    	     * methods that may be redefined or don't yet exist. See
    	     * [Peter Michaux's article](http://peter.michaux.ca/articles/lazy-function-definition-pattern)
    	     * for more details.
    	     *
    	     * The `_.bindKey.placeholder` value, which defaults to `_` in monolithic
    	     * builds, may be used as a placeholder for partially applied arguments.
    	     *
    	     * @static
    	     * @memberOf _
    	     * @since 0.10.0
    	     * @category Function
    	     * @param {Object} object The object to invoke the method on.
    	     * @param {string} key The key of the method.
    	     * @param {...*} [partials] The arguments to be partially applied.
    	     * @returns {Function} Returns the new bound function.
    	     * @example
    	     *
    	     * var object = {
    	     *   'user': 'fred',
    	     *   'greet': function(greeting, punctuation) {
    	     *     return greeting + ' ' + this.user + punctuation;
    	     *   }
    	     * };
    	     *
    	     * var bound = _.bindKey(object, 'greet', 'hi');
    	     * bound('!');
    	     * // => 'hi fred!'
    	     *
    	     * object.greet = function(greeting, punctuation) {
    	     *   return greeting + 'ya ' + this.user + punctuation;
    	     * };
    	     *
    	     * bound('!');
    	     * // => 'hiya fred!'
    	     *
    	     * // Bound with placeholders.
    	     * var bound = _.bindKey(object, 'greet', _, '!');
    	     * bound('hi');
    	     * // => 'hiya fred!'
    	     */var bindKey=baseRest(function(object,key,partials){var bitmask=WRAP_BIND_FLAG|WRAP_BIND_KEY_FLAG;if(partials.length){var holders=replaceHolders(partials,getHolder(bindKey));bitmask|=WRAP_PARTIAL_FLAG;}return createWrap(key,bitmask,object,partials,holders);});/**
    	     * Creates a function that accepts arguments of `func` and either invokes
    	     * `func` returning its result, if at least `arity` number of arguments have
    	     * been provided, or returns a function that accepts the remaining `func`
    	     * arguments, and so on. The arity of `func` may be specified if `func.length`
    	     * is not sufficient.
    	     *
    	     * The `_.curry.placeholder` value, which defaults to `_` in monolithic builds,
    	     * may be used as a placeholder for provided arguments.
    	     *
    	     * **Note:** This method doesn't set the "length" property of curried functions.
    	     *
    	     * @static
    	     * @memberOf _
    	     * @since 2.0.0
    	     * @category Function
    	     * @param {Function} func The function to curry.
    	     * @param {number} [arity=func.length] The arity of `func`.
    	     * @param- {Object} [guard] Enables use as an iteratee for methods like `_.map`.
    	     * @returns {Function} Returns the new curried function.
    	     * @example
    	     *
    	     * var abc = function(a, b, c) {
    	     *   return [a, b, c];
    	     * };
    	     *
    	     * var curried = _.curry(abc);
    	     *
    	     * curried(1)(2)(3);
    	     * // => [1, 2, 3]
    	     *
    	     * curried(1, 2)(3);
    	     * // => [1, 2, 3]
    	     *
    	     * curried(1, 2, 3);
    	     * // => [1, 2, 3]
    	     *
    	     * // Curried with placeholders.
    	     * curried(1)(_, 3)(2);
    	     * // => [1, 2, 3]
    	     */function curry(func,arity,guard){arity=guard?undefined$1:arity;var result=createWrap(func,WRAP_CURRY_FLAG,undefined$1,undefined$1,undefined$1,undefined$1,undefined$1,arity);result.placeholder=curry.placeholder;return result;}/**
    	     * This method is like `_.curry` except that arguments are applied to `func`
    	     * in the manner of `_.partialRight` instead of `_.partial`.
    	     *
    	     * The `_.curryRight.placeholder` value, which defaults to `_` in monolithic
    	     * builds, may be used as a placeholder for provided arguments.
    	     *
    	     * **Note:** This method doesn't set the "length" property of curried functions.
    	     *
    	     * @static
    	     * @memberOf _
    	     * @since 3.0.0
    	     * @category Function
    	     * @param {Function} func The function to curry.
    	     * @param {number} [arity=func.length] The arity of `func`.
    	     * @param- {Object} [guard] Enables use as an iteratee for methods like `_.map`.
    	     * @returns {Function} Returns the new curried function.
    	     * @example
    	     *
    	     * var abc = function(a, b, c) {
    	     *   return [a, b, c];
    	     * };
    	     *
    	     * var curried = _.curryRight(abc);
    	     *
    	     * curried(3)(2)(1);
    	     * // => [1, 2, 3]
    	     *
    	     * curried(2, 3)(1);
    	     * // => [1, 2, 3]
    	     *
    	     * curried(1, 2, 3);
    	     * // => [1, 2, 3]
    	     *
    	     * // Curried with placeholders.
    	     * curried(3)(1, _)(2);
    	     * // => [1, 2, 3]
    	     */function curryRight(func,arity,guard){arity=guard?undefined$1:arity;var result=createWrap(func,WRAP_CURRY_RIGHT_FLAG,undefined$1,undefined$1,undefined$1,undefined$1,undefined$1,arity);result.placeholder=curryRight.placeholder;return result;}/**
    	     * Creates a debounced function that delays invoking `func` until after `wait`
    	     * milliseconds have elapsed since the last time the debounced function was
    	     * invoked. The debounced function comes with a `cancel` method to cancel
    	     * delayed `func` invocations and a `flush` method to immediately invoke them.
    	     * Provide `options` to indicate whether `func` should be invoked on the
    	     * leading and/or trailing edge of the `wait` timeout. The `func` is invoked
    	     * with the last arguments provided to the debounced function. Subsequent
    	     * calls to the debounced function return the result of the last `func`
    	     * invocation.
    	     *
    	     * **Note:** If `leading` and `trailing` options are `true`, `func` is
    	     * invoked on the trailing edge of the timeout only if the debounced function
    	     * is invoked more than once during the `wait` timeout.
    	     *
    	     * If `wait` is `0` and `leading` is `false`, `func` invocation is deferred
    	     * until to the next tick, similar to `setTimeout` with a timeout of `0`.
    	     *
    	     * See [David Corbacho's article](https://css-tricks.com/debouncing-throttling-explained-examples/)
    	     * for details over the differences between `_.debounce` and `_.throttle`.
    	     *
    	     * @static
    	     * @memberOf _
    	     * @since 0.1.0
    	     * @category Function
    	     * @param {Function} func The function to debounce.
    	     * @param {number} [wait=0] The number of milliseconds to delay.
    	     * @param {Object} [options={}] The options object.
    	     * @param {boolean} [options.leading=false]
    	     *  Specify invoking on the leading edge of the timeout.
    	     * @param {number} [options.maxWait]
    	     *  The maximum time `func` is allowed to be delayed before it's invoked.
    	     * @param {boolean} [options.trailing=true]
    	     *  Specify invoking on the trailing edge of the timeout.
    	     * @returns {Function} Returns the new debounced function.
    	     * @example
    	     *
    	     * // Avoid costly calculations while the window size is in flux.
    	     * jQuery(window).on('resize', _.debounce(calculateLayout, 150));
    	     *
    	     * // Invoke `sendMail` when clicked, debouncing subsequent calls.
    	     * jQuery(element).on('click', _.debounce(sendMail, 300, {
    	     *   'leading': true,
    	     *   'trailing': false
    	     * }));
    	     *
    	     * // Ensure `batchLog` is invoked once after 1 second of debounced calls.
    	     * var debounced = _.debounce(batchLog, 250, { 'maxWait': 1000 });
    	     * var source = new EventSource('/stream');
    	     * jQuery(source).on('message', debounced);
    	     *
    	     * // Cancel the trailing debounced invocation.
    	     * jQuery(window).on('popstate', debounced.cancel);
    	     */function debounce(func,wait,options){var lastArgs,lastThis,maxWait,result,timerId,lastCallTime,lastInvokeTime=0,leading=false,maxing=false,trailing=true;if(typeof func!='function'){throw new TypeError(FUNC_ERROR_TEXT);}wait=toNumber(wait)||0;if(isObject(options)){leading=!!options.leading;maxing='maxWait'in options;maxWait=maxing?nativeMax(toNumber(options.maxWait)||0,wait):maxWait;trailing='trailing'in options?!!options.trailing:trailing;}function invokeFunc(time){var args=lastArgs,thisArg=lastThis;lastArgs=lastThis=undefined$1;lastInvokeTime=time;result=func.apply(thisArg,args);return result;}function leadingEdge(time){// Reset any `maxWait` timer.
    	lastInvokeTime=time;// Start the timer for the trailing edge.
    	timerId=setTimeout(timerExpired,wait);// Invoke the leading edge.
    	return leading?invokeFunc(time):result;}function remainingWait(time){var timeSinceLastCall=time-lastCallTime,timeSinceLastInvoke=time-lastInvokeTime,timeWaiting=wait-timeSinceLastCall;return maxing?nativeMin(timeWaiting,maxWait-timeSinceLastInvoke):timeWaiting;}function shouldInvoke(time){var timeSinceLastCall=time-lastCallTime,timeSinceLastInvoke=time-lastInvokeTime;// Either this is the first call, activity has stopped and we're at the
    	// trailing edge, the system time has gone backwards and we're treating
    	// it as the trailing edge, or we've hit the `maxWait` limit.
    	return lastCallTime===undefined$1||timeSinceLastCall>=wait||timeSinceLastCall<0||maxing&&timeSinceLastInvoke>=maxWait;}function timerExpired(){var time=now();if(shouldInvoke(time)){return trailingEdge(time);}// Restart the timer.
    	timerId=setTimeout(timerExpired,remainingWait(time));}function trailingEdge(time){timerId=undefined$1;// Only invoke if we have `lastArgs` which means `func` has been
    	// debounced at least once.
    	if(trailing&&lastArgs){return invokeFunc(time);}lastArgs=lastThis=undefined$1;return result;}function cancel(){if(timerId!==undefined$1){clearTimeout(timerId);}lastInvokeTime=0;lastArgs=lastCallTime=lastThis=timerId=undefined$1;}function flush(){return timerId===undefined$1?result:trailingEdge(now());}function debounced(){var time=now(),isInvoking=shouldInvoke(time);lastArgs=arguments;lastThis=this;lastCallTime=time;if(isInvoking){if(timerId===undefined$1){return leadingEdge(lastCallTime);}if(maxing){// Handle invocations in a tight loop.
    	clearTimeout(timerId);timerId=setTimeout(timerExpired,wait);return invokeFunc(lastCallTime);}}if(timerId===undefined$1){timerId=setTimeout(timerExpired,wait);}return result;}debounced.cancel=cancel;debounced.flush=flush;return debounced;}/**
    	     * Defers invoking the `func` until the current call stack has cleared. Any
    	     * additional arguments are provided to `func` when it's invoked.
    	     *
    	     * @static
    	     * @memberOf _
    	     * @since 0.1.0
    	     * @category Function
    	     * @param {Function} func The function to defer.
    	     * @param {...*} [args] The arguments to invoke `func` with.
    	     * @returns {number} Returns the timer id.
    	     * @example
    	     *
    	     * _.defer(function(text) {
    	     *   console.log(text);
    	     * }, 'deferred');
    	     * // => Logs 'deferred' after one millisecond.
    	     */var defer=baseRest(function(func,args){return baseDelay(func,1,args);});/**
    	     * Invokes `func` after `wait` milliseconds. Any additional arguments are
    	     * provided to `func` when it's invoked.
    	     *
    	     * @static
    	     * @memberOf _
    	     * @since 0.1.0
    	     * @category Function
    	     * @param {Function} func The function to delay.
    	     * @param {number} wait The number of milliseconds to delay invocation.
    	     * @param {...*} [args] The arguments to invoke `func` with.
    	     * @returns {number} Returns the timer id.
    	     * @example
    	     *
    	     * _.delay(function(text) {
    	     *   console.log(text);
    	     * }, 1000, 'later');
    	     * // => Logs 'later' after one second.
    	     */var delay=baseRest(function(func,wait,args){return baseDelay(func,toNumber(wait)||0,args);});/**
    	     * Creates a function that invokes `func` with arguments reversed.
    	     *
    	     * @static
    	     * @memberOf _
    	     * @since 4.0.0
    	     * @category Function
    	     * @param {Function} func The function to flip arguments for.
    	     * @returns {Function} Returns the new flipped function.
    	     * @example
    	     *
    	     * var flipped = _.flip(function() {
    	     *   return _.toArray(arguments);
    	     * });
    	     *
    	     * flipped('a', 'b', 'c', 'd');
    	     * // => ['d', 'c', 'b', 'a']
    	     */function flip(func){return createWrap(func,WRAP_FLIP_FLAG);}/**
    	     * Creates a function that memoizes the result of `func`. If `resolver` is
    	     * provided, it determines the cache key for storing the result based on the
    	     * arguments provided to the memoized function. By default, the first argument
    	     * provided to the memoized function is used as the map cache key. The `func`
    	     * is invoked with the `this` binding of the memoized function.
    	     *
    	     * **Note:** The cache is exposed as the `cache` property on the memoized
    	     * function. Its creation may be customized by replacing the `_.memoize.Cache`
    	     * constructor with one whose instances implement the
    	     * [`Map`](http://ecma-international.org/ecma-262/7.0/#sec-properties-of-the-map-prototype-object)
    	     * method interface of `clear`, `delete`, `get`, `has`, and `set`.
    	     *
    	     * @static
    	     * @memberOf _
    	     * @since 0.1.0
    	     * @category Function
    	     * @param {Function} func The function to have its output memoized.
    	     * @param {Function} [resolver] The function to resolve the cache key.
    	     * @returns {Function} Returns the new memoized function.
    	     * @example
    	     *
    	     * var object = { 'a': 1, 'b': 2 };
    	     * var other = { 'c': 3, 'd': 4 };
    	     *
    	     * var values = _.memoize(_.values);
    	     * values(object);
    	     * // => [1, 2]
    	     *
    	     * values(other);
    	     * // => [3, 4]
    	     *
    	     * object.a = 2;
    	     * values(object);
    	     * // => [1, 2]
    	     *
    	     * // Modify the result cache.
    	     * values.cache.set(object, ['a', 'b']);
    	     * values(object);
    	     * // => ['a', 'b']
    	     *
    	     * // Replace `_.memoize.Cache`.
    	     * _.memoize.Cache = WeakMap;
    	     */function memoize(func,resolver){if(typeof func!='function'||resolver!=null&&typeof resolver!='function'){throw new TypeError(FUNC_ERROR_TEXT);}var memoized=function(){var args=arguments,key=resolver?resolver.apply(this,args):args[0],cache=memoized.cache;if(cache.has(key)){return cache.get(key);}var result=func.apply(this,args);memoized.cache=cache.set(key,result)||cache;return result;};memoized.cache=new(memoize.Cache||MapCache)();return memoized;}// Expose `MapCache`.
    	memoize.Cache=MapCache;/**
    	     * Creates a function that negates the result of the predicate `func`. The
    	     * `func` predicate is invoked with the `this` binding and arguments of the
    	     * created function.
    	     *
    	     * @static
    	     * @memberOf _
    	     * @since 3.0.0
    	     * @category Function
    	     * @param {Function} predicate The predicate to negate.
    	     * @returns {Function} Returns the new negated function.
    	     * @example
    	     *
    	     * function isEven(n) {
    	     *   return n % 2 == 0;
    	     * }
    	     *
    	     * _.filter([1, 2, 3, 4, 5, 6], _.negate(isEven));
    	     * // => [1, 3, 5]
    	     */function negate(predicate){if(typeof predicate!='function'){throw new TypeError(FUNC_ERROR_TEXT);}return function(){var args=arguments;switch(args.length){case 0:return !predicate.call(this);case 1:return !predicate.call(this,args[0]);case 2:return !predicate.call(this,args[0],args[1]);case 3:return !predicate.call(this,args[0],args[1],args[2]);}return !predicate.apply(this,args);};}/**
    	     * Creates a function that is restricted to invoking `func` once. Repeat calls
    	     * to the function return the value of the first invocation. The `func` is
    	     * invoked with the `this` binding and arguments of the created function.
    	     *
    	     * @static
    	     * @memberOf _
    	     * @since 0.1.0
    	     * @category Function
    	     * @param {Function} func The function to restrict.
    	     * @returns {Function} Returns the new restricted function.
    	     * @example
    	     *
    	     * var initialize = _.once(createApplication);
    	     * initialize();
    	     * initialize();
    	     * // => `createApplication` is invoked once
    	     */function once(func){return before(2,func);}/**
    	     * Creates a function that invokes `func` with its arguments transformed.
    	     *
    	     * @static
    	     * @since 4.0.0
    	     * @memberOf _
    	     * @category Function
    	     * @param {Function} func The function to wrap.
    	     * @param {...(Function|Function[])} [transforms=[_.identity]]
    	     *  The argument transforms.
    	     * @returns {Function} Returns the new function.
    	     * @example
    	     *
    	     * function doubled(n) {
    	     *   return n * 2;
    	     * }
    	     *
    	     * function square(n) {
    	     *   return n * n;
    	     * }
    	     *
    	     * var func = _.overArgs(function(x, y) {
    	     *   return [x, y];
    	     * }, [square, doubled]);
    	     *
    	     * func(9, 3);
    	     * // => [81, 6]
    	     *
    	     * func(10, 5);
    	     * // => [100, 10]
    	     */var overArgs=castRest(function(func,transforms){transforms=transforms.length==1&&isArray(transforms[0])?arrayMap(transforms[0],baseUnary(getIteratee())):arrayMap(baseFlatten(transforms,1),baseUnary(getIteratee()));var funcsLength=transforms.length;return baseRest(function(args){var index=-1,length=nativeMin(args.length,funcsLength);while(++index<length){args[index]=transforms[index].call(this,args[index]);}return apply(func,this,args);});});/**
    	     * Creates a function that invokes `func` with `partials` prepended to the
    	     * arguments it receives. This method is like `_.bind` except it does **not**
    	     * alter the `this` binding.
    	     *
    	     * The `_.partial.placeholder` value, which defaults to `_` in monolithic
    	     * builds, may be used as a placeholder for partially applied arguments.
    	     *
    	     * **Note:** This method doesn't set the "length" property of partially
    	     * applied functions.
    	     *
    	     * @static
    	     * @memberOf _
    	     * @since 0.2.0
    	     * @category Function
    	     * @param {Function} func The function to partially apply arguments to.
    	     * @param {...*} [partials] The arguments to be partially applied.
    	     * @returns {Function} Returns the new partially applied function.
    	     * @example
    	     *
    	     * function greet(greeting, name) {
    	     *   return greeting + ' ' + name;
    	     * }
    	     *
    	     * var sayHelloTo = _.partial(greet, 'hello');
    	     * sayHelloTo('fred');
    	     * // => 'hello fred'
    	     *
    	     * // Partially applied with placeholders.
    	     * var greetFred = _.partial(greet, _, 'fred');
    	     * greetFred('hi');
    	     * // => 'hi fred'
    	     */var partial=baseRest(function(func,partials){var holders=replaceHolders(partials,getHolder(partial));return createWrap(func,WRAP_PARTIAL_FLAG,undefined$1,partials,holders);});/**
    	     * This method is like `_.partial` except that partially applied arguments
    	     * are appended to the arguments it receives.
    	     *
    	     * The `_.partialRight.placeholder` value, which defaults to `_` in monolithic
    	     * builds, may be used as a placeholder for partially applied arguments.
    	     *
    	     * **Note:** This method doesn't set the "length" property of partially
    	     * applied functions.
    	     *
    	     * @static
    	     * @memberOf _
    	     * @since 1.0.0
    	     * @category Function
    	     * @param {Function} func The function to partially apply arguments to.
    	     * @param {...*} [partials] The arguments to be partially applied.
    	     * @returns {Function} Returns the new partially applied function.
    	     * @example
    	     *
    	     * function greet(greeting, name) {
    	     *   return greeting + ' ' + name;
    	     * }
    	     *
    	     * var greetFred = _.partialRight(greet, 'fred');
    	     * greetFred('hi');
    	     * // => 'hi fred'
    	     *
    	     * // Partially applied with placeholders.
    	     * var sayHelloTo = _.partialRight(greet, 'hello', _);
    	     * sayHelloTo('fred');
    	     * // => 'hello fred'
    	     */var partialRight=baseRest(function(func,partials){var holders=replaceHolders(partials,getHolder(partialRight));return createWrap(func,WRAP_PARTIAL_RIGHT_FLAG,undefined$1,partials,holders);});/**
    	     * Creates a function that invokes `func` with arguments arranged according
    	     * to the specified `indexes` where the argument value at the first index is
    	     * provided as the first argument, the argument value at the second index is
    	     * provided as the second argument, and so on.
    	     *
    	     * @static
    	     * @memberOf _
    	     * @since 3.0.0
    	     * @category Function
    	     * @param {Function} func The function to rearrange arguments for.
    	     * @param {...(number|number[])} indexes The arranged argument indexes.
    	     * @returns {Function} Returns the new function.
    	     * @example
    	     *
    	     * var rearged = _.rearg(function(a, b, c) {
    	     *   return [a, b, c];
    	     * }, [2, 0, 1]);
    	     *
    	     * rearged('b', 'c', 'a')
    	     * // => ['a', 'b', 'c']
    	     */var rearg=flatRest(function(func,indexes){return createWrap(func,WRAP_REARG_FLAG,undefined$1,undefined$1,undefined$1,indexes);});/**
    	     * Creates a function that invokes `func` with the `this` binding of the
    	     * created function and arguments from `start` and beyond provided as
    	     * an array.
    	     *
    	     * **Note:** This method is based on the
    	     * [rest parameter](https://mdn.io/rest_parameters).
    	     *
    	     * @static
    	     * @memberOf _
    	     * @since 4.0.0
    	     * @category Function
    	     * @param {Function} func The function to apply a rest parameter to.
    	     * @param {number} [start=func.length-1] The start position of the rest parameter.
    	     * @returns {Function} Returns the new function.
    	     * @example
    	     *
    	     * var say = _.rest(function(what, names) {
    	     *   return what + ' ' + _.initial(names).join(', ') +
    	     *     (_.size(names) > 1 ? ', & ' : '') + _.last(names);
    	     * });
    	     *
    	     * say('hello', 'fred', 'barney', 'pebbles');
    	     * // => 'hello fred, barney, & pebbles'
    	     */function rest(func,start){if(typeof func!='function'){throw new TypeError(FUNC_ERROR_TEXT);}start=start===undefined$1?start:toInteger(start);return baseRest(func,start);}/**
    	     * Creates a function that invokes `func` with the `this` binding of the
    	     * create function and an array of arguments much like
    	     * [`Function#apply`](http://www.ecma-international.org/ecma-262/7.0/#sec-function.prototype.apply).
    	     *
    	     * **Note:** This method is based on the
    	     * [spread operator](https://mdn.io/spread_operator).
    	     *
    	     * @static
    	     * @memberOf _
    	     * @since 3.2.0
    	     * @category Function
    	     * @param {Function} func The function to spread arguments over.
    	     * @param {number} [start=0] The start position of the spread.
    	     * @returns {Function} Returns the new function.
    	     * @example
    	     *
    	     * var say = _.spread(function(who, what) {
    	     *   return who + ' says ' + what;
    	     * });
    	     *
    	     * say(['fred', 'hello']);
    	     * // => 'fred says hello'
    	     *
    	     * var numbers = Promise.all([
    	     *   Promise.resolve(40),
    	     *   Promise.resolve(36)
    	     * ]);
    	     *
    	     * numbers.then(_.spread(function(x, y) {
    	     *   return x + y;
    	     * }));
    	     * // => a Promise of 76
    	     */function spread(func,start){if(typeof func!='function'){throw new TypeError(FUNC_ERROR_TEXT);}start=start==null?0:nativeMax(toInteger(start),0);return baseRest(function(args){var array=args[start],otherArgs=castSlice(args,0,start);if(array){arrayPush(otherArgs,array);}return apply(func,this,otherArgs);});}/**
    	     * Creates a throttled function that only invokes `func` at most once per
    	     * every `wait` milliseconds. The throttled function comes with a `cancel`
    	     * method to cancel delayed `func` invocations and a `flush` method to
    	     * immediately invoke them. Provide `options` to indicate whether `func`
    	     * should be invoked on the leading and/or trailing edge of the `wait`
    	     * timeout. The `func` is invoked with the last arguments provided to the
    	     * throttled function. Subsequent calls to the throttled function return the
    	     * result of the last `func` invocation.
    	     *
    	     * **Note:** If `leading` and `trailing` options are `true`, `func` is
    	     * invoked on the trailing edge of the timeout only if the throttled function
    	     * is invoked more than once during the `wait` timeout.
    	     *
    	     * If `wait` is `0` and `leading` is `false`, `func` invocation is deferred
    	     * until to the next tick, similar to `setTimeout` with a timeout of `0`.
    	     *
    	     * See [David Corbacho's article](https://css-tricks.com/debouncing-throttling-explained-examples/)
    	     * for details over the differences between `_.throttle` and `_.debounce`.
    	     *
    	     * @static
    	     * @memberOf _
    	     * @since 0.1.0
    	     * @category Function
    	     * @param {Function} func The function to throttle.
    	     * @param {number} [wait=0] The number of milliseconds to throttle invocations to.
    	     * @param {Object} [options={}] The options object.
    	     * @param {boolean} [options.leading=true]
    	     *  Specify invoking on the leading edge of the timeout.
    	     * @param {boolean} [options.trailing=true]
    	     *  Specify invoking on the trailing edge of the timeout.
    	     * @returns {Function} Returns the new throttled function.
    	     * @example
    	     *
    	     * // Avoid excessively updating the position while scrolling.
    	     * jQuery(window).on('scroll', _.throttle(updatePosition, 100));
    	     *
    	     * // Invoke `renewToken` when the click event is fired, but not more than once every 5 minutes.
    	     * var throttled = _.throttle(renewToken, 300000, { 'trailing': false });
    	     * jQuery(element).on('click', throttled);
    	     *
    	     * // Cancel the trailing throttled invocation.
    	     * jQuery(window).on('popstate', throttled.cancel);
    	     */function throttle(func,wait,options){var leading=true,trailing=true;if(typeof func!='function'){throw new TypeError(FUNC_ERROR_TEXT);}if(isObject(options)){leading='leading'in options?!!options.leading:leading;trailing='trailing'in options?!!options.trailing:trailing;}return debounce(func,wait,{'leading':leading,'maxWait':wait,'trailing':trailing});}/**
    	     * Creates a function that accepts up to one argument, ignoring any
    	     * additional arguments.
    	     *
    	     * @static
    	     * @memberOf _
    	     * @since 4.0.0
    	     * @category Function
    	     * @param {Function} func The function to cap arguments for.
    	     * @returns {Function} Returns the new capped function.
    	     * @example
    	     *
    	     * _.map(['6', '8', '10'], _.unary(parseInt));
    	     * // => [6, 8, 10]
    	     */function unary(func){return ary(func,1);}/**
    	     * Creates a function that provides `value` to `wrapper` as its first
    	     * argument. Any additional arguments provided to the function are appended
    	     * to those provided to the `wrapper`. The wrapper is invoked with the `this`
    	     * binding of the created function.
    	     *
    	     * @static
    	     * @memberOf _
    	     * @since 0.1.0
    	     * @category Function
    	     * @param {*} value The value to wrap.
    	     * @param {Function} [wrapper=identity] The wrapper function.
    	     * @returns {Function} Returns the new function.
    	     * @example
    	     *
    	     * var p = _.wrap(_.escape, function(func, text) {
    	     *   return '<p>' + func(text) + '</p>';
    	     * });
    	     *
    	     * p('fred, barney, & pebbles');
    	     * // => '<p>fred, barney, &amp; pebbles</p>'
    	     */function wrap(value,wrapper){return partial(castFunction(wrapper),value);}/*------------------------------------------------------------------------*//**
    	     * Casts `value` as an array if it's not one.
    	     *
    	     * @static
    	     * @memberOf _
    	     * @since 4.4.0
    	     * @category Lang
    	     * @param {*} value The value to inspect.
    	     * @returns {Array} Returns the cast array.
    	     * @example
    	     *
    	     * _.castArray(1);
    	     * // => [1]
    	     *
    	     * _.castArray({ 'a': 1 });
    	     * // => [{ 'a': 1 }]
    	     *
    	     * _.castArray('abc');
    	     * // => ['abc']
    	     *
    	     * _.castArray(null);
    	     * // => [null]
    	     *
    	     * _.castArray(undefined);
    	     * // => [undefined]
    	     *
    	     * _.castArray();
    	     * // => []
    	     *
    	     * var array = [1, 2, 3];
    	     * console.log(_.castArray(array) === array);
    	     * // => true
    	     */function castArray(){if(!arguments.length){return [];}var value=arguments[0];return isArray(value)?value:[value];}/**
    	     * Creates a shallow clone of `value`.
    	     *
    	     * **Note:** This method is loosely based on the
    	     * [structured clone algorithm](https://mdn.io/Structured_clone_algorithm)
    	     * and supports cloning arrays, array buffers, booleans, date objects, maps,
    	     * numbers, `Object` objects, regexes, sets, strings, symbols, and typed
    	     * arrays. The own enumerable properties of `arguments` objects are cloned
    	     * as plain objects. An empty object is returned for uncloneable values such
    	     * as error objects, functions, DOM nodes, and WeakMaps.
    	     *
    	     * @static
    	     * @memberOf _
    	     * @since 0.1.0
    	     * @category Lang
    	     * @param {*} value The value to clone.
    	     * @returns {*} Returns the cloned value.
    	     * @see _.cloneDeep
    	     * @example
    	     *
    	     * var objects = [{ 'a': 1 }, { 'b': 2 }];
    	     *
    	     * var shallow = _.clone(objects);
    	     * console.log(shallow[0] === objects[0]);
    	     * // => true
    	     */function clone(value){return baseClone(value,CLONE_SYMBOLS_FLAG);}/**
    	     * This method is like `_.clone` except that it accepts `customizer` which
    	     * is invoked to produce the cloned value. If `customizer` returns `undefined`,
    	     * cloning is handled by the method instead. The `customizer` is invoked with
    	     * up to four arguments; (value [, index|key, object, stack]).
    	     *
    	     * @static
    	     * @memberOf _
    	     * @since 4.0.0
    	     * @category Lang
    	     * @param {*} value The value to clone.
    	     * @param {Function} [customizer] The function to customize cloning.
    	     * @returns {*} Returns the cloned value.
    	     * @see _.cloneDeepWith
    	     * @example
    	     *
    	     * function customizer(value) {
    	     *   if (_.isElement(value)) {
    	     *     return value.cloneNode(false);
    	     *   }
    	     * }
    	     *
    	     * var el = _.cloneWith(document.body, customizer);
    	     *
    	     * console.log(el === document.body);
    	     * // => false
    	     * console.log(el.nodeName);
    	     * // => 'BODY'
    	     * console.log(el.childNodes.length);
    	     * // => 0
    	     */function cloneWith(value,customizer){customizer=typeof customizer=='function'?customizer:undefined$1;return baseClone(value,CLONE_SYMBOLS_FLAG,customizer);}/**
    	     * This method is like `_.clone` except that it recursively clones `value`.
    	     *
    	     * @static
    	     * @memberOf _
    	     * @since 1.0.0
    	     * @category Lang
    	     * @param {*} value The value to recursively clone.
    	     * @returns {*} Returns the deep cloned value.
    	     * @see _.clone
    	     * @example
    	     *
    	     * var objects = [{ 'a': 1 }, { 'b': 2 }];
    	     *
    	     * var deep = _.cloneDeep(objects);
    	     * console.log(deep[0] === objects[0]);
    	     * // => false
    	     */function cloneDeep(value){return baseClone(value,CLONE_DEEP_FLAG|CLONE_SYMBOLS_FLAG);}/**
    	     * This method is like `_.cloneWith` except that it recursively clones `value`.
    	     *
    	     * @static
    	     * @memberOf _
    	     * @since 4.0.0
    	     * @category Lang
    	     * @param {*} value The value to recursively clone.
    	     * @param {Function} [customizer] The function to customize cloning.
    	     * @returns {*} Returns the deep cloned value.
    	     * @see _.cloneWith
    	     * @example
    	     *
    	     * function customizer(value) {
    	     *   if (_.isElement(value)) {
    	     *     return value.cloneNode(true);
    	     *   }
    	     * }
    	     *
    	     * var el = _.cloneDeepWith(document.body, customizer);
    	     *
    	     * console.log(el === document.body);
    	     * // => false
    	     * console.log(el.nodeName);
    	     * // => 'BODY'
    	     * console.log(el.childNodes.length);
    	     * // => 20
    	     */function cloneDeepWith(value,customizer){customizer=typeof customizer=='function'?customizer:undefined$1;return baseClone(value,CLONE_DEEP_FLAG|CLONE_SYMBOLS_FLAG,customizer);}/**
    	     * Checks if `object` conforms to `source` by invoking the predicate
    	     * properties of `source` with the corresponding property values of `object`.
    	     *
    	     * **Note:** This method is equivalent to `_.conforms` when `source` is
    	     * partially applied.
    	     *
    	     * @static
    	     * @memberOf _
    	     * @since 4.14.0
    	     * @category Lang
    	     * @param {Object} object The object to inspect.
    	     * @param {Object} source The object of property predicates to conform to.
    	     * @returns {boolean} Returns `true` if `object` conforms, else `false`.
    	     * @example
    	     *
    	     * var object = { 'a': 1, 'b': 2 };
    	     *
    	     * _.conformsTo(object, { 'b': function(n) { return n > 1; } });
    	     * // => true
    	     *
    	     * _.conformsTo(object, { 'b': function(n) { return n > 2; } });
    	     * // => false
    	     */function conformsTo(object,source){return source==null||baseConformsTo(object,source,keys(source));}/**
    	     * Performs a
    	     * [`SameValueZero`](http://ecma-international.org/ecma-262/7.0/#sec-samevaluezero)
    	     * comparison between two values to determine if they are equivalent.
    	     *
    	     * @static
    	     * @memberOf _
    	     * @since 4.0.0
    	     * @category Lang
    	     * @param {*} value The value to compare.
    	     * @param {*} other The other value to compare.
    	     * @returns {boolean} Returns `true` if the values are equivalent, else `false`.
    	     * @example
    	     *
    	     * var object = { 'a': 1 };
    	     * var other = { 'a': 1 };
    	     *
    	     * _.eq(object, object);
    	     * // => true
    	     *
    	     * _.eq(object, other);
    	     * // => false
    	     *
    	     * _.eq('a', 'a');
    	     * // => true
    	     *
    	     * _.eq('a', Object('a'));
    	     * // => false
    	     *
    	     * _.eq(NaN, NaN);
    	     * // => true
    	     */function eq(value,other){return value===other||value!==value&&other!==other;}/**
    	     * Checks if `value` is greater than `other`.
    	     *
    	     * @static
    	     * @memberOf _
    	     * @since 3.9.0
    	     * @category Lang
    	     * @param {*} value The value to compare.
    	     * @param {*} other The other value to compare.
    	     * @returns {boolean} Returns `true` if `value` is greater than `other`,
    	     *  else `false`.
    	     * @see _.lt
    	     * @example
    	     *
    	     * _.gt(3, 1);
    	     * // => true
    	     *
    	     * _.gt(3, 3);
    	     * // => false
    	     *
    	     * _.gt(1, 3);
    	     * // => false
    	     */var gt=createRelationalOperation(baseGt);/**
    	     * Checks if `value` is greater than or equal to `other`.
    	     *
    	     * @static
    	     * @memberOf _
    	     * @since 3.9.0
    	     * @category Lang
    	     * @param {*} value The value to compare.
    	     * @param {*} other The other value to compare.
    	     * @returns {boolean} Returns `true` if `value` is greater than or equal to
    	     *  `other`, else `false`.
    	     * @see _.lte
    	     * @example
    	     *
    	     * _.gte(3, 1);
    	     * // => true
    	     *
    	     * _.gte(3, 3);
    	     * // => true
    	     *
    	     * _.gte(1, 3);
    	     * // => false
    	     */var gte=createRelationalOperation(function(value,other){return value>=other;});/**
    	     * Checks if `value` is likely an `arguments` object.
    	     *
    	     * @static
    	     * @memberOf _
    	     * @since 0.1.0
    	     * @category Lang
    	     * @param {*} value The value to check.
    	     * @returns {boolean} Returns `true` if `value` is an `arguments` object,
    	     *  else `false`.
    	     * @example
    	     *
    	     * _.isArguments(function() { return arguments; }());
    	     * // => true
    	     *
    	     * _.isArguments([1, 2, 3]);
    	     * // => false
    	     */var isArguments=baseIsArguments(function(){return arguments;}())?baseIsArguments:function(value){return isObjectLike(value)&&hasOwnProperty.call(value,'callee')&&!propertyIsEnumerable.call(value,'callee');};/**
    	     * Checks if `value` is classified as an `Array` object.
    	     *
    	     * @static
    	     * @memberOf _
    	     * @since 0.1.0
    	     * @category Lang
    	     * @param {*} value The value to check.
    	     * @returns {boolean} Returns `true` if `value` is an array, else `false`.
    	     * @example
    	     *
    	     * _.isArray([1, 2, 3]);
    	     * // => true
    	     *
    	     * _.isArray(document.body.children);
    	     * // => false
    	     *
    	     * _.isArray('abc');
    	     * // => false
    	     *
    	     * _.isArray(_.noop);
    	     * // => false
    	     */var isArray=Array.isArray;/**
    	     * Checks if `value` is classified as an `ArrayBuffer` object.
    	     *
    	     * @static
    	     * @memberOf _
    	     * @since 4.3.0
    	     * @category Lang
    	     * @param {*} value The value to check.
    	     * @returns {boolean} Returns `true` if `value` is an array buffer, else `false`.
    	     * @example
    	     *
    	     * _.isArrayBuffer(new ArrayBuffer(2));
    	     * // => true
    	     *
    	     * _.isArrayBuffer(new Array(2));
    	     * // => false
    	     */var isArrayBuffer=nodeIsArrayBuffer?baseUnary(nodeIsArrayBuffer):baseIsArrayBuffer;/**
    	     * Checks if `value` is array-like. A value is considered array-like if it's
    	     * not a function and has a `value.length` that's an integer greater than or
    	     * equal to `0` and less than or equal to `Number.MAX_SAFE_INTEGER`.
    	     *
    	     * @static
    	     * @memberOf _
    	     * @since 4.0.0
    	     * @category Lang
    	     * @param {*} value The value to check.
    	     * @returns {boolean} Returns `true` if `value` is array-like, else `false`.
    	     * @example
    	     *
    	     * _.isArrayLike([1, 2, 3]);
    	     * // => true
    	     *
    	     * _.isArrayLike(document.body.children);
    	     * // => true
    	     *
    	     * _.isArrayLike('abc');
    	     * // => true
    	     *
    	     * _.isArrayLike(_.noop);
    	     * // => false
    	     */function isArrayLike(value){return value!=null&&isLength(value.length)&&!isFunction(value);}/**
    	     * This method is like `_.isArrayLike` except that it also checks if `value`
    	     * is an object.
    	     *
    	     * @static
    	     * @memberOf _
    	     * @since 4.0.0
    	     * @category Lang
    	     * @param {*} value The value to check.
    	     * @returns {boolean} Returns `true` if `value` is an array-like object,
    	     *  else `false`.
    	     * @example
    	     *
    	     * _.isArrayLikeObject([1, 2, 3]);
    	     * // => true
    	     *
    	     * _.isArrayLikeObject(document.body.children);
    	     * // => true
    	     *
    	     * _.isArrayLikeObject('abc');
    	     * // => false
    	     *
    	     * _.isArrayLikeObject(_.noop);
    	     * // => false
    	     */function isArrayLikeObject(value){return isObjectLike(value)&&isArrayLike(value);}/**
    	     * Checks if `value` is classified as a boolean primitive or object.
    	     *
    	     * @static
    	     * @memberOf _
    	     * @since 0.1.0
    	     * @category Lang
    	     * @param {*} value The value to check.
    	     * @returns {boolean} Returns `true` if `value` is a boolean, else `false`.
    	     * @example
    	     *
    	     * _.isBoolean(false);
    	     * // => true
    	     *
    	     * _.isBoolean(null);
    	     * // => false
    	     */function isBoolean(value){return value===true||value===false||isObjectLike(value)&&baseGetTag(value)==boolTag;}/**
    	     * Checks if `value` is a buffer.
    	     *
    	     * @static
    	     * @memberOf _
    	     * @since 4.3.0
    	     * @category Lang
    	     * @param {*} value The value to check.
    	     * @returns {boolean} Returns `true` if `value` is a buffer, else `false`.
    	     * @example
    	     *
    	     * _.isBuffer(new Buffer(2));
    	     * // => true
    	     *
    	     * _.isBuffer(new Uint8Array(2));
    	     * // => false
    	     */var isBuffer=nativeIsBuffer||stubFalse;/**
    	     * Checks if `value` is classified as a `Date` object.
    	     *
    	     * @static
    	     * @memberOf _
    	     * @since 0.1.0
    	     * @category Lang
    	     * @param {*} value The value to check.
    	     * @returns {boolean} Returns `true` if `value` is a date object, else `false`.
    	     * @example
    	     *
    	     * _.isDate(new Date);
    	     * // => true
    	     *
    	     * _.isDate('Mon April 23 2012');
    	     * // => false
    	     */var isDate=nodeIsDate?baseUnary(nodeIsDate):baseIsDate;/**
    	     * Checks if `value` is likely a DOM element.
    	     *
    	     * @static
    	     * @memberOf _
    	     * @since 0.1.0
    	     * @category Lang
    	     * @param {*} value The value to check.
    	     * @returns {boolean} Returns `true` if `value` is a DOM element, else `false`.
    	     * @example
    	     *
    	     * _.isElement(document.body);
    	     * // => true
    	     *
    	     * _.isElement('<body>');
    	     * // => false
    	     */function isElement(value){return isObjectLike(value)&&value.nodeType===1&&!isPlainObject(value);}/**
    	     * Checks if `value` is an empty object, collection, map, or set.
    	     *
    	     * Objects are considered empty if they have no own enumerable string keyed
    	     * properties.
    	     *
    	     * Array-like values such as `arguments` objects, arrays, buffers, strings, or
    	     * jQuery-like collections are considered empty if they have a `length` of `0`.
    	     * Similarly, maps and sets are considered empty if they have a `size` of `0`.
    	     *
    	     * @static
    	     * @memberOf _
    	     * @since 0.1.0
    	     * @category Lang
    	     * @param {*} value The value to check.
    	     * @returns {boolean} Returns `true` if `value` is empty, else `false`.
    	     * @example
    	     *
    	     * _.isEmpty(null);
    	     * // => true
    	     *
    	     * _.isEmpty(true);
    	     * // => true
    	     *
    	     * _.isEmpty(1);
    	     * // => true
    	     *
    	     * _.isEmpty([1, 2, 3]);
    	     * // => false
    	     *
    	     * _.isEmpty({ 'a': 1 });
    	     * // => false
    	     */function isEmpty(value){if(value==null){return true;}if(isArrayLike(value)&&(isArray(value)||typeof value=='string'||typeof value.splice=='function'||isBuffer(value)||isTypedArray(value)||isArguments(value))){return !value.length;}var tag=getTag(value);if(tag==mapTag||tag==setTag){return !value.size;}if(isPrototype(value)){return !baseKeys(value).length;}for(var key in value){if(hasOwnProperty.call(value,key)){return false;}}return true;}/**
    	     * Performs a deep comparison between two values to determine if they are
    	     * equivalent.
    	     *
    	     * **Note:** This method supports comparing arrays, array buffers, booleans,
    	     * date objects, error objects, maps, numbers, `Object` objects, regexes,
    	     * sets, strings, symbols, and typed arrays. `Object` objects are compared
    	     * by their own, not inherited, enumerable properties. Functions and DOM
    	     * nodes are compared by strict equality, i.e. `===`.
    	     *
    	     * @static
    	     * @memberOf _
    	     * @since 0.1.0
    	     * @category Lang
    	     * @param {*} value The value to compare.
    	     * @param {*} other The other value to compare.
    	     * @returns {boolean} Returns `true` if the values are equivalent, else `false`.
    	     * @example
    	     *
    	     * var object = { 'a': 1 };
    	     * var other = { 'a': 1 };
    	     *
    	     * _.isEqual(object, other);
    	     * // => true
    	     *
    	     * object === other;
    	     * // => false
    	     */function isEqual(value,other){return baseIsEqual(value,other);}/**
    	     * This method is like `_.isEqual` except that it accepts `customizer` which
    	     * is invoked to compare values. If `customizer` returns `undefined`, comparisons
    	     * are handled by the method instead. The `customizer` is invoked with up to
    	     * six arguments: (objValue, othValue [, index|key, object, other, stack]).
    	     *
    	     * @static
    	     * @memberOf _
    	     * @since 4.0.0
    	     * @category Lang
    	     * @param {*} value The value to compare.
    	     * @param {*} other The other value to compare.
    	     * @param {Function} [customizer] The function to customize comparisons.
    	     * @returns {boolean} Returns `true` if the values are equivalent, else `false`.
    	     * @example
    	     *
    	     * function isGreeting(value) {
    	     *   return /^h(?:i|ello)$/.test(value);
    	     * }
    	     *
    	     * function customizer(objValue, othValue) {
    	     *   if (isGreeting(objValue) && isGreeting(othValue)) {
    	     *     return true;
    	     *   }
    	     * }
    	     *
    	     * var array = ['hello', 'goodbye'];
    	     * var other = ['hi', 'goodbye'];
    	     *
    	     * _.isEqualWith(array, other, customizer);
    	     * // => true
    	     */function isEqualWith(value,other,customizer){customizer=typeof customizer=='function'?customizer:undefined$1;var result=customizer?customizer(value,other):undefined$1;return result===undefined$1?baseIsEqual(value,other,undefined$1,customizer):!!result;}/**
    	     * Checks if `value` is an `Error`, `EvalError`, `RangeError`, `ReferenceError`,
    	     * `SyntaxError`, `TypeError`, or `URIError` object.
    	     *
    	     * @static
    	     * @memberOf _
    	     * @since 3.0.0
    	     * @category Lang
    	     * @param {*} value The value to check.
    	     * @returns {boolean} Returns `true` if `value` is an error object, else `false`.
    	     * @example
    	     *
    	     * _.isError(new Error);
    	     * // => true
    	     *
    	     * _.isError(Error);
    	     * // => false
    	     */function isError(value){if(!isObjectLike(value)){return false;}var tag=baseGetTag(value);return tag==errorTag||tag==domExcTag||typeof value.message=='string'&&typeof value.name=='string'&&!isPlainObject(value);}/**
    	     * Checks if `value` is a finite primitive number.
    	     *
    	     * **Note:** This method is based on
    	     * [`Number.isFinite`](https://mdn.io/Number/isFinite).
    	     *
    	     * @static
    	     * @memberOf _
    	     * @since 0.1.0
    	     * @category Lang
    	     * @param {*} value The value to check.
    	     * @returns {boolean} Returns `true` if `value` is a finite number, else `false`.
    	     * @example
    	     *
    	     * _.isFinite(3);
    	     * // => true
    	     *
    	     * _.isFinite(Number.MIN_VALUE);
    	     * // => true
    	     *
    	     * _.isFinite(Infinity);
    	     * // => false
    	     *
    	     * _.isFinite('3');
    	     * // => false
    	     */function isFinite(value){return typeof value=='number'&&nativeIsFinite(value);}/**
    	     * Checks if `value` is classified as a `Function` object.
    	     *
    	     * @static
    	     * @memberOf _
    	     * @since 0.1.0
    	     * @category Lang
    	     * @param {*} value The value to check.
    	     * @returns {boolean} Returns `true` if `value` is a function, else `false`.
    	     * @example
    	     *
    	     * _.isFunction(_);
    	     * // => true
    	     *
    	     * _.isFunction(/abc/);
    	     * // => false
    	     */function isFunction(value){if(!isObject(value)){return false;}// The use of `Object#toString` avoids issues with the `typeof` operator
    	// in Safari 9 which returns 'object' for typed arrays and other constructors.
    	var tag=baseGetTag(value);return tag==funcTag||tag==genTag||tag==asyncTag||tag==proxyTag;}/**
    	     * Checks if `value` is an integer.
    	     *
    	     * **Note:** This method is based on
    	     * [`Number.isInteger`](https://mdn.io/Number/isInteger).
    	     *
    	     * @static
    	     * @memberOf _
    	     * @since 4.0.0
    	     * @category Lang
    	     * @param {*} value The value to check.
    	     * @returns {boolean} Returns `true` if `value` is an integer, else `false`.
    	     * @example
    	     *
    	     * _.isInteger(3);
    	     * // => true
    	     *
    	     * _.isInteger(Number.MIN_VALUE);
    	     * // => false
    	     *
    	     * _.isInteger(Infinity);
    	     * // => false
    	     *
    	     * _.isInteger('3');
    	     * // => false
    	     */function isInteger(value){return typeof value=='number'&&value==toInteger(value);}/**
    	     * Checks if `value` is a valid array-like length.
    	     *
    	     * **Note:** This method is loosely based on
    	     * [`ToLength`](http://ecma-international.org/ecma-262/7.0/#sec-tolength).
    	     *
    	     * @static
    	     * @memberOf _
    	     * @since 4.0.0
    	     * @category Lang
    	     * @param {*} value The value to check.
    	     * @returns {boolean} Returns `true` if `value` is a valid length, else `false`.
    	     * @example
    	     *
    	     * _.isLength(3);
    	     * // => true
    	     *
    	     * _.isLength(Number.MIN_VALUE);
    	     * // => false
    	     *
    	     * _.isLength(Infinity);
    	     * // => false
    	     *
    	     * _.isLength('3');
    	     * // => false
    	     */function isLength(value){return typeof value=='number'&&value>-1&&value%1==0&&value<=MAX_SAFE_INTEGER;}/**
    	     * Checks if `value` is the
    	     * [language type](http://www.ecma-international.org/ecma-262/7.0/#sec-ecmascript-language-types)
    	     * of `Object`. (e.g. arrays, functions, objects, regexes, `new Number(0)`, and `new String('')`)
    	     *
    	     * @static
    	     * @memberOf _
    	     * @since 0.1.0
    	     * @category Lang
    	     * @param {*} value The value to check.
    	     * @returns {boolean} Returns `true` if `value` is an object, else `false`.
    	     * @example
    	     *
    	     * _.isObject({});
    	     * // => true
    	     *
    	     * _.isObject([1, 2, 3]);
    	     * // => true
    	     *
    	     * _.isObject(_.noop);
    	     * // => true
    	     *
    	     * _.isObject(null);
    	     * // => false
    	     */function isObject(value){var type=typeof value;return value!=null&&(type=='object'||type=='function');}/**
    	     * Checks if `value` is object-like. A value is object-like if it's not `null`
    	     * and has a `typeof` result of "object".
    	     *
    	     * @static
    	     * @memberOf _
    	     * @since 4.0.0
    	     * @category Lang
    	     * @param {*} value The value to check.
    	     * @returns {boolean} Returns `true` if `value` is object-like, else `false`.
    	     * @example
    	     *
    	     * _.isObjectLike({});
    	     * // => true
    	     *
    	     * _.isObjectLike([1, 2, 3]);
    	     * // => true
    	     *
    	     * _.isObjectLike(_.noop);
    	     * // => false
    	     *
    	     * _.isObjectLike(null);
    	     * // => false
    	     */function isObjectLike(value){return value!=null&&typeof value=='object';}/**
    	     * Checks if `value` is classified as a `Map` object.
    	     *
    	     * @static
    	     * @memberOf _
    	     * @since 4.3.0
    	     * @category Lang
    	     * @param {*} value The value to check.
    	     * @returns {boolean} Returns `true` if `value` is a map, else `false`.
    	     * @example
    	     *
    	     * _.isMap(new Map);
    	     * // => true
    	     *
    	     * _.isMap(new WeakMap);
    	     * // => false
    	     */var isMap=nodeIsMap?baseUnary(nodeIsMap):baseIsMap;/**
    	     * Performs a partial deep comparison between `object` and `source` to
    	     * determine if `object` contains equivalent property values.
    	     *
    	     * **Note:** This method is equivalent to `_.matches` when `source` is
    	     * partially applied.
    	     *
    	     * Partial comparisons will match empty array and empty object `source`
    	     * values against any array or object value, respectively. See `_.isEqual`
    	     * for a list of supported value comparisons.
    	     *
    	     * @static
    	     * @memberOf _
    	     * @since 3.0.0
    	     * @category Lang
    	     * @param {Object} object The object to inspect.
    	     * @param {Object} source The object of property values to match.
    	     * @returns {boolean} Returns `true` if `object` is a match, else `false`.
    	     * @example
    	     *
    	     * var object = { 'a': 1, 'b': 2 };
    	     *
    	     * _.isMatch(object, { 'b': 2 });
    	     * // => true
    	     *
    	     * _.isMatch(object, { 'b': 1 });
    	     * // => false
    	     */function isMatch(object,source){return object===source||baseIsMatch(object,source,getMatchData(source));}/**
    	     * This method is like `_.isMatch` except that it accepts `customizer` which
    	     * is invoked to compare values. If `customizer` returns `undefined`, comparisons
    	     * are handled by the method instead. The `customizer` is invoked with five
    	     * arguments: (objValue, srcValue, index|key, object, source).
    	     *
    	     * @static
    	     * @memberOf _
    	     * @since 4.0.0
    	     * @category Lang
    	     * @param {Object} object The object to inspect.
    	     * @param {Object} source The object of property values to match.
    	     * @param {Function} [customizer] The function to customize comparisons.
    	     * @returns {boolean} Returns `true` if `object` is a match, else `false`.
    	     * @example
    	     *
    	     * function isGreeting(value) {
    	     *   return /^h(?:i|ello)$/.test(value);
    	     * }
    	     *
    	     * function customizer(objValue, srcValue) {
    	     *   if (isGreeting(objValue) && isGreeting(srcValue)) {
    	     *     return true;
    	     *   }
    	     * }
    	     *
    	     * var object = { 'greeting': 'hello' };
    	     * var source = { 'greeting': 'hi' };
    	     *
    	     * _.isMatchWith(object, source, customizer);
    	     * // => true
    	     */function isMatchWith(object,source,customizer){customizer=typeof customizer=='function'?customizer:undefined$1;return baseIsMatch(object,source,getMatchData(source),customizer);}/**
    	     * Checks if `value` is `NaN`.
    	     *
    	     * **Note:** This method is based on
    	     * [`Number.isNaN`](https://mdn.io/Number/isNaN) and is not the same as
    	     * global [`isNaN`](https://mdn.io/isNaN) which returns `true` for
    	     * `undefined` and other non-number values.
    	     *
    	     * @static
    	     * @memberOf _
    	     * @since 0.1.0
    	     * @category Lang
    	     * @param {*} value The value to check.
    	     * @returns {boolean} Returns `true` if `value` is `NaN`, else `false`.
    	     * @example
    	     *
    	     * _.isNaN(NaN);
    	     * // => true
    	     *
    	     * _.isNaN(new Number(NaN));
    	     * // => true
    	     *
    	     * isNaN(undefined);
    	     * // => true
    	     *
    	     * _.isNaN(undefined);
    	     * // => false
    	     */function isNaN(value){// An `NaN` primitive is the only value that is not equal to itself.
    	// Perform the `toStringTag` check first to avoid errors with some
    	// ActiveX objects in IE.
    	return isNumber(value)&&value!=+value;}/**
    	     * Checks if `value` is a pristine native function.
    	     *
    	     * **Note:** This method can't reliably detect native functions in the presence
    	     * of the core-js package because core-js circumvents this kind of detection.
    	     * Despite multiple requests, the core-js maintainer has made it clear: any
    	     * attempt to fix the detection will be obstructed. As a result, we're left
    	     * with little choice but to throw an error. Unfortunately, this also affects
    	     * packages, like [babel-polyfill](https://www.npmjs.com/package/babel-polyfill),
    	     * which rely on core-js.
    	     *
    	     * @static
    	     * @memberOf _
    	     * @since 3.0.0
    	     * @category Lang
    	     * @param {*} value The value to check.
    	     * @returns {boolean} Returns `true` if `value` is a native function,
    	     *  else `false`.
    	     * @example
    	     *
    	     * _.isNative(Array.prototype.push);
    	     * // => true
    	     *
    	     * _.isNative(_);
    	     * // => false
    	     */function isNative(value){if(isMaskable(value)){throw new Error(CORE_ERROR_TEXT);}return baseIsNative(value);}/**
    	     * Checks if `value` is `null`.
    	     *
    	     * @static
    	     * @memberOf _
    	     * @since 0.1.0
    	     * @category Lang
    	     * @param {*} value The value to check.
    	     * @returns {boolean} Returns `true` if `value` is `null`, else `false`.
    	     * @example
    	     *
    	     * _.isNull(null);
    	     * // => true
    	     *
    	     * _.isNull(void 0);
    	     * // => false
    	     */function isNull(value){return value===null;}/**
    	     * Checks if `value` is `null` or `undefined`.
    	     *
    	     * @static
    	     * @memberOf _
    	     * @since 4.0.0
    	     * @category Lang
    	     * @param {*} value The value to check.
    	     * @returns {boolean} Returns `true` if `value` is nullish, else `false`.
    	     * @example
    	     *
    	     * _.isNil(null);
    	     * // => true
    	     *
    	     * _.isNil(void 0);
    	     * // => true
    	     *
    	     * _.isNil(NaN);
    	     * // => false
    	     */function isNil(value){return value==null;}/**
    	     * Checks if `value` is classified as a `Number` primitive or object.
    	     *
    	     * **Note:** To exclude `Infinity`, `-Infinity`, and `NaN`, which are
    	     * classified as numbers, use the `_.isFinite` method.
    	     *
    	     * @static
    	     * @memberOf _
    	     * @since 0.1.0
    	     * @category Lang
    	     * @param {*} value The value to check.
    	     * @returns {boolean} Returns `true` if `value` is a number, else `false`.
    	     * @example
    	     *
    	     * _.isNumber(3);
    	     * // => true
    	     *
    	     * _.isNumber(Number.MIN_VALUE);
    	     * // => true
    	     *
    	     * _.isNumber(Infinity);
    	     * // => true
    	     *
    	     * _.isNumber('3');
    	     * // => false
    	     */function isNumber(value){return typeof value=='number'||isObjectLike(value)&&baseGetTag(value)==numberTag;}/**
    	     * Checks if `value` is a plain object, that is, an object created by the
    	     * `Object` constructor or one with a `[[Prototype]]` of `null`.
    	     *
    	     * @static
    	     * @memberOf _
    	     * @since 0.8.0
    	     * @category Lang
    	     * @param {*} value The value to check.
    	     * @returns {boolean} Returns `true` if `value` is a plain object, else `false`.
    	     * @example
    	     *
    	     * function Foo() {
    	     *   this.a = 1;
    	     * }
    	     *
    	     * _.isPlainObject(new Foo);
    	     * // => false
    	     *
    	     * _.isPlainObject([1, 2, 3]);
    	     * // => false
    	     *
    	     * _.isPlainObject({ 'x': 0, 'y': 0 });
    	     * // => true
    	     *
    	     * _.isPlainObject(Object.create(null));
    	     * // => true
    	     */function isPlainObject(value){if(!isObjectLike(value)||baseGetTag(value)!=objectTag){return false;}var proto=getPrototype(value);if(proto===null){return true;}var Ctor=hasOwnProperty.call(proto,'constructor')&&proto.constructor;return typeof Ctor=='function'&&Ctor instanceof Ctor&&funcToString.call(Ctor)==objectCtorString;}/**
    	     * Checks if `value` is classified as a `RegExp` object.
    	     *
    	     * @static
    	     * @memberOf _
    	     * @since 0.1.0
    	     * @category Lang
    	     * @param {*} value The value to check.
    	     * @returns {boolean} Returns `true` if `value` is a regexp, else `false`.
    	     * @example
    	     *
    	     * _.isRegExp(/abc/);
    	     * // => true
    	     *
    	     * _.isRegExp('/abc/');
    	     * // => false
    	     */var isRegExp=nodeIsRegExp?baseUnary(nodeIsRegExp):baseIsRegExp;/**
    	     * Checks if `value` is a safe integer. An integer is safe if it's an IEEE-754
    	     * double precision number which isn't the result of a rounded unsafe integer.
    	     *
    	     * **Note:** This method is based on
    	     * [`Number.isSafeInteger`](https://mdn.io/Number/isSafeInteger).
    	     *
    	     * @static
    	     * @memberOf _
    	     * @since 4.0.0
    	     * @category Lang
    	     * @param {*} value The value to check.
    	     * @returns {boolean} Returns `true` if `value` is a safe integer, else `false`.
    	     * @example
    	     *
    	     * _.isSafeInteger(3);
    	     * // => true
    	     *
    	     * _.isSafeInteger(Number.MIN_VALUE);
    	     * // => false
    	     *
    	     * _.isSafeInteger(Infinity);
    	     * // => false
    	     *
    	     * _.isSafeInteger('3');
    	     * // => false
    	     */function isSafeInteger(value){return isInteger(value)&&value>=-MAX_SAFE_INTEGER&&value<=MAX_SAFE_INTEGER;}/**
    	     * Checks if `value` is classified as a `Set` object.
    	     *
    	     * @static
    	     * @memberOf _
    	     * @since 4.3.0
    	     * @category Lang
    	     * @param {*} value The value to check.
    	     * @returns {boolean} Returns `true` if `value` is a set, else `false`.
    	     * @example
    	     *
    	     * _.isSet(new Set);
    	     * // => true
    	     *
    	     * _.isSet(new WeakSet);
    	     * // => false
    	     */var isSet=nodeIsSet?baseUnary(nodeIsSet):baseIsSet;/**
    	     * Checks if `value` is classified as a `String` primitive or object.
    	     *
    	     * @static
    	     * @since 0.1.0
    	     * @memberOf _
    	     * @category Lang
    	     * @param {*} value The value to check.
    	     * @returns {boolean} Returns `true` if `value` is a string, else `false`.
    	     * @example
    	     *
    	     * _.isString('abc');
    	     * // => true
    	     *
    	     * _.isString(1);
    	     * // => false
    	     */function isString(value){return typeof value=='string'||!isArray(value)&&isObjectLike(value)&&baseGetTag(value)==stringTag;}/**
    	     * Checks if `value` is classified as a `Symbol` primitive or object.
    	     *
    	     * @static
    	     * @memberOf _
    	     * @since 4.0.0
    	     * @category Lang
    	     * @param {*} value The value to check.
    	     * @returns {boolean} Returns `true` if `value` is a symbol, else `false`.
    	     * @example
    	     *
    	     * _.isSymbol(Symbol.iterator);
    	     * // => true
    	     *
    	     * _.isSymbol('abc');
    	     * // => false
    	     */function isSymbol(value){return typeof value=='symbol'||isObjectLike(value)&&baseGetTag(value)==symbolTag;}/**
    	     * Checks if `value` is classified as a typed array.
    	     *
    	     * @static
    	     * @memberOf _
    	     * @since 3.0.0
    	     * @category Lang
    	     * @param {*} value The value to check.
    	     * @returns {boolean} Returns `true` if `value` is a typed array, else `false`.
    	     * @example
    	     *
    	     * _.isTypedArray(new Uint8Array);
    	     * // => true
    	     *
    	     * _.isTypedArray([]);
    	     * // => false
    	     */var isTypedArray=nodeIsTypedArray?baseUnary(nodeIsTypedArray):baseIsTypedArray;/**
    	     * Checks if `value` is `undefined`.
    	     *
    	     * @static
    	     * @since 0.1.0
    	     * @memberOf _
    	     * @category Lang
    	     * @param {*} value The value to check.
    	     * @returns {boolean} Returns `true` if `value` is `undefined`, else `false`.
    	     * @example
    	     *
    	     * _.isUndefined(void 0);
    	     * // => true
    	     *
    	     * _.isUndefined(null);
    	     * // => false
    	     */function isUndefined(value){return value===undefined$1;}/**
    	     * Checks if `value` is classified as a `WeakMap` object.
    	     *
    	     * @static
    	     * @memberOf _
    	     * @since 4.3.0
    	     * @category Lang
    	     * @param {*} value The value to check.
    	     * @returns {boolean} Returns `true` if `value` is a weak map, else `false`.
    	     * @example
    	     *
    	     * _.isWeakMap(new WeakMap);
    	     * // => true
    	     *
    	     * _.isWeakMap(new Map);
    	     * // => false
    	     */function isWeakMap(value){return isObjectLike(value)&&getTag(value)==weakMapTag;}/**
    	     * Checks if `value` is classified as a `WeakSet` object.
    	     *
    	     * @static
    	     * @memberOf _
    	     * @since 4.3.0
    	     * @category Lang
    	     * @param {*} value The value to check.
    	     * @returns {boolean} Returns `true` if `value` is a weak set, else `false`.
    	     * @example
    	     *
    	     * _.isWeakSet(new WeakSet);
    	     * // => true
    	     *
    	     * _.isWeakSet(new Set);
    	     * // => false
    	     */function isWeakSet(value){return isObjectLike(value)&&baseGetTag(value)==weakSetTag;}/**
    	     * Checks if `value` is less than `other`.
    	     *
    	     * @static
    	     * @memberOf _
    	     * @since 3.9.0
    	     * @category Lang
    	     * @param {*} value The value to compare.
    	     * @param {*} other The other value to compare.
    	     * @returns {boolean} Returns `true` if `value` is less than `other`,
    	     *  else `false`.
    	     * @see _.gt
    	     * @example
    	     *
    	     * _.lt(1, 3);
    	     * // => true
    	     *
    	     * _.lt(3, 3);
    	     * // => false
    	     *
    	     * _.lt(3, 1);
    	     * // => false
    	     */var lt=createRelationalOperation(baseLt);/**
    	     * Checks if `value` is less than or equal to `other`.
    	     *
    	     * @static
    	     * @memberOf _
    	     * @since 3.9.0
    	     * @category Lang
    	     * @param {*} value The value to compare.
    	     * @param {*} other The other value to compare.
    	     * @returns {boolean} Returns `true` if `value` is less than or equal to
    	     *  `other`, else `false`.
    	     * @see _.gte
    	     * @example
    	     *
    	     * _.lte(1, 3);
    	     * // => true
    	     *
    	     * _.lte(3, 3);
    	     * // => true
    	     *
    	     * _.lte(3, 1);
    	     * // => false
    	     */var lte=createRelationalOperation(function(value,other){return value<=other;});/**
    	     * Converts `value` to an array.
    	     *
    	     * @static
    	     * @since 0.1.0
    	     * @memberOf _
    	     * @category Lang
    	     * @param {*} value The value to convert.
    	     * @returns {Array} Returns the converted array.
    	     * @example
    	     *
    	     * _.toArray({ 'a': 1, 'b': 2 });
    	     * // => [1, 2]
    	     *
    	     * _.toArray('abc');
    	     * // => ['a', 'b', 'c']
    	     *
    	     * _.toArray(1);
    	     * // => []
    	     *
    	     * _.toArray(null);
    	     * // => []
    	     */function toArray(value){if(!value){return [];}if(isArrayLike(value)){return isString(value)?stringToArray(value):copyArray(value);}if(symIterator&&value[symIterator]){return iteratorToArray(value[symIterator]());}var tag=getTag(value),func=tag==mapTag?mapToArray:tag==setTag?setToArray:values;return func(value);}/**
    	     * Converts `value` to a finite number.
    	     *
    	     * @static
    	     * @memberOf _
    	     * @since 4.12.0
    	     * @category Lang
    	     * @param {*} value The value to convert.
    	     * @returns {number} Returns the converted number.
    	     * @example
    	     *
    	     * _.toFinite(3.2);
    	     * // => 3.2
    	     *
    	     * _.toFinite(Number.MIN_VALUE);
    	     * // => 5e-324
    	     *
    	     * _.toFinite(Infinity);
    	     * // => 1.7976931348623157e+308
    	     *
    	     * _.toFinite('3.2');
    	     * // => 3.2
    	     */function toFinite(value){if(!value){return value===0?value:0;}value=toNumber(value);if(value===INFINITY||value===-INFINITY){var sign=value<0?-1:1;return sign*MAX_INTEGER;}return value===value?value:0;}/**
    	     * Converts `value` to an integer.
    	     *
    	     * **Note:** This method is loosely based on
    	     * [`ToInteger`](http://www.ecma-international.org/ecma-262/7.0/#sec-tointeger).
    	     *
    	     * @static
    	     * @memberOf _
    	     * @since 4.0.0
    	     * @category Lang
    	     * @param {*} value The value to convert.
    	     * @returns {number} Returns the converted integer.
    	     * @example
    	     *
    	     * _.toInteger(3.2);
    	     * // => 3
    	     *
    	     * _.toInteger(Number.MIN_VALUE);
    	     * // => 0
    	     *
    	     * _.toInteger(Infinity);
    	     * // => 1.7976931348623157e+308
    	     *
    	     * _.toInteger('3.2');
    	     * // => 3
    	     */function toInteger(value){var result=toFinite(value),remainder=result%1;return result===result?remainder?result-remainder:result:0;}/**
    	     * Converts `value` to an integer suitable for use as the length of an
    	     * array-like object.
    	     *
    	     * **Note:** This method is based on
    	     * [`ToLength`](http://ecma-international.org/ecma-262/7.0/#sec-tolength).
    	     *
    	     * @static
    	     * @memberOf _
    	     * @since 4.0.0
    	     * @category Lang
    	     * @param {*} value The value to convert.
    	     * @returns {number} Returns the converted integer.
    	     * @example
    	     *
    	     * _.toLength(3.2);
    	     * // => 3
    	     *
    	     * _.toLength(Number.MIN_VALUE);
    	     * // => 0
    	     *
    	     * _.toLength(Infinity);
    	     * // => 4294967295
    	     *
    	     * _.toLength('3.2');
    	     * // => 3
    	     */function toLength(value){return value?baseClamp(toInteger(value),0,MAX_ARRAY_LENGTH):0;}/**
    	     * Converts `value` to a number.
    	     *
    	     * @static
    	     * @memberOf _
    	     * @since 4.0.0
    	     * @category Lang
    	     * @param {*} value The value to process.
    	     * @returns {number} Returns the number.
    	     * @example
    	     *
    	     * _.toNumber(3.2);
    	     * // => 3.2
    	     *
    	     * _.toNumber(Number.MIN_VALUE);
    	     * // => 5e-324
    	     *
    	     * _.toNumber(Infinity);
    	     * // => Infinity
    	     *
    	     * _.toNumber('3.2');
    	     * // => 3.2
    	     */function toNumber(value){if(typeof value=='number'){return value;}if(isSymbol(value)){return NAN;}if(isObject(value)){var other=typeof value.valueOf=='function'?value.valueOf():value;value=isObject(other)?other+'':other;}if(typeof value!='string'){return value===0?value:+value;}value=baseTrim(value);var isBinary=reIsBinary.test(value);return isBinary||reIsOctal.test(value)?freeParseInt(value.slice(2),isBinary?2:8):reIsBadHex.test(value)?NAN:+value;}/**
    	     * Converts `value` to a plain object flattening inherited enumerable string
    	     * keyed properties of `value` to own properties of the plain object.
    	     *
    	     * @static
    	     * @memberOf _
    	     * @since 3.0.0
    	     * @category Lang
    	     * @param {*} value The value to convert.
    	     * @returns {Object} Returns the converted plain object.
    	     * @example
    	     *
    	     * function Foo() {
    	     *   this.b = 2;
    	     * }
    	     *
    	     * Foo.prototype.c = 3;
    	     *
    	     * _.assign({ 'a': 1 }, new Foo);
    	     * // => { 'a': 1, 'b': 2 }
    	     *
    	     * _.assign({ 'a': 1 }, _.toPlainObject(new Foo));
    	     * // => { 'a': 1, 'b': 2, 'c': 3 }
    	     */function toPlainObject(value){return copyObject(value,keysIn(value));}/**
    	     * Converts `value` to a safe integer. A safe integer can be compared and
    	     * represented correctly.
    	     *
    	     * @static
    	     * @memberOf _
    	     * @since 4.0.0
    	     * @category Lang
    	     * @param {*} value The value to convert.
    	     * @returns {number} Returns the converted integer.
    	     * @example
    	     *
    	     * _.toSafeInteger(3.2);
    	     * // => 3
    	     *
    	     * _.toSafeInteger(Number.MIN_VALUE);
    	     * // => 0
    	     *
    	     * _.toSafeInteger(Infinity);
    	     * // => 9007199254740991
    	     *
    	     * _.toSafeInteger('3.2');
    	     * // => 3
    	     */function toSafeInteger(value){return value?baseClamp(toInteger(value),-MAX_SAFE_INTEGER,MAX_SAFE_INTEGER):value===0?value:0;}/**
    	     * Converts `value` to a string. An empty string is returned for `null`
    	     * and `undefined` values. The sign of `-0` is preserved.
    	     *
    	     * @static
    	     * @memberOf _
    	     * @since 4.0.0
    	     * @category Lang
    	     * @param {*} value The value to convert.
    	     * @returns {string} Returns the converted string.
    	     * @example
    	     *
    	     * _.toString(null);
    	     * // => ''
    	     *
    	     * _.toString(-0);
    	     * // => '-0'
    	     *
    	     * _.toString([1, 2, 3]);
    	     * // => '1,2,3'
    	     */function toString(value){return value==null?'':baseToString(value);}/*------------------------------------------------------------------------*//**
    	     * Assigns own enumerable string keyed properties of source objects to the
    	     * destination object. Source objects are applied from left to right.
    	     * Subsequent sources overwrite property assignments of previous sources.
    	     *
    	     * **Note:** This method mutates `object` and is loosely based on
    	     * [`Object.assign`](https://mdn.io/Object/assign).
    	     *
    	     * @static
    	     * @memberOf _
    	     * @since 0.10.0
    	     * @category Object
    	     * @param {Object} object The destination object.
    	     * @param {...Object} [sources] The source objects.
    	     * @returns {Object} Returns `object`.
    	     * @see _.assignIn
    	     * @example
    	     *
    	     * function Foo() {
    	     *   this.a = 1;
    	     * }
    	     *
    	     * function Bar() {
    	     *   this.c = 3;
    	     * }
    	     *
    	     * Foo.prototype.b = 2;
    	     * Bar.prototype.d = 4;
    	     *
    	     * _.assign({ 'a': 0 }, new Foo, new Bar);
    	     * // => { 'a': 1, 'c': 3 }
    	     */var assign=createAssigner(function(object,source){if(isPrototype(source)||isArrayLike(source)){copyObject(source,keys(source),object);return;}for(var key in source){if(hasOwnProperty.call(source,key)){assignValue(object,key,source[key]);}}});/**
    	     * This method is like `_.assign` except that it iterates over own and
    	     * inherited source properties.
    	     *
    	     * **Note:** This method mutates `object`.
    	     *
    	     * @static
    	     * @memberOf _
    	     * @since 4.0.0
    	     * @alias extend
    	     * @category Object
    	     * @param {Object} object The destination object.
    	     * @param {...Object} [sources] The source objects.
    	     * @returns {Object} Returns `object`.
    	     * @see _.assign
    	     * @example
    	     *
    	     * function Foo() {
    	     *   this.a = 1;
    	     * }
    	     *
    	     * function Bar() {
    	     *   this.c = 3;
    	     * }
    	     *
    	     * Foo.prototype.b = 2;
    	     * Bar.prototype.d = 4;
    	     *
    	     * _.assignIn({ 'a': 0 }, new Foo, new Bar);
    	     * // => { 'a': 1, 'b': 2, 'c': 3, 'd': 4 }
    	     */var assignIn=createAssigner(function(object,source){copyObject(source,keysIn(source),object);});/**
    	     * This method is like `_.assignIn` except that it accepts `customizer`
    	     * which is invoked to produce the assigned values. If `customizer` returns
    	     * `undefined`, assignment is handled by the method instead. The `customizer`
    	     * is invoked with five arguments: (objValue, srcValue, key, object, source).
    	     *
    	     * **Note:** This method mutates `object`.
    	     *
    	     * @static
    	     * @memberOf _
    	     * @since 4.0.0
    	     * @alias extendWith
    	     * @category Object
    	     * @param {Object} object The destination object.
    	     * @param {...Object} sources The source objects.
    	     * @param {Function} [customizer] The function to customize assigned values.
    	     * @returns {Object} Returns `object`.
    	     * @see _.assignWith
    	     * @example
    	     *
    	     * function customizer(objValue, srcValue) {
    	     *   return _.isUndefined(objValue) ? srcValue : objValue;
    	     * }
    	     *
    	     * var defaults = _.partialRight(_.assignInWith, customizer);
    	     *
    	     * defaults({ 'a': 1 }, { 'b': 2 }, { 'a': 3 });
    	     * // => { 'a': 1, 'b': 2 }
    	     */var assignInWith=createAssigner(function(object,source,srcIndex,customizer){copyObject(source,keysIn(source),object,customizer);});/**
    	     * This method is like `_.assign` except that it accepts `customizer`
    	     * which is invoked to produce the assigned values. If `customizer` returns
    	     * `undefined`, assignment is handled by the method instead. The `customizer`
    	     * is invoked with five arguments: (objValue, srcValue, key, object, source).
    	     *
    	     * **Note:** This method mutates `object`.
    	     *
    	     * @static
    	     * @memberOf _
    	     * @since 4.0.0
    	     * @category Object
    	     * @param {Object} object The destination object.
    	     * @param {...Object} sources The source objects.
    	     * @param {Function} [customizer] The function to customize assigned values.
    	     * @returns {Object} Returns `object`.
    	     * @see _.assignInWith
    	     * @example
    	     *
    	     * function customizer(objValue, srcValue) {
    	     *   return _.isUndefined(objValue) ? srcValue : objValue;
    	     * }
    	     *
    	     * var defaults = _.partialRight(_.assignWith, customizer);
    	     *
    	     * defaults({ 'a': 1 }, { 'b': 2 }, { 'a': 3 });
    	     * // => { 'a': 1, 'b': 2 }
    	     */var assignWith=createAssigner(function(object,source,srcIndex,customizer){copyObject(source,keys(source),object,customizer);});/**
    	     * Creates an array of values corresponding to `paths` of `object`.
    	     *
    	     * @static
    	     * @memberOf _
    	     * @since 1.0.0
    	     * @category Object
    	     * @param {Object} object The object to iterate over.
    	     * @param {...(string|string[])} [paths] The property paths to pick.
    	     * @returns {Array} Returns the picked values.
    	     * @example
    	     *
    	     * var object = { 'a': [{ 'b': { 'c': 3 } }, 4] };
    	     *
    	     * _.at(object, ['a[0].b.c', 'a[1]']);
    	     * // => [3, 4]
    	     */var at=flatRest(baseAt);/**
    	     * Creates an object that inherits from the `prototype` object. If a
    	     * `properties` object is given, its own enumerable string keyed properties
    	     * are assigned to the created object.
    	     *
    	     * @static
    	     * @memberOf _
    	     * @since 2.3.0
    	     * @category Object
    	     * @param {Object} prototype The object to inherit from.
    	     * @param {Object} [properties] The properties to assign to the object.
    	     * @returns {Object} Returns the new object.
    	     * @example
    	     *
    	     * function Shape() {
    	     *   this.x = 0;
    	     *   this.y = 0;
    	     * }
    	     *
    	     * function Circle() {
    	     *   Shape.call(this);
    	     * }
    	     *
    	     * Circle.prototype = _.create(Shape.prototype, {
    	     *   'constructor': Circle
    	     * });
    	     *
    	     * var circle = new Circle;
    	     * circle instanceof Circle;
    	     * // => true
    	     *
    	     * circle instanceof Shape;
    	     * // => true
    	     */function create(prototype,properties){var result=baseCreate(prototype);return properties==null?result:baseAssign(result,properties);}/**
    	     * Assigns own and inherited enumerable string keyed properties of source
    	     * objects to the destination object for all destination properties that
    	     * resolve to `undefined`. Source objects are applied from left to right.
    	     * Once a property is set, additional values of the same property are ignored.
    	     *
    	     * **Note:** This method mutates `object`.
    	     *
    	     * @static
    	     * @since 0.1.0
    	     * @memberOf _
    	     * @category Object
    	     * @param {Object} object The destination object.
    	     * @param {...Object} [sources] The source objects.
    	     * @returns {Object} Returns `object`.
    	     * @see _.defaultsDeep
    	     * @example
    	     *
    	     * _.defaults({ 'a': 1 }, { 'b': 2 }, { 'a': 3 });
    	     * // => { 'a': 1, 'b': 2 }
    	     */var defaults=baseRest(function(object,sources){object=Object(object);var index=-1;var length=sources.length;var guard=length>2?sources[2]:undefined$1;if(guard&&isIterateeCall(sources[0],sources[1],guard)){length=1;}while(++index<length){var source=sources[index];var props=keysIn(source);var propsIndex=-1;var propsLength=props.length;while(++propsIndex<propsLength){var key=props[propsIndex];var value=object[key];if(value===undefined$1||eq(value,objectProto[key])&&!hasOwnProperty.call(object,key)){object[key]=source[key];}}}return object;});/**
    	     * This method is like `_.defaults` except that it recursively assigns
    	     * default properties.
    	     *
    	     * **Note:** This method mutates `object`.
    	     *
    	     * @static
    	     * @memberOf _
    	     * @since 3.10.0
    	     * @category Object
    	     * @param {Object} object The destination object.
    	     * @param {...Object} [sources] The source objects.
    	     * @returns {Object} Returns `object`.
    	     * @see _.defaults
    	     * @example
    	     *
    	     * _.defaultsDeep({ 'a': { 'b': 2 } }, { 'a': { 'b': 1, 'c': 3 } });
    	     * // => { 'a': { 'b': 2, 'c': 3 } }
    	     */var defaultsDeep=baseRest(function(args){args.push(undefined$1,customDefaultsMerge);return apply(mergeWith,undefined$1,args);});/**
    	     * This method is like `_.find` except that it returns the key of the first
    	     * element `predicate` returns truthy for instead of the element itself.
    	     *
    	     * @static
    	     * @memberOf _
    	     * @since 1.1.0
    	     * @category Object
    	     * @param {Object} object The object to inspect.
    	     * @param {Function} [predicate=_.identity] The function invoked per iteration.
    	     * @returns {string|undefined} Returns the key of the matched element,
    	     *  else `undefined`.
    	     * @example
    	     *
    	     * var users = {
    	     *   'barney':  { 'age': 36, 'active': true },
    	     *   'fred':    { 'age': 40, 'active': false },
    	     *   'pebbles': { 'age': 1,  'active': true }
    	     * };
    	     *
    	     * _.findKey(users, function(o) { return o.age < 40; });
    	     * // => 'barney' (iteration order is not guaranteed)
    	     *
    	     * // The `_.matches` iteratee shorthand.
    	     * _.findKey(users, { 'age': 1, 'active': true });
    	     * // => 'pebbles'
    	     *
    	     * // The `_.matchesProperty` iteratee shorthand.
    	     * _.findKey(users, ['active', false]);
    	     * // => 'fred'
    	     *
    	     * // The `_.property` iteratee shorthand.
    	     * _.findKey(users, 'active');
    	     * // => 'barney'
    	     */function findKey(object,predicate){return baseFindKey(object,getIteratee(predicate,3),baseForOwn);}/**
    	     * This method is like `_.findKey` except that it iterates over elements of
    	     * a collection in the opposite order.
    	     *
    	     * @static
    	     * @memberOf _
    	     * @since 2.0.0
    	     * @category Object
    	     * @param {Object} object The object to inspect.
    	     * @param {Function} [predicate=_.identity] The function invoked per iteration.
    	     * @returns {string|undefined} Returns the key of the matched element,
    	     *  else `undefined`.
    	     * @example
    	     *
    	     * var users = {
    	     *   'barney':  { 'age': 36, 'active': true },
    	     *   'fred':    { 'age': 40, 'active': false },
    	     *   'pebbles': { 'age': 1,  'active': true }
    	     * };
    	     *
    	     * _.findLastKey(users, function(o) { return o.age < 40; });
    	     * // => returns 'pebbles' assuming `_.findKey` returns 'barney'
    	     *
    	     * // The `_.matches` iteratee shorthand.
    	     * _.findLastKey(users, { 'age': 36, 'active': true });
    	     * // => 'barney'
    	     *
    	     * // The `_.matchesProperty` iteratee shorthand.
    	     * _.findLastKey(users, ['active', false]);
    	     * // => 'fred'
    	     *
    	     * // The `_.property` iteratee shorthand.
    	     * _.findLastKey(users, 'active');
    	     * // => 'pebbles'
    	     */function findLastKey(object,predicate){return baseFindKey(object,getIteratee(predicate,3),baseForOwnRight);}/**
    	     * Iterates over own and inherited enumerable string keyed properties of an
    	     * object and invokes `iteratee` for each property. The iteratee is invoked
    	     * with three arguments: (value, key, object). Iteratee functions may exit
    	     * iteration early by explicitly returning `false`.
    	     *
    	     * @static
    	     * @memberOf _
    	     * @since 0.3.0
    	     * @category Object
    	     * @param {Object} object The object to iterate over.
    	     * @param {Function} [iteratee=_.identity] The function invoked per iteration.
    	     * @returns {Object} Returns `object`.
    	     * @see _.forInRight
    	     * @example
    	     *
    	     * function Foo() {
    	     *   this.a = 1;
    	     *   this.b = 2;
    	     * }
    	     *
    	     * Foo.prototype.c = 3;
    	     *
    	     * _.forIn(new Foo, function(value, key) {
    	     *   console.log(key);
    	     * });
    	     * // => Logs 'a', 'b', then 'c' (iteration order is not guaranteed).
    	     */function forIn(object,iteratee){return object==null?object:baseFor(object,getIteratee(iteratee,3),keysIn);}/**
    	     * This method is like `_.forIn` except that it iterates over properties of
    	     * `object` in the opposite order.
    	     *
    	     * @static
    	     * @memberOf _
    	     * @since 2.0.0
    	     * @category Object
    	     * @param {Object} object The object to iterate over.
    	     * @param {Function} [iteratee=_.identity] The function invoked per iteration.
    	     * @returns {Object} Returns `object`.
    	     * @see _.forIn
    	     * @example
    	     *
    	     * function Foo() {
    	     *   this.a = 1;
    	     *   this.b = 2;
    	     * }
    	     *
    	     * Foo.prototype.c = 3;
    	     *
    	     * _.forInRight(new Foo, function(value, key) {
    	     *   console.log(key);
    	     * });
    	     * // => Logs 'c', 'b', then 'a' assuming `_.forIn` logs 'a', 'b', then 'c'.
    	     */function forInRight(object,iteratee){return object==null?object:baseForRight(object,getIteratee(iteratee,3),keysIn);}/**
    	     * Iterates over own enumerable string keyed properties of an object and
    	     * invokes `iteratee` for each property. The iteratee is invoked with three
    	     * arguments: (value, key, object). Iteratee functions may exit iteration
    	     * early by explicitly returning `false`.
    	     *
    	     * @static
    	     * @memberOf _
    	     * @since 0.3.0
    	     * @category Object
    	     * @param {Object} object The object to iterate over.
    	     * @param {Function} [iteratee=_.identity] The function invoked per iteration.
    	     * @returns {Object} Returns `object`.
    	     * @see _.forOwnRight
    	     * @example
    	     *
    	     * function Foo() {
    	     *   this.a = 1;
    	     *   this.b = 2;
    	     * }
    	     *
    	     * Foo.prototype.c = 3;
    	     *
    	     * _.forOwn(new Foo, function(value, key) {
    	     *   console.log(key);
    	     * });
    	     * // => Logs 'a' then 'b' (iteration order is not guaranteed).
    	     */function forOwn(object,iteratee){return object&&baseForOwn(object,getIteratee(iteratee,3));}/**
    	     * This method is like `_.forOwn` except that it iterates over properties of
    	     * `object` in the opposite order.
    	     *
    	     * @static
    	     * @memberOf _
    	     * @since 2.0.0
    	     * @category Object
    	     * @param {Object} object The object to iterate over.
    	     * @param {Function} [iteratee=_.identity] The function invoked per iteration.
    	     * @returns {Object} Returns `object`.
    	     * @see _.forOwn
    	     * @example
    	     *
    	     * function Foo() {
    	     *   this.a = 1;
    	     *   this.b = 2;
    	     * }
    	     *
    	     * Foo.prototype.c = 3;
    	     *
    	     * _.forOwnRight(new Foo, function(value, key) {
    	     *   console.log(key);
    	     * });
    	     * // => Logs 'b' then 'a' assuming `_.forOwn` logs 'a' then 'b'.
    	     */function forOwnRight(object,iteratee){return object&&baseForOwnRight(object,getIteratee(iteratee,3));}/**
    	     * Creates an array of function property names from own enumerable properties
    	     * of `object`.
    	     *
    	     * @static
    	     * @since 0.1.0
    	     * @memberOf _
    	     * @category Object
    	     * @param {Object} object The object to inspect.
    	     * @returns {Array} Returns the function names.
    	     * @see _.functionsIn
    	     * @example
    	     *
    	     * function Foo() {
    	     *   this.a = _.constant('a');
    	     *   this.b = _.constant('b');
    	     * }
    	     *
    	     * Foo.prototype.c = _.constant('c');
    	     *
    	     * _.functions(new Foo);
    	     * // => ['a', 'b']
    	     */function functions(object){return object==null?[]:baseFunctions(object,keys(object));}/**
    	     * Creates an array of function property names from own and inherited
    	     * enumerable properties of `object`.
    	     *
    	     * @static
    	     * @memberOf _
    	     * @since 4.0.0
    	     * @category Object
    	     * @param {Object} object The object to inspect.
    	     * @returns {Array} Returns the function names.
    	     * @see _.functions
    	     * @example
    	     *
    	     * function Foo() {
    	     *   this.a = _.constant('a');
    	     *   this.b = _.constant('b');
    	     * }
    	     *
    	     * Foo.prototype.c = _.constant('c');
    	     *
    	     * _.functionsIn(new Foo);
    	     * // => ['a', 'b', 'c']
    	     */function functionsIn(object){return object==null?[]:baseFunctions(object,keysIn(object));}/**
    	     * Gets the value at `path` of `object`. If the resolved value is
    	     * `undefined`, the `defaultValue` is returned in its place.
    	     *
    	     * @static
    	     * @memberOf _
    	     * @since 3.7.0
    	     * @category Object
    	     * @param {Object} object The object to query.
    	     * @param {Array|string} path The path of the property to get.
    	     * @param {*} [defaultValue] The value returned for `undefined` resolved values.
    	     * @returns {*} Returns the resolved value.
    	     * @example
    	     *
    	     * var object = { 'a': [{ 'b': { 'c': 3 } }] };
    	     *
    	     * _.get(object, 'a[0].b.c');
    	     * // => 3
    	     *
    	     * _.get(object, ['a', '0', 'b', 'c']);
    	     * // => 3
    	     *
    	     * _.get(object, 'a.b.c', 'default');
    	     * // => 'default'
    	     */function get(object,path,defaultValue){var result=object==null?undefined$1:baseGet(object,path);return result===undefined$1?defaultValue:result;}/**
    	     * Checks if `path` is a direct property of `object`.
    	     *
    	     * @static
    	     * @since 0.1.0
    	     * @memberOf _
    	     * @category Object
    	     * @param {Object} object The object to query.
    	     * @param {Array|string} path The path to check.
    	     * @returns {boolean} Returns `true` if `path` exists, else `false`.
    	     * @example
    	     *
    	     * var object = { 'a': { 'b': 2 } };
    	     * var other = _.create({ 'a': _.create({ 'b': 2 }) });
    	     *
    	     * _.has(object, 'a');
    	     * // => true
    	     *
    	     * _.has(object, 'a.b');
    	     * // => true
    	     *
    	     * _.has(object, ['a', 'b']);
    	     * // => true
    	     *
    	     * _.has(other, 'a');
    	     * // => false
    	     */function has(object,path){return object!=null&&hasPath(object,path,baseHas);}/**
    	     * Checks if `path` is a direct or inherited property of `object`.
    	     *
    	     * @static
    	     * @memberOf _
    	     * @since 4.0.0
    	     * @category Object
    	     * @param {Object} object The object to query.
    	     * @param {Array|string} path The path to check.
    	     * @returns {boolean} Returns `true` if `path` exists, else `false`.
    	     * @example
    	     *
    	     * var object = _.create({ 'a': _.create({ 'b': 2 }) });
    	     *
    	     * _.hasIn(object, 'a');
    	     * // => true
    	     *
    	     * _.hasIn(object, 'a.b');
    	     * // => true
    	     *
    	     * _.hasIn(object, ['a', 'b']);
    	     * // => true
    	     *
    	     * _.hasIn(object, 'b');
    	     * // => false
    	     */function hasIn(object,path){return object!=null&&hasPath(object,path,baseHasIn);}/**
    	     * Creates an object composed of the inverted keys and values of `object`.
    	     * If `object` contains duplicate values, subsequent values overwrite
    	     * property assignments of previous values.
    	     *
    	     * @static
    	     * @memberOf _
    	     * @since 0.7.0
    	     * @category Object
    	     * @param {Object} object The object to invert.
    	     * @returns {Object} Returns the new inverted object.
    	     * @example
    	     *
    	     * var object = { 'a': 1, 'b': 2, 'c': 1 };
    	     *
    	     * _.invert(object);
    	     * // => { '1': 'c', '2': 'b' }
    	     */var invert=createInverter(function(result,value,key){if(value!=null&&typeof value.toString!='function'){value=nativeObjectToString.call(value);}result[value]=key;},constant(identity));/**
    	     * This method is like `_.invert` except that the inverted object is generated
    	     * from the results of running each element of `object` thru `iteratee`. The
    	     * corresponding inverted value of each inverted key is an array of keys
    	     * responsible for generating the inverted value. The iteratee is invoked
    	     * with one argument: (value).
    	     *
    	     * @static
    	     * @memberOf _
    	     * @since 4.1.0
    	     * @category Object
    	     * @param {Object} object The object to invert.
    	     * @param {Function} [iteratee=_.identity] The iteratee invoked per element.
    	     * @returns {Object} Returns the new inverted object.
    	     * @example
    	     *
    	     * var object = { 'a': 1, 'b': 2, 'c': 1 };
    	     *
    	     * _.invertBy(object);
    	     * // => { '1': ['a', 'c'], '2': ['b'] }
    	     *
    	     * _.invertBy(object, function(value) {
    	     *   return 'group' + value;
    	     * });
    	     * // => { 'group1': ['a', 'c'], 'group2': ['b'] }
    	     */var invertBy=createInverter(function(result,value,key){if(value!=null&&typeof value.toString!='function'){value=nativeObjectToString.call(value);}if(hasOwnProperty.call(result,value)){result[value].push(key);}else {result[value]=[key];}},getIteratee);/**
    	     * Invokes the method at `path` of `object`.
    	     *
    	     * @static
    	     * @memberOf _
    	     * @since 4.0.0
    	     * @category Object
    	     * @param {Object} object The object to query.
    	     * @param {Array|string} path The path of the method to invoke.
    	     * @param {...*} [args] The arguments to invoke the method with.
    	     * @returns {*} Returns the result of the invoked method.
    	     * @example
    	     *
    	     * var object = { 'a': [{ 'b': { 'c': [1, 2, 3, 4] } }] };
    	     *
    	     * _.invoke(object, 'a[0].b.c.slice', 1, 3);
    	     * // => [2, 3]
    	     */var invoke=baseRest(baseInvoke);/**
    	     * Creates an array of the own enumerable property names of `object`.
    	     *
    	     * **Note:** Non-object values are coerced to objects. See the
    	     * [ES spec](http://ecma-international.org/ecma-262/7.0/#sec-object.keys)
    	     * for more details.
    	     *
    	     * @static
    	     * @since 0.1.0
    	     * @memberOf _
    	     * @category Object
    	     * @param {Object} object The object to query.
    	     * @returns {Array} Returns the array of property names.
    	     * @example
    	     *
    	     * function Foo() {
    	     *   this.a = 1;
    	     *   this.b = 2;
    	     * }
    	     *
    	     * Foo.prototype.c = 3;
    	     *
    	     * _.keys(new Foo);
    	     * // => ['a', 'b'] (iteration order is not guaranteed)
    	     *
    	     * _.keys('hi');
    	     * // => ['0', '1']
    	     */function keys(object){return isArrayLike(object)?arrayLikeKeys(object):baseKeys(object);}/**
    	     * Creates an array of the own and inherited enumerable property names of `object`.
    	     *
    	     * **Note:** Non-object values are coerced to objects.
    	     *
    	     * @static
    	     * @memberOf _
    	     * @since 3.0.0
    	     * @category Object
    	     * @param {Object} object The object to query.
    	     * @returns {Array} Returns the array of property names.
    	     * @example
    	     *
    	     * function Foo() {
    	     *   this.a = 1;
    	     *   this.b = 2;
    	     * }
    	     *
    	     * Foo.prototype.c = 3;
    	     *
    	     * _.keysIn(new Foo);
    	     * // => ['a', 'b', 'c'] (iteration order is not guaranteed)
    	     */function keysIn(object){return isArrayLike(object)?arrayLikeKeys(object,true):baseKeysIn(object);}/**
    	     * The opposite of `_.mapValues`; this method creates an object with the
    	     * same values as `object` and keys generated by running each own enumerable
    	     * string keyed property of `object` thru `iteratee`. The iteratee is invoked
    	     * with three arguments: (value, key, object).
    	     *
    	     * @static
    	     * @memberOf _
    	     * @since 3.8.0
    	     * @category Object
    	     * @param {Object} object The object to iterate over.
    	     * @param {Function} [iteratee=_.identity] The function invoked per iteration.
    	     * @returns {Object} Returns the new mapped object.
    	     * @see _.mapValues
    	     * @example
    	     *
    	     * _.mapKeys({ 'a': 1, 'b': 2 }, function(value, key) {
    	     *   return key + value;
    	     * });
    	     * // => { 'a1': 1, 'b2': 2 }
    	     */function mapKeys(object,iteratee){var result={};iteratee=getIteratee(iteratee,3);baseForOwn(object,function(value,key,object){baseAssignValue(result,iteratee(value,key,object),value);});return result;}/**
    	     * Creates an object with the same keys as `object` and values generated
    	     * by running each own enumerable string keyed property of `object` thru
    	     * `iteratee`. The iteratee is invoked with three arguments:
    	     * (value, key, object).
    	     *
    	     * @static
    	     * @memberOf _
    	     * @since 2.4.0
    	     * @category Object
    	     * @param {Object} object The object to iterate over.
    	     * @param {Function} [iteratee=_.identity] The function invoked per iteration.
    	     * @returns {Object} Returns the new mapped object.
    	     * @see _.mapKeys
    	     * @example
    	     *
    	     * var users = {
    	     *   'fred':    { 'user': 'fred',    'age': 40 },
    	     *   'pebbles': { 'user': 'pebbles', 'age': 1 }
    	     * };
    	     *
    	     * _.mapValues(users, function(o) { return o.age; });
    	     * // => { 'fred': 40, 'pebbles': 1 } (iteration order is not guaranteed)
    	     *
    	     * // The `_.property` iteratee shorthand.
    	     * _.mapValues(users, 'age');
    	     * // => { 'fred': 40, 'pebbles': 1 } (iteration order is not guaranteed)
    	     */function mapValues(object,iteratee){var result={};iteratee=getIteratee(iteratee,3);baseForOwn(object,function(value,key,object){baseAssignValue(result,key,iteratee(value,key,object));});return result;}/**
    	     * This method is like `_.assign` except that it recursively merges own and
    	     * inherited enumerable string keyed properties of source objects into the
    	     * destination object. Source properties that resolve to `undefined` are
    	     * skipped if a destination value exists. Array and plain object properties
    	     * are merged recursively. Other objects and value types are overridden by
    	     * assignment. Source objects are applied from left to right. Subsequent
    	     * sources overwrite property assignments of previous sources.
    	     *
    	     * **Note:** This method mutates `object`.
    	     *
    	     * @static
    	     * @memberOf _
    	     * @since 0.5.0
    	     * @category Object
    	     * @param {Object} object The destination object.
    	     * @param {...Object} [sources] The source objects.
    	     * @returns {Object} Returns `object`.
    	     * @example
    	     *
    	     * var object = {
    	     *   'a': [{ 'b': 2 }, { 'd': 4 }]
    	     * };
    	     *
    	     * var other = {
    	     *   'a': [{ 'c': 3 }, { 'e': 5 }]
    	     * };
    	     *
    	     * _.merge(object, other);
    	     * // => { 'a': [{ 'b': 2, 'c': 3 }, { 'd': 4, 'e': 5 }] }
    	     */var merge=createAssigner(function(object,source,srcIndex){baseMerge(object,source,srcIndex);});/**
    	     * This method is like `_.merge` except that it accepts `customizer` which
    	     * is invoked to produce the merged values of the destination and source
    	     * properties. If `customizer` returns `undefined`, merging is handled by the
    	     * method instead. The `customizer` is invoked with six arguments:
    	     * (objValue, srcValue, key, object, source, stack).
    	     *
    	     * **Note:** This method mutates `object`.
    	     *
    	     * @static
    	     * @memberOf _
    	     * @since 4.0.0
    	     * @category Object
    	     * @param {Object} object The destination object.
    	     * @param {...Object} sources The source objects.
    	     * @param {Function} customizer The function to customize assigned values.
    	     * @returns {Object} Returns `object`.
    	     * @example
    	     *
    	     * function customizer(objValue, srcValue) {
    	     *   if (_.isArray(objValue)) {
    	     *     return objValue.concat(srcValue);
    	     *   }
    	     * }
    	     *
    	     * var object = { 'a': [1], 'b': [2] };
    	     * var other = { 'a': [3], 'b': [4] };
    	     *
    	     * _.mergeWith(object, other, customizer);
    	     * // => { 'a': [1, 3], 'b': [2, 4] }
    	     */var mergeWith=createAssigner(function(object,source,srcIndex,customizer){baseMerge(object,source,srcIndex,customizer);});/**
    	     * The opposite of `_.pick`; this method creates an object composed of the
    	     * own and inherited enumerable property paths of `object` that are not omitted.
    	     *
    	     * **Note:** This method is considerably slower than `_.pick`.
    	     *
    	     * @static
    	     * @since 0.1.0
    	     * @memberOf _
    	     * @category Object
    	     * @param {Object} object The source object.
    	     * @param {...(string|string[])} [paths] The property paths to omit.
    	     * @returns {Object} Returns the new object.
    	     * @example
    	     *
    	     * var object = { 'a': 1, 'b': '2', 'c': 3 };
    	     *
    	     * _.omit(object, ['a', 'c']);
    	     * // => { 'b': '2' }
    	     */var omit=flatRest(function(object,paths){var result={};if(object==null){return result;}var isDeep=false;paths=arrayMap(paths,function(path){path=castPath(path,object);isDeep||(isDeep=path.length>1);return path;});copyObject(object,getAllKeysIn(object),result);if(isDeep){result=baseClone(result,CLONE_DEEP_FLAG|CLONE_FLAT_FLAG|CLONE_SYMBOLS_FLAG,customOmitClone);}var length=paths.length;while(length--){baseUnset(result,paths[length]);}return result;});/**
    	     * The opposite of `_.pickBy`; this method creates an object composed of
    	     * the own and inherited enumerable string keyed properties of `object` that
    	     * `predicate` doesn't return truthy for. The predicate is invoked with two
    	     * arguments: (value, key).
    	     *
    	     * @static
    	     * @memberOf _
    	     * @since 4.0.0
    	     * @category Object
    	     * @param {Object} object The source object.
    	     * @param {Function} [predicate=_.identity] The function invoked per property.
    	     * @returns {Object} Returns the new object.
    	     * @example
    	     *
    	     * var object = { 'a': 1, 'b': '2', 'c': 3 };
    	     *
    	     * _.omitBy(object, _.isNumber);
    	     * // => { 'b': '2' }
    	     */function omitBy(object,predicate){return pickBy(object,negate(getIteratee(predicate)));}/**
    	     * Creates an object composed of the picked `object` properties.
    	     *
    	     * @static
    	     * @since 0.1.0
    	     * @memberOf _
    	     * @category Object
    	     * @param {Object} object The source object.
    	     * @param {...(string|string[])} [paths] The property paths to pick.
    	     * @returns {Object} Returns the new object.
    	     * @example
    	     *
    	     * var object = { 'a': 1, 'b': '2', 'c': 3 };
    	     *
    	     * _.pick(object, ['a', 'c']);
    	     * // => { 'a': 1, 'c': 3 }
    	     */var pick=flatRest(function(object,paths){return object==null?{}:basePick(object,paths);});/**
    	     * Creates an object composed of the `object` properties `predicate` returns
    	     * truthy for. The predicate is invoked with two arguments: (value, key).
    	     *
    	     * @static
    	     * @memberOf _
    	     * @since 4.0.0
    	     * @category Object
    	     * @param {Object} object The source object.
    	     * @param {Function} [predicate=_.identity] The function invoked per property.
    	     * @returns {Object} Returns the new object.
    	     * @example
    	     *
    	     * var object = { 'a': 1, 'b': '2', 'c': 3 };
    	     *
    	     * _.pickBy(object, _.isNumber);
    	     * // => { 'a': 1, 'c': 3 }
    	     */function pickBy(object,predicate){if(object==null){return {};}var props=arrayMap(getAllKeysIn(object),function(prop){return [prop];});predicate=getIteratee(predicate);return basePickBy(object,props,function(value,path){return predicate(value,path[0]);});}/**
    	     * This method is like `_.get` except that if the resolved value is a
    	     * function it's invoked with the `this` binding of its parent object and
    	     * its result is returned.
    	     *
    	     * @static
    	     * @since 0.1.0
    	     * @memberOf _
    	     * @category Object
    	     * @param {Object} object The object to query.
    	     * @param {Array|string} path The path of the property to resolve.
    	     * @param {*} [defaultValue] The value returned for `undefined` resolved values.
    	     * @returns {*} Returns the resolved value.
    	     * @example
    	     *
    	     * var object = { 'a': [{ 'b': { 'c1': 3, 'c2': _.constant(4) } }] };
    	     *
    	     * _.result(object, 'a[0].b.c1');
    	     * // => 3
    	     *
    	     * _.result(object, 'a[0].b.c2');
    	     * // => 4
    	     *
    	     * _.result(object, 'a[0].b.c3', 'default');
    	     * // => 'default'
    	     *
    	     * _.result(object, 'a[0].b.c3', _.constant('default'));
    	     * // => 'default'
    	     */function result(object,path,defaultValue){path=castPath(path,object);var index=-1,length=path.length;// Ensure the loop is entered when path is empty.
    	if(!length){length=1;object=undefined$1;}while(++index<length){var value=object==null?undefined$1:object[toKey(path[index])];if(value===undefined$1){index=length;value=defaultValue;}object=isFunction(value)?value.call(object):value;}return object;}/**
    	     * Sets the value at `path` of `object`. If a portion of `path` doesn't exist,
    	     * it's created. Arrays are created for missing index properties while objects
    	     * are created for all other missing properties. Use `_.setWith` to customize
    	     * `path` creation.
    	     *
    	     * **Note:** This method mutates `object`.
    	     *
    	     * @static
    	     * @memberOf _
    	     * @since 3.7.0
    	     * @category Object
    	     * @param {Object} object The object to modify.
    	     * @param {Array|string} path The path of the property to set.
    	     * @param {*} value The value to set.
    	     * @returns {Object} Returns `object`.
    	     * @example
    	     *
    	     * var object = { 'a': [{ 'b': { 'c': 3 } }] };
    	     *
    	     * _.set(object, 'a[0].b.c', 4);
    	     * console.log(object.a[0].b.c);
    	     * // => 4
    	     *
    	     * _.set(object, ['x', '0', 'y', 'z'], 5);
    	     * console.log(object.x[0].y.z);
    	     * // => 5
    	     */function set(object,path,value){return object==null?object:baseSet(object,path,value);}/**
    	     * This method is like `_.set` except that it accepts `customizer` which is
    	     * invoked to produce the objects of `path`.  If `customizer` returns `undefined`
    	     * path creation is handled by the method instead. The `customizer` is invoked
    	     * with three arguments: (nsValue, key, nsObject).
    	     *
    	     * **Note:** This method mutates `object`.
    	     *
    	     * @static
    	     * @memberOf _
    	     * @since 4.0.0
    	     * @category Object
    	     * @param {Object} object The object to modify.
    	     * @param {Array|string} path The path of the property to set.
    	     * @param {*} value The value to set.
    	     * @param {Function} [customizer] The function to customize assigned values.
    	     * @returns {Object} Returns `object`.
    	     * @example
    	     *
    	     * var object = {};
    	     *
    	     * _.setWith(object, '[0][1]', 'a', Object);
    	     * // => { '0': { '1': 'a' } }
    	     */function setWith(object,path,value,customizer){customizer=typeof customizer=='function'?customizer:undefined$1;return object==null?object:baseSet(object,path,value,customizer);}/**
    	     * Creates an array of own enumerable string keyed-value pairs for `object`
    	     * which can be consumed by `_.fromPairs`. If `object` is a map or set, its
    	     * entries are returned.
    	     *
    	     * @static
    	     * @memberOf _
    	     * @since 4.0.0
    	     * @alias entries
    	     * @category Object
    	     * @param {Object} object The object to query.
    	     * @returns {Array} Returns the key-value pairs.
    	     * @example
    	     *
    	     * function Foo() {
    	     *   this.a = 1;
    	     *   this.b = 2;
    	     * }
    	     *
    	     * Foo.prototype.c = 3;
    	     *
    	     * _.toPairs(new Foo);
    	     * // => [['a', 1], ['b', 2]] (iteration order is not guaranteed)
    	     */var toPairs=createToPairs(keys);/**
    	     * Creates an array of own and inherited enumerable string keyed-value pairs
    	     * for `object` which can be consumed by `_.fromPairs`. If `object` is a map
    	     * or set, its entries are returned.
    	     *
    	     * @static
    	     * @memberOf _
    	     * @since 4.0.0
    	     * @alias entriesIn
    	     * @category Object
    	     * @param {Object} object The object to query.
    	     * @returns {Array} Returns the key-value pairs.
    	     * @example
    	     *
    	     * function Foo() {
    	     *   this.a = 1;
    	     *   this.b = 2;
    	     * }
    	     *
    	     * Foo.prototype.c = 3;
    	     *
    	     * _.toPairsIn(new Foo);
    	     * // => [['a', 1], ['b', 2], ['c', 3]] (iteration order is not guaranteed)
    	     */var toPairsIn=createToPairs(keysIn);/**
    	     * An alternative to `_.reduce`; this method transforms `object` to a new
    	     * `accumulator` object which is the result of running each of its own
    	     * enumerable string keyed properties thru `iteratee`, with each invocation
    	     * potentially mutating the `accumulator` object. If `accumulator` is not
    	     * provided, a new object with the same `[[Prototype]]` will be used. The
    	     * iteratee is invoked with four arguments: (accumulator, value, key, object).
    	     * Iteratee functions may exit iteration early by explicitly returning `false`.
    	     *
    	     * @static
    	     * @memberOf _
    	     * @since 1.3.0
    	     * @category Object
    	     * @param {Object} object The object to iterate over.
    	     * @param {Function} [iteratee=_.identity] The function invoked per iteration.
    	     * @param {*} [accumulator] The custom accumulator value.
    	     * @returns {*} Returns the accumulated value.
    	     * @example
    	     *
    	     * _.transform([2, 3, 4], function(result, n) {
    	     *   result.push(n *= n);
    	     *   return n % 2 == 0;
    	     * }, []);
    	     * // => [4, 9]
    	     *
    	     * _.transform({ 'a': 1, 'b': 2, 'c': 1 }, function(result, value, key) {
    	     *   (result[value] || (result[value] = [])).push(key);
    	     * }, {});
    	     * // => { '1': ['a', 'c'], '2': ['b'] }
    	     */function transform(object,iteratee,accumulator){var isArr=isArray(object),isArrLike=isArr||isBuffer(object)||isTypedArray(object);iteratee=getIteratee(iteratee,4);if(accumulator==null){var Ctor=object&&object.constructor;if(isArrLike){accumulator=isArr?new Ctor():[];}else if(isObject(object)){accumulator=isFunction(Ctor)?baseCreate(getPrototype(object)):{};}else {accumulator={};}}(isArrLike?arrayEach:baseForOwn)(object,function(value,index,object){return iteratee(accumulator,value,index,object);});return accumulator;}/**
    	     * Removes the property at `path` of `object`.
    	     *
    	     * **Note:** This method mutates `object`.
    	     *
    	     * @static
    	     * @memberOf _
    	     * @since 4.0.0
    	     * @category Object
    	     * @param {Object} object The object to modify.
    	     * @param {Array|string} path The path of the property to unset.
    	     * @returns {boolean} Returns `true` if the property is deleted, else `false`.
    	     * @example
    	     *
    	     * var object = { 'a': [{ 'b': { 'c': 7 } }] };
    	     * _.unset(object, 'a[0].b.c');
    	     * // => true
    	     *
    	     * console.log(object);
    	     * // => { 'a': [{ 'b': {} }] };
    	     *
    	     * _.unset(object, ['a', '0', 'b', 'c']);
    	     * // => true
    	     *
    	     * console.log(object);
    	     * // => { 'a': [{ 'b': {} }] };
    	     */function unset(object,path){return object==null?true:baseUnset(object,path);}/**
    	     * This method is like `_.set` except that accepts `updater` to produce the
    	     * value to set. Use `_.updateWith` to customize `path` creation. The `updater`
    	     * is invoked with one argument: (value).
    	     *
    	     * **Note:** This method mutates `object`.
    	     *
    	     * @static
    	     * @memberOf _
    	     * @since 4.6.0
    	     * @category Object
    	     * @param {Object} object The object to modify.
    	     * @param {Array|string} path The path of the property to set.
    	     * @param {Function} updater The function to produce the updated value.
    	     * @returns {Object} Returns `object`.
    	     * @example
    	     *
    	     * var object = { 'a': [{ 'b': { 'c': 3 } }] };
    	     *
    	     * _.update(object, 'a[0].b.c', function(n) { return n * n; });
    	     * console.log(object.a[0].b.c);
    	     * // => 9
    	     *
    	     * _.update(object, 'x[0].y.z', function(n) { return n ? n + 1 : 0; });
    	     * console.log(object.x[0].y.z);
    	     * // => 0
    	     */function update(object,path,updater){return object==null?object:baseUpdate(object,path,castFunction(updater));}/**
    	     * This method is like `_.update` except that it accepts `customizer` which is
    	     * invoked to produce the objects of `path`.  If `customizer` returns `undefined`
    	     * path creation is handled by the method instead. The `customizer` is invoked
    	     * with three arguments: (nsValue, key, nsObject).
    	     *
    	     * **Note:** This method mutates `object`.
    	     *
    	     * @static
    	     * @memberOf _
    	     * @since 4.6.0
    	     * @category Object
    	     * @param {Object} object The object to modify.
    	     * @param {Array|string} path The path of the property to set.
    	     * @param {Function} updater The function to produce the updated value.
    	     * @param {Function} [customizer] The function to customize assigned values.
    	     * @returns {Object} Returns `object`.
    	     * @example
    	     *
    	     * var object = {};
    	     *
    	     * _.updateWith(object, '[0][1]', _.constant('a'), Object);
    	     * // => { '0': { '1': 'a' } }
    	     */function updateWith(object,path,updater,customizer){customizer=typeof customizer=='function'?customizer:undefined$1;return object==null?object:baseUpdate(object,path,castFunction(updater),customizer);}/**
    	     * Creates an array of the own enumerable string keyed property values of `object`.
    	     *
    	     * **Note:** Non-object values are coerced to objects.
    	     *
    	     * @static
    	     * @since 0.1.0
    	     * @memberOf _
    	     * @category Object
    	     * @param {Object} object The object to query.
    	     * @returns {Array} Returns the array of property values.
    	     * @example
    	     *
    	     * function Foo() {
    	     *   this.a = 1;
    	     *   this.b = 2;
    	     * }
    	     *
    	     * Foo.prototype.c = 3;
    	     *
    	     * _.values(new Foo);
    	     * // => [1, 2] (iteration order is not guaranteed)
    	     *
    	     * _.values('hi');
    	     * // => ['h', 'i']
    	     */function values(object){return object==null?[]:baseValues(object,keys(object));}/**
    	     * Creates an array of the own and inherited enumerable string keyed property
    	     * values of `object`.
    	     *
    	     * **Note:** Non-object values are coerced to objects.
    	     *
    	     * @static
    	     * @memberOf _
    	     * @since 3.0.0
    	     * @category Object
    	     * @param {Object} object The object to query.
    	     * @returns {Array} Returns the array of property values.
    	     * @example
    	     *
    	     * function Foo() {
    	     *   this.a = 1;
    	     *   this.b = 2;
    	     * }
    	     *
    	     * Foo.prototype.c = 3;
    	     *
    	     * _.valuesIn(new Foo);
    	     * // => [1, 2, 3] (iteration order is not guaranteed)
    	     */function valuesIn(object){return object==null?[]:baseValues(object,keysIn(object));}/*------------------------------------------------------------------------*//**
    	     * Clamps `number` within the inclusive `lower` and `upper` bounds.
    	     *
    	     * @static
    	     * @memberOf _
    	     * @since 4.0.0
    	     * @category Number
    	     * @param {number} number The number to clamp.
    	     * @param {number} [lower] The lower bound.
    	     * @param {number} upper The upper bound.
    	     * @returns {number} Returns the clamped number.
    	     * @example
    	     *
    	     * _.clamp(-10, -5, 5);
    	     * // => -5
    	     *
    	     * _.clamp(10, -5, 5);
    	     * // => 5
    	     */function clamp(number,lower,upper){if(upper===undefined$1){upper=lower;lower=undefined$1;}if(upper!==undefined$1){upper=toNumber(upper);upper=upper===upper?upper:0;}if(lower!==undefined$1){lower=toNumber(lower);lower=lower===lower?lower:0;}return baseClamp(toNumber(number),lower,upper);}/**
    	     * Checks if `n` is between `start` and up to, but not including, `end`. If
    	     * `end` is not specified, it's set to `start` with `start` then set to `0`.
    	     * If `start` is greater than `end` the params are swapped to support
    	     * negative ranges.
    	     *
    	     * @static
    	     * @memberOf _
    	     * @since 3.3.0
    	     * @category Number
    	     * @param {number} number The number to check.
    	     * @param {number} [start=0] The start of the range.
    	     * @param {number} end The end of the range.
    	     * @returns {boolean} Returns `true` if `number` is in the range, else `false`.
    	     * @see _.range, _.rangeRight
    	     * @example
    	     *
    	     * _.inRange(3, 2, 4);
    	     * // => true
    	     *
    	     * _.inRange(4, 8);
    	     * // => true
    	     *
    	     * _.inRange(4, 2);
    	     * // => false
    	     *
    	     * _.inRange(2, 2);
    	     * // => false
    	     *
    	     * _.inRange(1.2, 2);
    	     * // => true
    	     *
    	     * _.inRange(5.2, 4);
    	     * // => false
    	     *
    	     * _.inRange(-3, -2, -6);
    	     * // => true
    	     */function inRange(number,start,end){start=toFinite(start);if(end===undefined$1){end=start;start=0;}else {end=toFinite(end);}number=toNumber(number);return baseInRange(number,start,end);}/**
    	     * Produces a random number between the inclusive `lower` and `upper` bounds.
    	     * If only one argument is provided a number between `0` and the given number
    	     * is returned. If `floating` is `true`, or either `lower` or `upper` are
    	     * floats, a floating-point number is returned instead of an integer.
    	     *
    	     * **Note:** JavaScript follows the IEEE-754 standard for resolving
    	     * floating-point values which can produce unexpected results.
    	     *
    	     * @static
    	     * @memberOf _
    	     * @since 0.7.0
    	     * @category Number
    	     * @param {number} [lower=0] The lower bound.
    	     * @param {number} [upper=1] The upper bound.
    	     * @param {boolean} [floating] Specify returning a floating-point number.
    	     * @returns {number} Returns the random number.
    	     * @example
    	     *
    	     * _.random(0, 5);
    	     * // => an integer between 0 and 5
    	     *
    	     * _.random(5);
    	     * // => also an integer between 0 and 5
    	     *
    	     * _.random(5, true);
    	     * // => a floating-point number between 0 and 5
    	     *
    	     * _.random(1.2, 5.2);
    	     * // => a floating-point number between 1.2 and 5.2
    	     */function random(lower,upper,floating){if(floating&&typeof floating!='boolean'&&isIterateeCall(lower,upper,floating)){upper=floating=undefined$1;}if(floating===undefined$1){if(typeof upper=='boolean'){floating=upper;upper=undefined$1;}else if(typeof lower=='boolean'){floating=lower;lower=undefined$1;}}if(lower===undefined$1&&upper===undefined$1){lower=0;upper=1;}else {lower=toFinite(lower);if(upper===undefined$1){upper=lower;lower=0;}else {upper=toFinite(upper);}}if(lower>upper){var temp=lower;lower=upper;upper=temp;}if(floating||lower%1||upper%1){var rand=nativeRandom();return nativeMin(lower+rand*(upper-lower+freeParseFloat('1e-'+((rand+'').length-1))),upper);}return baseRandom(lower,upper);}/*------------------------------------------------------------------------*//**
    	     * Converts `string` to [camel case](https://en.wikipedia.org/wiki/CamelCase).
    	     *
    	     * @static
    	     * @memberOf _
    	     * @since 3.0.0
    	     * @category String
    	     * @param {string} [string=''] The string to convert.
    	     * @returns {string} Returns the camel cased string.
    	     * @example
    	     *
    	     * _.camelCase('Foo Bar');
    	     * // => 'fooBar'
    	     *
    	     * _.camelCase('--foo-bar--');
    	     * // => 'fooBar'
    	     *
    	     * _.camelCase('__FOO_BAR__');
    	     * // => 'fooBar'
    	     */var camelCase=createCompounder(function(result,word,index){word=word.toLowerCase();return result+(index?capitalize(word):word);});/**
    	     * Converts the first character of `string` to upper case and the remaining
    	     * to lower case.
    	     *
    	     * @static
    	     * @memberOf _
    	     * @since 3.0.0
    	     * @category String
    	     * @param {string} [string=''] The string to capitalize.
    	     * @returns {string} Returns the capitalized string.
    	     * @example
    	     *
    	     * _.capitalize('FRED');
    	     * // => 'Fred'
    	     */function capitalize(string){return upperFirst(toString(string).toLowerCase());}/**
    	     * Deburrs `string` by converting
    	     * [Latin-1 Supplement](https://en.wikipedia.org/wiki/Latin-1_Supplement_(Unicode_block)#Character_table)
    	     * and [Latin Extended-A](https://en.wikipedia.org/wiki/Latin_Extended-A)
    	     * letters to basic Latin letters and removing
    	     * [combining diacritical marks](https://en.wikipedia.org/wiki/Combining_Diacritical_Marks).
    	     *
    	     * @static
    	     * @memberOf _
    	     * @since 3.0.0
    	     * @category String
    	     * @param {string} [string=''] The string to deburr.
    	     * @returns {string} Returns the deburred string.
    	     * @example
    	     *
    	     * _.deburr('déjà vu');
    	     * // => 'deja vu'
    	     */function deburr(string){string=toString(string);return string&&string.replace(reLatin,deburrLetter).replace(reComboMark,'');}/**
    	     * Checks if `string` ends with the given target string.
    	     *
    	     * @static
    	     * @memberOf _
    	     * @since 3.0.0
    	     * @category String
    	     * @param {string} [string=''] The string to inspect.
    	     * @param {string} [target] The string to search for.
    	     * @param {number} [position=string.length] The position to search up to.
    	     * @returns {boolean} Returns `true` if `string` ends with `target`,
    	     *  else `false`.
    	     * @example
    	     *
    	     * _.endsWith('abc', 'c');
    	     * // => true
    	     *
    	     * _.endsWith('abc', 'b');
    	     * // => false
    	     *
    	     * _.endsWith('abc', 'b', 2);
    	     * // => true
    	     */function endsWith(string,target,position){string=toString(string);target=baseToString(target);var length=string.length;position=position===undefined$1?length:baseClamp(toInteger(position),0,length);var end=position;position-=target.length;return position>=0&&string.slice(position,end)==target;}/**
    	     * Converts the characters "&", "<", ">", '"', and "'" in `string` to their
    	     * corresponding HTML entities.
    	     *
    	     * **Note:** No other characters are escaped. To escape additional
    	     * characters use a third-party library like [_he_](https://mths.be/he).
    	     *
    	     * Though the ">" character is escaped for symmetry, characters like
    	     * ">" and "/" don't need escaping in HTML and have no special meaning
    	     * unless they're part of a tag or unquoted attribute value. See
    	     * [Mathias Bynens's article](https://mathiasbynens.be/notes/ambiguous-ampersands)
    	     * (under "semi-related fun fact") for more details.
    	     *
    	     * When working with HTML you should always
    	     * [quote attribute values](http://wonko.com/post/html-escaping) to reduce
    	     * XSS vectors.
    	     *
    	     * @static
    	     * @since 0.1.0
    	     * @memberOf _
    	     * @category String
    	     * @param {string} [string=''] The string to escape.
    	     * @returns {string} Returns the escaped string.
    	     * @example
    	     *
    	     * _.escape('fred, barney, & pebbles');
    	     * // => 'fred, barney, &amp; pebbles'
    	     */function escape(string){string=toString(string);return string&&reHasUnescapedHtml.test(string)?string.replace(reUnescapedHtml,escapeHtmlChar):string;}/**
    	     * Escapes the `RegExp` special characters "^", "$", "\", ".", "*", "+",
    	     * "?", "(", ")", "[", "]", "{", "}", and "|" in `string`.
    	     *
    	     * @static
    	     * @memberOf _
    	     * @since 3.0.0
    	     * @category String
    	     * @param {string} [string=''] The string to escape.
    	     * @returns {string} Returns the escaped string.
    	     * @example
    	     *
    	     * _.escapeRegExp('[lodash](https://lodash.com/)');
    	     * // => '\[lodash\]\(https://lodash\.com/\)'
    	     */function escapeRegExp(string){string=toString(string);return string&&reHasRegExpChar.test(string)?string.replace(reRegExpChar,'\\$&'):string;}/**
    	     * Converts `string` to
    	     * [kebab case](https://en.wikipedia.org/wiki/Letter_case#Special_case_styles).
    	     *
    	     * @static
    	     * @memberOf _
    	     * @since 3.0.0
    	     * @category String
    	     * @param {string} [string=''] The string to convert.
    	     * @returns {string} Returns the kebab cased string.
    	     * @example
    	     *
    	     * _.kebabCase('Foo Bar');
    	     * // => 'foo-bar'
    	     *
    	     * _.kebabCase('fooBar');
    	     * // => 'foo-bar'
    	     *
    	     * _.kebabCase('__FOO_BAR__');
    	     * // => 'foo-bar'
    	     */var kebabCase=createCompounder(function(result,word,index){return result+(index?'-':'')+word.toLowerCase();});/**
    	     * Converts `string`, as space separated words, to lower case.
    	     *
    	     * @static
    	     * @memberOf _
    	     * @since 4.0.0
    	     * @category String
    	     * @param {string} [string=''] The string to convert.
    	     * @returns {string} Returns the lower cased string.
    	     * @example
    	     *
    	     * _.lowerCase('--Foo-Bar--');
    	     * // => 'foo bar'
    	     *
    	     * _.lowerCase('fooBar');
    	     * // => 'foo bar'
    	     *
    	     * _.lowerCase('__FOO_BAR__');
    	     * // => 'foo bar'
    	     */var lowerCase=createCompounder(function(result,word,index){return result+(index?' ':'')+word.toLowerCase();});/**
    	     * Converts the first character of `string` to lower case.
    	     *
    	     * @static
    	     * @memberOf _
    	     * @since 4.0.0
    	     * @category String
    	     * @param {string} [string=''] The string to convert.
    	     * @returns {string} Returns the converted string.
    	     * @example
    	     *
    	     * _.lowerFirst('Fred');
    	     * // => 'fred'
    	     *
    	     * _.lowerFirst('FRED');
    	     * // => 'fRED'
    	     */var lowerFirst=createCaseFirst('toLowerCase');/**
    	     * Pads `string` on the left and right sides if it's shorter than `length`.
    	     * Padding characters are truncated if they can't be evenly divided by `length`.
    	     *
    	     * @static
    	     * @memberOf _
    	     * @since 3.0.0
    	     * @category String
    	     * @param {string} [string=''] The string to pad.
    	     * @param {number} [length=0] The padding length.
    	     * @param {string} [chars=' '] The string used as padding.
    	     * @returns {string} Returns the padded string.
    	     * @example
    	     *
    	     * _.pad('abc', 8);
    	     * // => '  abc   '
    	     *
    	     * _.pad('abc', 8, '_-');
    	     * // => '_-abc_-_'
    	     *
    	     * _.pad('abc', 3);
    	     * // => 'abc'
    	     */function pad(string,length,chars){string=toString(string);length=toInteger(length);var strLength=length?stringSize(string):0;if(!length||strLength>=length){return string;}var mid=(length-strLength)/2;return createPadding(nativeFloor(mid),chars)+string+createPadding(nativeCeil(mid),chars);}/**
    	     * Pads `string` on the right side if it's shorter than `length`. Padding
    	     * characters are truncated if they exceed `length`.
    	     *
    	     * @static
    	     * @memberOf _
    	     * @since 4.0.0
    	     * @category String
    	     * @param {string} [string=''] The string to pad.
    	     * @param {number} [length=0] The padding length.
    	     * @param {string} [chars=' '] The string used as padding.
    	     * @returns {string} Returns the padded string.
    	     * @example
    	     *
    	     * _.padEnd('abc', 6);
    	     * // => 'abc   '
    	     *
    	     * _.padEnd('abc', 6, '_-');
    	     * // => 'abc_-_'
    	     *
    	     * _.padEnd('abc', 3);
    	     * // => 'abc'
    	     */function padEnd(string,length,chars){string=toString(string);length=toInteger(length);var strLength=length?stringSize(string):0;return length&&strLength<length?string+createPadding(length-strLength,chars):string;}/**
    	     * Pads `string` on the left side if it's shorter than `length`. Padding
    	     * characters are truncated if they exceed `length`.
    	     *
    	     * @static
    	     * @memberOf _
    	     * @since 4.0.0
    	     * @category String
    	     * @param {string} [string=''] The string to pad.
    	     * @param {number} [length=0] The padding length.
    	     * @param {string} [chars=' '] The string used as padding.
    	     * @returns {string} Returns the padded string.
    	     * @example
    	     *
    	     * _.padStart('abc', 6);
    	     * // => '   abc'
    	     *
    	     * _.padStart('abc', 6, '_-');
    	     * // => '_-_abc'
    	     *
    	     * _.padStart('abc', 3);
    	     * // => 'abc'
    	     */function padStart(string,length,chars){string=toString(string);length=toInteger(length);var strLength=length?stringSize(string):0;return length&&strLength<length?createPadding(length-strLength,chars)+string:string;}/**
    	     * Converts `string` to an integer of the specified radix. If `radix` is
    	     * `undefined` or `0`, a `radix` of `10` is used unless `value` is a
    	     * hexadecimal, in which case a `radix` of `16` is used.
    	     *
    	     * **Note:** This method aligns with the
    	     * [ES5 implementation](https://es5.github.io/#x15.1.2.2) of `parseInt`.
    	     *
    	     * @static
    	     * @memberOf _
    	     * @since 1.1.0
    	     * @category String
    	     * @param {string} string The string to convert.
    	     * @param {number} [radix=10] The radix to interpret `value` by.
    	     * @param- {Object} [guard] Enables use as an iteratee for methods like `_.map`.
    	     * @returns {number} Returns the converted integer.
    	     * @example
    	     *
    	     * _.parseInt('08');
    	     * // => 8
    	     *
    	     * _.map(['6', '08', '10'], _.parseInt);
    	     * // => [6, 8, 10]
    	     */function parseInt(string,radix,guard){if(guard||radix==null){radix=0;}else if(radix){radix=+radix;}return nativeParseInt(toString(string).replace(reTrimStart,''),radix||0);}/**
    	     * Repeats the given string `n` times.
    	     *
    	     * @static
    	     * @memberOf _
    	     * @since 3.0.0
    	     * @category String
    	     * @param {string} [string=''] The string to repeat.
    	     * @param {number} [n=1] The number of times to repeat the string.
    	     * @param- {Object} [guard] Enables use as an iteratee for methods like `_.map`.
    	     * @returns {string} Returns the repeated string.
    	     * @example
    	     *
    	     * _.repeat('*', 3);
    	     * // => '***'
    	     *
    	     * _.repeat('abc', 2);
    	     * // => 'abcabc'
    	     *
    	     * _.repeat('abc', 0);
    	     * // => ''
    	     */function repeat(string,n,guard){if(guard?isIterateeCall(string,n,guard):n===undefined$1){n=1;}else {n=toInteger(n);}return baseRepeat(toString(string),n);}/**
    	     * Replaces matches for `pattern` in `string` with `replacement`.
    	     *
    	     * **Note:** This method is based on
    	     * [`String#replace`](https://mdn.io/String/replace).
    	     *
    	     * @static
    	     * @memberOf _
    	     * @since 4.0.0
    	     * @category String
    	     * @param {string} [string=''] The string to modify.
    	     * @param {RegExp|string} pattern The pattern to replace.
    	     * @param {Function|string} replacement The match replacement.
    	     * @returns {string} Returns the modified string.
    	     * @example
    	     *
    	     * _.replace('Hi Fred', 'Fred', 'Barney');
    	     * // => 'Hi Barney'
    	     */function replace(){var args=arguments,string=toString(args[0]);return args.length<3?string:string.replace(args[1],args[2]);}/**
    	     * Converts `string` to
    	     * [snake case](https://en.wikipedia.org/wiki/Snake_case).
    	     *
    	     * @static
    	     * @memberOf _
    	     * @since 3.0.0
    	     * @category String
    	     * @param {string} [string=''] The string to convert.
    	     * @returns {string} Returns the snake cased string.
    	     * @example
    	     *
    	     * _.snakeCase('Foo Bar');
    	     * // => 'foo_bar'
    	     *
    	     * _.snakeCase('fooBar');
    	     * // => 'foo_bar'
    	     *
    	     * _.snakeCase('--FOO-BAR--');
    	     * // => 'foo_bar'
    	     */var snakeCase=createCompounder(function(result,word,index){return result+(index?'_':'')+word.toLowerCase();});/**
    	     * Splits `string` by `separator`.
    	     *
    	     * **Note:** This method is based on
    	     * [`String#split`](https://mdn.io/String/split).
    	     *
    	     * @static
    	     * @memberOf _
    	     * @since 4.0.0
    	     * @category String
    	     * @param {string} [string=''] The string to split.
    	     * @param {RegExp|string} separator The separator pattern to split by.
    	     * @param {number} [limit] The length to truncate results to.
    	     * @returns {Array} Returns the string segments.
    	     * @example
    	     *
    	     * _.split('a-b-c', '-', 2);
    	     * // => ['a', 'b']
    	     */function split(string,separator,limit){if(limit&&typeof limit!='number'&&isIterateeCall(string,separator,limit)){separator=limit=undefined$1;}limit=limit===undefined$1?MAX_ARRAY_LENGTH:limit>>>0;if(!limit){return [];}string=toString(string);if(string&&(typeof separator=='string'||separator!=null&&!isRegExp(separator))){separator=baseToString(separator);if(!separator&&hasUnicode(string)){return castSlice(stringToArray(string),0,limit);}}return string.split(separator,limit);}/**
    	     * Converts `string` to
    	     * [start case](https://en.wikipedia.org/wiki/Letter_case#Stylistic_or_specialised_usage).
    	     *
    	     * @static
    	     * @memberOf _
    	     * @since 3.1.0
    	     * @category String
    	     * @param {string} [string=''] The string to convert.
    	     * @returns {string} Returns the start cased string.
    	     * @example
    	     *
    	     * _.startCase('--foo-bar--');
    	     * // => 'Foo Bar'
    	     *
    	     * _.startCase('fooBar');
    	     * // => 'Foo Bar'
    	     *
    	     * _.startCase('__FOO_BAR__');
    	     * // => 'FOO BAR'
    	     */var startCase=createCompounder(function(result,word,index){return result+(index?' ':'')+upperFirst(word);});/**
    	     * Checks if `string` starts with the given target string.
    	     *
    	     * @static
    	     * @memberOf _
    	     * @since 3.0.0
    	     * @category String
    	     * @param {string} [string=''] The string to inspect.
    	     * @param {string} [target] The string to search for.
    	     * @param {number} [position=0] The position to search from.
    	     * @returns {boolean} Returns `true` if `string` starts with `target`,
    	     *  else `false`.
    	     * @example
    	     *
    	     * _.startsWith('abc', 'a');
    	     * // => true
    	     *
    	     * _.startsWith('abc', 'b');
    	     * // => false
    	     *
    	     * _.startsWith('abc', 'b', 1);
    	     * // => true
    	     */function startsWith(string,target,position){string=toString(string);position=position==null?0:baseClamp(toInteger(position),0,string.length);target=baseToString(target);return string.slice(position,position+target.length)==target;}/**
    	     * Creates a compiled template function that can interpolate data properties
    	     * in "interpolate" delimiters, HTML-escape interpolated data properties in
    	     * "escape" delimiters, and execute JavaScript in "evaluate" delimiters. Data
    	     * properties may be accessed as free variables in the template. If a setting
    	     * object is given, it takes precedence over `_.templateSettings` values.
    	     *
    	     * **Note:** In the development build `_.template` utilizes
    	     * [sourceURLs](http://www.html5rocks.com/en/tutorials/developertools/sourcemaps/#toc-sourceurl)
    	     * for easier debugging.
    	     *
    	     * For more information on precompiling templates see
    	     * [lodash's custom builds documentation](https://lodash.com/custom-builds).
    	     *
    	     * For more information on Chrome extension sandboxes see
    	     * [Chrome's extensions documentation](https://developer.chrome.com/extensions/sandboxingEval).
    	     *
    	     * @static
    	     * @since 0.1.0
    	     * @memberOf _
    	     * @category String
    	     * @param {string} [string=''] The template string.
    	     * @param {Object} [options={}] The options object.
    	     * @param {RegExp} [options.escape=_.templateSettings.escape]
    	     *  The HTML "escape" delimiter.
    	     * @param {RegExp} [options.evaluate=_.templateSettings.evaluate]
    	     *  The "evaluate" delimiter.
    	     * @param {Object} [options.imports=_.templateSettings.imports]
    	     *  An object to import into the template as free variables.
    	     * @param {RegExp} [options.interpolate=_.templateSettings.interpolate]
    	     *  The "interpolate" delimiter.
    	     * @param {string} [options.sourceURL='lodash.templateSources[n]']
    	     *  The sourceURL of the compiled template.
    	     * @param {string} [options.variable='obj']
    	     *  The data object variable name.
    	     * @param- {Object} [guard] Enables use as an iteratee for methods like `_.map`.
    	     * @returns {Function} Returns the compiled template function.
    	     * @example
    	     *
    	     * // Use the "interpolate" delimiter to create a compiled template.
    	     * var compiled = _.template('hello <%= user %>!');
    	     * compiled({ 'user': 'fred' });
    	     * // => 'hello fred!'
    	     *
    	     * // Use the HTML "escape" delimiter to escape data property values.
    	     * var compiled = _.template('<b><%- value %></b>');
    	     * compiled({ 'value': '<script>' });
    	     * // => '<b>&lt;script&gt;</b>'
    	     *
    	     * // Use the "evaluate" delimiter to execute JavaScript and generate HTML.
    	     * var compiled = _.template('<% _.forEach(users, function(user) { %><li><%- user %></li><% }); %>');
    	     * compiled({ 'users': ['fred', 'barney'] });
    	     * // => '<li>fred</li><li>barney</li>'
    	     *
    	     * // Use the internal `print` function in "evaluate" delimiters.
    	     * var compiled = _.template('<% print("hello " + user); %>!');
    	     * compiled({ 'user': 'barney' });
    	     * // => 'hello barney!'
    	     *
    	     * // Use the ES template literal delimiter as an "interpolate" delimiter.
    	     * // Disable support by replacing the "interpolate" delimiter.
    	     * var compiled = _.template('hello ${ user }!');
    	     * compiled({ 'user': 'pebbles' });
    	     * // => 'hello pebbles!'
    	     *
    	     * // Use backslashes to treat delimiters as plain text.
    	     * var compiled = _.template('<%= "\\<%- value %\\>" %>');
    	     * compiled({ 'value': 'ignored' });
    	     * // => '<%- value %>'
    	     *
    	     * // Use the `imports` option to import `jQuery` as `jq`.
    	     * var text = '<% jq.each(users, function(user) { %><li><%- user %></li><% }); %>';
    	     * var compiled = _.template(text, { 'imports': { 'jq': jQuery } });
    	     * compiled({ 'users': ['fred', 'barney'] });
    	     * // => '<li>fred</li><li>barney</li>'
    	     *
    	     * // Use the `sourceURL` option to specify a custom sourceURL for the template.
    	     * var compiled = _.template('hello <%= user %>!', { 'sourceURL': '/basic/greeting.jst' });
    	     * compiled(data);
    	     * // => Find the source of "greeting.jst" under the Sources tab or Resources panel of the web inspector.
    	     *
    	     * // Use the `variable` option to ensure a with-statement isn't used in the compiled template.
    	     * var compiled = _.template('hi <%= data.user %>!', { 'variable': 'data' });
    	     * compiled.source;
    	     * // => function(data) {
    	     * //   var __t, __p = '';
    	     * //   __p += 'hi ' + ((__t = ( data.user )) == null ? '' : __t) + '!';
    	     * //   return __p;
    	     * // }
    	     *
    	     * // Use custom template delimiters.
    	     * _.templateSettings.interpolate = /{{([\s\S]+?)}}/g;
    	     * var compiled = _.template('hello {{ user }}!');
    	     * compiled({ 'user': 'mustache' });
    	     * // => 'hello mustache!'
    	     *
    	     * // Use the `source` property to inline compiled templates for meaningful
    	     * // line numbers in error messages and stack traces.
    	     * fs.writeFileSync(path.join(process.cwd(), 'jst.js'), '\
    	     *   var JST = {\
    	     *     "main": ' + _.template(mainText).source + '\
    	     *   };\
    	     * ');
    	     */function template(string,options,guard){// Based on John Resig's `tmpl` implementation
    	// (http://ejohn.org/blog/javascript-micro-templating/)
    	// and Laura Doktorova's doT.js (https://github.com/olado/doT).
    	var settings=lodash.templateSettings;if(guard&&isIterateeCall(string,options,guard)){options=undefined$1;}string=toString(string);options=assignInWith({},options,settings,customDefaultsAssignIn);var imports=assignInWith({},options.imports,settings.imports,customDefaultsAssignIn),importsKeys=keys(imports),importsValues=baseValues(imports,importsKeys);var isEscaping,isEvaluating,index=0,interpolate=options.interpolate||reNoMatch,source="__p += '";// Compile the regexp to match each delimiter.
    	var reDelimiters=RegExp((options.escape||reNoMatch).source+'|'+interpolate.source+'|'+(interpolate===reInterpolate?reEsTemplate:reNoMatch).source+'|'+(options.evaluate||reNoMatch).source+'|$','g');// Use a sourceURL for easier debugging.
    	// The sourceURL gets injected into the source that's eval-ed, so be careful
    	// to normalize all kinds of whitespace, so e.g. newlines (and unicode versions of it) can't sneak in
    	// and escape the comment, thus injecting code that gets evaled.
    	var sourceURL='//# sourceURL='+(hasOwnProperty.call(options,'sourceURL')?(options.sourceURL+'').replace(/\s/g,' '):'lodash.templateSources['+ ++templateCounter+']')+'\n';string.replace(reDelimiters,function(match,escapeValue,interpolateValue,esTemplateValue,evaluateValue,offset){interpolateValue||(interpolateValue=esTemplateValue);// Escape characters that can't be included in string literals.
    	source+=string.slice(index,offset).replace(reUnescapedString,escapeStringChar);// Replace delimiters with snippets.
    	if(escapeValue){isEscaping=true;source+="' +\n__e("+escapeValue+") +\n'";}if(evaluateValue){isEvaluating=true;source+="';\n"+evaluateValue+";\n__p += '";}if(interpolateValue){source+="' +\n((__t = ("+interpolateValue+")) == null ? '' : __t) +\n'";}index=offset+match.length;// The JS engine embedded in Adobe products needs `match` returned in
    	// order to produce the correct `offset` value.
    	return match;});source+="';\n";// If `variable` is not specified wrap a with-statement around the generated
    	// code to add the data object to the top of the scope chain.
    	var variable=hasOwnProperty.call(options,'variable')&&options.variable;if(!variable){source='with (obj) {\n'+source+'\n}\n';}// Throw an error if a forbidden character was found in `variable`, to prevent
    	// potential command injection attacks.
    	else if(reForbiddenIdentifierChars.test(variable)){throw new Error(INVALID_TEMPL_VAR_ERROR_TEXT);}// Cleanup code by stripping empty strings.
    	source=(isEvaluating?source.replace(reEmptyStringLeading,''):source).replace(reEmptyStringMiddle,'$1').replace(reEmptyStringTrailing,'$1;');// Frame code as the function body.
    	source='function('+(variable||'obj')+') {\n'+(variable?'':'obj || (obj = {});\n')+"var __t, __p = ''"+(isEscaping?', __e = _.escape':'')+(isEvaluating?', __j = Array.prototype.join;\n'+"function print() { __p += __j.call(arguments, '') }\n":';\n')+source+'return __p\n}';var result=attempt(function(){return Function(importsKeys,sourceURL+'return '+source).apply(undefined$1,importsValues);});// Provide the compiled function's source by its `toString` method or
    	// the `source` property as a convenience for inlining compiled templates.
    	result.source=source;if(isError(result)){throw result;}return result;}/**
    	     * Converts `string`, as a whole, to lower case just like
    	     * [String#toLowerCase](https://mdn.io/toLowerCase).
    	     *
    	     * @static
    	     * @memberOf _
    	     * @since 4.0.0
    	     * @category String
    	     * @param {string} [string=''] The string to convert.
    	     * @returns {string} Returns the lower cased string.
    	     * @example
    	     *
    	     * _.toLower('--Foo-Bar--');
    	     * // => '--foo-bar--'
    	     *
    	     * _.toLower('fooBar');
    	     * // => 'foobar'
    	     *
    	     * _.toLower('__FOO_BAR__');
    	     * // => '__foo_bar__'
    	     */function toLower(value){return toString(value).toLowerCase();}/**
    	     * Converts `string`, as a whole, to upper case just like
    	     * [String#toUpperCase](https://mdn.io/toUpperCase).
    	     *
    	     * @static
    	     * @memberOf _
    	     * @since 4.0.0
    	     * @category String
    	     * @param {string} [string=''] The string to convert.
    	     * @returns {string} Returns the upper cased string.
    	     * @example
    	     *
    	     * _.toUpper('--foo-bar--');
    	     * // => '--FOO-BAR--'
    	     *
    	     * _.toUpper('fooBar');
    	     * // => 'FOOBAR'
    	     *
    	     * _.toUpper('__foo_bar__');
    	     * // => '__FOO_BAR__'
    	     */function toUpper(value){return toString(value).toUpperCase();}/**
    	     * Removes leading and trailing whitespace or specified characters from `string`.
    	     *
    	     * @static
    	     * @memberOf _
    	     * @since 3.0.0
    	     * @category String
    	     * @param {string} [string=''] The string to trim.
    	     * @param {string} [chars=whitespace] The characters to trim.
    	     * @param- {Object} [guard] Enables use as an iteratee for methods like `_.map`.
    	     * @returns {string} Returns the trimmed string.
    	     * @example
    	     *
    	     * _.trim('  abc  ');
    	     * // => 'abc'
    	     *
    	     * _.trim('-_-abc-_-', '_-');
    	     * // => 'abc'
    	     *
    	     * _.map(['  foo  ', '  bar  '], _.trim);
    	     * // => ['foo', 'bar']
    	     */function trim(string,chars,guard){string=toString(string);if(string&&(guard||chars===undefined$1)){return baseTrim(string);}if(!string||!(chars=baseToString(chars))){return string;}var strSymbols=stringToArray(string),chrSymbols=stringToArray(chars),start=charsStartIndex(strSymbols,chrSymbols),end=charsEndIndex(strSymbols,chrSymbols)+1;return castSlice(strSymbols,start,end).join('');}/**
    	     * Removes trailing whitespace or specified characters from `string`.
    	     *
    	     * @static
    	     * @memberOf _
    	     * @since 4.0.0
    	     * @category String
    	     * @param {string} [string=''] The string to trim.
    	     * @param {string} [chars=whitespace] The characters to trim.
    	     * @param- {Object} [guard] Enables use as an iteratee for methods like `_.map`.
    	     * @returns {string} Returns the trimmed string.
    	     * @example
    	     *
    	     * _.trimEnd('  abc  ');
    	     * // => '  abc'
    	     *
    	     * _.trimEnd('-_-abc-_-', '_-');
    	     * // => '-_-abc'
    	     */function trimEnd(string,chars,guard){string=toString(string);if(string&&(guard||chars===undefined$1)){return string.slice(0,trimmedEndIndex(string)+1);}if(!string||!(chars=baseToString(chars))){return string;}var strSymbols=stringToArray(string),end=charsEndIndex(strSymbols,stringToArray(chars))+1;return castSlice(strSymbols,0,end).join('');}/**
    	     * Removes leading whitespace or specified characters from `string`.
    	     *
    	     * @static
    	     * @memberOf _
    	     * @since 4.0.0
    	     * @category String
    	     * @param {string} [string=''] The string to trim.
    	     * @param {string} [chars=whitespace] The characters to trim.
    	     * @param- {Object} [guard] Enables use as an iteratee for methods like `_.map`.
    	     * @returns {string} Returns the trimmed string.
    	     * @example
    	     *
    	     * _.trimStart('  abc  ');
    	     * // => 'abc  '
    	     *
    	     * _.trimStart('-_-abc-_-', '_-');
    	     * // => 'abc-_-'
    	     */function trimStart(string,chars,guard){string=toString(string);if(string&&(guard||chars===undefined$1)){return string.replace(reTrimStart,'');}if(!string||!(chars=baseToString(chars))){return string;}var strSymbols=stringToArray(string),start=charsStartIndex(strSymbols,stringToArray(chars));return castSlice(strSymbols,start).join('');}/**
    	     * Truncates `string` if it's longer than the given maximum string length.
    	     * The last characters of the truncated string are replaced with the omission
    	     * string which defaults to "...".
    	     *
    	     * @static
    	     * @memberOf _
    	     * @since 4.0.0
    	     * @category String
    	     * @param {string} [string=''] The string to truncate.
    	     * @param {Object} [options={}] The options object.
    	     * @param {number} [options.length=30] The maximum string length.
    	     * @param {string} [options.omission='...'] The string to indicate text is omitted.
    	     * @param {RegExp|string} [options.separator] The separator pattern to truncate to.
    	     * @returns {string} Returns the truncated string.
    	     * @example
    	     *
    	     * _.truncate('hi-diddly-ho there, neighborino');
    	     * // => 'hi-diddly-ho there, neighbo...'
    	     *
    	     * _.truncate('hi-diddly-ho there, neighborino', {
    	     *   'length': 24,
    	     *   'separator': ' '
    	     * });
    	     * // => 'hi-diddly-ho there,...'
    	     *
    	     * _.truncate('hi-diddly-ho there, neighborino', {
    	     *   'length': 24,
    	     *   'separator': /,? +/
    	     * });
    	     * // => 'hi-diddly-ho there...'
    	     *
    	     * _.truncate('hi-diddly-ho there, neighborino', {
    	     *   'omission': ' [...]'
    	     * });
    	     * // => 'hi-diddly-ho there, neig [...]'
    	     */function truncate(string,options){var length=DEFAULT_TRUNC_LENGTH,omission=DEFAULT_TRUNC_OMISSION;if(isObject(options)){var separator='separator'in options?options.separator:separator;length='length'in options?toInteger(options.length):length;omission='omission'in options?baseToString(options.omission):omission;}string=toString(string);var strLength=string.length;if(hasUnicode(string)){var strSymbols=stringToArray(string);strLength=strSymbols.length;}if(length>=strLength){return string;}var end=length-stringSize(omission);if(end<1){return omission;}var result=strSymbols?castSlice(strSymbols,0,end).join(''):string.slice(0,end);if(separator===undefined$1){return result+omission;}if(strSymbols){end+=result.length-end;}if(isRegExp(separator)){if(string.slice(end).search(separator)){var match,substring=result;if(!separator.global){separator=RegExp(separator.source,toString(reFlags.exec(separator))+'g');}separator.lastIndex=0;while(match=separator.exec(substring)){var newEnd=match.index;}result=result.slice(0,newEnd===undefined$1?end:newEnd);}}else if(string.indexOf(baseToString(separator),end)!=end){var index=result.lastIndexOf(separator);if(index>-1){result=result.slice(0,index);}}return result+omission;}/**
    	     * The inverse of `_.escape`; this method converts the HTML entities
    	     * `&amp;`, `&lt;`, `&gt;`, `&quot;`, and `&#39;` in `string` to
    	     * their corresponding characters.
    	     *
    	     * **Note:** No other HTML entities are unescaped. To unescape additional
    	     * HTML entities use a third-party library like [_he_](https://mths.be/he).
    	     *
    	     * @static
    	     * @memberOf _
    	     * @since 0.6.0
    	     * @category String
    	     * @param {string} [string=''] The string to unescape.
    	     * @returns {string} Returns the unescaped string.
    	     * @example
    	     *
    	     * _.unescape('fred, barney, &amp; pebbles');
    	     * // => 'fred, barney, & pebbles'
    	     */function unescape(string){string=toString(string);return string&&reHasEscapedHtml.test(string)?string.replace(reEscapedHtml,unescapeHtmlChar):string;}/**
    	     * Converts `string`, as space separated words, to upper case.
    	     *
    	     * @static
    	     * @memberOf _
    	     * @since 4.0.0
    	     * @category String
    	     * @param {string} [string=''] The string to convert.
    	     * @returns {string} Returns the upper cased string.
    	     * @example
    	     *
    	     * _.upperCase('--foo-bar');
    	     * // => 'FOO BAR'
    	     *
    	     * _.upperCase('fooBar');
    	     * // => 'FOO BAR'
    	     *
    	     * _.upperCase('__foo_bar__');
    	     * // => 'FOO BAR'
    	     */var upperCase=createCompounder(function(result,word,index){return result+(index?' ':'')+word.toUpperCase();});/**
    	     * Converts the first character of `string` to upper case.
    	     *
    	     * @static
    	     * @memberOf _
    	     * @since 4.0.0
    	     * @category String
    	     * @param {string} [string=''] The string to convert.
    	     * @returns {string} Returns the converted string.
    	     * @example
    	     *
    	     * _.upperFirst('fred');
    	     * // => 'Fred'
    	     *
    	     * _.upperFirst('FRED');
    	     * // => 'FRED'
    	     */var upperFirst=createCaseFirst('toUpperCase');/**
    	     * Splits `string` into an array of its words.
    	     *
    	     * @static
    	     * @memberOf _
    	     * @since 3.0.0
    	     * @category String
    	     * @param {string} [string=''] The string to inspect.
    	     * @param {RegExp|string} [pattern] The pattern to match words.
    	     * @param- {Object} [guard] Enables use as an iteratee for methods like `_.map`.
    	     * @returns {Array} Returns the words of `string`.
    	     * @example
    	     *
    	     * _.words('fred, barney, & pebbles');
    	     * // => ['fred', 'barney', 'pebbles']
    	     *
    	     * _.words('fred, barney, & pebbles', /[^, ]+/g);
    	     * // => ['fred', 'barney', '&', 'pebbles']
    	     */function words(string,pattern,guard){string=toString(string);pattern=guard?undefined$1:pattern;if(pattern===undefined$1){return hasUnicodeWord(string)?unicodeWords(string):asciiWords(string);}return string.match(pattern)||[];}/*------------------------------------------------------------------------*//**
    	     * Attempts to invoke `func`, returning either the result or the caught error
    	     * object. Any additional arguments are provided to `func` when it's invoked.
    	     *
    	     * @static
    	     * @memberOf _
    	     * @since 3.0.0
    	     * @category Util
    	     * @param {Function} func The function to attempt.
    	     * @param {...*} [args] The arguments to invoke `func` with.
    	     * @returns {*} Returns the `func` result or error object.
    	     * @example
    	     *
    	     * // Avoid throwing errors for invalid selectors.
    	     * var elements = _.attempt(function(selector) {
    	     *   return document.querySelectorAll(selector);
    	     * }, '>_>');
    	     *
    	     * if (_.isError(elements)) {
    	     *   elements = [];
    	     * }
    	     */var attempt=baseRest(function(func,args){try{return apply(func,undefined$1,args);}catch(e){return isError(e)?e:new Error(e);}});/**
    	     * Binds methods of an object to the object itself, overwriting the existing
    	     * method.
    	     *
    	     * **Note:** This method doesn't set the "length" property of bound functions.
    	     *
    	     * @static
    	     * @since 0.1.0
    	     * @memberOf _
    	     * @category Util
    	     * @param {Object} object The object to bind and assign the bound methods to.
    	     * @param {...(string|string[])} methodNames The object method names to bind.
    	     * @returns {Object} Returns `object`.
    	     * @example
    	     *
    	     * var view = {
    	     *   'label': 'docs',
    	     *   'click': function() {
    	     *     console.log('clicked ' + this.label);
    	     *   }
    	     * };
    	     *
    	     * _.bindAll(view, ['click']);
    	     * jQuery(element).on('click', view.click);
    	     * // => Logs 'clicked docs' when clicked.
    	     */var bindAll=flatRest(function(object,methodNames){arrayEach(methodNames,function(key){key=toKey(key);baseAssignValue(object,key,bind(object[key],object));});return object;});/**
    	     * Creates a function that iterates over `pairs` and invokes the corresponding
    	     * function of the first predicate to return truthy. The predicate-function
    	     * pairs are invoked with the `this` binding and arguments of the created
    	     * function.
    	     *
    	     * @static
    	     * @memberOf _
    	     * @since 4.0.0
    	     * @category Util
    	     * @param {Array} pairs The predicate-function pairs.
    	     * @returns {Function} Returns the new composite function.
    	     * @example
    	     *
    	     * var func = _.cond([
    	     *   [_.matches({ 'a': 1 }),           _.constant('matches A')],
    	     *   [_.conforms({ 'b': _.isNumber }), _.constant('matches B')],
    	     *   [_.stubTrue,                      _.constant('no match')]
    	     * ]);
    	     *
    	     * func({ 'a': 1, 'b': 2 });
    	     * // => 'matches A'
    	     *
    	     * func({ 'a': 0, 'b': 1 });
    	     * // => 'matches B'
    	     *
    	     * func({ 'a': '1', 'b': '2' });
    	     * // => 'no match'
    	     */function cond(pairs){var length=pairs==null?0:pairs.length,toIteratee=getIteratee();pairs=!length?[]:arrayMap(pairs,function(pair){if(typeof pair[1]!='function'){throw new TypeError(FUNC_ERROR_TEXT);}return [toIteratee(pair[0]),pair[1]];});return baseRest(function(args){var index=-1;while(++index<length){var pair=pairs[index];if(apply(pair[0],this,args)){return apply(pair[1],this,args);}}});}/**
    	     * Creates a function that invokes the predicate properties of `source` with
    	     * the corresponding property values of a given object, returning `true` if
    	     * all predicates return truthy, else `false`.
    	     *
    	     * **Note:** The created function is equivalent to `_.conformsTo` with
    	     * `source` partially applied.
    	     *
    	     * @static
    	     * @memberOf _
    	     * @since 4.0.0
    	     * @category Util
    	     * @param {Object} source The object of property predicates to conform to.
    	     * @returns {Function} Returns the new spec function.
    	     * @example
    	     *
    	     * var objects = [
    	     *   { 'a': 2, 'b': 1 },
    	     *   { 'a': 1, 'b': 2 }
    	     * ];
    	     *
    	     * _.filter(objects, _.conforms({ 'b': function(n) { return n > 1; } }));
    	     * // => [{ 'a': 1, 'b': 2 }]
    	     */function conforms(source){return baseConforms(baseClone(source,CLONE_DEEP_FLAG));}/**
    	     * Creates a function that returns `value`.
    	     *
    	     * @static
    	     * @memberOf _
    	     * @since 2.4.0
    	     * @category Util
    	     * @param {*} value The value to return from the new function.
    	     * @returns {Function} Returns the new constant function.
    	     * @example
    	     *
    	     * var objects = _.times(2, _.constant({ 'a': 1 }));
    	     *
    	     * console.log(objects);
    	     * // => [{ 'a': 1 }, { 'a': 1 }]
    	     *
    	     * console.log(objects[0] === objects[1]);
    	     * // => true
    	     */function constant(value){return function(){return value;};}/**
    	     * Checks `value` to determine whether a default value should be returned in
    	     * its place. The `defaultValue` is returned if `value` is `NaN`, `null`,
    	     * or `undefined`.
    	     *
    	     * @static
    	     * @memberOf _
    	     * @since 4.14.0
    	     * @category Util
    	     * @param {*} value The value to check.
    	     * @param {*} defaultValue The default value.
    	     * @returns {*} Returns the resolved value.
    	     * @example
    	     *
    	     * _.defaultTo(1, 10);
    	     * // => 1
    	     *
    	     * _.defaultTo(undefined, 10);
    	     * // => 10
    	     */function defaultTo(value,defaultValue){return value==null||value!==value?defaultValue:value;}/**
    	     * Creates a function that returns the result of invoking the given functions
    	     * with the `this` binding of the created function, where each successive
    	     * invocation is supplied the return value of the previous.
    	     *
    	     * @static
    	     * @memberOf _
    	     * @since 3.0.0
    	     * @category Util
    	     * @param {...(Function|Function[])} [funcs] The functions to invoke.
    	     * @returns {Function} Returns the new composite function.
    	     * @see _.flowRight
    	     * @example
    	     *
    	     * function square(n) {
    	     *   return n * n;
    	     * }
    	     *
    	     * var addSquare = _.flow([_.add, square]);
    	     * addSquare(1, 2);
    	     * // => 9
    	     */var flow=createFlow();/**
    	     * This method is like `_.flow` except that it creates a function that
    	     * invokes the given functions from right to left.
    	     *
    	     * @static
    	     * @since 3.0.0
    	     * @memberOf _
    	     * @category Util
    	     * @param {...(Function|Function[])} [funcs] The functions to invoke.
    	     * @returns {Function} Returns the new composite function.
    	     * @see _.flow
    	     * @example
    	     *
    	     * function square(n) {
    	     *   return n * n;
    	     * }
    	     *
    	     * var addSquare = _.flowRight([square, _.add]);
    	     * addSquare(1, 2);
    	     * // => 9
    	     */var flowRight=createFlow(true);/**
    	     * This method returns the first argument it receives.
    	     *
    	     * @static
    	     * @since 0.1.0
    	     * @memberOf _
    	     * @category Util
    	     * @param {*} value Any value.
    	     * @returns {*} Returns `value`.
    	     * @example
    	     *
    	     * var object = { 'a': 1 };
    	     *
    	     * console.log(_.identity(object) === object);
    	     * // => true
    	     */function identity(value){return value;}/**
    	     * Creates a function that invokes `func` with the arguments of the created
    	     * function. If `func` is a property name, the created function returns the
    	     * property value for a given element. If `func` is an array or object, the
    	     * created function returns `true` for elements that contain the equivalent
    	     * source properties, otherwise it returns `false`.
    	     *
    	     * @static
    	     * @since 4.0.0
    	     * @memberOf _
    	     * @category Util
    	     * @param {*} [func=_.identity] The value to convert to a callback.
    	     * @returns {Function} Returns the callback.
    	     * @example
    	     *
    	     * var users = [
    	     *   { 'user': 'barney', 'age': 36, 'active': true },
    	     *   { 'user': 'fred',   'age': 40, 'active': false }
    	     * ];
    	     *
    	     * // The `_.matches` iteratee shorthand.
    	     * _.filter(users, _.iteratee({ 'user': 'barney', 'active': true }));
    	     * // => [{ 'user': 'barney', 'age': 36, 'active': true }]
    	     *
    	     * // The `_.matchesProperty` iteratee shorthand.
    	     * _.filter(users, _.iteratee(['user', 'fred']));
    	     * // => [{ 'user': 'fred', 'age': 40 }]
    	     *
    	     * // The `_.property` iteratee shorthand.
    	     * _.map(users, _.iteratee('user'));
    	     * // => ['barney', 'fred']
    	     *
    	     * // Create custom iteratee shorthands.
    	     * _.iteratee = _.wrap(_.iteratee, function(iteratee, func) {
    	     *   return !_.isRegExp(func) ? iteratee(func) : function(string) {
    	     *     return func.test(string);
    	     *   };
    	     * });
    	     *
    	     * _.filter(['abc', 'def'], /ef/);
    	     * // => ['def']
    	     */function iteratee(func){return baseIteratee(typeof func=='function'?func:baseClone(func,CLONE_DEEP_FLAG));}/**
    	     * Creates a function that performs a partial deep comparison between a given
    	     * object and `source`, returning `true` if the given object has equivalent
    	     * property values, else `false`.
    	     *
    	     * **Note:** The created function is equivalent to `_.isMatch` with `source`
    	     * partially applied.
    	     *
    	     * Partial comparisons will match empty array and empty object `source`
    	     * values against any array or object value, respectively. See `_.isEqual`
    	     * for a list of supported value comparisons.
    	     *
    	     * **Note:** Multiple values can be checked by combining several matchers
    	     * using `_.overSome`
    	     *
    	     * @static
    	     * @memberOf _
    	     * @since 3.0.0
    	     * @category Util
    	     * @param {Object} source The object of property values to match.
    	     * @returns {Function} Returns the new spec function.
    	     * @example
    	     *
    	     * var objects = [
    	     *   { 'a': 1, 'b': 2, 'c': 3 },
    	     *   { 'a': 4, 'b': 5, 'c': 6 }
    	     * ];
    	     *
    	     * _.filter(objects, _.matches({ 'a': 4, 'c': 6 }));
    	     * // => [{ 'a': 4, 'b': 5, 'c': 6 }]
    	     *
    	     * // Checking for several possible values
    	     * _.filter(objects, _.overSome([_.matches({ 'a': 1 }), _.matches({ 'a': 4 })]));
    	     * // => [{ 'a': 1, 'b': 2, 'c': 3 }, { 'a': 4, 'b': 5, 'c': 6 }]
    	     */function matches(source){return baseMatches(baseClone(source,CLONE_DEEP_FLAG));}/**
    	     * Creates a function that performs a partial deep comparison between the
    	     * value at `path` of a given object to `srcValue`, returning `true` if the
    	     * object value is equivalent, else `false`.
    	     *
    	     * **Note:** Partial comparisons will match empty array and empty object
    	     * `srcValue` values against any array or object value, respectively. See
    	     * `_.isEqual` for a list of supported value comparisons.
    	     *
    	     * **Note:** Multiple values can be checked by combining several matchers
    	     * using `_.overSome`
    	     *
    	     * @static
    	     * @memberOf _
    	     * @since 3.2.0
    	     * @category Util
    	     * @param {Array|string} path The path of the property to get.
    	     * @param {*} srcValue The value to match.
    	     * @returns {Function} Returns the new spec function.
    	     * @example
    	     *
    	     * var objects = [
    	     *   { 'a': 1, 'b': 2, 'c': 3 },
    	     *   { 'a': 4, 'b': 5, 'c': 6 }
    	     * ];
    	     *
    	     * _.find(objects, _.matchesProperty('a', 4));
    	     * // => { 'a': 4, 'b': 5, 'c': 6 }
    	     *
    	     * // Checking for several possible values
    	     * _.filter(objects, _.overSome([_.matchesProperty('a', 1), _.matchesProperty('a', 4)]));
    	     * // => [{ 'a': 1, 'b': 2, 'c': 3 }, { 'a': 4, 'b': 5, 'c': 6 }]
    	     */function matchesProperty(path,srcValue){return baseMatchesProperty(path,baseClone(srcValue,CLONE_DEEP_FLAG));}/**
    	     * Creates a function that invokes the method at `path` of a given object.
    	     * Any additional arguments are provided to the invoked method.
    	     *
    	     * @static
    	     * @memberOf _
    	     * @since 3.7.0
    	     * @category Util
    	     * @param {Array|string} path The path of the method to invoke.
    	     * @param {...*} [args] The arguments to invoke the method with.
    	     * @returns {Function} Returns the new invoker function.
    	     * @example
    	     *
    	     * var objects = [
    	     *   { 'a': { 'b': _.constant(2) } },
    	     *   { 'a': { 'b': _.constant(1) } }
    	     * ];
    	     *
    	     * _.map(objects, _.method('a.b'));
    	     * // => [2, 1]
    	     *
    	     * _.map(objects, _.method(['a', 'b']));
    	     * // => [2, 1]
    	     */var method=baseRest(function(path,args){return function(object){return baseInvoke(object,path,args);};});/**
    	     * The opposite of `_.method`; this method creates a function that invokes
    	     * the method at a given path of `object`. Any additional arguments are
    	     * provided to the invoked method.
    	     *
    	     * @static
    	     * @memberOf _
    	     * @since 3.7.0
    	     * @category Util
    	     * @param {Object} object The object to query.
    	     * @param {...*} [args] The arguments to invoke the method with.
    	     * @returns {Function} Returns the new invoker function.
    	     * @example
    	     *
    	     * var array = _.times(3, _.constant),
    	     *     object = { 'a': array, 'b': array, 'c': array };
    	     *
    	     * _.map(['a[2]', 'c[0]'], _.methodOf(object));
    	     * // => [2, 0]
    	     *
    	     * _.map([['a', '2'], ['c', '0']], _.methodOf(object));
    	     * // => [2, 0]
    	     */var methodOf=baseRest(function(object,args){return function(path){return baseInvoke(object,path,args);};});/**
    	     * Adds all own enumerable string keyed function properties of a source
    	     * object to the destination object. If `object` is a function, then methods
    	     * are added to its prototype as well.
    	     *
    	     * **Note:** Use `_.runInContext` to create a pristine `lodash` function to
    	     * avoid conflicts caused by modifying the original.
    	     *
    	     * @static
    	     * @since 0.1.0
    	     * @memberOf _
    	     * @category Util
    	     * @param {Function|Object} [object=lodash] The destination object.
    	     * @param {Object} source The object of functions to add.
    	     * @param {Object} [options={}] The options object.
    	     * @param {boolean} [options.chain=true] Specify whether mixins are chainable.
    	     * @returns {Function|Object} Returns `object`.
    	     * @example
    	     *
    	     * function vowels(string) {
    	     *   return _.filter(string, function(v) {
    	     *     return /[aeiou]/i.test(v);
    	     *   });
    	     * }
    	     *
    	     * _.mixin({ 'vowels': vowels });
    	     * _.vowels('fred');
    	     * // => ['e']
    	     *
    	     * _('fred').vowels().value();
    	     * // => ['e']
    	     *
    	     * _.mixin({ 'vowels': vowels }, { 'chain': false });
    	     * _('fred').vowels();
    	     * // => ['e']
    	     */function mixin(object,source,options){var props=keys(source),methodNames=baseFunctions(source,props);if(options==null&&!(isObject(source)&&(methodNames.length||!props.length))){options=source;source=object;object=this;methodNames=baseFunctions(source,keys(source));}var chain=!(isObject(options)&&'chain'in options)||!!options.chain,isFunc=isFunction(object);arrayEach(methodNames,function(methodName){var func=source[methodName];object[methodName]=func;if(isFunc){object.prototype[methodName]=function(){var chainAll=this.__chain__;if(chain||chainAll){var result=object(this.__wrapped__),actions=result.__actions__=copyArray(this.__actions__);actions.push({'func':func,'args':arguments,'thisArg':object});result.__chain__=chainAll;return result;}return func.apply(object,arrayPush([this.value()],arguments));};}});return object;}/**
    	     * Reverts the `_` variable to its previous value and returns a reference to
    	     * the `lodash` function.
    	     *
    	     * @static
    	     * @since 0.1.0
    	     * @memberOf _
    	     * @category Util
    	     * @returns {Function} Returns the `lodash` function.
    	     * @example
    	     *
    	     * var lodash = _.noConflict();
    	     */function noConflict(){if(root._===this){root._=oldDash;}return this;}/**
    	     * This method returns `undefined`.
    	     *
    	     * @static
    	     * @memberOf _
    	     * @since 2.3.0
    	     * @category Util
    	     * @example
    	     *
    	     * _.times(2, _.noop);
    	     * // => [undefined, undefined]
    	     */function noop(){// No operation performed.
    	}/**
    	     * Creates a function that gets the argument at index `n`. If `n` is negative,
    	     * the nth argument from the end is returned.
    	     *
    	     * @static
    	     * @memberOf _
    	     * @since 4.0.0
    	     * @category Util
    	     * @param {number} [n=0] The index of the argument to return.
    	     * @returns {Function} Returns the new pass-thru function.
    	     * @example
    	     *
    	     * var func = _.nthArg(1);
    	     * func('a', 'b', 'c', 'd');
    	     * // => 'b'
    	     *
    	     * var func = _.nthArg(-2);
    	     * func('a', 'b', 'c', 'd');
    	     * // => 'c'
    	     */function nthArg(n){n=toInteger(n);return baseRest(function(args){return baseNth(args,n);});}/**
    	     * Creates a function that invokes `iteratees` with the arguments it receives
    	     * and returns their results.
    	     *
    	     * @static
    	     * @memberOf _
    	     * @since 4.0.0
    	     * @category Util
    	     * @param {...(Function|Function[])} [iteratees=[_.identity]]
    	     *  The iteratees to invoke.
    	     * @returns {Function} Returns the new function.
    	     * @example
    	     *
    	     * var func = _.over([Math.max, Math.min]);
    	     *
    	     * func(1, 2, 3, 4);
    	     * // => [4, 1]
    	     */var over=createOver(arrayMap);/**
    	     * Creates a function that checks if **all** of the `predicates` return
    	     * truthy when invoked with the arguments it receives.
    	     *
    	     * Following shorthands are possible for providing predicates.
    	     * Pass an `Object` and it will be used as an parameter for `_.matches` to create the predicate.
    	     * Pass an `Array` of parameters for `_.matchesProperty` and the predicate will be created using them.
    	     *
    	     * @static
    	     * @memberOf _
    	     * @since 4.0.0
    	     * @category Util
    	     * @param {...(Function|Function[])} [predicates=[_.identity]]
    	     *  The predicates to check.
    	     * @returns {Function} Returns the new function.
    	     * @example
    	     *
    	     * var func = _.overEvery([Boolean, isFinite]);
    	     *
    	     * func('1');
    	     * // => true
    	     *
    	     * func(null);
    	     * // => false
    	     *
    	     * func(NaN);
    	     * // => false
    	     */var overEvery=createOver(arrayEvery);/**
    	     * Creates a function that checks if **any** of the `predicates` return
    	     * truthy when invoked with the arguments it receives.
    	     *
    	     * Following shorthands are possible for providing predicates.
    	     * Pass an `Object` and it will be used as an parameter for `_.matches` to create the predicate.
    	     * Pass an `Array` of parameters for `_.matchesProperty` and the predicate will be created using them.
    	     *
    	     * @static
    	     * @memberOf _
    	     * @since 4.0.0
    	     * @category Util
    	     * @param {...(Function|Function[])} [predicates=[_.identity]]
    	     *  The predicates to check.
    	     * @returns {Function} Returns the new function.
    	     * @example
    	     *
    	     * var func = _.overSome([Boolean, isFinite]);
    	     *
    	     * func('1');
    	     * // => true
    	     *
    	     * func(null);
    	     * // => true
    	     *
    	     * func(NaN);
    	     * // => false
    	     *
    	     * var matchesFunc = _.overSome([{ 'a': 1 }, { 'a': 2 }])
    	     * var matchesPropertyFunc = _.overSome([['a', 1], ['a', 2]])
    	     */var overSome=createOver(arraySome);/**
    	     * Creates a function that returns the value at `path` of a given object.
    	     *
    	     * @static
    	     * @memberOf _
    	     * @since 2.4.0
    	     * @category Util
    	     * @param {Array|string} path The path of the property to get.
    	     * @returns {Function} Returns the new accessor function.
    	     * @example
    	     *
    	     * var objects = [
    	     *   { 'a': { 'b': 2 } },
    	     *   { 'a': { 'b': 1 } }
    	     * ];
    	     *
    	     * _.map(objects, _.property('a.b'));
    	     * // => [2, 1]
    	     *
    	     * _.map(_.sortBy(objects, _.property(['a', 'b'])), 'a.b');
    	     * // => [1, 2]
    	     */function property(path){return isKey(path)?baseProperty(toKey(path)):basePropertyDeep(path);}/**
    	     * The opposite of `_.property`; this method creates a function that returns
    	     * the value at a given path of `object`.
    	     *
    	     * @static
    	     * @memberOf _
    	     * @since 3.0.0
    	     * @category Util
    	     * @param {Object} object The object to query.
    	     * @returns {Function} Returns the new accessor function.
    	     * @example
    	     *
    	     * var array = [0, 1, 2],
    	     *     object = { 'a': array, 'b': array, 'c': array };
    	     *
    	     * _.map(['a[2]', 'c[0]'], _.propertyOf(object));
    	     * // => [2, 0]
    	     *
    	     * _.map([['a', '2'], ['c', '0']], _.propertyOf(object));
    	     * // => [2, 0]
    	     */function propertyOf(object){return function(path){return object==null?undefined$1:baseGet(object,path);};}/**
    	     * Creates an array of numbers (positive and/or negative) progressing from
    	     * `start` up to, but not including, `end`. A step of `-1` is used if a negative
    	     * `start` is specified without an `end` or `step`. If `end` is not specified,
    	     * it's set to `start` with `start` then set to `0`.
    	     *
    	     * **Note:** JavaScript follows the IEEE-754 standard for resolving
    	     * floating-point values which can produce unexpected results.
    	     *
    	     * @static
    	     * @since 0.1.0
    	     * @memberOf _
    	     * @category Util
    	     * @param {number} [start=0] The start of the range.
    	     * @param {number} end The end of the range.
    	     * @param {number} [step=1] The value to increment or decrement by.
    	     * @returns {Array} Returns the range of numbers.
    	     * @see _.inRange, _.rangeRight
    	     * @example
    	     *
    	     * _.range(4);
    	     * // => [0, 1, 2, 3]
    	     *
    	     * _.range(-4);
    	     * // => [0, -1, -2, -3]
    	     *
    	     * _.range(1, 5);
    	     * // => [1, 2, 3, 4]
    	     *
    	     * _.range(0, 20, 5);
    	     * // => [0, 5, 10, 15]
    	     *
    	     * _.range(0, -4, -1);
    	     * // => [0, -1, -2, -3]
    	     *
    	     * _.range(1, 4, 0);
    	     * // => [1, 1, 1]
    	     *
    	     * _.range(0);
    	     * // => []
    	     */var range=createRange();/**
    	     * This method is like `_.range` except that it populates values in
    	     * descending order.
    	     *
    	     * @static
    	     * @memberOf _
    	     * @since 4.0.0
    	     * @category Util
    	     * @param {number} [start=0] The start of the range.
    	     * @param {number} end The end of the range.
    	     * @param {number} [step=1] The value to increment or decrement by.
    	     * @returns {Array} Returns the range of numbers.
    	     * @see _.inRange, _.range
    	     * @example
    	     *
    	     * _.rangeRight(4);
    	     * // => [3, 2, 1, 0]
    	     *
    	     * _.rangeRight(-4);
    	     * // => [-3, -2, -1, 0]
    	     *
    	     * _.rangeRight(1, 5);
    	     * // => [4, 3, 2, 1]
    	     *
    	     * _.rangeRight(0, 20, 5);
    	     * // => [15, 10, 5, 0]
    	     *
    	     * _.rangeRight(0, -4, -1);
    	     * // => [-3, -2, -1, 0]
    	     *
    	     * _.rangeRight(1, 4, 0);
    	     * // => [1, 1, 1]
    	     *
    	     * _.rangeRight(0);
    	     * // => []
    	     */var rangeRight=createRange(true);/**
    	     * This method returns a new empty array.
    	     *
    	     * @static
    	     * @memberOf _
    	     * @since 4.13.0
    	     * @category Util
    	     * @returns {Array} Returns the new empty array.
    	     * @example
    	     *
    	     * var arrays = _.times(2, _.stubArray);
    	     *
    	     * console.log(arrays);
    	     * // => [[], []]
    	     *
    	     * console.log(arrays[0] === arrays[1]);
    	     * // => false
    	     */function stubArray(){return [];}/**
    	     * This method returns `false`.
    	     *
    	     * @static
    	     * @memberOf _
    	     * @since 4.13.0
    	     * @category Util
    	     * @returns {boolean} Returns `false`.
    	     * @example
    	     *
    	     * _.times(2, _.stubFalse);
    	     * // => [false, false]
    	     */function stubFalse(){return false;}/**
    	     * This method returns a new empty object.
    	     *
    	     * @static
    	     * @memberOf _
    	     * @since 4.13.0
    	     * @category Util
    	     * @returns {Object} Returns the new empty object.
    	     * @example
    	     *
    	     * var objects = _.times(2, _.stubObject);
    	     *
    	     * console.log(objects);
    	     * // => [{}, {}]
    	     *
    	     * console.log(objects[0] === objects[1]);
    	     * // => false
    	     */function stubObject(){return {};}/**
    	     * This method returns an empty string.
    	     *
    	     * @static
    	     * @memberOf _
    	     * @since 4.13.0
    	     * @category Util
    	     * @returns {string} Returns the empty string.
    	     * @example
    	     *
    	     * _.times(2, _.stubString);
    	     * // => ['', '']
    	     */function stubString(){return '';}/**
    	     * This method returns `true`.
    	     *
    	     * @static
    	     * @memberOf _
    	     * @since 4.13.0
    	     * @category Util
    	     * @returns {boolean} Returns `true`.
    	     * @example
    	     *
    	     * _.times(2, _.stubTrue);
    	     * // => [true, true]
    	     */function stubTrue(){return true;}/**
    	     * Invokes the iteratee `n` times, returning an array of the results of
    	     * each invocation. The iteratee is invoked with one argument; (index).
    	     *
    	     * @static
    	     * @since 0.1.0
    	     * @memberOf _
    	     * @category Util
    	     * @param {number} n The number of times to invoke `iteratee`.
    	     * @param {Function} [iteratee=_.identity] The function invoked per iteration.
    	     * @returns {Array} Returns the array of results.
    	     * @example
    	     *
    	     * _.times(3, String);
    	     * // => ['0', '1', '2']
    	     *
    	     *  _.times(4, _.constant(0));
    	     * // => [0, 0, 0, 0]
    	     */function times(n,iteratee){n=toInteger(n);if(n<1||n>MAX_SAFE_INTEGER){return [];}var index=MAX_ARRAY_LENGTH,length=nativeMin(n,MAX_ARRAY_LENGTH);iteratee=getIteratee(iteratee);n-=MAX_ARRAY_LENGTH;var result=baseTimes(length,iteratee);while(++index<n){iteratee(index);}return result;}/**
    	     * Converts `value` to a property path array.
    	     *
    	     * @static
    	     * @memberOf _
    	     * @since 4.0.0
    	     * @category Util
    	     * @param {*} value The value to convert.
    	     * @returns {Array} Returns the new property path array.
    	     * @example
    	     *
    	     * _.toPath('a.b.c');
    	     * // => ['a', 'b', 'c']
    	     *
    	     * _.toPath('a[0].b.c');
    	     * // => ['a', '0', 'b', 'c']
    	     */function toPath(value){if(isArray(value)){return arrayMap(value,toKey);}return isSymbol(value)?[value]:copyArray(stringToPath(toString(value)));}/**
    	     * Generates a unique ID. If `prefix` is given, the ID is appended to it.
    	     *
    	     * @static
    	     * @since 0.1.0
    	     * @memberOf _
    	     * @category Util
    	     * @param {string} [prefix=''] The value to prefix the ID with.
    	     * @returns {string} Returns the unique ID.
    	     * @example
    	     *
    	     * _.uniqueId('contact_');
    	     * // => 'contact_104'
    	     *
    	     * _.uniqueId();
    	     * // => '105'
    	     */function uniqueId(prefix){var id=++idCounter;return toString(prefix)+id;}/*------------------------------------------------------------------------*//**
    	     * Adds two numbers.
    	     *
    	     * @static
    	     * @memberOf _
    	     * @since 3.4.0
    	     * @category Math
    	     * @param {number} augend The first number in an addition.
    	     * @param {number} addend The second number in an addition.
    	     * @returns {number} Returns the total.
    	     * @example
    	     *
    	     * _.add(6, 4);
    	     * // => 10
    	     */var add=createMathOperation(function(augend,addend){return augend+addend;},0);/**
    	     * Computes `number` rounded up to `precision`.
    	     *
    	     * @static
    	     * @memberOf _
    	     * @since 3.10.0
    	     * @category Math
    	     * @param {number} number The number to round up.
    	     * @param {number} [precision=0] The precision to round up to.
    	     * @returns {number} Returns the rounded up number.
    	     * @example
    	     *
    	     * _.ceil(4.006);
    	     * // => 5
    	     *
    	     * _.ceil(6.004, 2);
    	     * // => 6.01
    	     *
    	     * _.ceil(6040, -2);
    	     * // => 6100
    	     */var ceil=createRound('ceil');/**
    	     * Divide two numbers.
    	     *
    	     * @static
    	     * @memberOf _
    	     * @since 4.7.0
    	     * @category Math
    	     * @param {number} dividend The first number in a division.
    	     * @param {number} divisor The second number in a division.
    	     * @returns {number} Returns the quotient.
    	     * @example
    	     *
    	     * _.divide(6, 4);
    	     * // => 1.5
    	     */var divide=createMathOperation(function(dividend,divisor){return dividend/divisor;},1);/**
    	     * Computes `number` rounded down to `precision`.
    	     *
    	     * @static
    	     * @memberOf _
    	     * @since 3.10.0
    	     * @category Math
    	     * @param {number} number The number to round down.
    	     * @param {number} [precision=0] The precision to round down to.
    	     * @returns {number} Returns the rounded down number.
    	     * @example
    	     *
    	     * _.floor(4.006);
    	     * // => 4
    	     *
    	     * _.floor(0.046, 2);
    	     * // => 0.04
    	     *
    	     * _.floor(4060, -2);
    	     * // => 4000
    	     */var floor=createRound('floor');/**
    	     * Computes the maximum value of `array`. If `array` is empty or falsey,
    	     * `undefined` is returned.
    	     *
    	     * @static
    	     * @since 0.1.0
    	     * @memberOf _
    	     * @category Math
    	     * @param {Array} array The array to iterate over.
    	     * @returns {*} Returns the maximum value.
    	     * @example
    	     *
    	     * _.max([4, 2, 8, 6]);
    	     * // => 8
    	     *
    	     * _.max([]);
    	     * // => undefined
    	     */function max(array){return array&&array.length?baseExtremum(array,identity,baseGt):undefined$1;}/**
    	     * This method is like `_.max` except that it accepts `iteratee` which is
    	     * invoked for each element in `array` to generate the criterion by which
    	     * the value is ranked. The iteratee is invoked with one argument: (value).
    	     *
    	     * @static
    	     * @memberOf _
    	     * @since 4.0.0
    	     * @category Math
    	     * @param {Array} array The array to iterate over.
    	     * @param {Function} [iteratee=_.identity] The iteratee invoked per element.
    	     * @returns {*} Returns the maximum value.
    	     * @example
    	     *
    	     * var objects = [{ 'n': 1 }, { 'n': 2 }];
    	     *
    	     * _.maxBy(objects, function(o) { return o.n; });
    	     * // => { 'n': 2 }
    	     *
    	     * // The `_.property` iteratee shorthand.
    	     * _.maxBy(objects, 'n');
    	     * // => { 'n': 2 }
    	     */function maxBy(array,iteratee){return array&&array.length?baseExtremum(array,getIteratee(iteratee,2),baseGt):undefined$1;}/**
    	     * Computes the mean of the values in `array`.
    	     *
    	     * @static
    	     * @memberOf _
    	     * @since 4.0.0
    	     * @category Math
    	     * @param {Array} array The array to iterate over.
    	     * @returns {number} Returns the mean.
    	     * @example
    	     *
    	     * _.mean([4, 2, 8, 6]);
    	     * // => 5
    	     */function mean(array){return baseMean(array,identity);}/**
    	     * This method is like `_.mean` except that it accepts `iteratee` which is
    	     * invoked for each element in `array` to generate the value to be averaged.
    	     * The iteratee is invoked with one argument: (value).
    	     *
    	     * @static
    	     * @memberOf _
    	     * @since 4.7.0
    	     * @category Math
    	     * @param {Array} array The array to iterate over.
    	     * @param {Function} [iteratee=_.identity] The iteratee invoked per element.
    	     * @returns {number} Returns the mean.
    	     * @example
    	     *
    	     * var objects = [{ 'n': 4 }, { 'n': 2 }, { 'n': 8 }, { 'n': 6 }];
    	     *
    	     * _.meanBy(objects, function(o) { return o.n; });
    	     * // => 5
    	     *
    	     * // The `_.property` iteratee shorthand.
    	     * _.meanBy(objects, 'n');
    	     * // => 5
    	     */function meanBy(array,iteratee){return baseMean(array,getIteratee(iteratee,2));}/**
    	     * Computes the minimum value of `array`. If `array` is empty or falsey,
    	     * `undefined` is returned.
    	     *
    	     * @static
    	     * @since 0.1.0
    	     * @memberOf _
    	     * @category Math
    	     * @param {Array} array The array to iterate over.
    	     * @returns {*} Returns the minimum value.
    	     * @example
    	     *
    	     * _.min([4, 2, 8, 6]);
    	     * // => 2
    	     *
    	     * _.min([]);
    	     * // => undefined
    	     */function min(array){return array&&array.length?baseExtremum(array,identity,baseLt):undefined$1;}/**
    	     * This method is like `_.min` except that it accepts `iteratee` which is
    	     * invoked for each element in `array` to generate the criterion by which
    	     * the value is ranked. The iteratee is invoked with one argument: (value).
    	     *
    	     * @static
    	     * @memberOf _
    	     * @since 4.0.0
    	     * @category Math
    	     * @param {Array} array The array to iterate over.
    	     * @param {Function} [iteratee=_.identity] The iteratee invoked per element.
    	     * @returns {*} Returns the minimum value.
    	     * @example
    	     *
    	     * var objects = [{ 'n': 1 }, { 'n': 2 }];
    	     *
    	     * _.minBy(objects, function(o) { return o.n; });
    	     * // => { 'n': 1 }
    	     *
    	     * // The `_.property` iteratee shorthand.
    	     * _.minBy(objects, 'n');
    	     * // => { 'n': 1 }
    	     */function minBy(array,iteratee){return array&&array.length?baseExtremum(array,getIteratee(iteratee,2),baseLt):undefined$1;}/**
    	     * Multiply two numbers.
    	     *
    	     * @static
    	     * @memberOf _
    	     * @since 4.7.0
    	     * @category Math
    	     * @param {number} multiplier The first number in a multiplication.
    	     * @param {number} multiplicand The second number in a multiplication.
    	     * @returns {number} Returns the product.
    	     * @example
    	     *
    	     * _.multiply(6, 4);
    	     * // => 24
    	     */var multiply=createMathOperation(function(multiplier,multiplicand){return multiplier*multiplicand;},1);/**
    	     * Computes `number` rounded to `precision`.
    	     *
    	     * @static
    	     * @memberOf _
    	     * @since 3.10.0
    	     * @category Math
    	     * @param {number} number The number to round.
    	     * @param {number} [precision=0] The precision to round to.
    	     * @returns {number} Returns the rounded number.
    	     * @example
    	     *
    	     * _.round(4.006);
    	     * // => 4
    	     *
    	     * _.round(4.006, 2);
    	     * // => 4.01
    	     *
    	     * _.round(4060, -2);
    	     * // => 4100
    	     */var round=createRound('round');/**
    	     * Subtract two numbers.
    	     *
    	     * @static
    	     * @memberOf _
    	     * @since 4.0.0
    	     * @category Math
    	     * @param {number} minuend The first number in a subtraction.
    	     * @param {number} subtrahend The second number in a subtraction.
    	     * @returns {number} Returns the difference.
    	     * @example
    	     *
    	     * _.subtract(6, 4);
    	     * // => 2
    	     */var subtract=createMathOperation(function(minuend,subtrahend){return minuend-subtrahend;},0);/**
    	     * Computes the sum of the values in `array`.
    	     *
    	     * @static
    	     * @memberOf _
    	     * @since 3.4.0
    	     * @category Math
    	     * @param {Array} array The array to iterate over.
    	     * @returns {number} Returns the sum.
    	     * @example
    	     *
    	     * _.sum([4, 2, 8, 6]);
    	     * // => 20
    	     */function sum(array){return array&&array.length?baseSum(array,identity):0;}/**
    	     * This method is like `_.sum` except that it accepts `iteratee` which is
    	     * invoked for each element in `array` to generate the value to be summed.
    	     * The iteratee is invoked with one argument: (value).
    	     *
    	     * @static
    	     * @memberOf _
    	     * @since 4.0.0
    	     * @category Math
    	     * @param {Array} array The array to iterate over.
    	     * @param {Function} [iteratee=_.identity] The iteratee invoked per element.
    	     * @returns {number} Returns the sum.
    	     * @example
    	     *
    	     * var objects = [{ 'n': 4 }, { 'n': 2 }, { 'n': 8 }, { 'n': 6 }];
    	     *
    	     * _.sumBy(objects, function(o) { return o.n; });
    	     * // => 20
    	     *
    	     * // The `_.property` iteratee shorthand.
    	     * _.sumBy(objects, 'n');
    	     * // => 20
    	     */function sumBy(array,iteratee){return array&&array.length?baseSum(array,getIteratee(iteratee,2)):0;}/*------------------------------------------------------------------------*/// Add methods that return wrapped values in chain sequences.
    	lodash.after=after;lodash.ary=ary;lodash.assign=assign;lodash.assignIn=assignIn;lodash.assignInWith=assignInWith;lodash.assignWith=assignWith;lodash.at=at;lodash.before=before;lodash.bind=bind;lodash.bindAll=bindAll;lodash.bindKey=bindKey;lodash.castArray=castArray;lodash.chain=chain;lodash.chunk=chunk;lodash.compact=compact;lodash.concat=concat;lodash.cond=cond;lodash.conforms=conforms;lodash.constant=constant;lodash.countBy=countBy;lodash.create=create;lodash.curry=curry;lodash.curryRight=curryRight;lodash.debounce=debounce;lodash.defaults=defaults;lodash.defaultsDeep=defaultsDeep;lodash.defer=defer;lodash.delay=delay;lodash.difference=difference;lodash.differenceBy=differenceBy;lodash.differenceWith=differenceWith;lodash.drop=drop;lodash.dropRight=dropRight;lodash.dropRightWhile=dropRightWhile;lodash.dropWhile=dropWhile;lodash.fill=fill;lodash.filter=filter;lodash.flatMap=flatMap;lodash.flatMapDeep=flatMapDeep;lodash.flatMapDepth=flatMapDepth;lodash.flatten=flatten;lodash.flattenDeep=flattenDeep;lodash.flattenDepth=flattenDepth;lodash.flip=flip;lodash.flow=flow;lodash.flowRight=flowRight;lodash.fromPairs=fromPairs;lodash.functions=functions;lodash.functionsIn=functionsIn;lodash.groupBy=groupBy;lodash.initial=initial;lodash.intersection=intersection;lodash.intersectionBy=intersectionBy;lodash.intersectionWith=intersectionWith;lodash.invert=invert;lodash.invertBy=invertBy;lodash.invokeMap=invokeMap;lodash.iteratee=iteratee;lodash.keyBy=keyBy;lodash.keys=keys;lodash.keysIn=keysIn;lodash.map=map;lodash.mapKeys=mapKeys;lodash.mapValues=mapValues;lodash.matches=matches;lodash.matchesProperty=matchesProperty;lodash.memoize=memoize;lodash.merge=merge;lodash.mergeWith=mergeWith;lodash.method=method;lodash.methodOf=methodOf;lodash.mixin=mixin;lodash.negate=negate;lodash.nthArg=nthArg;lodash.omit=omit;lodash.omitBy=omitBy;lodash.once=once;lodash.orderBy=orderBy;lodash.over=over;lodash.overArgs=overArgs;lodash.overEvery=overEvery;lodash.overSome=overSome;lodash.partial=partial;lodash.partialRight=partialRight;lodash.partition=partition;lodash.pick=pick;lodash.pickBy=pickBy;lodash.property=property;lodash.propertyOf=propertyOf;lodash.pull=pull;lodash.pullAll=pullAll;lodash.pullAllBy=pullAllBy;lodash.pullAllWith=pullAllWith;lodash.pullAt=pullAt;lodash.range=range;lodash.rangeRight=rangeRight;lodash.rearg=rearg;lodash.reject=reject;lodash.remove=remove;lodash.rest=rest;lodash.reverse=reverse;lodash.sampleSize=sampleSize;lodash.set=set;lodash.setWith=setWith;lodash.shuffle=shuffle;lodash.slice=slice;lodash.sortBy=sortBy;lodash.sortedUniq=sortedUniq;lodash.sortedUniqBy=sortedUniqBy;lodash.split=split;lodash.spread=spread;lodash.tail=tail;lodash.take=take;lodash.takeRight=takeRight;lodash.takeRightWhile=takeRightWhile;lodash.takeWhile=takeWhile;lodash.tap=tap;lodash.throttle=throttle;lodash.thru=thru;lodash.toArray=toArray;lodash.toPairs=toPairs;lodash.toPairsIn=toPairsIn;lodash.toPath=toPath;lodash.toPlainObject=toPlainObject;lodash.transform=transform;lodash.unary=unary;lodash.union=union;lodash.unionBy=unionBy;lodash.unionWith=unionWith;lodash.uniq=uniq;lodash.uniqBy=uniqBy;lodash.uniqWith=uniqWith;lodash.unset=unset;lodash.unzip=unzip;lodash.unzipWith=unzipWith;lodash.update=update;lodash.updateWith=updateWith;lodash.values=values;lodash.valuesIn=valuesIn;lodash.without=without;lodash.words=words;lodash.wrap=wrap;lodash.xor=xor;lodash.xorBy=xorBy;lodash.xorWith=xorWith;lodash.zip=zip;lodash.zipObject=zipObject;lodash.zipObjectDeep=zipObjectDeep;lodash.zipWith=zipWith;// Add aliases.
    	lodash.entries=toPairs;lodash.entriesIn=toPairsIn;lodash.extend=assignIn;lodash.extendWith=assignInWith;// Add methods to `lodash.prototype`.
    	mixin(lodash,lodash);/*------------------------------------------------------------------------*/// Add methods that return unwrapped values in chain sequences.
    	lodash.add=add;lodash.attempt=attempt;lodash.camelCase=camelCase;lodash.capitalize=capitalize;lodash.ceil=ceil;lodash.clamp=clamp;lodash.clone=clone;lodash.cloneDeep=cloneDeep;lodash.cloneDeepWith=cloneDeepWith;lodash.cloneWith=cloneWith;lodash.conformsTo=conformsTo;lodash.deburr=deburr;lodash.defaultTo=defaultTo;lodash.divide=divide;lodash.endsWith=endsWith;lodash.eq=eq;lodash.escape=escape;lodash.escapeRegExp=escapeRegExp;lodash.every=every;lodash.find=find;lodash.findIndex=findIndex;lodash.findKey=findKey;lodash.findLast=findLast;lodash.findLastIndex=findLastIndex;lodash.findLastKey=findLastKey;lodash.floor=floor;lodash.forEach=forEach;lodash.forEachRight=forEachRight;lodash.forIn=forIn;lodash.forInRight=forInRight;lodash.forOwn=forOwn;lodash.forOwnRight=forOwnRight;lodash.get=get;lodash.gt=gt;lodash.gte=gte;lodash.has=has;lodash.hasIn=hasIn;lodash.head=head;lodash.identity=identity;lodash.includes=includes;lodash.indexOf=indexOf;lodash.inRange=inRange;lodash.invoke=invoke;lodash.isArguments=isArguments;lodash.isArray=isArray;lodash.isArrayBuffer=isArrayBuffer;lodash.isArrayLike=isArrayLike;lodash.isArrayLikeObject=isArrayLikeObject;lodash.isBoolean=isBoolean;lodash.isBuffer=isBuffer;lodash.isDate=isDate;lodash.isElement=isElement;lodash.isEmpty=isEmpty;lodash.isEqual=isEqual;lodash.isEqualWith=isEqualWith;lodash.isError=isError;lodash.isFinite=isFinite;lodash.isFunction=isFunction;lodash.isInteger=isInteger;lodash.isLength=isLength;lodash.isMap=isMap;lodash.isMatch=isMatch;lodash.isMatchWith=isMatchWith;lodash.isNaN=isNaN;lodash.isNative=isNative;lodash.isNil=isNil;lodash.isNull=isNull;lodash.isNumber=isNumber;lodash.isObject=isObject;lodash.isObjectLike=isObjectLike;lodash.isPlainObject=isPlainObject;lodash.isRegExp=isRegExp;lodash.isSafeInteger=isSafeInteger;lodash.isSet=isSet;lodash.isString=isString;lodash.isSymbol=isSymbol;lodash.isTypedArray=isTypedArray;lodash.isUndefined=isUndefined;lodash.isWeakMap=isWeakMap;lodash.isWeakSet=isWeakSet;lodash.join=join;lodash.kebabCase=kebabCase;lodash.last=last;lodash.lastIndexOf=lastIndexOf;lodash.lowerCase=lowerCase;lodash.lowerFirst=lowerFirst;lodash.lt=lt;lodash.lte=lte;lodash.max=max;lodash.maxBy=maxBy;lodash.mean=mean;lodash.meanBy=meanBy;lodash.min=min;lodash.minBy=minBy;lodash.stubArray=stubArray;lodash.stubFalse=stubFalse;lodash.stubObject=stubObject;lodash.stubString=stubString;lodash.stubTrue=stubTrue;lodash.multiply=multiply;lodash.nth=nth;lodash.noConflict=noConflict;lodash.noop=noop;lodash.now=now;lodash.pad=pad;lodash.padEnd=padEnd;lodash.padStart=padStart;lodash.parseInt=parseInt;lodash.random=random;lodash.reduce=reduce;lodash.reduceRight=reduceRight;lodash.repeat=repeat;lodash.replace=replace;lodash.result=result;lodash.round=round;lodash.runInContext=runInContext;lodash.sample=sample;lodash.size=size;lodash.snakeCase=snakeCase;lodash.some=some;lodash.sortedIndex=sortedIndex;lodash.sortedIndexBy=sortedIndexBy;lodash.sortedIndexOf=sortedIndexOf;lodash.sortedLastIndex=sortedLastIndex;lodash.sortedLastIndexBy=sortedLastIndexBy;lodash.sortedLastIndexOf=sortedLastIndexOf;lodash.startCase=startCase;lodash.startsWith=startsWith;lodash.subtract=subtract;lodash.sum=sum;lodash.sumBy=sumBy;lodash.template=template;lodash.times=times;lodash.toFinite=toFinite;lodash.toInteger=toInteger;lodash.toLength=toLength;lodash.toLower=toLower;lodash.toNumber=toNumber;lodash.toSafeInteger=toSafeInteger;lodash.toString=toString;lodash.toUpper=toUpper;lodash.trim=trim;lodash.trimEnd=trimEnd;lodash.trimStart=trimStart;lodash.truncate=truncate;lodash.unescape=unescape;lodash.uniqueId=uniqueId;lodash.upperCase=upperCase;lodash.upperFirst=upperFirst;// Add aliases.
    	lodash.each=forEach;lodash.eachRight=forEachRight;lodash.first=head;mixin(lodash,function(){var source={};baseForOwn(lodash,function(func,methodName){if(!hasOwnProperty.call(lodash.prototype,methodName)){source[methodName]=func;}});return source;}(),{'chain':false});/*------------------------------------------------------------------------*//**
    	     * The semantic version number.
    	     *
    	     * @static
    	     * @memberOf _
    	     * @type {string}
    	     */lodash.VERSION=VERSION;// Assign default placeholders.
    	arrayEach(['bind','bindKey','curry','curryRight','partial','partialRight'],function(methodName){lodash[methodName].placeholder=lodash;});// Add `LazyWrapper` methods for `_.drop` and `_.take` variants.
    	arrayEach(['drop','take'],function(methodName,index){LazyWrapper.prototype[methodName]=function(n){n=n===undefined$1?1:nativeMax(toInteger(n),0);var result=this.__filtered__&&!index?new LazyWrapper(this):this.clone();if(result.__filtered__){result.__takeCount__=nativeMin(n,result.__takeCount__);}else {result.__views__.push({'size':nativeMin(n,MAX_ARRAY_LENGTH),'type':methodName+(result.__dir__<0?'Right':'')});}return result;};LazyWrapper.prototype[methodName+'Right']=function(n){return this.reverse()[methodName](n).reverse();};});// Add `LazyWrapper` methods that accept an `iteratee` value.
    	arrayEach(['filter','map','takeWhile'],function(methodName,index){var type=index+1,isFilter=type==LAZY_FILTER_FLAG||type==LAZY_WHILE_FLAG;LazyWrapper.prototype[methodName]=function(iteratee){var result=this.clone();result.__iteratees__.push({'iteratee':getIteratee(iteratee,3),'type':type});result.__filtered__=result.__filtered__||isFilter;return result;};});// Add `LazyWrapper` methods for `_.head` and `_.last`.
    	arrayEach(['head','last'],function(methodName,index){var takeName='take'+(index?'Right':'');LazyWrapper.prototype[methodName]=function(){return this[takeName](1).value()[0];};});// Add `LazyWrapper` methods for `_.initial` and `_.tail`.
    	arrayEach(['initial','tail'],function(methodName,index){var dropName='drop'+(index?'':'Right');LazyWrapper.prototype[methodName]=function(){return this.__filtered__?new LazyWrapper(this):this[dropName](1);};});LazyWrapper.prototype.compact=function(){return this.filter(identity);};LazyWrapper.prototype.find=function(predicate){return this.filter(predicate).head();};LazyWrapper.prototype.findLast=function(predicate){return this.reverse().find(predicate);};LazyWrapper.prototype.invokeMap=baseRest(function(path,args){if(typeof path=='function'){return new LazyWrapper(this);}return this.map(function(value){return baseInvoke(value,path,args);});});LazyWrapper.prototype.reject=function(predicate){return this.filter(negate(getIteratee(predicate)));};LazyWrapper.prototype.slice=function(start,end){start=toInteger(start);var result=this;if(result.__filtered__&&(start>0||end<0)){return new LazyWrapper(result);}if(start<0){result=result.takeRight(-start);}else if(start){result=result.drop(start);}if(end!==undefined$1){end=toInteger(end);result=end<0?result.dropRight(-end):result.take(end-start);}return result;};LazyWrapper.prototype.takeRightWhile=function(predicate){return this.reverse().takeWhile(predicate).reverse();};LazyWrapper.prototype.toArray=function(){return this.take(MAX_ARRAY_LENGTH);};// Add `LazyWrapper` methods to `lodash.prototype`.
    	baseForOwn(LazyWrapper.prototype,function(func,methodName){var checkIteratee=/^(?:filter|find|map|reject)|While$/.test(methodName),isTaker=/^(?:head|last)$/.test(methodName),lodashFunc=lodash[isTaker?'take'+(methodName=='last'?'Right':''):methodName],retUnwrapped=isTaker||/^find/.test(methodName);if(!lodashFunc){return;}lodash.prototype[methodName]=function(){var value=this.__wrapped__,args=isTaker?[1]:arguments,isLazy=value instanceof LazyWrapper,iteratee=args[0],useLazy=isLazy||isArray(value);var interceptor=function(value){var result=lodashFunc.apply(lodash,arrayPush([value],args));return isTaker&&chainAll?result[0]:result;};if(useLazy&&checkIteratee&&typeof iteratee=='function'&&iteratee.length!=1){// Avoid lazy use if the iteratee has a "length" value other than `1`.
    	isLazy=useLazy=false;}var chainAll=this.__chain__,isHybrid=!!this.__actions__.length,isUnwrapped=retUnwrapped&&!chainAll,onlyLazy=isLazy&&!isHybrid;if(!retUnwrapped&&useLazy){value=onlyLazy?value:new LazyWrapper(this);var result=func.apply(value,args);result.__actions__.push({'func':thru,'args':[interceptor],'thisArg':undefined$1});return new LodashWrapper(result,chainAll);}if(isUnwrapped&&onlyLazy){return func.apply(this,args);}result=this.thru(interceptor);return isUnwrapped?isTaker?result.value()[0]:result.value():result;};});// Add `Array` methods to `lodash.prototype`.
    	arrayEach(['pop','push','shift','sort','splice','unshift'],function(methodName){var func=arrayProto[methodName],chainName=/^(?:push|sort|unshift)$/.test(methodName)?'tap':'thru',retUnwrapped=/^(?:pop|shift)$/.test(methodName);lodash.prototype[methodName]=function(){var args=arguments;if(retUnwrapped&&!this.__chain__){var value=this.value();return func.apply(isArray(value)?value:[],args);}return this[chainName](function(value){return func.apply(isArray(value)?value:[],args);});};});// Map minified method names to their real names.
    	baseForOwn(LazyWrapper.prototype,function(func,methodName){var lodashFunc=lodash[methodName];if(lodashFunc){var key=lodashFunc.name+'';if(!hasOwnProperty.call(realNames,key)){realNames[key]=[];}realNames[key].push({'name':methodName,'func':lodashFunc});}});realNames[createHybrid(undefined$1,WRAP_BIND_KEY_FLAG).name]=[{'name':'wrapper','func':undefined$1}];// Add methods to `LazyWrapper`.
    	LazyWrapper.prototype.clone=lazyClone;LazyWrapper.prototype.reverse=lazyReverse;LazyWrapper.prototype.value=lazyValue;// Add chain sequence methods to the `lodash` wrapper.
    	lodash.prototype.at=wrapperAt;lodash.prototype.chain=wrapperChain;lodash.prototype.commit=wrapperCommit;lodash.prototype.next=wrapperNext;lodash.prototype.plant=wrapperPlant;lodash.prototype.reverse=wrapperReverse;lodash.prototype.toJSON=lodash.prototype.valueOf=lodash.prototype.value=wrapperValue;// Add lazy aliases.
    	lodash.prototype.first=lodash.prototype.head;if(symIterator){lodash.prototype[symIterator]=wrapperToIterator;}return lodash;};/*--------------------------------------------------------------------------*/// Export lodash.
    	var _=runInContext();// Some AMD build optimizers, like r.js, check for condition patterns like:
    	if(freeModule){// Export for Node.js.
    	(freeModule.exports=_)._=_;// Export for CommonJS support.
    	freeExports._=_;}else {// Export to the global object.
    	root._=_;}}).call(commonjsGlobal); 
    } (lodash, lodash.exports));

    var lodashExports = lodash.exports;

    var commonVariables$m = {
      fontFamily: 'v-sans, system-ui, -apple-system, BlinkMacSystemFont, "Segoe UI", sans-serif, "Apple Color Emoji", "Segoe UI Emoji", "Segoe UI Symbol"',
      fontFamilyMono: "v-mono, SFMono-Regular, Menlo, Consolas, Courier, monospace",
      fontWeight: "400",
      fontWeightStrong: "500",
      cubicBezierEaseInOut: "cubic-bezier(.4, 0, .2, 1)",
      cubicBezierEaseOut: "cubic-bezier(0, 0, .2, 1)",
      cubicBezierEaseIn: "cubic-bezier(.4, 0, 1, 1)",
      borderRadius: "3px",
      borderRadiusSmall: "2px",
      fontSize: "14px",
      fontSizeMini: "12px",
      fontSizeTiny: "12px",
      fontSizeSmall: "14px",
      fontSizeMedium: "14px",
      fontSizeLarge: "15px",
      fontSizeHuge: "16px",
      lineHeight: "1.6",
      heightMini: "16px",
      // private now, it's too small
      heightTiny: "22px",
      heightSmall: "28px",
      heightMedium: "34px",
      heightLarge: "40px",
      heightHuge: "46px"
    };

    const {
      fontSize,
      fontFamily,
      lineHeight: lineHeight$1
    } = commonVariables$m;
    var globalStyle = c$1("body", `
  margin: 0;
  font-size: ${fontSize};
  font-family: ${fontFamily};
  line-height: ${lineHeight$1};
  -webkit-text-size-adjust: 100%;
  -webkit-tap-highlight-color: transparent;
`, [c$1("input", `
    font-family: inherit;
    font-size: inherit;
  `)]);

    const configProviderInjectionKey = createInjectionKey("n-config-provider");

    const cssrAnchorMetaName = "naive-ui-style";

    function createTheme$1(theme) {
      return theme;
    }
    function useTheme(resolveId, mountId, style, defaultTheme, props, clsPrefixRef) {
      const ssrAdapter = useSsrAdapter();
      const NConfigProvider = vue.inject(configProviderInjectionKey, null);
      if (style) {
        const mountStyle = () => {
          const clsPrefix = clsPrefixRef?.value;
          style.mount({
            id: clsPrefix === void 0 ? mountId : clsPrefix + mountId,
            head: true,
            props: {
              bPrefix: clsPrefix ? `.${clsPrefix}-` : void 0
            },
            anchorMetaName: cssrAnchorMetaName,
            ssr: ssrAdapter,
            parent: NConfigProvider?.styleMountTarget
          });
          if (!NConfigProvider?.preflightStyleDisabled) {
            globalStyle.mount({
              id: "n-global",
              head: true,
              anchorMetaName: cssrAnchorMetaName,
              ssr: ssrAdapter,
              parent: NConfigProvider?.styleMountTarget
            });
          }
        };
        if (ssrAdapter) {
          mountStyle();
        } else {
          vue.onBeforeMount(mountStyle);
        }
      }
      const mergedThemeRef = vue.computed(() => {
        const {
          theme: {
            common: selfCommon,
            self,
            peers = {}
          } = {},
          themeOverrides: selfOverrides = {},
          builtinThemeOverrides: builtinOverrides = {}
        } = props;
        const {
          common: selfCommonOverrides,
          peers: peersOverrides
        } = selfOverrides;
        const {
          common: globalCommon = void 0,
          [resolveId]: {
            common: globalSelfCommon = void 0,
            self: globalSelf = void 0,
            peers: globalPeers = {}
          } = {}
        } = NConfigProvider?.mergedThemeRef.value || {};
        const {
          common: globalCommonOverrides = void 0,
          [resolveId]: globalSelfOverrides = {}
        } = NConfigProvider?.mergedThemeOverridesRef.value || {};
        const {
          common: globalSelfCommonOverrides,
          peers: globalPeersOverrides = {}
        } = globalSelfOverrides;
        const mergedCommon = lodashExports.merge({}, selfCommon || globalSelfCommon || globalCommon || defaultTheme.common, globalCommonOverrides, globalSelfCommonOverrides, selfCommonOverrides);
        const mergedSelf = lodashExports.merge(
        // {}, executed every time, no need for empty obj
        (self || globalSelf || defaultTheme.self)?.(mergedCommon), builtinOverrides, globalSelfOverrides, selfOverrides);
        return {
          common: mergedCommon,
          self: mergedSelf,
          peers: lodashExports.merge({}, defaultTheme.peers, globalPeers, peers),
          peerOverrides: lodashExports.merge({}, builtinOverrides.peers, globalPeersOverrides, peersOverrides)
        };
      });
      return mergedThemeRef;
    }
    useTheme.props = {
      theme: Object,
      themeOverrides: Object,
      builtinThemeOverrides: Object
    };

    const defaultClsPrefix = "n";
    function useConfig(props = {}, options = {
      defaultBordered: true
    }) {
      const NConfigProvider = vue.inject(configProviderInjectionKey, null);
      return {
        // NConfigProvider,
        inlineThemeDisabled: NConfigProvider?.inlineThemeDisabled,
        mergedRtlRef: NConfigProvider?.mergedRtlRef,
        mergedComponentPropsRef: NConfigProvider?.mergedComponentPropsRef,
        mergedBreakpointsRef: NConfigProvider?.mergedBreakpointsRef,
        mergedBorderedRef: vue.computed(() => {
          const {
            bordered
          } = props;
          if (bordered !== void 0) return bordered;
          return NConfigProvider?.mergedBorderedRef.value ?? options.defaultBordered ?? true;
        }),
        mergedClsPrefixRef: NConfigProvider ? NConfigProvider.mergedClsPrefixRef : vue.shallowRef(defaultClsPrefix),
        namespaceRef: vue.computed(() => NConfigProvider?.mergedNamespaceRef.value)
      };
    }
    function useMergedClsPrefix() {
      const NConfigProvider = vue.inject(configProviderInjectionKey, null);
      return NConfigProvider ? NConfigProvider.mergedClsPrefixRef : vue.shallowRef(defaultClsPrefix);
    }

    const zhCN$1 = {
      name: "zh-CN",
      global: {
        undo: "\u64A4\u9500",
        redo: "\u91CD\u505A",
        confirm: "\u786E\u8BA4",
        clear: "\u6E05\u9664"
      },
      Popconfirm: {
        positiveText: "\u786E\u8BA4",
        negativeText: "\u53D6\u6D88"
      },
      Cascader: {
        placeholder: "\u8BF7\u9009\u62E9",
        loading: "\u52A0\u8F7D\u4E2D",
        loadingRequiredMessage: label => `\u52A0\u8F7D\u5168\u90E8 ${label} \u7684\u5B50\u8282\u70B9\u540E\u624D\u53EF\u9009\u4E2D`
      },
      Time: {
        dateFormat: "yyyy-MM-dd",
        dateTimeFormat: "yyyy-MM-dd HH:mm:ss"
      },
      DatePicker: {
        yearFormat: "yyyy\u5E74",
        monthFormat: "MMM",
        dayFormat: "eeeeee",
        yearTypeFormat: "yyyy",
        monthTypeFormat: "yyyy-MM",
        dateFormat: "yyyy-MM-dd",
        dateTimeFormat: "yyyy-MM-dd HH:mm:ss",
        quarterFormat: "yyyy-qqq",
        weekFormat: "YYYY-w\u5468",
        clear: "\u6E05\u9664",
        now: "\u6B64\u523B",
        confirm: "\u786E\u8BA4",
        selectTime: "\u9009\u62E9\u65F6\u95F4",
        selectDate: "\u9009\u62E9\u65E5\u671F",
        datePlaceholder: "\u9009\u62E9\u65E5\u671F",
        datetimePlaceholder: "\u9009\u62E9\u65E5\u671F\u65F6\u95F4",
        monthPlaceholder: "\u9009\u62E9\u6708\u4EFD",
        yearPlaceholder: "\u9009\u62E9\u5E74\u4EFD",
        quarterPlaceholder: "\u9009\u62E9\u5B63\u5EA6",
        weekPlaceholder: "\u9009\u62E9\u5468",
        startDatePlaceholder: "\u5F00\u59CB\u65E5\u671F",
        endDatePlaceholder: "\u7ED3\u675F\u65E5\u671F",
        startDatetimePlaceholder: "\u5F00\u59CB\u65E5\u671F\u65F6\u95F4",
        endDatetimePlaceholder: "\u7ED3\u675F\u65E5\u671F\u65F6\u95F4",
        startMonthPlaceholder: "\u5F00\u59CB\u6708\u4EFD",
        endMonthPlaceholder: "\u7ED3\u675F\u6708\u4EFD",
        monthBeforeYear: false,
        firstDayOfWeek: 0,
        today: "\u4ECA\u5929"
      },
      DataTable: {
        checkTableAll: "\u9009\u62E9\u5168\u90E8\u8868\u683C\u6570\u636E",
        uncheckTableAll: "\u53D6\u6D88\u9009\u62E9\u5168\u90E8\u8868\u683C\u6570\u636E",
        confirm: "\u786E\u8BA4",
        clear: "\u91CD\u7F6E"
      },
      LegacyTransfer: {
        sourceTitle: "\u6E90\u9879",
        targetTitle: "\u76EE\u6807\u9879"
      },
      Transfer: {
        selectAll: "\u5168\u9009",
        clearAll: "\u6E05\u9664",
        unselectAll: "\u53D6\u6D88\u5168\u9009",
        total: num => `\u5171 ${num} \u9879`,
        selected: num => `\u5DF2\u9009 ${num} \u9879`
      },
      Empty: {
        description: "\u65E0\u6570\u636E"
      },
      Select: {
        placeholder: "\u8BF7\u9009\u62E9"
      },
      TimePicker: {
        placeholder: "\u8BF7\u9009\u62E9\u65F6\u95F4",
        positiveText: "\u786E\u8BA4",
        negativeText: "\u53D6\u6D88",
        now: "\u6B64\u523B",
        clear: "\u6E05\u9664"
      },
      Pagination: {
        goto: "\u8DF3\u81F3",
        selectionSuffix: "\u9875"
      },
      DynamicTags: {
        add: "\u6DFB\u52A0"
      },
      Log: {
        loading: "\u52A0\u8F7D\u4E2D"
      },
      Input: {
        placeholder: "\u8BF7\u8F93\u5165"
      },
      InputNumber: {
        placeholder: "\u8BF7\u8F93\u5165"
      },
      DynamicInput: {
        create: "\u6DFB\u52A0"
      },
      ThemeEditor: {
        title: "\u4E3B\u9898\u7F16\u8F91\u5668",
        clearAllVars: "\u6E05\u9664\u5168\u90E8\u53D8\u91CF",
        clearSearch: "\u6E05\u9664\u641C\u7D22",
        filterCompName: "\u8FC7\u6EE4\u7EC4\u4EF6\u540D",
        filterVarName: "\u8FC7\u6EE4\u53D8\u91CF\u540D",
        import: "\u5BFC\u5165",
        export: "\u5BFC\u51FA",
        restore: "\u6062\u590D\u9ED8\u8BA4"
      },
      Image: {
        tipPrevious: "\u4E0A\u4E00\u5F20\uFF08\u2190\uFF09",
        tipNext: "\u4E0B\u4E00\u5F20\uFF08\u2192\uFF09",
        tipCounterclockwise: "\u5411\u5DE6\u65CB\u8F6C",
        tipClockwise: "\u5411\u53F3\u65CB\u8F6C",
        tipZoomOut: "\u7F29\u5C0F",
        tipZoomIn: "\u653E\u5927",
        tipDownload: "\u4E0B\u8F7D",
        tipClose: "\u5173\u95ED\uFF08Esc\uFF09",
        tipOriginalSize: "\u7F29\u653E\u5230\u539F\u59CB\u5C3A\u5BF8"
      }
    };

    const zhTW$1 = {
      name: "zh-TW",
      global: {
        undo: "\u5FA9\u539F",
        redo: "\u91CD\u505A",
        confirm: "\u78BA\u5B9A",
        clear: "\u6E05\u9664"
      },
      Popconfirm: {
        positiveText: "\u78BA\u5B9A",
        negativeText: "\u53D6\u6D88"
      },
      Cascader: {
        placeholder: "\u8ACB\u9078\u64C7",
        loading: "\u8F09\u5165\u4E2D",
        loadingRequiredMessage: label => `\u8F09\u5165\u5168\u90E8 ${label} \u7684\u5B50\u7BC0\u9EDE\u5F8C\u624D\u53EF\u9078\u64C7`
      },
      Time: {
        dateFormat: "yyyy-MM-dd",
        dateTimeFormat: "yyyy-MM-dd HH:mm:ss"
      },
      DatePicker: {
        yearFormat: "yyyy \u5E74",
        monthFormat: "MMM",
        dayFormat: "eeeeee",
        yearTypeFormat: "yyyy",
        monthTypeFormat: "yyyy-MM",
        dateFormat: "yyyy-MM-dd",
        dateTimeFormat: "yyyy-MM-dd HH:mm:ss",
        quarterFormat: "yyyy-qqq",
        weekFormat: "YYYY-w",
        clear: "\u6E05\u9664",
        now: "\u73FE\u5728",
        confirm: "\u78BA\u5B9A",
        selectTime: "\u9078\u64C7\u6642\u9593",
        selectDate: "\u9078\u64C7\u65E5\u671F",
        datePlaceholder: "\u9078\u64C7\u65E5\u671F",
        datetimePlaceholder: "\u9078\u64C7\u65E5\u671F\u6642\u9593",
        monthPlaceholder: "\u9078\u64C7\u6708\u4EFD",
        yearPlaceholder: "\u9078\u64C7\u5E74\u4EFD",
        quarterPlaceholder: "\u9078\u64C7\u5B63\u5EA6",
        weekPlaceholder: "Select Week",
        startDatePlaceholder: "\u958B\u59CB\u65E5\u671F",
        endDatePlaceholder: "\u7D50\u675F\u65E5\u671F",
        startDatetimePlaceholder: "\u958B\u59CB\u65E5\u671F\u6642\u9593",
        endDatetimePlaceholder: "\u7D50\u675F\u65E5\u671F\u6642\u9593",
        startMonthPlaceholder: "\u958B\u59CB\u6708\u4EFD",
        endMonthPlaceholder: "\u7D50\u675F\u6708\u4EFD",
        monthBeforeYear: false,
        firstDayOfWeek: 0,
        today: "\u4ECA\u5929"
      },
      DataTable: {
        checkTableAll: "\u9078\u64C7\u5168\u90E8\u8868\u683C\u8CC7\u6599",
        uncheckTableAll: "\u53D6\u6D88\u9078\u64C7\u5168\u90E8\u8868\u683C\u8CC7\u6599",
        confirm: "\u78BA\u5B9A",
        clear: "\u91CD\u8A2D"
      },
      LegacyTransfer: {
        sourceTitle: "\u4F86\u6E90",
        targetTitle: "\u76EE\u6A19"
      },
      Transfer: {
        selectAll: "\u5168\u9078",
        unselectAll: "\u53D6\u6D88\u5168\u9078",
        clearAll: "\u6E05\u9664\u5168\u90E8",
        total: num => `\u5171 ${num} \u9805`,
        selected: num => `\u5DF2\u9078 ${num} \u9805`
      },
      Empty: {
        description: "\u7121\u8CC7\u6599"
      },
      Select: {
        placeholder: "\u8ACB\u9078\u64C7"
      },
      TimePicker: {
        placeholder: "\u8ACB\u9078\u64C7\u6642\u9593",
        positiveText: "\u78BA\u5B9A",
        negativeText: "\u53D6\u6D88",
        now: "\u73FE\u5728",
        clear: "\u6E05\u9664"
      },
      Pagination: {
        goto: "\u8DF3\u81F3",
        selectionSuffix: "\u9801"
      },
      DynamicTags: {
        add: "\u65B0\u589E"
      },
      Log: {
        loading: "\u8F09\u5165\u4E2D"
      },
      Input: {
        placeholder: "\u8ACB\u8F38\u5165"
      },
      InputNumber: {
        placeholder: "\u8ACB\u8F38\u5165"
      },
      DynamicInput: {
        create: "\u65B0\u589E"
      },
      ThemeEditor: {
        title: "\u4E3B\u984C\u7DE8\u8F2F\u5668",
        clearAllVars: "\u6E05\u9664\u5168\u90E8\u8B8A\u6578",
        clearSearch: "\u6E05\u9664\u641C\u5C0B",
        filterCompName: "\u904E\u6FFE\u7D44\u4EF6\u540D\u7A31",
        filterVarName: "\u904E\u6FFE\u8B8A\u6578\u540D\u7A31",
        import: "\u532F\u5165",
        export: "\u532F\u51FA",
        restore: "\u6062\u5FA9\u9810\u8A2D"
      },
      Image: {
        tipPrevious: "\u4E0A\u4E00\u5F35\uFF08\u2190\uFF09",
        tipNext: "\u4E0B\u4E00\u5F35\uFF08\u2192\uFF09",
        tipCounterclockwise: "\u5411\u5DE6\u65CB\u8F49",
        tipClockwise: "\u5411\u53F3\u65CB\u8F49",
        tipZoomOut: "\u7E2E\u5C0F",
        tipZoomIn: "\u653E\u5927",
        tipDownload: "\u4E0B\u8F09",
        tipClose: "\u95DC\u9589\uFF08Esc\uFF09",
        tipOriginalSize: "\u7E2E\u653E\u5230\u539F\u59CB\u5C3A\u5BF8"
      }
    };

    const enUS$1 = {
      name: "en-US",
      global: {
        undo: "Undo",
        redo: "Redo",
        confirm: "Confirm",
        clear: "Clear"
      },
      Popconfirm: {
        positiveText: "Confirm",
        negativeText: "Cancel"
      },
      Cascader: {
        placeholder: "Please Select",
        loading: "Loading",
        loadingRequiredMessage: label => `Please load all ${label}'s descendants before checking it.`
      },
      Time: {
        dateFormat: "yyyy-MM-dd",
        dateTimeFormat: "yyyy-MM-dd HH:mm:ss"
      },
      DatePicker: {
        yearFormat: "yyyy",
        monthFormat: "MMM",
        dayFormat: "eeeeee",
        yearTypeFormat: "yyyy",
        monthTypeFormat: "yyyy-MM",
        dateFormat: "yyyy-MM-dd",
        dateTimeFormat: "yyyy-MM-dd HH:mm:ss",
        quarterFormat: "yyyy-qqq",
        weekFormat: "YYYY-w",
        clear: "Clear",
        now: "Now",
        confirm: "Confirm",
        selectTime: "Select Time",
        selectDate: "Select Date",
        datePlaceholder: "Select Date",
        datetimePlaceholder: "Select Date and Time",
        monthPlaceholder: "Select Month",
        yearPlaceholder: "Select Year",
        quarterPlaceholder: "Select Quarter",
        weekPlaceholder: "Select Week",
        startDatePlaceholder: "Start Date",
        endDatePlaceholder: "End Date",
        startDatetimePlaceholder: "Start Date and Time",
        endDatetimePlaceholder: "End Date and Time",
        startMonthPlaceholder: "Start Month",
        endMonthPlaceholder: "End Month",
        monthBeforeYear: true,
        firstDayOfWeek: 6,
        today: "Today"
      },
      DataTable: {
        checkTableAll: "Select all in the table",
        uncheckTableAll: "Unselect all in the table",
        confirm: "Confirm",
        clear: "Clear"
      },
      LegacyTransfer: {
        sourceTitle: "Source",
        targetTitle: "Target"
      },
      Transfer: {
        selectAll: "Select all",
        unselectAll: "Unselect all",
        clearAll: "Clear",
        total: num => `Total ${num} items`,
        selected: num => `${num} items selected`
      },
      Empty: {
        description: "No Data"
      },
      Select: {
        placeholder: "Please Select"
      },
      TimePicker: {
        placeholder: "Select Time",
        positiveText: "OK",
        negativeText: "Cancel",
        now: "Now",
        clear: "Clear"
      },
      Pagination: {
        goto: "Goto",
        selectionSuffix: "page"
      },
      DynamicTags: {
        add: "Add"
      },
      Log: {
        loading: "Loading"
      },
      Input: {
        placeholder: "Please Input"
      },
      InputNumber: {
        placeholder: "Please Input"
      },
      DynamicInput: {
        create: "Create"
      },
      ThemeEditor: {
        title: "Theme Editor",
        clearAllVars: "Clear All Variables",
        clearSearch: "Clear Search",
        filterCompName: "Filter Component Name",
        filterVarName: "Filter Variable Name",
        import: "Import",
        export: "Export",
        restore: "Reset to Default"
      },
      Image: {
        tipPrevious: "Previous picture (\u2190)",
        tipNext: "Next picture (\u2192)",
        tipCounterclockwise: "Counterclockwise",
        tipClockwise: "Clockwise",
        tipZoomOut: "Zoom out",
        tipZoomIn: "Zoom in",
        tipDownload: "Download",
        tipClose: "Close (Esc)",
        // TODO: translation
        tipOriginalSize: "Zoom to original size"
      }
    };

    const ruRu = {
      name: "ru-RU",
      global: {
        undo: "\u041E\u0442\u043C\u0435\u043D\u0438\u0442\u044C",
        redo: "\u0412\u0435\u0440\u043D\u0443\u0442\u044C",
        confirm: "\u041F\u043E\u0434\u0442\u0432\u0435\u0440\u0434\u0438\u0442\u044C",
        clear: "\u041E\u0447\u0438\u0441\u0442\u0438\u0442\u044C"
      },
      Popconfirm: {
        positiveText: "\u041F\u043E\u0434\u0442\u0432\u0435\u0440\u0434\u0438\u0442\u044C",
        negativeText: "\u041E\u0442\u043C\u0435\u043D\u0430"
      },
      Cascader: {
        placeholder: "\u0412\u044B\u0431\u0440\u0430\u0442\u044C",
        loading: "\u0417\u0430\u0433\u0440\u0443\u0437\u043A\u0430",
        loadingRequiredMessage: label => `\u0417\u0430\u0433\u0440\u0443\u0437\u0438\u0442\u0435 \u0432\u0441\u0435 \u0434\u043E\u0447\u0435\u0440\u043D\u0438\u0435 \u0443\u0437\u043B\u044B ${label} \u043F\u0440\u0435\u0436\u0434\u0435 \u0447\u0435\u043C \u043E\u043D\u0438 \u0441\u0442\u0430\u043D\u0443\u0442 \u043D\u0435\u043E\u0431\u044F\u0437\u0430\u0442\u0435\u043B\u044C\u043D\u044B\u043C\u0438`
      },
      Time: {
        dateFormat: "yyyy-MM-dd",
        dateTimeFormat: "yyyy-MM-dd HH:mm:ss"
      },
      DatePicker: {
        yearFormat: "yyyy",
        monthFormat: "MMM",
        dayFormat: "eeeeee",
        yearTypeFormat: "yyyy",
        monthTypeFormat: "yyyy-MM",
        dateFormat: "yyyy-MM-dd",
        dateTimeFormat: "yyyy-MM-dd HH:mm:ss",
        quarterFormat: "yyyy-qqq",
        weekFormat: "YYYY-w",
        clear: "\u041E\u0447\u0438\u0441\u0442\u0438\u0442\u044C",
        now: "\u0421\u0435\u0439\u0447\u0430\u0441",
        confirm: "\u041F\u043E\u0434\u0442\u0432\u0435\u0440\u0434\u0438\u0442\u044C",
        selectTime: "\u0412\u044B\u0431\u0440\u0430\u0442\u044C \u0432\u0440\u0435\u043C\u044F",
        selectDate: "\u0412\u044B\u0431\u0440\u0430\u0442\u044C \u0434\u0430\u0442\u0443",
        datePlaceholder: "\u0412\u044B\u0431\u0440\u0430\u0442\u044C \u0434\u0430\u0442\u0443",
        datetimePlaceholder: "\u0412\u044B\u0431\u0440\u0430\u0442\u044C \u0434\u0430\u0442\u0443 \u0438 \u0432\u0440\u0435\u043C\u044F",
        monthPlaceholder: "\u0412\u044B\u0431\u0435\u0440\u0438\u0442\u0435 \u043C\u0435\u0441\u044F\u0446",
        yearPlaceholder: "\u0412\u044B\u0431\u0435\u0440\u0438\u0442\u0435 \u0433\u043E\u0434",
        quarterPlaceholder: "\u0412\u044B\u0431\u0435\u0440\u0438\u0442\u0435 \u043A\u0432\u0430\u0440\u0442\u0430\u043B",
        weekPlaceholder: "Select Week",
        startDatePlaceholder: "\u0414\u0430\u0442\u0430 \u043D\u0430\u0447\u0430\u043B\u0430",
        endDatePlaceholder: "\u0414\u0430\u0442\u0430 \u043E\u043A\u043E\u043D\u0447\u0430\u043D\u0438\u044F",
        startDatetimePlaceholder: "\u0414\u0430\u0442\u0430 \u0438 \u0432\u0440\u0435\u043C\u044F \u043D\u0430\u0447\u0430\u043B\u0430",
        endDatetimePlaceholder: "\u0414\u0430\u0442\u0430 \u0438 \u0432\u0440\u0435\u043C\u044F \u043E\u043A\u043E\u043D\u0447\u0430\u043D\u0438\u044F",
        startMonthPlaceholder: "\u041D\u0430\u0447\u0430\u043B\u043E \u043C\u0435\u0441\u044F\u0446\u0430",
        endMonthPlaceholder: "\u041A\u043E\u043D\u0435\u0446 \u043C\u0435\u0441\u044F\u0446\u0430",
        monthBeforeYear: true,
        firstDayOfWeek: 0,
        today: "\u0421\u0435\u0433\u043E\u0434\u043D\u044F"
      },
      DataTable: {
        checkTableAll: "\u0412\u044B\u0431\u0440\u0430\u0442\u044C \u0432\u0441\u0435 \u0432 \u0442\u0430\u0431\u043B\u0438\u0446\u0435",
        uncheckTableAll: "\u041E\u0442\u043C\u0435\u043D\u0438\u0442\u044C \u0432\u0441\u0435 \u0432 \u0442\u0430\u0431\u043B\u0438\u0446\u0435",
        confirm: "\u041F\u043E\u0434\u0442\u0432\u0435\u0440\u0434\u0438\u0442\u044C",
        clear: "\u041E\u0447\u0438\u0441\u0442\u0438\u0442\u044C"
      },
      LegacyTransfer: {
        sourceTitle: "\u0418\u0441\u0442\u043E\u0447\u043D\u0438\u043A",
        targetTitle: "\u041D\u0430\u0437\u043D\u0430\u0447\u0435\u043D\u0438\u0435"
      },
      Transfer: {
        selectAll: "\u0412\u044B\u0431\u0440\u0430\u0442\u044C \u0432\u0441\u0435",
        unselectAll: "\u0421\u043D\u044F\u0442\u044C \u0432\u0441\u0435",
        clearAll: "\u041E\u0447\u0438\u0441\u0442\u0438\u0442\u044C",
        total: num => `\u0412\u0441\u0435\u0433\u043E ${num} \u044D\u043B\u0435\u043C\u0435\u043D\u0442\u043E\u0432`,
        selected: num => `${num} \u0432\u044B\u0431\u0440\u0430\u043D\u043E \u044D\u043B\u0435\u043C\u0435\u043D\u0442\u043E\u0432`
      },
      Empty: {
        description: "\u041D\u0435\u0442 \u0434\u0430\u043D\u043D\u044B\u0445"
      },
      Select: {
        placeholder: "\u0412\u044B\u0431\u0440\u0430\u0442\u044C"
      },
      TimePicker: {
        placeholder: "\u0412\u044B\u0431\u0440\u0430\u0442\u044C \u0432\u0440\u0435\u043C\u044F",
        positiveText: "OK",
        negativeText: "\u041E\u0442\u043C\u0435\u043D\u0438\u0442\u044C",
        now: "\u0421\u0435\u0439\u0447\u0430\u0441",
        clear: "\u041E\u0447\u0438\u0441\u0442\u0438\u0442\u044C"
      },
      Pagination: {
        goto: "\u041F\u0435\u0440\u0435\u0439\u0442\u0438",
        selectionSuffix: "\u0441\u0442\u0440\u0430\u043D\u0438\u0446\u0430"
      },
      DynamicTags: {
        add: "\u0414\u043E\u0431\u0430\u0432\u0438\u0442\u044C"
      },
      Log: {
        loading: "\u0417\u0430\u0433\u0440\u0443\u0437\u043A\u0430"
      },
      Input: {
        placeholder: "\u0412\u0432\u0435\u0441\u0442\u0438"
      },
      InputNumber: {
        placeholder: "\u0412\u0432\u0435\u0441\u0442\u0438"
      },
      DynamicInput: {
        create: "\u0421\u043E\u0437\u0434\u0430\u0442\u044C"
      },
      ThemeEditor: {
        title: "\u0420\u0435\u0434\u0430\u043A\u0442\u043E\u0440 \u0442\u0435\u043C\u044B",
        clearAllVars: "\u041E\u0447\u0438\u0441\u0442\u0438\u0442\u044C \u0432\u0441\u0435",
        clearSearch: "\u041E\u0447\u0438\u0441\u0442\u0438\u0442\u044C \u043F\u043E\u0438\u0441\u043A",
        filterCompName: "\u0424\u0438\u043B\u044C\u0442\u0440\u043E\u0432\u0430\u0442\u044C \u043F\u043E \u0438\u043C\u0435\u043D\u0438 \u043A\u043E\u043C\u043F\u043E\u043D\u0435\u043D\u0442\u0430",
        filterVarName: "\u0424\u0438\u043B\u044C\u0442\u0440\u043E\u0432\u0430\u0442\u044C \u0438\u043C\u0435\u043D\u0430 \u043F\u0435\u0440\u0435\u043C\u0435\u043D\u043D\u044B\u0445",
        import: "\u0418\u043C\u043F\u043E\u0440\u0442",
        export: "\u042D\u043A\u0441\u043F\u043E\u0440\u0442",
        restore: "\u0421\u0431\u0440\u043E\u0441\u0438\u0442\u044C"
      },
      Image: {
        tipPrevious: "\u041F\u0440\u0435\u0434\u044B\u0434\u0443\u0449\u0435\u0435 \u0438\u0437\u043E\u0431\u0440\u0430\u0436\u0435\u043D\u0438\u0435 (\u2190)",
        tipNext: "\u0421\u043B\u0435\u0434\u0443\u044E\u0449\u0435\u0435 \u0438\u0437\u043E\u0431\u0440\u0430\u0436\u0435\u043D\u0438\u0435 (\u2192)",
        tipCounterclockwise: "\u041F\u0440\u043E\u0442\u0438\u0432 \u0447\u0430\u0441\u043E\u0432\u043E\u0439 \u0441\u0442\u0440\u0435\u043B\u043A\u0438",
        tipClockwise: "\u041F\u043E \u0447\u0430\u0441\u043E\u0432\u043E\u0439 \u0441\u0442\u0440\u0435\u043B\u043A\u0435",
        tipZoomOut: "\u041E\u0442\u0434\u0430\u043B\u0438\u0442\u044C",
        tipZoomIn: "\u041F\u0440\u0438\u0431\u043B\u0438\u0437\u0438\u0442\u044C",
        tipDownload: "\u0421\u043A\u0430\u0447\u0430\u0442\u044C",
        tipClose: "\u0417\u0430\u043A\u0440\u044B\u0442\u044C (Esc)",
        tipOriginalSize: "\u0412\u0435\u0440\u043D\u0443\u0442\u044C \u0438\u0441\u0445\u043E\u0434\u043D\u044B\u0439 \u0440\u0430\u0437\u043C\u0435\u0440"
      }
    };

    const ukUA = {
      name: "uk-UA",
      global: {
        undo: "\u0412\u0456\u0434\u043C\u0456\u043D\u0438\u0442\u0438",
        redo: "\u041F\u043E\u0432\u0442\u043E\u0440\u0438\u0442\u0438",
        confirm: "\u041F\u0456\u0434\u0442\u0432\u0435\u0440\u0434\u0438\u0442\u0438",
        clear: "\u0421\u0442\u0435\u0440\u0442\u0438"
      },
      Popconfirm: {
        positiveText: "\u041F\u0456\u0434\u0442\u0432\u0435\u0440\u0434\u0438\u0442\u0438",
        negativeText: "\u0421\u043A\u0430\u0441\u0443\u0432\u0430\u0442\u0438"
      },
      Cascader: {
        placeholder: "\u041E\u0431\u0440\u0430\u0442\u0438",
        loading: "\u0417\u0430\u0432\u0430\u043D\u0442\u0430\u0436\u0435\u043D\u043D\u044F",
        loadingRequiredMessage: label => `\u0417\u0430\u0432\u0430\u043D\u0442\u0430\u0436\u0442\u0435 \u0432\u0441\u0456 \u0434\u043E\u0447\u0456\u0440\u043D\u0456 \u0432\u0443\u0437\u043B\u0438 ${label} \u043F\u0435\u0440\u0435\u0434 \u043F\u0435\u0440\u0435\u0432\u0456\u0440\u043A\u043E\u044E`
      },
      Time: {
        dateFormat: "yyyy-MM-dd",
        dateTimeFormat: "yyyy-MM-dd HH:mm:ss"
      },
      DatePicker: {
        yearFormat: "yyyy",
        monthFormat: "MMM",
        dayFormat: "eeeeee",
        yearTypeFormat: "yyyy",
        monthTypeFormat: "yyyy-MM",
        dateFormat: "yyyy-MM-dd",
        dateTimeFormat: "yyyy-MM-dd HH:mm:ss",
        quarterFormat: "yyyy-qqq",
        weekFormat: "YYYY-w",
        clear: "\u0421\u0442\u0435\u0440\u0442\u0438",
        now: "\u0417\u0430\u0440\u0430\u0437",
        confirm: "\u041F\u0456\u0434\u0442\u0432\u0435\u0440\u0434\u0438\u0442\u0438",
        selectTime: "\u041E\u0431\u0440\u0430\u0442\u0438 \u0447\u0430\u0441",
        selectDate: "\u041E\u0431\u0440\u0430\u0442\u0438 \u0434\u0430\u0442\u0443",
        datePlaceholder: "\u041E\u0431\u0440\u0430\u0442\u0438 \u0434\u0430\u0442\u0443",
        datetimePlaceholder: "\u041E\u0431\u0440\u0430\u0442\u0438 \u0434\u0430\u0442\u0443 \u0456 \u0447\u0430\u0441",
        monthPlaceholder: "\u041E\u0431\u0435\u0440\u0456\u0442\u044C \u043C\u0456\u0441\u044F\u0446\u044C",
        yearPlaceholder: "\u041E\u0431\u0435\u0440\u0456\u0442\u044C \u0440\u0456\u043A",
        quarterPlaceholder: "\u041E\u0431\u0435\u0440\u0456\u0442\u044C \u043A\u0432\u0430\u0440\u0442\u0430\u043B",
        weekPlaceholder: "Select Week",
        startDatePlaceholder: "\u0414\u0430\u0442\u0430 \u043F\u043E\u0447\u0430\u0442\u043A\u0443",
        endDatePlaceholder: "\u0414\u0430\u0442\u0430 \u0437\u0430\u0432\u0435\u0440\u0448\u0435\u043D\u043D\u044F",
        startDatetimePlaceholder: "\u0414\u0430\u0442\u0430 \u0456 \u0447\u0430\u0441 \u043F\u043E\u0447\u0430\u0442\u043A\u0443",
        endDatetimePlaceholder: "\u0414\u0430\u0442\u0430 \u0456 \u0447\u0430\u0441 \u0437\u0430\u0432\u0435\u0440\u0448\u0435\u043D\u043D\u044F",
        startMonthPlaceholder: "\u041F\u0435\u0440\u0448\u0438\u0439 \u043C\u0456\u0441\u044F\u0446\u044C",
        endMonthPlaceholder: "\u041E\u0441\u0442\u0430\u043D\u043D\u0456\u0439 \u043C\u0456\u0441\u044F\u0446\u044C",
        monthBeforeYear: true,
        firstDayOfWeek: 0,
        today: "\u0421\u044C\u043E\u0433\u043E\u0434\u043D\u0456"
      },
      DataTable: {
        checkTableAll: "\u041E\u0431\u0440\u0430\u0442\u0438 \u0432\u0441\u0435 \u0432 \u0442\u0430\u0431\u043B\u0438\u0446\u0456",
        uncheckTableAll: "\u0421\u043A\u0430\u0441\u0443\u0432\u0430\u0442\u0438 \u0432\u0438\u0431\u0456\u0440 \u0443\u0441\u044C\u043E\u0433\u043E \u0432 \u0442\u0430\u0431\u043B\u0438\u0446\u0456",
        confirm: "\u041F\u0456\u0434\u0442\u0432\u0435\u0440\u0434\u0438\u0442\u0438",
        clear: "\u0421\u0442\u0435\u0440\u0442\u0438"
      },
      LegacyTransfer: {
        sourceTitle: "\u0414\u0436\u0435\u0440\u0435\u043B\u043E",
        targetTitle: "\u0426\u0456\u043B\u044C"
      },
      // TODO: translation
      Transfer: {
        selectAll: "Select all",
        unselectAll: "Unselect all",
        clearAll: "Clear",
        total: num => `Total ${num} items`,
        selected: num => `${num} items selected`
      },
      Empty: {
        description: "\u041D\u0435\u043C\u0430\u0454 \u0434\u0430\u043D\u0438\u0445"
      },
      Select: {
        placeholder: "\u041E\u0431\u0440\u0430\u0442\u0438"
      },
      TimePicker: {
        placeholder: "\u041E\u0431\u0440\u0430\u0442\u0438 \u0447\u0430\u0441",
        positiveText: "OK",
        negativeText: "\u0421\u043A\u0430\u0441\u0443\u0432\u0430\u0442\u0438",
        now: "\u0417\u0430\u0440\u0430\u0437",
        clear: "\u0421\u0442\u0435\u0440\u0442\u0438"
      },
      Pagination: {
        goto: "\u041F\u0435\u0440\u0435\u0439\u0442\u0438",
        selectionSuffix: "\u0441\u0442\u043E\u0440\u0456\u043D\u043A\u0430"
      },
      DynamicTags: {
        add: "\u0414\u043E\u0434\u0430\u0442\u0438"
      },
      Log: {
        loading: "\u0417\u0430\u0432\u0430\u043D\u0442\u0430\u0436\u0435\u043D\u043D\u044F"
      },
      Input: {
        placeholder: "\u0412\u0432\u0435\u0441\u0442\u0438"
      },
      InputNumber: {
        placeholder: "\u0412\u0432\u0435\u0441\u0442\u0438"
      },
      DynamicInput: {
        create: "\u0421\u0442\u0432\u043E\u0440\u0438\u0442\u0438"
      },
      ThemeEditor: {
        title: "\u0420\u0435\u0434\u0430\u043A\u0442\u043E\u0440 \u0442\u0435\u043C\u0438",
        clearAllVars: "\u0421\u0442\u0435\u0440\u0442\u0438 \u0432\u0441\u0435",
        clearSearch: "\u041E\u0447\u0438\u0441\u0442\u0438\u0442\u0438 \u043F\u043E\u0448\u0443\u043A",
        filterCompName: "\u0424\u0456\u043B\u044C\u0442\u0440\u0443\u0432\u0430\u0442\u0438 \u0437\u0430 \u0456\u043C\u0435\u043D\u0435\u043C \u043A\u043E\u043C\u043F\u043E\u043D\u0435\u043D\u0442\u0430",
        filterVarName: "\u0424\u0456\u043B\u044C\u0442\u0440\u0443\u0432\u0430\u0442\u0438 \u0456\u043C\u0435\u043D\u0430 \u0437\u043C\u0456\u043D\u043D\u0438\u0445",
        import: "\u0406\u043C\u043F\u043E\u0440\u0442",
        export: "\u0415\u043A\u0441\u043F\u043E\u0440\u0442",
        restore: "\u0421\u043A\u0438\u043D\u0443\u0442\u0438"
      },
      Image: {
        tipPrevious: "\u041F\u043E\u043F\u0435\u0440\u0435\u0434\u043D\u0454 \u0437\u043E\u0431\u0440\u0430\u0436\u0435\u043D\u043D\u044F (\u2190)",
        tipNext: "\u041D\u0430\u0441\u0442\u0443\u043F\u043D\u0435 \u0437\u043E\u0431\u0440\u0430\u0436\u0435\u043D\u043D\u044F (\u2192)",
        tipCounterclockwise: "\u041F\u0440\u043E\u0442\u0438 \u0433\u043E\u0434\u0438\u043D\u043D\u0438\u043A\u043E\u0432\u043E\u0457 \u0441\u0442\u0440\u0456\u043B\u043A\u0438",
        tipClockwise: "\u0417\u0430 \u0433\u043E\u0434\u0438\u043D\u043D\u0438\u043A\u043E\u0432\u043E\u044E \u0441\u0442\u0440\u0456\u043B\u043A\u043E\u044E",
        tipZoomOut: "\u0412\u0456\u0434\u0434\u0430\u043B\u0438\u0442\u0438",
        tipZoomIn: "\u041D\u0430\u0431\u043B\u0438\u0437\u0438\u0442\u0438",
        tipDownload: "\u0437\u0432\u0430\u043D\u0442\u0430\u0436\u0438\u0442\u0438",
        tipClose: "\u0417\u0430\u043A\u0440\u0438\u0442\u0438 (Esc)",
        tipOriginalSize: "\u041C\u0430\u0441\u0448\u0442\u0430\u0431\u0443\u0432\u0430\u0442\u0438 \u0434\u043E \u043E\u0440\u0438\u0433\u0456\u043D\u0430\u043B\u044C\u043D\u043E\u0433\u043E \u0440\u043E\u0437\u043C\u0456\u0440\u0443"
      }
    };

    const uzUZ = {
      name: "uz-UZ",
      global: {
        undo: "Bekor qilish",
        redo: "Qaytadan amalga oshirish",
        confirm: "Tasdiqlash",
        clear: "Tozalash"
      },
      Popconfirm: {
        positiveText: "Tasdiqlash",
        negativeText: "Bekor qilish"
      },
      Cascader: {
        placeholder: "Tanlash",
        loading: "Yuklanmoqda",
        loadingRequiredMessage: label => `${label} ning barcha farzand elementlarini yuklang, keyin ularga to'liq ruxsat bering`
      },
      Time: {
        dateFormat: "yyyy-MM-dd",
        dateTimeFormat: "yyyy-MM-dd HH:mm:ss"
      },
      DatePicker: {
        yearFormat: "yyyy",
        monthFormat: "MMM",
        dayFormat: "eeeeee",
        yearTypeFormat: "yyyy",
        monthTypeFormat: "yyyy-MM",
        dateFormat: "yyyy-MM-dd",
        dateTimeFormat: "yyyy-MM-dd HH:mm:ss",
        quarterFormat: "yyyy-qqq",
        weekFormat: "YYYY-w",
        clear: "Tozalash",
        now: "Hozir",
        confirm: "Tasdiqlash",
        selectTime: "Vaqtni tanlash",
        selectDate: "Sana tanlash",
        datePlaceholder: "Sana tanlash",
        datetimePlaceholder: "Sana va vaqtni tanlash",
        monthPlaceholder: "Oy tanlash",
        yearPlaceholder: "Yil tanlash",
        quarterPlaceholder: "Chet tanlash",
        weekPlaceholder: "Hafta tanlash",
        startDatePlaceholder: "Boshlang`ich sana",
        endDatePlaceholder: "Tugash sanasi",
        startDatetimePlaceholder: "Boshlang`ich sana va vaqti",
        endDatetimePlaceholder: "Tugash sanasi va vaqti",
        startMonthPlaceholder: "Boshlang`ich oy",
        endMonthPlaceholder: "Oxirgi oy",
        monthBeforeYear: true,
        firstDayOfWeek: 1,
        // in Uzbekistan monday
        today: "Bugun"
      },
      DataTable: {
        checkTableAll: "Jadvaldagi barchasini tanlash",
        uncheckTableAll: "Jadvaldagi barchasini bekor qilish",
        confirm: "Tasdiqlash",
        clear: "Tozalash"
      },
      LegacyTransfer: {
        sourceTitle: "Manba",
        targetTitle: "Mansub"
      },
      Transfer: {
        selectAll: "Barchasini tanlash",
        unselectAll: "Barchasini bekor qilish",
        clearAll: "Tozalash",
        total: num => `Jami ${num} ta`,
        selected: num => `Tanlandi ${num} ta`
      },
      Empty: {
        description: "Ma`lumotlar yo`q"
      },
      Select: {
        placeholder: "Tanlash"
      },
      TimePicker: {
        placeholder: "Vaqtni tanlash",
        positiveText: "OK",
        negativeText: "Bekor qilish",
        now: "Hozir",
        clear: "Tozalash"
      },
      Pagination: {
        goto: "O`tish",
        selectionSuffix: "sahifa"
      },
      DynamicTags: {
        add: "Qo`shish"
      },
      Log: {
        loading: "Yuklanmoqda"
      },
      Input: {
        placeholder: "Kiriting"
      },
      InputNumber: {
        placeholder: "Kiriting"
      },
      DynamicInput: {
        create: "Yaratish"
      },
      ThemeEditor: {
        title: "Mavzu muharriri",
        clearAllVars: "Barchasini tozalash",
        clearSearch: "Qidiruvni tozalash",
        filterCompName: "Komponent nomi bo`yicha filtr",
        filterVarName: "O`zgaruvchi nomlarini filtrlash",
        import: "Import qilish",
        export: "Eksport qilish",
        restore: "Qayta tiklash"
      },
      Image: {
        tipPrevious: "Oldingi rasm (\u2190)",
        tipNext: "Keyingi rasm (\u2192)",
        tipCounterclockwise: "Soat yonida",
        tipClockwise: "Soat bo`yicha",
        tipZoomOut: "Yaqinlash",
        tipZoomIn: "Uzoq qilish",
        tipDownload: "Yuklab olish",
        tipClose: "Yopish (Esc)",
        tipOriginalSize: "Asl o`lchamga qaytish"
      }
    };

    const jaJP = {
      name: "ja-JP",
      global: {
        undo: "\u5143\u306B\u623B\u3059",
        redo: "\u3084\u308A\u76F4\u3059",
        confirm: "OK",
        clear: "\u30AF\u30EA\u30A2"
      },
      Popconfirm: {
        positiveText: "OK",
        negativeText: "\u30AD\u30E3\u30F3\u30BB\u30EB"
      },
      Cascader: {
        placeholder: "\u9078\u629E\u3057\u3066\u304F\u3060\u3055\u3044",
        loading: "\u30ED\u30FC\u30C9\u4E2D",
        loadingRequiredMessage: label => `\u3059\u3079\u3066\u306E ${label} \u30B5\u30D6\u30CE\u30FC\u30C9\u3092\u30ED\u30FC\u30C9\u3057\u3066\u304B\u3089\u9078\u629E\u3067\u304D\u307E\u3059\u3002`
      },
      Time: {
        dateFormat: "yyyy-MM-dd",
        dateTimeFormat: "yyyy-MM-dd HH:mm:ss"
      },
      DatePicker: {
        yearFormat: "yyyy\u5E74",
        monthFormat: "MMM",
        dayFormat: "eeeeee",
        yearTypeFormat: "yyyy",
        monthTypeFormat: "yyyy-MM",
        dateFormat: "yyyy-MM-dd",
        dateTimeFormat: "yyyy-MM-dd HH:mm:ss",
        quarterFormat: "yyyy-qqq",
        weekFormat: "YYYY-w",
        clear: "\u30AF\u30EA\u30A2",
        now: "\u73FE\u5728",
        confirm: "OK",
        selectTime: "\u6642\u9593\u3092\u9078\u629E",
        selectDate: "\u65E5\u4ED8\u3092\u9078\u629E",
        datePlaceholder: "\u65E5\u4ED8\u3092\u9078\u629E",
        datetimePlaceholder: "\u9078\u629E",
        monthPlaceholder: "\u6708\u3092\u9078\u629E",
        yearPlaceholder: "\u5E74\u3092\u9078\u629E",
        quarterPlaceholder: "\u56DB\u534A\u671F\u3092\u9078\u629E",
        weekPlaceholder: "Select Week",
        startDatePlaceholder: "\u958B\u59CB\u65E5",
        endDatePlaceholder: "\u7D42\u4E86\u65E5",
        startDatetimePlaceholder: "\u958B\u59CB\u6642\u9593",
        endDatetimePlaceholder: "\u7D42\u4E86\u6642\u9593",
        startMonthPlaceholder: "\u958B\u59CB\u6708",
        endMonthPlaceholder: "\u7D42\u4E86\u6708",
        monthBeforeYear: false,
        firstDayOfWeek: 0,
        today: "\u4ECA\u65E5"
      },
      DataTable: {
        checkTableAll: "\u5168\u9078\u629E",
        uncheckTableAll: "\u5168\u9078\u629E\u53D6\u6D88",
        confirm: "OK",
        clear: "\u30EA\u30BB\u30C3\u30C8"
      },
      LegacyTransfer: {
        sourceTitle: "\u5143",
        targetTitle: "\u5148"
      },
      Transfer: {
        selectAll: "\u5168\u9078\u629E",
        unselectAll: "\u5168\u9078\u629E\u53D6\u6D88",
        clearAll: "\u30EA\u30BB\u30C3\u30C8",
        total: num => `\u5408\u8A08 ${num} \u9805\u76EE`,
        selected: num => `${num} \u500B\u306E\u9805\u76EE\u3092\u9078\u629E`
      },
      Empty: {
        description: "\u30C7\u30FC\u30BF\u306A\u3057"
      },
      Select: {
        placeholder: "\u9078\u629E\u3057\u3066\u304F\u3060\u3055\u3044"
      },
      TimePicker: {
        placeholder: "\u9078\u629E\u3057\u3066\u304F\u3060\u3055\u3044",
        positiveText: "OK",
        negativeText: "\u30AD\u30E3\u30F3\u30BB\u30EB",
        now: "\u73FE\u5728",
        clear: "\u30AF\u30EA\u30A2"
      },
      Pagination: {
        goto: "\u30DA\u30FC\u30B8\u30B8\u30E3\u30F3\u30D7",
        selectionSuffix: "\u30DA\u30FC\u30B8"
      },
      DynamicTags: {
        add: "\u8FFD\u52A0"
      },
      Log: {
        loading: "\u30ED\u30FC\u30C9\u4E2D"
      },
      Input: {
        placeholder: "\u5165\u529B\u3057\u3066\u304F\u3060\u3055\u3044"
      },
      InputNumber: {
        placeholder: "\u5165\u529B\u3057\u3066\u304F\u3060\u3055\u3044"
      },
      DynamicInput: {
        create: "\u8FFD\u52A0"
      },
      ThemeEditor: {
        title: "\u30C6\u30FC\u30DE\u30A8\u30C7\u30A3\u30BF",
        clearAllVars: "\u5168\u4EF6\u5909\u6570\u30AF\u30EA\u30A2",
        clearSearch: "\u691C\u7D22\u30AF\u30EA\u30A2",
        filterCompName: "\u30B3\u30F3\u30DD\u30CD\u30F3\u30C8\u540D\u3092\u30D5\u30A3\u30EB\u30BF",
        filterVarName: "\u5909\u6570\u3092\u30D5\u30A3\u30EB\u30BF",
        import: "\u30A4\u30F3\u30DD\u30FC\u30C8",
        export: "\u30A8\u30AF\u30B9\u30DD\u30FC\u30C8",
        restore: "\u30C7\u30D5\u30A9\u30EB\u30C8"
      },
      Image: {
        tipPrevious: "\u524D\u306E\u753B\u50CF (\u2190)",
        tipNext: "\u6B21\u306E\u753B\u50CF (\u2192)",
        tipCounterclockwise: "\u5DE6\u306B\u56DE\u8EE2",
        tipClockwise: "\u53F3\u306B\u56DE\u8EE2",
        tipZoomOut: "\u7E2E\u5C0F",
        tipZoomIn: "\u62E1\u5927",
        tipDownload: "\u30C0\u30A6\u30F3\u30ED\u30FC\u30C9",
        tipClose: "\u9589\u3058\u308B (Esc)",
        tipOriginalSize: "\u5143\u306E\u30B5\u30A4\u30BA\u306B\u623B\u3059"
      }
    };

    const koKR = {
      name: "ko-KR",
      global: {
        undo: "\uC2E4\uD589 \uCDE8\uC18C",
        redo: "\uB2E4\uC2DC \uC2E4\uD589",
        confirm: "\uD655\uC778",
        clear: "\uC9C0\uC6B0\uAE30"
      },
      Popconfirm: {
        positiveText: "\uD655\uC778",
        negativeText: "\uCDE8\uC18C"
      },
      Cascader: {
        placeholder: "\uC120\uD0DD\uD574 \uC8FC\uC138\uC694",
        loading: "\uBD88\uB7EC\uC624\uB294 \uC911",
        loadingRequiredMessage: label => `${label}\uC758 \uBAA8\uB4E0 \uD558\uC704 \uD56D\uBAA9\uC744 \uBD88\uB7EC\uC628 \uB4A4\uC5D0 \uC120\uD0DD\uD560 \uC218 \uC788\uC2B5\uB2C8\uB2E4.`
      },
      Time: {
        dateFormat: "yyyy-MM-dd",
        dateTimeFormat: "yyyy-MM-dd HH:mm:ss"
      },
      DatePicker: {
        yearFormat: "yyyy\uB144",
        monthFormat: "MMM",
        dayFormat: "eeeeee",
        yearTypeFormat: "yyyy",
        monthTypeFormat: "yyyy-MM",
        dateFormat: "yyyy-MM-dd",
        dateTimeFormat: "yyyy-MM-dd HH:mm:ss",
        quarterFormat: "yyyy-qqq",
        weekFormat: "YYYY-w",
        clear: "\uC9C0\uC6B0\uAE30",
        now: "\uD604\uC7AC",
        confirm: "\uD655\uC778",
        selectTime: "\uC2DC\uAC04 \uC120\uD0DD",
        selectDate: "\uB0A0\uC9DC \uC120\uD0DD",
        datePlaceholder: "\uB0A0\uC9DC \uC120\uD0DD",
        datetimePlaceholder: "\uB0A0\uC9DC \uBC0F \uC2DC\uAC04 \uC120\uD0DD",
        monthPlaceholder: "\uC6D4 \uC120\uD0DD",
        yearPlaceholder: "\uB144 \uC120\uD0DD",
        quarterPlaceholder: "\uBD84\uAE30 \uC120\uD0DD",
        weekPlaceholder: "Select Week",
        startDatePlaceholder: "\uC2DC\uC791 \uB0A0\uC9DC",
        endDatePlaceholder: "\uC885\uB8CC \uB0A0\uC9DC",
        startDatetimePlaceholder: "\uC2DC\uC791 \uB0A0\uC9DC \uBC0F \uC2DC\uAC04",
        endDatetimePlaceholder: "\uC885\uB8CC \uB0A0\uC9DC \uBC0F \uC2DC\uAC04",
        // FIXME: translation needed
        startMonthPlaceholder: "\uC2DC\uC791 \uC6D4",
        endMonthPlaceholder: "\uC885\uB8CC \uC6D4",
        monthBeforeYear: false,
        firstDayOfWeek: 6,
        today: "\uC624\uB298"
      },
      DataTable: {
        checkTableAll: "\uBAA8\uB450 \uC120\uD0DD",
        uncheckTableAll: "\uBAA8\uB450 \uC120\uD0DD \uD574\uC81C",
        confirm: "\uD655\uC778",
        clear: "\uC9C0\uC6B0\uAE30"
      },
      LegacyTransfer: {
        sourceTitle: "\uC6D0\uBCF8",
        targetTitle: "\uD0C0\uAE43"
      },
      // TODO: translation
      Transfer: {
        selectAll: "\uC804\uCCB4 \uC120\uD0DD",
        unselectAll: "\uC804\uCCB4 \uD574\uC81C",
        clearAll: "\uC804\uCCB4 \uC0AD\uC81C",
        total: num => `\uCD1D ${num} \uAC1C`,
        selected: num => `${num} \uAC1C \uC120\uD0DD`
      },
      Empty: {
        description: "\uB370\uC774\uD130 \uC5C6\uC74C"
      },
      Select: {
        placeholder: "\uC120\uD0DD\uD574 \uC8FC\uC138\uC694"
      },
      TimePicker: {
        placeholder: "\uC2DC\uAC04 \uC120\uD0DD",
        positiveText: "\uD655\uC778",
        negativeText: "\uCDE8\uC18C",
        now: "\uD604\uC7AC \uC2DC\uAC04",
        clear: "\uC9C0\uC6B0\uAE30"
      },
      Pagination: {
        goto: "\uC774\uB3D9",
        selectionSuffix: "\uD398\uC774\uC9C0"
      },
      DynamicTags: {
        add: "\uCD94\uAC00"
      },
      Log: {
        loading: "\uBD88\uB7EC\uC624\uB294 \uC911"
      },
      Input: {
        placeholder: "\uC785\uB825\uD574 \uC8FC\uC138\uC694"
      },
      InputNumber: {
        placeholder: "\uC785\uB825\uD574 \uC8FC\uC138\uC694"
      },
      DynamicInput: {
        create: "\uCD94\uAC00"
      },
      ThemeEditor: {
        title: "\uD14C\uB9C8 \uD3B8\uC9D1\uAE30",
        clearAllVars: "\uBAA8\uB4E0 \uBCC0\uC218 \uC9C0\uC6B0\uAE30",
        clearSearch: "\uAC80\uC0C9 \uC9C0\uC6B0\uAE30",
        filterCompName: "\uAD6C\uC131 \uC694\uC18C \uC774\uB984 \uD544\uD130",
        filterVarName: "\uBCC0\uC218 \uC774\uB984 \uD544\uD130",
        import: "\uAC00\uC838\uC624\uAE30",
        export: "\uB0B4\uBCF4\uB0B4\uAE30",
        restore: "\uAE30\uBCF8\uC73C\uB85C \uC7AC\uC124\uC815"
      },
      Image: {
        tipPrevious: "\uC774\uC804 (\u2190)",
        tipNext: "\uB2E4\uC74C (\u2192)",
        tipCounterclockwise: "\uC2DC\uACC4 \uBC18\uB300 \uBC29\uD5A5\uC73C\uB85C \uD68C\uC804",
        tipClockwise: "\uC2DC\uACC4 \uBC29\uD5A5\uC73C\uB85C \uD68C\uC804",
        tipZoomOut: "\uCD95\uC18C",
        tipZoomIn: "\uD655\uB300",
        tipDownload: "\uB2E4\uC6B4\uB85C\uB4DC",
        tipClose: "\uB2EB\uAE30 (Esc)",
        // TODO: translation
        tipOriginalSize: "\uC6D0\uBCF8 \uD06C\uAE30\uB85C \uD655\uB300"
      }
    };

    const idID = {
      name: "id-ID",
      global: {
        undo: "Membatalkan",
        redo: "Mem-perbarui",
        confirm: "Setuju",
        clear: "Bersihkan"
      },
      Popconfirm: {
        positiveText: "Setuju",
        negativeText: "Batalkan"
      },
      Cascader: {
        placeholder: "Mohon Pilih",
        loading: "Memuat",
        loadingRequiredMessage: label => `Mohon muat semua ${label} sebelum memeriksa.`
      },
      Time: {
        dateFormat: "dd-MM-yyyy",
        dateTimeFormat: "dd-MM-yyyy HH:mm:ss"
      },
      DatePicker: {
        yearFormat: "yyyy",
        monthFormat: "MMM",
        dayFormat: "eeeeee",
        yearTypeFormat: "yyyy",
        monthTypeFormat: "MM-yyyy",
        dateFormat: "dd-MM-yyyy",
        dateTimeFormat: "dd-MM-yyyy HH:mm:ss",
        quarterFormat: "yyyy-qqq",
        weekFormat: "YYYY-w",
        clear: "Bersihkan",
        now: "Sekarang",
        confirm: "Setuju",
        selectTime: "Pilih Waktu",
        selectDate: "Pilih Tanggal",
        datePlaceholder: "Pilih Tanggal",
        datetimePlaceholder: "Pilih Tanggal dan Waktu",
        monthPlaceholder: "Pilih Bulan",
        // FIXME: translation needed
        yearPlaceholder: "Pilih tahun",
        quarterPlaceholder: "Pilih perempat tahun",
        weekPlaceholder: "Select Week",
        startDatePlaceholder: "Tanggal Mulai",
        endDatePlaceholder: "Tanggal Selesai",
        startDatetimePlaceholder: "Tanggal dan Waktu Mulai",
        endDatetimePlaceholder: "Tanggal dan Waktu Selesai",
        // FIXME: translation needed
        startMonthPlaceholder: "Awal bulan",
        endMonthPlaceholder: "Akhir bulan",
        monthBeforeYear: true,
        firstDayOfWeek: 6,
        today: "Hari ini"
      },
      DataTable: {
        checkTableAll: "Pilih semua pada tabel",
        uncheckTableAll: "Batalkan pilihan semua",
        confirm: "Setuju",
        clear: "Bersihkan"
      },
      LegacyTransfer: {
        sourceTitle: "Sumber",
        targetTitle: "Tujuan"
      },
      // TODO: translation
      Transfer: {
        selectAll: "Pilih semua",
        unselectAll: "Batalkan pilihan",
        clearAll: "Bersihkan",
        total: num => `Total ${num} item`,
        selected: num => `${num} item dipilih`
      },
      Empty: {
        description: "Tidak ada data"
      },
      Select: {
        placeholder: "Mohon Pilih"
      },
      TimePicker: {
        placeholder: "Pilih Waktu",
        positiveText: "OK",
        negativeText: "Batalkan",
        now: "Sekarang",
        clear: "Bersihkan"
      },
      Pagination: {
        goto: "Ke",
        selectionSuffix: "halaman"
      },
      DynamicTags: {
        add: "Tambah"
      },
      Log: {
        loading: "Memuat"
      },
      Input: {
        placeholder: "Mohon isi"
      },
      InputNumber: {
        placeholder: "Mohon isi"
      },
      DynamicInput: {
        create: "Buat baru"
      },
      ThemeEditor: {
        title: "Pengaturan Tema",
        clearAllVars: "Bersihkan semua variabel",
        clearSearch: "Bersihkan pencarian",
        filterCompName: "Saring nama komponen",
        filterVarName: "Saring nama variabel",
        import: "Impor",
        export: "Ekspor",
        restore: "Setel ulang ke awal"
      },
      Image: {
        tipPrevious: "Gambar sebelumnya (\u2190)",
        tipNext: "Gambar berikutnya (\u2192)",
        tipCounterclockwise: "Berlawanan arah jarum jam",
        tipClockwise: "Searah jarum jam",
        tipZoomOut: "Zoom out",
        tipZoomIn: "Zoom in",
        tipDownload: "Download",
        tipClose: "Tutup (Esc)",
        // TODO: translation
        tipOriginalSize: "Zoom ke ukuran asli"
      }
    };

    const deDE = {
      name: "de-DE",
      global: {
        undo: "R\xFCckg\xE4ngig",
        redo: "Wiederholen",
        confirm: "Best\xE4tigen",
        clear: "L\xF6schen"
      },
      Popconfirm: {
        positiveText: "Best\xE4tigen",
        negativeText: "Abbrechen"
      },
      Cascader: {
        placeholder: "Bitte ausw\xE4hlen",
        loading: "Wird geladen",
        loadingRequiredMessage: label => `Bitte laden Sie alle Unterpunkte von ${label}, bevor Sie es ausw\xE4hlen.`
      },
      Time: {
        dateFormat: "dd.MM.yyyy",
        dateTimeFormat: "dd.MM.yyyy HH:mm:ss"
      },
      DatePicker: {
        yearFormat: "yyyy",
        monthFormat: "MMM",
        dayFormat: "eeeeee",
        yearTypeFormat: "yyyy",
        monthTypeFormat: "MM-yyyy",
        dateFormat: "dd.MM.yyyy",
        dateTimeFormat: "dd.MM.yyyy HH:mm:ss",
        quarterFormat: "yyyy-qqq",
        weekFormat: "YYYY-w",
        clear: "L\xF6schen",
        now: "Jetzt",
        confirm: "Best\xE4tigen",
        selectTime: "Uhrzeit ausw\xE4hlen",
        selectDate: "Datum ausw\xE4hlen",
        datePlaceholder: "Datum ausw\xE4hlen",
        datetimePlaceholder: "Datum und Uhrzeit ausw\xE4hlen",
        monthPlaceholder: "Monat ausw\xE4hlen",
        yearPlaceholder: "Jahr ausw\xE4hlen",
        quarterPlaceholder: "Quartal ausw\xE4hlen",
        weekPlaceholder: "Select Week",
        startDatePlaceholder: "Anfangsdatum",
        endDatePlaceholder: "Enddatum",
        startDatetimePlaceholder: "Anfangsdatum und Uhrzeit",
        endDatetimePlaceholder: "Enddatum und Uhrzeit",
        startMonthPlaceholder: "Anfangsmonat",
        endMonthPlaceholder: "Endmonat",
        monthBeforeYear: true,
        firstDayOfWeek: 0,
        today: "Heute"
      },
      DataTable: {
        checkTableAll: "Alles ausw\xE4hlen",
        uncheckTableAll: "Auswahl aufheben",
        confirm: "Best\xE4tigen",
        clear: "L\xF6schen"
      },
      LegacyTransfer: {
        sourceTitle: "Quelle",
        targetTitle: "Ziel"
      },
      Transfer: {
        selectAll: "Alle ausw\xE4hlen",
        unselectAll: "Alle abw\xE4hlen",
        clearAll: "Leeren",
        total: num => `Insgesamt ${num} Eintr\xE4ge`,
        selected: num => `${num} Eintr\xE4ge ausgew\xE4hlt`
      },
      Empty: {
        description: "Keine Daten"
      },
      Select: {
        placeholder: "Bitte ausw\xE4hlen"
      },
      TimePicker: {
        placeholder: "Uhrzeit ausw\xE4hlen",
        positiveText: "OK",
        negativeText: "Abbrechen",
        now: "Jetzt",
        clear: "L\xF6schen"
      },
      Pagination: {
        goto: "Gehe zu",
        selectionSuffix: "Seite"
      },
      DynamicTags: {
        add: "Hinzuf\xFCgen"
      },
      Log: {
        loading: "Wird geladen"
      },
      Input: {
        placeholder: "Bitte ausf\xFCllen"
      },
      InputNumber: {
        placeholder: "Bitte ausf\xFCllen"
      },
      DynamicInput: {
        create: "Erstellen"
      },
      ThemeEditor: {
        title: "Theme Editor",
        clearAllVars: "Alle Variablen l\xF6schen",
        clearSearch: "Suche l\xF6schen",
        filterCompName: "Filter Komponentenname",
        filterVarName: "Filter Variablenname",
        import: "Importieren",
        export: "Exportieren",
        restore: "Auf Standard zur\xFCcksetzen"
      },
      Image: {
        tipPrevious: "Vorheriges Bild (\u2190)",
        tipNext: "N\xE4chstes Bild (\u2192)",
        tipCounterclockwise: "Gegen Uhrzeigersinn",
        tipClockwise: "Uhrzeigersinn",
        tipZoomOut: "Rauszoomen",
        tipZoomIn: "Reinzoomen",
        tipDownload: "Download",
        tipClose: "Schlie\xDFen (Esc)",
        tipOriginalSize: "Zoom zur\xFCcksetzen"
      }
    };

    const nbNO = {
      name: "nb-NO",
      global: {
        undo: "Angre",
        redo: "Utf\xF8r likevel",
        confirm: "Bekreft",
        clear: "T\xF8m"
      },
      Popconfirm: {
        positiveText: "Bekreft",
        negativeText: "Avbryt"
      },
      Cascader: {
        placeholder: "Vennligst velg",
        loading: "Laster",
        loadingRequiredMessage: label => `Vennligst last alle underpunkter av ${label} f\xF8r du velger oppf\xF8ringen.`
      },
      Time: {
        dateFormat: "dd.MM.yyyy",
        dateTimeFormat: "dd.MM.yyyy HH:mm:ss"
      },
      DatePicker: {
        yearFormat: "yyyy",
        monthFormat: "MMM",
        dayFormat: "eeeeee",
        yearTypeFormat: "yyyy",
        monthTypeFormat: "MM.yyyy",
        dateFormat: "dd.MM.yyyy",
        dateTimeFormat: "dd.MM.yyyy HH:mm:ss",
        quarterFormat: "yyyy-qqq",
        weekFormat: "YYYY-w",
        clear: "T\xF8m",
        now: "N\xE5",
        confirm: "Bekreft",
        selectTime: "Velg tid",
        selectDate: "Velg dato",
        datePlaceholder: "Velg dato",
        datetimePlaceholder: "Velg dato og tid",
        monthPlaceholder: "Velg m\xE5ned",
        // FIXME: translation needed
        yearPlaceholder: "Select Year",
        quarterPlaceholder: "Select Quarter",
        weekPlaceholder: "Select Week",
        startDatePlaceholder: "Startdato",
        endDatePlaceholder: "Sluttdato",
        startDatetimePlaceholder: "Startdato og -tid",
        endDatetimePlaceholder: "Sluttdato og -tid",
        // FIXME: translation needed
        startMonthPlaceholder: "Start Month",
        endMonthPlaceholder: "End Month",
        monthBeforeYear: true,
        firstDayOfWeek: 0,
        today: "I dag"
      },
      DataTable: {
        checkTableAll: "Velg alt",
        uncheckTableAll: "Velg ingenting",
        confirm: "Bekreft",
        clear: "T\xF8m"
      },
      LegacyTransfer: {
        sourceTitle: "Kilde",
        targetTitle: "M\xE5l"
      },
      // TODO: translation
      Transfer: {
        selectAll: "Select all",
        unselectAll: "Unselect all",
        clearAll: "Clear",
        total: num => `Total ${num} items`,
        selected: num => `${num} items selected`
      },
      Empty: {
        description: "Ingen data"
      },
      Select: {
        placeholder: "Vennligst velg"
      },
      TimePicker: {
        placeholder: "Velg tid",
        positiveText: "OK",
        negativeText: "Avbryt",
        now: "N\xE5",
        clear: "T\xF8m"
      },
      Pagination: {
        goto: "G\xE5 til",
        selectionSuffix: "side"
      },
      DynamicTags: {
        add: "Legg til"
      },
      Log: {
        loading: "Laster"
      },
      Input: {
        placeholder: "Vennligst fyll ut"
      },
      InputNumber: {
        placeholder: "Vennligst fyll ut"
      },
      DynamicInput: {
        create: "Opprett"
      },
      ThemeEditor: {
        title: "Temaredigerer",
        clearAllVars: "Nullstill alle variabler",
        clearSearch: "T\xF8m s\xF8k",
        filterCompName: "Filtrer etter komponentnavn",
        filterVarName: "Filtrer etter variabelnavn",
        import: "Importer",
        export: "Eksporter",
        restore: "Nullstill til standardvalg"
      },
      Image: {
        tipPrevious: "Previous picture (\u2190)",
        tipNext: "Next picture (\u2192)",
        tipCounterclockwise: "Counterclockwise",
        tipClockwise: "Clockwise",
        tipZoomOut: "Zoom out",
        tipZoomIn: "Zoom in",
        tipDownload: "Download",
        tipClose: "Close (Esc)",
        // TODO: translation
        tipOriginalSize: "Zoom to original size"
      }
    };

    const nlNL = {
      name: "nl-NL",
      global: {
        undo: "Ongedaan Maken",
        redo: "Opnieuw Doen",
        confirm: "Bevestig",
        clear: "Wis"
      },
      Popconfirm: {
        positiveText: "Bevestig",
        negativeText: "Annuleer"
      },
      Cascader: {
        placeholder: "Selecteer a.u.b.",
        loading: "Laden",
        loadingRequiredMessage: label => `Laad alle afstammelingen van ${label} alvorens het te selecteren.`
      },
      Time: {
        dateFormat: "dd/MM/yyyy",
        dateTimeFormat: "dd/MM/yyyy HH:mm:ss"
      },
      DatePicker: {
        yearFormat: "yyyy",
        monthFormat: "MMM",
        dayFormat: "eeeeee",
        yearTypeFormat: "yyyy",
        monthTypeFormat: "MM/yyyy",
        dateFormat: "dd/MM/yyyy",
        dateTimeFormat: "dd/MM/yyyy HH:mm:ss",
        quarterFormat: "qqq yyyy",
        weekFormat: "YYYY-w",
        clear: "Wis",
        now: "Nu",
        confirm: "Bevestig",
        selectTime: "Selecteer Uur",
        selectDate: "Selecteer Datum",
        datePlaceholder: "Selecteer Datum",
        datetimePlaceholder: "Selecteer Dag en Uur",
        monthPlaceholder: "Selecteer Maand",
        yearPlaceholder: "Selecteer Jaar",
        quarterPlaceholder: "Selecteer Kwartaal",
        weekPlaceholder: "Select Week",
        startDatePlaceholder: "Begindatum",
        endDatePlaceholder: "Einddatum",
        startDatetimePlaceholder: "Begindatum en Uur",
        endDatetimePlaceholder: "Einddatum en Uur",
        startMonthPlaceholder: "Begin Maand",
        endMonthPlaceholder: "Eind Maand",
        monthBeforeYear: true,
        firstDayOfWeek: 0,
        today: "Vandaag"
      },
      DataTable: {
        checkTableAll: "Selecteer alles in de tabel",
        uncheckTableAll: "Alles in de tabel deselecteren",
        confirm: "Bevestig",
        clear: "Wis"
      },
      LegacyTransfer: {
        sourceTitle: "Bron",
        targetTitle: "Doel"
      },
      // TODO: translation
      Transfer: {
        selectAll: "Select all",
        unselectAll: "Unselect all",
        clearAll: "Clear",
        total: num => `Total ${num} items`,
        selected: num => `${num} items selected`
      },
      Empty: {
        description: "Geen Data"
      },
      Select: {
        placeholder: "Selecteer a.u.b."
      },
      TimePicker: {
        placeholder: "Selecteer Uur",
        positiveText: "OK",
        negativeText: "Annuleer",
        now: "Nu",
        clear: "Wis"
      },
      Pagination: {
        goto: "Ga naar",
        selectionSuffix: "pagina"
      },
      DynamicTags: {
        add: "Toevoegen"
      },
      Log: {
        loading: "Laden"
      },
      Input: {
        placeholder: "Invoeren a.u.b."
      },
      InputNumber: {
        placeholder: "Invoeren a.u.b."
      },
      DynamicInput: {
        create: "Cre\xEBer"
      },
      ThemeEditor: {
        title: "Thema Bewerker",
        clearAllVars: "Wis Alle Variabelen",
        clearSearch: "Wis Zoekopdracht",
        filterCompName: "Filter Componentsnaam",
        filterVarName: "Filter Variablenaam",
        import: "Importen",
        export: "Exporteren",
        restore: "Reset naar Standaard"
      },
      Image: {
        tipPrevious: "Vorige afbeelding (\u2190)",
        tipNext: "Volgende afbeelding (\u2192)",
        tipCounterclockwise: "Tegen de klok in",
        tipClockwise: "Met de klok mee",
        tipZoomOut: "Uitzoomen",
        tipZoomIn: "Inzoomen",
        tipDownload: "Download",
        tipClose: "Sluiten (Esc)",
        // TODO: translation
        tipOriginalSize: "Zoom to original size"
      }
    };

    const frFR = {
      name: "fr-FR",
      global: {
        undo: "D\xE9faire",
        redo: "Refaire",
        confirm: "Confirmer",
        clear: "Effacer"
      },
      Popconfirm: {
        positiveText: "Confirmer",
        negativeText: "Annuler"
      },
      Cascader: {
        placeholder: "S\xE9lectionner",
        loading: "Chargement",
        loadingRequiredMessage: label => `Charger tous les enfants de ${label} avant de le s\xE9lectionner`
      },
      Time: {
        dateFormat: "dd/MM/yyyy",
        dateTimeFormat: "dd/MM/yyyy HH:mm:ss"
      },
      DatePicker: {
        yearFormat: "yyyy",
        monthFormat: "MMM",
        dayFormat: "eeeeee",
        yearTypeFormat: "yyyy",
        monthTypeFormat: "MM/yyyy",
        dateFormat: "dd/MM/yyyy",
        dateTimeFormat: "dd/MM/yyyy HH:mm:ss",
        quarterFormat: "qqq yyyy",
        weekFormat: "YYYY-w",
        clear: "Effacer",
        now: "Maintenant",
        confirm: "Confirmer",
        selectTime: "S\xE9lectionner l'heure",
        selectDate: "S\xE9lectionner la date",
        datePlaceholder: "S\xE9lectionner la date",
        datetimePlaceholder: "S\xE9lectionner la date et l'heure",
        monthPlaceholder: "S\xE9lectionner le mois",
        yearPlaceholder: "S\xE9lectionner l'ann\xE9e",
        quarterPlaceholder: "S\xE9lectionner le trimestre",
        weekPlaceholder: "Select Week",
        startDatePlaceholder: "Date de d\xE9but",
        endDatePlaceholder: "Date de fin",
        startDatetimePlaceholder: "Date et heure de d\xE9but",
        endDatetimePlaceholder: "Date et heure de fin",
        startMonthPlaceholder: "Mois de d\xE9but",
        endMonthPlaceholder: "Mois de fin",
        monthBeforeYear: true,
        firstDayOfWeek: 0,
        today: "Aujourd'hui"
      },
      DataTable: {
        checkTableAll: "S\xE9lectionner tout",
        uncheckTableAll: "D\xE9s\xE9lectionner tout",
        confirm: "Confirmer",
        clear: "Effacer"
      },
      LegacyTransfer: {
        sourceTitle: "Source",
        targetTitle: "Cible"
      },
      Transfer: {
        selectAll: "S\xE9lectionner tout",
        unselectAll: "D\xE9s\xE9lectionner tout",
        clearAll: "Effacer",
        total: num => `Total ${num} \xE9l\xE9ments`,
        selected: num => `${num} \xE9l\xE9ments s\xE9lectionn\xE9s`
      },
      Empty: {
        description: "Aucune donn\xE9e"
      },
      Select: {
        placeholder: "S\xE9lectionner"
      },
      TimePicker: {
        placeholder: "S\xE9lectionner l'heure",
        positiveText: "OK",
        negativeText: "Annuler",
        now: "Maintenant",
        clear: "Effacer"
      },
      Pagination: {
        goto: "Aller \xE0",
        selectionSuffix: "page"
      },
      DynamicTags: {
        add: "Ajouter"
      },
      Log: {
        loading: "Chargement"
      },
      Input: {
        placeholder: "Saisir"
      },
      InputNumber: {
        placeholder: "Saisir"
      },
      DynamicInput: {
        create: "Cr\xE9er"
      },
      ThemeEditor: {
        title: "\xC9diteur de th\xE8me",
        clearAllVars: "Effacer toutes les variables",
        clearSearch: "Effacer la recherche",
        filterCompName: "Filtrer par nom de composant",
        filterVarName: "Filtrer par nom de variable",
        import: "Importer",
        export: "Exporter",
        restore: "R\xE9initialiser"
      },
      Image: {
        tipPrevious: "Image pr\xE9c\xE9dente (\u2190)",
        tipNext: "Image suivante (\u2192)",
        tipCounterclockwise: "Sens antihoraire",
        tipClockwise: "Sens horaire",
        tipZoomOut: "D\xE9zoomer",
        tipZoomIn: "Zoomer",
        tipDownload: "Descargar",
        tipClose: "Fermer (\xC9chap.)",
        tipOriginalSize: "Zoom \xE0 la taille originale"
      }
    };

    const esAR = {
      name: "es-AR",
      global: {
        undo: "Deshacer",
        redo: "Rehacer",
        confirm: "Confirmar",
        clear: "Borrar"
      },
      Popconfirm: {
        positiveText: "Confirmar",
        negativeText: "Cancelar"
      },
      Cascader: {
        placeholder: "Seleccionar por favor",
        loading: "Cargando",
        loadingRequiredMessage: label => `Por favor, cargue los descendientes de ${label} antes de marcarlo.`
      },
      Time: {
        dateFormat: "yyyy-MM-dd",
        dateTimeFormat: "yyyy-MM-dd HH:mm:ss"
      },
      DatePicker: {
        yearFormat: "yyyy",
        monthFormat: "MMM",
        dayFormat: "eeeeee",
        yearTypeFormat: "yyyy",
        monthTypeFormat: "yyyy-MM",
        dateFormat: "yyyy-MM-dd",
        dateTimeFormat: "yyyy-MM-dd HH:mm:ss",
        quarterFormat: "yyyy-qqq",
        weekFormat: "YYYY-w",
        clear: "Borrar",
        now: "Ahora",
        confirm: "Confirmar",
        selectTime: "Seleccionar hora",
        selectDate: "Seleccionar fecha",
        datePlaceholder: "Seleccionar fecha",
        datetimePlaceholder: "Seleccionar fecha y hora",
        monthPlaceholder: "Seleccionar mes",
        yearPlaceholder: "Seleccionar a\xF1o",
        quarterPlaceholder: "Seleccionar Trimestre",
        weekPlaceholder: "Select Week",
        startDatePlaceholder: "Fecha de inicio",
        endDatePlaceholder: "Fecha final",
        startDatetimePlaceholder: "Fecha y hora de inicio",
        endDatetimePlaceholder: "Fecha y hora final",
        monthBeforeYear: true,
        // FIXME: translation needed
        startMonthPlaceholder: "Start Month",
        endMonthPlaceholder: "End Month",
        // 0 is Monday / 0 es Lunes
        firstDayOfWeek: 6,
        today: "Hoy"
      },
      DataTable: {
        checkTableAll: "Seleccionar todo de la tabla",
        uncheckTableAll: "Deseleccionar todo de la tabla",
        confirm: "Confirmar",
        clear: "Limpiar"
      },
      LegacyTransfer: {
        sourceTitle: "Fuente",
        targetTitle: "Objetivo"
      },
      // TODO: translation
      Transfer: {
        selectAll: "Select all",
        unselectAll: "Unselect all",
        clearAll: "Clear",
        total: num => `Total ${num} items`,
        selected: num => `${num} items selected`
      },
      Empty: {
        description: "Sin datos"
      },
      Select: {
        placeholder: "Seleccionar por favor"
      },
      TimePicker: {
        placeholder: "Seleccionar hora",
        positiveText: "OK",
        negativeText: "Cancelar",
        now: "Ahora",
        clear: "Borrar"
      },
      Pagination: {
        goto: "Ir a",
        selectionSuffix: "p\xE1gina"
      },
      DynamicTags: {
        add: "Agregar"
      },
      Log: {
        loading: "Cargando"
      },
      Input: {
        placeholder: "Ingrese datos por favor"
      },
      InputNumber: {
        placeholder: "Ingrese datos por favor"
      },
      DynamicInput: {
        create: "Crear"
      },
      ThemeEditor: {
        title: "Editor de Tema",
        clearAllVars: "Limpiar todas las variables",
        clearSearch: "Limpiar b\xFAsqueda",
        filterCompName: "Filtro para nombre del componente",
        filterVarName: "Filtro para nombre de la variable",
        import: "Importar",
        export: "Exportar",
        restore: "Restablecer los valores por defecto"
      },
      Image: {
        tipPrevious: "Imagen anterior (\u2190)",
        tipNext: "Siguiente imagen (\u2192)",
        tipCounterclockwise: "Sentido antihorario",
        tipClockwise: "Sentido horario",
        tipZoomOut: "Alejar",
        tipZoomIn: "Acercar",
        tipDownload: "Descargar",
        tipClose: "Cerrar (Esc)",
        // TODO: translation
        tipOriginalSize: "Zoom to original size"
      }
    };

    const itIT = {
      name: "it-IT",
      global: {
        undo: "Annulla",
        redo: "Ripeti",
        confirm: "Conferma",
        clear: "Cancella"
      },
      Popconfirm: {
        positiveText: "Conferma",
        negativeText: "Annulla"
      },
      Cascader: {
        placeholder: "Si prega di selezionare",
        loading: "Caricamento",
        loadingRequiredMessage: label => `Carica tutti i discendenti di ${label} prima di controllarlo.`
      },
      Time: {
        dateFormat: "dd/MM/yyyy",
        dateTimeFormat: "dd/MM/yyyy HH:mm:ss"
      },
      DatePicker: {
        yearFormat: "yyyy",
        monthFormat: "MMM",
        dayFormat: "eeeeee",
        yearTypeFormat: "yyyy",
        monthTypeFormat: "yyyy-MM",
        dateFormat: "yyyy-MM-dd",
        dateTimeFormat: "yyyy-MM-dd HH:mm:ss",
        quarterFormat: "yyyy-qqq",
        weekFormat: "YYYY-w",
        clear: "Cancella",
        now: "Adesso",
        confirm: "Conferma",
        selectTime: "Seleziona ora",
        selectDate: "Seleziona data",
        datePlaceholder: "Seleziona data",
        datetimePlaceholder: "Seleziona data e ora",
        monthPlaceholder: "Seleziona mese",
        yearPlaceholder: "Seleziona anno",
        quarterPlaceholder: "Seleziona trimestre",
        weekPlaceholder: "Select Week",
        startDatePlaceholder: "Data inizio",
        endDatePlaceholder: "Data fine",
        startDatetimePlaceholder: "Data e ora di inizio",
        endDatetimePlaceholder: "Data e ora di fine",
        startMonthPlaceholder: "Mese di inizio",
        endMonthPlaceholder: "Mese di fine",
        monthBeforeYear: true,
        firstDayOfWeek: 0,
        today: "Oggi"
      },
      DataTable: {
        checkTableAll: "Seleziona tutto nella tabella",
        uncheckTableAll: "Deseleziona tutto nella tabella",
        confirm: "Conferma",
        clear: "Cancella"
      },
      LegacyTransfer: {
        sourceTitle: "Fonte",
        targetTitle: "Destinazione"
      },
      Transfer: {
        selectAll: "Seleziona tutto",
        unselectAll: "Deseleziona tutto",
        clearAll: "Pulisci",
        total: num => {
          if (num !== 1) return `${num} elementi in totale`;
          return "1 elemento in totale";
        },
        selected: num => {
          if (num !== 1) return `${num} elementi selezionati`;
          return "1 elemento selezionato";
        }
      },
      Empty: {
        description: "Nessun Dato"
      },
      Select: {
        placeholder: "Si prega di selezionare"
      },
      TimePicker: {
        placeholder: "Seleziona ora",
        positiveText: "OK",
        negativeText: "Annulla",
        now: "Ora",
        clear: "Cancella"
      },
      Pagination: {
        goto: "Vai a",
        selectionSuffix: "per pagina"
      },
      DynamicTags: {
        add: "Aggiungi"
      },
      Log: {
        loading: "Caricamento"
      },
      Input: {
        placeholder: "Si prega di inserire"
      },
      InputNumber: {
        placeholder: "Si prega di inserire"
      },
      DynamicInput: {
        create: "Crea"
      },
      ThemeEditor: {
        title: "Editor Tema",
        clearAllVars: "Cancella tutte le variabili",
        clearSearch: "Cancella ricerca",
        filterCompName: "Filtra componenti",
        filterVarName: "Filtra variabili",
        import: "Importa",
        export: "Esporta",
        restore: "Ripristina"
      },
      Image: {
        tipPrevious: "Immagine precedente (\u2190)",
        tipNext: "Immagine successiva (\u2192)",
        tipCounterclockwise: "Ruota a sinistra",
        tipClockwise: "Ruota a destra",
        tipZoomOut: "Ingrandisci",
        tipZoomIn: "Riduci",
        tipDownload: "Download",
        tipClose: "Chiudi (Esc)",
        tipOriginalSize: "Torna alla dimensione originale"
      }
    };

    const skSK = {
      name: "sk-SK",
      global: {
        undo: "Sp\xE4\u0165",
        redo: "Obnovi\u0165",
        confirm: "Potvrdi\u0165",
        clear: "Vy\u010Disti\u0165"
      },
      Popconfirm: {
        positiveText: "Potvrdi\u0165",
        negativeText: "Zru\u0161i\u0165"
      },
      Cascader: {
        placeholder: "Pros\xEDm vyberte",
        loading: "Na\u010D\xEDtavanie",
        loadingRequiredMessage: label => `Pros\xEDm na\u010D\xEDtajte v\u0161etk\xFDch ${label} potomkov pred kontrolou.`
      },
      Time: {
        dateFormat: "d-M-yyyy",
        dateTimeFormat: "d-M-yyyy HH:mm:ss"
      },
      DatePicker: {
        yearFormat: "yyyy",
        monthFormat: "MMM",
        dayFormat: "EEEE",
        yearTypeFormat: "yyyy",
        monthTypeFormat: "MMM-yyyy",
        dateFormat: "d-M-yyyy",
        dateTimeFormat: "d-M-yyyy HH:mm:ss",
        quarterFormat: "qqq-yyyy",
        weekFormat: "YYYY-w",
        clear: "Vy\u010Disti\u0165",
        now: "Teraz",
        confirm: "Potvrdi\u0165",
        selectTime: "Vybra\u0165 \u010Das",
        selectDate: "Vybra\u0165 d\xE1tum",
        datePlaceholder: "Vyberte \u010Das",
        datetimePlaceholder: "Vyberte d\xE1tum a \u010Das",
        monthPlaceholder: "Vyberte mesiac",
        yearPlaceholder: "Vyberte rok",
        quarterPlaceholder: "Vyberte \u0161tvr\u0165rok",
        weekPlaceholder: "Vyberte t\xFD\u017Ede\u0148",
        startDatePlaceholder: "D\xE1tum za\u010Diatku",
        endDatePlaceholder: "D\xE1tum ukon\u010Denia",
        startDatetimePlaceholder: "D\xE1tum a \u010Das za\u010Diatku",
        endDatetimePlaceholder: "D\xE1tum a \u010Das ukon\u010Denia ",
        startMonthPlaceholder: "Za\u010Diatok mesiaca",
        endMonthPlaceholder: "Koniec mesiaca",
        monthBeforeYear: true,
        firstDayOfWeek: 6,
        today: "Dnes"
      },
      DataTable: {
        checkTableAll: "Vybra\u0165 v\u0161etko v tabu\u013Eke",
        uncheckTableAll: "Zru\u0161i\u0165 v\xFDber v\u0161etk\xE9ho v tabu\u013Eke ",
        confirm: "Potvrdi\u0165",
        clear: "Vy\u010Disti\u0165"
      },
      LegacyTransfer: {
        sourceTitle: "Zdroj",
        targetTitle: "Cie\u013E"
      },
      Transfer: {
        selectAll: "Vybra\u0165 v\u0161etko",
        unselectAll: "odzna\u010Di\u0165 v\u0161etko",
        clearAll: "Vy\u010Disti\u0165",
        total: num => `Celkom ${num} polo\u017Eiek`,
        selected: num => `Vybrat\xFDch ${num} polo\u017Eiek`
      },
      Empty: {
        description: "\u017Diadne d\xE1ta"
      },
      Select: {
        placeholder: "Pros\xEDm vyberte"
      },
      TimePicker: {
        placeholder: "Vybra\u0165 \u010Das",
        positiveText: "OK",
        negativeText: "Zru\u0161i\u0165",
        now: "Teraz",
        clear: "Vy\u010Disti\u0165"
      },
      Pagination: {
        goto: "\xCDs\u0165 na",
        selectionSuffix: "Strana"
      },
      DynamicTags: {
        add: "Prida\u0165"
      },
      Log: {
        loading: "Na\u010D\xEDtavanie"
      },
      Input: {
        placeholder: "Zadajte"
      },
      InputNumber: {
        placeholder: "Zadajte"
      },
      DynamicInput: {
        create: "Vytvori\u0165"
      },
      ThemeEditor: {
        title: "Editor t\xE9m",
        clearAllVars: "Vymaza\u0165 v\u0161etky premenn\xE9",
        clearSearch: "Vymaza\u0165 vyh\u013Ead\xE1vanie",
        filterCompName: "Filtrova\u0165 n\xE1zov komponentu",
        filterVarName: "Filtrova\u0165 n\xE1zov premennej",
        import: "Importova\u0165",
        export: "Exportova\u0165",
        restore: "Obnovi\u0165 p\xF4vodn\xE9 nastavenia"
      },
      Image: {
        tipPrevious: "Predch\xE1dzaj\xFAci obr\xE1zok (\u2190)",
        tipNext: "\u010Eal\u0161\xED obr\xE1zok (\u2192)",
        tipCounterclockwise: "Proti smeru hodinov\xFDch ru\u010Di\u010Diek",
        tipClockwise: "V smere hodinov\xFDch ru\u010Di\u010Diek",
        tipZoomOut: "Oddiali\u0165",
        tipZoomIn: "Pribl\xED\u017Ei\u0165",
        tipDownload: "S\u0165ahova\u0165",
        tipClose: "Zavrie\u0165 (Esc)",
        tipOriginalSize: "Pribl\xED\u017Ei\u0165 na p\xF4vodn\xFA ve\u013Ekos\u0165"
      }
    };

    const csCZ = {
      name: "cs-CZ",
      global: {
        undo: "Zp\u011Bt",
        redo: "Obnovit",
        confirm: "Potvrdit",
        clear: "Vy\u010Distit"
      },
      Popconfirm: {
        positiveText: "Potvrdit",
        negativeText: "Zru\u0161it"
      },
      Cascader: {
        placeholder: "Pros\xEDm vyberte",
        loading: "Na\u010D\xEDt\xE1n\xED",
        loadingRequiredMessage: label => `Pros\xEDm na\u010Dt\u011Bte p\u0159ed kontrolou v\u0161echny potomky pro ${label}.`
      },
      Time: {
        dateFormat: "d-M-yyyy",
        dateTimeFormat: "d-M-yyyy HH:mm:ss"
      },
      DatePicker: {
        yearFormat: "yyyy",
        monthFormat: "MMM",
        dayFormat: "EEEE",
        yearTypeFormat: "yyyy",
        monthTypeFormat: "MMM-yyyy",
        dateFormat: "d-M-yyyy",
        dateTimeFormat: "d-M-yyyy HH:mm:ss",
        quarterFormat: "qqq-yyyy",
        weekFormat: "YYYY-w",
        clear: "Vy\u010Distit",
        now: "Te\u010F",
        confirm: "Potvrdit",
        selectTime: "Vybrat \u010Das",
        selectDate: "Vybrat datum",
        datePlaceholder: "Vyberte \u010Das",
        datetimePlaceholder: "Vyberte datum a \u010Das",
        monthPlaceholder: "Vyberte m\u011Bs\xEDc",
        yearPlaceholder: "Vyberte rok",
        quarterPlaceholder: "Vyberte \u010Dtvrtlet\xED",
        weekPlaceholder: "Vyberte t\xFDden",
        startDatePlaceholder: "Datum za\u010D\xE1tku",
        endDatePlaceholder: "Datum ukon\u010Den\xED",
        startDatetimePlaceholder: "Datum a \u010Das za\u010D\xE1tku",
        endDatetimePlaceholder: "Datum a \u010Das ukon\u010Den\xED ",
        startMonthPlaceholder: "Za\u010D\xE1tek m\u011Bs\xEDce",
        endMonthPlaceholder: "Konec m\u011Bs\xEDce",
        monthBeforeYear: true,
        firstDayOfWeek: 6,
        today: "Dnes"
      },
      DataTable: {
        checkTableAll: "Vybrat v\u0161e v tabulce",
        uncheckTableAll: "Zru\u0161it v\xFDb\u011Br v\u0161eho v tabulce ",
        confirm: "Potvrdit",
        clear: "Vy\u010Distit"
      },
      LegacyTransfer: {
        sourceTitle: "Zdroj",
        targetTitle: "C\xEDl"
      },
      Transfer: {
        selectAll: "Vybrat v\u0161e",
        unselectAll: "Odzna\u010Dit v\u0161e",
        clearAll: "Vy\u010Distit",
        total: num => `Celkem ${num} polo\u017Eek`,
        selected: num => `${num} polo\u017Eek vybr\xE1no`
      },
      Empty: {
        description: "\u017D\xE1dn\xE1 data"
      },
      Select: {
        placeholder: "Pros\xEDm vyberte"
      },
      TimePicker: {
        placeholder: "Vybrat \u010Das",
        positiveText: "OK",
        negativeText: "Zru\u0161it",
        now: "Te\u010F",
        clear: "Vy\u010Distit"
      },
      Pagination: {
        goto: "J\xEDt na",
        selectionSuffix: "Strana"
      },
      DynamicTags: {
        add: "P\u0159idat"
      },
      Log: {
        loading: "Na\u010D\xEDt\xE1n\xED"
      },
      Input: {
        placeholder: "Zadejte"
      },
      InputNumber: {
        placeholder: "Zadejte"
      },
      DynamicInput: {
        create: "Vytvo\u0159it"
      },
      ThemeEditor: {
        title: "Editor t\xE9mat",
        clearAllVars: "Vymazat v\u0161echny prom\u011Bnn\xE9",
        clearSearch: "Vymazat vyhled\xE1v\xE1n\xED",
        filterCompName: "Filtrovat n\xE1zev komponenty",
        filterVarName: "Filztrovat n\xE1zev prom\u011Bnn\xE9",
        import: "Importovat",
        export: "Exportovat",
        restore: "Obnovit p\u016Fvodn\xED nastaven\xED"
      },
      Image: {
        tipPrevious: "P\u0159edchoz\xED obr\xE1zek (\u2190)",
        tipNext: "Dal\u0161\xED obr\xE1zek (\u2192)",
        tipCounterclockwise: "Proti sm\u011Bru hodinov\xFDch ru\u010Di\u010Dek",
        tipClockwise: "Ve sm\u011Bru hodinov\xFDch ru\u010Di\u010Dek",
        tipZoomOut: "Odd\xE1lit",
        tipZoomIn: "P\u0159ibl\xED\u017Eit",
        tipDownload: "St\xE1hnout",
        tipClose: "Zav\u0159\xEDt (Esc)",
        tipOriginalSize: "P\u0159ibl\xED\u017Eit na p\u016Fvodn\xED velikost"
      }
    };

    const enGB$1 = {
      name: "en-GB",
      global: {
        undo: "Undo",
        redo: "Redo",
        confirm: "Confirm",
        clear: "Clear"
      },
      Popconfirm: {
        positiveText: "Confirm",
        negativeText: "Cancel"
      },
      Cascader: {
        placeholder: "Please Select",
        loading: "Loading",
        loadingRequiredMessage: label => `Please load all ${label}'s descendants before checking it.`
      },
      Time: {
        dateFormat: "yyyy/MM/dd",
        dateTimeFormat: "yyyy/MM/dd HH:mm:ss"
      },
      DatePicker: {
        yearFormat: "yyyy",
        monthFormat: "MMM",
        dayFormat: "eeeeee",
        yearTypeFormat: "yyyy",
        monthTypeFormat: "yyyy/MM",
        dateFormat: "yyyy/MM/dd",
        dateTimeFormat: "yyyy/MM/dd HH:mm:ss",
        quarterFormat: "yyyy-qqq",
        weekFormat: "YYYY-w",
        clear: "Clear",
        now: "Now",
        confirm: "Confirm",
        selectTime: "Select Time",
        selectDate: "Select Date",
        datePlaceholder: "Select Date",
        datetimePlaceholder: "Select Date and Time",
        monthPlaceholder: "Select Month",
        yearPlaceholder: "Select Year",
        quarterPlaceholder: "Select Quarter",
        weekPlaceholder: "Select Week",
        startDatePlaceholder: "Start Date",
        endDatePlaceholder: "End Date",
        startDatetimePlaceholder: "Start Date and Time",
        endDatetimePlaceholder: "End Date and Time",
        startMonthPlaceholder: "Start Month",
        endMonthPlaceholder: "End Month",
        monthBeforeYear: true,
        firstDayOfWeek: 0,
        today: "Today"
      },
      DataTable: {
        checkTableAll: "Select all in the table",
        uncheckTableAll: "Unselect all in the table",
        confirm: "Confirm",
        clear: "Clear"
      },
      LegacyTransfer: {
        sourceTitle: "Source",
        targetTitle: "Target"
      },
      Transfer: {
        selectAll: "Select all",
        unselectAll: "Unselect all",
        clearAll: "Clear",
        total: num => `Total ${num} items`,
        selected: num => `${num} items selected`
      },
      Empty: {
        description: "No Data"
      },
      Select: {
        placeholder: "Please Select"
      },
      TimePicker: {
        placeholder: "Select Time",
        positiveText: "OK",
        negativeText: "Cancel",
        now: "Now",
        clear: "Clear"
      },
      Pagination: {
        goto: "Goto",
        selectionSuffix: "page"
      },
      DynamicTags: {
        add: "Add"
      },
      Log: {
        loading: "Loading"
      },
      Input: {
        placeholder: "Please Input"
      },
      InputNumber: {
        placeholder: "Please Input"
      },
      DynamicInput: {
        create: "Create"
      },
      ThemeEditor: {
        title: "Theme Editor",
        clearAllVars: "Clear All Variables",
        clearSearch: "Clear Search",
        filterCompName: "Filter Component Name",
        filterVarName: "Filter Variable Name",
        import: "Import",
        export: "Export",
        restore: "Reset to Default"
      },
      Image: {
        tipPrevious: "Previous picture (\u2190)",
        tipNext: "Next picture (\u2192)",
        tipCounterclockwise: "Counterclockwise",
        tipClockwise: "Clockwise",
        tipZoomOut: "Zoom out",
        tipZoomIn: "Zoom in",
        tipDownload: "Download",
        tipClose: "Close (Esc)",
        // TODO: translation
        tipOriginalSize: "Zoom to original size"
      }
    };

    const plPL = {
      name: "pl-PL",
      global: {
        undo: "Cofnij",
        redo: "Pon\xF3w",
        confirm: "Potwierd\u017A",
        clear: "Wyczy\u015B\u0107"
      },
      Popconfirm: {
        positiveText: "Potwierd\u017A",
        negativeText: "Anuluj"
      },
      Cascader: {
        placeholder: "Wybierz",
        loading: "\u0141adowanie",
        loadingRequiredMessage: label => `Prosz\u0119 za\u0142adowa\u0107 wszystkie ${label}'s elementy przed sprawdzeniem.`
      },
      Time: {
        dateFormat: "yyyy-MM-dd",
        dateTimeFormat: "yyyy-MM-dd HH:mm:ss"
      },
      DatePicker: {
        yearFormat: "yyyy",
        monthFormat: "MMM",
        dayFormat: "eeeeee",
        yearTypeFormat: "yyyy",
        monthTypeFormat: "yyyy-MM",
        dateFormat: "yyyy-MM-dd",
        dateTimeFormat: "yyyy-MM-dd HH:mm:ss",
        quarterFormat: "yyyy-qqq",
        weekFormat: "YYYY-w",
        clear: "Wyczy\u015B\u0107",
        now: "Teraz",
        confirm: "Potwierd\u017A",
        selectTime: "Wybierz czas",
        selectDate: "Wybierz dat\u0119",
        datePlaceholder: "Wybierz dat\u0119",
        datetimePlaceholder: "Wybierz dat\u0119 i czas",
        monthPlaceholder: "Wybierz miesi\u0105c",
        yearPlaceholder: "Wybierz rok",
        quarterPlaceholder: "Wybierz kwarta\u0142",
        weekPlaceholder: "Select Week",
        startDatePlaceholder: "Data rozpocz\u0119cia",
        endDatePlaceholder: "Data zako\u0144czenia",
        startDatetimePlaceholder: "Data i godzina rozpocz\u0119cia",
        endDatetimePlaceholder: "Data i godzina zako\u0144czenia",
        // FIXME: translation needed
        startMonthPlaceholder: "Start Month",
        endMonthPlaceholder: "End Month",
        monthBeforeYear: true,
        firstDayOfWeek: 6,
        today: "Dzi\u015B"
      },
      DataTable: {
        checkTableAll: "Zaznacz wszystkie w tabeli",
        uncheckTableAll: "Odznacz wszystkie w tabeli",
        confirm: "Potwierd\u017A",
        clear: "Wyczy\u015B\u0107"
      },
      LegacyTransfer: {
        sourceTitle: "\u0179r\xF3d\u0142o",
        targetTitle: "Cel"
      },
      // TODO: translation
      Transfer: {
        selectAll: "Select all",
        unselectAll: "Unselect all",
        clearAll: "Clear",
        total: num => `Total ${num} items`,
        selected: num => `${num} items selected`
      },
      Empty: {
        description: "Brak danych"
      },
      Select: {
        placeholder: "Wybierz"
      },
      TimePicker: {
        placeholder: "Wybierz czas",
        positiveText: "Potwierd\u017A",
        negativeText: "Anuluj",
        now: "Teraz",
        clear: "Wyczy\u015B\u0107"
      },
      Pagination: {
        goto: "Id\u017A do",
        selectionSuffix: "strona"
      },
      DynamicTags: {
        add: "Dodaj"
      },
      Log: {
        loading: "\u0141adowanie"
      },
      Input: {
        placeholder: "Wprowad\u017A dane"
      },
      InputNumber: {
        placeholder: "Wprowad\u017A dane"
      },
      DynamicInput: {
        create: "Utw\xF3rz"
      },
      ThemeEditor: {
        title: "Edytor motyw\xF3w",
        clearAllVars: "Wyczy\u015B\u0107 wszystkie zmienne",
        clearSearch: "Wyczy\u015B\u0107 wyszukiwanie",
        filterCompName: "Nazwa komponentu filtra",
        filterVarName: "Nazwa zmiennej filtra",
        import: "Importuj",
        export: "Eksportuj",
        restore: "Przywr\xF3\u0107 ustawienia domy\u015Blne"
      },
      Image: {
        tipPrevious: "Poprzednie zdj\u0119cie (\u2190)",
        tipNext: "Nast\u0119pne zdj\u0119cie (\u2192)",
        tipCounterclockwise: "Przeciwnie do ruchu wskaz\xF3wek zegara",
        tipClockwise: "Zgodnie z ruchem wskaz\xF3wek zegara",
        tipZoomOut: "Powi\u0119ksz",
        tipZoomIn: "Pomniejsz",
        tipDownload: "Pobierz",
        tipClose: "Zamknij (Esc)",
        // TODO: translation
        tipOriginalSize: "Zoom to original size"
      }
    };

    const ptBR$1 = {
      name: "pt-BR",
      global: {
        undo: "Desfazer",
        redo: "Refazer",
        confirm: "Confirmar",
        clear: "Limpar"
      },
      Popconfirm: {
        positiveText: "Confirmar",
        negativeText: "Cancelar"
      },
      Cascader: {
        placeholder: "Por favor selecione",
        loading: "Carregando",
        loadingRequiredMessage: label => `Carregue todos os descendentes de ${label} antes de verificar.`
      },
      Time: {
        dateFormat: "dd/MM/yyyy",
        dateTimeFormat: "dd/MM/yyyy HH:mm:ss"
      },
      DatePicker: {
        yearFormat: "yyyy",
        monthFormat: "MMM",
        dayFormat: "eeeeee",
        yearTypeFormat: "yyyy",
        monthTypeFormat: "yyyy/MM",
        dateFormat: "dd/MM/yyyy",
        dateTimeFormat: "dd/MM/yyyy HH:mm:ss",
        quarterFormat: "yyyy/qqq",
        weekFormat: "YYYY-w",
        clear: "Limpar",
        now: "Agora",
        confirm: "Confirmar",
        selectTime: "Selecione a hora",
        selectDate: "Selecione a data",
        datePlaceholder: "Selecione a data",
        datetimePlaceholder: "Selecione a data e hora",
        monthPlaceholder: "Selecione o m\xEAs",
        yearPlaceholder: "Selecione o ano",
        quarterPlaceholder: "Selecione o trimestre",
        weekPlaceholder: "Select Week",
        startDatePlaceholder: "Selecione a data de in\xEDcio",
        endDatePlaceholder: "Selecione a data de t\xE9rmino",
        startDatetimePlaceholder: "Selecione a data e hora de in\xEDcio",
        endDatetimePlaceholder: "Selecione a data e hora de t\xE9rmino",
        startMonthPlaceholder: "Selecione o m\xEAs de in\xEDcio",
        endMonthPlaceholder: "Selecione o m\xEAs de t\xE9rmino",
        monthBeforeYear: true,
        firstDayOfWeek: 6,
        today: "Hoje"
      },
      DataTable: {
        checkTableAll: "Selecionar todos na tabela",
        uncheckTableAll: "Desmarcar todos na tabela",
        confirm: "Confirmar",
        clear: "Limpar"
      },
      LegacyTransfer: {
        sourceTitle: "Origem",
        targetTitle: "Destino"
      },
      Transfer: {
        selectAll: "Selecionar todos",
        unselectAll: "Desmarcar todos",
        clearAll: "Limpar",
        total: num => `Total ${num} itens`,
        selected: num => `${num} itens selecionados`
      },
      Empty: {
        description: "N\xE3o h\xE1 dados"
      },
      Select: {
        placeholder: "Por favor selecione"
      },
      TimePicker: {
        placeholder: "Selecione a hora",
        positiveText: "OK",
        negativeText: "Cancelar",
        now: "Agora",
        clear: "Limpar"
      },
      Pagination: {
        goto: "Ir para",
        selectionSuffix: "p\xE1gina"
      },
      DynamicTags: {
        add: "Adicionar"
      },
      Log: {
        loading: "Carregando"
      },
      Input: {
        placeholder: "Por favor digite"
      },
      InputNumber: {
        placeholder: "Por favor digite"
      },
      DynamicInput: {
        create: "Criar"
      },
      ThemeEditor: {
        title: "Editor de temas",
        clearAllVars: "Limpar todas as vari\xE1veis",
        clearSearch: "Limpar pesquisa",
        filterCompName: "Filtrar nome do componente",
        filterVarName: "Filtrar nome da vari\xE1vel",
        import: "Importar",
        export: "Exportar",
        restore: "Restaurar"
      },
      Image: {
        tipPrevious: "Foto anterior (\u2190)",
        tipNext: "Pr\xF3xima foto (\u2192)",
        tipCounterclockwise: "Sentido anti-hor\xE1rio",
        tipClockwise: "Sentido hor\xE1rio",
        tipZoomOut: "Reduzir o zoom",
        tipZoomIn: "Aumentar o zoom",
        tipDownload: "Download",
        tipClose: "Fechar (Esc)",
        tipOriginalSize: "Exibir no tamanho original"
      }
    };

    const thTH = {
      name: "thTH",
      global: {
        undo: "\u0E40\u0E25\u0E34\u0E01\u0E17\u0E33",
        redo: "\u0E17\u0E33\u0E0B\u0E49\u0E33",
        confirm: "\u0E22\u0E37\u0E19\u0E22\u0E31\u0E19",
        clear: "\u0E25\u0E49\u0E32\u0E07"
      },
      Popconfirm: {
        positiveText: "\u0E22\u0E37\u0E19\u0E22\u0E31\u0E19",
        negativeText: "\u0E22\u0E01\u0E40\u0E25\u0E34\u0E01"
      },
      Cascader: {
        placeholder: "\u0E01\u0E23\u0E38\u0E13\u0E32\u0E40\u0E25\u0E37\u0E2D\u0E01",
        loading: "\u0E01\u0E33\u0E25\u0E31\u0E07\u0E42\u0E2B\u0E25\u0E14",
        loadingRequiredMessage: label => `Please load all ${label}'s descendants before checking it.`
      },
      Time: {
        dateFormat: "dd-MMMM-yyyy",
        dateTimeFormat: "dd-MMMM-yyyy HH:mm:ss"
      },
      DatePicker: {
        yearFormat: "yyyy",
        monthFormat: "MMM",
        dayFormat: "eeeeee",
        yearTypeFormat: "yyyy",
        monthTypeFormat: "yyyy-MM",
        dateFormat: "dd/MMMM/yyyy",
        dateTimeFormat: "dd/MMMM/yyyy HH:mm:ss",
        quarterFormat: "yyyy-qqq",
        weekFormat: "YYYY-w",
        clear: "\u0E25\u0E49\u0E32\u0E07",
        now: "\u0E27\u0E31\u0E19\u0E19\u0E35\u0E49",
        confirm: "\u0E22\u0E37\u0E19\u0E22\u0E31\u0E19",
        selectTime: "\u0E40\u0E27\u0E25\u0E32",
        selectDate: "\u0E27\u0E31\u0E19\u0E17\u0E35\u0E48",
        datePlaceholder: "\u0E27\u0E31\u0E19\u0E17\u0E35\u0E48",
        datetimePlaceholder: "\u0E40\u0E27\u0E25\u0E32-\u0E27\u0E31\u0E19\u0E17\u0E35\u0E48",
        monthPlaceholder: "\u0E40\u0E14\u0E37\u0E2D\u0E19",
        yearPlaceholder: "\u0E1B\u0E35",
        quarterPlaceholder: "\u0E44\u0E15\u0E23\u0E21\u0E32\u0E2A",
        weekPlaceholder: "Select Week",
        startDatePlaceholder: "\u0E27\u0E31\u0E19\u0E17\u0E35\u0E48\u0E40\u0E23\u0E34\u0E48\u0E21\u0E15\u0E49\u0E19",
        endDatePlaceholder: "\u0E27\u0E31\u0E19\u0E17\u0E35\u0E48\u0E2A\u0E34\u0E49\u0E19\u0E2A\u0E38\u0E14",
        startDatetimePlaceholder: "\u0E27\u0E31\u0E19\u0E17\u0E35\u0E48\u0E40\u0E23\u0E34\u0E48\u0E21\u0E15\u0E49\u0E19\u0E41\u0E25\u0E30\u0E2A\u0E34\u0E49\u0E19\u0E2A\u0E38\u0E14",
        endDatetimePlaceholder: "\u0E27\u0E31\u0E19\u0E17\u0E35\u0E48\u0E2A\u0E34\u0E49\u0E19\u0E2A\u0E38\u0E14\u0E41\u0E25\u0E30\u0E40\u0E27\u0E25\u0E32",
        // FIXME: translation needed
        startMonthPlaceholder: "Start Month",
        endMonthPlaceholder: "End Month",
        monthBeforeYear: true,
        firstDayOfWeek: 6,
        today: "\u0E27\u0E31\u0E19\u0E19\u0E35\u0E49"
      },
      DataTable: {
        checkTableAll: "\u0E40\u0E25\u0E37\u0E2D\u0E01\u0E17\u0E31\u0E49\u0E07\u0E2B\u0E21\u0E14",
        uncheckTableAll: "\u0E44\u0E21\u0E48\u0E40\u0E25\u0E37\u0E2D\u0E01\u0E17\u0E31\u0E49\u0E07\u0E2B\u0E21\u0E14",
        confirm: "\u0E22\u0E37\u0E19\u0E22\u0E31\u0E19",
        clear: "\u0E25\u0E49\u0E32\u0E07\u0E02\u0E49\u0E2D\u0E21\u0E39\u0E25"
      },
      LegacyTransfer: {
        sourceTitle: "Source",
        targetTitle: "Target"
      },
      // TODO: translation
      Transfer: {
        selectAll: "Select all",
        unselectAll: "Unselect all",
        clearAll: "Clear",
        total: num => `Total ${num} items`,
        selected: num => `${num} items selected`
      },
      Empty: {
        description: "\u0E44\u0E21\u0E48\u0E21\u0E35\u0E02\u0E49\u0E2D\u0E21\u0E39\u0E25"
      },
      Select: {
        placeholder: "\u0E01\u0E23\u0E38\u0E13\u0E32\u0E40\u0E25\u0E37\u0E2D\u0E01"
      },
      TimePicker: {
        placeholder: "\u0E40\u0E27\u0E25\u0E32",
        positiveText: "\u0E15\u0E01\u0E25\u0E07",
        negativeText: "\u0E22\u0E01\u0E40\u0E25\u0E34\u0E01",
        now: "\u0E27\u0E31\u0E19\u0E19\u0E35\u0E49",
        clear: "\u0E25\u0E49\u0E32\u0E07"
      },
      Pagination: {
        goto: "\u0E44\u0E1B\u0E22\u0E31\u0E07",
        selectionSuffix: "\u0E2B\u0E19\u0E49\u0E32"
      },
      DynamicTags: {
        add: "\u0E40\u0E1E\u0E34\u0E48\u0E21"
      },
      Log: {
        loading: "\u0E01\u0E33\u0E25\u0E31\u0E07\u0E42\u0E2B\u0E25\u0E14"
      },
      Input: {
        placeholder: "\u0E01\u0E23\u0E38\u0E13\u0E32\u0E01\u0E23\u0E2D\u0E01"
      },
      InputNumber: {
        placeholder: "\u0E01\u0E23\u0E38\u0E13\u0E32\u0E01\u0E23\u0E2D\u0E01"
      },
      DynamicInput: {
        create: "\u0E2A\u0E23\u0E49\u0E32\u0E07"
      },
      ThemeEditor: {
        title: "\u0E41\u0E01\u0E49\u0E44\u0E02\u0E18\u0E35\u0E21",
        clearAllVars: "\u0E25\u0E49\u0E32\u0E07\u0E02\u0E49\u0E2D\u0E21\u0E39\u0E25\u0E15\u0E31\u0E27\u0E41\u0E1B\u0E23",
        clearSearch: "\u0E25\u0E49\u0E32\u0E07\u0E02\u0E49\u0E2D\u0E21\u0E39\u0E25\u0E04\u0E49\u0E19\u0E2B\u0E32",
        filterCompName: "\u0E01\u0E23\u0E2D\u0E07\u0E42\u0E14\u0E22\u0E0A\u0E37\u0E48\u0E2D Component",
        filterVarName: "\u0E01\u0E23\u0E2D\u0E07\u0E42\u0E14\u0E22\u0E0A\u0E37\u0E48\u0E2D\u0E15\u0E31\u0E27\u0E41\u0E1B\u0E23",
        import: "\u0E19\u0E33\u0E40\u0E02\u0E49\u0E32",
        export: "\u0E2A\u0E48\u0E07\u0E2D\u0E2D\u0E01",
        restore: "\u0E23\u0E35\u0E40\u0E0B\u0E47\u0E15"
      },
      // TODO: translation
      Image: {
        tipPrevious: "\u0E01\u0E48\u0E2D\u0E19\u0E2B\u0E19\u0E49\u0E32 (\u2190)",
        tipNext: "\u0E16\u0E31\u0E14\u0E44\u0E1B (\u2192)",
        tipCounterclockwise: "\u0E2B\u0E21\u0E38\u0E19 (\u21BA)",
        tipClockwise: "\u0E2B\u0E21\u0E38\u0E19 (\u21BB)",
        tipZoomOut: "\u0E0B\u0E39\u0E21\u0E2D\u0E2D\u0E01",
        tipZoomIn: "\u0E0B\u0E39\u0E21\u0E40\u0E02\u0E49\u0E32",
        tipDownload: "\u0E14\u0E32\u0E27\u0E19\u0E4C\u0E42\u0E2B\u0E25\u0E14",
        tipClose: "\u0E1B\u0E34\u0E14 (Esc)",
        // TODO: translation
        tipOriginalSize: "Zoom to original size"
      }
    };

    const arDZ$1 = {
      name: "ar-DZ",
      global: {
        undo: "\u062A\u0631\u0627\u062C\u0639",
        redo: "\u0625\u0639\u0627\u062F\u0629",
        confirm: "\u062A\u0623\u0643\u064A\u062F",
        clear: "\u0645\u0633\u062D"
      },
      Popconfirm: {
        positiveText: "\u062A\u0623\u0643\u064A\u062F",
        negativeText: "\u0625\u0644\u063A\u0627\u0621"
      },
      Cascader: {
        placeholder: "\u064A\u0631\u062C\u0649 \u0627\u0644\u062A\u062D\u062F\u064A\u062F",
        loading: "\u062C\u0627\u0631\u064A \u0627\u0644\u062A\u062D\u0645\u064A\u0644",
        loadingRequiredMessage: label => `\u064A\u0631\u062C\u0649 \u062A\u062D\u0645\u064A\u0644 \u062C\u0645\u064A\u0639 \u0627\u0644\u0640 ${label} \u0627\u0644\u0641\u0631\u0639\u064A\u0629 \u0642\u0628\u0644 \u0627\u0644\u062A\u062D\u0642\u0642 \u0645\u0646\u0647\u0627.`
      },
      Time: {
        dateFormat: "yyyy-MM-dd",
        dateTimeFormat: "yyyy-MM-dd HH:mm:ss"
      },
      DatePicker: {
        yearFormat: "yyyy",
        monthFormat: "MMM",
        dayFormat: "eeeeee",
        yearTypeFormat: "yyyy",
        monthTypeFormat: "yyyy-MM",
        dateFormat: "yyyy-MM-dd",
        dateTimeFormat: "yyyy-MM-dd HH:mm:ss",
        quarterFormat: "yyyy-qqq",
        weekFormat: "YYYY-w",
        clear: "\u0645\u0633\u062D",
        now: "\u0627\u0644\u0622\u0646",
        confirm: "\u062A\u0623\u0643\u064A\u062F",
        selectTime: "\u0625\u062E\u062A\u064A\u0627\u0631 \u0627\u0644\u0648\u0642\u062A",
        selectDate: "\u0625\u062E\u062A\u064A\u0627\u0631 \u0627\u0644\u062A\u0627\u0631\u064A\u062E",
        datePlaceholder: "\u0625\u062E\u062A\u064A\u0627\u0631 \u0627\u0644\u062A\u0627\u0631\u064A\u062E",
        datetimePlaceholder: "\u0625\u062E\u062A\u064A\u0627\u0631 \u0627\u0644\u062A\u0627\u0631\u064A\u062E \u0648\u0627\u0644\u0648\u0642\u062A",
        monthPlaceholder: "\u0625\u062E\u062A\u064A\u0627\u0631 \u0627\u0644\u0634\u0647\u0631",
        yearPlaceholder: "\u0625\u062E\u062A\u064A\u0627\u0631 \u0627\u0644\u0633\u0646\u0629",
        quarterPlaceholder: "\u0625\u062E\u062A\u064A\u0627\u0631 \u0627\u0644\u0631\u0628\u0639",
        weekPlaceholder: "Select Week",
        startDatePlaceholder: "\u062A\u0627\u0631\u064A\u062E \u0627\u0644\u0628\u062F\u0621",
        endDatePlaceholder: "\u062A\u0627\u0631\u064A\u062E \u0627\u0644\u0625\u0646\u062A\u0647\u0627\u0621",
        startDatetimePlaceholder: "\u062A\u0627\u0631\u064A\u062E \u0648\u0648\u0642\u062A \u0627\u0644\u0628\u062F\u0621",
        endDatetimePlaceholder: "\u062A\u0627\u0631\u064A\u062E \u0648\u0648\u0642\u062A \u0627\u0644\u0625\u0646\u062A\u0647\u0627\u0621",
        startMonthPlaceholder: "\u0634\u0647\u0631 \u0627\u0644\u0628\u062F\u0621",
        endMonthPlaceholder: "\u0634\u0647\u0631 \u0627\u0644\u0625\u0646\u062A\u0647\u0627\u0621",
        monthBeforeYear: true,
        firstDayOfWeek: 6,
        today: "\u0627\u0644\u064A\u0648\u0645"
      },
      DataTable: {
        checkTableAll: "\u062A\u062D\u062F\u064A\u062F \u0643\u0644 \u0627\u0644\u0639\u0646\u0627\u0635\u0631 \u0641\u064A \u0627\u0644\u062C\u062F\u0648\u0644",
        uncheckTableAll: "\u0625\u0644\u063A\u0627\u0621 \u062A\u062D\u062F\u064A\u062F \u0643\u0644 \u0627\u0644\u0639\u0646\u0627\u0635\u0631 \u0641\u064A \u0627\u0644\u062C\u062F\u0648\u0644",
        confirm: "\u062A\u0623\u0643\u064A\u062F",
        clear: "\u0645\u0633\u062D"
      },
      LegacyTransfer: {
        sourceTitle: "\u0627\u0644\u0645\u0635\u062F\u0631",
        targetTitle: "\u0627\u0644\u0647\u062F\u0641"
      },
      Transfer: {
        selectAll: "\u062A\u062D\u062F\u064A\u062F \u0627\u0644\u0643\u0644",
        unselectAll: "\u0625\u0644\u063A\u0627\u0621 \u062A\u062D\u062F\u064A\u062F \u0627\u0644\u0643\u0644",
        clearAll: "\u0645\u0633\u062D",
        total: num => `\u0625\u062C\u0645\u0627\u0644\u064A ${num} \u0639\u0646\u0635\u0631`,
        selected: num => `${num} \u0639\u0646\u0635\u0631 \u0645\u062D\u062F\u062F`
      },
      Empty: {
        description: "\u0644\u0627 \u062A\u0648\u062C\u062F \u0628\u064A\u0627\u0646\u0627\u062A"
      },
      Select: {
        placeholder: "\u064A\u0631\u062C\u0649 \u0627\u0644\u0625\u062E\u062A\u064A\u0627\u0631"
      },
      TimePicker: {
        placeholder: "\u0625\u062E\u062A\u064A\u0627\u0631 \u0627\u0644\u0648\u0642\u062A",
        positiveText: "\u062A\u0623\u0643\u064A\u062F",
        negativeText: "\u0625\u0644\u063A\u0627\u0621",
        now: "\u0627\u0644\u0622\u0646",
        clear: "\u0645\u0633\u062D"
      },
      Pagination: {
        goto: "\u0625\u0630\u0647\u0628 \u0625\u0644\u0649",
        selectionSuffix: "\u0635\u0641\u062D\u0629"
      },
      DynamicTags: {
        add: "\u0625\u0636\u0627\u0641\u0629"
      },
      Log: {
        loading: "\u062C\u0627\u0631\u064A \u0627\u0644\u062A\u062D\u0645\u064A\u0644"
      },
      Input: {
        placeholder: "\u064A\u0631\u062C\u0649 \u0627\u0644\u0625\u062F\u062E\u0627\u0644"
      },
      InputNumber: {
        placeholder: "\u064A\u0631\u062C\u0649 \u0627\u0644\u0625\u062F\u062E\u0627\u0644"
      },
      DynamicInput: {
        create: "\u0625\u0646\u0634\u0627\u0621"
      },
      ThemeEditor: {
        title: "\u0645\u062D\u0631\u0631 \u0627\u0644\u0646\u0645\u0637",
        clearAllVars: "\u0645\u0633\u062D \u062C\u0645\u064A\u0639 \u0627\u0644\u0645\u062A\u063A\u064A\u0631\u0627\u062A",
        clearSearch: "\u0645\u0633\u062D \u0627\u0644\u0628\u062D\u062B",
        filterCompName: "\u062A\u0635\u0641\u064A\u0629 \u0625\u0633\u0645 \u0627\u0644\u0645\u0643\u0648\u0646",
        filterVarName: "\u062A\u0635\u0641\u064A\u0629 \u0625\u0633\u0645 \u0627\u0644\u0645\u062A\u063A\u064A\u0631",
        import: "\u0625\u0633\u062A\u064A\u0631\u0627\u062F",
        export: "\u062A\u0635\u062F\u064A\u0631",
        restore: "\u0625\u0639\u0627\u062F\u0629 \u062A\u0639\u064A\u064A\u0646 \u0625\u0644\u0649 \u0627\u0644\u0625\u0641\u062A\u0631\u0627\u0636\u064A"
      },
      Image: {
        tipPrevious: "(\u2192) \u0627\u0644\u0635\u0648\u0631\u0629 \u0627\u0644\u0633\u0627\u0628\u0642\u0629",
        tipNext: "(\u2190) \u0627\u0644\u0635\u0648\u0631\u0629 \u0627\u0644\u062A\u0627\u0644\u064A\u0629",
        tipCounterclockwise: "\u0639\u0643\u0633 \u0639\u0642\u0627\u0631\u0628 \u0627\u0644\u0633\u0627\u0639\u0629",
        tipClockwise: "\u0625\u062A\u062C\u0627\u0647 \u0639\u0642\u0627\u0631\u0628 \u0627\u0644\u0633\u0627\u0639\u0629",
        tipZoomOut: "\u062A\u0643\u0628\u064A\u0631",
        tipZoomIn: "\u062A\u0635\u063A\u064A\u0631",
        tipDownload: "\u0644\u0644\u062A\u062D\u0645\u064A\u0644",
        tipClose: "\u0625\u063A\u0644\u0627\u0642 (Esc \u0632\u0631)",
        tipOriginalSize: "\u062A\u0643\u0628\u064A\u0631 \u0625\u0644\u0649 \u0627\u0644\u062D\u062C\u0645 \u0627\u0644\u0623\u0635\u0644\u064A"
      }
    };

    const trTR = {
      name: "tr-TR",
      global: {
        undo: "Vazge\xE7",
        redo: "Tekrar Dene",
        confirm: "Kabul Et",
        clear: "Temizle"
      },
      Popconfirm: {
        positiveText: "Kabul Et",
        negativeText: "\u0130ptal"
      },
      Cascader: {
        placeholder: "L\xFCtfen Se\xE7in",
        loading: "Y\xFCkleniyor",
        loadingRequiredMessage: label => `L\xFCtfen kontrol etmeden \xF6nce t\xFCm ${label}lar\u0131 y\xFCkleyin.`
      },
      Time: {
        dateFormat: "yyyy-MM-dd",
        dateTimeFormat: "yyyy-MM-dd HH:mm:ss"
      },
      DatePicker: {
        yearFormat: "yyyy",
        monthFormat: "MMM",
        dayFormat: "eeeeee",
        yearTypeFormat: "yyyy",
        monthTypeFormat: "yyyy-MM",
        dateFormat: "yyyy-MM-dd",
        dateTimeFormat: "yyyy-MM-dd HH:mm:ss",
        quarterFormat: "yyyy-qqq",
        weekFormat: "YYYY-w",
        clear: "Temizle",
        now: "\u015Eimdi",
        confirm: "Onayla",
        selectTime: "Saat Se\xE7in",
        selectDate: "Tarih Se\xE7in",
        datePlaceholder: "Tarih Se\xE7",
        datetimePlaceholder: "Tarih ve Saat Se\xE7in",
        monthPlaceholder: "Ay Se\xE7in",
        yearPlaceholder: "Y\u0131l Se\xE7in",
        quarterPlaceholder: "Mevsim Se\xE7in",
        weekPlaceholder: "Select Week",
        startDatePlaceholder: "Ba\u015Flang\u0131\xE7 Tarihi",
        endDatePlaceholder: "Biti\u015F Tarihi",
        startDatetimePlaceholder: "Ba\u015Flang\u0131\xE7 Tarih ve Saati",
        endDatetimePlaceholder: "Biti\u015F Tarihi ve Saati",
        startMonthPlaceholder: "Ba\u015Flang\u0131\xE7 Ay\u0131",
        endMonthPlaceholder: "Biti\u015F Ay\u0131",
        monthBeforeYear: true,
        firstDayOfWeek: 6,
        today: "Bug\xFCn"
      },
      DataTable: {
        checkTableAll: "T\xFCm Tabloyu Se\xE7",
        uncheckTableAll: "Tablodaki t\xFCm se\xE7imleri kald\u0131r",
        confirm: "Onayla",
        clear: "Temizle"
      },
      LegacyTransfer: {
        sourceTitle: "Kaynak",
        targetTitle: "Hedef"
      },
      Transfer: {
        selectAll: "T\xFCm\xFCn\xFC Se\xE7",
        unselectAll: "T\xFCm Se\xE7imi Kald\u0131r",
        clearAll: "Temizle",
        total: num => `Toplam ${num} \xF6\u011Fe`,
        selected: num => `${num} \xF6\u011Fe Se\xE7ili`
      },
      Empty: {
        description: "Veri Yok"
      },
      Select: {
        placeholder: "L\xFCtfen se\xE7in"
      },
      TimePicker: {
        placeholder: "Saat Se\xE7",
        positiveText: "Tamam",
        negativeText: "\u0130ptal",
        now: "\u015Eimdi",
        clear: "Temizle"
      },
      Pagination: {
        goto: "git",
        selectionSuffix: "sayfa"
      },
      DynamicTags: {
        add: "Ekle"
      },
      Log: {
        loading: "Y\xFCkleniyor"
      },
      Input: {
        placeholder: "L\xFCtfen Girin"
      },
      InputNumber: {
        placeholder: "L\xFCtfen Girin"
      },
      DynamicInput: {
        create: "Olu\u015Ftur"
      },
      ThemeEditor: {
        title: "Tema Edit\xF6r\xFC",
        clearAllVars: "T\xFCm De\u011Fi\u015Fkenleri Temizle",
        clearSearch: "Aramay\u0131 Temizle",
        filterCompName: "Bile\u015Fen Ad\u0131n\u0131 Filtrele",
        filterVarName: "De\u011Fi\u015Fken Ad\u0131n\u0131 Filtrele",
        import: "Ekle",
        export: "\xC7\u0131kar",
        restore: "Varsay\u0131lana s\u0131f\u0131rla"
      },
      Image: {
        tipPrevious: "\xD6nceki resim (\u2190)",
        tipNext: "Sonraki resim (\u2192)",
        tipCounterclockwise: "saat y\xF6n\xFCn\xFCn tersine",
        tipClockwise: "saat y\xF6n\xFCn\xFCne",
        tipZoomOut: "Uzakla\u015Ft\u0131r",
        tipZoomIn: "Yak\u0131nla\u015Ft\u0131r",
        tipDownload: "indir",
        tipClose: "Kapat (Esc)",
        // TODO: translation
        tipOriginalSize: "Orijinal boyuta yak\u0131nla\u015Ft\u0131r"
      }
    };

    const eo$1 = {
      name: "eo",
      global: {
        undo: "Malfari",
        redo: "Refari",
        confirm: "Konfirmi",
        clear: "Malplenigi"
      },
      Popconfirm: {
        positiveText: "Konfirmi",
        negativeText: "Nuligi"
      },
      Cascader: {
        placeholder: "Bonvolu elekti",
        loading: "\u015Cargi\u011Do",
        loadingRequiredMessage: label => `Bonvolu \u015Dargi \u0109iujn idojn de ${label} anta\u016D ol elekti \u011Din.`
      },
      Time: {
        dateFormat: "yyyy-MM-dd",
        dateTimeFormat: "yyyy-MM-dd HH:mm:ss"
      },
      DatePicker: {
        yearFormat: "yyyy",
        monthFormat: "MMM",
        dayFormat: "eeeeee",
        yearTypeFormat: "yyyy",
        monthTypeFormat: "yyyy-MM",
        dateFormat: "yyyy-MM-dd",
        dateTimeFormat: "yyyy-MM-dd HH:mm:ss",
        quarterFormat: "yyyy-qqq",
        weekFormat: "YYYY-w",
        clear: "Malplenigi",
        now: "Nun",
        confirm: "Konfirmi",
        selectTime: "Elekti tempon",
        selectDate: "Elekti daton",
        datePlaceholder: "Elekti daton",
        datetimePlaceholder: "Elekti daton kaj tempon",
        monthPlaceholder: "Elekti monaton",
        yearPlaceholder: "Elekti jaron",
        quarterPlaceholder: "Elekti jarkvaronon",
        weekPlaceholder: "Select Week",
        startDatePlaceholder: "Komenca dato",
        endDatePlaceholder: "Fina dato",
        startDatetimePlaceholder: "Komencaj dato kaj tempo",
        endDatetimePlaceholder: "Finaj dato kaj tempo",
        // FIXME: translation needed
        startMonthPlaceholder: "Start Month",
        endMonthPlaceholder: "End Month",
        monthBeforeYear: true,
        firstDayOfWeek: 0,
        today: "Hodia\u016D"
      },
      DataTable: {
        checkTableAll: "Elekti \u0109iujn en la tabelo",
        uncheckTableAll: "Malelekti \u0109iujn en la tabelo",
        confirm: "Konfirmi",
        clear: "Malplenigi"
      },
      LegacyTransfer: {
        sourceTitle: "Source",
        targetTitle: "Target"
      },
      // TODO: translation
      Transfer: {
        selectAll: "Select all",
        unselectAll: "Unselect all",
        clearAll: "Clear",
        total: num => `Total ${num} items`,
        selected: num => `${num} items selected`
      },
      Empty: {
        description: "Neniu datumo"
      },
      Select: {
        placeholder: "Bonvolu elekti"
      },
      TimePicker: {
        placeholder: "Elekti tempon",
        positiveText: "Bone",
        negativeText: "Nuligi",
        now: "Nun",
        clear: "Malplenigi"
      },
      Pagination: {
        goto: "Iri al",
        selectionSuffix: "pa\u011Do"
      },
      DynamicTags: {
        add: "Aldoni"
      },
      Log: {
        loading: "\u015Cargado"
      },
      Input: {
        placeholder: "Bonvolu entajpi"
      },
      InputNumber: {
        placeholder: "Bonvolu entajpi"
      },
      DynamicInput: {
        create: "Krei"
      },
      ThemeEditor: {
        title: "Etosredaktilo",
        clearAllVars: "Malplenigi \u0109iujn variablojn",
        clearSearch: "Malplenigi ser\u0109on",
        filterCompName: "Filtri nomojn de kompona\u0135oj",
        filterVarName: "Filtri nomojn de variabloj",
        import: "Importi",
        export: "Eksporti",
        restore: "Restarigi defa\u016Dltajn valorojn"
      },
      Image: {
        tipPrevious: "Anta\u016Da bildo (\u2190)",
        tipNext: "Sekva bildo (\u2192)",
        tipCounterclockwise: "Maldekstrume",
        tipClockwise: "Dekstrume",
        tipZoomOut: "Malzomi",
        tipZoomIn: "Zomi",
        tipDownload: "El\u015Duti",
        tipClose: "Fermi (Esc)",
        tipOriginalSize: "Zoom to original size"
      }
    };

    const viVN = {
      name: "vi-VN",
      global: {
        undo: "Ho\xE0n t\xE1c",
        redo: "L\xE0m l\u1EA1i",
        confirm: "X\xE1c nh\u1EADn",
        clear: "x\xF3a"
      },
      Popconfirm: {
        positiveText: "X\xE1c nh\u1EADn",
        negativeText: "H\u1EE7y"
      },
      Cascader: {
        placeholder: "Vui l\xF2ng ch\u1ECDn",
        loading: "\u0110ang t\u1EA3i",
        loadingRequiredMessage: label => `Vui l\xF2ng t\u1EA3i t\u1EA5t c\u1EA3 th\xF4ng tin con c\u1EE7a ${label} tr\u01B0\u1EDBc.`
      },
      Time: {
        dateFormat: "",
        dateTimeFormat: "HH:mm:ss dd-MM-yyyy"
      },
      DatePicker: {
        yearFormat: "yyyy",
        monthFormat: "MMM",
        dayFormat: "eeeeee",
        yearTypeFormat: "yyyy",
        monthTypeFormat: "MM-yyyy",
        dateFormat: "dd-MM-yyyy",
        dateTimeFormat: "HH:mm:ss dd-MM-yyyy",
        quarterFormat: "qqq-yyyy",
        weekFormat: "YYYY-w",
        clear: "X\xF3a",
        now: "H\xF4m nay",
        confirm: "X\xE1c nh\u1EADn",
        selectTime: "Ch\u1ECDn gi\u1EDD",
        selectDate: "Ch\u1ECDn ng\xE0y",
        datePlaceholder: "Ch\u1ECDn ng\xE0y",
        datetimePlaceholder: "Ch\u1ECDn ng\xE0y gi\u1EDD",
        monthPlaceholder: "Ch\u1ECDn th\xE1ng",
        yearPlaceholder: "Ch\u1ECDn n\u0103m",
        quarterPlaceholder: "Ch\u1ECDn qu\xFD",
        weekPlaceholder: "Select Week",
        startDatePlaceholder: "Ng\xE0y b\u1EAFt \u0111\u1EA7u",
        endDatePlaceholder: "Ng\xE0y k\u1EBFt th\xFAc",
        startDatetimePlaceholder: "Th\u1EDDi gian b\u1EAFt \u0111\u1EA7u",
        endDatetimePlaceholder: "Th\u1EDDi gian k\u1EBFt th\xFAc",
        startMonthPlaceholder: "Th\xE1ng b\u1EAFt \u0111\u1EA7u",
        endMonthPlaceholder: "Th\xE1ng k\u1EBFt th\xFAc",
        monthBeforeYear: true,
        firstDayOfWeek: 0,
        today: "H\xF4m nay"
      },
      DataTable: {
        checkTableAll: "Ch\u1ECDn t\u1EA5t c\u1EA3 c\xF3 trong b\u1EA3ng",
        uncheckTableAll: "B\u1ECF ch\u1ECDn t\u1EA5t c\u1EA3 c\xF3 trong b\u1EA3ng",
        confirm: "X\xE1c nh\u1EADn",
        clear: "X\xF3a"
      },
      LegacyTransfer: {
        sourceTitle: "Ngu\u1ED3n",
        targetTitle: "\u0110\xEDch"
      },
      Transfer: {
        selectAll: "Ch\u1ECDn t\u1EA5t c\u1EA3",
        unselectAll: "B\u1ECF ch\u1ECDn t\u1EA5t c\u1EA3",
        clearAll: "Xo\xE1 t\u1EA5t c\u1EA3",
        total: num => `T\u1ED5ng c\u1ED9ng ${num} m\u1EE5c`,
        selected: num => `${num} m\u1EE5c \u0111\u01B0\u1EE3c ch\u1ECDn`
      },
      Empty: {
        description: "Kh\xF4ng c\xF3 d\u1EEF li\u1EC7u"
      },
      Select: {
        placeholder: "Vui l\xF2ng ch\u1ECDn"
      },
      TimePicker: {
        placeholder: "Ch\u1ECDn th\u1EDDi gian",
        positiveText: "OK",
        negativeText: "H\u1EE7y",
        now: "Hi\u1EC7n t\u1EA1i",
        clear: "X\xF3a"
      },
      Pagination: {
        goto: "\u0110i \u0111\u1EBFn trang",
        selectionSuffix: "trang"
      },
      DynamicTags: {
        add: "Th\xEAm"
      },
      Log: {
        loading: "\u0110ang t\u1EA3i"
      },
      Input: {
        placeholder: "Vui l\xF2ng nh\u1EADp"
      },
      InputNumber: {
        placeholder: "Vui l\xF2ng nh\u1EADp"
      },
      DynamicInput: {
        create: "T\u1EA1o"
      },
      ThemeEditor: {
        title: "T\xF9y ch\u1EC9nh giao di\u1EC7n",
        clearAllVars: "X\xF3a t\u1EA5t c\u1EA3 c\xE1c bi\u1EBFn",
        clearSearch: "X\xF3a t\xECm ki\u1EBFm",
        filterCompName: "L\u1ECDc t\xEAn component",
        filterVarName: "L\u1ECDc t\xEAn bi\u1EBFn",
        import: "Nh\u1EADp",
        export: "Xu\u1EA5t",
        restore: "\u0110\u1EB7t l\u1EA1i m\u1EB7c \u0111\u1ECBnh"
      },
      Image: {
        tipPrevious: "H\xECnh tr\u01B0\u1EDBc (\u2190)",
        tipNext: "H\xECnh ti\u1EBFp (\u2192)",
        tipCounterclockwise: "Counterclockwise",
        tipClockwise: "Chi\u1EC1u kim \u0111\u1ED3ng h\u1ED3",
        tipZoomOut: "Thu nh\u1ECF",
        tipZoomIn: "Ph\xF3ng to",
        tipDownload: "T\u1EA3i v\u1EC1",
        tipClose: "\u0110\xF3ng (Esc)",
        tipOriginalSize: "Xem k\xEDch th\u01B0\u1EDBc g\u1ED1c"
      }
    };

    const faIR$1 = {
      name: "fa-IR",
      global: {
        undo: "\u0644\u063A\u0648 \u0627\u0646\u062C\u0627\u0645 \u0634\u062F\u0647",
        redo: "\u0627\u0646\u062C\u0627\u0645 \u062F\u0648\u0628\u0627\u0631\u0647",
        confirm: "\u062A\u0623\u06CC\u06CC\u062F",
        clear: "\u067E\u0627\u06A9 \u06A9\u0631\u062F\u0646"
      },
      Popconfirm: {
        positiveText: "\u062A\u0623\u06CC\u06CC\u062F",
        negativeText: "\u0644\u063A\u0648"
      },
      Cascader: {
        placeholder: "\u0644\u0637\u0641\u0627 \u0627\u0646\u062A\u062E\u0627\u0628 \u06A9\u0646\u06CC\u062F",
        loading: "\u0628\u0627\u0631\u06AF\u0630\u0627\u0631\u06CC",
        loadingRequiredMessage: label => `\u067E\u0633 \u0627\u0632 \u0628\u0627\u0631\u06AF\u06CC\u0631\u06CC \u06A9\u0627\u0645\u0644 \u0632\u06CC\u0631\u0645\u062C\u0645\u0648\u0639\u0647 \u0647\u0627\u06CC ${label} \u0645\u06CC \u062A\u0648\u0627\u0646\u06CC\u062F \u0627\u0646\u062A\u062E\u0627\u0628 \u06A9\u0646\u06CC\u062F `
      },
      Time: {
        dateFormat: "yyyy/MM/dd",
        dateTimeFormat: "yyyy/MM/dd\u060C H:mm:ss"
      },
      DatePicker: {
        yearFormat: "yyyy \u0633\u0627\u0644",
        monthFormat: "MMM",
        dayFormat: "eeeeee",
        yearTypeFormat: "yyyy",
        monthTypeFormat: "MM/yyyy",
        dateFormat: "yyyy/MM/dd",
        dateTimeFormat: "yyyy/MM/dd HH:mm:ss",
        quarterFormat: "\u0633\u0647 \u0645\u0627\u0647\u0647 yyyy",
        weekFormat: "YYYY-w",
        clear: "\u067E\u0627\u06A9 \u06A9\u0631\u062F\u0646",
        now: "\u0627\u06A9\u0646\u0648\u0646",
        confirm: "\u062A\u0623\u06CC\u06CC\u062F",
        selectTime: "\u0627\u0646\u062A\u062E\u0627\u0628 \u0632\u0645\u0627\u0646",
        selectDate: "\u0627\u0646\u062A\u062E\u0627\u0628 \u062A\u0627\u0631\u06CC\u062E",
        datePlaceholder: "\u0627\u0646\u062A\u062E\u0627\u0628 \u062A\u0627\u0631\u06CC\u062E",
        datetimePlaceholder: "\u0627\u0646\u062A\u062E\u0627\u0628 \u062A\u0627\u0631\u06CC\u062E \u0648 \u0632\u0645\u0627\u0646",
        monthPlaceholder: "\u0627\u0646\u062A\u062E\u0627\u0628 \u0645\u0627\u0647",
        yearPlaceholder: "\u0627\u0646\u062A\u062E\u0627\u0628 \u0633\u0627\u0644",
        quarterPlaceholder: "\u0627\u0646\u062A\u062E\u0627\u0628 \u0633\u0647\u200C\u0645\u0627\u0647\u0647",
        weekPlaceholder: "Select Week",
        startDatePlaceholder: "\u062A\u0627\u0631\u06CC\u062E \u0634\u0631\u0648\u0639",
        endDatePlaceholder: "\u062A\u0627\u0631\u06CC\u062E \u067E\u0627\u06CC\u0627\u0646",
        startDatetimePlaceholder: "\u0632\u0645\u0627\u0646 \u0634\u0631\u0648\u0639",
        endDatetimePlaceholder: "\u0632\u0645\u0627\u0646 \u067E\u0627\u06CC\u0627\u0646",
        startMonthPlaceholder: "\u0645\u0627\u0647 \u0634\u0631\u0648\u0639",
        endMonthPlaceholder: "\u0645\u0627\u0647 \u067E\u0627\u06CC\u0627\u0646",
        monthBeforeYear: false,
        firstDayOfWeek: 6,
        today: "\u0627\u0645\u0631\u0648\u0632"
      },
      DataTable: {
        checkTableAll: "\u0627\u0646\u062A\u062E\u0627\u0628 \u0647\u0645\u0647 \u062F\u0627\u062F\u0647\u200C\u0647\u0627\u06CC \u062C\u062F\u0648\u0644",
        uncheckTableAll: "\u0639\u062F\u0645 \u0627\u0646\u062A\u062E\u0627\u0628 \u0647\u0645\u0647 \u062F\u0627\u062F\u0647\u200C\u0647\u0627\u06CC \u062C\u062F\u0648\u0644",
        confirm: "\u062A\u0623\u06CC\u06CC\u062F",
        clear: "\u062A\u0646\u0638\u06CC\u0645 \u0645\u062C\u062F\u062F"
      },
      LegacyTransfer: {
        sourceTitle: "\u0622\u06CC\u062A\u0645 \u0645\u0646\u0628\u0639",
        targetTitle: "\u0622\u06CC\u062A\u0645 \u0645\u0642\u0635\u062F"
      },
      Transfer: {
        selectAll: "\u0627\u0646\u062A\u062E\u0627\u0628 \u0647\u0645\u0647",
        clearAll: "\u062D\u0630\u0641 \u0647\u0645\u0647",
        unselectAll: "\u0639\u062F\u0645 \u0627\u0646\u062A\u062E\u0627\u0628 \u0647\u0645\u0647",
        total: num => `\u06A9\u0644 ${num} \u0645\u0648\u0631\u062F`,
        selected: num => `\u0627\u0646\u062A\u062E\u0627\u0628 \u0634\u062F\u0647 ${num} \u0645\u0648\u0631\u062F`
      },
      Empty: {
        description: "\u0627\u0637\u0644\u0627\u0639\u0627\u062A\u06CC \u0648\u062C\u0648\u062F \u0646\u062F\u0627\u0631\u062F"
      },
      Select: {
        placeholder: "\u0644\u0637\u0641\u0627\u064B \u0627\u0646\u062A\u062E\u0627\u0628 \u06A9\u0646\u06CC\u062F"
      },
      TimePicker: {
        placeholder: "\u0644\u0637\u0641\u0627\u064B \u0632\u0645\u0627\u0646 \u0645\u0648\u0631\u062F \u0646\u0638\u0631 \u0631\u0627 \u0627\u0646\u062A\u062E\u0627\u0628 \u06A9\u0646\u06CC\u062F",
        positiveText: "\u062A\u0623\u06CC\u06CC\u062F",
        negativeText: "\u0644\u063A\u0648",
        now: "\u0647\u0645\u06CC\u0646 \u0627\u0644\u0627\u0646",
        clear: "\u067E\u0627\u06A9 \u06A9\u0631\u062F\u0646"
      },
      Pagination: {
        goto: "\u0631\u0641\u062A\u0646 \u0628\u0647 \u0635\u0641\u062D\u0647",
        selectionSuffix: "\u0635\u0641\u062D\u0647"
      },
      DynamicTags: {
        add: "\u0627\u0641\u0632\u0648\u062F\u0646"
      },
      Log: {
        loading: "\u062F\u0631 \u062D\u0627\u0644 \u0628\u0627\u0631\u06AF\u0630\u0627\u0631\u06CC"
      },
      Input: {
        placeholder: "\u0644\u0637\u0641\u0627\u064B \u0648\u0627\u0631\u062F \u06A9\u0646\u06CC\u062F"
      },
      InputNumber: {
        placeholder: "\u0644\u0637\u0641\u0627\u064B \u0648\u0627\u0631\u062F \u06A9\u0646\u06CC\u062F"
      },
      DynamicInput: {
        create: "\u0627\u0641\u0632\u0648\u062F\u0646"
      },
      ThemeEditor: {
        title: "\u0648\u06CC\u0631\u0627\u06CC\u0634\u06AF\u0631 \u067E\u0648\u0633\u062A\u0647",
        clearAllVars: "\u067E\u0627\u06A9 \u06A9\u0631\u062F\u0646 \u0647\u0645\u0647 \u0645\u062A\u063A\u06CC\u0631\u0647\u0627",
        clearSearch: "\u067E\u0627\u06A9 \u06A9\u0631\u062F\u0646 \u062C\u0633\u062A\u062C\u0648",
        filterCompName: "\u0641\u06CC\u0644\u062A\u0631 \u0646\u0627\u0645 \u06A9\u0627\u0645\u067E\u0648\u0646\u0646\u062A",
        filterVarName: "\u0641\u06CC\u0644\u062A\u0631 \u0646\u0627\u0645 \u0645\u062A\u063A\u06CC\u0631",
        import: "\u0648\u0631\u0648\u062F",
        export: "\u062E\u0631\u0648\u062C",
        restore: "\u0628\u0627\u0632\u06AF\u0631\u062F\u0627\u0646\u06CC \u0628\u0647 \u062D\u0627\u0644\u062A \u067E\u06CC\u0634\u200C\u0641\u0631\u0636"
      },
      Image: {
        tipPrevious: "\u062A\u0635\u0648\u06CC\u0631 \u0642\u0628\u0644\u06CC (\u2190)",
        tipNext: "\u062A\u0635\u0648\u06CC\u0631 \u0628\u0639\u062F\u06CC (\u2192)",
        tipCounterclockwise: "\u0686\u0631\u062E\u0634 \u0628\u0647 \u0633\u0645\u062A \u0686\u067E",
        tipClockwise: "\u0686\u0631\u062E\u0634 \u0628\u0647 \u0633\u0645\u062A \u0631\u0627\u0633\u062A",
        tipZoomOut: "\u06A9\u0648\u0686\u06A9 \u0646\u0645\u0627\u06CC\u06CC \u062A\u0635\u0648\u06CC\u0631",
        tipZoomIn: "\u0628\u0632\u0631\u06AF \u0646\u0645\u0627\u06CC\u06CC \u062A\u0635\u0648\u06CC\u0631",
        tipDownload: "\u0628\u0627\u0631\u06AF\u06CC\u0631\u06CC",
        tipClose: "\u0628\u0633\u062A\u0646 (Esc)",
        tipOriginalSize: "\u0627\u0646\u062F\u0627\u0632\u0647 \u0627\u0635\u0644\u06CC \u062A\u0635\u0648\u06CC\u0631"
      }
    };

    const svSE = {
      name: "sv-SE",
      global: {
        undo: "\xC5ngra",
        redo: "G\xF6r om",
        confirm: "Bekr\xE4fta",
        clear: "Rensa"
      },
      Popconfirm: {
        positiveText: "Bekr\xE4fta",
        negativeText: "Avbryt"
      },
      Cascader: {
        placeholder: "V\xE4nligen v\xE4lj",
        loading: "Laddar",
        loadingRequiredMessage: label => `V\xE4nligen ladda alla underpunkter till ${label} innan du v\xE4ljer punkten.`
      },
      Time: {
        dateFormat: "yyyy-MM-dd",
        dateTimeFormat: "yyyy-MM-dd HH:mm:ss"
      },
      DatePicker: {
        yearFormat: "yyyy",
        monthFormat: "MMM",
        dayFormat: "eeeeee",
        yearTypeFormat: "yyyy",
        monthTypeFormat: "yyyy-MM",
        dateFormat: "yyyy-MM-dd",
        dateTimeFormat: "yyyy-MM-dd HH:mm:ss",
        quarterFormat: "yyyy-qqq",
        weekFormat: "YYYY-w",
        clear: "Rensa",
        now: "Nu",
        confirm: "Bekr\xE4fta",
        selectTime: "V\xE4lj tid",
        selectDate: "V\xE4lj datum",
        datePlaceholder: "V\xE4lj datum",
        datetimePlaceholder: "V\xE4lj datum och tid",
        monthPlaceholder: "V\xE4lj m\xE5nad",
        yearPlaceholder: "V\xE4lj \xE5r",
        quarterPlaceholder: "V\xE4lj kvartal",
        weekPlaceholder: "Select Week",
        startDatePlaceholder: "Startdatum",
        endDatePlaceholder: "Slutdatum",
        startDatetimePlaceholder: "Startdatum och -tid",
        endDatetimePlaceholder: "Slutdatum och -tid",
        startMonthPlaceholder: "Startm\xE5nad",
        endMonthPlaceholder: "Slutm\xE5nad",
        monthBeforeYear: true,
        firstDayOfWeek: 0,
        today: "I dag"
      },
      DataTable: {
        checkTableAll: "V\xE4lj allt",
        uncheckTableAll: "V\xE4lj inget",
        confirm: "Bekr\xE4fta",
        clear: "Rensa"
      },
      LegacyTransfer: {
        sourceTitle: "K\xE4lla",
        targetTitle: "M\xE5l"
      },
      Transfer: {
        selectAll: "V\xE4lj allt",
        unselectAll: "V\xE4lj inget",
        clearAll: "Rensa",
        total: num => `Totalt ${num} rader`,
        selected: num => `${num} rader valda`
      },
      Empty: {
        description: "Ingen data"
      },
      Select: {
        placeholder: "V\xE4nligen v\xE4lj"
      },
      TimePicker: {
        placeholder: "V\xE4lj tid",
        positiveText: "OK",
        negativeText: "Avbryt",
        now: "Nu",
        clear: "Rensa"
      },
      Pagination: {
        goto: "G\xE5 till",
        selectionSuffix: "sida"
      },
      DynamicTags: {
        add: "L\xE4gg till"
      },
      Log: {
        loading: "Laddar"
      },
      Input: {
        placeholder: "V\xE4nligen fyll i"
      },
      InputNumber: {
        placeholder: "V\xE4nligen fyll i"
      },
      DynamicInput: {
        create: "Skapa"
      },
      ThemeEditor: {
        title: "Temaverktyg",
        clearAllVars: "Nollst\xE4ll alla variabler",
        clearSearch: "Rensa s\xF6kning",
        filterCompName: "Filtrera efter komponentnamn",
        filterVarName: "Filtrera efter variabelnamn",
        import: "Importera",
        export: "Exportera",
        restore: "Nollst\xE4ll till ursprungsval"
      },
      Image: {
        tipPrevious: "F\xF6rg\xE5ende bild (\u2190)",
        tipNext: "N\xE4sta bild (\u2192)",
        tipCounterclockwise: "Moturs",
        tipClockwise: "Medurs",
        tipZoomOut: "Zooma in",
        tipZoomIn: "Zooma ut",
        tipDownload: "Ladda ned",
        tipClose: "St\xE4ng (Esc)",
        tipOriginalSize: "Zooma till ursprunglig storlek"
      }
    };

    const etEE = {
      name: "et-EE",
      global: {
        undo: "V\xF5ta tagasi",
        redo: "Rakenda uuesti",
        confirm: "Kinnita",
        clear: "T\xFChjenda"
      },
      Popconfirm: {
        positiveText: "Kinnita",
        negativeText: "Katkesta"
      },
      Cascader: {
        placeholder: "Vali",
        loading: "Laeb",
        loadingRequiredMessage: label => `Enne kontrollimist pead elemendi ${label} k\xF5ik alamad laadima.`
      },
      Time: {
        dateFormat: "dd.MM.yyyy",
        dateTimeFormat: "dd.MM.yyyy HH:mm:ss"
      },
      DatePicker: {
        yearFormat: "yyyy",
        monthFormat: "MMM",
        dayFormat: "eeeeee",
        yearTypeFormat: "yyyy",
        monthTypeFormat: "MM-yyyy",
        dateFormat: "dd.MM.yyyy",
        dateTimeFormat: "dd.MM.yyyy HH:mm:ss",
        quarterFormat: "yyyy-qqq",
        weekFormat: "YYYY-w",
        clear: "T\xFChjenda",
        now: "N\xFC\xFCd",
        confirm: "Kinnita",
        selectTime: "Vali aeg",
        selectDate: "Vali kuup\xE4ev",
        datePlaceholder: "Vali kuup\xE4ev",
        datetimePlaceholder: "Vali aeg ja kuup\xE4ev",
        monthPlaceholder: "Vali kuu",
        yearPlaceholder: "Vali aasta",
        quarterPlaceholder: "Vali kvartal",
        weekPlaceholder: "Vali n\xE4dal",
        startDatePlaceholder: "Alguskpv",
        endDatePlaceholder: "L\xF5ppkpv",
        startDatetimePlaceholder: "Alguskpv ja -aeg",
        endDatetimePlaceholder: "L\xF5ppkpv ja -aeg",
        startMonthPlaceholder: "Alguskuu",
        endMonthPlaceholder: "L\xF5ppkuu",
        monthBeforeYear: true,
        firstDayOfWeek: 0,
        today: "T\xE4na"
      },
      DataTable: {
        checkTableAll: "Vali tabelis k\xF5ik",
        uncheckTableAll: "T\xFChista tabeli valik",
        confirm: "Kinnita",
        clear: "T\xFChjenda"
      },
      LegacyTransfer: {
        sourceTitle: "Kust",
        targetTitle: "Kuhu"
      },
      Transfer: {
        selectAll: "Vali k\xF5ik",
        unselectAll: "T\xFChista valik",
        clearAll: "T\xFChjenda",
        total: num => `Kokku ${num} rida`,
        selected: num => `${num} rida valitud`
      },
      Empty: {
        description: "Andmeid pole"
      },
      Select: {
        placeholder: "Vali"
      },
      TimePicker: {
        placeholder: "Vali aeg",
        positiveText: "OK",
        negativeText: "Katkesta",
        now: "N\xFC\xFCd",
        clear: "T\xFChjenda"
      },
      Pagination: {
        goto: "Mine",
        selectionSuffix: "lk"
      },
      DynamicTags: {
        add: "Lisa"
      },
      Log: {
        loading: "Laeb"
      },
      Input: {
        placeholder: "Sisesta"
      },
      InputNumber: {
        placeholder: "Sisesta"
      },
      DynamicInput: {
        create: "Loo"
      },
      ThemeEditor: {
        title: "Teemaredaktor",
        clearAllVars: "T\xFChjenda k\xF5ik muutujad",
        clearSearch: "T\xFChjenda otsing",
        filterCompName: "Filter komponendi nimega",
        filterVarName: "Filter muutuja nimega",
        import: "Import",
        export: "Eksport",
        restore: "Taasta originaal"
      },
      Image: {
        tipPrevious: "Eelmine pilt (\u2190)",
        tipNext: "J\xE4rgmine pilt (\u2192)",
        tipCounterclockwise: "Vastup\xE4eva",
        tipClockwise: "P\xE4rip\xE4eva",
        tipZoomOut: "Suumi v\xE4lja",
        tipZoomIn: "Suumi sisse",
        tipDownload: "Lae alla",
        tipClose: "Sulge (Esc)",
        tipOriginalSize: "Algsuurus"
      }
    };

    const azAZ = {
      name: "az-AZ",
      global: {
        undo: "Geri al",
        redo: "T\u0259krar et",
        confirm: "T\u0259sdiql\u0259",
        clear: "T\u0259mizl\u0259"
      },
      Popconfirm: {
        positiveText: "T\u0259sdiql\u0259",
        negativeText: "\u0130mtina et"
      },
      Cascader: {
        placeholder: "Z\u0259hm\u0259t olmasa, se\xE7in",
        loading: "Y\xFCkl\u0259nir",
        loadingRequiredMessage: label => `Z\u0259hm\u0259t olmasa, yoxlamadan \u0259vv\u0259l b\xFCt\xFCn ${label}l\u0259ri y\xFCkl\u0259yin.`
      },
      Time: {
        dateFormat: "yyyy-MM-dd",
        dateTimeFormat: "yyyy-MM-dd HH:mm:ss"
      },
      DatePicker: {
        yearFormat: "yyyy",
        monthFormat: "MMM",
        dayFormat: "eeeeee",
        yearTypeFormat: "yyyy",
        monthTypeFormat: "yyyy-MM",
        dateFormat: "dd MMMM yyyy",
        dateTimeFormat: "yyyy-MM-dd HH:mm:ss",
        quarterFormat: "yyyy-qqq",
        weekFormat: "YYYY-w",
        clear: "T\u0259mizl\u0259",
        now: "\u0130ndi",
        confirm: "T\u0259sdiql\u0259",
        selectTime: "Vaxt\u0131 se\xE7in",
        selectDate: "Tarixi se\xE7in",
        datePlaceholder: "Tarixi se\xE7",
        datetimePlaceholder: "Tarix v\u0259 vaxt\u0131 se\xE7in",
        monthPlaceholder: "Ayi se\xE7in",
        yearPlaceholder: "\u0130li se\xE7in",
        quarterPlaceholder: "R\xFCbu se\xE7in",
        weekPlaceholder: "H\u0259ft\u0259ni se\xE7in",
        startDatePlaceholder: "Ba\u015Flan\u011F\u0131c tarixi",
        endDatePlaceholder: "Son tarix",
        startDatetimePlaceholder: "Ba\u015Flan\u011F\u0131c tarixi v\u0259 vaxt\u0131",
        endDatetimePlaceholder: "Biti\u015F tarixi v\u0259 vaxt\u0131",
        startMonthPlaceholder: "Ba\u015Flan\u011F\u0131c ay\u0131",
        endMonthPlaceholder: "Son ay",
        monthBeforeYear: true,
        firstDayOfWeek: 0,
        today: "Bu g\xFCn"
      },
      DataTable: {
        checkTableAll: "B\xFCt\xFCn c\u0259dv\u0259li se\xE7",
        uncheckTableAll: "C\u0259dv\u0259ld\u0259ki b\xFCt\xFCn se\xE7iml\u0259ri l\u0259\u011Fv et",
        confirm: "T\u0259sdiql\u0259",
        clear: "T\u0259mizl\u0259"
      },
      LegacyTransfer: {
        sourceTitle: "M\u0259nb\u0259",
        targetTitle: "H\u0259d\u0259f"
      },
      Transfer: {
        selectAll: "Ham\u0131s\u0131n\u0131 se\xE7",
        unselectAll: "Ham\u0131s\u0131n\u0131 l\u0259\u011Fv et",
        clearAll: "T\u0259mizl\u0259",
        total: num => `C\u0259mi ${num} madd\u0259`,
        selected: num => `${num} madd\u0259 se\xE7ildi`
      },
      Empty: {
        description: "M\u0259lumat yoxdur"
      },
      Select: {
        placeholder: "Z\u0259hm\u0259t olmasa, se\xE7in"
      },
      TimePicker: {
        placeholder: "Vaxt\u0131 se\xE7in",
        positiveText: "T\u0259sdiql\u0259",
        negativeText: "\u0130mtina et",
        now: "\u0130ndi",
        clear: "T\u0259mizl\u0259"
      },
      Pagination: {
        goto: "Get",
        selectionSuffix: "s\u0259hif\u0259"
      },
      DynamicTags: {
        add: "\u018Flav\u0259 et"
      },
      Log: {
        loading: "Y\xFCkl\u0259nir"
      },
      Input: {
        placeholder: "Z\u0259hm\u0259t olmasa, daxil edin"
      },
      InputNumber: {
        placeholder: "Z\u0259hm\u0259t olmasa, daxil edin"
      },
      DynamicInput: {
        create: "Yarat"
      },
      ThemeEditor: {
        title: "Tema redaktoru",
        clearAllVars: "B\xFCt\xFCn d\u0259yi\u015F\u0259nl\u0259ri t\u0259mizl\u0259",
        clearSearch: "Axtar\u0131\u015F\u0131 t\u0259mizl\u0259",
        filterCompName: "Komponent ad\u0131n\u0131 filtrl\u0259",
        filterVarName: "D\u0259yi\u015F\u0259n ad\u0131n\u0131 filtrl\u0259",
        import: "\u0130dxal et",
        export: "\u0130xrac et",
        restore: "Varsay\u0131lan\u0131 b\u0259rpa et"
      },
      Image: {
        tipPrevious: "\u018Fvv\u0259lki \u015F\u0259kil (\u2190)",
        tipNext: "N\xF6vb\u0259ti \u015F\u0259kil (\u2192)",
        tipCounterclockwise: "Saat \u0259qr\u0259binin \u0259ksin\u0259",
        tipClockwise: "Saat \u0259qr\u0259binin istiqam\u0259tind\u0259",
        tipZoomOut: "Uzaqla\u015Fd\u0131r",
        tipZoomIn: "Yax\u0131nla\u015Fd\u0131r",
        tipDownload: "Y\xFCkl\u0259",
        tipClose: "Ba\u011Fla (Esc)",
        tipOriginalSize: "Orijinal \xF6l\xE7\xFCy\u0259 yax\u0131nla\u015Fd\u0131r"
      }
    };

    function buildFormatLongFn(args) {
      return (options = {}) => {
        // TODO: Remove String()
        const width = options.width ? String(options.width) : args.defaultWidth;
        const format = args.formats[width] || args.formats[args.defaultWidth];
        return format;
      };
    }

    /* eslint-disable no-unused-vars */

    /**
     * The localize function argument callback which allows to convert raw value to
     * the actual type.
     *
     * @param value - The value to convert
     *
     * @returns The converted value
     */

    /**
     * The map of localized values for each width.
     */

    /**
     * The index type of the locale unit value. It types conversion of units of
     * values that don't start at 0 (i.e. quarters).
     */

    /**
     * Converts the unit value to the tuple of values.
     */

    /**
     * The tuple of localized era values. The first element represents BC,
     * the second element represents AD.
     */

    /**
     * The tuple of localized quarter values. The first element represents Q1.
     */

    /**
     * The tuple of localized day values. The first element represents Sunday.
     */

    /**
     * The tuple of localized month values. The first element represents January.
     */

    function buildLocalizeFn(args) {
      return (value, options) => {
        const context = options?.context ? String(options.context) : "standalone";
        let valuesArray;
        if (context === "formatting" && args.formattingValues) {
          const defaultWidth = args.defaultFormattingWidth || args.defaultWidth;
          const width = options?.width ? String(options.width) : defaultWidth;
          valuesArray = args.formattingValues[width] || args.formattingValues[defaultWidth];
        } else {
          const defaultWidth = args.defaultWidth;
          const width = options?.width ? String(options.width) : args.defaultWidth;
          valuesArray = args.values[width] || args.values[defaultWidth];
        }
        const index = args.argumentCallback ? args.argumentCallback(value) : value;

        // @ts-expect-error - For some reason TypeScript just don't want to match it, no matter how hard we try. I challenge you to try to remove it!
        return valuesArray[index];
      };
    }

    function buildMatchFn(args) {
      return (string, options = {}) => {
        const width = options.width;
        const matchPattern = width && args.matchPatterns[width] || args.matchPatterns[args.defaultMatchWidth];
        const matchResult = string.match(matchPattern);
        if (!matchResult) {
          return null;
        }
        const matchedString = matchResult[0];
        const parsePatterns = width && args.parsePatterns[width] || args.parsePatterns[args.defaultParseWidth];
        const key = Array.isArray(parsePatterns) ? findIndex(parsePatterns, pattern => pattern.test(matchedString)) :
        // eslint-disable-next-line @typescript-eslint/no-explicit-any -- I challange you to fix the type
        findKey(parsePatterns, pattern => pattern.test(matchedString));
        let value;
        value = args.valueCallback ? args.valueCallback(key) : key;
        value = options.valueCallback ?
        // eslint-disable-next-line @typescript-eslint/no-explicit-any -- I challange you to fix the type
        options.valueCallback(value) : value;
        const rest = string.slice(matchedString.length);
        return {
          value,
          rest
        };
      };
    }
    function findKey(object, predicate) {
      for (const key in object) {
        if (Object.prototype.hasOwnProperty.call(object, key) && predicate(object[key])) {
          return key;
        }
      }
      return undefined;
    }
    function findIndex(array, predicate) {
      for (let key = 0; key < array.length; key++) {
        if (predicate(array[key])) {
          return key;
        }
      }
      return undefined;
    }

    function buildMatchPatternFn(args) {
      return (string, options = {}) => {
        const matchResult = string.match(args.matchPattern);
        if (!matchResult) return null;
        const matchedString = matchResult[0];
        const parseResult = string.match(args.parsePattern);
        if (!parseResult) return null;
        let value = args.valueCallback ? args.valueCallback(parseResult[0]) : parseResult[0];

        // eslint-disable-next-line @typescript-eslint/no-explicit-any -- I challange you to fix the type
        value = options.valueCallback ? options.valueCallback(value) : value;
        const rest = string.slice(matchedString.length);
        return {
          value,
          rest
        };
      };
    }

    const formatDistanceLocale$r = {
      lessThanXSeconds: {
        one: "أقل من ثانية واحدة",
        two: "أقل من ثانتين",
        threeToTen: "أقل من {{count}} ثواني",
        other: "أقل من {{count}} ثانية"
      },
      xSeconds: {
        one: "ثانية واحدة",
        two: "ثانتين",
        threeToTen: "{{count}} ثواني",
        other: "{{count}} ثانية"
      },
      halfAMinute: "نصف دقيقة",
      lessThanXMinutes: {
        one: "أقل من دقيقة",
        two: "أقل من دقيقتين",
        threeToTen: "أقل من {{count}} دقائق",
        other: "أقل من {{count}} دقيقة"
      },
      xMinutes: {
        one: "دقيقة واحدة",
        two: "دقيقتين",
        threeToTen: "{{count}} دقائق",
        other: "{{count}} دقيقة"
      },
      aboutXHours: {
        one: "ساعة واحدة تقريباً",
        two: "ساعتين تقريباً",
        threeToTen: "{{count}} ساعات تقريباً",
        other: "{{count}} ساعة تقريباً"
      },
      xHours: {
        one: "ساعة واحدة",
        two: "ساعتين",
        threeToTen: "{{count}} ساعات",
        other: "{{count}} ساعة"
      },
      xDays: {
        one: "يوم واحد",
        two: "يومين",
        threeToTen: "{{count}} أيام",
        other: "{{count}} يوم"
      },
      aboutXWeeks: {
        one: "أسبوع واحد تقريباً",
        two: "أسبوعين تقريباً",
        threeToTen: "{{count}} أسابيع تقريباً",
        other: "{{count}} أسبوع تقريباً"
      },
      xWeeks: {
        one: "أسبوع واحد",
        two: "أسبوعين",
        threeToTen: "{{count}} أسابيع",
        other: "{{count}} أسبوع"
      },
      aboutXMonths: {
        one: "شهر واحد تقريباً",
        two: "شهرين تقريباً",
        threeToTen: "{{count}} أشهر تقريباً",
        other: "{{count}} شهر تقريباً"
      },
      xMonths: {
        one: "شهر واحد",
        two: "شهرين",
        threeToTen: "{{count}} أشهر",
        other: "{{count}} شهر"
      },
      aboutXYears: {
        one: "عام واحد تقريباً",
        two: "عامين تقريباً",
        threeToTen: "{{count}} أعوام تقريباً",
        other: "{{count}} عام تقريباً"
      },
      xYears: {
        one: "عام واحد",
        two: "عامين",
        threeToTen: "{{count}} أعوام",
        other: "{{count}} عام"
      },
      overXYears: {
        one: "أكثر من عام",
        two: "أكثر من عامين",
        threeToTen: "أكثر من {{count}} أعوام",
        other: "أكثر من {{count}} عام"
      },
      almostXYears: {
        one: "عام واحد تقريباً",
        two: "عامين تقريباً",
        threeToTen: "{{count}} أعوام تقريباً",
        other: "{{count}} عام تقريباً"
      }
    };
    const formatDistance$r = (token, count, options) => {
      options = options || {};
      const usageGroup = formatDistanceLocale$r[token];
      let result;
      if (typeof usageGroup === "string") {
        result = usageGroup;
      } else if (count === 1) {
        result = usageGroup.one;
      } else if (count === 2) {
        result = usageGroup.two;
      } else if (count <= 10) {
        result = usageGroup.threeToTen.replace("{{count}}", String(count));
      } else {
        result = usageGroup.other.replace("{{count}}", String(count));
      }
      if (options.addSuffix) {
        if (options.comparison && options.comparison > 0) {
          return "في خلال " + result;
        } else {
          return "منذ " + result;
        }
      }
      return result;
    };

    const dateFormats$s = {
      full: "EEEE, MMMM do, y",
      long: "MMMM do, y",
      medium: "MMM d, y",
      short: "MM/dd/yyyy"
    };
    const timeFormats$s = {
      full: "h:mm:ss a zzzz",
      long: "h:mm:ss a z",
      medium: "h:mm:ss a",
      short: "h:mm a"
    };
    const dateTimeFormats$s = {
      full: "{{date}} 'عند' {{time}}",
      long: "{{date}} 'عند' {{time}}",
      medium: "{{date}}, {{time}}",
      short: "{{date}}, {{time}}"
    };
    const formatLong$s = {
      date: buildFormatLongFn({
        formats: dateFormats$s,
        defaultWidth: "full"
      }),
      time: buildFormatLongFn({
        formats: timeFormats$s,
        defaultWidth: "full"
      }),
      dateTime: buildFormatLongFn({
        formats: dateTimeFormats$s,
        defaultWidth: "full"
      })
    };

    const formatRelativeLocale$r = {
      lastWeek: "'أخر' eeee 'عند' p",
      yesterday: "'أمس عند' p",
      today: "'اليوم عند' p",
      tomorrow: "'غداً عند' p",
      nextWeek: "eeee 'عند' p",
      other: "P"
    };
    const formatRelative$r = (token, _date, _baseDate, _options) => {
      return formatRelativeLocale$r[token];
    };

    const eraValues$r = {
      narrow: ["ق", "ب"],
      abbreviated: ["ق.م.", "ب.م."],
      wide: ["قبل الميلاد", "بعد الميلاد"]
    };
    const quarterValues$r = {
      narrow: ["1", "2", "3", "4"],
      abbreviated: ["ر1", "ر2", "ر3", "ر4"],
      wide: ["الربع الأول", "الربع الثاني", "الربع الثالث", "الربع الرابع"]
    };
    const monthValues$r = {
      narrow: ["ج", "ف", "م", "أ", "م", "ج", "ج", "أ", "س", "أ", "ن", "د"],
      abbreviated: ["جانـ", "فيفـ", "مارس", "أفريل", "مايـ", "جوانـ", "جويـ", "أوت", "سبتـ", "أكتـ", "نوفـ", "ديسـ"],
      wide: ["جانفي", "فيفري", "مارس", "أفريل", "ماي", "جوان", "جويلية", "أوت", "سبتمبر", "أكتوبر", "نوفمبر", "ديسمبر"]
    };
    const dayValues$r = {
      narrow: ["ح", "ن", "ث", "ر", "خ", "ج", "س"],
      short: ["أحد", "اثنين", "ثلاثاء", "أربعاء", "خميس", "جمعة", "سبت"],
      abbreviated: ["أحد", "اثنـ", "ثلا", "أربـ", "خميـ", "جمعة", "سبت"],
      wide: ["الأحد", "الاثنين", "الثلاثاء", "الأربعاء", "الخميس", "الجمعة", "السبت"]
    };
    const dayPeriodValues$r = {
      narrow: {
        am: "ص",
        pm: "م",
        midnight: "ن",
        noon: "ظ",
        morning: "صباحاً",
        afternoon: "بعد الظهر",
        evening: "مساءاً",
        night: "ليلاً"
      },
      abbreviated: {
        am: "ص",
        pm: "م",
        midnight: "نصف الليل",
        noon: "ظهر",
        morning: "صباحاً",
        afternoon: "بعد الظهر",
        evening: "مساءاً",
        night: "ليلاً"
      },
      wide: {
        am: "ص",
        pm: "م",
        midnight: "نصف الليل",
        noon: "ظهر",
        morning: "صباحاً",
        afternoon: "بعد الظهر",
        evening: "مساءاً",
        night: "ليلاً"
      }
    };
    const formattingDayPeriodValues$m = {
      narrow: {
        am: "ص",
        pm: "م",
        midnight: "ن",
        noon: "ظ",
        morning: "في الصباح",
        afternoon: "بعد الظـهر",
        evening: "في المساء",
        night: "في الليل"
      },
      abbreviated: {
        am: "ص",
        pm: "م",
        midnight: "نصف الليل",
        noon: "ظهر",
        morning: "في الصباح",
        afternoon: "بعد الظهر",
        evening: "في المساء",
        night: "في الليل"
      },
      wide: {
        am: "ص",
        pm: "م",
        midnight: "نصف الليل",
        noon: "ظهر",
        morning: "صباحاً",
        afternoon: "بعد الظـهر",
        evening: "في المساء",
        night: "في الليل"
      }
    };
    const ordinalNumber$r = dirtyNumber => {
      return String(dirtyNumber);
    };
    const localize$r = {
      ordinalNumber: ordinalNumber$r,
      era: buildLocalizeFn({
        values: eraValues$r,
        defaultWidth: "wide"
      }),
      quarter: buildLocalizeFn({
        values: quarterValues$r,
        defaultWidth: "wide",
        argumentCallback: quarter => Number(quarter) - 1
      }),
      month: buildLocalizeFn({
        values: monthValues$r,
        defaultWidth: "wide"
      }),
      day: buildLocalizeFn({
        values: dayValues$r,
        defaultWidth: "wide"
      }),
      dayPeriod: buildLocalizeFn({
        values: dayPeriodValues$r,
        defaultWidth: "wide",
        formattingValues: formattingDayPeriodValues$m,
        defaultFormattingWidth: "wide"
      })
    };

    const matchOrdinalNumberPattern$r = /^(\d+)(th|st|nd|rd)?/i;
    const parseOrdinalNumberPattern$r = /\d+/i;
    const matchEraPatterns$r = {
      narrow: /^(ق|ب)/i,
      abbreviated: /^(ق\.?\s?م\.?|ق\.?\s?م\.?\s?|a\.?\s?d\.?|c\.?\s?)/i,
      wide: /^(قبل الميلاد|قبل الميلاد|بعد الميلاد|بعد الميلاد)/i
    };
    const parseEraPatterns$r = {
      any: [/^قبل/i, /^بعد/i]
    };
    const matchQuarterPatterns$r = {
      narrow: /^[1234]/i,
      abbreviated: /^ر[1234]/i,
      wide: /^الربع [1234]/i
    };
    const parseQuarterPatterns$r = {
      any: [/1/i, /2/i, /3/i, /4/i]
    };
    const matchMonthPatterns$r = {
      narrow: /^[جفمأسند]/i,
      abbreviated: /^(جان|فيف|مار|أفر|ماي|جوا|جوي|أوت|سبت|أكت|نوف|ديس)/i,
      wide: /^(جانفي|فيفري|مارس|أفريل|ماي|جوان|جويلية|أوت|سبتمبر|أكتوبر|نوفمبر|ديسمبر)/i
    };
    const parseMonthPatterns$r = {
      narrow: [/^ج/i, /^ف/i, /^م/i, /^أ/i, /^م/i, /^ج/i, /^ج/i, /^أ/i, /^س/i, /^أ/i, /^ن/i, /^د/i],
      any: [/^جان/i, /^فيف/i, /^مار/i, /^أفر/i, /^ماي/i, /^جوا/i, /^جوي/i, /^أوت/i, /^سبت/i, /^أكت/i, /^نوف/i, /^ديس/i]
    };
    const matchDayPatterns$r = {
      narrow: /^[حنثرخجس]/i,
      short: /^(أحد|اثنين|ثلاثاء|أربعاء|خميس|جمعة|سبت)/i,
      abbreviated: /^(أحد|اثن|ثلا|أرب|خمي|جمعة|سبت)/i,
      wide: /^(الأحد|الاثنين|الثلاثاء|الأربعاء|الخميس|الجمعة|السبت)/i
    };
    const parseDayPatterns$r = {
      narrow: [/^ح/i, /^ن/i, /^ث/i, /^ر/i, /^خ/i, /^ج/i, /^س/i],
      wide: [/^الأحد/i, /^الاثنين/i, /^الثلاثاء/i, /^الأربعاء/i, /^الخميس/i, /^الجمعة/i, /^السبت/i],
      any: [/^أح/i, /^اث/i, /^ث/i, /^أر/i, /^خ/i, /^ج/i, /^س/i]
    };
    const matchDayPeriodPatterns$r = {
      narrow: /^(a|p|mi|n|(in the|at) (morning|afternoon|evening|night))/i,
      any: /^([ap]\.?\s?m\.?|midnight|noon|(in the|at) (morning|afternoon|evening|night))/i
    };
    const parseDayPeriodPatterns$r = {
      any: {
        am: /^a/i,
        pm: /^p/i,
        midnight: /^mi/i,
        noon: /^no/i,
        morning: /morning/i,
        afternoon: /afternoon/i,
        evening: /evening/i,
        night: /night/i
      }
    };
    const match$r = {
      ordinalNumber: buildMatchPatternFn({
        matchPattern: matchOrdinalNumberPattern$r,
        parsePattern: parseOrdinalNumberPattern$r,
        valueCallback: value => parseInt(value, 10)
      }),
      era: buildMatchFn({
        matchPatterns: matchEraPatterns$r,
        defaultMatchWidth: "wide",
        parsePatterns: parseEraPatterns$r,
        defaultParseWidth: "any"
      }),
      quarter: buildMatchFn({
        matchPatterns: matchQuarterPatterns$r,
        defaultMatchWidth: "wide",
        parsePatterns: parseQuarterPatterns$r,
        defaultParseWidth: "any",
        valueCallback: index => Number(index) + 1
      }),
      month: buildMatchFn({
        matchPatterns: matchMonthPatterns$r,
        defaultMatchWidth: "wide",
        parsePatterns: parseMonthPatterns$r,
        defaultParseWidth: "any"
      }),
      day: buildMatchFn({
        matchPatterns: matchDayPatterns$r,
        defaultMatchWidth: "wide",
        parsePatterns: parseDayPatterns$r,
        defaultParseWidth: "any"
      }),
      dayPeriod: buildMatchFn({
        matchPatterns: matchDayPeriodPatterns$r,
        defaultMatchWidth: "any",
        parsePatterns: parseDayPeriodPatterns$r,
        defaultParseWidth: "any"
      })
    };

    /**
     * @category Locales
     * @summary Arabic locale (Algerian Arabic).
     * @language Algerian Arabic
     * @iso-639-2 ara
     * @author Badreddine Boumaza [@badre429](https://github.com/badre429)
     * @author Ahmed ElShahat [@elshahat](https://github.com/elshahat)
     */
    const arDZ = {
      code: "ar-DZ",
      formatDistance: formatDistance$r,
      formatLong: formatLong$s,
      formatRelative: formatRelative$r,
      localize: localize$r,
      match: match$r,
      options: {
        weekStartsOn: 0 /* Sunday */,
        firstWeekContainsDate: 1
      }
    };

    const formatDistanceLocale$q = {
      lessThanXSeconds: {
        one: "bir saniyədən az",
        other: "{{count}} bir saniyədən az"
      },
      xSeconds: {
        one: "1 saniyə",
        other: "{{count}} saniyə"
      },
      halfAMinute: "yarım dəqiqə",
      lessThanXMinutes: {
        one: "bir dəqiqədən az",
        other: "{{count}} bir dəqiqədən az"
      },
      xMinutes: {
        one: "bir dəqiqə",
        other: "{{count}} dəqiqə"
      },
      aboutXHours: {
        one: "təxminən 1 saat",
        other: "təxminən {{count}} saat"
      },
      xHours: {
        one: "1 saat",
        other: "{{count}} saat"
      },
      xDays: {
        one: "1 gün",
        other: "{{count}} gün"
      },
      aboutXWeeks: {
        one: "təxminən 1 həftə",
        other: "təxminən {{count}} həftə"
      },
      xWeeks: {
        one: "1 həftə",
        other: "{{count}} həftə"
      },
      aboutXMonths: {
        one: "təxminən 1 ay",
        other: "təxminən {{count}} ay"
      },
      xMonths: {
        one: "1 ay",
        other: "{{count}} ay"
      },
      aboutXYears: {
        one: "təxminən 1 il",
        other: "təxminən {{count}} il"
      },
      xYears: {
        one: "1 il",
        other: "{{count}} il"
      },
      overXYears: {
        one: "1 ildən çox",
        other: "{{count}} ildən çox"
      },
      almostXYears: {
        one: "demək olar ki 1 il",
        other: "demək olar ki {{count}} il"
      }
    };
    const formatDistance$q = (token, count, options) => {
      let result;
      const tokenValue = formatDistanceLocale$q[token];
      if (typeof tokenValue === "string") {
        result = tokenValue;
      } else if (count === 1) {
        result = tokenValue.one;
      } else {
        result = tokenValue.other.replace("{{count}}", String(count));
      }
      if (options?.addSuffix) {
        if (options.comparison && options.comparison > 0) {
          return result + " sonra";
        } else {
          return result + " əvvəl";
        }
      }
      return result;
    };

    const dateFormats$r = {
      full: "EEEE, do MMMM y 'il'",
      long: "do MMMM y 'il'",
      medium: "d MMM y 'il'",
      short: "dd.MM.yyyy"
    };
    const timeFormats$r = {
      full: "H:mm:ss zzzz",
      long: "H:mm:ss z",
      medium: "H:mm:ss",
      short: "H:mm"
    };
    const dateTimeFormats$r = {
      full: "{{date}} {{time}} - 'də'",
      long: "{{date}} {{time}} - 'də'",
      medium: "{{date}}, {{time}}",
      short: "{{date}}, {{time}}"
    };
    const formatLong$r = {
      date: buildFormatLongFn({
        formats: dateFormats$r,
        defaultWidth: "full"
      }),
      time: buildFormatLongFn({
        formats: timeFormats$r,
        defaultWidth: "full"
      }),
      dateTime: buildFormatLongFn({
        formats: dateTimeFormats$r,
        defaultWidth: "full"
      })
    };

    const formatRelativeLocale$q = {
      lastWeek: "'sonuncu' eeee p -'də'",
      yesterday: "'dünən' p -'də'",
      today: "'bugün' p -'də'",
      tomorrow: "'sabah' p -'də'",
      nextWeek: "eeee p -'də'",
      other: "P"
    };
    const formatRelative$q = (token, _date, _baseDate, _options) => formatRelativeLocale$q[token];

    const eraValues$q = {
      narrow: ["e.ə", "b.e"],
      abbreviated: ["e.ə", "b.e"],
      wide: ["eramızdan əvvəl", "bizim era"]
    };
    const quarterValues$q = {
      narrow: ["1", "2", "3", "4"],
      abbreviated: ["K1", "K2", "K3", "K4"],
      wide: ["1ci kvartal", "2ci kvartal", "3cü kvartal", "4cü kvartal"]
    };
    const monthValues$q = {
      narrow: ["Y", "F", "M", "A", "M", "İ", "İ", "A", "S", "O", "N", "D"],
      abbreviated: ["Yan", "Fev", "Mar", "Apr", "May", "İyun", "İyul", "Avq", "Sen", "Okt", "Noy", "Dek"],
      wide: ["Yanvar", "Fevral", "Mart", "Aprel", "May", "İyun", "İyul", "Avqust", "Sentyabr", "Oktyabr", "Noyabr", "Dekabr"]
    };
    const dayValues$q = {
      narrow: ["B.", "B.e", "Ç.a", "Ç.", "C.a", "C.", "Ş."],
      short: ["B.", "B.e", "Ç.a", "Ç.", "C.a", "C.", "Ş."],
      abbreviated: ["Baz", "Baz.e", "Çər.a", "Çər", "Cüm.a", "Cüm", "Şə"],
      wide: ["Bazar", "Bazar ertəsi", "Çərşənbə axşamı", "Çərşənbə", "Cümə axşamı", "Cümə", "Şənbə"]
    };
    const dayPeriodValues$q = {
      narrow: {
        am: "am",
        pm: "pm",
        midnight: "gecəyarı",
        noon: "gün",
        morning: "səhər",
        afternoon: "gündüz",
        evening: "axşam",
        night: "gecə"
      },
      abbreviated: {
        am: "AM",
        pm: "PM",
        midnight: "gecəyarı",
        noon: "gün",
        morning: "səhər",
        afternoon: "gündüz",
        evening: "axşam",
        night: "gecə"
      },
      wide: {
        am: "a.m.",
        pm: "p.m.",
        midnight: "gecəyarı",
        noon: "gün",
        morning: "səhər",
        afternoon: "gündüz",
        evening: "axşam",
        night: "gecə"
      }
    };
    const formattingDayPeriodValues$l = {
      narrow: {
        am: "a",
        pm: "p",
        midnight: "gecəyarı",
        noon: "gün",
        morning: "səhər",
        afternoon: "gündüz",
        evening: "axşam",
        night: "gecə"
      },
      abbreviated: {
        am: "AM",
        pm: "PM",
        midnight: "gecəyarı",
        noon: "gün",
        morning: "səhər",
        afternoon: "gündüz",
        evening: "axşam",
        night: "gecə"
      },
      wide: {
        am: "a.m.",
        pm: "p.m.",
        midnight: "gecəyarı",
        noon: "gün",
        morning: "səhər",
        afternoon: "gündüz",
        evening: "axşam",
        night: "gecə"
      }
    };
    const suffixes = {
      1: "-inci",
      5: "-inci",
      8: "-inci",
      70: "-inci",
      80: "-inci",
      2: "-nci",
      7: "-nci",
      20: "-nci",
      50: "-nci",
      3: "-üncü",
      4: "-üncü",
      100: "-üncü",
      6: "-ncı",
      9: "-uncu",
      10: "-uncu",
      30: "-uncu",
      60: "-ıncı",
      90: "-ıncı"
    };
    const getSuffix = number => {
      if (number === 0) {
        // special case for zero
        return number + "-ıncı";
      }
      const a = number % 10;
      const b = number % 100 - a;
      const c = number >= 100 ? 100 : null;
      if (suffixes[a]) {
        return suffixes[a];
      } else if (suffixes[b]) {
        return suffixes[b];
      } else if (c !== null) {
        return suffixes[c];
      }
      return "";
    };
    const ordinalNumber$q = (dirtyNumber, _options) => {
      const number = Number(dirtyNumber);
      const suffix = getSuffix(number);
      return number + suffix;
    };
    const localize$q = {
      ordinalNumber: ordinalNumber$q,
      era: buildLocalizeFn({
        values: eraValues$q,
        defaultWidth: "wide"
      }),
      quarter: buildLocalizeFn({
        values: quarterValues$q,
        defaultWidth: "wide",
        argumentCallback: quarter => quarter - 1
      }),
      month: buildLocalizeFn({
        values: monthValues$q,
        defaultWidth: "wide"
      }),
      day: buildLocalizeFn({
        values: dayValues$q,
        defaultWidth: "wide"
      }),
      dayPeriod: buildLocalizeFn({
        values: dayPeriodValues$q,
        defaultWidth: "wide",
        formattingValues: formattingDayPeriodValues$l,
        defaultFormattingWidth: "wide"
      })
    };

    const matchOrdinalNumberPattern$q = /^(\d+)(-?(ci|inci|nci|uncu|üncü|ncı))?/i;
    const parseOrdinalNumberPattern$q = /\d+/i;
    const matchEraPatterns$q = {
      narrow: /^(b|a)$/i,
      abbreviated: /^(b\.?\s?c\.?|b\.?\s?c\.?\s?e\.?|a\.?\s?d\.?|c\.?\s?e\.?)$/i,
      wide: /^(bizim eradan əvvəl|bizim era)$/i
    };
    const parseEraPatterns$q = {
      any: [/^b$/i, /^(a|c)$/i]
    };
    const matchQuarterPatterns$q = {
      narrow: /^[1234]$/i,
      abbreviated: /^K[1234]$/i,
      wide: /^[1234](ci)? kvartal$/i
    };
    const parseQuarterPatterns$q = {
      any: [/1/i, /2/i, /3/i, /4/i]
    };
    const matchMonthPatterns$q = {
      narrow: /^[(?-i)yfmaisond]$/i,
      abbreviated: /^(Yan|Fev|Mar|Apr|May|İyun|İyul|Avq|Sen|Okt|Noy|Dek)$/i,
      wide: /^(Yanvar|Fevral|Mart|Aprel|May|İyun|İyul|Avgust|Sentyabr|Oktyabr|Noyabr|Dekabr)$/i
    };
    const parseMonthPatterns$q = {
      narrow: [/^[(?-i)y]$/i, /^[(?-i)f]$/i, /^[(?-i)m]$/i, /^[(?-i)a]$/i, /^[(?-i)m]$/i, /^[(?-i)i]$/i, /^[(?-i)i]$/i, /^[(?-i)a]$/i, /^[(?-i)s]$/i, /^[(?-i)o]$/i, /^[(?-i)n]$/i, /^[(?-i)d]$/i],
      abbreviated: [/^Yan$/i, /^Fev$/i, /^Mar$/i, /^Apr$/i, /^May$/i, /^İyun$/i, /^İyul$/i, /^Avg$/i, /^Sen$/i, /^Okt$/i, /^Noy$/i, /^Dek$/i],
      wide: [/^Yanvar$/i, /^Fevral$/i, /^Mart$/i, /^Aprel$/i, /^May$/i, /^İyun$/i, /^İyul$/i, /^Avgust$/i, /^Sentyabr$/i, /^Oktyabr$/i, /^Noyabr$/i, /^Dekabr$/i]
    };
    const matchDayPatterns$q = {
      narrow: /^(B\.|B\.e|Ç\.a|Ç\.|C\.a|C\.|Ş\.)$/i,
      short: /^(B\.|B\.e|Ç\.a|Ç\.|C\.a|C\.|Ş\.)$/i,
      abbreviated: /^(Baz\.e|Çər|Çər\.a|Cüm|Cüm\.a|Şə)$/i,
      wide: /^(Bazar|Bazar ertəsi|Çərşənbə axşamı|Çərşənbə|Cümə axşamı|Cümə|Şənbə)$/i
    };
    const parseDayPatterns$q = {
      narrow: [/^B\.$/i, /^B\.e$/i, /^Ç\.a$/i, /^Ç\.$/i, /^C\.a$/i, /^C\.$/i, /^Ş\.$/i],
      abbreviated: [/^Baz$/i, /^Baz\.e$/i, /^Çər\.a$/i, /^Çər$/i, /^Cüm\.a$/i, /^Cüm$/i, /^Şə$/i],
      wide: [/^Bazar$/i, /^Bazar ertəsi$/i, /^Çərşənbə axşamı$/i, /^Çərşənbə$/i, /^Cümə axşamı$/i, /^Cümə$/i, /^Şənbə$/i],
      any: [/^B\.$/i, /^B\.e$/i, /^Ç\.a$/i, /^Ç\.$/i, /^C\.a$/i, /^C\.$/i, /^Ş\.$/i]
    };
    const matchDayPeriodPatterns$q = {
      narrow: /^(a|p|gecəyarı|gün|səhər|gündüz|axşam|gecə)$/i,
      any: /^(am|pm|a\.m\.|p\.m\.|AM|PM|gecəyarı|gün|səhər|gündüz|axşam|gecə)$/i
    };
    const parseDayPeriodPatterns$q = {
      any: {
        am: /^a$/i,
        pm: /^p$/i,
        midnight: /^gecəyarı$/i,
        noon: /^gün$/i,
        morning: /səhər$/i,
        afternoon: /gündüz$/i,
        evening: /axşam$/i,
        night: /gecə$/i
      }
    };
    const match$q = {
      ordinalNumber: buildMatchPatternFn({
        matchPattern: matchOrdinalNumberPattern$q,
        parsePattern: parseOrdinalNumberPattern$q,
        valueCallback: value => parseInt(value, 10)
      }),
      era: buildMatchFn({
        matchPatterns: matchEraPatterns$q,
        defaultMatchWidth: "wide",
        parsePatterns: parseEraPatterns$q,
        defaultParseWidth: "any"
      }),
      quarter: buildMatchFn({
        matchPatterns: matchQuarterPatterns$q,
        defaultMatchWidth: "wide",
        parsePatterns: parseQuarterPatterns$q,
        defaultParseWidth: "any",
        valueCallback: index => index + 1
      }),
      month: buildMatchFn({
        matchPatterns: matchMonthPatterns$q,
        defaultMatchWidth: "wide",
        parsePatterns: parseMonthPatterns$q,
        defaultParseWidth: "narrow"
      }),
      day: buildMatchFn({
        matchPatterns: matchDayPatterns$q,
        defaultMatchWidth: "wide",
        parsePatterns: parseDayPatterns$q,
        defaultParseWidth: "any"
      }),
      dayPeriod: buildMatchFn({
        matchPatterns: matchDayPeriodPatterns$q,
        defaultMatchWidth: "any",
        parsePatterns: parseDayPeriodPatterns$q,
        defaultParseWidth: "any"
      })
    };

    /**
     * @category Locales
     * @summary Azerbaijani locale.
     * @language Azerbaijani
     * @iso-639-2 aze
     */

    const az = {
      code: "az",
      formatDistance: formatDistance$q,
      formatLong: formatLong$r,
      formatRelative: formatRelative$q,
      localize: localize$q,
      match: match$q,
      options: {
        weekStartsOn: 1,
        firstWeekContainsDate: 1
      }
    };

    /**
     * @name toDate
     * @category Common Helpers
     * @summary Convert the given argument to an instance of Date.
     *
     * @description
     * Convert the given argument to an instance of Date.
     *
     * If the argument is an instance of Date, the function returns its clone.
     *
     * If the argument is a number, it is treated as a timestamp.
     *
     * If the argument is none of the above, the function returns Invalid Date.
     *
     * **Note**: *all* Date arguments passed to any *date-fns* function is processed by `toDate`.
     *
     * @typeParam DateType - The `Date` type, the function operates on. Gets inferred from passed arguments. Allows to use extensions like [`UTCDate`](https://github.com/date-fns/utc).
     *
     * @param argument - The value to convert
     *
     * @returns The parsed date in the local time zone
     *
     * @example
     * // Clone the date:
     * const result = toDate(new Date(2014, 1, 11, 11, 30, 30))
     * //=> Tue Feb 11 2014 11:30:30
     *
     * @example
     * // Convert the timestamp to date:
     * const result = toDate(1392098430000)
     * //=> Tue Feb 11 2014 11:30:30
     */
    function toDate$1(argument) {
      const argStr = Object.prototype.toString.call(argument);

      // Clone the date
      if (argument instanceof Date || typeof argument === "object" && argStr === "[object Date]") {
        // Prevent the date to lose the milliseconds when passed to new Date() in IE10
        return new argument.constructor(+argument);
      } else if (typeof argument === "number" || argStr === "[object Number]" || typeof argument === "string" || argStr === "[object String]") {
        // TODO: Can we get rid of as?
        return new Date(argument);
      } else {
        // TODO: Can we get rid of as?
        return new Date(NaN);
      }
    }

    let defaultOptions = {};
    function getDefaultOptions$1() {
      return defaultOptions;
    }

    /**
     * The {@link startOfWeek} function options.
     */

    /**
     * @name startOfWeek
     * @category Week Helpers
     * @summary Return the start of a week for the given date.
     *
     * @description
     * Return the start of a week for the given date.
     * The result will be in the local timezone.
     *
     * @typeParam DateType - The `Date` type, the function operates on. Gets inferred from passed arguments. Allows to use extensions like [`UTCDate`](https://github.com/date-fns/utc).
     *
     * @param date - The original date
     * @param options - An object with options
     *
     * @returns The start of a week
     *
     * @example
     * // The start of a week for 2 September 2014 11:55:00:
     * const result = startOfWeek(new Date(2014, 8, 2, 11, 55, 0))
     * //=> Sun Aug 31 2014 00:00:00
     *
     * @example
     * // If the week starts on Monday, the start of the week for 2 September 2014 11:55:00:
     * const result = startOfWeek(new Date(2014, 8, 2, 11, 55, 0), { weekStartsOn: 1 })
     * //=> Mon Sep 01 2014 00:00:00
     */
    function startOfWeek(date, options) {
      const defaultOptions = getDefaultOptions$1();
      const weekStartsOn = options?.weekStartsOn ?? options?.locale?.options?.weekStartsOn ?? defaultOptions.weekStartsOn ?? defaultOptions.locale?.options?.weekStartsOn ?? 0;
      const _date = toDate$1(date);
      const day = _date.getDay();
      const diff = (day < weekStartsOn ? 7 : 0) + day - weekStartsOn;
      _date.setDate(_date.getDate() - diff);
      _date.setHours(0, 0, 0, 0);
      return _date;
    }

    /**
     * The {@link isSameWeek} function options.
     */

    /**
     * @name isSameWeek
     * @category Week Helpers
     * @summary Are the given dates in the same week (and month and year)?
     *
     * @description
     * Are the given dates in the same week (and month and year)?
     *
     * @typeParam DateType - The `Date` type, the function operates on. Gets inferred from passed arguments. Allows to use extensions like [`UTCDate`](https://github.com/date-fns/utc).
     *
     * @param dateLeft - The first date to check
     * @param dateRight - The second date to check
     * @param options - An object with options
     *
     * @returns The dates are in the same week (and month and year)
     *
     * @example
     * // Are 31 August 2014 and 4 September 2014 in the same week?
     * const result = isSameWeek(new Date(2014, 7, 31), new Date(2014, 8, 4))
     * //=> true
     *
     * @example
     * // If week starts with Monday,
     * // are 31 August 2014 and 4 September 2014 in the same week?
     * const result = isSameWeek(new Date(2014, 7, 31), new Date(2014, 8, 4), {
     *   weekStartsOn: 1
     * })
     * //=> false
     *
     * @example
     * // Are 1 January 2014 and 1 January 2015 in the same week?
     * const result = isSameWeek(new Date(2014, 0, 1), new Date(2015, 0, 1))
     * //=> false
     */
    function isSameWeek(dateLeft, dateRight, options) {
      const dateLeftStartOfWeek = startOfWeek(dateLeft, options);
      const dateRightStartOfWeek = startOfWeek(dateRight, options);
      return +dateLeftStartOfWeek === +dateRightStartOfWeek;
    }

    const formatDistanceLocale$p = {
      lessThanXSeconds: {
        one: {
          regular: "méně než 1 sekunda",
          past: "před méně než 1 sekundou",
          future: "za méně než 1 sekundu"
        },
        few: {
          regular: "méně než {{count}} sekundy",
          past: "před méně než {{count}} sekundami",
          future: "za méně než {{count}} sekundy"
        },
        many: {
          regular: "méně než {{count}} sekund",
          past: "před méně než {{count}} sekundami",
          future: "za méně než {{count}} sekund"
        }
      },
      xSeconds: {
        one: {
          regular: "1 sekunda",
          past: "před 1 sekundou",
          future: "za 1 sekundu"
        },
        few: {
          regular: "{{count}} sekundy",
          past: "před {{count}} sekundami",
          future: "za {{count}} sekundy"
        },
        many: {
          regular: "{{count}} sekund",
          past: "před {{count}} sekundami",
          future: "za {{count}} sekund"
        }
      },
      halfAMinute: {
        type: "other",
        other: {
          regular: "půl minuty",
          past: "před půl minutou",
          future: "za půl minuty"
        }
      },
      lessThanXMinutes: {
        one: {
          regular: "méně než 1 minuta",
          past: "před méně než 1 minutou",
          future: "za méně než 1 minutu"
        },
        few: {
          regular: "méně než {{count}} minuty",
          past: "před méně než {{count}} minutami",
          future: "za méně než {{count}} minuty"
        },
        many: {
          regular: "méně než {{count}} minut",
          past: "před méně než {{count}} minutami",
          future: "za méně než {{count}} minut"
        }
      },
      xMinutes: {
        one: {
          regular: "1 minuta",
          past: "před 1 minutou",
          future: "za 1 minutu"
        },
        few: {
          regular: "{{count}} minuty",
          past: "před {{count}} minutami",
          future: "za {{count}} minuty"
        },
        many: {
          regular: "{{count}} minut",
          past: "před {{count}} minutami",
          future: "za {{count}} minut"
        }
      },
      aboutXHours: {
        one: {
          regular: "přibližně 1 hodina",
          past: "přibližně před 1 hodinou",
          future: "přibližně za 1 hodinu"
        },
        few: {
          regular: "přibližně {{count}} hodiny",
          past: "přibližně před {{count}} hodinami",
          future: "přibližně za {{count}} hodiny"
        },
        many: {
          regular: "přibližně {{count}} hodin",
          past: "přibližně před {{count}} hodinami",
          future: "přibližně za {{count}} hodin"
        }
      },
      xHours: {
        one: {
          regular: "1 hodina",
          past: "před 1 hodinou",
          future: "za 1 hodinu"
        },
        few: {
          regular: "{{count}} hodiny",
          past: "před {{count}} hodinami",
          future: "za {{count}} hodiny"
        },
        many: {
          regular: "{{count}} hodin",
          past: "před {{count}} hodinami",
          future: "za {{count}} hodin"
        }
      },
      xDays: {
        one: {
          regular: "1 den",
          past: "před 1 dnem",
          future: "za 1 den"
        },
        few: {
          regular: "{{count}} dny",
          past: "před {{count}} dny",
          future: "za {{count}} dny"
        },
        many: {
          regular: "{{count}} dní",
          past: "před {{count}} dny",
          future: "za {{count}} dní"
        }
      },
      aboutXWeeks: {
        one: {
          regular: "přibližně 1 týden",
          past: "přibližně před 1 týdnem",
          future: "přibližně za 1 týden"
        },
        few: {
          regular: "přibližně {{count}} týdny",
          past: "přibližně před {{count}} týdny",
          future: "přibližně za {{count}} týdny"
        },
        many: {
          regular: "přibližně {{count}} týdnů",
          past: "přibližně před {{count}} týdny",
          future: "přibližně za {{count}} týdnů"
        }
      },
      xWeeks: {
        one: {
          regular: "1 týden",
          past: "před 1 týdnem",
          future: "za 1 týden"
        },
        few: {
          regular: "{{count}} týdny",
          past: "před {{count}} týdny",
          future: "za {{count}} týdny"
        },
        many: {
          regular: "{{count}} týdnů",
          past: "před {{count}} týdny",
          future: "za {{count}} týdnů"
        }
      },
      aboutXMonths: {
        one: {
          regular: "přibližně 1 měsíc",
          past: "přibližně před 1 měsícem",
          future: "přibližně za 1 měsíc"
        },
        few: {
          regular: "přibližně {{count}} měsíce",
          past: "přibližně před {{count}} měsíci",
          future: "přibližně za {{count}} měsíce"
        },
        many: {
          regular: "přibližně {{count}} měsíců",
          past: "přibližně před {{count}} měsíci",
          future: "přibližně za {{count}} měsíců"
        }
      },
      xMonths: {
        one: {
          regular: "1 měsíc",
          past: "před 1 měsícem",
          future: "za 1 měsíc"
        },
        few: {
          regular: "{{count}} měsíce",
          past: "před {{count}} měsíci",
          future: "za {{count}} měsíce"
        },
        many: {
          regular: "{{count}} měsíců",
          past: "před {{count}} měsíci",
          future: "za {{count}} měsíců"
        }
      },
      aboutXYears: {
        one: {
          regular: "přibližně 1 rok",
          past: "přibližně před 1 rokem",
          future: "přibližně za 1 rok"
        },
        few: {
          regular: "přibližně {{count}} roky",
          past: "přibližně před {{count}} roky",
          future: "přibližně za {{count}} roky"
        },
        many: {
          regular: "přibližně {{count}} roků",
          past: "přibližně před {{count}} roky",
          future: "přibližně za {{count}} roků"
        }
      },
      xYears: {
        one: {
          regular: "1 rok",
          past: "před 1 rokem",
          future: "za 1 rok"
        },
        few: {
          regular: "{{count}} roky",
          past: "před {{count}} roky",
          future: "za {{count}} roky"
        },
        many: {
          regular: "{{count}} roků",
          past: "před {{count}} roky",
          future: "za {{count}} roků"
        }
      },
      overXYears: {
        one: {
          regular: "více než 1 rok",
          past: "před více než 1 rokem",
          future: "za více než 1 rok"
        },
        few: {
          regular: "více než {{count}} roky",
          past: "před více než {{count}} roky",
          future: "za více než {{count}} roky"
        },
        many: {
          regular: "více než {{count}} roků",
          past: "před více než {{count}} roky",
          future: "za více než {{count}} roků"
        }
      },
      almostXYears: {
        one: {
          regular: "skoro 1 rok",
          past: "skoro před 1 rokem",
          future: "skoro za 1 rok"
        },
        few: {
          regular: "skoro {{count}} roky",
          past: "skoro před {{count}} roky",
          future: "skoro za {{count}} roky"
        },
        many: {
          regular: "skoro {{count}} roků",
          past: "skoro před {{count}} roky",
          future: "skoro za {{count}} roků"
        }
      }
    };
    const formatDistance$p = (token, count, options) => {
      let pluralResult;
      const tokenValue = formatDistanceLocale$p[token];

      // cs pluralization
      if (tokenValue.type === "other") {
        pluralResult = tokenValue.other;
      } else if (count === 1) {
        pluralResult = tokenValue.one;
      } else if (count > 1 && count < 5) {
        pluralResult = tokenValue.few;
      } else {
        pluralResult = tokenValue.many;
      }

      // times
      const suffixExist = options?.addSuffix === true;
      const comparison = options?.comparison;
      let timeResult;
      if (suffixExist && comparison === -1) {
        timeResult = pluralResult.past;
      } else if (suffixExist && comparison === 1) {
        timeResult = pluralResult.future;
      } else {
        timeResult = pluralResult.regular;
      }
      return timeResult.replace("{{count}}", String(count));
    };

    const dateFormats$q = {
      full: "EEEE, d. MMMM yyyy",
      long: "d. MMMM yyyy",
      medium: "d. M. yyyy",
      short: "dd.MM.yyyy"
    };
    const timeFormats$q = {
      full: "H:mm:ss zzzz",
      long: "H:mm:ss z",
      medium: "H:mm:ss",
      short: "H:mm"
    };
    const dateTimeFormats$q = {
      full: "{{date}} 'v' {{time}}",
      long: "{{date}} 'v' {{time}}",
      medium: "{{date}}, {{time}}",
      short: "{{date}}, {{time}}"
    };
    const formatLong$q = {
      date: buildFormatLongFn({
        formats: dateFormats$q,
        defaultWidth: "full"
      }),
      time: buildFormatLongFn({
        formats: timeFormats$q,
        defaultWidth: "full"
      }),
      dateTime: buildFormatLongFn({
        formats: dateTimeFormats$q,
        defaultWidth: "full"
      })
    };

    const accusativeWeekdays$3 = ["neděli", "pondělí", "úterý", "středu", "čtvrtek", "pátek", "sobotu"];
    const formatRelativeLocale$p = {
      lastWeek: "'poslední' eeee 've' p",
      yesterday: "'včera v' p",
      today: "'dnes v' p",
      tomorrow: "'zítra v' p",
      nextWeek: date => {
        const day = date.getDay();
        return "'v " + accusativeWeekdays$3[day] + " o' p";
      },
      other: "P"
    };
    const formatRelative$p = (token, date) => {
      const format = formatRelativeLocale$p[token];
      if (typeof format === "function") {
        return format(date);
      }
      return format;
    };

    const eraValues$p = {
      narrow: ["př. n. l.", "n. l."],
      abbreviated: ["př. n. l.", "n. l."],
      wide: ["před naším letopočtem", "našeho letopočtu"]
    };
    const quarterValues$p = {
      narrow: ["1", "2", "3", "4"],
      abbreviated: ["1. čtvrtletí", "2. čtvrtletí", "3. čtvrtletí", "4. čtvrtletí"],
      wide: ["1. čtvrtletí", "2. čtvrtletí", "3. čtvrtletí", "4. čtvrtletí"]
    };
    const monthValues$p = {
      narrow: ["L", "Ú", "B", "D", "K", "Č", "Č", "S", "Z", "Ř", "L", "P"],
      abbreviated: ["led", "úno", "bře", "dub", "kvě", "čvn", "čvc", "srp", "zář", "říj", "lis", "pro"],
      wide: ["leden", "únor", "březen", "duben", "květen", "červen", "červenec", "srpen", "září", "říjen", "listopad", "prosinec"]
    };
    const formattingMonthValues$5 = {
      narrow: ["L", "Ú", "B", "D", "K", "Č", "Č", "S", "Z", "Ř", "L", "P"],
      abbreviated: ["led", "úno", "bře", "dub", "kvě", "čvn", "čvc", "srp", "zář", "říj", "lis", "pro"],
      wide: ["ledna", "února", "března", "dubna", "května", "června", "července", "srpna", "září", "října", "listopadu", "prosince"]
    };
    const dayValues$p = {
      narrow: ["ne", "po", "út", "st", "čt", "pá", "so"],
      short: ["ne", "po", "út", "st", "čt", "pá", "so"],
      abbreviated: ["ned", "pon", "úte", "stř", "čtv", "pát", "sob"],
      wide: ["neděle", "pondělí", "úterý", "středa", "čtvrtek", "pátek", "sobota"]
    };
    const dayPeriodValues$p = {
      narrow: {
        am: "dop.",
        pm: "odp.",
        midnight: "půlnoc",
        noon: "poledne",
        morning: "ráno",
        afternoon: "odpoledne",
        evening: "večer",
        night: "noc"
      },
      abbreviated: {
        am: "dop.",
        pm: "odp.",
        midnight: "půlnoc",
        noon: "poledne",
        morning: "ráno",
        afternoon: "odpoledne",
        evening: "večer",
        night: "noc"
      },
      wide: {
        am: "dopoledne",
        pm: "odpoledne",
        midnight: "půlnoc",
        noon: "poledne",
        morning: "ráno",
        afternoon: "odpoledne",
        evening: "večer",
        night: "noc"
      }
    };
    const formattingDayPeriodValues$k = {
      narrow: {
        am: "dop.",
        pm: "odp.",
        midnight: "půlnoc",
        noon: "poledne",
        morning: "ráno",
        afternoon: "odpoledne",
        evening: "večer",
        night: "noc"
      },
      abbreviated: {
        am: "dop.",
        pm: "odp.",
        midnight: "půlnoc",
        noon: "poledne",
        morning: "ráno",
        afternoon: "odpoledne",
        evening: "večer",
        night: "noc"
      },
      wide: {
        am: "dopoledne",
        pm: "odpoledne",
        midnight: "půlnoc",
        noon: "poledne",
        morning: "ráno",
        afternoon: "odpoledne",
        evening: "večer",
        night: "noc"
      }
    };
    const ordinalNumber$p = (dirtyNumber, _options) => {
      const number = Number(dirtyNumber);
      return number + ".";
    };
    const localize$p = {
      ordinalNumber: ordinalNumber$p,
      era: buildLocalizeFn({
        values: eraValues$p,
        defaultWidth: "wide"
      }),
      quarter: buildLocalizeFn({
        values: quarterValues$p,
        defaultWidth: "wide",
        argumentCallback: quarter => quarter - 1
      }),
      month: buildLocalizeFn({
        values: monthValues$p,
        defaultWidth: "wide",
        formattingValues: formattingMonthValues$5,
        defaultFormattingWidth: "wide"
      }),
      day: buildLocalizeFn({
        values: dayValues$p,
        defaultWidth: "wide"
      }),
      dayPeriod: buildLocalizeFn({
        values: dayPeriodValues$p,
        defaultWidth: "wide",
        formattingValues: formattingDayPeriodValues$k,
        defaultFormattingWidth: "wide"
      })
    };

    const matchOrdinalNumberPattern$p = /^(\d+)\.?/i;
    const parseOrdinalNumberPattern$p = /\d+/i;
    const matchEraPatterns$p = {
      narrow: /^(p[řr](\.|ed) Kr\.|p[řr](\.|ed) n\. l\.|po Kr\.|n\. l\.)/i,
      abbreviated: /^(p[řr](\.|ed) Kr\.|p[řr](\.|ed) n\. l\.|po Kr\.|n\. l\.)/i,
      wide: /^(p[řr](\.|ed) Kristem|p[řr](\.|ed) na[šs][íi]m letopo[čc]tem|po Kristu|na[šs]eho letopo[čc]tu)/i
    };
    const parseEraPatterns$p = {
      any: [/^p[řr]/i, /^(po|n)/i]
    };
    const matchQuarterPatterns$p = {
      narrow: /^[1234]/i,
      abbreviated: /^[1234]\. [čc]tvrtlet[íi]/i,
      wide: /^[1234]\. [čc]tvrtlet[íi]/i
    };
    const parseQuarterPatterns$p = {
      any: [/1/i, /2/i, /3/i, /4/i]
    };
    const matchMonthPatterns$p = {
      narrow: /^[lúubdkčcszřrlp]/i,
      abbreviated: /^(led|[úu]no|b[řr]e|dub|kv[ěe]|[čc]vn|[čc]vc|srp|z[áa][řr]|[řr][íi]j|lis|pro)/i,
      wide: /^(leden|ledna|[úu]nora?|b[řr]ezen|b[řr]ezna|duben|dubna|kv[ěe]ten|kv[ěe]tna|[čc]erven(ec|ce)?|[čc]ervna|srpen|srpna|z[áa][řr][íi]|[řr][íi]jen|[řr][íi]jna|listopad(a|u)?|prosinec|prosince)/i
    };
    const parseMonthPatterns$p = {
      narrow: [/^l/i, /^[úu]/i, /^b/i, /^d/i, /^k/i, /^[čc]/i, /^[čc]/i, /^s/i, /^z/i, /^[řr]/i, /^l/i, /^p/i],
      any: [/^led/i, /^[úu]n/i, /^b[řr]e/i, /^dub/i, /^kv[ěe]/i, /^[čc]vn|[čc]erven(?!\w)|[čc]ervna/i, /^[čc]vc|[čc]erven(ec|ce)/i, /^srp/i, /^z[áa][řr]/i, /^[řr][íi]j/i, /^lis/i, /^pro/i]
    };
    const matchDayPatterns$p = {
      narrow: /^[npuúsčps]/i,
      short: /^(ne|po|[úu]t|st|[čc]t|p[áa]|so)/i,
      abbreviated: /^(ned|pon|[úu]te|st[rř]|[čc]tv|p[áa]t|sob)/i,
      wide: /^(ned[ěe]le|pond[ěe]l[íi]|[úu]ter[ýy]|st[řr]eda|[čc]tvrtek|p[áa]tek|sobota)/i
    };
    const parseDayPatterns$p = {
      narrow: [/^n/i, /^p/i, /^[úu]/i, /^s/i, /^[čc]/i, /^p/i, /^s/i],
      any: [/^ne/i, /^po/i, /^[úu]t/i, /^st/i, /^[čc]t/i, /^p[áa]/i, /^so/i]
    };
    const matchDayPeriodPatterns$p = {
      any: /^dopoledne|dop\.?|odpoledne|odp\.?|p[ůu]lnoc|poledne|r[áa]no|odpoledne|ve[čc]er|(v )?noci?/i
    };
    const parseDayPeriodPatterns$p = {
      any: {
        am: /^dop/i,
        pm: /^odp/i,
        midnight: /^p[ůu]lnoc/i,
        noon: /^poledne/i,
        morning: /r[áa]no/i,
        afternoon: /odpoledne/i,
        evening: /ve[čc]er/i,
        night: /noc/i
      }
    };
    const match$p = {
      ordinalNumber: buildMatchPatternFn({
        matchPattern: matchOrdinalNumberPattern$p,
        parsePattern: parseOrdinalNumberPattern$p,
        valueCallback: value => parseInt(value, 10)
      }),
      era: buildMatchFn({
        matchPatterns: matchEraPatterns$p,
        defaultMatchWidth: "wide",
        parsePatterns: parseEraPatterns$p,
        defaultParseWidth: "any"
      }),
      quarter: buildMatchFn({
        matchPatterns: matchQuarterPatterns$p,
        defaultMatchWidth: "wide",
        parsePatterns: parseQuarterPatterns$p,
        defaultParseWidth: "any",
        valueCallback: index => index + 1
      }),
      month: buildMatchFn({
        matchPatterns: matchMonthPatterns$p,
        defaultMatchWidth: "wide",
        parsePatterns: parseMonthPatterns$p,
        defaultParseWidth: "any"
      }),
      day: buildMatchFn({
        matchPatterns: matchDayPatterns$p,
        defaultMatchWidth: "wide",
        parsePatterns: parseDayPatterns$p,
        defaultParseWidth: "any"
      }),
      dayPeriod: buildMatchFn({
        matchPatterns: matchDayPeriodPatterns$p,
        defaultMatchWidth: "any",
        parsePatterns: parseDayPeriodPatterns$p,
        defaultParseWidth: "any"
      })
    };

    /**
     * @category Locales
     * @summary Czech locale.
     * @language Czech
     * @iso-639-2 ces
     * @author David Rus [@davidrus](https://github.com/davidrus)
     * @author Pavel Hrách [@SilenY](https://github.com/SilenY)
     * @author Jozef Bíroš [@JozefBiros](https://github.com/JozefBiros)
     */
    const cs = {
      code: "cs",
      formatDistance: formatDistance$p,
      formatLong: formatLong$q,
      formatRelative: formatRelative$p,
      localize: localize$p,
      match: match$p,
      options: {
        weekStartsOn: 1 /* Monday */,
        firstWeekContainsDate: 4
      }
    };

    const formatDistanceLocale$o = {
      lessThanXSeconds: {
        standalone: {
          one: "weniger als 1 Sekunde",
          other: "weniger als {{count}} Sekunden"
        },
        withPreposition: {
          one: "weniger als 1 Sekunde",
          other: "weniger als {{count}} Sekunden"
        }
      },
      xSeconds: {
        standalone: {
          one: "1 Sekunde",
          other: "{{count}} Sekunden"
        },
        withPreposition: {
          one: "1 Sekunde",
          other: "{{count}} Sekunden"
        }
      },
      halfAMinute: {
        standalone: "eine halbe Minute",
        withPreposition: "einer halben Minute"
      },
      lessThanXMinutes: {
        standalone: {
          one: "weniger als 1 Minute",
          other: "weniger als {{count}} Minuten"
        },
        withPreposition: {
          one: "weniger als 1 Minute",
          other: "weniger als {{count}} Minuten"
        }
      },
      xMinutes: {
        standalone: {
          one: "1 Minute",
          other: "{{count}} Minuten"
        },
        withPreposition: {
          one: "1 Minute",
          other: "{{count}} Minuten"
        }
      },
      aboutXHours: {
        standalone: {
          one: "etwa 1 Stunde",
          other: "etwa {{count}} Stunden"
        },
        withPreposition: {
          one: "etwa 1 Stunde",
          other: "etwa {{count}} Stunden"
        }
      },
      xHours: {
        standalone: {
          one: "1 Stunde",
          other: "{{count}} Stunden"
        },
        withPreposition: {
          one: "1 Stunde",
          other: "{{count}} Stunden"
        }
      },
      xDays: {
        standalone: {
          one: "1 Tag",
          other: "{{count}} Tage"
        },
        withPreposition: {
          one: "1 Tag",
          other: "{{count}} Tagen"
        }
      },
      aboutXWeeks: {
        standalone: {
          one: "etwa 1 Woche",
          other: "etwa {{count}} Wochen"
        },
        withPreposition: {
          one: "etwa 1 Woche",
          other: "etwa {{count}} Wochen"
        }
      },
      xWeeks: {
        standalone: {
          one: "1 Woche",
          other: "{{count}} Wochen"
        },
        withPreposition: {
          one: "1 Woche",
          other: "{{count}} Wochen"
        }
      },
      aboutXMonths: {
        standalone: {
          one: "etwa 1 Monat",
          other: "etwa {{count}} Monate"
        },
        withPreposition: {
          one: "etwa 1 Monat",
          other: "etwa {{count}} Monaten"
        }
      },
      xMonths: {
        standalone: {
          one: "1 Monat",
          other: "{{count}} Monate"
        },
        withPreposition: {
          one: "1 Monat",
          other: "{{count}} Monaten"
        }
      },
      aboutXYears: {
        standalone: {
          one: "etwa 1 Jahr",
          other: "etwa {{count}} Jahre"
        },
        withPreposition: {
          one: "etwa 1 Jahr",
          other: "etwa {{count}} Jahren"
        }
      },
      xYears: {
        standalone: {
          one: "1 Jahr",
          other: "{{count}} Jahre"
        },
        withPreposition: {
          one: "1 Jahr",
          other: "{{count}} Jahren"
        }
      },
      overXYears: {
        standalone: {
          one: "mehr als 1 Jahr",
          other: "mehr als {{count}} Jahre"
        },
        withPreposition: {
          one: "mehr als 1 Jahr",
          other: "mehr als {{count}} Jahren"
        }
      },
      almostXYears: {
        standalone: {
          one: "fast 1 Jahr",
          other: "fast {{count}} Jahre"
        },
        withPreposition: {
          one: "fast 1 Jahr",
          other: "fast {{count}} Jahren"
        }
      }
    };
    const formatDistance$o = (token, count, options) => {
      let result;
      const tokenValue = options?.addSuffix ? formatDistanceLocale$o[token].withPreposition : formatDistanceLocale$o[token].standalone;
      if (typeof tokenValue === "string") {
        result = tokenValue;
      } else if (count === 1) {
        result = tokenValue.one;
      } else {
        result = tokenValue.other.replace("{{count}}", String(count));
      }
      if (options?.addSuffix) {
        if (options.comparison && options.comparison > 0) {
          return "in " + result;
        } else {
          return "vor " + result;
        }
      }
      return result;
    };

    // DIN 5008: https://de.wikipedia.org/wiki/Datumsformat#DIN_5008
    const dateFormats$p = {
      full: "EEEE, do MMMM y",
      // Montag, 7. Januar 2018
      long: "do MMMM y",
      // 7. Januar 2018
      medium: "do MMM y",
      // 7. Jan. 2018
      short: "dd.MM.y" // 07.01.2018
    };
    const timeFormats$p = {
      full: "HH:mm:ss zzzz",
      long: "HH:mm:ss z",
      medium: "HH:mm:ss",
      short: "HH:mm"
    };
    const dateTimeFormats$p = {
      full: "{{date}} 'um' {{time}}",
      long: "{{date}} 'um' {{time}}",
      medium: "{{date}} {{time}}",
      short: "{{date}} {{time}}"
    };
    const formatLong$p = {
      date: buildFormatLongFn({
        formats: dateFormats$p,
        defaultWidth: "full"
      }),
      time: buildFormatLongFn({
        formats: timeFormats$p,
        defaultWidth: "full"
      }),
      dateTime: buildFormatLongFn({
        formats: dateTimeFormats$p,
        defaultWidth: "full"
      })
    };

    const formatRelativeLocale$o = {
      lastWeek: "'letzten' eeee 'um' p",
      yesterday: "'gestern um' p",
      today: "'heute um' p",
      tomorrow: "'morgen um' p",
      nextWeek: "eeee 'um' p",
      other: "P"
    };
    const formatRelative$o = (token, _date, _baseDate, _options) => formatRelativeLocale$o[token];

    const eraValues$o = {
      narrow: ["v.Chr.", "n.Chr."],
      abbreviated: ["v.Chr.", "n.Chr."],
      wide: ["vor Christus", "nach Christus"]
    };
    const quarterValues$o = {
      narrow: ["1", "2", "3", "4"],
      abbreviated: ["Q1", "Q2", "Q3", "Q4"],
      wide: ["1. Quartal", "2. Quartal", "3. Quartal", "4. Quartal"]
    };

    // Note: in German, the names of days of the week and months are capitalized.
    // If you are making a new locale based on this one, check if the same is true for the language you're working on.
    // Generally, formatted dates should look like they are in the middle of a sentence,
    // e.g. in Spanish language the weekdays and months should be in the lowercase.
    const monthValues$o = {
      narrow: ["J", "F", "M", "A", "M", "J", "J", "A", "S", "O", "N", "D"],
      abbreviated: ["Jan", "Feb", "Mär", "Apr", "Mai", "Jun", "Jul", "Aug", "Sep", "Okt", "Nov", "Dez"],
      wide: ["Januar", "Februar", "März", "April", "Mai", "Juni", "Juli", "August", "September", "Oktober", "November", "Dezember"]
    };

    // https://st.unicode.org/cldr-apps/v#/de/Gregorian/
    const formattingMonthValues$4 = {
      narrow: monthValues$o.narrow,
      abbreviated: ["Jan.", "Feb.", "März", "Apr.", "Mai", "Juni", "Juli", "Aug.", "Sep.", "Okt.", "Nov.", "Dez."],
      wide: monthValues$o.wide
    };
    const dayValues$o = {
      narrow: ["S", "M", "D", "M", "D", "F", "S"],
      short: ["So", "Mo", "Di", "Mi", "Do", "Fr", "Sa"],
      abbreviated: ["So.", "Mo.", "Di.", "Mi.", "Do.", "Fr.", "Sa."],
      wide: ["Sonntag", "Montag", "Dienstag", "Mittwoch", "Donnerstag", "Freitag", "Samstag"]
    };

    // https://www.unicode.org/cldr/charts/32/summary/de.html#1881
    const dayPeriodValues$o = {
      narrow: {
        am: "vm.",
        pm: "nm.",
        midnight: "Mitternacht",
        noon: "Mittag",
        morning: "Morgen",
        afternoon: "Nachm.",
        evening: "Abend",
        night: "Nacht"
      },
      abbreviated: {
        am: "vorm.",
        pm: "nachm.",
        midnight: "Mitternacht",
        noon: "Mittag",
        morning: "Morgen",
        afternoon: "Nachmittag",
        evening: "Abend",
        night: "Nacht"
      },
      wide: {
        am: "vormittags",
        pm: "nachmittags",
        midnight: "Mitternacht",
        noon: "Mittag",
        morning: "Morgen",
        afternoon: "Nachmittag",
        evening: "Abend",
        night: "Nacht"
      }
    };
    const formattingDayPeriodValues$j = {
      narrow: {
        am: "vm.",
        pm: "nm.",
        midnight: "Mitternacht",
        noon: "Mittag",
        morning: "morgens",
        afternoon: "nachm.",
        evening: "abends",
        night: "nachts"
      },
      abbreviated: {
        am: "vorm.",
        pm: "nachm.",
        midnight: "Mitternacht",
        noon: "Mittag",
        morning: "morgens",
        afternoon: "nachmittags",
        evening: "abends",
        night: "nachts"
      },
      wide: {
        am: "vormittags",
        pm: "nachmittags",
        midnight: "Mitternacht",
        noon: "Mittag",
        morning: "morgens",
        afternoon: "nachmittags",
        evening: "abends",
        night: "nachts"
      }
    };
    const ordinalNumber$o = dirtyNumber => {
      const number = Number(dirtyNumber);
      return number + ".";
    };
    const localize$o = {
      ordinalNumber: ordinalNumber$o,
      era: buildLocalizeFn({
        values: eraValues$o,
        defaultWidth: "wide"
      }),
      quarter: buildLocalizeFn({
        values: quarterValues$o,
        defaultWidth: "wide",
        argumentCallback: quarter => quarter - 1
      }),
      month: buildLocalizeFn({
        values: monthValues$o,
        formattingValues: formattingMonthValues$4,
        defaultWidth: "wide"
      }),
      day: buildLocalizeFn({
        values: dayValues$o,
        defaultWidth: "wide"
      }),
      dayPeriod: buildLocalizeFn({
        values: dayPeriodValues$o,
        defaultWidth: "wide",
        formattingValues: formattingDayPeriodValues$j,
        defaultFormattingWidth: "wide"
      })
    };

    const matchOrdinalNumberPattern$o = /^(\d+)(\.)?/i;
    const parseOrdinalNumberPattern$o = /\d+/i;
    const matchEraPatterns$o = {
      narrow: /^(v\.? ?Chr\.?|n\.? ?Chr\.?)/i,
      abbreviated: /^(v\.? ?Chr\.?|n\.? ?Chr\.?)/i,
      wide: /^(vor Christus|vor unserer Zeitrechnung|nach Christus|unserer Zeitrechnung)/i
    };
    const parseEraPatterns$o = {
      any: [/^v/i, /^n/i]
    };
    const matchQuarterPatterns$o = {
      narrow: /^[1234]/i,
      abbreviated: /^q[1234]/i,
      wide: /^[1234](\.)? Quartal/i
    };
    const parseQuarterPatterns$o = {
      any: [/1/i, /2/i, /3/i, /4/i]
    };
    const matchMonthPatterns$o = {
      narrow: /^[jfmasond]/i,
      abbreviated: /^(j[aä]n|feb|mär[z]?|apr|mai|jun[i]?|jul[i]?|aug|sep|okt|nov|dez)\.?/i,
      wide: /^(januar|februar|märz|april|mai|juni|juli|august|september|oktober|november|dezember)/i
    };
    const parseMonthPatterns$o = {
      narrow: [/^j/i, /^f/i, /^m/i, /^a/i, /^m/i, /^j/i, /^j/i, /^a/i, /^s/i, /^o/i, /^n/i, /^d/i],
      any: [/^j[aä]/i, /^f/i, /^mär/i, /^ap/i, /^mai/i, /^jun/i, /^jul/i, /^au/i, /^s/i, /^o/i, /^n/i, /^d/i]
    };
    const matchDayPatterns$o = {
      narrow: /^[smdmf]/i,
      short: /^(so|mo|di|mi|do|fr|sa)/i,
      abbreviated: /^(son?|mon?|die?|mit?|don?|fre?|sam?)\.?/i,
      wide: /^(sonntag|montag|dienstag|mittwoch|donnerstag|freitag|samstag)/i
    };
    const parseDayPatterns$o = {
      any: [/^so/i, /^mo/i, /^di/i, /^mi/i, /^do/i, /^f/i, /^sa/i]
    };
    const matchDayPeriodPatterns$o = {
      narrow: /^(vm\.?|nm\.?|Mitternacht|Mittag|morgens|nachm\.?|abends|nachts)/i,
      abbreviated: /^(vorm\.?|nachm\.?|Mitternacht|Mittag|morgens|nachm\.?|abends|nachts)/i,
      wide: /^(vormittags|nachmittags|Mitternacht|Mittag|morgens|nachmittags|abends|nachts)/i
    };
    const parseDayPeriodPatterns$o = {
      any: {
        am: /^v/i,
        pm: /^n/i,
        midnight: /^Mitte/i,
        noon: /^Mitta/i,
        morning: /morgens/i,
        afternoon: /nachmittags/i,
        // will never be matched. Afternoon is matched by `pm`
        evening: /abends/i,
        night: /nachts/i // will never be matched. Night is matched by `pm`
      }
    };
    const match$o = {
      ordinalNumber: buildMatchPatternFn({
        matchPattern: matchOrdinalNumberPattern$o,
        parsePattern: parseOrdinalNumberPattern$o,
        valueCallback: value => parseInt(value)
      }),
      era: buildMatchFn({
        matchPatterns: matchEraPatterns$o,
        defaultMatchWidth: "wide",
        parsePatterns: parseEraPatterns$o,
        defaultParseWidth: "any"
      }),
      quarter: buildMatchFn({
        matchPatterns: matchQuarterPatterns$o,
        defaultMatchWidth: "wide",
        parsePatterns: parseQuarterPatterns$o,
        defaultParseWidth: "any",
        valueCallback: index => index + 1
      }),
      month: buildMatchFn({
        matchPatterns: matchMonthPatterns$o,
        defaultMatchWidth: "wide",
        parsePatterns: parseMonthPatterns$o,
        defaultParseWidth: "any"
      }),
      day: buildMatchFn({
        matchPatterns: matchDayPatterns$o,
        defaultMatchWidth: "wide",
        parsePatterns: parseDayPatterns$o,
        defaultParseWidth: "any"
      }),
      dayPeriod: buildMatchFn({
        matchPatterns: matchDayPeriodPatterns$o,
        defaultMatchWidth: "wide",
        parsePatterns: parseDayPeriodPatterns$o,
        defaultParseWidth: "any"
      })
    };

    /**
     * @category Locales
     * @summary German locale.
     * @language German
     * @iso-639-2 deu
     * @author Thomas Eilmsteiner [@DeMuu](https://github.com/DeMuu)
     * @author Asia [@asia-t](https://github.com/asia-t)
     * @author Van Vuong Ngo [@vanvuongngo](https://github.com/vanvuongngo)
     * @author RomanErnst [@pex](https://github.com/pex)
     * @author Philipp Keck [@Philipp91](https://github.com/Philipp91)
     */
    const de = {
      code: "de",
      formatDistance: formatDistance$o,
      formatLong: formatLong$p,
      formatRelative: formatRelative$o,
      localize: localize$o,
      match: match$o,
      options: {
        weekStartsOn: 1 /* Monday */,
        firstWeekContainsDate: 4
      }
    };

    const formatDistanceLocale$n = {
      lessThanXSeconds: {
        one: "less than a second",
        other: "less than {{count}} seconds"
      },
      xSeconds: {
        one: "1 second",
        other: "{{count}} seconds"
      },
      halfAMinute: "half a minute",
      lessThanXMinutes: {
        one: "less than a minute",
        other: "less than {{count}} minutes"
      },
      xMinutes: {
        one: "1 minute",
        other: "{{count}} minutes"
      },
      aboutXHours: {
        one: "about 1 hour",
        other: "about {{count}} hours"
      },
      xHours: {
        one: "1 hour",
        other: "{{count}} hours"
      },
      xDays: {
        one: "1 day",
        other: "{{count}} days"
      },
      aboutXWeeks: {
        one: "about 1 week",
        other: "about {{count}} weeks"
      },
      xWeeks: {
        one: "1 week",
        other: "{{count}} weeks"
      },
      aboutXMonths: {
        one: "about 1 month",
        other: "about {{count}} months"
      },
      xMonths: {
        one: "1 month",
        other: "{{count}} months"
      },
      aboutXYears: {
        one: "about 1 year",
        other: "about {{count}} years"
      },
      xYears: {
        one: "1 year",
        other: "{{count}} years"
      },
      overXYears: {
        one: "over 1 year",
        other: "over {{count}} years"
      },
      almostXYears: {
        one: "almost 1 year",
        other: "almost {{count}} years"
      }
    };
    const formatDistance$n = (token, count, options) => {
      let result;
      const tokenValue = formatDistanceLocale$n[token];
      if (typeof tokenValue === "string") {
        result = tokenValue;
      } else if (count === 1) {
        result = tokenValue.one;
      } else {
        result = tokenValue.other.replace("{{count}}", count.toString());
      }
      if (options?.addSuffix) {
        if (options.comparison && options.comparison > 0) {
          return "in " + result;
        } else {
          return result + " ago";
        }
      }
      return result;
    };

    const formatRelativeLocale$n = {
      lastWeek: "'last' eeee 'at' p",
      yesterday: "'yesterday at' p",
      today: "'today at' p",
      tomorrow: "'tomorrow at' p",
      nextWeek: "eeee 'at' p",
      other: "P"
    };
    const formatRelative$n = (token, _date, _baseDate, _options) => formatRelativeLocale$n[token];

    const eraValues$n = {
      narrow: ["B", "A"],
      abbreviated: ["BC", "AD"],
      wide: ["Before Christ", "Anno Domini"]
    };
    const quarterValues$n = {
      narrow: ["1", "2", "3", "4"],
      abbreviated: ["Q1", "Q2", "Q3", "Q4"],
      wide: ["1st quarter", "2nd quarter", "3rd quarter", "4th quarter"]
    };

    // Note: in English, the names of days of the week and months are capitalized.
    // If you are making a new locale based on this one, check if the same is true for the language you're working on.
    // Generally, formatted dates should look like they are in the middle of a sentence,
    // e.g. in Spanish language the weekdays and months should be in the lowercase.
    const monthValues$n = {
      narrow: ["J", "F", "M", "A", "M", "J", "J", "A", "S", "O", "N", "D"],
      abbreviated: ["Jan", "Feb", "Mar", "Apr", "May", "Jun", "Jul", "Aug", "Sep", "Oct", "Nov", "Dec"],
      wide: ["January", "February", "March", "April", "May", "June", "July", "August", "September", "October", "November", "December"]
    };
    const dayValues$n = {
      narrow: ["S", "M", "T", "W", "T", "F", "S"],
      short: ["Su", "Mo", "Tu", "We", "Th", "Fr", "Sa"],
      abbreviated: ["Sun", "Mon", "Tue", "Wed", "Thu", "Fri", "Sat"],
      wide: ["Sunday", "Monday", "Tuesday", "Wednesday", "Thursday", "Friday", "Saturday"]
    };
    const dayPeriodValues$n = {
      narrow: {
        am: "a",
        pm: "p",
        midnight: "mi",
        noon: "n",
        morning: "morning",
        afternoon: "afternoon",
        evening: "evening",
        night: "night"
      },
      abbreviated: {
        am: "AM",
        pm: "PM",
        midnight: "midnight",
        noon: "noon",
        morning: "morning",
        afternoon: "afternoon",
        evening: "evening",
        night: "night"
      },
      wide: {
        am: "a.m.",
        pm: "p.m.",
        midnight: "midnight",
        noon: "noon",
        morning: "morning",
        afternoon: "afternoon",
        evening: "evening",
        night: "night"
      }
    };
    const formattingDayPeriodValues$i = {
      narrow: {
        am: "a",
        pm: "p",
        midnight: "mi",
        noon: "n",
        morning: "in the morning",
        afternoon: "in the afternoon",
        evening: "in the evening",
        night: "at night"
      },
      abbreviated: {
        am: "AM",
        pm: "PM",
        midnight: "midnight",
        noon: "noon",
        morning: "in the morning",
        afternoon: "in the afternoon",
        evening: "in the evening",
        night: "at night"
      },
      wide: {
        am: "a.m.",
        pm: "p.m.",
        midnight: "midnight",
        noon: "noon",
        morning: "in the morning",
        afternoon: "in the afternoon",
        evening: "in the evening",
        night: "at night"
      }
    };
    const ordinalNumber$n = (dirtyNumber, _options) => {
      const number = Number(dirtyNumber);

      // If ordinal numbers depend on context, for example,
      // if they are different for different grammatical genders,
      // use `options.unit`.
      //
      // `unit` can be 'year', 'quarter', 'month', 'week', 'date', 'dayOfYear',
      // 'day', 'hour', 'minute', 'second'.

      const rem100 = number % 100;
      if (rem100 > 20 || rem100 < 10) {
        switch (rem100 % 10) {
          case 1:
            return number + "st";
          case 2:
            return number + "nd";
          case 3:
            return number + "rd";
        }
      }
      return number + "th";
    };
    const localize$n = {
      ordinalNumber: ordinalNumber$n,
      era: buildLocalizeFn({
        values: eraValues$n,
        defaultWidth: "wide"
      }),
      quarter: buildLocalizeFn({
        values: quarterValues$n,
        defaultWidth: "wide",
        argumentCallback: quarter => quarter - 1
      }),
      month: buildLocalizeFn({
        values: monthValues$n,
        defaultWidth: "wide"
      }),
      day: buildLocalizeFn({
        values: dayValues$n,
        defaultWidth: "wide"
      }),
      dayPeriod: buildLocalizeFn({
        values: dayPeriodValues$n,
        defaultWidth: "wide",
        formattingValues: formattingDayPeriodValues$i,
        defaultFormattingWidth: "wide"
      })
    };

    const matchOrdinalNumberPattern$n = /^(\d+)(th|st|nd|rd)?/i;
    const parseOrdinalNumberPattern$n = /\d+/i;
    const matchEraPatterns$n = {
      narrow: /^(b|a)/i,
      abbreviated: /^(b\.?\s?c\.?|b\.?\s?c\.?\s?e\.?|a\.?\s?d\.?|c\.?\s?e\.?)/i,
      wide: /^(before christ|before common era|anno domini|common era)/i
    };
    const parseEraPatterns$n = {
      any: [/^b/i, /^(a|c)/i]
    };
    const matchQuarterPatterns$n = {
      narrow: /^[1234]/i,
      abbreviated: /^q[1234]/i,
      wide: /^[1234](th|st|nd|rd)? quarter/i
    };
    const parseQuarterPatterns$n = {
      any: [/1/i, /2/i, /3/i, /4/i]
    };
    const matchMonthPatterns$n = {
      narrow: /^[jfmasond]/i,
      abbreviated: /^(jan|feb|mar|apr|may|jun|jul|aug|sep|oct|nov|dec)/i,
      wide: /^(january|february|march|april|may|june|july|august|september|october|november|december)/i
    };
    const parseMonthPatterns$n = {
      narrow: [/^j/i, /^f/i, /^m/i, /^a/i, /^m/i, /^j/i, /^j/i, /^a/i, /^s/i, /^o/i, /^n/i, /^d/i],
      any: [/^ja/i, /^f/i, /^mar/i, /^ap/i, /^may/i, /^jun/i, /^jul/i, /^au/i, /^s/i, /^o/i, /^n/i, /^d/i]
    };
    const matchDayPatterns$n = {
      narrow: /^[smtwf]/i,
      short: /^(su|mo|tu|we|th|fr|sa)/i,
      abbreviated: /^(sun|mon|tue|wed|thu|fri|sat)/i,
      wide: /^(sunday|monday|tuesday|wednesday|thursday|friday|saturday)/i
    };
    const parseDayPatterns$n = {
      narrow: [/^s/i, /^m/i, /^t/i, /^w/i, /^t/i, /^f/i, /^s/i],
      any: [/^su/i, /^m/i, /^tu/i, /^w/i, /^th/i, /^f/i, /^sa/i]
    };
    const matchDayPeriodPatterns$n = {
      narrow: /^(a|p|mi|n|(in the|at) (morning|afternoon|evening|night))/i,
      any: /^([ap]\.?\s?m\.?|midnight|noon|(in the|at) (morning|afternoon|evening|night))/i
    };
    const parseDayPeriodPatterns$n = {
      any: {
        am: /^a/i,
        pm: /^p/i,
        midnight: /^mi/i,
        noon: /^no/i,
        morning: /morning/i,
        afternoon: /afternoon/i,
        evening: /evening/i,
        night: /night/i
      }
    };
    const match$n = {
      ordinalNumber: buildMatchPatternFn({
        matchPattern: matchOrdinalNumberPattern$n,
        parsePattern: parseOrdinalNumberPattern$n,
        valueCallback: value => parseInt(value, 10)
      }),
      era: buildMatchFn({
        matchPatterns: matchEraPatterns$n,
        defaultMatchWidth: "wide",
        parsePatterns: parseEraPatterns$n,
        defaultParseWidth: "any"
      }),
      quarter: buildMatchFn({
        matchPatterns: matchQuarterPatterns$n,
        defaultMatchWidth: "wide",
        parsePatterns: parseQuarterPatterns$n,
        defaultParseWidth: "any",
        valueCallback: index => index + 1
      }),
      month: buildMatchFn({
        matchPatterns: matchMonthPatterns$n,
        defaultMatchWidth: "wide",
        parsePatterns: parseMonthPatterns$n,
        defaultParseWidth: "any"
      }),
      day: buildMatchFn({
        matchPatterns: matchDayPatterns$n,
        defaultMatchWidth: "wide",
        parsePatterns: parseDayPatterns$n,
        defaultParseWidth: "any"
      }),
      dayPeriod: buildMatchFn({
        matchPatterns: matchDayPeriodPatterns$n,
        defaultMatchWidth: "any",
        parsePatterns: parseDayPeriodPatterns$n,
        defaultParseWidth: "any"
      })
    };

    const dateFormats$o = {
      full: "EEEE, d MMMM yyyy",
      long: "d MMMM yyyy",
      medium: "d MMM yyyy",
      short: "dd/MM/yyyy"
    };
    const timeFormats$o = {
      full: "HH:mm:ss zzzz",
      long: "HH:mm:ss z",
      medium: "HH:mm:ss",
      short: "HH:mm"
    };
    const dateTimeFormats$o = {
      full: "{{date}} 'at' {{time}}",
      long: "{{date}} 'at' {{time}}",
      medium: "{{date}}, {{time}}",
      short: "{{date}}, {{time}}"
    };
    const formatLong$o = {
      date: buildFormatLongFn({
        formats: dateFormats$o,
        defaultWidth: "full"
      }),
      time: buildFormatLongFn({
        formats: timeFormats$o,
        defaultWidth: "full"
      }),
      dateTime: buildFormatLongFn({
        formats: dateTimeFormats$o,
        defaultWidth: "full"
      })
    };

    /**
     * @category Locales
     * @summary English locale (United Kingdom).
     * @language English
     * @iso-639-2 eng
     * @author Alex [@glintik](https://github.com/glintik)
     */
    const enGB = {
      code: "en-GB",
      formatDistance: formatDistance$n,
      formatLong: formatLong$o,
      formatRelative: formatRelative$n,
      localize: localize$n,
      match: match$n,
      options: {
        weekStartsOn: 1 /* Monday */,
        firstWeekContainsDate: 4
      }
    };

    const dateFormats$n = {
      full: "EEEE, MMMM do, y",
      long: "MMMM do, y",
      medium: "MMM d, y",
      short: "MM/dd/yyyy"
    };
    const timeFormats$n = {
      full: "h:mm:ss a zzzz",
      long: "h:mm:ss a z",
      medium: "h:mm:ss a",
      short: "h:mm a"
    };
    const dateTimeFormats$n = {
      full: "{{date}} 'at' {{time}}",
      long: "{{date}} 'at' {{time}}",
      medium: "{{date}}, {{time}}",
      short: "{{date}}, {{time}}"
    };
    const formatLong$n = {
      date: buildFormatLongFn({
        formats: dateFormats$n,
        defaultWidth: "full"
      }),
      time: buildFormatLongFn({
        formats: timeFormats$n,
        defaultWidth: "full"
      }),
      dateTime: buildFormatLongFn({
        formats: dateTimeFormats$n,
        defaultWidth: "full"
      })
    };

    /**
     * @category Locales
     * @summary English locale (United States).
     * @language English
     * @iso-639-2 eng
     * @author Sasha Koss [@kossnocorp](https://github.com/kossnocorp)
     * @author Lesha Koss [@leshakoss](https://github.com/leshakoss)
     */
    const enUS = {
      code: "en-US",
      formatDistance: formatDistance$n,
      formatLong: formatLong$n,
      formatRelative: formatRelative$n,
      localize: localize$n,
      match: match$n,
      options: {
        weekStartsOn: 0 /* Sunday */,
        firstWeekContainsDate: 1
      }
    };

    const formatDistanceLocale$m = {
      lessThanXSeconds: {
        one: "malpli ol sekundo",
        other: "malpli ol {{count}} sekundoj"
      },
      xSeconds: {
        one: "1 sekundo",
        other: "{{count}} sekundoj"
      },
      halfAMinute: "duonminuto",
      lessThanXMinutes: {
        one: "malpli ol minuto",
        other: "malpli ol {{count}} minutoj"
      },
      xMinutes: {
        one: "1 minuto",
        other: "{{count}} minutoj"
      },
      aboutXHours: {
        one: "proksimume 1 horo",
        other: "proksimume {{count}} horoj"
      },
      xHours: {
        one: "1 horo",
        other: "{{count}} horoj"
      },
      xDays: {
        one: "1 tago",
        other: "{{count}} tagoj"
      },
      aboutXMonths: {
        one: "proksimume 1 monato",
        other: "proksimume {{count}} monatoj"
      },
      xWeeks: {
        one: "1 semajno",
        other: "{{count}} semajnoj"
      },
      aboutXWeeks: {
        one: "proksimume 1 semajno",
        other: "proksimume {{count}} semajnoj"
      },
      xMonths: {
        one: "1 monato",
        other: "{{count}} monatoj"
      },
      aboutXYears: {
        one: "proksimume 1 jaro",
        other: "proksimume {{count}} jaroj"
      },
      xYears: {
        one: "1 jaro",
        other: "{{count}} jaroj"
      },
      overXYears: {
        one: "pli ol 1 jaro",
        other: "pli ol {{count}} jaroj"
      },
      almostXYears: {
        one: "preskaŭ 1 jaro",
        other: "preskaŭ {{count}} jaroj"
      }
    };
    const formatDistance$m = (token, count, options) => {
      let result;
      const tokenValue = formatDistanceLocale$m[token];
      if (typeof tokenValue === "string") {
        result = tokenValue;
      } else if (count === 1) {
        result = tokenValue.one;
      } else {
        result = tokenValue.other.replace("{{count}}", String(count));
      }
      if (options?.addSuffix) {
        if (options?.comparison && options.comparison > 0) {
          return "post " + result;
        } else {
          return "antaŭ " + result;
        }
      }
      return result;
    };

    const dateFormats$m = {
      full: "EEEE, do 'de' MMMM y",
      long: "y-MMMM-dd",
      medium: "y-MMM-dd",
      short: "yyyy-MM-dd"
    };
    const timeFormats$m = {
      full: "Ho 'horo kaj' m:ss zzzz",
      long: "HH:mm:ss z",
      medium: "HH:mm:ss",
      short: "HH:mm"
    };
    const dateTimeFormats$m = {
      any: "{{date}} {{time}}"
    };
    const formatLong$m = {
      date: buildFormatLongFn({
        formats: dateFormats$m,
        defaultWidth: "full"
      }),
      time: buildFormatLongFn({
        formats: timeFormats$m,
        defaultWidth: "full"
      }),
      dateTime: buildFormatLongFn({
        formats: dateTimeFormats$m,
        defaultWidth: "any"
      })
    };

    const formatRelativeLocale$m = {
      lastWeek: "'pasinta' eeee 'je' p",
      yesterday: "'hieraŭ je' p",
      today: "'hodiaŭ je' p",
      tomorrow: "'morgaŭ je' p",
      nextWeek: "eeee 'je' p",
      other: "P"
    };
    const formatRelative$m = (token, _date, _baseDate, _options) => formatRelativeLocale$m[token];

    const eraValues$m = {
      narrow: ["aK", "pK"],
      abbreviated: ["a.K.E.", "p.K.E."],
      wide: ["antaŭ Komuna Erao", "Komuna Erao"]
    };
    const quarterValues$m = {
      narrow: ["1", "2", "3", "4"],
      abbreviated: ["K1", "K2", "K3", "K4"],
      wide: ["1-a kvaronjaro", "2-a kvaronjaro", "3-a kvaronjaro", "4-a kvaronjaro"]
    };
    const monthValues$m = {
      narrow: ["J", "F", "M", "A", "M", "J", "J", "A", "S", "O", "N", "D"],
      abbreviated: ["jan", "feb", "mar", "apr", "maj", "jun", "jul", "aŭg", "sep", "okt", "nov", "dec"],
      wide: ["januaro", "februaro", "marto", "aprilo", "majo", "junio", "julio", "aŭgusto", "septembro", "oktobro", "novembro", "decembro"]
    };
    const dayValues$m = {
      narrow: ["D", "L", "M", "M", "Ĵ", "V", "S"],
      short: ["di", "lu", "ma", "me", "ĵa", "ve", "sa"],
      abbreviated: ["dim", "lun", "mar", "mer", "ĵaŭ", "ven", "sab"],
      wide: ["dimanĉo", "lundo", "mardo", "merkredo", "ĵaŭdo", "vendredo", "sabato"]
    };
    const dayPeriodValues$m = {
      narrow: {
        am: "a",
        pm: "p",
        midnight: "noktomezo",
        noon: "tagmezo",
        morning: "matene",
        afternoon: "posttagmeze",
        evening: "vespere",
        night: "nokte"
      },
      abbreviated: {
        am: "a.t.m.",
        pm: "p.t.m.",
        midnight: "noktomezo",
        noon: "tagmezo",
        morning: "matene",
        afternoon: "posttagmeze",
        evening: "vespere",
        night: "nokte"
      },
      wide: {
        am: "antaŭtagmeze",
        pm: "posttagmeze",
        midnight: "noktomezo",
        noon: "tagmezo",
        morning: "matene",
        afternoon: "posttagmeze",
        evening: "vespere",
        night: "nokte"
      }
    };
    const ordinalNumber$m = dirtyNumber => {
      const number = Number(dirtyNumber);
      return number + "-a";
    };
    const localize$m = {
      ordinalNumber: ordinalNumber$m,
      era: buildLocalizeFn({
        values: eraValues$m,
        defaultWidth: "wide"
      }),
      quarter: buildLocalizeFn({
        values: quarterValues$m,
        defaultWidth: "wide",
        argumentCallback: function (quarter) {
          return Number(quarter) - 1;
        }
      }),
      month: buildLocalizeFn({
        values: monthValues$m,
        defaultWidth: "wide"
      }),
      day: buildLocalizeFn({
        values: dayValues$m,
        defaultWidth: "wide"
      }),
      dayPeriod: buildLocalizeFn({
        values: dayPeriodValues$m,
        defaultWidth: "wide"
      })
    };

    const matchOrdinalNumberPattern$m = /^(\d+)(-?a)?/i;
    const parseOrdinalNumberPattern$m = /\d+/i;
    const matchEraPatterns$m = {
      narrow: /^([ap]k)/i,
      abbreviated: /^([ap]\.?\s?k\.?\s?e\.?)/i,
      wide: /^((antaǔ |post )?komuna erao)/i
    };
    const parseEraPatterns$m = {
      any: [/^a/i, /^[kp]/i]
    };
    const matchQuarterPatterns$m = {
      narrow: /^[1234]/i,
      abbreviated: /^k[1234]/i,
      wide: /^[1234](-?a)? kvaronjaro/i
    };
    const parseQuarterPatterns$m = {
      any: [/1/i, /2/i, /3/i, /4/i]
    };
    const matchMonthPatterns$m = {
      narrow: /^[jfmasond]/i,
      abbreviated: /^(jan|feb|mar|apr|maj|jun|jul|a(ŭ|ux|uh|u)g|sep|okt|nov|dec)/i,
      wide: /^(januaro|februaro|marto|aprilo|majo|junio|julio|a(ŭ|ux|uh|u)gusto|septembro|oktobro|novembro|decembro)/i
    };
    const parseMonthPatterns$m = {
      narrow: [/^j/i, /^f/i, /^m/i, /^a/i, /^m/i, /^j/i, /^j/i, /^a/i, /^s/i, /^o/i, /^n/i, /^d/i],
      any: [/^ja/i, /^f/i, /^mar/i, /^ap/i, /^maj/i, /^jun/i, /^jul/i, /^a(u|ŭ)/i, /^s/i, /^o/i, /^n/i, /^d/i]
    };
    const matchDayPatterns$m = {
      narrow: /^[dlmĵjvs]/i,
      short: /^(di|lu|ma|me|(ĵ|jx|jh|j)a|ve|sa)/i,
      abbreviated: /^(dim|lun|mar|mer|(ĵ|jx|jh|j)a(ŭ|ux|uh|u)|ven|sab)/i,
      wide: /^(diman(ĉ|cx|ch|c)o|lundo|mardo|merkredo|(ĵ|jx|jh|j)a(ŭ|ux|uh|u)do|vendredo|sabato)/i
    };
    const parseDayPatterns$m = {
      narrow: [/^d/i, /^l/i, /^m/i, /^m/i, /^(j|ĵ)/i, /^v/i, /^s/i],
      any: [/^d/i, /^l/i, /^ma/i, /^me/i, /^(j|ĵ)/i, /^v/i, /^s/i]
    };
    const matchDayPeriodPatterns$m = {
      narrow: /^([ap]|(posttagmez|noktomez|tagmez|maten|vesper|nokt)[eo])/i,
      abbreviated: /^([ap][.\s]?t[.\s]?m[.\s]?|(posttagmez|noktomez|tagmez|maten|vesper|nokt)[eo])/i,
      wide: /^(anta(ŭ|ux)tagmez|posttagmez|noktomez|tagmez|maten|vesper|nokt)[eo]/i
    };
    const parseDayPeriodPatterns$m = {
      any: {
        am: /^a/i,
        pm: /^p/i,
        midnight: /^noktom/i,
        noon: /^t/i,
        morning: /^m/i,
        afternoon: /^posttagmeze/i,
        evening: /^v/i,
        night: /^n/i
      }
    };
    const match$m = {
      ordinalNumber: buildMatchPatternFn({
        matchPattern: matchOrdinalNumberPattern$m,
        parsePattern: parseOrdinalNumberPattern$m,
        valueCallback: function (value) {
          return parseInt(value, 10);
        }
      }),
      era: buildMatchFn({
        matchPatterns: matchEraPatterns$m,
        defaultMatchWidth: "wide",
        parsePatterns: parseEraPatterns$m,
        defaultParseWidth: "any"
      }),
      quarter: buildMatchFn({
        matchPatterns: matchQuarterPatterns$m,
        defaultMatchWidth: "wide",
        parsePatterns: parseQuarterPatterns$m,
        defaultParseWidth: "any",
        valueCallback: function (index) {
          return index + 1;
        }
      }),
      month: buildMatchFn({
        matchPatterns: matchMonthPatterns$m,
        defaultMatchWidth: "wide",
        parsePatterns: parseMonthPatterns$m,
        defaultParseWidth: "any"
      }),
      day: buildMatchFn({
        matchPatterns: matchDayPatterns$m,
        defaultMatchWidth: "wide",
        parsePatterns: parseDayPatterns$m,
        defaultParseWidth: "any"
      }),
      dayPeriod: buildMatchFn({
        matchPatterns: matchDayPeriodPatterns$m,
        defaultMatchWidth: "wide",
        parsePatterns: parseDayPeriodPatterns$m,
        defaultParseWidth: "any"
      })
    };

    /**
     * @category Locales
     * @summary Esperanto locale.
     * @language Esperanto
     * @iso-639-2 epo
     * @author date-fns
     */
    const eo = {
      code: "eo",
      formatDistance: formatDistance$m,
      formatLong: formatLong$m,
      formatRelative: formatRelative$m,
      localize: localize$m,
      match: match$m,
      options: {
        weekStartsOn: 1 /* Monday */,
        firstWeekContainsDate: 4
      }
    };

    const formatDistanceLocale$l = {
      lessThanXSeconds: {
        one: "menos de un segundo",
        other: "menos de {{count}} segundos"
      },
      xSeconds: {
        one: "1 segundo",
        other: "{{count}} segundos"
      },
      halfAMinute: "medio minuto",
      lessThanXMinutes: {
        one: "menos de un minuto",
        other: "menos de {{count}} minutos"
      },
      xMinutes: {
        one: "1 minuto",
        other: "{{count}} minutos"
      },
      aboutXHours: {
        one: "alrededor de 1 hora",
        other: "alrededor de {{count}} horas"
      },
      xHours: {
        one: "1 hora",
        other: "{{count}} horas"
      },
      xDays: {
        one: "1 día",
        other: "{{count}} días"
      },
      aboutXWeeks: {
        one: "alrededor de 1 semana",
        other: "alrededor de {{count}} semanas"
      },
      xWeeks: {
        one: "1 semana",
        other: "{{count}} semanas"
      },
      aboutXMonths: {
        one: "alrededor de 1 mes",
        other: "alrededor de {{count}} meses"
      },
      xMonths: {
        one: "1 mes",
        other: "{{count}} meses"
      },
      aboutXYears: {
        one: "alrededor de 1 año",
        other: "alrededor de {{count}} años"
      },
      xYears: {
        one: "1 año",
        other: "{{count}} años"
      },
      overXYears: {
        one: "más de 1 año",
        other: "más de {{count}} años"
      },
      almostXYears: {
        one: "casi 1 año",
        other: "casi {{count}} años"
      }
    };
    const formatDistance$l = (token, count, options) => {
      let result;
      const tokenValue = formatDistanceLocale$l[token];
      if (typeof tokenValue === "string") {
        result = tokenValue;
      } else if (count === 1) {
        result = tokenValue.one;
      } else {
        result = tokenValue.other.replace("{{count}}", count.toString());
      }
      if (options?.addSuffix) {
        if (options.comparison && options.comparison > 0) {
          return "en " + result;
        } else {
          return "hace " + result;
        }
      }
      return result;
    };

    const dateFormats$l = {
      full: "EEEE, d 'de' MMMM 'de' y",
      long: "d 'de' MMMM 'de' y",
      medium: "d MMM y",
      short: "dd/MM/y"
    };
    const timeFormats$l = {
      full: "HH:mm:ss zzzz",
      long: "HH:mm:ss z",
      medium: "HH:mm:ss",
      short: "HH:mm"
    };
    const dateTimeFormats$l = {
      full: "{{date}} 'a las' {{time}}",
      long: "{{date}} 'a las' {{time}}",
      medium: "{{date}}, {{time}}",
      short: "{{date}}, {{time}}"
    };
    const formatLong$l = {
      date: buildFormatLongFn({
        formats: dateFormats$l,
        defaultWidth: "full"
      }),
      time: buildFormatLongFn({
        formats: timeFormats$l,
        defaultWidth: "full"
      }),
      dateTime: buildFormatLongFn({
        formats: dateTimeFormats$l,
        defaultWidth: "full"
      })
    };

    const formatRelativeLocale$l = {
      lastWeek: "'el' eeee 'pasado a la' p",
      yesterday: "'ayer a la' p",
      today: "'hoy a la' p",
      tomorrow: "'mañana a la' p",
      nextWeek: "eeee 'a la' p",
      other: "P"
    };
    const formatRelativeLocalePlural = {
      lastWeek: "'el' eeee 'pasado a las' p",
      yesterday: "'ayer a las' p",
      today: "'hoy a las' p",
      tomorrow: "'mañana a las' p",
      nextWeek: "eeee 'a las' p",
      other: "P"
    };
    const formatRelative$l = (token, date, _baseDate, _options) => {
      if (date.getHours() !== 1) {
        return formatRelativeLocalePlural[token];
      } else {
        return formatRelativeLocale$l[token];
      }
    };

    const eraValues$l = {
      narrow: ["AC", "DC"],
      abbreviated: ["AC", "DC"],
      wide: ["antes de cristo", "después de cristo"]
    };
    const quarterValues$l = {
      narrow: ["1", "2", "3", "4"],
      abbreviated: ["T1", "T2", "T3", "T4"],
      wide: ["1º trimestre", "2º trimestre", "3º trimestre", "4º trimestre"]
    };
    const monthValues$l = {
      narrow: ["e", "f", "m", "a", "m", "j", "j", "a", "s", "o", "n", "d"],
      abbreviated: ["ene", "feb", "mar", "abr", "may", "jun", "jul", "ago", "sep", "oct", "nov", "dic"],
      wide: ["enero", "febrero", "marzo", "abril", "mayo", "junio", "julio", "agosto", "septiembre", "octubre", "noviembre", "diciembre"]
    };
    const dayValues$l = {
      narrow: ["d", "l", "m", "m", "j", "v", "s"],
      short: ["do", "lu", "ma", "mi", "ju", "vi", "sá"],
      abbreviated: ["dom", "lun", "mar", "mié", "jue", "vie", "sáb"],
      wide: ["domingo", "lunes", "martes", "miércoles", "jueves", "viernes", "sábado"]
    };
    const dayPeriodValues$l = {
      narrow: {
        am: "a",
        pm: "p",
        midnight: "mn",
        noon: "md",
        morning: "mañana",
        afternoon: "tarde",
        evening: "tarde",
        night: "noche"
      },
      abbreviated: {
        am: "AM",
        pm: "PM",
        midnight: "medianoche",
        noon: "mediodia",
        morning: "mañana",
        afternoon: "tarde",
        evening: "tarde",
        night: "noche"
      },
      wide: {
        am: "a.m.",
        pm: "p.m.",
        midnight: "medianoche",
        noon: "mediodia",
        morning: "mañana",
        afternoon: "tarde",
        evening: "tarde",
        night: "noche"
      }
    };
    const formattingDayPeriodValues$h = {
      narrow: {
        am: "a",
        pm: "p",
        midnight: "mn",
        noon: "md",
        morning: "de la mañana",
        afternoon: "de la tarde",
        evening: "de la tarde",
        night: "de la noche"
      },
      abbreviated: {
        am: "AM",
        pm: "PM",
        midnight: "medianoche",
        noon: "mediodia",
        morning: "de la mañana",
        afternoon: "de la tarde",
        evening: "de la tarde",
        night: "de la noche"
      },
      wide: {
        am: "a.m.",
        pm: "p.m.",
        midnight: "medianoche",
        noon: "mediodia",
        morning: "de la mañana",
        afternoon: "de la tarde",
        evening: "de la tarde",
        night: "de la noche"
      }
    };
    const ordinalNumber$l = (dirtyNumber, _options) => {
      const number = Number(dirtyNumber);
      return number + "º";
    };
    const localize$l = {
      ordinalNumber: ordinalNumber$l,
      era: buildLocalizeFn({
        values: eraValues$l,
        defaultWidth: "wide"
      }),
      quarter: buildLocalizeFn({
        values: quarterValues$l,
        defaultWidth: "wide",
        argumentCallback: quarter => Number(quarter) - 1
      }),
      month: buildLocalizeFn({
        values: monthValues$l,
        defaultWidth: "wide"
      }),
      day: buildLocalizeFn({
        values: dayValues$l,
        defaultWidth: "wide"
      }),
      dayPeriod: buildLocalizeFn({
        values: dayPeriodValues$l,
        defaultWidth: "wide",
        formattingValues: formattingDayPeriodValues$h,
        defaultFormattingWidth: "wide"
      })
    };

    const matchOrdinalNumberPattern$l = /^(\d+)(º)?/i;
    const parseOrdinalNumberPattern$l = /\d+/i;
    const matchEraPatterns$l = {
      narrow: /^(ac|dc|a|d)/i,
      abbreviated: /^(a\.?\s?c\.?|a\.?\s?e\.?\s?c\.?|d\.?\s?c\.?|e\.?\s?c\.?)/i,
      wide: /^(antes de cristo|antes de la era com[uú]n|despu[eé]s de cristo|era com[uú]n)/i
    };
    const parseEraPatterns$l = {
      any: [/^ac/i, /^dc/i],
      wide: [/^(antes de cristo|antes de la era com[uú]n)/i, /^(despu[eé]s de cristo|era com[uú]n)/i]
    };
    const matchQuarterPatterns$l = {
      narrow: /^[1234]/i,
      abbreviated: /^T[1234]/i,
      wide: /^[1234](º)? trimestre/i
    };
    const parseQuarterPatterns$l = {
      any: [/1/i, /2/i, /3/i, /4/i]
    };
    const matchMonthPatterns$l = {
      narrow: /^[efmajsond]/i,
      abbreviated: /^(ene|feb|mar|abr|may|jun|jul|ago|sep|oct|nov|dic)/i,
      wide: /^(enero|febrero|marzo|abril|mayo|junio|julio|agosto|septiembre|octubre|noviembre|diciembre)/i
    };
    const parseMonthPatterns$l = {
      narrow: [/^e/i, /^f/i, /^m/i, /^a/i, /^m/i, /^j/i, /^j/i, /^a/i, /^s/i, /^o/i, /^n/i, /^d/i],
      any: [/^en/i, /^feb/i, /^mar/i, /^abr/i, /^may/i, /^jun/i, /^jul/i, /^ago/i, /^sep/i, /^oct/i, /^nov/i, /^dic/i]
    };
    const matchDayPatterns$l = {
      narrow: /^[dlmjvs]/i,
      short: /^(do|lu|ma|mi|ju|vi|s[áa])/i,
      abbreviated: /^(dom|lun|mar|mi[ée]|jue|vie|s[áa]b)/i,
      wide: /^(domingo|lunes|martes|mi[ée]rcoles|jueves|viernes|s[áa]bado)/i
    };
    const parseDayPatterns$l = {
      narrow: [/^d/i, /^l/i, /^m/i, /^m/i, /^j/i, /^v/i, /^s/i],
      any: [/^do/i, /^lu/i, /^ma/i, /^mi/i, /^ju/i, /^vi/i, /^sa/i]
    };
    const matchDayPeriodPatterns$l = {
      narrow: /^(a|p|mn|md|(de la|a las) (mañana|tarde|noche))/i,
      any: /^([ap]\.?\s?m\.?|medianoche|mediodia|(de la|a las) (mañana|tarde|noche))/i
    };
    const parseDayPeriodPatterns$l = {
      any: {
        am: /^a/i,
        pm: /^p/i,
        midnight: /^mn/i,
        noon: /^md/i,
        morning: /mañana/i,
        afternoon: /tarde/i,
        evening: /tarde/i,
        night: /noche/i
      }
    };
    const match$l = {
      ordinalNumber: buildMatchPatternFn({
        matchPattern: matchOrdinalNumberPattern$l,
        parsePattern: parseOrdinalNumberPattern$l,
        valueCallback: function (value) {
          return parseInt(value, 10);
        }
      }),
      era: buildMatchFn({
        matchPatterns: matchEraPatterns$l,
        defaultMatchWidth: "wide",
        parsePatterns: parseEraPatterns$l,
        defaultParseWidth: "any"
      }),
      quarter: buildMatchFn({
        matchPatterns: matchQuarterPatterns$l,
        defaultMatchWidth: "wide",
        parsePatterns: parseQuarterPatterns$l,
        defaultParseWidth: "any",
        valueCallback: index => index + 1
      }),
      month: buildMatchFn({
        matchPatterns: matchMonthPatterns$l,
        defaultMatchWidth: "wide",
        parsePatterns: parseMonthPatterns$l,
        defaultParseWidth: "any"
      }),
      day: buildMatchFn({
        matchPatterns: matchDayPatterns$l,
        defaultMatchWidth: "wide",
        parsePatterns: parseDayPatterns$l,
        defaultParseWidth: "any"
      }),
      dayPeriod: buildMatchFn({
        matchPatterns: matchDayPeriodPatterns$l,
        defaultMatchWidth: "any",
        parsePatterns: parseDayPeriodPatterns$l,
        defaultParseWidth: "any"
      })
    };

    /**
     * @category Locales
     * @summary Spanish locale.
     * @language Spanish
     * @iso-639-2 spa
     * @author Juan Angosto [@juanangosto](https://github.com/juanangosto)
     * @author Guillermo Grau [@guigrpa](https://github.com/guigrpa)
     * @author Fernando Agüero [@fjaguero](https://github.com/fjaguero)
     * @author Gastón Haro [@harogaston](https://github.com/harogaston)
     * @author Yago Carballo [@YagoCarballo](https://github.com/YagoCarballo)
     */
    const es = {
      code: "es",
      formatDistance: formatDistance$l,
      formatLong: formatLong$l,
      formatRelative: formatRelative$l,
      localize: localize$l,
      match: match$l,
      options: {
        weekStartsOn: 1 /* Monday */,
        firstWeekContainsDate: 1
      }
    };

    const formatDistanceLocale$k = {
      lessThanXSeconds: {
        standalone: {
          one: "vähem kui üks sekund",
          other: "vähem kui {{count}} sekundit"
        },
        withPreposition: {
          one: "vähem kui ühe sekundi",
          other: "vähem kui {{count}} sekundi"
        }
      },
      xSeconds: {
        standalone: {
          one: "üks sekund",
          other: "{{count}} sekundit"
        },
        withPreposition: {
          one: "ühe sekundi",
          other: "{{count}} sekundi"
        }
      },
      halfAMinute: {
        standalone: "pool minutit",
        withPreposition: "poole minuti"
      },
      lessThanXMinutes: {
        standalone: {
          one: "vähem kui üks minut",
          other: "vähem kui {{count}} minutit"
        },
        withPreposition: {
          one: "vähem kui ühe minuti",
          other: "vähem kui {{count}} minuti"
        }
      },
      xMinutes: {
        standalone: {
          one: "üks minut",
          other: "{{count}} minutit"
        },
        withPreposition: {
          one: "ühe minuti",
          other: "{{count}} minuti"
        }
      },
      aboutXHours: {
        standalone: {
          one: "umbes üks tund",
          other: "umbes {{count}} tundi"
        },
        withPreposition: {
          one: "umbes ühe tunni",
          other: "umbes {{count}} tunni"
        }
      },
      xHours: {
        standalone: {
          one: "üks tund",
          other: "{{count}} tundi"
        },
        withPreposition: {
          one: "ühe tunni",
          other: "{{count}} tunni"
        }
      },
      xDays: {
        standalone: {
          one: "üks päev",
          other: "{{count}} päeva"
        },
        withPreposition: {
          one: "ühe päeva",
          other: "{{count}} päeva"
        }
      },
      aboutXWeeks: {
        standalone: {
          one: "umbes üks nädal",
          other: "umbes {{count}} nädalat"
        },
        withPreposition: {
          one: "umbes ühe nädala",
          other: "umbes {{count}} nädala"
        }
      },
      xWeeks: {
        standalone: {
          one: "üks nädal",
          other: "{{count}} nädalat"
        },
        withPreposition: {
          one: "ühe nädala",
          other: "{{count}} nädala"
        }
      },
      aboutXMonths: {
        standalone: {
          one: "umbes üks kuu",
          other: "umbes {{count}} kuud"
        },
        withPreposition: {
          one: "umbes ühe kuu",
          other: "umbes {{count}} kuu"
        }
      },
      xMonths: {
        standalone: {
          one: "üks kuu",
          other: "{{count}} kuud"
        },
        withPreposition: {
          one: "ühe kuu",
          other: "{{count}} kuu"
        }
      },
      aboutXYears: {
        standalone: {
          one: "umbes üks aasta",
          other: "umbes {{count}} aastat"
        },
        withPreposition: {
          one: "umbes ühe aasta",
          other: "umbes {{count}} aasta"
        }
      },
      xYears: {
        standalone: {
          one: "üks aasta",
          other: "{{count}} aastat"
        },
        withPreposition: {
          one: "ühe aasta",
          other: "{{count}} aasta"
        }
      },
      overXYears: {
        standalone: {
          one: "rohkem kui üks aasta",
          other: "rohkem kui {{count}} aastat"
        },
        withPreposition: {
          one: "rohkem kui ühe aasta",
          other: "rohkem kui {{count}} aasta"
        }
      },
      almostXYears: {
        standalone: {
          one: "peaaegu üks aasta",
          other: "peaaegu {{count}} aastat"
        },
        withPreposition: {
          one: "peaaegu ühe aasta",
          other: "peaaegu {{count}} aasta"
        }
      }
    };
    const formatDistance$k = (token, count, options) => {
      const usageGroup = options?.addSuffix ? formatDistanceLocale$k[token].withPreposition : formatDistanceLocale$k[token].standalone;
      let result;
      if (typeof usageGroup === "string") {
        result = usageGroup;
      } else if (count === 1) {
        result = usageGroup.one;
      } else {
        result = usageGroup.other.replace("{{count}}", String(count));
      }
      if (options?.addSuffix) {
        if (options.comparison && options.comparison > 0) {
          return result + " pärast";
        } else {
          return result + " eest";
        }
      }
      return result;
    };

    const dateFormats$k = {
      full: "EEEE, d. MMMM y",
      long: "d. MMMM y",
      medium: "d. MMM y",
      short: "dd.MM.y"
    };
    const timeFormats$k = {
      full: "HH:mm:ss zzzz",
      long: "HH:mm:ss z",
      medium: "HH:mm:ss",
      short: "HH:mm"
    };
    const dateTimeFormats$k = {
      full: "{{date}} 'kell' {{time}}",
      long: "{{date}} 'kell' {{time}}",
      medium: "{{date}}. {{time}}",
      short: "{{date}}. {{time}}"
    };
    const formatLong$k = {
      date: buildFormatLongFn({
        formats: dateFormats$k,
        defaultWidth: "full"
      }),
      time: buildFormatLongFn({
        formats: timeFormats$k,
        defaultWidth: "full"
      }),
      dateTime: buildFormatLongFn({
        formats: dateTimeFormats$k,
        defaultWidth: "full"
      })
    };

    const formatRelativeLocale$k = {
      lastWeek: "'eelmine' eeee 'kell' p",
      yesterday: "'eile kell' p",
      today: "'täna kell' p",
      tomorrow: "'homme kell' p",
      nextWeek: "'järgmine' eeee 'kell' p",
      other: "P"
    };
    const formatRelative$k = (token, _date, _baseDate, _options) => formatRelativeLocale$k[token];

    const eraValues$k = {
      narrow: ["e.m.a", "m.a.j"],
      abbreviated: ["e.m.a", "m.a.j"],
      wide: ["enne meie ajaarvamist", "meie ajaarvamise järgi"]
    };
    const quarterValues$k = {
      narrow: ["1", "2", "3", "4"],
      abbreviated: ["K1", "K2", "K3", "K4"],
      wide: ["1. kvartal", "2. kvartal", "3. kvartal", "4. kvartal"]
    };
    const monthValues$k = {
      narrow: ["J", "V", "M", "A", "M", "J", "J", "A", "S", "O", "N", "D"],
      abbreviated: ["jaan", "veebr", "märts", "apr", "mai", "juuni", "juuli", "aug", "sept", "okt", "nov", "dets"],
      wide: ["jaanuar", "veebruar", "märts", "aprill", "mai", "juuni", "juuli", "august", "september", "oktoober", "november", "detsember"]
    };
    const dayValues$k = {
      narrow: ["P", "E", "T", "K", "N", "R", "L"],
      short: ["P", "E", "T", "K", "N", "R", "L"],
      abbreviated: ["pühap.", "esmasp.", "teisip.", "kolmap.", "neljap.", "reede.", "laup."],
      wide: ["pühapäev", "esmaspäev", "teisipäev", "kolmapäev", "neljapäev", "reede", "laupäev"]
    };
    const dayPeriodValues$k = {
      narrow: {
        am: "AM",
        pm: "PM",
        midnight: "kesköö",
        noon: "keskpäev",
        morning: "hommik",
        afternoon: "pärastlõuna",
        evening: "õhtu",
        night: "öö"
      },
      abbreviated: {
        am: "AM",
        pm: "PM",
        midnight: "kesköö",
        noon: "keskpäev",
        morning: "hommik",
        afternoon: "pärastlõuna",
        evening: "õhtu",
        night: "öö"
      },
      wide: {
        am: "AM",
        pm: "PM",
        midnight: "kesköö",
        noon: "keskpäev",
        morning: "hommik",
        afternoon: "pärastlõuna",
        evening: "õhtu",
        night: "öö"
      }
    };
    const formattingDayPeriodValues$g = {
      narrow: {
        am: "AM",
        pm: "PM",
        midnight: "keskööl",
        noon: "keskpäeval",
        morning: "hommikul",
        afternoon: "pärastlõunal",
        evening: "õhtul",
        night: "öösel"
      },
      abbreviated: {
        am: "AM",
        pm: "PM",
        midnight: "keskööl",
        noon: "keskpäeval",
        morning: "hommikul",
        afternoon: "pärastlõunal",
        evening: "õhtul",
        night: "öösel"
      },
      wide: {
        am: "AM",
        pm: "PM",
        midnight: "keskööl",
        noon: "keskpäeval",
        morning: "hommikul",
        afternoon: "pärastlõunal",
        evening: "õhtul",
        night: "öösel"
      }
    };
    const ordinalNumber$k = (dirtyNumber, _options) => {
      const number = Number(dirtyNumber);
      return number + ".";
    };
    const localize$k = {
      ordinalNumber: ordinalNumber$k,
      era: buildLocalizeFn({
        values: eraValues$k,
        defaultWidth: "wide"
      }),
      quarter: buildLocalizeFn({
        values: quarterValues$k,
        defaultWidth: "wide",
        argumentCallback: quarter => quarter - 1
      }),
      month: buildLocalizeFn({
        values: monthValues$k,
        defaultWidth: "wide",
        formattingValues: monthValues$k,
        defaultFormattingWidth: "wide"
      }),
      day: buildLocalizeFn({
        values: dayValues$k,
        defaultWidth: "wide",
        formattingValues: dayValues$k,
        defaultFormattingWidth: "wide"
      }),
      dayPeriod: buildLocalizeFn({
        values: dayPeriodValues$k,
        defaultWidth: "wide",
        formattingValues: formattingDayPeriodValues$g,
        defaultFormattingWidth: "wide"
      })
    };

    const matchOrdinalNumberPattern$k = /^\d+\./i;
    const parseOrdinalNumberPattern$k = /\d+/i;
    const matchEraPatterns$k = {
      narrow: /^(e\.m\.a|m\.a\.j|eKr|pKr)/i,
      abbreviated: /^(e\.m\.a|m\.a\.j|eKr|pKr)/i,
      wide: /^(enne meie ajaarvamist|meie ajaarvamise järgi|enne Kristust|pärast Kristust)/i
    };
    const parseEraPatterns$k = {
      any: [/^e/i, /^(m|p)/i]
    };
    const matchQuarterPatterns$k = {
      narrow: /^[1234]/i,
      abbreviated: /^K[1234]/i,
      wide: /^[1234](\.)? kvartal/i
    };
    const parseQuarterPatterns$k = {
      any: [/1/i, /2/i, /3/i, /4/i]
    };
    const matchMonthPatterns$k = {
      narrow: /^[jvmasond]/i,
      abbreviated: /^(jaan|veebr|märts|apr|mai|juuni|juuli|aug|sept|okt|nov|dets)/i,
      wide: /^(jaanuar|veebruar|märts|aprill|mai|juuni|juuli|august|september|oktoober|november|detsember)/i
    };
    const parseMonthPatterns$k = {
      narrow: [/^j/i, /^v/i, /^m/i, /^a/i, /^m/i, /^j/i, /^j/i, /^a/i, /^s/i, /^o/i, /^n/i, /^d/i],
      any: [/^ja/i, /^v/i, /^mär/i, /^ap/i, /^mai/i, /^juun/i, /^juul/i, /^au/i, /^s/i, /^o/i, /^n/i, /^d/i]
    };
    const matchDayPatterns$k = {
      narrow: /^[petknrl]/i,
      short: /^[petknrl]/i,
      abbreviated: /^(püh?|esm?|tei?|kolm?|nel?|ree?|laup?)\.?/i,
      wide: /^(pühapäev|esmaspäev|teisipäev|kolmapäev|neljapäev|reede|laupäev)/i
    };
    const parseDayPatterns$k = {
      any: [/^p/i, /^e/i, /^t/i, /^k/i, /^n/i, /^r/i, /^l/i]
    };
    const matchDayPeriodPatterns$k = {
      any: /^(am|pm|keskööl?|keskpäev(al)?|hommik(ul)?|pärastlõunal?|õhtul?|öö(sel)?)/i
    };
    const parseDayPeriodPatterns$k = {
      any: {
        am: /^a/i,
        pm: /^p/i,
        midnight: /^keskö/i,
        noon: /^keskp/i,
        morning: /hommik/i,
        afternoon: /pärastlõuna/i,
        evening: /õhtu/i,
        night: /öö/i
      }
    };
    const match$k = {
      ordinalNumber: buildMatchPatternFn({
        matchPattern: matchOrdinalNumberPattern$k,
        parsePattern: parseOrdinalNumberPattern$k,
        valueCallback: value => parseInt(value, 10)
      }),
      era: buildMatchFn({
        matchPatterns: matchEraPatterns$k,
        defaultMatchWidth: "wide",
        parsePatterns: parseEraPatterns$k,
        defaultParseWidth: "any"
      }),
      quarter: buildMatchFn({
        matchPatterns: matchQuarterPatterns$k,
        defaultMatchWidth: "wide",
        parsePatterns: parseQuarterPatterns$k,
        defaultParseWidth: "any",
        valueCallback: index => index + 1
      }),
      month: buildMatchFn({
        matchPatterns: matchMonthPatterns$k,
        defaultMatchWidth: "wide",
        parsePatterns: parseMonthPatterns$k,
        defaultParseWidth: "any"
      }),
      day: buildMatchFn({
        matchPatterns: matchDayPatterns$k,
        defaultMatchWidth: "wide",
        parsePatterns: parseDayPatterns$k,
        defaultParseWidth: "any"
      }),
      dayPeriod: buildMatchFn({
        matchPatterns: matchDayPeriodPatterns$k,
        defaultMatchWidth: "any",
        parsePatterns: parseDayPeriodPatterns$k,
        defaultParseWidth: "any"
      })
    };

    /**
     * @category Locales
     * @summary Estonian locale.
     * @language Estonian
     * @iso-639-2 est
     * @author Priit Hansen [@HansenPriit](https://github.com/priithansen)
     */
    const et = {
      code: "et",
      formatDistance: formatDistance$k,
      formatLong: formatLong$k,
      formatRelative: formatRelative$k,
      localize: localize$k,
      match: match$k,
      options: {
        weekStartsOn: 1 /* Monday */,
        firstWeekContainsDate: 4
      }
    };

    const formatDistanceLocale$j = {
      lessThanXSeconds: {
        one: "کمتر از یک ثانیه",
        other: "کمتر از {{count}} ثانیه"
      },
      xSeconds: {
        one: "1 ثانیه",
        other: "{{count}} ثانیه"
      },
      halfAMinute: "نیم دقیقه",
      lessThanXMinutes: {
        one: "کمتر از یک دقیقه",
        other: "کمتر از {{count}} دقیقه"
      },
      xMinutes: {
        one: "1 دقیقه",
        other: "{{count}} دقیقه"
      },
      aboutXHours: {
        one: "حدود 1 ساعت",
        other: "حدود {{count}} ساعت"
      },
      xHours: {
        one: "1 ساعت",
        other: "{{count}} ساعت"
      },
      xDays: {
        one: "1 روز",
        other: "{{count}} روز"
      },
      aboutXWeeks: {
        one: "حدود 1 هفته",
        other: "حدود {{count}} هفته"
      },
      xWeeks: {
        one: "1 هفته",
        other: "{{count}} هفته"
      },
      aboutXMonths: {
        one: "حدود 1 ماه",
        other: "حدود {{count}} ماه"
      },
      xMonths: {
        one: "1 ماه",
        other: "{{count}} ماه"
      },
      aboutXYears: {
        one: "حدود 1 سال",
        other: "حدود {{count}} سال"
      },
      xYears: {
        one: "1 سال",
        other: "{{count}} سال"
      },
      overXYears: {
        one: "بیشتر از 1 سال",
        other: "بیشتر از {{count}} سال"
      },
      almostXYears: {
        one: "نزدیک 1 سال",
        other: "نزدیک {{count}} سال"
      }
    };
    const formatDistance$j = (token, count, options) => {
      let result;
      const tokenValue = formatDistanceLocale$j[token];
      if (typeof tokenValue === "string") {
        result = tokenValue;
      } else if (count === 1) {
        result = tokenValue.one;
      } else {
        result = tokenValue.other.replace("{{count}}", String(count));
      }
      if (options?.addSuffix) {
        if (options.comparison && options.comparison > 0) {
          return "در " + result;
        } else {
          return result + " قبل";
        }
      }
      return result;
    };

    const dateFormats$j = {
      full: "EEEE do MMMM y",
      long: "do MMMM y",
      medium: "d MMM y",
      short: "yyyy/MM/dd"
    };
    const timeFormats$j = {
      full: "h:mm:ss a zzzz",
      long: "h:mm:ss a z",
      medium: "h:mm:ss a",
      short: "h:mm a"
    };
    const dateTimeFormats$j = {
      full: "{{date}} 'در' {{time}}",
      long: "{{date}} 'در' {{time}}",
      medium: "{{date}}, {{time}}",
      short: "{{date}}, {{time}}"
    };
    const formatLong$j = {
      date: buildFormatLongFn({
        formats: dateFormats$j,
        defaultWidth: "full"
      }),
      time: buildFormatLongFn({
        formats: timeFormats$j,
        defaultWidth: "full"
      }),
      dateTime: buildFormatLongFn({
        formats: dateTimeFormats$j,
        defaultWidth: "full"
      })
    };

    const formatRelativeLocale$j = {
      lastWeek: "eeee 'گذشته در' p",
      yesterday: "'دیروز در' p",
      today: "'امروز در' p",
      tomorrow: "'فردا در' p",
      nextWeek: "eeee 'در' p",
      other: "P"
    };
    const formatRelative$j = (token, _date, _baseDate, _options) => formatRelativeLocale$j[token];

    const eraValues$j = {
      narrow: ["ق", "ب"],
      abbreviated: ["ق.م.", "ب.م."],
      wide: ["قبل از میلاد", "بعد از میلاد"]
    };
    const quarterValues$j = {
      narrow: ["1", "2", "3", "4"],
      abbreviated: ["س‌م1", "س‌م2", "س‌م3", "س‌م4"],
      wide: ["سه‌ماهه 1", "سه‌ماهه 2", "سه‌ماهه 3", "سه‌ماهه 4"]
    };

    // Note: in English, the names of days of the week and months are capitalized.
    // If you are making a new locale based on this one, check if the same is true for the language you're working on.
    // Generally, formatted dates should look like they are in the middle of a sentence,
    // e.g. in Spanish language the weekdays and months should be in the lowercase.
    const monthValues$j = {
      narrow: ["ژ", "ف", "م", "آ", "م", "ج", "ج", "آ", "س", "ا", "ن", "د"],
      abbreviated: ["ژانـ", "فور", "مارس", "آپر", "می", "جون", "جولـ", "آگو", "سپتـ", "اکتـ", "نوامـ", "دسامـ"],
      wide: ["ژانویه", "فوریه", "مارس", "آپریل", "می", "جون", "جولای", "آگوست", "سپتامبر", "اکتبر", "نوامبر", "دسامبر"]
    };
    const dayValues$j = {
      narrow: ["ی", "د", "س", "چ", "پ", "ج", "ش"],
      short: ["1ش", "2ش", "3ش", "4ش", "5ش", "ج", "ش"],
      abbreviated: ["یکشنبه", "دوشنبه", "سه‌شنبه", "چهارشنبه", "پنجشنبه", "جمعه", "شنبه"],
      wide: ["یکشنبه", "دوشنبه", "سه‌شنبه", "چهارشنبه", "پنجشنبه", "جمعه", "شنبه"]
    };
    const dayPeriodValues$j = {
      narrow: {
        am: "ق",
        pm: "ب",
        midnight: "ن",
        noon: "ظ",
        morning: "ص",
        afternoon: "ب.ظ.",
        evening: "ع",
        night: "ش"
      },
      abbreviated: {
        am: "ق.ظ.",
        pm: "ب.ظ.",
        midnight: "نیمه‌شب",
        noon: "ظهر",
        morning: "صبح",
        afternoon: "بعدازظهر",
        evening: "عصر",
        night: "شب"
      },
      wide: {
        am: "قبل‌ازظهر",
        pm: "بعدازظهر",
        midnight: "نیمه‌شب",
        noon: "ظهر",
        morning: "صبح",
        afternoon: "بعدازظهر",
        evening: "عصر",
        night: "شب"
      }
    };
    const formattingDayPeriodValues$f = {
      narrow: {
        am: "ق",
        pm: "ب",
        midnight: "ن",
        noon: "ظ",
        morning: "ص",
        afternoon: "ب.ظ.",
        evening: "ع",
        night: "ش"
      },
      abbreviated: {
        am: "ق.ظ.",
        pm: "ب.ظ.",
        midnight: "نیمه‌شب",
        noon: "ظهر",
        morning: "صبح",
        afternoon: "بعدازظهر",
        evening: "عصر",
        night: "شب"
      },
      wide: {
        am: "قبل‌ازظهر",
        pm: "بعدازظهر",
        midnight: "نیمه‌شب",
        noon: "ظهر",
        morning: "صبح",
        afternoon: "بعدازظهر",
        evening: "عصر",
        night: "شب"
      }
    };
    const ordinalNumber$j = (dirtyNumber, _options) => {
      return String(dirtyNumber);
    };
    const localize$j = {
      ordinalNumber: ordinalNumber$j,
      era: buildLocalizeFn({
        values: eraValues$j,
        defaultWidth: "wide"
      }),
      quarter: buildLocalizeFn({
        values: quarterValues$j,
        defaultWidth: "wide",
        argumentCallback: quarter => quarter - 1
      }),
      month: buildLocalizeFn({
        values: monthValues$j,
        defaultWidth: "wide"
      }),
      day: buildLocalizeFn({
        values: dayValues$j,
        defaultWidth: "wide"
      }),
      dayPeriod: buildLocalizeFn({
        values: dayPeriodValues$j,
        defaultWidth: "wide",
        formattingValues: formattingDayPeriodValues$f,
        defaultFormattingWidth: "wide"
      })
    };

    const matchOrdinalNumberPattern$j = /^(\d+)(th|st|nd|rd)?/i;
    const parseOrdinalNumberPattern$j = /\d+/i;
    const matchEraPatterns$j = {
      narrow: /^(ق|ب)/i,
      abbreviated: /^(ق\.?\s?م\.?|ق\.?\s?د\.?\s?م\.?|م\.?\s?|د\.?\s?م\.?)/i,
      wide: /^(قبل از میلاد|قبل از دوران مشترک|میلادی|دوران مشترک|بعد از میلاد)/i
    };
    const parseEraPatterns$j = {
      any: [/^قبل/i, /^بعد/i]
    };
    const matchQuarterPatterns$j = {
      narrow: /^[1234]/i,
      abbreviated: /^س‌م[1234]/i,
      wide: /^سه‌ماهه [1234]/i
    };
    const parseQuarterPatterns$j = {
      any: [/1/i, /2/i, /3/i, /4/i]
    };
    const matchMonthPatterns$j = {
      narrow: /^[جژفمآاماسند]/i,
      abbreviated: /^(جنو|ژانـ|ژانویه|فوریه|فور|مارس|آوریل|آپر|مه|می|ژوئن|جون|جول|جولـ|ژوئیه|اوت|آگو|سپتمبر|سپتامبر|اکتبر|اکتوبر|نوامبر|نوامـ|دسامبر|دسامـ|دسم)/i,
      wide: /^(ژانویه|جنوری|فبروری|فوریه|مارچ|مارس|آپریل|اپریل|ایپریل|آوریل|مه|می|ژوئن|جون|جولای|ژوئیه|آگست|اگست|آگوست|اوت|سپتمبر|سپتامبر|اکتبر|اکتوبر|نوامبر|نومبر|دسامبر|دسمبر)/i
    };
    const parseMonthPatterns$j = {
      narrow: [/^(ژ|ج)/i, /^ف/i, /^م/i, /^(آ|ا)/i, /^م/i, /^(ژ|ج)/i, /^(ج|ژ)/i, /^(آ|ا)/i, /^س/i, /^ا/i, /^ن/i, /^د/i],
      any: [/^ژا/i, /^ف/i, /^ما/i, /^آپ/i, /^(می|مه)/i, /^(ژوئن|جون)/i, /^(ژوئی|جول)/i, /^(اوت|آگ)/i, /^س/i, /^(اوک|اک)/i, /^ن/i, /^د/i]
    };
    const matchDayPatterns$j = {
      narrow: /^[شیدسچپج]/i,
      short: /^(ش|ج|1ش|2ش|3ش|4ش|5ش)/i,
      abbreviated: /^(یکشنبه|دوشنبه|سه‌شنبه|چهارشنبه|پنج‌شنبه|جمعه|شنبه)/i,
      wide: /^(یکشنبه|دوشنبه|سه‌شنبه|چهارشنبه|پنج‌شنبه|جمعه|شنبه)/i
    };
    const parseDayPatterns$j = {
      narrow: [/^ی/i, /^دو/i, /^س/i, /^چ/i, /^پ/i, /^ج/i, /^ش/i],
      any: [/^(ی|1ش|یکشنبه)/i, /^(د|2ش|دوشنبه)/i, /^(س|3ش|سه‌شنبه)/i, /^(چ|4ش|چهارشنبه)/i, /^(پ|5ش|پنجشنبه)/i, /^(ج|جمعه)/i, /^(ش|شنبه)/i]
    };
    const matchDayPeriodPatterns$j = {
      narrow: /^(ب|ق|ن|ظ|ص|ب.ظ.|ع|ش)/i,
      abbreviated: /^(ق.ظ.|ب.ظ.|نیمه‌شب|ظهر|صبح|بعدازظهر|عصر|شب)/i,
      wide: /^(قبل‌ازظهر|نیمه‌شب|ظهر|صبح|بعدازظهر|عصر|شب)/i
    };
    const parseDayPeriodPatterns$j = {
      any: {
        am: /^(ق|ق.ظ.|قبل‌ازظهر)/i,
        pm: /^(ب|ب.ظ.|بعدازظهر)/i,
        midnight: /^(‌نیمه‌شب|ن)/i,
        noon: /^(ظ|ظهر)/i,
        morning: /(ص|صبح)/i,
        afternoon: /(ب|ب.ظ.|بعدازظهر)/i,
        evening: /(ع|عصر)/i,
        night: /(ش|شب)/i
      }
    };
    const match$j = {
      ordinalNumber: buildMatchPatternFn({
        matchPattern: matchOrdinalNumberPattern$j,
        parsePattern: parseOrdinalNumberPattern$j,
        valueCallback: value => parseInt(value, 10)
      }),
      era: buildMatchFn({
        matchPatterns: matchEraPatterns$j,
        defaultMatchWidth: "wide",
        parsePatterns: parseEraPatterns$j,
        defaultParseWidth: "any"
      }),
      quarter: buildMatchFn({
        matchPatterns: matchQuarterPatterns$j,
        defaultMatchWidth: "wide",
        parsePatterns: parseQuarterPatterns$j,
        defaultParseWidth: "any",
        valueCallback: index => index + 1
      }),
      month: buildMatchFn({
        matchPatterns: matchMonthPatterns$j,
        defaultMatchWidth: "wide",
        parsePatterns: parseMonthPatterns$j,
        defaultParseWidth: "any"
      }),
      day: buildMatchFn({
        matchPatterns: matchDayPatterns$j,
        defaultMatchWidth: "wide",
        parsePatterns: parseDayPatterns$j,
        defaultParseWidth: "any"
      }),
      dayPeriod: buildMatchFn({
        matchPatterns: matchDayPeriodPatterns$j,
        defaultMatchWidth: "wide",
        parsePatterns: parseDayPeriodPatterns$j,
        defaultParseWidth: "any"
      })
    };

    /**
     * @category Locales
     * @summary Persian/Farsi locale (Iran).
     * @language Persian
     * @iso-639-2 ira
     * @author Morteza Ziyae [@mort3za](https://github.com/mort3za)
     */
    const faIR = {
      code: "fa-IR",
      formatDistance: formatDistance$j,
      formatLong: formatLong$j,
      formatRelative: formatRelative$j,
      localize: localize$j,
      match: match$j,
      options: {
        weekStartsOn: 6 /* Saturday */,
        firstWeekContainsDate: 1
      }
    };

    const formatDistanceLocale$i = {
      lessThanXSeconds: {
        one: "moins d’une seconde",
        other: "moins de {{count}} secondes"
      },
      xSeconds: {
        one: "1 seconde",
        other: "{{count}} secondes"
      },
      halfAMinute: "30 secondes",
      lessThanXMinutes: {
        one: "moins d’une minute",
        other: "moins de {{count}} minutes"
      },
      xMinutes: {
        one: "1 minute",
        other: "{{count}} minutes"
      },
      aboutXHours: {
        one: "environ 1 heure",
        other: "environ {{count}} heures"
      },
      xHours: {
        one: "1 heure",
        other: "{{count}} heures"
      },
      xDays: {
        one: "1 jour",
        other: "{{count}} jours"
      },
      aboutXWeeks: {
        one: "environ 1 semaine",
        other: "environ {{count}} semaines"
      },
      xWeeks: {
        one: "1 semaine",
        other: "{{count}} semaines"
      },
      aboutXMonths: {
        one: "environ 1 mois",
        other: "environ {{count}} mois"
      },
      xMonths: {
        one: "1 mois",
        other: "{{count}} mois"
      },
      aboutXYears: {
        one: "environ 1 an",
        other: "environ {{count}} ans"
      },
      xYears: {
        one: "1 an",
        other: "{{count}} ans"
      },
      overXYears: {
        one: "plus d’un an",
        other: "plus de {{count}} ans"
      },
      almostXYears: {
        one: "presqu’un an",
        other: "presque {{count}} ans"
      }
    };
    const formatDistance$i = (token, count, options) => {
      let result;
      const form = formatDistanceLocale$i[token];
      if (typeof form === "string") {
        result = form;
      } else if (count === 1) {
        result = form.one;
      } else {
        result = form.other.replace("{{count}}", String(count));
      }
      if (options?.addSuffix) {
        if (options.comparison && options.comparison > 0) {
          return "dans " + result;
        } else {
          return "il y a " + result;
        }
      }
      return result;
    };

    const dateFormats$i = {
      full: "EEEE d MMMM y",
      long: "d MMMM y",
      medium: "d MMM y",
      short: "dd/MM/y"
    };
    const timeFormats$i = {
      full: "HH:mm:ss zzzz",
      long: "HH:mm:ss z",
      medium: "HH:mm:ss",
      short: "HH:mm"
    };
    const dateTimeFormats$i = {
      full: "{{date}} 'à' {{time}}",
      long: "{{date}} 'à' {{time}}",
      medium: "{{date}}, {{time}}",
      short: "{{date}}, {{time}}"
    };
    const formatLong$i = {
      date: buildFormatLongFn({
        formats: dateFormats$i,
        defaultWidth: "full"
      }),
      time: buildFormatLongFn({
        formats: timeFormats$i,
        defaultWidth: "full"
      }),
      dateTime: buildFormatLongFn({
        formats: dateTimeFormats$i,
        defaultWidth: "full"
      })
    };

    const formatRelativeLocale$i = {
      lastWeek: "eeee 'dernier à' p",
      yesterday: "'hier à' p",
      today: "'aujourd’hui à' p",
      tomorrow: "'demain à' p'",
      nextWeek: "eeee 'prochain à' p",
      other: "P"
    };
    const formatRelative$i = (token, _date, _baseDate, _options) => formatRelativeLocale$i[token];

    const eraValues$i = {
      narrow: ["av. J.-C", "ap. J.-C"],
      abbreviated: ["av. J.-C", "ap. J.-C"],
      wide: ["avant Jésus-Christ", "après Jésus-Christ"]
    };
    const quarterValues$i = {
      narrow: ["T1", "T2", "T3", "T4"],
      abbreviated: ["1er trim.", "2ème trim.", "3ème trim.", "4ème trim."],
      wide: ["1er trimestre", "2ème trimestre", "3ème trimestre", "4ème trimestre"]
    };
    const monthValues$i = {
      narrow: ["J", "F", "M", "A", "M", "J", "J", "A", "S", "O", "N", "D"],
      abbreviated: ["janv.", "févr.", "mars", "avr.", "mai", "juin", "juil.", "août", "sept.", "oct.", "nov.", "déc."],
      wide: ["janvier", "février", "mars", "avril", "mai", "juin", "juillet", "août", "septembre", "octobre", "novembre", "décembre"]
    };
    const dayValues$i = {
      narrow: ["D", "L", "M", "M", "J", "V", "S"],
      short: ["di", "lu", "ma", "me", "je", "ve", "sa"],
      abbreviated: ["dim.", "lun.", "mar.", "mer.", "jeu.", "ven.", "sam."],
      wide: ["dimanche", "lundi", "mardi", "mercredi", "jeudi", "vendredi", "samedi"]
    };
    const dayPeriodValues$i = {
      narrow: {
        am: "AM",
        pm: "PM",
        midnight: "minuit",
        noon: "midi",
        morning: "mat.",
        afternoon: "ap.m.",
        evening: "soir",
        night: "mat."
      },
      abbreviated: {
        am: "AM",
        pm: "PM",
        midnight: "minuit",
        noon: "midi",
        morning: "matin",
        afternoon: "après-midi",
        evening: "soir",
        night: "matin"
      },
      wide: {
        am: "AM",
        pm: "PM",
        midnight: "minuit",
        noon: "midi",
        morning: "du matin",
        afternoon: "de l’après-midi",
        evening: "du soir",
        night: "du matin"
      }
    };
    const ordinalNumber$i = (dirtyNumber, options) => {
      const number = Number(dirtyNumber);
      const unit = options?.unit;
      if (number === 0) return "0";
      const feminineUnits = ["year", "week", "hour", "minute", "second"];
      let suffix;
      if (number === 1) {
        suffix = unit && feminineUnits.includes(unit) ? "ère" : "er";
      } else {
        suffix = "ème";
      }
      return number + suffix;
    };
    const LONG_MONTHS_TOKENS = ["MMM", "MMMM"];
    const localize$i = {
      preprocessor: (date, parts) => {
        // Replaces the `do` tokens with `d` when used with long month tokens and the day of the month is greater than one.
        // Use case "do MMMM" => 1er août, 29 août
        // see https://github.com/date-fns/date-fns/issues/1391

        if (date.getDate() === 1) return parts;
        const hasLongMonthToken = parts.some(part => part.isToken && LONG_MONTHS_TOKENS.includes(part.value));
        if (!hasLongMonthToken) return parts;
        return parts.map(part => part.isToken && part.value === "do" ? {
          isToken: true,
          value: "d"
        } : part);
      },
      ordinalNumber: ordinalNumber$i,
      era: buildLocalizeFn({
        values: eraValues$i,
        defaultWidth: "wide"
      }),
      quarter: buildLocalizeFn({
        values: quarterValues$i,
        defaultWidth: "wide",
        argumentCallback: quarter => quarter - 1
      }),
      month: buildLocalizeFn({
        values: monthValues$i,
        defaultWidth: "wide"
      }),
      day: buildLocalizeFn({
        values: dayValues$i,
        defaultWidth: "wide"
      }),
      dayPeriod: buildLocalizeFn({
        values: dayPeriodValues$i,
        defaultWidth: "wide"
      })
    };

    const matchOrdinalNumberPattern$i = /^(\d+)(ième|ère|ème|er|e)?/i;
    const parseOrdinalNumberPattern$i = /\d+/i;
    const matchEraPatterns$i = {
      narrow: /^(av\.J\.C|ap\.J\.C|ap\.J\.-C)/i,
      abbreviated: /^(av\.J\.-C|av\.J-C|apr\.J\.-C|apr\.J-C|ap\.J-C)/i,
      wide: /^(avant Jésus-Christ|après Jésus-Christ)/i
    };
    const parseEraPatterns$i = {
      any: [/^av/i, /^ap/i]
    };
    const matchQuarterPatterns$i = {
      narrow: /^T?[1234]/i,
      abbreviated: /^[1234](er|ème|e)? trim\.?/i,
      wide: /^[1234](er|ème|e)? trimestre/i
    };
    const parseQuarterPatterns$i = {
      any: [/1/i, /2/i, /3/i, /4/i]
    };
    const matchMonthPatterns$i = {
      narrow: /^[jfmasond]/i,
      abbreviated: /^(janv|févr|mars|avr|mai|juin|juill|juil|août|sept|oct|nov|déc)\.?/i,
      wide: /^(janvier|février|mars|avril|mai|juin|juillet|août|septembre|octobre|novembre|décembre)/i
    };
    const parseMonthPatterns$i = {
      narrow: [/^j/i, /^f/i, /^m/i, /^a/i, /^m/i, /^j/i, /^j/i, /^a/i, /^s/i, /^o/i, /^n/i, /^d/i],
      any: [/^ja/i, /^f/i, /^mar/i, /^av/i, /^ma/i, /^juin/i, /^juil/i, /^ao/i, /^s/i, /^o/i, /^n/i, /^d/i]
    };
    const matchDayPatterns$i = {
      narrow: /^[lmjvsd]/i,
      short: /^(di|lu|ma|me|je|ve|sa)/i,
      abbreviated: /^(dim|lun|mar|mer|jeu|ven|sam)\.?/i,
      wide: /^(dimanche|lundi|mardi|mercredi|jeudi|vendredi|samedi)/i
    };
    const parseDayPatterns$i = {
      narrow: [/^d/i, /^l/i, /^m/i, /^m/i, /^j/i, /^v/i, /^s/i],
      any: [/^di/i, /^lu/i, /^ma/i, /^me/i, /^je/i, /^ve/i, /^sa/i]
    };
    const matchDayPeriodPatterns$i = {
      narrow: /^(a|p|minuit|midi|mat\.?|ap\.?m\.?|soir|nuit)/i,
      any: /^([ap]\.?\s?m\.?|du matin|de l'après[-\s]midi|du soir|de la nuit)/i
    };
    const parseDayPeriodPatterns$i = {
      any: {
        am: /^a/i,
        pm: /^p/i,
        midnight: /^min/i,
        noon: /^mid/i,
        morning: /mat/i,
        afternoon: /ap/i,
        evening: /soir/i,
        night: /nuit/i
      }
    };
    const match$i = {
      ordinalNumber: buildMatchPatternFn({
        matchPattern: matchOrdinalNumberPattern$i,
        parsePattern: parseOrdinalNumberPattern$i,
        valueCallback: value => parseInt(value)
      }),
      era: buildMatchFn({
        matchPatterns: matchEraPatterns$i,
        defaultMatchWidth: "wide",
        parsePatterns: parseEraPatterns$i,
        defaultParseWidth: "any"
      }),
      quarter: buildMatchFn({
        matchPatterns: matchQuarterPatterns$i,
        defaultMatchWidth: "wide",
        parsePatterns: parseQuarterPatterns$i,
        defaultParseWidth: "any",
        valueCallback: index => index + 1
      }),
      month: buildMatchFn({
        matchPatterns: matchMonthPatterns$i,
        defaultMatchWidth: "wide",
        parsePatterns: parseMonthPatterns$i,
        defaultParseWidth: "any"
      }),
      day: buildMatchFn({
        matchPatterns: matchDayPatterns$i,
        defaultMatchWidth: "wide",
        parsePatterns: parseDayPatterns$i,
        defaultParseWidth: "any"
      }),
      dayPeriod: buildMatchFn({
        matchPatterns: matchDayPeriodPatterns$i,
        defaultMatchWidth: "any",
        parsePatterns: parseDayPeriodPatterns$i,
        defaultParseWidth: "any"
      })
    };

    /**
     * @category Locales
     * @summary French locale.
     * @language French
     * @iso-639-2 fra
     * @author Jean Dupouy [@izeau](https://github.com/izeau)
     * @author François B [@fbonzon](https://github.com/fbonzon)
     */
    const fr = {
      code: "fr",
      formatDistance: formatDistance$i,
      formatLong: formatLong$i,
      formatRelative: formatRelative$i,
      localize: localize$i,
      match: match$i,
      options: {
        weekStartsOn: 1 /* Monday */,
        firstWeekContainsDate: 4
      }
    };

    const formatDistanceLocale$h = {
      lessThanXSeconds: {
        one: "kurang dari 1 detik",
        other: "kurang dari {{count}} detik"
      },
      xSeconds: {
        one: "1 detik",
        other: "{{count}} detik"
      },
      halfAMinute: "setengah menit",
      lessThanXMinutes: {
        one: "kurang dari 1 menit",
        other: "kurang dari {{count}} menit"
      },
      xMinutes: {
        one: "1 menit",
        other: "{{count}} menit"
      },
      aboutXHours: {
        one: "sekitar 1 jam",
        other: "sekitar {{count}} jam"
      },
      xHours: {
        one: "1 jam",
        other: "{{count}} jam"
      },
      xDays: {
        one: "1 hari",
        other: "{{count}} hari"
      },
      aboutXWeeks: {
        one: "sekitar 1 minggu",
        other: "sekitar {{count}} minggu"
      },
      xWeeks: {
        one: "1 minggu",
        other: "{{count}} minggu"
      },
      aboutXMonths: {
        one: "sekitar 1 bulan",
        other: "sekitar {{count}} bulan"
      },
      xMonths: {
        one: "1 bulan",
        other: "{{count}} bulan"
      },
      aboutXYears: {
        one: "sekitar 1 tahun",
        other: "sekitar {{count}} tahun"
      },
      xYears: {
        one: "1 tahun",
        other: "{{count}} tahun"
      },
      overXYears: {
        one: "lebih dari 1 tahun",
        other: "lebih dari {{count}} tahun"
      },
      almostXYears: {
        one: "hampir 1 tahun",
        other: "hampir {{count}} tahun"
      }
    };
    const formatDistance$h = (token, count, options) => {
      let result;
      const tokenValue = formatDistanceLocale$h[token];
      if (typeof tokenValue === "string") {
        result = tokenValue;
      } else if (count === 1) {
        result = tokenValue.one;
      } else {
        result = tokenValue.other.replace("{{count}}", count.toString());
      }
      if (options?.addSuffix) {
        if (options.comparison && options.comparison > 0) {
          return "dalam waktu " + result;
        } else {
          return result + " yang lalu";
        }
      }
      return result;
    };

    const dateFormats$h = {
      full: "EEEE, d MMMM yyyy",
      long: "d MMMM yyyy",
      medium: "d MMM yyyy",
      short: "d/M/yyyy"
    };
    const timeFormats$h = {
      full: "HH.mm.ss",
      long: "HH.mm.ss",
      medium: "HH.mm",
      short: "HH.mm"
    };
    const dateTimeFormats$h = {
      full: "{{date}} 'pukul' {{time}}",
      long: "{{date}} 'pukul' {{time}}",
      medium: "{{date}}, {{time}}",
      short: "{{date}}, {{time}}"
    };
    const formatLong$h = {
      date: buildFormatLongFn({
        formats: dateFormats$h,
        defaultWidth: "full"
      }),
      time: buildFormatLongFn({
        formats: timeFormats$h,
        defaultWidth: "full"
      }),
      dateTime: buildFormatLongFn({
        formats: dateTimeFormats$h,
        defaultWidth: "full"
      })
    };

    const formatRelativeLocale$h = {
      lastWeek: "eeee 'lalu pukul' p",
      yesterday: "'Kemarin pukul' p",
      today: "'Hari ini pukul' p",
      tomorrow: "'Besok pukul' p",
      nextWeek: "eeee 'pukul' p",
      other: "P"
    };
    const formatRelative$h = (token, _date, _baseDate, _options) => formatRelativeLocale$h[token];

    // All data for localization are taken from this page
    // https://www.unicode.org/cldr/charts/32/summary/id.html
    const eraValues$h = {
      narrow: ["SM", "M"],
      abbreviated: ["SM", "M"],
      wide: ["Sebelum Masehi", "Masehi"]
    };
    const quarterValues$h = {
      narrow: ["1", "2", "3", "4"],
      abbreviated: ["K1", "K2", "K3", "K4"],
      wide: ["Kuartal ke-1", "Kuartal ke-2", "Kuartal ke-3", "Kuartal ke-4"]
    };

    // Note: in Indonesian, the names of days of the week and months are capitalized.
    // If you are making a new locale based on this one, check if the same is true for the language you're working on.
    // Generally, formatted dates should look like they are in the middle of a sentence,
    // e.g. in Spanish language the weekdays and months should be in the lowercase.
    const monthValues$h = {
      narrow: ["J", "F", "M", "A", "M", "J", "J", "A", "S", "O", "N", "D"],
      abbreviated: ["Jan", "Feb", "Mar", "Apr", "Mei", "Jun", "Jul", "Agt", "Sep", "Okt", "Nov", "Des"],
      wide: ["Januari", "Februari", "Maret", "April", "Mei", "Juni", "Juli", "Agustus", "September", "Oktober", "November", "Desember"]
    };
    const dayValues$h = {
      narrow: ["M", "S", "S", "R", "K", "J", "S"],
      short: ["Min", "Sen", "Sel", "Rab", "Kam", "Jum", "Sab"],
      abbreviated: ["Min", "Sen", "Sel", "Rab", "Kam", "Jum", "Sab"],
      wide: ["Minggu", "Senin", "Selasa", "Rabu", "Kamis", "Jumat", "Sabtu"]
    };
    const dayPeriodValues$h = {
      narrow: {
        am: "AM",
        pm: "PM",
        midnight: "tengah malam",
        noon: "tengah hari",
        morning: "pagi",
        afternoon: "siang",
        evening: "sore",
        night: "malam"
      },
      abbreviated: {
        am: "AM",
        pm: "PM",
        midnight: "tengah malam",
        noon: "tengah hari",
        morning: "pagi",
        afternoon: "siang",
        evening: "sore",
        night: "malam"
      },
      wide: {
        am: "AM",
        pm: "PM",
        midnight: "tengah malam",
        noon: "tengah hari",
        morning: "pagi",
        afternoon: "siang",
        evening: "sore",
        night: "malam"
      }
    };
    const formattingDayPeriodValues$e = {
      narrow: {
        am: "AM",
        pm: "PM",
        midnight: "tengah malam",
        noon: "tengah hari",
        morning: "pagi",
        afternoon: "siang",
        evening: "sore",
        night: "malam"
      },
      abbreviated: {
        am: "AM",
        pm: "PM",
        midnight: "tengah malam",
        noon: "tengah hari",
        morning: "pagi",
        afternoon: "siang",
        evening: "sore",
        night: "malam"
      },
      wide: {
        am: "AM",
        pm: "PM",
        midnight: "tengah malam",
        noon: "tengah hari",
        morning: "pagi",
        afternoon: "siang",
        evening: "sore",
        night: "malam"
      }
    };
    const ordinalNumber$h = (dirtyNumber, _options) => {
      const number = Number(dirtyNumber);

      // Can't use "pertama", "kedua" because can't be parsed

      return "ke-" + number;
    };
    const localize$h = {
      ordinalNumber: ordinalNumber$h,
      era: buildLocalizeFn({
        values: eraValues$h,
        defaultWidth: "wide"
      }),
      quarter: buildLocalizeFn({
        values: quarterValues$h,
        defaultWidth: "wide",
        argumentCallback: quarter => quarter - 1
      }),
      month: buildLocalizeFn({
        values: monthValues$h,
        defaultWidth: "wide"
      }),
      day: buildLocalizeFn({
        values: dayValues$h,
        defaultWidth: "wide"
      }),
      dayPeriod: buildLocalizeFn({
        values: dayPeriodValues$h,
        defaultWidth: "wide",
        formattingValues: formattingDayPeriodValues$e,
        defaultFormattingWidth: "wide"
      })
    };

    const matchOrdinalNumberPattern$h = /^ke-(\d+)?/i;
    const parseOrdinalNumberPattern$h = /\d+/i;
    const matchEraPatterns$h = {
      narrow: /^(sm|m)/i,
      abbreviated: /^(s\.?\s?m\.?|s\.?\s?e\.?\s?u\.?|m\.?|e\.?\s?u\.?)/i,
      wide: /^(sebelum masehi|sebelum era umum|masehi|era umum)/i
    };
    const parseEraPatterns$h = {
      any: [/^s/i, /^(m|e)/i]
    };
    const matchQuarterPatterns$h = {
      narrow: /^[1234]/i,
      abbreviated: /^K-?\s[1234]/i,
      wide: /^Kuartal ke-?\s?[1234]/i
    };
    const parseQuarterPatterns$h = {
      any: [/1/i, /2/i, /3/i, /4/i]
    };
    const matchMonthPatterns$h = {
      narrow: /^[jfmasond]/i,
      abbreviated: /^(jan|feb|mar|apr|mei|jun|jul|agt|sep|okt|nov|des)/i,
      wide: /^(januari|februari|maret|april|mei|juni|juli|agustus|september|oktober|november|desember)/i
    };
    const parseMonthPatterns$h = {
      narrow: [/^j/i, /^f/i, /^m/i, /^a/i, /^m/i, /^j/i, /^j/i, /^a/i, /^s/i, /^o/i, /^n/i, /^d/i],
      any: [/^ja/i, /^f/i, /^ma/i, /^ap/i, /^me/i, /^jun/i, /^jul/i, /^ag/i, /^s/i, /^o/i, /^n/i, /^d/i]
    };
    const matchDayPatterns$h = {
      narrow: /^[srkjm]/i,
      short: /^(min|sen|sel|rab|kam|jum|sab)/i,
      abbreviated: /^(min|sen|sel|rab|kam|jum|sab)/i,
      wide: /^(minggu|senin|selasa|rabu|kamis|jumat|sabtu)/i
    };
    const parseDayPatterns$h = {
      narrow: [/^m/i, /^s/i, /^s/i, /^r/i, /^k/i, /^j/i, /^s/i],
      any: [/^m/i, /^sen/i, /^sel/i, /^r/i, /^k/i, /^j/i, /^sa/i]
    };
    const matchDayPeriodPatterns$h = {
      narrow: /^(a|p|tengah m|tengah h|(di(\swaktu)?) (pagi|siang|sore|malam))/i,
      any: /^([ap]\.?\s?m\.?|tengah malam|tengah hari|(di(\swaktu)?) (pagi|siang|sore|malam))/i
    };
    const parseDayPeriodPatterns$h = {
      any: {
        am: /^a/i,
        pm: /^pm/i,
        midnight: /^tengah m/i,
        noon: /^tengah h/i,
        morning: /pagi/i,
        afternoon: /siang/i,
        evening: /sore/i,
        night: /malam/i
      }
    };
    const match$h = {
      ordinalNumber: buildMatchPatternFn({
        matchPattern: matchOrdinalNumberPattern$h,
        parsePattern: parseOrdinalNumberPattern$h,
        valueCallback: value => parseInt(value, 10)
      }),
      era: buildMatchFn({
        matchPatterns: matchEraPatterns$h,
        defaultMatchWidth: "wide",
        parsePatterns: parseEraPatterns$h,
        defaultParseWidth: "any"
      }),
      quarter: buildMatchFn({
        matchPatterns: matchQuarterPatterns$h,
        defaultMatchWidth: "wide",
        parsePatterns: parseQuarterPatterns$h,
        defaultParseWidth: "any",
        valueCallback: index => index + 1
      }),
      month: buildMatchFn({
        matchPatterns: matchMonthPatterns$h,
        defaultMatchWidth: "wide",
        parsePatterns: parseMonthPatterns$h,
        defaultParseWidth: "any"
      }),
      day: buildMatchFn({
        matchPatterns: matchDayPatterns$h,
        defaultMatchWidth: "wide",
        parsePatterns: parseDayPatterns$h,
        defaultParseWidth: "any"
      }),
      dayPeriod: buildMatchFn({
        matchPatterns: matchDayPeriodPatterns$h,
        defaultMatchWidth: "any",
        parsePatterns: parseDayPeriodPatterns$h,
        defaultParseWidth: "any"
      })
    };

    /**
     * @category Locales
     * @summary Indonesian locale.
     * @language Indonesian
     * @iso-639-2 ind
     * @author Rahmat Budiharso [@rbudiharso](https://github.com/rbudiharso)
     * @author Benget Nata [@bentinata](https://github.com/bentinata)
     * @author Budi Irawan [@deerawan](https://github.com/deerawan)
     * @author Try Ajitiono [@imballinst](https://github.com/imballinst)
     */
    const id = {
      code: "id",
      formatDistance: formatDistance$h,
      formatLong: formatLong$h,
      formatRelative: formatRelative$h,
      localize: localize$h,
      match: match$h,
      options: {
        weekStartsOn: 1 /* Monday */,
        firstWeekContainsDate: 1
      }
    };

    const formatDistanceLocale$g = {
      lessThanXSeconds: {
        one: "meno di un secondo",
        other: "meno di {{count}} secondi"
      },
      xSeconds: {
        one: "un secondo",
        other: "{{count}} secondi"
      },
      halfAMinute: "alcuni secondi",
      lessThanXMinutes: {
        one: "meno di un minuto",
        other: "meno di {{count}} minuti"
      },
      xMinutes: {
        one: "un minuto",
        other: "{{count}} minuti"
      },
      aboutXHours: {
        one: "circa un'ora",
        other: "circa {{count}} ore"
      },
      xHours: {
        one: "un'ora",
        other: "{{count}} ore"
      },
      xDays: {
        one: "un giorno",
        other: "{{count}} giorni"
      },
      aboutXWeeks: {
        one: "circa una settimana",
        other: "circa {{count}} settimane"
      },
      xWeeks: {
        one: "una settimana",
        other: "{{count}} settimane"
      },
      aboutXMonths: {
        one: "circa un mese",
        other: "circa {{count}} mesi"
      },
      xMonths: {
        one: "un mese",
        other: "{{count}} mesi"
      },
      aboutXYears: {
        one: "circa un anno",
        other: "circa {{count}} anni"
      },
      xYears: {
        one: "un anno",
        other: "{{count}} anni"
      },
      overXYears: {
        one: "più di un anno",
        other: "più di {{count}} anni"
      },
      almostXYears: {
        one: "quasi un anno",
        other: "quasi {{count}} anni"
      }
    };
    const formatDistance$g = (token, count, options) => {
      let result;
      const tokenValue = formatDistanceLocale$g[token];
      if (typeof tokenValue === "string") {
        result = tokenValue;
      } else if (count === 1) {
        result = tokenValue.one;
      } else {
        result = tokenValue.other.replace("{{count}}", count.toString());
      }
      if (options?.addSuffix) {
        if (options.comparison && options.comparison > 0) {
          return "tra " + result;
        } else {
          return result + " fa";
        }
      }
      return result;
    };

    const dateFormats$g = {
      full: "EEEE d MMMM y",
      long: "d MMMM y",
      medium: "d MMM y",
      short: "dd/MM/y"
    };
    const timeFormats$g = {
      full: "HH:mm:ss zzzz",
      long: "HH:mm:ss z",
      medium: "HH:mm:ss",
      short: "HH:mm"
    };
    const dateTimeFormats$g = {
      full: "{{date}} {{time}}",
      long: "{{date}} {{time}}",
      medium: "{{date}} {{time}}",
      short: "{{date}} {{time}}"
    };
    const formatLong$g = {
      date: buildFormatLongFn({
        formats: dateFormats$g,
        defaultWidth: "full"
      }),
      time: buildFormatLongFn({
        formats: timeFormats$g,
        defaultWidth: "full"
      }),
      dateTime: buildFormatLongFn({
        formats: dateTimeFormats$g,
        defaultWidth: "full"
      })
    };

    const weekdays = ["domenica", "lunedì", "martedì", "mercoledì", "giovedì", "venerdì", "sabato"];
    function lastWeek$3(day) {
      switch (day) {
        case 0:
          return "'domenica scorsa alle' p";
        default:
          return "'" + weekdays[day] + " scorso alle' p";
      }
    }
    function thisWeek$3(day) {
      return "'" + weekdays[day] + " alle' p";
    }
    function nextWeek$3(day) {
      switch (day) {
        case 0:
          return "'domenica prossima alle' p";
        default:
          return "'" + weekdays[day] + " prossimo alle' p";
      }
    }
    const formatRelativeLocale$g = {
      lastWeek: (date, baseDate, options) => {
        const day = date.getDay();
        if (isSameWeek(date, baseDate, options)) {
          return thisWeek$3(day);
        } else {
          return lastWeek$3(day);
        }
      },
      yesterday: "'ieri alle' p",
      today: "'oggi alle' p",
      tomorrow: "'domani alle' p",
      nextWeek: (date, baseDate, options) => {
        const day = date.getDay();
        if (isSameWeek(date, baseDate, options)) {
          return thisWeek$3(day);
        } else {
          return nextWeek$3(day);
        }
      },
      other: "P"
    };
    const formatRelative$g = (token, date, baseDate, options) => {
      const format = formatRelativeLocale$g[token];
      if (typeof format === "function") {
        return format(date, baseDate, options);
      }
      return format;
    };

    const eraValues$g = {
      narrow: ["aC", "dC"],
      abbreviated: ["a.C.", "d.C."],
      wide: ["avanti Cristo", "dopo Cristo"]
    };
    const quarterValues$g = {
      narrow: ["1", "2", "3", "4"],
      abbreviated: ["T1", "T2", "T3", "T4"],
      wide: ["1º trimestre", "2º trimestre", "3º trimestre", "4º trimestre"]
    };
    const monthValues$g = {
      narrow: ["G", "F", "M", "A", "M", "G", "L", "A", "S", "O", "N", "D"],
      abbreviated: ["gen", "feb", "mar", "apr", "mag", "giu", "lug", "ago", "set", "ott", "nov", "dic"],
      wide: ["gennaio", "febbraio", "marzo", "aprile", "maggio", "giugno", "luglio", "agosto", "settembre", "ottobre", "novembre", "dicembre"]
    };
    const dayValues$g = {
      narrow: ["D", "L", "M", "M", "G", "V", "S"],
      short: ["dom", "lun", "mar", "mer", "gio", "ven", "sab"],
      abbreviated: ["dom", "lun", "mar", "mer", "gio", "ven", "sab"],
      wide: ["domenica", "lunedì", "martedì", "mercoledì", "giovedì", "venerdì", "sabato"]
    };
    const dayPeriodValues$g = {
      narrow: {
        am: "m.",
        pm: "p.